"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Comprehend = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [comprehend](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Comprehend extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [comprehend](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'comprehend';
        this.actionList = {
            "BatchDetectDominantLanguage": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectDominantLanguage.html",
                "description": "Detects the language or languages present in the list of text documents.",
                "accessLevel": "Read"
            },
            "BatchDetectEntities": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectEntities.html",
                "description": "Detects the named entities (\"People\", \"Places\", \"Locations\", etc) within the given list of text documents.",
                "accessLevel": "Read"
            },
            "BatchDetectKeyPhrases": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectKeyPhrases.html",
                "description": "Detects the phrases in the list of text documents that are most indicative of the content.",
                "accessLevel": "Read"
            },
            "BatchDetectSentiment": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectSentiment.html",
                "description": "Detects the sentiment of a text in the list of documents (Positive, Negative, Neutral, or Mixed).",
                "accessLevel": "Read"
            },
            "BatchDetectSyntax": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectSyntax.html",
                "description": "Detects syntactic information (like Part of Speech, Tokens) in a list of text documents.",
                "accessLevel": "Read"
            },
            "ClassifyDocument": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ClassifyDocument.html",
                "description": "Creates a new document classification request to analyze a single document in real-time, using a previously created and trained custom model and an endpoint.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "document-classifier-endpoint": {
                        "required": true
                    }
                }
            },
            "CreateDocumentClassifier": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateDocumentClassifier.html",
                "description": "Creates a new document classifier that you can use to categorize documents.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateEndpoint": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateEndpoint.html",
                "description": "Creates a model-specific endpoint for synchronous inference for a previously trained custom model",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document-classifier": {
                        "required": true
                    },
                    "entity-recognizer": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateEntityRecognizer": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateEntityRecognizer.html",
                "description": "Creates an entity recognizer using submitted files.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteDocumentClassifier": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteDocumentClassifier.html",
                "description": "Deletes a previously created document classifier.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document-classifier": {
                        "required": true
                    }
                }
            },
            "DeleteEndpoint": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteEndpoint.html",
                "description": "Deletes a model-specific endpoint for a previously-trained custom model. All endpoints must be deleted in order for the model to be deleted.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document-classifier-endpoint": {
                        "required": true
                    },
                    "entity-recognizer-endpoint": {
                        "required": true
                    }
                }
            },
            "DeleteEntityRecognizer": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteEntityRecognizer.html",
                "description": "Deletes a submitted entity recognizer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "entity-recognizer": {
                        "required": true
                    }
                }
            },
            "DescribeDocumentClassificationJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDocumentClassificationJob.html",
                "description": "Gets the properties associated with a document classification job.",
                "accessLevel": "Read"
            },
            "DescribeDocumentClassifier": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDocumentClassifier.html",
                "description": "Gets the properties associated with a document classifier.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "document-classifier": {
                        "required": true
                    }
                }
            },
            "DescribeDominantLanguageDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDominantLanguageDetectionJob.html",
                "description": "Gets the properties associated with a dominant language detection job.",
                "accessLevel": "Read"
            },
            "DescribeEndpoint": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEndpoint.html",
                "description": "Gets the properties associated with a specific endpoint. Use this operation to get the status of an endpoint.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "document-classifier-endpoint": {
                        "required": true
                    },
                    "entity-recognizer-endpoint": {
                        "required": true
                    }
                }
            },
            "DescribeEntitiesDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEntitiesDetectionJob.html",
                "description": "Gets the properties associated with an entities detection job.",
                "accessLevel": "Read"
            },
            "DescribeEntityRecognizer": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEntityRecognizer.html",
                "description": "Provides details about an entity recognizer including status, S3 buckets containing training data, recognizer metadata, metrics, and so on.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "entity-recognizer": {
                        "required": true
                    }
                }
            },
            "DescribeKeyPhrasesDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeKeyPhrasesDetectionJob.html",
                "description": "Gets the properties associated with a key phrases detection job.",
                "accessLevel": "Read"
            },
            "DescribeSentimentDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeSentimentDetectionJob.html",
                "description": "Gets the properties associated with a sentiment detection job.",
                "accessLevel": "Read"
            },
            "DescribeTopicsDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeTopicsDetectionJob.html",
                "description": "Gets the properties associated with a topic detection job.",
                "accessLevel": "Read"
            },
            "DetectDominantLanguage": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectDominantLanguage.html",
                "description": "Detects the language or languages present in the text.",
                "accessLevel": "Read"
            },
            "DetectEntities": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectEntities.html",
                "description": "Detects the named entities (\"People\", \"Places\", \"Locations\", etc) within the given text document.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "entity-recognizer-endpoint": {
                        "required": false
                    }
                }
            },
            "DetectKeyPhrases": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectKeyPhrases.html",
                "description": "Detects the phrases in the text that are most indicative of the content.",
                "accessLevel": "Read"
            },
            "DetectSentiment": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectSentiment.html",
                "description": "Detects the sentiment of a text in a document (Positive, Negative, Neutral, or Mixed).",
                "accessLevel": "Read"
            },
            "DetectSyntax": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectSyntax.html",
                "description": "Detects syntactic information (like Part of Speech, Tokens) in a text document.",
                "accessLevel": "Read"
            },
            "ListDocumentClassificationJobs": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDocumentClassificationJobs.html",
                "description": "Gets a list of the document classification jobs that you have submitted.",
                "accessLevel": "List"
            },
            "ListDocumentClassifiers": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDocumentClassifiers.html",
                "description": "Gets a list of the document classifiers that you have created.",
                "accessLevel": "List"
            },
            "ListDominantLanguageDetectionJobs": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDominantLanguageDetectionJobs.html",
                "description": "Gets a list of the dominant language detection jobs that you have submitted.",
                "accessLevel": "List"
            },
            "ListEndpoints": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEndpoints.html",
                "description": "Gets a list of all existing endpoints that you've created.",
                "accessLevel": "List"
            },
            "ListEntitiesDetectionJobs": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEntitiesDetectionJobs.html",
                "description": "Gets a list of the entity detection jobs that you have submitted.",
                "accessLevel": "List"
            },
            "ListEntityRecognizers": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEntityRecognizers.html",
                "description": "Gets a list of the properties of all entity recognizers that you created, including recognizers currently in training.",
                "accessLevel": "List"
            },
            "ListKeyPhrasesDetectionJobs": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListKeyPhrasesDetectionJobs.html",
                "description": "Get a list of key phrase detection jobs that you have submitted.",
                "accessLevel": "List"
            },
            "ListSentimentDetectionJobs": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListSentimentDetectionJobs.html",
                "description": "Gets a list of sentiment detection jobs that you have submitted.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListTagsForResource.html",
                "description": "Lists tags for a resource",
                "accessLevel": "List",
                "resourceTypes": {
                    "document-classifier": {
                        "required": false
                    },
                    "document-classifier-endpoint": {
                        "required": false
                    },
                    "entity-recognizer": {
                        "required": false
                    },
                    "entity-recognizer-endpoint": {
                        "required": false
                    }
                }
            },
            "ListTopicsDetectionJobs": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListTopicsDetectionJobs.html",
                "description": "Gets a list of the topic detection jobs that you have submitted.",
                "accessLevel": "List"
            },
            "StartDocumentClassificationJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StartDocumentClassificationJob.html",
                "description": "Starts an asynchronous document classification job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document-classifier": {
                        "required": true
                    }
                }
            },
            "StartDominantLanguageDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StartDominantLanguageDetectionJob.html",
                "description": "Starts an asynchronous dominant language detection job for a collection of documents.",
                "accessLevel": "Write"
            },
            "StartEntitiesDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StartEntitiesDetectionJob.html",
                "description": "Starts an asynchronous entity detection job for a collection of documents.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "entity-recognizer": {
                        "required": false
                    }
                }
            },
            "StartKeyPhrasesDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StartKeyPhrasesDetectionJob.html",
                "description": "Starts an asynchronous key phrase detection job for a collection of documents.",
                "accessLevel": "Write"
            },
            "StartSentimentDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StartSentimentDetectionJob.html",
                "description": "Starts an asynchronous sentiment detection job for a collection of documents.",
                "accessLevel": "Write"
            },
            "StartTopicsDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StartTopicsDetectionJob.html",
                "description": "Starts an asynchronous job to detect the most common topics in the collection of documents and the phrases associated with each topic.",
                "accessLevel": "Write"
            },
            "StopDominantLanguageDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StopDominantLanguageDetectionJob.html",
                "description": "Stops a dominant language detection job.",
                "accessLevel": "Write"
            },
            "StopEntitiesDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StopEntitiesDetectionJob.html",
                "description": "Stops an entity detection job.",
                "accessLevel": "Write"
            },
            "StopKeyPhrasesDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StopKeyPhrasesDetectionJob.html",
                "description": "Stops a key phrase detection job.",
                "accessLevel": "Write"
            },
            "StopSentimentDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StopSentimentDetectionJob.html",
                "description": "Stops a sentiment detection job.",
                "accessLevel": "Write"
            },
            "StopTrainingDocumentClassifier": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StopTrainingDocumentClassifier.html",
                "description": "Stop a previously created document classifier training job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document-classifier": {
                        "required": true
                    }
                }
            },
            "StopTrainingEntityRecognizer": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StopTrainingEntityRecognizer.html",
                "description": "Stop a previously created entity recognizer training job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "entity-recognizer": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_TagResource.html",
                "description": "Tags a resource with given key value pairs",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "document-classifier": {
                        "required": false
                    },
                    "document-classifier-endpoint": {
                        "required": false
                    },
                    "entity-recognizer": {
                        "required": false
                    },
                    "entity-recognizer-endpoint": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_UntagResource.html",
                "description": "Untags a resource with given key",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "document-classifier": {
                        "required": false
                    },
                    "document-classifier-endpoint": {
                        "required": false
                    },
                    "entity-recognizer": {
                        "required": false
                    },
                    "entity-recognizer-endpoint": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateEndpoint": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_UpdateEndpoint.html",
                "description": "Updates information about the specified endpoint.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document-classifier-endpoint": {
                        "required": true
                    },
                    "entity-recognizer-endpoint": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "document-classifier": {
                "name": "document-classifier",
                "url": "",
                "arn": "arn:${Partition}:comprehend:${Region}:${Account}:document-classifier/${DocumentClassifierName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "entity-recognizer": {
                "name": "entity-recognizer",
                "url": "",
                "arn": "arn:${Partition}:comprehend:${Region}:${Account}:entity-recognizer/${EntityRecognizerName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "document-classifier-endpoint": {
                "name": "document-classifier-endpoint",
                "url": "",
                "arn": "arn:${Partition}:comprehend:${Region}:${Account}:document-classifier-endpoint/${DocumentClassifierEndpointName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "entity-recognizer-endpoint": {
                "name": "entity-recognizer-endpoint",
                "url": "",
                "arn": "arn:${Partition}:comprehend:${Region}:${Account}:entity-recognizer-endpoint/${EntityRecognizerEndpointName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Detects the language or languages present in the list of text documents.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectDominantLanguage.html
     */
    batchDetectDominantLanguage() {
        this.add('comprehend:BatchDetectDominantLanguage');
        return this;
    }
    /**
     * Detects the named entities ("People", "Places", "Locations", etc) within the given list of text documents.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectEntities.html
     */
    batchDetectEntities() {
        this.add('comprehend:BatchDetectEntities');
        return this;
    }
    /**
     * Detects the phrases in the list of text documents that are most indicative of the content.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectKeyPhrases.html
     */
    batchDetectKeyPhrases() {
        this.add('comprehend:BatchDetectKeyPhrases');
        return this;
    }
    /**
     * Detects the sentiment of a text in the list of documents (Positive, Negative, Neutral, or Mixed).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectSentiment.html
     */
    batchDetectSentiment() {
        this.add('comprehend:BatchDetectSentiment');
        return this;
    }
    /**
     * Detects syntactic information (like Part of Speech, Tokens) in a list of text documents.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectSyntax.html
     */
    batchDetectSyntax() {
        this.add('comprehend:BatchDetectSyntax');
        return this;
    }
    /**
     * Creates a new document classification request to analyze a single document in real-time, using a previously created and trained custom model and an endpoint.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ClassifyDocument.html
     */
    classifyDocument() {
        this.add('comprehend:ClassifyDocument');
        return this;
    }
    /**
     * Creates a new document classifier that you can use to categorize documents.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateDocumentClassifier.html
     */
    createDocumentClassifier() {
        this.add('comprehend:CreateDocumentClassifier');
        return this;
    }
    /**
     * Creates a model-specific endpoint for synchronous inference for a previously trained custom model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateEndpoint.html
     */
    createEndpoint() {
        this.add('comprehend:CreateEndpoint');
        return this;
    }
    /**
     * Creates an entity recognizer using submitted files.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateEntityRecognizer.html
     */
    createEntityRecognizer() {
        this.add('comprehend:CreateEntityRecognizer');
        return this;
    }
    /**
     * Deletes a previously created document classifier.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteDocumentClassifier.html
     */
    deleteDocumentClassifier() {
        this.add('comprehend:DeleteDocumentClassifier');
        return this;
    }
    /**
     * Deletes a model-specific endpoint for a previously-trained custom model. All endpoints must be deleted in order for the model to be deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteEndpoint.html
     */
    deleteEndpoint() {
        this.add('comprehend:DeleteEndpoint');
        return this;
    }
    /**
     * Deletes a submitted entity recognizer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteEntityRecognizer.html
     */
    deleteEntityRecognizer() {
        this.add('comprehend:DeleteEntityRecognizer');
        return this;
    }
    /**
     * Gets the properties associated with a document classification job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDocumentClassificationJob.html
     */
    describeDocumentClassificationJob() {
        this.add('comprehend:DescribeDocumentClassificationJob');
        return this;
    }
    /**
     * Gets the properties associated with a document classifier.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDocumentClassifier.html
     */
    describeDocumentClassifier() {
        this.add('comprehend:DescribeDocumentClassifier');
        return this;
    }
    /**
     * Gets the properties associated with a dominant language detection job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDominantLanguageDetectionJob.html
     */
    describeDominantLanguageDetectionJob() {
        this.add('comprehend:DescribeDominantLanguageDetectionJob');
        return this;
    }
    /**
     * Gets the properties associated with a specific endpoint. Use this operation to get the status of an endpoint.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEndpoint.html
     */
    describeEndpoint() {
        this.add('comprehend:DescribeEndpoint');
        return this;
    }
    /**
     * Gets the properties associated with an entities detection job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEntitiesDetectionJob.html
     */
    describeEntitiesDetectionJob() {
        this.add('comprehend:DescribeEntitiesDetectionJob');
        return this;
    }
    /**
     * Provides details about an entity recognizer including status, S3 buckets containing training data, recognizer metadata, metrics, and so on.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEntityRecognizer.html
     */
    describeEntityRecognizer() {
        this.add('comprehend:DescribeEntityRecognizer');
        return this;
    }
    /**
     * Gets the properties associated with a key phrases detection job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeKeyPhrasesDetectionJob.html
     */
    describeKeyPhrasesDetectionJob() {
        this.add('comprehend:DescribeKeyPhrasesDetectionJob');
        return this;
    }
    /**
     * Gets the properties associated with a sentiment detection job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeSentimentDetectionJob.html
     */
    describeSentimentDetectionJob() {
        this.add('comprehend:DescribeSentimentDetectionJob');
        return this;
    }
    /**
     * Gets the properties associated with a topic detection job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeTopicsDetectionJob.html
     */
    describeTopicsDetectionJob() {
        this.add('comprehend:DescribeTopicsDetectionJob');
        return this;
    }
    /**
     * Detects the language or languages present in the text.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectDominantLanguage.html
     */
    detectDominantLanguage() {
        this.add('comprehend:DetectDominantLanguage');
        return this;
    }
    /**
     * Detects the named entities ("People", "Places", "Locations", etc) within the given text document.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectEntities.html
     */
    detectEntities() {
        this.add('comprehend:DetectEntities');
        return this;
    }
    /**
     * Detects the phrases in the text that are most indicative of the content.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectKeyPhrases.html
     */
    detectKeyPhrases() {
        this.add('comprehend:DetectKeyPhrases');
        return this;
    }
    /**
     * Detects the sentiment of a text in a document (Positive, Negative, Neutral, or Mixed).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectSentiment.html
     */
    detectSentiment() {
        this.add('comprehend:DetectSentiment');
        return this;
    }
    /**
     * Detects syntactic information (like Part of Speech, Tokens) in a text document.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectSyntax.html
     */
    detectSyntax() {
        this.add('comprehend:DetectSyntax');
        return this;
    }
    /**
     * Gets a list of the document classification jobs that you have submitted.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDocumentClassificationJobs.html
     */
    listDocumentClassificationJobs() {
        this.add('comprehend:ListDocumentClassificationJobs');
        return this;
    }
    /**
     * Gets a list of the document classifiers that you have created.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDocumentClassifiers.html
     */
    listDocumentClassifiers() {
        this.add('comprehend:ListDocumentClassifiers');
        return this;
    }
    /**
     * Gets a list of the dominant language detection jobs that you have submitted.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDominantLanguageDetectionJobs.html
     */
    listDominantLanguageDetectionJobs() {
        this.add('comprehend:ListDominantLanguageDetectionJobs');
        return this;
    }
    /**
     * Gets a list of all existing endpoints that you've created.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEndpoints.html
     */
    listEndpoints() {
        this.add('comprehend:ListEndpoints');
        return this;
    }
    /**
     * Gets a list of the entity detection jobs that you have submitted.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEntitiesDetectionJobs.html
     */
    listEntitiesDetectionJobs() {
        this.add('comprehend:ListEntitiesDetectionJobs');
        return this;
    }
    /**
     * Gets a list of the properties of all entity recognizers that you created, including recognizers currently in training.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEntityRecognizers.html
     */
    listEntityRecognizers() {
        this.add('comprehend:ListEntityRecognizers');
        return this;
    }
    /**
     * Get a list of key phrase detection jobs that you have submitted.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListKeyPhrasesDetectionJobs.html
     */
    listKeyPhrasesDetectionJobs() {
        this.add('comprehend:ListKeyPhrasesDetectionJobs');
        return this;
    }
    /**
     * Gets a list of sentiment detection jobs that you have submitted.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListSentimentDetectionJobs.html
     */
    listSentimentDetectionJobs() {
        this.add('comprehend:ListSentimentDetectionJobs');
        return this;
    }
    /**
     * Lists tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('comprehend:ListTagsForResource');
        return this;
    }
    /**
     * Gets a list of the topic detection jobs that you have submitted.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListTopicsDetectionJobs.html
     */
    listTopicsDetectionJobs() {
        this.add('comprehend:ListTopicsDetectionJobs');
        return this;
    }
    /**
     * Starts an asynchronous document classification job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartDocumentClassificationJob.html
     */
    startDocumentClassificationJob() {
        this.add('comprehend:StartDocumentClassificationJob');
        return this;
    }
    /**
     * Starts an asynchronous dominant language detection job for a collection of documents.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartDominantLanguageDetectionJob.html
     */
    startDominantLanguageDetectionJob() {
        this.add('comprehend:StartDominantLanguageDetectionJob');
        return this;
    }
    /**
     * Starts an asynchronous entity detection job for a collection of documents.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartEntitiesDetectionJob.html
     */
    startEntitiesDetectionJob() {
        this.add('comprehend:StartEntitiesDetectionJob');
        return this;
    }
    /**
     * Starts an asynchronous key phrase detection job for a collection of documents.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartKeyPhrasesDetectionJob.html
     */
    startKeyPhrasesDetectionJob() {
        this.add('comprehend:StartKeyPhrasesDetectionJob');
        return this;
    }
    /**
     * Starts an asynchronous sentiment detection job for a collection of documents.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartSentimentDetectionJob.html
     */
    startSentimentDetectionJob() {
        this.add('comprehend:StartSentimentDetectionJob');
        return this;
    }
    /**
     * Starts an asynchronous job to detect the most common topics in the collection of documents and the phrases associated with each topic.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartTopicsDetectionJob.html
     */
    startTopicsDetectionJob() {
        this.add('comprehend:StartTopicsDetectionJob');
        return this;
    }
    /**
     * Stops a dominant language detection job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopDominantLanguageDetectionJob.html
     */
    stopDominantLanguageDetectionJob() {
        this.add('comprehend:StopDominantLanguageDetectionJob');
        return this;
    }
    /**
     * Stops an entity detection job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopEntitiesDetectionJob.html
     */
    stopEntitiesDetectionJob() {
        this.add('comprehend:StopEntitiesDetectionJob');
        return this;
    }
    /**
     * Stops a key phrase detection job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopKeyPhrasesDetectionJob.html
     */
    stopKeyPhrasesDetectionJob() {
        this.add('comprehend:StopKeyPhrasesDetectionJob');
        return this;
    }
    /**
     * Stops a sentiment detection job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopSentimentDetectionJob.html
     */
    stopSentimentDetectionJob() {
        this.add('comprehend:StopSentimentDetectionJob');
        return this;
    }
    /**
     * Stop a previously created document classifier training job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopTrainingDocumentClassifier.html
     */
    stopTrainingDocumentClassifier() {
        this.add('comprehend:StopTrainingDocumentClassifier');
        return this;
    }
    /**
     * Stop a previously created entity recognizer training job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopTrainingEntityRecognizer.html
     */
    stopTrainingEntityRecognizer() {
        this.add('comprehend:StopTrainingEntityRecognizer');
        return this;
    }
    /**
     * Tags a resource with given key value pairs
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_TagResource.html
     */
    tagResource() {
        this.add('comprehend:TagResource');
        return this;
    }
    /**
     * Untags a resource with given key
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_UntagResource.html
     */
    untagResource() {
        this.add('comprehend:UntagResource');
        return this;
    }
    /**
     * Updates information about the specified endpoint.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_UpdateEndpoint.html
     */
    updateEndpoint() {
        this.add('comprehend:UpdateEndpoint');
        return this;
    }
    /**
     * Adds a resource of type document-classifier to the statement
     *
     * @param documentClassifierName - Identifier for the documentClassifierName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentClassifier(documentClassifierName, account, region, partition) {
        var arn = 'arn:${Partition}:comprehend:${Region}:${Account}:document-classifier/${DocumentClassifierName}';
        arn = arn.replace('${DocumentClassifierName}', documentClassifierName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type entity-recognizer to the statement
     *
     * @param entityRecognizerName - Identifier for the entityRecognizerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEntityRecognizer(entityRecognizerName, account, region, partition) {
        var arn = 'arn:${Partition}:comprehend:${Region}:${Account}:entity-recognizer/${EntityRecognizerName}';
        arn = arn.replace('${EntityRecognizerName}', entityRecognizerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type document-classifier-endpoint to the statement
     *
     * @param documentClassifierEndpointName - Identifier for the documentClassifierEndpointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentClassifierEndpoint(documentClassifierEndpointName, account, region, partition) {
        var arn = 'arn:${Partition}:comprehend:${Region}:${Account}:document-classifier-endpoint/${DocumentClassifierEndpointName}';
        arn = arn.replace('${DocumentClassifierEndpointName}', documentClassifierEndpointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type entity-recognizer-endpoint to the statement
     *
     * @param entityRecognizerEndpointName - Identifier for the entityRecognizerEndpointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEntityRecognizerEndpoint(entityRecognizerEndpointName, account, region, partition) {
        var arn = 'arn:${Partition}:comprehend:${Region}:${Account}:entity-recognizer-endpoint/${EntityRecognizerEndpointName}';
        arn = arn.replace('${EntityRecognizerEndpointName}', entityRecognizerEndpointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters create requests based on the allowed set of values for each of the mandatory tags.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .createDocumentClassifier()
     * - .createEndpoint()
     * - .createEntityRecognizer()
     * - .tagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tag value associated with the resource.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to resource types:
     * - document-classifier
     * - entity-recognizer
     * - document-classifier-endpoint
     * - entity-recognizer-endpoint
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters create requests based on the presence of mandatory tags in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .createDocumentClassifier()
     * - .createEndpoint()
     * - .createEntityRecognizer()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Comprehend = Comprehend;
//# sourceMappingURL=data:application/json;base64,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