"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Es = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [es](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticsearchservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Es extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [es](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticsearchservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'es';
        this.actionList = {
            "AcceptInboundCrossClusterSearchConnection": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-acceptinboundcrossclustersearchconnection",
                "description": "Allows the destination domain owner to accept an inbound cross-cluster search connection request",
                "accessLevel": "Write"
            },
            "AddTags": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-addtags",
                "description": "Grants permission to attach resource tags to an Amazon ES domain.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CreateElasticsearchDomain": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createelasticsearchdomain",
                "description": "Grants permission to create an Amazon ES domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "CreateElasticsearchServiceRole": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createelasticsearchservicerole",
                "description": "Grants permission to create the service-linked role required for Amazon ES domains that use VPC access.",
                "accessLevel": "Write"
            },
            "CreateOutboundCrossClusterSearchConnection": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createoutboundcrossclustersearchconnection",
                "description": "Creates a new cross-cluster search connection from a source domain to a destination domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteElasticsearchDomain": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteelasticsearchdomain",
                "description": "Grants permission to delete an Amazon ES domain and all of its data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteElasticsearchServiceRole": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteelasticsearchservicerole",
                "description": "Grants permission to delete the service-linked role required for Amazon ES domains that use VPC access.",
                "accessLevel": "Write"
            },
            "DeleteInboundCrossClusterSearchConnection": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteinboundcrossclustersearchconnection",
                "description": "Allows the destination domain owner to delete an existing inbound cross-cluster search connection",
                "accessLevel": "Write"
            },
            "DeleteOutboundCrossClusterSearchConnection": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteoutboundcrossclustersearchconnection",
                "description": "Allows the source domain owner to delete an existing outbound cross-cluster search connection",
                "accessLevel": "Write"
            },
            "DescribeElasticsearchDomain": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomain",
                "description": "Grants permission to view a description of the domain configuration for the specified Amazon ES domain, including the domain ID, domain service endpoint, and domain ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeElasticsearchDomainConfig": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomainconfig",
                "description": "Grants permission to view a description of the configuration options and status of an Amazon ES domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeElasticsearchDomains": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomain",
                "description": "Grants permission to view a description of the domain configuration for up to five specified Amazon ES domains.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeElasticsearchInstanceTypeLimits": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeinstancetypelimits",
                "description": "Grants permission to view the instance count, storage, and master node limits for a given Elasticsearch version and instance type.",
                "accessLevel": "List"
            },
            "DescribeInboundCrossClusterSearchConnections": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeinboundcrossclustersearchconnections",
                "description": "Lists all the inbound cross-cluster search connections for a destination domain",
                "accessLevel": "List"
            },
            "DescribeOutboundCrossClusterSearchConnections": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeoutboundcrossclustersearchconnections",
                "description": "Lists all the outbound cross-cluster search connections for a source domain",
                "accessLevel": "List"
            },
            "DescribeReservedElasticsearchInstanceOfferings": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describereservedelasticsearchinstanceofferings",
                "description": "Grants permission to fetch reserved instance offerings for ES",
                "accessLevel": "List"
            },
            "DescribeReservedElasticsearchInstances": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describereservedelasticsearchinstances",
                "description": "Grants permission to fetch ES reserved instances already purchased by customer",
                "accessLevel": "List"
            },
            "ESCrossClusterGet": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send cross-cluster requests to a destination domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpDelete": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP DELETE requests to the Elasticsearch APIs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpGet": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP GET requests to the Elasticsearch APIs.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpHead": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP HEAD requests to the Elasticsearch APIs.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpPatch": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP PATCH requests to the Elasticsearch APIs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpPost": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP POST requests to the Elasticsearch APIs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ESHttpPut": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference",
                "description": "Grants permission to send HTTP PUT requests to the Elasticsearch APIs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "GetCompatibleElasticsearchVersions": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getcompatibleelasticsearchversions",
                "description": "Grants permission to fetch list of compatible elastic search versions to which Amazon ES domain can be upgraded",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "GetUpgradeHistory": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getupgradehistory",
                "description": "Grants permission to fetch upgrade history for given ES domain",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "GetUpgradeStatus": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getupgradestatus",
                "description": "Grants permission to fetch upgrade status for given ES domain",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ListDomainNames": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listdomainnames",
                "description": "Grants permission to display the names of all Amazon ES domains that the current user owns.",
                "accessLevel": "List"
            },
            "ListElasticsearchInstanceTypeDetails": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchinstancetypedetails",
                "description": "Grants permission to list all instance types and available features for a given Elasticsearch version.",
                "accessLevel": "List"
            },
            "ListElasticsearchInstanceTypes": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchinstancetypes",
                "description": "Grants permission to list all Elasticsearch instance types that are supported for a given Elasticsearch version.",
                "accessLevel": "List"
            },
            "ListElasticsearchVersions": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchversions",
                "description": "Grants permission to list all supported Elasticsearch versions on Amazon ES.",
                "accessLevel": "List"
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listtags",
                "description": "Grants permission to display all of the tags for an Amazon ES domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "PurchaseReservedElasticsearchInstanceOffering": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-purchasereservedelasticsearchinstance",
                "description": "Grants permission to purchase ES reserved instances",
                "accessLevel": "Write"
            },
            "RejectInboundCrossClusterSearchConnection": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-rejectinboundcrossclustersearchconnection",
                "description": "Allows the destination domain owner to reject an inbound cross-cluster search connection request",
                "accessLevel": "Write"
            },
            "RemoveTags": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listtags",
                "description": "Grants permission to remove tags from Amazon ES domains.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UpdateElasticsearchDomainConfig": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-updateelasticsearchdomainconfig",
                "description": "Grants permission to modify the configuration of an Amazon ES domain, such as the instance type or number of instances.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UpgradeElasticsearchDomain": {
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-upgradeelasticsearchdomain",
                "description": "Grants permission to initiate upgrade of elastic search domain to given version",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "domain": {
                "name": "domain",
                "url": "https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html",
                "arn": "arn:${Partition}:es:${Region}:${Account}:domain/${DomainName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Allows the destination domain owner to accept an inbound cross-cluster search connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-acceptinboundcrossclustersearchconnection
     */
    acceptInboundCrossClusterSearchConnection() {
        this.add('es:AcceptInboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Grants permission to attach resource tags to an Amazon ES domain.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-addtags
     */
    addTags() {
        this.add('es:AddTags');
        return this;
    }
    /**
     * Grants permission to create an Amazon ES domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createelasticsearchdomain
     */
    createElasticsearchDomain() {
        this.add('es:CreateElasticsearchDomain');
        return this;
    }
    /**
     * Grants permission to create the service-linked role required for Amazon ES domains that use VPC access.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createelasticsearchservicerole
     */
    createElasticsearchServiceRole() {
        this.add('es:CreateElasticsearchServiceRole');
        return this;
    }
    /**
     * Creates a new cross-cluster search connection from a source domain to a destination domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createoutboundcrossclustersearchconnection
     */
    createOutboundCrossClusterSearchConnection() {
        this.add('es:CreateOutboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Grants permission to delete an Amazon ES domain and all of its data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteelasticsearchdomain
     */
    deleteElasticsearchDomain() {
        this.add('es:DeleteElasticsearchDomain');
        return this;
    }
    /**
     * Grants permission to delete the service-linked role required for Amazon ES domains that use VPC access.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteelasticsearchservicerole
     */
    deleteElasticsearchServiceRole() {
        this.add('es:DeleteElasticsearchServiceRole');
        return this;
    }
    /**
     * Allows the destination domain owner to delete an existing inbound cross-cluster search connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteinboundcrossclustersearchconnection
     */
    deleteInboundCrossClusterSearchConnection() {
        this.add('es:DeleteInboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Allows the source domain owner to delete an existing outbound cross-cluster search connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteoutboundcrossclustersearchconnection
     */
    deleteOutboundCrossClusterSearchConnection() {
        this.add('es:DeleteOutboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Grants permission to view a description of the domain configuration for the specified Amazon ES domain, including the domain ID, domain service endpoint, and domain ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomain
     */
    describeElasticsearchDomain() {
        this.add('es:DescribeElasticsearchDomain');
        return this;
    }
    /**
     * Grants permission to view a description of the configuration options and status of an Amazon ES domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomainconfig
     */
    describeElasticsearchDomainConfig() {
        this.add('es:DescribeElasticsearchDomainConfig');
        return this;
    }
    /**
     * Grants permission to view a description of the domain configuration for up to five specified Amazon ES domains.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomain
     */
    describeElasticsearchDomains() {
        this.add('es:DescribeElasticsearchDomains');
        return this;
    }
    /**
     * Grants permission to view the instance count, storage, and master node limits for a given Elasticsearch version and instance type.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeinstancetypelimits
     */
    describeElasticsearchInstanceTypeLimits() {
        this.add('es:DescribeElasticsearchInstanceTypeLimits');
        return this;
    }
    /**
     * Lists all the inbound cross-cluster search connections for a destination domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeinboundcrossclustersearchconnections
     */
    describeInboundCrossClusterSearchConnections() {
        this.add('es:DescribeInboundCrossClusterSearchConnections');
        return this;
    }
    /**
     * Lists all the outbound cross-cluster search connections for a source domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeoutboundcrossclustersearchconnections
     */
    describeOutboundCrossClusterSearchConnections() {
        this.add('es:DescribeOutboundCrossClusterSearchConnections');
        return this;
    }
    /**
     * Grants permission to fetch reserved instance offerings for ES
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describereservedelasticsearchinstanceofferings
     */
    describeReservedElasticsearchInstanceOfferings() {
        this.add('es:DescribeReservedElasticsearchInstanceOfferings');
        return this;
    }
    /**
     * Grants permission to fetch ES reserved instances already purchased by customer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describereservedelasticsearchinstances
     */
    describeReservedElasticsearchInstances() {
        this.add('es:DescribeReservedElasticsearchInstances');
        return this;
    }
    /**
     * Grants permission to send cross-cluster requests to a destination domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    eSCrossClusterGet() {
        this.add('es:ESCrossClusterGet');
        return this;
    }
    /**
     * Grants permission to send HTTP DELETE requests to the Elasticsearch APIs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    eSHttpDelete() {
        this.add('es:ESHttpDelete');
        return this;
    }
    /**
     * Grants permission to send HTTP GET requests to the Elasticsearch APIs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    eSHttpGet() {
        this.add('es:ESHttpGet');
        return this;
    }
    /**
     * Grants permission to send HTTP HEAD requests to the Elasticsearch APIs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    eSHttpHead() {
        this.add('es:ESHttpHead');
        return this;
    }
    /**
     * Grants permission to send HTTP PATCH requests to the Elasticsearch APIs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    eSHttpPatch() {
        this.add('es:ESHttpPatch');
        return this;
    }
    /**
     * Grants permission to send HTTP POST requests to the Elasticsearch APIs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    eSHttpPost() {
        this.add('es:ESHttpPost');
        return this;
    }
    /**
     * Grants permission to send HTTP PUT requests to the Elasticsearch APIs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    eSHttpPut() {
        this.add('es:ESHttpPut');
        return this;
    }
    /**
     * Grants permission to fetch list of compatible elastic search versions to which Amazon ES domain can be upgraded
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getcompatibleelasticsearchversions
     */
    getCompatibleElasticsearchVersions() {
        this.add('es:GetCompatibleElasticsearchVersions');
        return this;
    }
    /**
     * Grants permission to fetch upgrade history for given ES domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getupgradehistory
     */
    getUpgradeHistory() {
        this.add('es:GetUpgradeHistory');
        return this;
    }
    /**
     * Grants permission to fetch upgrade status for given ES domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getupgradestatus
     */
    getUpgradeStatus() {
        this.add('es:GetUpgradeStatus');
        return this;
    }
    /**
     * Grants permission to display the names of all Amazon ES domains that the current user owns.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listdomainnames
     */
    listDomainNames() {
        this.add('es:ListDomainNames');
        return this;
    }
    /**
     * Grants permission to list all instance types and available features for a given Elasticsearch version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchinstancetypedetails
     */
    listElasticsearchInstanceTypeDetails() {
        this.add('es:ListElasticsearchInstanceTypeDetails');
        return this;
    }
    /**
     * Grants permission to list all Elasticsearch instance types that are supported for a given Elasticsearch version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchinstancetypes
     */
    listElasticsearchInstanceTypes() {
        this.add('es:ListElasticsearchInstanceTypes');
        return this;
    }
    /**
     * Grants permission to list all supported Elasticsearch versions on Amazon ES.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchversions
     */
    listElasticsearchVersions() {
        this.add('es:ListElasticsearchVersions');
        return this;
    }
    /**
     * Grants permission to display all of the tags for an Amazon ES domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listtags
     */
    listTags() {
        this.add('es:ListTags');
        return this;
    }
    /**
     * Grants permission to purchase ES reserved instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-purchasereservedelasticsearchinstance
     */
    purchaseReservedElasticsearchInstanceOffering() {
        this.add('es:PurchaseReservedElasticsearchInstanceOffering');
        return this;
    }
    /**
     * Allows the destination domain owner to reject an inbound cross-cluster search connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-rejectinboundcrossclustersearchconnection
     */
    rejectInboundCrossClusterSearchConnection() {
        this.add('es:RejectInboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Grants permission to remove tags from Amazon ES domains.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listtags
     */
    removeTags() {
        this.add('es:RemoveTags');
        return this;
    }
    /**
     * Grants permission to modify the configuration of an Amazon ES domain, such as the instance type or number of instances.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-updateelasticsearchdomainconfig
     */
    updateElasticsearchDomainConfig() {
        this.add('es:UpdateElasticsearchDomainConfig');
        return this;
    }
    /**
     * Grants permission to initiate upgrade of elastic search domain to given version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-upgradeelasticsearchdomain
     */
    upgradeElasticsearchDomain() {
        this.add('es:UpgradeElasticsearchDomain');
        return this;
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDomain(domainName, account, region, partition) {
        var arn = 'arn:${Partition}:es:${Region}:${Account}:domain/${DomainName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Es = Es;
//# sourceMappingURL=data:application/json;base64,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