"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Frauddetector = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [frauddetector](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonfrauddetector.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Frauddetector extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [frauddetector](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonfrauddetector.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'frauddetector';
        this.actionList = {
            "BatchCreateVariable": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_BatchCreateVariable",
                "description": "Creates a batch of variables.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "BatchGetVariable": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_BatchGetVariable",
                "description": "Gets a batch of variables.",
                "accessLevel": "List",
                "resourceTypes": {
                    "variable": {
                        "required": false
                    }
                }
            },
            "CreateDetectorVersion": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateDetectorVersion",
                "description": "Creates a detector version. The detector version starts in a DRAFT status.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateModel": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateModel",
                "description": "Creates a model using the specified model type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateModelVersion": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateModelVersion",
                "description": "Creates a version of the model using the specified model type and model id.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateRule": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateRule.html",
                "description": "Creates a rule for use with the specified detector.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateVariable": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateVariable.html",
                "description": "Creates a variable.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteDetector": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteDetector",
                "description": "Deletes the detector. Before deleting a detector, you must first delete all detector versions and rule versions associated with the detector.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "DeleteDetectorVersion": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteDetectorVersion",
                "description": "Deletes the detector version. You cannot delete detector versions that are in ACTIVE status.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector-version": {
                        "required": true
                    }
                }
            },
            "DeleteEvent": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteEvent",
                "description": "Deletes the specified event.",
                "accessLevel": "Write"
            },
            "DeleteRule": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteRule",
                "description": "Deletes the rule. You cannot delete a rule if it is used by an ACTIVE or INACTIVE detector version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "DescribeDetector": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_DescribeDetector",
                "description": "Gets all versions for a specified detector.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "DescribeModelVersions": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_DescribeModelVersions",
                "description": "Gets all of the model versions for the specified model type or for the specified model type and model ID. You can also get details for a single, specified model version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "model-version": {
                        "required": false
                    }
                }
            },
            "GetDetectorVersion": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetDetectorVersion",
                "description": "Gets a particular detector version.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector-version": {
                        "required": true
                    }
                }
            },
            "GetDetectors": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetDetectors",
                "description": "Gets all detectors or a single detector if a detectorId is specified. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetDetectorsResponse as part of your request. A null pagination token fetches the records from the beginning.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector": {
                        "required": false
                    }
                }
            },
            "GetEntityTypes": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEntityTypes",
                "description": "Gets all entity types or a specific entity type if a name is specified. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetEntityTypesResponse as part of your request. A null pagination token fetches the records from the beginning.",
                "accessLevel": "List",
                "resourceTypes": {
                    "entity-type": {
                        "required": false
                    }
                }
            },
            "GetEventPrediction": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEventPrediction",
                "description": "Evaluates an event against a detector version. If a version ID is not provided, the detector’s (ACTIVE) version is used.",
                "accessLevel": "Read"
            },
            "GetEventTypes": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEventTypes",
                "description": "Gets all event types or a specific event type if name is provided. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetEventTypesResponse as part of your request. A null pagination token fetches the records from the beginning.",
                "accessLevel": "List",
                "resourceTypes": {
                    "event-type": {
                        "required": false
                    }
                }
            },
            "GetExternalModels": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetExternalModels",
                "description": "Gets the details for one or more Amazon SageMaker models that have been imported into the service. This is a paginated API. If you provide a null maxResults, this actions retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetExternalModelsResult as part of your request. A null pagination token fetches the records from the beginning.",
                "accessLevel": "List",
                "resourceTypes": {
                    "external-model": {
                        "required": false
                    }
                }
            },
            "GetKMSEncryptionKey": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetKMSEncryptionKey",
                "description": "Gets the encryption key if a Key Management Service (KMS) customer master key (CMK) has been specified to be used to encrypt content in Amazon Fraud Detector.",
                "accessLevel": "Read"
            },
            "GetLabels": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetLabels",
                "description": "Gets all labels or a specific label if name is provided. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 50 records per page. If you provide a maxResults, the value must be between 10 and 50. To get the next page results, provide the pagination token from the GetGetLabelsResponse as part of your request. A null pagination token fetches the records from the beginning.",
                "accessLevel": "List",
                "resourceTypes": {
                    "label": {
                        "required": false
                    }
                }
            },
            "GetModelVersion": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetModelVersion",
                "description": "Gets the details of the specified model version.",
                "accessLevel": "List",
                "resourceTypes": {
                    "model-version": {
                        "required": true
                    }
                }
            },
            "GetModels": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetModels",
                "description": "Gets one or more models. Gets all models for the AWS account if no model type and no model id provided. Gets all models for the AWS account and model type, if the model type is specified but model id is not provided. Gets a specific model if (model type, model id) tuple is specified.",
                "accessLevel": "List",
                "resourceTypes": {
                    "model": {
                        "required": false
                    }
                }
            },
            "GetOutcomes": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetOutcomes",
                "description": "Gets one or more outcomes. This is a paginated API. If you provide a null maxResults, this actions retrieves a maximum of 100 records per page. If you provide a maxResults, the value must be between 50 and 100. To get the next page results, provide the pagination token from the GetOutcomesResult as part of your request. A null pagination token fetches the records from the beginning.",
                "accessLevel": "List",
                "resourceTypes": {
                    "outcome": {
                        "required": false
                    }
                }
            },
            "GetRules": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetRules.html",
                "description": "Get all rules for a detector (paginated) if ruleId and ruleVersion are not specified. Gets all rules for the detector and the ruleId if present (paginated). Gets a specific rule if both the ruleId and the ruleVersion are specified.",
                "accessLevel": "List",
                "resourceTypes": {
                    "rule": {
                        "required": false
                    }
                }
            },
            "GetVariables": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetVariables",
                "description": "Gets all of the variables or the specific variable. This is a paginated API. Providing null maxSizePerPage results in retrieving maximum of 100 records per page. If you provide maxSizePerPage the value must be between 50 and 100. To get the next page result, a provide a pagination token from GetVariablesResult as part of your request. Null pagination token fetches the records from the beginning.",
                "accessLevel": "List",
                "resourceTypes": {
                    "variable": {
                        "required": false
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_ListTagsForResource.html",
                "description": "Lists all tags associated with the resource. This is a paginated API. To get the next page results, provide the pagination token from the response as part of your request. A null pagination token fetches the records from the beginning.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector": {
                        "required": false
                    },
                    "detector-version": {
                        "required": false
                    },
                    "entity-type": {
                        "required": false
                    },
                    "event-type": {
                        "required": false
                    },
                    "external-model": {
                        "required": false
                    },
                    "label": {
                        "required": false
                    },
                    "model": {
                        "required": false
                    },
                    "model-version": {
                        "required": false
                    },
                    "outcome": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "variable": {
                        "required": false
                    }
                }
            },
            "PutDetector": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_PutDetector.html",
                "description": "Creates or updates a detector.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutEntityType": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_PutEntityType.html",
                "description": "Creates or updates an entity type. An entity represents who is performing the event. As part of a fraud prediction, you pass the entity ID to indicate the specific entity who performed the event. An entity type classifies the entity. Example classifications include customer, merchant, or account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "entity-type": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutEventType": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_PutEventType.html",
                "description": "Creates or updates an event type. An event is a business activity that is evaluated for fraud risk. With Amazon Fraud Detector, you generate fraud predictions for events. An event type defines the structure for an event sent to Amazon Fraud Detector. This includes the variables sent as part of the event, the entity performing the event (such as a customer), and the labels that classify the event. Example event types include online payment transactions, account registrations, and authentications.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "event-type": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutExternalModel": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_PutExternalModel.html",
                "description": "Creates or updates an Amazon SageMaker model endpoint. You can also use this action to update the configuration of the model endpoint, including the IAM role and/or the mapped variables.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "external-model": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutKMSEncryptionKey": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_PutKMSEncryptionKey.html",
                "description": "Specifies the Key Management Service (KMS) customer master key (CMK) to be used to encrypt content in Amazon Fraud Detector.",
                "accessLevel": "Write"
            },
            "PutLabel": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_PutLabel.html",
                "description": "Creates or updates label. A label classifies an event as fraudulent or legitimate. Labels are associated with event types and used to train supervised machine learning models in Amazon Fraud Detector.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "label": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutOutcome": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_PutOutcome.html",
                "description": "Creates or updates an outcome.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "outcome": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_TagResource.html",
                "description": "Assigns tags to a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "detector": {
                        "required": false
                    },
                    "detector-version": {
                        "required": false
                    },
                    "entity-type": {
                        "required": false
                    },
                    "event-type": {
                        "required": false
                    },
                    "external-model": {
                        "required": false
                    },
                    "label": {
                        "required": false
                    },
                    "model": {
                        "required": false
                    },
                    "model-version": {
                        "required": false
                    },
                    "outcome": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "variable": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UntagResource.html",
                "description": "Removes tags from a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "detector": {
                        "required": false
                    },
                    "detector-version": {
                        "required": false
                    },
                    "entity-type": {
                        "required": false
                    },
                    "event-type": {
                        "required": false
                    },
                    "external-model": {
                        "required": false
                    },
                    "label": {
                        "required": false
                    },
                    "model": {
                        "required": false
                    },
                    "model-version": {
                        "required": false
                    },
                    "outcome": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "variable": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UpdateDetectorVersion": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersion",
                "description": "Updates a detector version. The detector version attributes that you can update include models, external model endpoints, rules, rule execution mode, and description. You can only update a DRAFT detector version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "UpdateDetectorVersionMetadata": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersionMetadata",
                "description": "Updates the detector version's description. You can update the metadata for any detector version (DRAFT, ACTIVE, or INACTIVE).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector-version": {
                        "required": true
                    }
                }
            },
            "UpdateDetectorVersionStatus": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersionStatus",
                "description": "Updates the detector version’s status. You can perform the following promotions or demotions using UpdateDetectorVersionStatus: DRAFT to ACTIVE, ACTIVE to INACTIVE, and INACTIVE to ACTIVE.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector-version": {
                        "required": true
                    }
                }
            },
            "UpdateModel": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModel",
                "description": "Updates a model. You can update the description attribute using this action.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                }
            },
            "UpdateModelVersion": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModelVersion",
                "description": "Updates a model version. Updating a model version retrains an existing model version using updated training data and produces a new minor version of the model. You can update the training data set location and data access role attributes using this action. This action creates and trains a new minor version of the model, for example version 1.01, 1.02, 1.03.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateModelVersionStatus": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModelVersionStatus",
                "description": "Updates the status of a model version.",
                "accessLevel": "Write"
            },
            "UpdateRuleMetadata": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateRuleMetadata.html",
                "description": "Updates a rule's metadata. The description attribute can be updated.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "UpdateRuleVersion": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateRuleVersion.html",
                "description": "Updates a rule version resulting in a new rule version. Updates a rule version resulting in a new rule version (version 1, 2, 3 ...).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateVariable": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateVariable.html",
                "description": "Updates a variable.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "variable": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "detector": {
                "name": "detector",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:detector/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "detector-version": {
                "name": "detector-version",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:detector-version/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "entity-type": {
                "name": "entity-type",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:entity-type/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "external-model": {
                "name": "external-model",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:external-model/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "event-type": {
                "name": "event-type",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:event-type/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "label": {
                "name": "label",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:label/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "model": {
                "name": "model",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:model/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "model-version": {
                "name": "model-version",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:model-version/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "outcome": {
                "name": "outcome",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:outcome/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "rule": {
                "name": "rule",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:rule/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "variable": {
                "name": "variable",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:variable/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a batch of variables.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_BatchCreateVariable
     */
    batchCreateVariable() {
        this.add('frauddetector:BatchCreateVariable');
        return this;
    }
    /**
     * Gets a batch of variables.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_BatchGetVariable
     */
    batchGetVariable() {
        this.add('frauddetector:BatchGetVariable');
        return this;
    }
    /**
     * Creates a detector version. The detector version starts in a DRAFT status.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateDetectorVersion
     */
    createDetectorVersion() {
        this.add('frauddetector:CreateDetectorVersion');
        return this;
    }
    /**
     * Creates a model using the specified model type.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateModel
     */
    createModel() {
        this.add('frauddetector:CreateModel');
        return this;
    }
    /**
     * Creates a version of the model using the specified model type and model id.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateModelVersion
     */
    createModelVersion() {
        this.add('frauddetector:CreateModelVersion');
        return this;
    }
    /**
     * Creates a rule for use with the specified detector.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateRule.html
     */
    createRule() {
        this.add('frauddetector:CreateRule');
        return this;
    }
    /**
     * Creates a variable.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateVariable.html
     */
    createVariable() {
        this.add('frauddetector:CreateVariable');
        return this;
    }
    /**
     * Deletes the detector. Before deleting a detector, you must first delete all detector versions and rule versions associated with the detector.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteDetector
     */
    deleteDetector() {
        this.add('frauddetector:DeleteDetector');
        return this;
    }
    /**
     * Deletes the detector version. You cannot delete detector versions that are in ACTIVE status.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteDetectorVersion
     */
    deleteDetectorVersion() {
        this.add('frauddetector:DeleteDetectorVersion');
        return this;
    }
    /**
     * Deletes the specified event.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteEvent
     */
    deleteEvent() {
        this.add('frauddetector:DeleteEvent');
        return this;
    }
    /**
     * Deletes the rule. You cannot delete a rule if it is used by an ACTIVE or INACTIVE detector version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteRule
     */
    deleteRule() {
        this.add('frauddetector:DeleteRule');
        return this;
    }
    /**
     * Gets all versions for a specified detector.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DescribeDetector
     */
    describeDetector() {
        this.add('frauddetector:DescribeDetector');
        return this;
    }
    /**
     * Gets all of the model versions for the specified model type or for the specified model type and model ID. You can also get details for a single, specified model version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DescribeModelVersions
     */
    describeModelVersions() {
        this.add('frauddetector:DescribeModelVersions');
        return this;
    }
    /**
     * Gets a particular detector version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetDetectorVersion
     */
    getDetectorVersion() {
        this.add('frauddetector:GetDetectorVersion');
        return this;
    }
    /**
     * Gets all detectors or a single detector if a detectorId is specified. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetDetectorsResponse as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetDetectors
     */
    getDetectors() {
        this.add('frauddetector:GetDetectors');
        return this;
    }
    /**
     * Gets all entity types or a specific entity type if a name is specified. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetEntityTypesResponse as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEntityTypes
     */
    getEntityTypes() {
        this.add('frauddetector:GetEntityTypes');
        return this;
    }
    /**
     * Evaluates an event against a detector version. If a version ID is not provided, the detector’s (ACTIVE) version is used.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEventPrediction
     */
    getEventPrediction() {
        this.add('frauddetector:GetEventPrediction');
        return this;
    }
    /**
     * Gets all event types or a specific event type if name is provided. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetEventTypesResponse as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEventTypes
     */
    getEventTypes() {
        this.add('frauddetector:GetEventTypes');
        return this;
    }
    /**
     * Gets the details for one or more Amazon SageMaker models that have been imported into the service. This is a paginated API. If you provide a null maxResults, this actions retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetExternalModelsResult as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetExternalModels
     */
    getExternalModels() {
        this.add('frauddetector:GetExternalModels');
        return this;
    }
    /**
     * Gets the encryption key if a Key Management Service (KMS) customer master key (CMK) has been specified to be used to encrypt content in Amazon Fraud Detector.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetKMSEncryptionKey
     */
    getKMSEncryptionKey() {
        this.add('frauddetector:GetKMSEncryptionKey');
        return this;
    }
    /**
     * Gets all labels or a specific label if name is provided. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 50 records per page. If you provide a maxResults, the value must be between 10 and 50. To get the next page results, provide the pagination token from the GetGetLabelsResponse as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetLabels
     */
    getLabels() {
        this.add('frauddetector:GetLabels');
        return this;
    }
    /**
     * Gets the details of the specified model version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetModelVersion
     */
    getModelVersion() {
        this.add('frauddetector:GetModelVersion');
        return this;
    }
    /**
     * Gets one or more models. Gets all models for the AWS account if no model type and no model id provided. Gets all models for the AWS account and model type, if the model type is specified but model id is not provided. Gets a specific model if (model type, model id) tuple is specified.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetModels
     */
    getModels() {
        this.add('frauddetector:GetModels');
        return this;
    }
    /**
     * Gets one or more outcomes. This is a paginated API. If you provide a null maxResults, this actions retrieves a maximum of 100 records per page. If you provide a maxResults, the value must be between 50 and 100. To get the next page results, provide the pagination token from the GetOutcomesResult as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetOutcomes
     */
    getOutcomes() {
        this.add('frauddetector:GetOutcomes');
        return this;
    }
    /**
     * Get all rules for a detector (paginated) if ruleId and ruleVersion are not specified. Gets all rules for the detector and the ruleId if present (paginated). Gets a specific rule if both the ruleId and the ruleVersion are specified.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetRules.html
     */
    getRules() {
        this.add('frauddetector:GetRules');
        return this;
    }
    /**
     * Gets all of the variables or the specific variable. This is a paginated API. Providing null maxSizePerPage results in retrieving maximum of 100 records per page. If you provide maxSizePerPage the value must be between 50 and 100. To get the next page result, a provide a pagination token from GetVariablesResult as part of your request. Null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetVariables
     */
    getVariables() {
        this.add('frauddetector:GetVariables');
        return this;
    }
    /**
     * Lists all tags associated with the resource. This is a paginated API. To get the next page results, provide the pagination token from the response as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('frauddetector:ListTagsForResource');
        return this;
    }
    /**
     * Creates or updates a detector.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutDetector.html
     */
    putDetector() {
        this.add('frauddetector:PutDetector');
        return this;
    }
    /**
     * Creates or updates an entity type. An entity represents who is performing the event. As part of a fraud prediction, you pass the entity ID to indicate the specific entity who performed the event. An entity type classifies the entity. Example classifications include customer, merchant, or account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutEntityType.html
     */
    putEntityType() {
        this.add('frauddetector:PutEntityType');
        return this;
    }
    /**
     * Creates or updates an event type. An event is a business activity that is evaluated for fraud risk. With Amazon Fraud Detector, you generate fraud predictions for events. An event type defines the structure for an event sent to Amazon Fraud Detector. This includes the variables sent as part of the event, the entity performing the event (such as a customer), and the labels that classify the event. Example event types include online payment transactions, account registrations, and authentications.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutEventType.html
     */
    putEventType() {
        this.add('frauddetector:PutEventType');
        return this;
    }
    /**
     * Creates or updates an Amazon SageMaker model endpoint. You can also use this action to update the configuration of the model endpoint, including the IAM role and/or the mapped variables.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutExternalModel.html
     */
    putExternalModel() {
        this.add('frauddetector:PutExternalModel');
        return this;
    }
    /**
     * Specifies the Key Management Service (KMS) customer master key (CMK) to be used to encrypt content in Amazon Fraud Detector.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutKMSEncryptionKey.html
     */
    putKMSEncryptionKey() {
        this.add('frauddetector:PutKMSEncryptionKey');
        return this;
    }
    /**
     * Creates or updates label. A label classifies an event as fraudulent or legitimate. Labels are associated with event types and used to train supervised machine learning models in Amazon Fraud Detector.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutLabel.html
     */
    putLabel() {
        this.add('frauddetector:PutLabel');
        return this;
    }
    /**
     * Creates or updates an outcome.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutOutcome.html
     */
    putOutcome() {
        this.add('frauddetector:PutOutcome');
        return this;
    }
    /**
     * Assigns tags to a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_TagResource.html
     */
    tagResource() {
        this.add('frauddetector:TagResource');
        return this;
    }
    /**
     * Removes tags from a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UntagResource.html
     */
    untagResource() {
        this.add('frauddetector:UntagResource');
        return this;
    }
    /**
     * Updates a detector version. The detector version attributes that you can update include models, external model endpoints, rules, rule execution mode, and description. You can only update a DRAFT detector version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersion
     */
    updateDetectorVersion() {
        this.add('frauddetector:UpdateDetectorVersion');
        return this;
    }
    /**
     * Updates the detector version's description. You can update the metadata for any detector version (DRAFT, ACTIVE, or INACTIVE).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersionMetadata
     */
    updateDetectorVersionMetadata() {
        this.add('frauddetector:UpdateDetectorVersionMetadata');
        return this;
    }
    /**
     * Updates the detector version’s status. You can perform the following promotions or demotions using UpdateDetectorVersionStatus: DRAFT to ACTIVE, ACTIVE to INACTIVE, and INACTIVE to ACTIVE.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersionStatus
     */
    updateDetectorVersionStatus() {
        this.add('frauddetector:UpdateDetectorVersionStatus');
        return this;
    }
    /**
     * Updates a model. You can update the description attribute using this action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModel
     */
    updateModel() {
        this.add('frauddetector:UpdateModel');
        return this;
    }
    /**
     * Updates a model version. Updating a model version retrains an existing model version using updated training data and produces a new minor version of the model. You can update the training data set location and data access role attributes using this action. This action creates and trains a new minor version of the model, for example version 1.01, 1.02, 1.03.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModelVersion
     */
    updateModelVersion() {
        this.add('frauddetector:UpdateModelVersion');
        return this;
    }
    /**
     * Updates the status of a model version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModelVersionStatus
     */
    updateModelVersionStatus() {
        this.add('frauddetector:UpdateModelVersionStatus');
        return this;
    }
    /**
     * Updates a rule's metadata. The description attribute can be updated.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateRuleMetadata.html
     */
    updateRuleMetadata() {
        this.add('frauddetector:UpdateRuleMetadata');
        return this;
    }
    /**
     * Updates a rule version resulting in a new rule version. Updates a rule version resulting in a new rule version (version 1, 2, 3 ...).
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateRuleVersion.html
     */
    updateRuleVersion() {
        this.add('frauddetector:UpdateRuleVersion');
        return this;
    }
    /**
     * Updates a variable.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateVariable.html
     */
    updateVariable() {
        this.add('frauddetector:UpdateVariable');
        return this;
    }
    /**
     * Adds a resource of type detector to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetector(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:detector/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type detector-version to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetectorVersion(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:detector-version/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type entity-type to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEntityType(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:entity-type/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type external-model to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onExternalModel(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:external-model/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type event-type to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventType(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:event-type/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type label to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLabel(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:label/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type model to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModel(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:model/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type model-version to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModelVersion(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:model-version/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type outcome to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOutcome(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:outcome/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rule to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRule(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:rule/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type variable to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVariable(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:variable/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .batchCreateVariable()
     * - .createDetectorVersion()
     * - .createModel()
     * - .createModelVersion()
     * - .createRule()
     * - .createVariable()
     * - .putDetector()
     * - .putEntityType()
     * - .putEventType()
     * - .putExternalModel()
     * - .putLabel()
     * - .putOutcome()
     * - .tagResource()
     * - .untagResource()
     * - .updateModelVersion()
     * - .updateRuleVersion()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - detector
     * - detector-version
     * - entity-type
     * - external-model
     * - event-type
     * - label
     * - model
     * - model-version
     * - outcome
     * - rule
     * - variable
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .batchCreateVariable()
     * - .createDetectorVersion()
     * - .createModel()
     * - .createModelVersion()
     * - .createRule()
     * - .createVariable()
     * - .putDetector()
     * - .putEntityType()
     * - .putEventType()
     * - .putExternalModel()
     * - .putLabel()
     * - .putOutcome()
     * - .tagResource()
     * - .untagResource()
     * - .updateModelVersion()
     * - .updateRuleVersion()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Frauddetector = Frauddetector;
//# sourceMappingURL=data:application/json;base64,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