"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Gamelift = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [gamelift](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazongamelift.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Gamelift extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [gamelift](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazongamelift.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'gamelift';
        this.actionList = {
            "AcceptMatch": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_AcceptMatch.html",
                "description": "Registers player acceptance or rejection of a proposed FlexMatch match.",
                "accessLevel": "Write"
            },
            "CreateAlias": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateAlias.html",
                "description": "Defines a new alias for a fleet.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateBuild": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateBuild.html",
                "description": "Creates a new game build using files stored in an Amazon S3 bucket.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateFleet": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateFleet.html",
                "description": "Creates a new fleet of computing resources to run your game servers.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateGameSession": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateGameSession.html",
                "description": "Starts a new game session on a specified fleet.",
                "accessLevel": "Write"
            },
            "CreateGameSessionQueue": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateGameSessionQueue.html",
                "description": "Sets up a new queue for processing new game session placement requests.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateMatchmakingConfiguration": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateMatchmakingConfiguration.html",
                "description": "Creates a new FlexMatch matchmaker.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateMatchmakingRuleSet": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateMatchmakingRuleSet.html",
                "description": "Creates a new matchmaking rule set for FlexMatch.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePlayerSession": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreatePlayerSession.html",
                "description": "Reserves an available game session slot for a player.",
                "accessLevel": "Write"
            },
            "CreatePlayerSessions": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreatePlayerSessions.html",
                "description": "Reserves available game session slots for multiple players.",
                "accessLevel": "Write"
            },
            "CreateScript": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateScript.html",
                "description": "Creates a new Realtime Servers script.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateVpcPeeringAuthorization": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateVpcPeeringAuthorization.html",
                "description": "Allows GameLift to create or delete a peering connection between a GameLift fleet VPC and a VPC on another AWS account.",
                "accessLevel": "Write"
            },
            "CreateVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateVpcPeeringConnection.html",
                "description": "Establishes a peering connection between your GameLift fleet VPC and a VPC on another account.",
                "accessLevel": "Write"
            },
            "DeleteAlias": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteAlias.html",
                "description": "Deletes an alias.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    }
                }
            },
            "DeleteBuild": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteBuild.html",
                "description": "Deletes a game build.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "build": {
                        "required": true
                    }
                }
            },
            "DeleteFleet": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteFleet.html",
                "description": "Deletes an empty fleet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DeleteGameSessionQueue": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteGameSessionQueue.html",
                "description": "Deletes an existing game session queue.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "gameSessionQueue": {
                        "required": true
                    }
                }
            },
            "DeleteMatchmakingConfiguration": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteMatchmakingConfiguration.html",
                "description": "Deletes an existing FlexMatch matchmaker.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "matchmakingConfiguration": {
                        "required": true
                    }
                }
            },
            "DeleteMatchmakingRuleSet": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteMatchmakingRuleSet.html",
                "description": "Deletes an existing FlexMatch matchmaking rule set.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "matchmakingRuleSet": {
                        "required": true
                    }
                }
            },
            "DeleteScalingPolicy": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteScalingPolicy.html",
                "description": "Deletes a set of auto-scaling rules.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DeleteScript": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteScript.html",
                "description": "Deletes a Realtime Servers script.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "script": {
                        "required": true
                    }
                }
            },
            "DeleteVpcPeeringAuthorization": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteVpcPeeringAuthorization.html",
                "description": "Cancels a VPC peering authorization.",
                "accessLevel": "Write"
            },
            "DeleteVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteVpcPeeringConnection.html",
                "description": "Removes a peering connection between VPCs.",
                "accessLevel": "Write"
            },
            "DescribeAlias": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeAlias.html",
                "description": "Retrieves properties for an alias.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    }
                }
            },
            "DescribeBuild": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeBuild.html",
                "description": "Retrieves properties for a game build.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "build": {
                        "required": true
                    }
                }
            },
            "DescribeEC2InstanceLimits": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeEC2InstanceLimits.html",
                "description": "Retrieves the maximum allowed and current usage for EC2 instance types.",
                "accessLevel": "Read"
            },
            "DescribeFleetAttributes": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetAttributes.html",
                "description": "Retrieves general properties, including status, for fleets.",
                "accessLevel": "Read"
            },
            "DescribeFleetCapacity": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetCapacity.html",
                "description": "Retrieves the current capacity setting for fleets.",
                "accessLevel": "Read"
            },
            "DescribeFleetEvents": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetEvents.html",
                "description": "Retrieves entries from a fleet's event log.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeFleetPortSettings": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetPortSettings.html",
                "description": "Retrieves the inbound connection permissions for a fleet.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeFleetUtilization": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetUtilization.html",
                "description": "Retrieves utilization statistics for fleets.",
                "accessLevel": "Read"
            },
            "DescribeGameSessionDetails": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessionDetails.html",
                "description": "Retrieves properties for game sessions in a fleet, including the protection policy.",
                "accessLevel": "Read"
            },
            "DescribeGameSessionPlacement": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessionPlacement.html",
                "description": "Retrieves details of a game session placement request.",
                "accessLevel": "Read"
            },
            "DescribeGameSessionQueues": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessionQueues.html",
                "description": "Retrieves properties for game session queues.",
                "accessLevel": "Read"
            },
            "DescribeGameSessions": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessions.html",
                "description": "Retrieves properties for game sessions in a fleet.",
                "accessLevel": "Read"
            },
            "DescribeInstances": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeInstances.html",
                "description": "Retrieves information about instances in a fleet.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeMatchmaking": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmaking.html",
                "description": "Retrieves details of matchmaking tickets.",
                "accessLevel": "Read"
            },
            "DescribeMatchmakingConfigurations": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmakingConfigurations.html",
                "description": "Retrieves properties for FlexMatch matchmakers.",
                "accessLevel": "Read"
            },
            "DescribeMatchmakingRuleSets": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmakingRuleSets.html",
                "description": "Retrieves properties for FlexMatch matchmaking rule sets.",
                "accessLevel": "Read"
            },
            "DescribePlayerSessions": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribePlayerSessions.html",
                "description": "Retrieves properties for player sessions in a game session.",
                "accessLevel": "Read"
            },
            "DescribeRuntimeConfiguration": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeRuntimeConfiguration.html",
                "description": "Retrieves the current runtime configuration for a fleet.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeScalingPolicies": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeScalingPolicies.html",
                "description": "Retrieves all scaling policies that are applied to a fleet.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeScript": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeScript.html",
                "description": "Retrieves properties for a Realtime Servers script.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "script": {
                        "required": true
                    }
                }
            },
            "DescribeVpcPeeringAuthorizations": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeVpcPeeringAuthorizations.html",
                "description": "Retrieves valid VPC peering authorizations.",
                "accessLevel": "Read"
            },
            "DescribeVpcPeeringConnections": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeVpcPeeringConnections.html",
                "description": "Retrieves details on active or pending VPC peering connections.",
                "accessLevel": "Read"
            },
            "GetGameSessionLogUrl": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_GetGameSessionLogUrl.html",
                "description": "Retrieves the location of stored logs for a game session.",
                "accessLevel": "Read"
            },
            "GetInstanceAccess": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_GetInstanceAccess.html",
                "description": "Requests remote access to a specified fleet instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "ListAliases": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListAliases.html",
                "description": "Retrieves all aliases that are defined in the current region.",
                "accessLevel": "List"
            },
            "ListBuilds": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListBuilds.html",
                "description": "Retrieves all game build in the current region.",
                "accessLevel": "List"
            },
            "ListFleets": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListFleets.html",
                "description": "Retrieves a list of fleet IDs for all fleets in the current region.",
                "accessLevel": "List"
            },
            "ListScripts": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListScripts.html",
                "description": "Retrieves properties for all Realtime Servers scripts in the current region.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListTagsForResource.html",
                "description": "List Tags for GameLift Resource",
                "accessLevel": "List",
                "resourceTypes": {
                    "alias": {
                        "required": false
                    },
                    "build": {
                        "required": false
                    },
                    "fleet": {
                        "required": false
                    },
                    "gameSessionQueue": {
                        "required": false
                    },
                    "matchmakingConfiguration": {
                        "required": false
                    },
                    "matchmakingRuleSet": {
                        "required": false
                    },
                    "script": {
                        "required": false
                    }
                }
            },
            "PutScalingPolicy": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_PutScalingPolicy.html",
                "description": "Creates or updates a fleet auto-scaling policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "RequestUploadCredentials": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_RequestUploadCredentials.html",
                "description": "Retrieves fresh upload credentials to use when uploading a new game build.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "build": {
                        "required": true
                    }
                }
            },
            "ResolveAlias": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_ResolveAlias.html",
                "description": "Retrieves the fleet ID associated with an alias.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    }
                }
            },
            "SearchGameSessions": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_SearchGameSessions.html",
                "description": "Retrieves game sessions that match a set of search criteria.",
                "accessLevel": "Read"
            },
            "StartFleetActions": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartFleetActions.html",
                "description": "Resumes auto-scaling activity on a fleet after it was suspended with StopFleetActions().",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "StartGameSessionPlacement": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartGameSessionPlacement.html",
                "description": "Sends a game session placement request to a game session queue.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "gameSessionQueue": {
                        "required": true
                    }
                }
            },
            "StartMatchBackfill": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartMatchBackfill.html",
                "description": "Requests FlexMatch matchmaking to fill available player slots in an existing game session.",
                "accessLevel": "Write"
            },
            "StartMatchmaking": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartMatchmaking.html",
                "description": "Requests FlexMatch matchmaking for one or a group of players and game session placement for a resulting match.",
                "accessLevel": "Write"
            },
            "StopFleetActions": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html",
                "description": "Suspends auto-scaling activity on a fleet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "StopGameSessionPlacement": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopGameSessionPlacement.html",
                "description": "Cancels a game session placement request that is in progress.",
                "accessLevel": "Write"
            },
            "StopMatchmaking": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopMatchmaking.html",
                "description": "Cancels a matchmaking or match backfill request that is in progress.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_TagResource.html",
                "description": "Tags GameLift Resources",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "alias": {
                        "required": false
                    },
                    "build": {
                        "required": false
                    },
                    "fleet": {
                        "required": false
                    },
                    "gameSessionQueue": {
                        "required": false
                    },
                    "matchmakingConfiguration": {
                        "required": false
                    },
                    "matchmakingRuleSet": {
                        "required": false
                    },
                    "script": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UntagResource.html",
                "description": "Untagging GameLift Resources",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "alias": {
                        "required": false
                    },
                    "build": {
                        "required": false
                    },
                    "fleet": {
                        "required": false
                    },
                    "gameSessionQueue": {
                        "required": false
                    },
                    "matchmakingConfiguration": {
                        "required": false
                    },
                    "matchmakingRuleSet": {
                        "required": false
                    },
                    "script": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateAlias": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateAlias.html",
                "description": "Updates the properties of an existing alias.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    }
                }
            },
            "UpdateBuild": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateBuild.html",
                "description": "Updates an existing build's metadata.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "build": {
                        "required": true
                    }
                }
            },
            "UpdateFleetAttributes": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateFleetAttributes.html",
                "description": "Updates the general properties of an existing fleet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateFleetCapacity": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateFleetCapacity.html",
                "description": "Adjusts a fleet's capacity settings.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateFleetPortSettings": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateFleetPortSettings.html",
                "description": "Adjusts a fleet's port settings.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateGameSession": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateGameSession.html",
                "description": "Updates the properties of an existing game session.",
                "accessLevel": "Write"
            },
            "UpdateGameSessionQueue": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateGameSessionQueue.html",
                "description": "Updates properties of an existing game session queue.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "gameSessionQueue": {
                        "required": true
                    }
                }
            },
            "UpdateMatchmakingConfiguration": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateMatchmakingConfiguration.html",
                "description": "Updates properties of an existing FlexMatch matchmaking configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "matchmakingConfiguration": {
                        "required": true
                    }
                }
            },
            "UpdateRuntimeConfiguration": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateRuntimeConfiguration.html",
                "description": "Updates how server processes are configured on instances in an existing fleet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateScript": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateScript.html",
                "description": "Updates the metadata and content of an existing Realtime Servers script.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "script": {
                        "required": true
                    }
                }
            },
            "ValidateMatchmakingRuleSet": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_ValidateMatchmakingRuleSet.html",
                "description": "Validates the syntax of a FlexMatch matchmaking rule set.",
                "accessLevel": "Read"
            }
        };
        this.resourceTypes = {
            "alias": {
                "name": "alias",
                "url": "https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html",
                "arn": "arn:${Partition}:gamelift:${Region}::alias/${AliasId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "build": {
                "name": "build",
                "url": "https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html",
                "arn": "arn:${Partition}:gamelift:${Region}:${AccountId}:build/${BuildId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "script": {
                "name": "script",
                "url": "https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html",
                "arn": "arn:${Partition}:gamelift:${Region}:${AccountId}:script/${ScriptId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "fleet": {
                "name": "fleet",
                "url": "https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html",
                "arn": "arn:${Partition}:gamelift:${Region}:${Account}:fleet/${FleetId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "gameSessionQueue": {
                "name": "gameSessionQueue",
                "url": "https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html",
                "arn": "arn:${Partition}:gamelift:${Region}:${Account}:gamesessionqueue/${GameSessionQueueName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "matchmakingConfiguration": {
                "name": "matchmakingConfiguration",
                "url": "https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html",
                "arn": "arn:${Partition}:gamelift:${Region}:${Account}:matchmakingconfiguration/${MatchmakingConfigurationName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "matchmakingRuleSet": {
                "name": "matchmakingRuleSet",
                "url": "https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html",
                "arn": "arn:${Partition}:gamelift:${Region}:${Account}:matchmakingruleset/${MatchmakingRuleSetName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Registers player acceptance or rejection of a proposed FlexMatch match.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_AcceptMatch.html
     */
    acceptMatch() {
        this.add('gamelift:AcceptMatch');
        return this;
    }
    /**
     * Defines a new alias for a fleet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateAlias.html
     */
    createAlias() {
        this.add('gamelift:CreateAlias');
        return this;
    }
    /**
     * Creates a new game build using files stored in an Amazon S3 bucket.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateBuild.html
     */
    createBuild() {
        this.add('gamelift:CreateBuild');
        return this;
    }
    /**
     * Creates a new fleet of computing resources to run your game servers.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateFleet.html
     */
    createFleet() {
        this.add('gamelift:CreateFleet');
        return this;
    }
    /**
     * Starts a new game session on a specified fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateGameSession.html
     */
    createGameSession() {
        this.add('gamelift:CreateGameSession');
        return this;
    }
    /**
     * Sets up a new queue for processing new game session placement requests.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateGameSessionQueue.html
     */
    createGameSessionQueue() {
        this.add('gamelift:CreateGameSessionQueue');
        return this;
    }
    /**
     * Creates a new FlexMatch matchmaker.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateMatchmakingConfiguration.html
     */
    createMatchmakingConfiguration() {
        this.add('gamelift:CreateMatchmakingConfiguration');
        return this;
    }
    /**
     * Creates a new matchmaking rule set for FlexMatch.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateMatchmakingRuleSet.html
     */
    createMatchmakingRuleSet() {
        this.add('gamelift:CreateMatchmakingRuleSet');
        return this;
    }
    /**
     * Reserves an available game session slot for a player.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreatePlayerSession.html
     */
    createPlayerSession() {
        this.add('gamelift:CreatePlayerSession');
        return this;
    }
    /**
     * Reserves available game session slots for multiple players.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreatePlayerSessions.html
     */
    createPlayerSessions() {
        this.add('gamelift:CreatePlayerSessions');
        return this;
    }
    /**
     * Creates a new Realtime Servers script.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateScript.html
     */
    createScript() {
        this.add('gamelift:CreateScript');
        return this;
    }
    /**
     * Allows GameLift to create or delete a peering connection between a GameLift fleet VPC and a VPC on another AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateVpcPeeringAuthorization.html
     */
    createVpcPeeringAuthorization() {
        this.add('gamelift:CreateVpcPeeringAuthorization');
        return this;
    }
    /**
     * Establishes a peering connection between your GameLift fleet VPC and a VPC on another account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateVpcPeeringConnection.html
     */
    createVpcPeeringConnection() {
        this.add('gamelift:CreateVpcPeeringConnection');
        return this;
    }
    /**
     * Deletes an alias.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteAlias.html
     */
    deleteAlias() {
        this.add('gamelift:DeleteAlias');
        return this;
    }
    /**
     * Deletes a game build.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteBuild.html
     */
    deleteBuild() {
        this.add('gamelift:DeleteBuild');
        return this;
    }
    /**
     * Deletes an empty fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteFleet.html
     */
    deleteFleet() {
        this.add('gamelift:DeleteFleet');
        return this;
    }
    /**
     * Deletes an existing game session queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteGameSessionQueue.html
     */
    deleteGameSessionQueue() {
        this.add('gamelift:DeleteGameSessionQueue');
        return this;
    }
    /**
     * Deletes an existing FlexMatch matchmaker.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteMatchmakingConfiguration.html
     */
    deleteMatchmakingConfiguration() {
        this.add('gamelift:DeleteMatchmakingConfiguration');
        return this;
    }
    /**
     * Deletes an existing FlexMatch matchmaking rule set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteMatchmakingRuleSet.html
     */
    deleteMatchmakingRuleSet() {
        this.add('gamelift:DeleteMatchmakingRuleSet');
        return this;
    }
    /**
     * Deletes a set of auto-scaling rules.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteScalingPolicy.html
     */
    deleteScalingPolicy() {
        this.add('gamelift:DeleteScalingPolicy');
        return this;
    }
    /**
     * Deletes a Realtime Servers script.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteScript.html
     */
    deleteScript() {
        this.add('gamelift:DeleteScript');
        return this;
    }
    /**
     * Cancels a VPC peering authorization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteVpcPeeringAuthorization.html
     */
    deleteVpcPeeringAuthorization() {
        this.add('gamelift:DeleteVpcPeeringAuthorization');
        return this;
    }
    /**
     * Removes a peering connection between VPCs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteVpcPeeringConnection.html
     */
    deleteVpcPeeringConnection() {
        this.add('gamelift:DeleteVpcPeeringConnection');
        return this;
    }
    /**
     * Retrieves properties for an alias.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeAlias.html
     */
    describeAlias() {
        this.add('gamelift:DescribeAlias');
        return this;
    }
    /**
     * Retrieves properties for a game build.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeBuild.html
     */
    describeBuild() {
        this.add('gamelift:DescribeBuild');
        return this;
    }
    /**
     * Retrieves the maximum allowed and current usage for EC2 instance types.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeEC2InstanceLimits.html
     */
    describeEC2InstanceLimits() {
        this.add('gamelift:DescribeEC2InstanceLimits');
        return this;
    }
    /**
     * Retrieves general properties, including status, for fleets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetAttributes.html
     */
    describeFleetAttributes() {
        this.add('gamelift:DescribeFleetAttributes');
        return this;
    }
    /**
     * Retrieves the current capacity setting for fleets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetCapacity.html
     */
    describeFleetCapacity() {
        this.add('gamelift:DescribeFleetCapacity');
        return this;
    }
    /**
     * Retrieves entries from a fleet's event log.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetEvents.html
     */
    describeFleetEvents() {
        this.add('gamelift:DescribeFleetEvents');
        return this;
    }
    /**
     * Retrieves the inbound connection permissions for a fleet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetPortSettings.html
     */
    describeFleetPortSettings() {
        this.add('gamelift:DescribeFleetPortSettings');
        return this;
    }
    /**
     * Retrieves utilization statistics for fleets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetUtilization.html
     */
    describeFleetUtilization() {
        this.add('gamelift:DescribeFleetUtilization');
        return this;
    }
    /**
     * Retrieves properties for game sessions in a fleet, including the protection policy.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessionDetails.html
     */
    describeGameSessionDetails() {
        this.add('gamelift:DescribeGameSessionDetails');
        return this;
    }
    /**
     * Retrieves details of a game session placement request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessionPlacement.html
     */
    describeGameSessionPlacement() {
        this.add('gamelift:DescribeGameSessionPlacement');
        return this;
    }
    /**
     * Retrieves properties for game session queues.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessionQueues.html
     */
    describeGameSessionQueues() {
        this.add('gamelift:DescribeGameSessionQueues');
        return this;
    }
    /**
     * Retrieves properties for game sessions in a fleet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessions.html
     */
    describeGameSessions() {
        this.add('gamelift:DescribeGameSessions');
        return this;
    }
    /**
     * Retrieves information about instances in a fleet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeInstances.html
     */
    describeInstances() {
        this.add('gamelift:DescribeInstances');
        return this;
    }
    /**
     * Retrieves details of matchmaking tickets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmaking.html
     */
    describeMatchmaking() {
        this.add('gamelift:DescribeMatchmaking');
        return this;
    }
    /**
     * Retrieves properties for FlexMatch matchmakers.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmakingConfigurations.html
     */
    describeMatchmakingConfigurations() {
        this.add('gamelift:DescribeMatchmakingConfigurations');
        return this;
    }
    /**
     * Retrieves properties for FlexMatch matchmaking rule sets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmakingRuleSets.html
     */
    describeMatchmakingRuleSets() {
        this.add('gamelift:DescribeMatchmakingRuleSets');
        return this;
    }
    /**
     * Retrieves properties for player sessions in a game session.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribePlayerSessions.html
     */
    describePlayerSessions() {
        this.add('gamelift:DescribePlayerSessions');
        return this;
    }
    /**
     * Retrieves the current runtime configuration for a fleet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeRuntimeConfiguration.html
     */
    describeRuntimeConfiguration() {
        this.add('gamelift:DescribeRuntimeConfiguration');
        return this;
    }
    /**
     * Retrieves all scaling policies that are applied to a fleet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeScalingPolicies.html
     */
    describeScalingPolicies() {
        this.add('gamelift:DescribeScalingPolicies');
        return this;
    }
    /**
     * Retrieves properties for a Realtime Servers script.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeScript.html
     */
    describeScript() {
        this.add('gamelift:DescribeScript');
        return this;
    }
    /**
     * Retrieves valid VPC peering authorizations.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeVpcPeeringAuthorizations.html
     */
    describeVpcPeeringAuthorizations() {
        this.add('gamelift:DescribeVpcPeeringAuthorizations');
        return this;
    }
    /**
     * Retrieves details on active or pending VPC peering connections.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeVpcPeeringConnections.html
     */
    describeVpcPeeringConnections() {
        this.add('gamelift:DescribeVpcPeeringConnections');
        return this;
    }
    /**
     * Retrieves the location of stored logs for a game session.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_GetGameSessionLogUrl.html
     */
    getGameSessionLogUrl() {
        this.add('gamelift:GetGameSessionLogUrl');
        return this;
    }
    /**
     * Requests remote access to a specified fleet instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_GetInstanceAccess.html
     */
    getInstanceAccess() {
        this.add('gamelift:GetInstanceAccess');
        return this;
    }
    /**
     * Retrieves all aliases that are defined in the current region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListAliases.html
     */
    listAliases() {
        this.add('gamelift:ListAliases');
        return this;
    }
    /**
     * Retrieves all game build in the current region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListBuilds.html
     */
    listBuilds() {
        this.add('gamelift:ListBuilds');
        return this;
    }
    /**
     * Retrieves a list of fleet IDs for all fleets in the current region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListFleets.html
     */
    listFleets() {
        this.add('gamelift:ListFleets');
        return this;
    }
    /**
     * Retrieves properties for all Realtime Servers scripts in the current region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListScripts.html
     */
    listScripts() {
        this.add('gamelift:ListScripts');
        return this;
    }
    /**
     * List Tags for GameLift Resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('gamelift:ListTagsForResource');
        return this;
    }
    /**
     * Creates or updates a fleet auto-scaling policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_PutScalingPolicy.html
     */
    putScalingPolicy() {
        this.add('gamelift:PutScalingPolicy');
        return this;
    }
    /**
     * Retrieves fresh upload credentials to use when uploading a new game build.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_RequestUploadCredentials.html
     */
    requestUploadCredentials() {
        this.add('gamelift:RequestUploadCredentials');
        return this;
    }
    /**
     * Retrieves the fleet ID associated with an alias.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ResolveAlias.html
     */
    resolveAlias() {
        this.add('gamelift:ResolveAlias');
        return this;
    }
    /**
     * Retrieves game sessions that match a set of search criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_SearchGameSessions.html
     */
    searchGameSessions() {
        this.add('gamelift:SearchGameSessions');
        return this;
    }
    /**
     * Resumes auto-scaling activity on a fleet after it was suspended with StopFleetActions().
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartFleetActions.html
     */
    startFleetActions() {
        this.add('gamelift:StartFleetActions');
        return this;
    }
    /**
     * Sends a game session placement request to a game session queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartGameSessionPlacement.html
     */
    startGameSessionPlacement() {
        this.add('gamelift:StartGameSessionPlacement');
        return this;
    }
    /**
     * Requests FlexMatch matchmaking to fill available player slots in an existing game session.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartMatchBackfill.html
     */
    startMatchBackfill() {
        this.add('gamelift:StartMatchBackfill');
        return this;
    }
    /**
     * Requests FlexMatch matchmaking for one or a group of players and game session placement for a resulting match.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartMatchmaking.html
     */
    startMatchmaking() {
        this.add('gamelift:StartMatchmaking');
        return this;
    }
    /**
     * Suspends auto-scaling activity on a fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html
     */
    stopFleetActions() {
        this.add('gamelift:StopFleetActions');
        return this;
    }
    /**
     * Cancels a game session placement request that is in progress.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopGameSessionPlacement.html
     */
    stopGameSessionPlacement() {
        this.add('gamelift:StopGameSessionPlacement');
        return this;
    }
    /**
     * Cancels a matchmaking or match backfill request that is in progress.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopMatchmaking.html
     */
    stopMatchmaking() {
        this.add('gamelift:StopMatchmaking');
        return this;
    }
    /**
     * Tags GameLift Resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_TagResource.html
     */
    tagResource() {
        this.add('gamelift:TagResource');
        return this;
    }
    /**
     * Untagging GameLift Resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UntagResource.html
     */
    untagResource() {
        this.add('gamelift:UntagResource');
        return this;
    }
    /**
     * Updates the properties of an existing alias.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateAlias.html
     */
    updateAlias() {
        this.add('gamelift:UpdateAlias');
        return this;
    }
    /**
     * Updates an existing build's metadata.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateBuild.html
     */
    updateBuild() {
        this.add('gamelift:UpdateBuild');
        return this;
    }
    /**
     * Updates the general properties of an existing fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateFleetAttributes.html
     */
    updateFleetAttributes() {
        this.add('gamelift:UpdateFleetAttributes');
        return this;
    }
    /**
     * Adjusts a fleet's capacity settings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateFleetCapacity.html
     */
    updateFleetCapacity() {
        this.add('gamelift:UpdateFleetCapacity');
        return this;
    }
    /**
     * Adjusts a fleet's port settings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateFleetPortSettings.html
     */
    updateFleetPortSettings() {
        this.add('gamelift:UpdateFleetPortSettings');
        return this;
    }
    /**
     * Updates the properties of an existing game session.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateGameSession.html
     */
    updateGameSession() {
        this.add('gamelift:UpdateGameSession');
        return this;
    }
    /**
     * Updates properties of an existing game session queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateGameSessionQueue.html
     */
    updateGameSessionQueue() {
        this.add('gamelift:UpdateGameSessionQueue');
        return this;
    }
    /**
     * Updates properties of an existing FlexMatch matchmaking configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateMatchmakingConfiguration.html
     */
    updateMatchmakingConfiguration() {
        this.add('gamelift:UpdateMatchmakingConfiguration');
        return this;
    }
    /**
     * Updates how server processes are configured on instances in an existing fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateRuntimeConfiguration.html
     */
    updateRuntimeConfiguration() {
        this.add('gamelift:UpdateRuntimeConfiguration');
        return this;
    }
    /**
     * Updates the metadata and content of an existing Realtime Servers script.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateScript.html
     */
    updateScript() {
        this.add('gamelift:UpdateScript');
        return this;
    }
    /**
     * Validates the syntax of a FlexMatch matchmaking rule set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ValidateMatchmakingRuleSet.html
     */
    validateMatchmakingRuleSet() {
        this.add('gamelift:ValidateMatchmakingRuleSet');
        return this;
    }
    /**
     * Adds a resource of type alias to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html
     *
     * @param aliasId - Identifier for the aliasId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAlias(aliasId, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}::alias/${AliasId}';
        arn = arn.replace('${AliasId}', aliasId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type build to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html
     *
     * @param buildId - Identifier for the buildId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBuild(buildId, accountId, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${AccountId}:build/${BuildId}';
        arn = arn.replace('${BuildId}', buildId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type script to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html
     *
     * @param scriptId - Identifier for the scriptId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onScript(scriptId, accountId, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${AccountId}:script/${ScriptId}';
        arn = arn.replace('${ScriptId}', scriptId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html
     *
     * @param fleetId - Identifier for the fleetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFleet(fleetId, account, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${Account}:fleet/${FleetId}';
        arn = arn.replace('${FleetId}', fleetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type gameSessionQueue to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html
     *
     * @param gameSessionQueueName - Identifier for the gameSessionQueueName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGameSessionQueue(gameSessionQueueName, account, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${Account}:gamesessionqueue/${GameSessionQueueName}';
        arn = arn.replace('${GameSessionQueueName}', gameSessionQueueName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type matchmakingConfiguration to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html
     *
     * @param matchmakingConfigurationName - Identifier for the matchmakingConfigurationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMatchmakingConfiguration(matchmakingConfigurationName, account, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${Account}:matchmakingconfiguration/${MatchmakingConfigurationName}';
        arn = arn.replace('${MatchmakingConfigurationName}', matchmakingConfigurationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type matchmakingRuleSet to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html
     *
     * @param matchmakingRuleSetName - Identifier for the matchmakingRuleSetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMatchmakingRuleSet(matchmakingRuleSetName, account, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${Account}:matchmakingruleset/${MatchmakingRuleSetName}';
        arn = arn.replace('${MatchmakingRuleSetName}', matchmakingRuleSetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .createAlias()
     * - .createBuild()
     * - .createFleet()
     * - .createGameSessionQueue()
     * - .createMatchmakingConfiguration()
     * - .createMatchmakingRuleSet()
     * - .createScript()
     * - .tagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - alias
     * - build
     * - script
     * - fleet
     * - gameSessionQueue
     * - matchmakingConfiguration
     * - matchmakingRuleSet
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .createAlias()
     * - .createBuild()
     * - .createFleet()
     * - .createGameSessionQueue()
     * - .createMatchmakingConfiguration()
     * - .createMatchmakingRuleSet()
     * - .createScript()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Gamelift = Gamelift;
//# sourceMappingURL=data:application/json;base64,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