"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rds = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [rds](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonrds.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Rds extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [rds](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonrds.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'rds';
        this.actionList = {
            "AddRoleToDBCluster": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddRoleToDBCluster.html",
                "description": "Grants permission to associate an Identity and Access Management (IAM) role from an Aurora DB cluster",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "AddRoleToDBInstance": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddRoleToDBInstance.html",
                "description": "Grants permission to associate an AWS Identity and Access Management (IAM) role with a DB instance",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "db": {
                        "required": true
                    }
                }
            },
            "AddSourceIdentifierToSubscription": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddSourceIdentifierToSubscription.html",
                "description": "Grants permission to add a source identifier to an existing RDS event notification subscription",
                "accessLevel": "Write",
                "resourceTypes": {
                    "es": {
                        "required": true
                    }
                }
            },
            "AddTagsToResource": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddTagsToResource.html",
                "description": "Grants permission to add metadata tags to an Amazon RDS resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "db": {
                        "required": false
                    },
                    "es": {
                        "required": false
                    },
                    "og": {
                        "required": false
                    },
                    "pg": {
                        "required": false
                    },
                    "proxy": {
                        "required": false
                    },
                    "ri": {
                        "required": false
                    },
                    "secgrp": {
                        "required": false
                    },
                    "snapshot": {
                        "required": false
                    },
                    "subgrp": {
                        "required": false
                    },
                    "target-group": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "ApplyPendingMaintenanceAction": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ApplyPendingMaintenanceAction.html",
                "description": "Grants permission to apply a pending maintenance action to a resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "db": {
                        "required": true
                    }
                }
            },
            "AuthorizeDBSecurityGroupIngress": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AuthorizeDBSecurityGroupIngress.html",
                "description": "Grants permission to enable ingress to a DBSecurityGroup using one of two forms of authorization",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "secgrp": {
                        "required": true
                    }
                }
            },
            "BacktrackDBCluster": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_BacktrackDBCluster.html",
                "description": "Grants permission to backtrack a DB cluster to a specific time, without creating a new DB cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "CancelExportTask": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CancelExportTask.html",
                "description": "Grants permission to cancel an export task in progress",
                "accessLevel": "Write"
            },
            "CopyDBClusterParameterGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBClusterParameterGroup.html",
                "description": "Grants permission to copy the specified DB cluster parameter group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster-pg": {
                        "required": true
                    }
                }
            },
            "CopyDBClusterSnapshot": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBClusterSnapshot.html",
                "description": "Grants permission to create a snapshot of a DB cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster-snapshot": {
                        "required": true
                    }
                }
            },
            "CopyDBParameterGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBParameterGroup.html",
                "description": "Grants permission to copy the specified DB parameter group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pg": {
                        "required": true
                    }
                }
            },
            "CopyDBSnapshot": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBSnapshot.html",
                "description": "Grants permission to copy the specified DB snapshot",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "CopyOptionGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyOptionGroup.html",
                "description": "Grants permission to copy the specified option group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "og": {
                        "required": true
                    }
                }
            },
            "CreateDBCluster": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBCluster.html",
                "description": "Grants permission to create a new Amazon Aurora DB cluster",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    },
                    "cluster-pg": {
                        "required": true
                    },
                    "og": {
                        "required": true
                    },
                    "subgrp": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}",
                    "rds:DatabaseEngine",
                    "rds:DatabaseName",
                    "rds:StorageEncrypted"
                ]
            },
            "CreateDBClusterEndpoint": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBClusterEndpoint.html",
                "description": "Grants permission to create a new custom endpoint and associates it with an Amazon Aurora DB cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    },
                    "cluster-endpoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "rds:EndpointType",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDBClusterParameterGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBClusterParameterGroup.html",
                "description": "Grants permission to create a new DB cluster parameter group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster-pg": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "CreateDBClusterSnapshot": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBClusterSnapshot.html",
                "description": "Grants permission to create a snapshot of a DB cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    },
                    "cluster-snapshot": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "CreateDBInstance": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html",
                "description": "Grants permission to create a new DB instance",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "db": {
                        "required": true
                    },
                    "og": {
                        "required": true
                    },
                    "pg": {
                        "required": true
                    },
                    "secgrp": {
                        "required": true
                    },
                    "subgrp": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "CreateDBInstanceReadReplica": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstanceReadReplica.html",
                "description": "Grants permission to create a DB instance that acts as a Read Replica of a source DB instance",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "db": {
                        "required": true
                    },
                    "og": {
                        "required": true
                    },
                    "subgrp": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "CreateDBParameterGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBParameterGroup.html",
                "description": "Grants permission to create a new DB parameter group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pg": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "CreateDBProxy": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBProxy.html",
                "description": "Grants permission to create a database proxy",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDBSecurityGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBSecurityGroup.html",
                "description": "Grants permission to create a new DB security group. DB security groups control access to a DB instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "secgrp": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "CreateDBSnapshot": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBSnapshot.html",
                "description": "Grants permission to create a DBSnapshot",
                "accessLevel": "Write",
                "resourceTypes": {
                    "db": {
                        "required": true
                    },
                    "snapshot": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "CreateDBSubnetGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBSubnetGroup.html",
                "description": "Grants permission to create a new DB subnet group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subgrp": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "CreateEventSubscription": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateEventSubscription.html",
                "description": "Grants permission to create an RDS event notification subscription",
                "accessLevel": "Write",
                "resourceTypes": {
                    "es": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "CreateGlobalCluster": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateGlobalCluster.html",
                "description": "Grants permission to create an Aurora global database spread across multiple regions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    },
                    "global-cluster": {
                        "required": true
                    }
                }
            },
            "CreateOptionGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateOptionGroup.html",
                "description": "Grants permission to create a new option group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "og": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "DeleteDBCluster": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBCluster.html",
                "description": "Grants permission to delete a previously provisioned DB cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    },
                    "cluster-snapshot": {
                        "required": true
                    }
                }
            },
            "DeleteDBClusterEndpoint": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBClusterEndpoint.html",
                "description": "Grants permission to delete a custom endpoint and removes it from an Amazon Aurora DB cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster-endpoint": {
                        "required": true
                    }
                }
            },
            "DeleteDBClusterParameterGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBClusterParameterGroup.html",
                "description": "Grants permission to delete a specified DB cluster parameter group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster-pg": {
                        "required": true
                    }
                }
            },
            "DeleteDBClusterSnapshot": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBClusterSnapshot.html",
                "description": "Grants permission to delete a DB cluster snapshot",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster-snapshot": {
                        "required": true
                    }
                }
            },
            "DeleteDBInstance": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBInstance.html",
                "description": "Grants permission to delete a previously provisioned DB instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "db": {
                        "required": true
                    }
                }
            },
            "DeleteDBInstanceAutomatedBackup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBInstanceAutomatedBackup.html",
                "description": "Grants permission to deletes automated backups based on the source instance's DbiResourceId value or the restorable instance's resource ID",
                "accessLevel": "Write"
            },
            "DeleteDBParameterGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBParameterGroup.html",
                "description": "Grants permission to delete a specified DBParameterGroup",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pg": {
                        "required": true
                    }
                }
            },
            "DeleteDBProxy": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBProxy.html",
                "description": "Grants permission to delete a database proxy",
                "accessLevel": "Write",
                "resourceTypes": {
                    "proxy": {
                        "required": true
                    }
                }
            },
            "DeleteDBSecurityGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBSecurityGroup.html",
                "description": "Grants permission to delete a DB security group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "secgrp": {
                        "required": true
                    }
                }
            },
            "DeleteDBSnapshot": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBSnapshot.html",
                "description": "Grants permission to delete a DBSnapshot",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "DeleteDBSubnetGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBSubnetGroup.html",
                "description": "Grants permission to delete a DB subnet group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subgrp": {
                        "required": true
                    }
                }
            },
            "DeleteEventSubscription": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteEventSubscription.html",
                "description": "Grants permission to delete an RDS event notification subscription",
                "accessLevel": "Write",
                "resourceTypes": {
                    "es": {
                        "required": true
                    }
                }
            },
            "DeleteGlobalCluster": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteGlobalCluster.html",
                "description": "Grants permission to delete a global database cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-cluster": {
                        "required": true
                    }
                }
            },
            "DeleteOptionGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteOptionGroup.html",
                "description": "Grants permission to delete an existing option group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "og": {
                        "required": true
                    }
                }
            },
            "DeregisterDBProxyTargets": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeregisterDBProxyTargets.html",
                "description": "Grants permission to remove targets from a database proxy target group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    },
                    "db": {
                        "required": true
                    },
                    "proxy": {
                        "required": true
                    },
                    "target-group": {
                        "required": true
                    }
                }
            },
            "DescribeAccountAttributes": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeAccountAttributes.html",
                "description": "Grants permission to list all of the attributes for a customer account",
                "accessLevel": "List"
            },
            "DescribeCertificates": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeCertificates.html",
                "description": "Lists the set of CA certificates provided by Amazon RDS for this AWS account",
                "accessLevel": "List"
            },
            "DescribeDBClusterBacktracks": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterBacktracks.html",
                "description": "Grants permission to return information about backtracks for a DB cluster",
                "accessLevel": "List",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "DescribeDBClusterEndpoints": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterEndpoints.html",
                "description": "Grants permission to return information about endpoints for an Amazon Aurora DB cluster",
                "accessLevel": "List"
            },
            "DescribeDBClusterParameterGroups": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterParameterGroups.html",
                "description": "Grants permission to return a list of DBClusterParameterGroup descriptions",
                "accessLevel": "List",
                "resourceTypes": {
                    "cluster-pg": {
                        "required": true
                    }
                }
            },
            "DescribeDBClusterParameters": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterParameters.html",
                "description": "Grants permission to return the detailed parameter list for a particular DB cluster parameter group",
                "accessLevel": "List",
                "resourceTypes": {
                    "cluster-pg": {
                        "required": true
                    }
                }
            },
            "DescribeDBClusterSnapshotAttributes": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterSnapshotAttributes.html",
                "description": "Grants permission to return a list of DB cluster snapshot attribute names and values for a manual DB cluster snapshot",
                "accessLevel": "List",
                "resourceTypes": {
                    "cluster-snapshot": {
                        "required": true
                    }
                }
            },
            "DescribeDBClusterSnapshots": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterSnapshots.html",
                "description": "Grants permission to return information about DB cluster snapshots",
                "accessLevel": "List"
            },
            "DescribeDBClusters": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusters.html",
                "description": "Grants permission to return information about provisioned Aurora DB clusters",
                "accessLevel": "List",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "DescribeDBEngineVersions": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBEngineVersions.html",
                "description": "Grants permission to return a list of the available DB engines",
                "accessLevel": "List",
                "resourceTypes": {
                    "pg": {
                        "required": true
                    }
                }
            },
            "DescribeDBInstanceAutomatedBackups": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstanceAutomatedBackups.html",
                "description": "Grants permission to return a list of automated backups for both current and deleted instances",
                "accessLevel": "List"
            },
            "DescribeDBInstances": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstances.html",
                "description": "Grants permission to return information about provisioned RDS instances",
                "accessLevel": "List"
            },
            "DescribeDBLogFiles": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBLogFiles.html",
                "description": "Grants permission to return a list of DB log files for the DB instance",
                "accessLevel": "List",
                "resourceTypes": {
                    "db": {
                        "required": true
                    }
                }
            },
            "DescribeDBParameterGroups": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBParameterGroups.html",
                "description": "Grants permission to return a list of DBParameterGroup descriptions",
                "accessLevel": "List",
                "resourceTypes": {
                    "pg": {
                        "required": true
                    }
                }
            },
            "DescribeDBParameters": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBParameters.html",
                "description": "Grants permission to return the detailed parameter list for a particular DB parameter group",
                "accessLevel": "List",
                "resourceTypes": {
                    "pg": {
                        "required": true
                    }
                }
            },
            "DescribeDBProxies": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxies.html",
                "description": "Grants permission to view proxies",
                "accessLevel": "List",
                "resourceTypes": {
                    "proxy": {
                        "required": true
                    }
                }
            },
            "DescribeDBProxyTargetGroups": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxyTargetGroups.html",
                "description": "Grants permission to view database proxy target group details",
                "accessLevel": "List",
                "resourceTypes": {
                    "proxy": {
                        "required": true
                    }
                }
            },
            "DescribeDBProxyTargets": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxyTargets.html",
                "description": "Grants permission to view database proxy target details",
                "accessLevel": "List",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    },
                    "db": {
                        "required": true
                    },
                    "proxy": {
                        "required": true
                    },
                    "target-group": {
                        "required": true
                    }
                }
            },
            "DescribeDBSecurityGroups": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSecurityGroups.html",
                "description": "Grants permission to return a list of DBSecurityGroup descriptions",
                "accessLevel": "List",
                "resourceTypes": {
                    "secgrp": {
                        "required": true
                    }
                }
            },
            "DescribeDBSnapshotAttributes": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSnapshotAttributes.html",
                "description": "Grants permission to return a list of DB snapshot attribute names and values for a manual DB snapshot",
                "accessLevel": "List",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "DescribeDBSnapshots": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSnapshots.html",
                "description": "Grants permission to return information about DB snapshots",
                "accessLevel": "List",
                "resourceTypes": {
                    "db": {
                        "required": true
                    },
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "DescribeDBSubnetGroups": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSubnetGroups.html",
                "description": "Grants permission to return a list of DBSubnetGroup descriptions",
                "accessLevel": "List",
                "resourceTypes": {
                    "subgrp": {
                        "required": true
                    }
                }
            },
            "DescribeEngineDefaultClusterParameters": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEngineDefaultClusterParameters.html",
                "description": "Grants permission to return the default engine and system parameter information for the cluster database engine",
                "accessLevel": "List"
            },
            "DescribeEngineDefaultParameters": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEngineDefaultParameters.html",
                "description": "Grants permission to return the default engine and system parameter information for the specified database engine",
                "accessLevel": "List"
            },
            "DescribeEventCategories": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEventCategories.html",
                "description": "Grants permission to display a list of categories for all event source types, or, if specified, for a specified source type",
                "accessLevel": "List"
            },
            "DescribeEventSubscriptions": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEventSubscriptions.html",
                "description": "Grants permission to list all the subscription descriptions for a customer account",
                "accessLevel": "List",
                "resourceTypes": {
                    "es": {
                        "required": true
                    }
                }
            },
            "DescribeEvents": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEvents.html",
                "description": "Grants permission to return events related to DB instances, DB security groups, DB snapshots, and DB parameter groups for the past 14 days",
                "accessLevel": "List"
            },
            "DescribeExportTasks": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeExportTasks.html",
                "description": "Grants permission to return information about the export tasks",
                "accessLevel": "List"
            },
            "DescribeGlobalClusters": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeGlobalClusters.html",
                "description": "Grants permission to return information about Aurora global database clusters",
                "accessLevel": "List"
            },
            "DescribeOptionGroupOptions": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeOptionGroupOptions.html",
                "description": "Grants permission to describe all available options",
                "accessLevel": "List",
                "resourceTypes": {
                    "og": {
                        "required": true
                    }
                }
            },
            "DescribeOptionGroups": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeOptionGroups.html",
                "description": "Grants permission to describe the available option groups",
                "accessLevel": "List",
                "resourceTypes": {
                    "og": {
                        "required": true
                    }
                }
            },
            "DescribeOrderableDBInstanceOptions": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeOrderableDBInstanceOptions.html",
                "description": "Grants permission to return a list of orderable DB instance options for the specified engine",
                "accessLevel": "List"
            },
            "DescribePendingMaintenanceActions": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribePendingMaintenanceActions.html",
                "description": "Grants permission to return a list of resources (for example, DB instances) that have at least one pending maintenance action",
                "accessLevel": "List",
                "resourceTypes": {
                    "db": {
                        "required": true
                    }
                }
            },
            "DescribeReservedDBInstances": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeReservedDBInstances.html",
                "description": "Grants permission to return information about reserved DB instances for this account, or about a specified reserved DB instance",
                "accessLevel": "List",
                "resourceTypes": {
                    "ri": {
                        "required": true
                    }
                }
            },
            "DescribeReservedDBInstancesOfferings": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeReservedDBInstancesOfferings.html",
                "description": "Grants permission to list available reserved DB instance offerings",
                "accessLevel": "List"
            },
            "DescribeSourceRegions": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeSourceRegions.html",
                "description": "Grants permission to return a list of the source AWS Regions where the current AWS Region can create a Read Replica or copy a DB snapshot from",
                "accessLevel": "List"
            },
            "DescribeValidDBInstanceModifications": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeValidDBInstanceModifications.html",
                "description": "Grants permission to list available modifications you can make to your DB instance",
                "accessLevel": "List",
                "resourceTypes": {
                    "db": {
                        "required": true
                    }
                }
            },
            "DownloadDBLogFilePortion": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DownloadDBLogFilePortion.html",
                "description": "Grants permission to download all or a portion of the specified log file, up to 1 MB in size",
                "accessLevel": "Read",
                "resourceTypes": {
                    "db": {
                        "required": true
                    }
                }
            },
            "FailoverDBCluster": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_FailoverDBCluster.html",
                "description": "Grants permission to force a failover for a DB cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list all tags on an Amazon RDS resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "db": {
                        "required": false
                    },
                    "es": {
                        "required": false
                    },
                    "og": {
                        "required": false
                    },
                    "pg": {
                        "required": false
                    },
                    "proxy": {
                        "required": false
                    },
                    "ri": {
                        "required": false
                    },
                    "secgrp": {
                        "required": false
                    },
                    "snapshot": {
                        "required": false
                    },
                    "subgrp": {
                        "required": false
                    },
                    "target-group": {
                        "required": false
                    }
                }
            },
            "ModifyCurrentDBClusterCapacity": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyCurrentDBClusterCapacity.html",
                "description": "Grants permission to modify current cluster capacity for an Amazon Aurora Severless DB cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "ModifyDBCluster": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBCluster.html",
                "description": "Grants permission to modify a setting for an Amazon Aurora DB cluster",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    },
                    "cluster-pg": {
                        "required": true
                    },
                    "og": {
                        "required": true
                    }
                }
            },
            "ModifyDBClusterEndpoint": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBClusterEndpoint.html",
                "description": "Grants permission to modify the properties of an endpoint in an Amazon Aurora DB cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster-endpoint": {
                        "required": true
                    }
                }
            },
            "ModifyDBClusterParameterGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBClusterParameterGroup.html",
                "description": "Grants permission to modify the parameters of a DB cluster parameter group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster-pg": {
                        "required": true
                    }
                }
            },
            "ModifyDBClusterSnapshotAttribute": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBClusterSnapshotAttribute.html",
                "description": "Grants permission to add an attribute and values to, or removes an attribute and values from, a manual DB cluster snapshot",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster-snapshot": {
                        "required": true
                    }
                }
            },
            "ModifyDBInstance": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBInstance.html",
                "description": "Grants permission to modify settings for a DB instance",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "db": {
                        "required": true
                    },
                    "og": {
                        "required": true
                    },
                    "pg": {
                        "required": true
                    },
                    "secgrp": {
                        "required": true
                    }
                }
            },
            "ModifyDBParameterGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBParameterGroup.html",
                "description": "Grants permission to modify the parameters of a DB parameter group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pg": {
                        "required": true
                    }
                }
            },
            "ModifyDBProxy": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBProxy.html",
                "description": "Grants permission to modify database proxy",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "proxy": {
                        "required": true
                    }
                }
            },
            "ModifyDBProxyTargetGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBProxyTargetGroup.html",
                "description": "Grants permission to modify target group for a database proxy",
                "accessLevel": "Write",
                "resourceTypes": {
                    "target-group": {
                        "required": true
                    }
                }
            },
            "ModifyDBSnapshot": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSnapshot.html",
                "description": "Grants permission to update a manual DB snapshot, which can be encrypted or not encrypted, with a new engine version",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "ModifyDBSnapshotAttribute": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSnapshotAttribute.html",
                "description": "Grants permission to add an attribute and values to, or removes an attribute and values from, a manual DB snapshot",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "ModifyDBSubnetGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSubnetGroup.html",
                "description": "Grants permission to modify an existing DB subnet group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subgrp": {
                        "required": true
                    }
                }
            },
            "ModifyEventSubscription": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyEventSubscription.html",
                "description": "Grants permission to modify an existing RDS event notification subscription",
                "accessLevel": "Write",
                "resourceTypes": {
                    "es": {
                        "required": true
                    }
                }
            },
            "ModifyGlobalCluster": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyGlobalCluster.html",
                "description": "Grants permission to modify a setting for an Amazon Aurora global cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-cluster": {
                        "required": true
                    }
                }
            },
            "ModifyOptionGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyOptionGroup.html",
                "description": "Grants permission to modify an existing option group",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "og": {
                        "required": true
                    }
                }
            },
            "PromoteReadReplica": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_PromoteReadReplica.html",
                "description": "Grants permission to promote a Read Replica DB instance to a standalone DB instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "db": {
                        "required": true
                    }
                }
            },
            "PromoteReadReplicaDBCluster": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_PromoteReadReplicaDBCluster.html",
                "description": "Grants permission to promote a Read Replica DB cluster to a standalone DB cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "PurchaseReservedDBInstancesOffering": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_PurchaseReservedDBInstancesOffering.html",
                "description": "Grants permission to purchase a reserved DB instance offering",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ri": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "RebootDBInstance": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RebootDBInstance.html",
                "description": "Grants permission to restart the database engine service",
                "accessLevel": "Write",
                "resourceTypes": {
                    "db": {
                        "required": true
                    }
                }
            },
            "RegisterDBProxyTargets": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RegisterDBProxyTargets.html",
                "description": "Grants permission to add targets to a database proxy target group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "target-group": {
                        "required": true
                    }
                }
            },
            "RemoveFromGlobalCluster": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveFromGlobalCluster.html",
                "description": "Grants permission to detach an Aurora secondary cluster from an Aurora global database cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    },
                    "global-cluster": {
                        "required": true
                    }
                }
            },
            "RemoveRoleFromDBCluster": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveRoleFromDBCluster.html",
                "description": "Grants permission to disassociate an AWS Identity and Access Management (IAM) role from an Amazon Aurora DB cluster",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "RemoveRoleFromDBInstance": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveRoleFromDBInstance.html",
                "description": "Grants permission to disassociate an AWS Identity and Access Management (IAM) role from a DB instance",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "db": {
                        "required": true
                    }
                }
            },
            "RemoveSourceIdentifierFromSubscription": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveSourceIdentifierFromSubscription.html",
                "description": "Grants permission to remove a source identifier from an existing RDS event notification subscription",
                "accessLevel": "Write",
                "resourceTypes": {
                    "es": {
                        "required": true
                    }
                }
            },
            "RemoveTagsFromResource": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveTagsFromResource.html",
                "description": "Grants permission to remove metadata tags from an Amazon RDS resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "db": {
                        "required": false
                    },
                    "es": {
                        "required": false
                    },
                    "og": {
                        "required": false
                    },
                    "pg": {
                        "required": false
                    },
                    "proxy": {
                        "required": false
                    },
                    "ri": {
                        "required": false
                    },
                    "secgrp": {
                        "required": false
                    },
                    "snapshot": {
                        "required": false
                    },
                    "subgrp": {
                        "required": false
                    },
                    "target-group": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "ResetDBClusterParameterGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ResetDBClusterParameterGroup.html",
                "description": "Grants permission to modify the parameters of a DB cluster parameter group to the default value",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster-pg": {
                        "required": true
                    }
                }
            },
            "ResetDBParameterGroup": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ResetDBParameterGroup.html",
                "description": "Grants permission to modify the parameters of a DB parameter group to the engine/system default value",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pg": {
                        "required": true
                    }
                }
            },
            "RestoreDBClusterFromS3": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBClusterFromS3.html",
                "description": "Grants permission to create an Amazon Aurora DB cluster from data stored in an Amazon S3 bucket",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}",
                    "rds:DatabaseEngine",
                    "rds:DatabaseName",
                    "rds:StorageEncrypted"
                ]
            },
            "RestoreDBClusterFromSnapshot": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBClusterFromSnapshot.html",
                "description": "Grants permission to create a new DB cluster from a DB cluster snapshot",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    },
                    "cluster-snapshot": {
                        "required": true
                    },
                    "og": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "RestoreDBClusterToPointInTime": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBClusterToPointInTime.html",
                "description": "Grants permission to restore a DB cluster to an arbitrary point in time",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    },
                    "og": {
                        "required": true
                    },
                    "subgrp": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "RestoreDBInstanceFromDBSnapshot": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceFromDBSnapshot.html",
                "description": "Grants permission to create a new DB instance from a DB snapshot",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "db": {
                        "required": true
                    },
                    "og": {
                        "required": true
                    },
                    "snapshot": {
                        "required": true
                    },
                    "subgrp": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "RestoreDBInstanceFromS3": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceFromS3.html",
                "description": "Grants permission to create a new DB instance from an Amazon S3 bucket",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "db": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "RestoreDBInstanceToPointInTime": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceToPointInTime.html",
                "description": "Grants permission to restore a DB instance to an arbitrary point in time",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "db": {
                        "required": true
                    },
                    "og": {
                        "required": true
                    },
                    "subgrp": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "rds:req-tag/${TagKey}"
                ]
            },
            "RevokeDBSecurityGroupIngress": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RevokeDBSecurityGroupIngress.html",
                "description": "Grants permission to revoke ingress from a DBSecurityGroup for previously authorized IP ranges or EC2 or VPC Security Groups",
                "accessLevel": "Write",
                "resourceTypes": {
                    "secgrp": {
                        "required": true
                    }
                }
            },
            "StartActivityStream": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartActivityStream.html",
                "description": "Grants permission to start Activity Stream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "StartDBCluster": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartDBCluster.html",
                "description": "Starts the DB cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "StartDBInstance": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartDBInstance.html",
                "description": "Grants permission to start the DB instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "db": {
                        "required": true
                    }
                }
            },
            "StartExportTask": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartExportTask.html",
                "description": "Grants permission to start a new Export task for a DB snapshot",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ]
            },
            "StopActivityStream": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopActivityStream.html",
                "description": "Grants permission to stop Activity Stream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "StopDBCluster": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopDBCluster.html",
                "description": "Grants permission to stop the DB cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "StopDBInstance": {
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopDBInstance.html",
                "description": "Grants permission to stop the DB instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "db": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "cluster": {
                "name": "cluster",
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Aurora.Managing.html",
                "arn": "arn:${Partition}:rds:${Region}:${Account}:cluster:${DbClusterInstanceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "rds:cluster-tag/${TagKey}"
                ]
            },
            "cluster-endpoint": {
                "name": "cluster-endpoint",
                "url": "",
                "arn": "arn:${Partition}:rds:${Region}:${Account}:cluster-endpoint:${DbClusterEndpoint}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "cluster-pg": {
                "name": "cluster-pg",
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html",
                "arn": "arn:${Partition}:rds:${Region}:${Account}:cluster-pg:${ClusterParameterGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "rds:cluster-pg-tag/${TagKey}"
                ]
            },
            "cluster-snapshot": {
                "name": "cluster-snapshot",
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html",
                "arn": "arn:${Partition}:rds:${Region}:${Account}:cluster-snapshot:${ClusterSnapshotName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "rds:cluster-snapshot-tag/${TagKey}"
                ]
            },
            "db": {
                "name": "db",
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.html",
                "arn": "arn:${Partition}:rds:${Region}:${Account}:db:${DbInstanceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "rds:DatabaseClass",
                    "rds:DatabaseEngine",
                    "rds:DatabaseName",
                    "rds:MultiAz",
                    "rds:Piops",
                    "rds:StorageEncrypted",
                    "rds:StorageSize",
                    "rds:Vpc",
                    "rds:db-tag/${TagKey}"
                ]
            },
            "es": {
                "name": "es",
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Events.html",
                "arn": "arn:${Partition}:rds:${Region}:${Account}:es:${SubscriptionName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "rds:es-tag/${TagKey}"
                ]
            },
            "global-cluster": {
                "name": "global-cluster",
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html",
                "arn": "arn:${Partition}:rds:${Account}:global-cluster:${GlobalCluster}",
                "conditionKeys": []
            },
            "og": {
                "name": "og",
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithOptionGroups.html",
                "arn": "arn:${Partition}:rds:${Region}:${Account}:og:${OptionGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "rds:og-tag/${TagKey}"
                ]
            },
            "pg": {
                "name": "pg",
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html",
                "arn": "arn:${Partition}:rds:${Region}:${Account}:pg:${ParameterGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "rds:pg-tag/${TagKey}"
                ]
            },
            "proxy": {
                "name": "proxy",
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBProxy.html",
                "arn": "arn:${Partition}:rds:${Region}:${Account}:db-proxy:${DbProxyId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "ri": {
                "name": "ri",
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithReservedDBInstances.html",
                "arn": "arn:${Partition}:rds:${Region}:${Account}:ri:${ReservedDbInstanceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "rds:ri-tag/${TagKey}"
                ]
            },
            "secgrp": {
                "name": "secgrp",
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithSecurityGroups.html",
                "arn": "arn:${Partition}:rds:${Region}:${Account}:secgrp:${SecurityGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "rds:secgrp-tag/${TagKey}"
                ]
            },
            "snapshot": {
                "name": "snapshot",
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html",
                "arn": "arn:${Partition}:rds:${Region}:${Account}:snapshot:${SnapshotName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "rds:snapshot-tag/${TagKey}"
                ]
            },
            "subgrp": {
                "name": "subgrp",
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.Scenarios.html#USER_VPC.Scenario1",
                "arn": "arn:${Partition}:rds:${Region}:${Account}:subgrp:${SubnetGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "rds:subgrp-tag/${TagKey}"
                ]
            },
            "target": {
                "name": "target",
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBProxy.html",
                "arn": "arn:${Partition}:rds:${Region}:${Account}:target:${TargetId}",
                "conditionKeys": []
            },
            "target-group": {
                "name": "target-group",
                "url": "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBProxy.html",
                "arn": "arn:${Partition}:rds:${Region}:${Account}:target-group:${TargetGroupId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to associate an Identity and Access Management (IAM) role from an Aurora DB cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddRoleToDBCluster.html
     */
    addRoleToDBCluster() {
        this.add('rds:AddRoleToDBCluster');
        return this;
    }
    /**
     * Grants permission to associate an AWS Identity and Access Management (IAM) role with a DB instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddRoleToDBInstance.html
     */
    addRoleToDBInstance() {
        this.add('rds:AddRoleToDBInstance');
        return this;
    }
    /**
     * Grants permission to add a source identifier to an existing RDS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddSourceIdentifierToSubscription.html
     */
    addSourceIdentifierToSubscription() {
        this.add('rds:AddSourceIdentifierToSubscription');
        return this;
    }
    /**
     * Grants permission to add metadata tags to an Amazon RDS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddTagsToResource.html
     */
    addTagsToResource() {
        this.add('rds:AddTagsToResource');
        return this;
    }
    /**
     * Grants permission to apply a pending maintenance action to a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ApplyPendingMaintenanceAction.html
     */
    applyPendingMaintenanceAction() {
        this.add('rds:ApplyPendingMaintenanceAction');
        return this;
    }
    /**
     * Grants permission to enable ingress to a DBSecurityGroup using one of two forms of authorization
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AuthorizeDBSecurityGroupIngress.html
     */
    authorizeDBSecurityGroupIngress() {
        this.add('rds:AuthorizeDBSecurityGroupIngress');
        return this;
    }
    /**
     * Grants permission to backtrack a DB cluster to a specific time, without creating a new DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_BacktrackDBCluster.html
     */
    backtrackDBCluster() {
        this.add('rds:BacktrackDBCluster');
        return this;
    }
    /**
     * Grants permission to cancel an export task in progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CancelExportTask.html
     */
    cancelExportTask() {
        this.add('rds:CancelExportTask');
        return this;
    }
    /**
     * Grants permission to copy the specified DB cluster parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBClusterParameterGroup.html
     */
    copyDBClusterParameterGroup() {
        this.add('rds:CopyDBClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to create a snapshot of a DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBClusterSnapshot.html
     */
    copyDBClusterSnapshot() {
        this.add('rds:CopyDBClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to copy the specified DB parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBParameterGroup.html
     */
    copyDBParameterGroup() {
        this.add('rds:CopyDBParameterGroup');
        return this;
    }
    /**
     * Grants permission to copy the specified DB snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBSnapshot.html
     */
    copyDBSnapshot() {
        this.add('rds:CopyDBSnapshot');
        return this;
    }
    /**
     * Grants permission to copy the specified option group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyOptionGroup.html
     */
    copyOptionGroup() {
        this.add('rds:CopyOptionGroup');
        return this;
    }
    /**
     * Grants permission to create a new Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     * - .ifDatabaseEngine()
     * - .ifDatabaseName()
     * - .ifStorageEncrypted()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBCluster.html
     */
    createDBCluster() {
        this.add('rds:CreateDBCluster');
        return this;
    }
    /**
     * Grants permission to create a new custom endpoint and associates it with an Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEndpointType()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBClusterEndpoint.html
     */
    createDBClusterEndpoint() {
        this.add('rds:CreateDBClusterEndpoint');
        return this;
    }
    /**
     * Grants permission to create a new DB cluster parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBClusterParameterGroup.html
     */
    createDBClusterParameterGroup() {
        this.add('rds:CreateDBClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to create a snapshot of a DB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBClusterSnapshot.html
     */
    createDBClusterSnapshot() {
        this.add('rds:CreateDBClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to create a new DB instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html
     */
    createDBInstance() {
        this.add('rds:CreateDBInstance');
        return this;
    }
    /**
     * Grants permission to create a DB instance that acts as a Read Replica of a source DB instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstanceReadReplica.html
     */
    createDBInstanceReadReplica() {
        this.add('rds:CreateDBInstanceReadReplica');
        return this;
    }
    /**
     * Grants permission to create a new DB parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBParameterGroup.html
     */
    createDBParameterGroup() {
        this.add('rds:CreateDBParameterGroup');
        return this;
    }
    /**
     * Grants permission to create a database proxy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBProxy.html
     */
    createDBProxy() {
        this.add('rds:CreateDBProxy');
        return this;
    }
    /**
     * Grants permission to create a new DB security group. DB security groups control access to a DB instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBSecurityGroup.html
     */
    createDBSecurityGroup() {
        this.add('rds:CreateDBSecurityGroup');
        return this;
    }
    /**
     * Grants permission to create a DBSnapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBSnapshot.html
     */
    createDBSnapshot() {
        this.add('rds:CreateDBSnapshot');
        return this;
    }
    /**
     * Grants permission to create a new DB subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBSubnetGroup.html
     */
    createDBSubnetGroup() {
        this.add('rds:CreateDBSubnetGroup');
        return this;
    }
    /**
     * Grants permission to create an RDS event notification subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateEventSubscription.html
     */
    createEventSubscription() {
        this.add('rds:CreateEventSubscription');
        return this;
    }
    /**
     * Grants permission to create an Aurora global database spread across multiple regions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateGlobalCluster.html
     */
    createGlobalCluster() {
        this.add('rds:CreateGlobalCluster');
        return this;
    }
    /**
     * Grants permission to create a new option group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateOptionGroup.html
     */
    createOptionGroup() {
        this.add('rds:CreateOptionGroup');
        return this;
    }
    /**
     * Grants permission to delete a previously provisioned DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBCluster.html
     */
    deleteDBCluster() {
        this.add('rds:DeleteDBCluster');
        return this;
    }
    /**
     * Grants permission to delete a custom endpoint and removes it from an Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBClusterEndpoint.html
     */
    deleteDBClusterEndpoint() {
        this.add('rds:DeleteDBClusterEndpoint');
        return this;
    }
    /**
     * Grants permission to delete a specified DB cluster parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBClusterParameterGroup.html
     */
    deleteDBClusterParameterGroup() {
        this.add('rds:DeleteDBClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to delete a DB cluster snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBClusterSnapshot.html
     */
    deleteDBClusterSnapshot() {
        this.add('rds:DeleteDBClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to delete a previously provisioned DB instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBInstance.html
     */
    deleteDBInstance() {
        this.add('rds:DeleteDBInstance');
        return this;
    }
    /**
     * Grants permission to deletes automated backups based on the source instance's DbiResourceId value or the restorable instance's resource ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBInstanceAutomatedBackup.html
     */
    deleteDBInstanceAutomatedBackup() {
        this.add('rds:DeleteDBInstanceAutomatedBackup');
        return this;
    }
    /**
     * Grants permission to delete a specified DBParameterGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBParameterGroup.html
     */
    deleteDBParameterGroup() {
        this.add('rds:DeleteDBParameterGroup');
        return this;
    }
    /**
     * Grants permission to delete a database proxy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBProxy.html
     */
    deleteDBProxy() {
        this.add('rds:DeleteDBProxy');
        return this;
    }
    /**
     * Grants permission to delete a DB security group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBSecurityGroup.html
     */
    deleteDBSecurityGroup() {
        this.add('rds:DeleteDBSecurityGroup');
        return this;
    }
    /**
     * Grants permission to delete a DBSnapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBSnapshot.html
     */
    deleteDBSnapshot() {
        this.add('rds:DeleteDBSnapshot');
        return this;
    }
    /**
     * Grants permission to delete a DB subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBSubnetGroup.html
     */
    deleteDBSubnetGroup() {
        this.add('rds:DeleteDBSubnetGroup');
        return this;
    }
    /**
     * Grants permission to delete an RDS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteEventSubscription.html
     */
    deleteEventSubscription() {
        this.add('rds:DeleteEventSubscription');
        return this;
    }
    /**
     * Grants permission to delete a global database cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteGlobalCluster.html
     */
    deleteGlobalCluster() {
        this.add('rds:DeleteGlobalCluster');
        return this;
    }
    /**
     * Grants permission to delete an existing option group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteOptionGroup.html
     */
    deleteOptionGroup() {
        this.add('rds:DeleteOptionGroup');
        return this;
    }
    /**
     * Grants permission to remove targets from a database proxy target group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeregisterDBProxyTargets.html
     */
    deregisterDBProxyTargets() {
        this.add('rds:DeregisterDBProxyTargets');
        return this;
    }
    /**
     * Grants permission to list all of the attributes for a customer account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeAccountAttributes.html
     */
    describeAccountAttributes() {
        this.add('rds:DescribeAccountAttributes');
        return this;
    }
    /**
     * Lists the set of CA certificates provided by Amazon RDS for this AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeCertificates.html
     */
    describeCertificates() {
        this.add('rds:DescribeCertificates');
        return this;
    }
    /**
     * Grants permission to return information about backtracks for a DB cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterBacktracks.html
     */
    describeDBClusterBacktracks() {
        this.add('rds:DescribeDBClusterBacktracks');
        return this;
    }
    /**
     * Grants permission to return information about endpoints for an Amazon Aurora DB cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterEndpoints.html
     */
    describeDBClusterEndpoints() {
        this.add('rds:DescribeDBClusterEndpoints');
        return this;
    }
    /**
     * Grants permission to return a list of DBClusterParameterGroup descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterParameterGroups.html
     */
    describeDBClusterParameterGroups() {
        this.add('rds:DescribeDBClusterParameterGroups');
        return this;
    }
    /**
     * Grants permission to return the detailed parameter list for a particular DB cluster parameter group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterParameters.html
     */
    describeDBClusterParameters() {
        this.add('rds:DescribeDBClusterParameters');
        return this;
    }
    /**
     * Grants permission to return a list of DB cluster snapshot attribute names and values for a manual DB cluster snapshot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterSnapshotAttributes.html
     */
    describeDBClusterSnapshotAttributes() {
        this.add('rds:DescribeDBClusterSnapshotAttributes');
        return this;
    }
    /**
     * Grants permission to return information about DB cluster snapshots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterSnapshots.html
     */
    describeDBClusterSnapshots() {
        this.add('rds:DescribeDBClusterSnapshots');
        return this;
    }
    /**
     * Grants permission to return information about provisioned Aurora DB clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusters.html
     */
    describeDBClusters() {
        this.add('rds:DescribeDBClusters');
        return this;
    }
    /**
     * Grants permission to return a list of the available DB engines
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBEngineVersions.html
     */
    describeDBEngineVersions() {
        this.add('rds:DescribeDBEngineVersions');
        return this;
    }
    /**
     * Grants permission to return a list of automated backups for both current and deleted instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstanceAutomatedBackups.html
     */
    describeDBInstanceAutomatedBackups() {
        this.add('rds:DescribeDBInstanceAutomatedBackups');
        return this;
    }
    /**
     * Grants permission to return information about provisioned RDS instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstances.html
     */
    describeDBInstances() {
        this.add('rds:DescribeDBInstances');
        return this;
    }
    /**
     * Grants permission to return a list of DB log files for the DB instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBLogFiles.html
     */
    describeDBLogFiles() {
        this.add('rds:DescribeDBLogFiles');
        return this;
    }
    /**
     * Grants permission to return a list of DBParameterGroup descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBParameterGroups.html
     */
    describeDBParameterGroups() {
        this.add('rds:DescribeDBParameterGroups');
        return this;
    }
    /**
     * Grants permission to return the detailed parameter list for a particular DB parameter group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBParameters.html
     */
    describeDBParameters() {
        this.add('rds:DescribeDBParameters');
        return this;
    }
    /**
     * Grants permission to view proxies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxies.html
     */
    describeDBProxies() {
        this.add('rds:DescribeDBProxies');
        return this;
    }
    /**
     * Grants permission to view database proxy target group details
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxyTargetGroups.html
     */
    describeDBProxyTargetGroups() {
        this.add('rds:DescribeDBProxyTargetGroups');
        return this;
    }
    /**
     * Grants permission to view database proxy target details
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxyTargets.html
     */
    describeDBProxyTargets() {
        this.add('rds:DescribeDBProxyTargets');
        return this;
    }
    /**
     * Grants permission to return a list of DBSecurityGroup descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSecurityGroups.html
     */
    describeDBSecurityGroups() {
        this.add('rds:DescribeDBSecurityGroups');
        return this;
    }
    /**
     * Grants permission to return a list of DB snapshot attribute names and values for a manual DB snapshot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSnapshotAttributes.html
     */
    describeDBSnapshotAttributes() {
        this.add('rds:DescribeDBSnapshotAttributes');
        return this;
    }
    /**
     * Grants permission to return information about DB snapshots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSnapshots.html
     */
    describeDBSnapshots() {
        this.add('rds:DescribeDBSnapshots');
        return this;
    }
    /**
     * Grants permission to return a list of DBSubnetGroup descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSubnetGroups.html
     */
    describeDBSubnetGroups() {
        this.add('rds:DescribeDBSubnetGroups');
        return this;
    }
    /**
     * Grants permission to return the default engine and system parameter information for the cluster database engine
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEngineDefaultClusterParameters.html
     */
    describeEngineDefaultClusterParameters() {
        this.add('rds:DescribeEngineDefaultClusterParameters');
        return this;
    }
    /**
     * Grants permission to return the default engine and system parameter information for the specified database engine
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEngineDefaultParameters.html
     */
    describeEngineDefaultParameters() {
        this.add('rds:DescribeEngineDefaultParameters');
        return this;
    }
    /**
     * Grants permission to display a list of categories for all event source types, or, if specified, for a specified source type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEventCategories.html
     */
    describeEventCategories() {
        this.add('rds:DescribeEventCategories');
        return this;
    }
    /**
     * Grants permission to list all the subscription descriptions for a customer account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEventSubscriptions.html
     */
    describeEventSubscriptions() {
        this.add('rds:DescribeEventSubscriptions');
        return this;
    }
    /**
     * Grants permission to return events related to DB instances, DB security groups, DB snapshots, and DB parameter groups for the past 14 days
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEvents.html
     */
    describeEvents() {
        this.add('rds:DescribeEvents');
        return this;
    }
    /**
     * Grants permission to return information about the export tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeExportTasks.html
     */
    describeExportTasks() {
        this.add('rds:DescribeExportTasks');
        return this;
    }
    /**
     * Grants permission to return information about Aurora global database clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeGlobalClusters.html
     */
    describeGlobalClusters() {
        this.add('rds:DescribeGlobalClusters');
        return this;
    }
    /**
     * Grants permission to describe all available options
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeOptionGroupOptions.html
     */
    describeOptionGroupOptions() {
        this.add('rds:DescribeOptionGroupOptions');
        return this;
    }
    /**
     * Grants permission to describe the available option groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeOptionGroups.html
     */
    describeOptionGroups() {
        this.add('rds:DescribeOptionGroups');
        return this;
    }
    /**
     * Grants permission to return a list of orderable DB instance options for the specified engine
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeOrderableDBInstanceOptions.html
     */
    describeOrderableDBInstanceOptions() {
        this.add('rds:DescribeOrderableDBInstanceOptions');
        return this;
    }
    /**
     * Grants permission to return a list of resources (for example, DB instances) that have at least one pending maintenance action
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribePendingMaintenanceActions.html
     */
    describePendingMaintenanceActions() {
        this.add('rds:DescribePendingMaintenanceActions');
        return this;
    }
    /**
     * Grants permission to return information about reserved DB instances for this account, or about a specified reserved DB instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeReservedDBInstances.html
     */
    describeReservedDBInstances() {
        this.add('rds:DescribeReservedDBInstances');
        return this;
    }
    /**
     * Grants permission to list available reserved DB instance offerings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeReservedDBInstancesOfferings.html
     */
    describeReservedDBInstancesOfferings() {
        this.add('rds:DescribeReservedDBInstancesOfferings');
        return this;
    }
    /**
     * Grants permission to return a list of the source AWS Regions where the current AWS Region can create a Read Replica or copy a DB snapshot from
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeSourceRegions.html
     */
    describeSourceRegions() {
        this.add('rds:DescribeSourceRegions');
        return this;
    }
    /**
     * Grants permission to list available modifications you can make to your DB instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeValidDBInstanceModifications.html
     */
    describeValidDBInstanceModifications() {
        this.add('rds:DescribeValidDBInstanceModifications');
        return this;
    }
    /**
     * Grants permission to download all or a portion of the specified log file, up to 1 MB in size
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DownloadDBLogFilePortion.html
     */
    downloadDBLogFilePortion() {
        this.add('rds:DownloadDBLogFilePortion');
        return this;
    }
    /**
     * Grants permission to force a failover for a DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_FailoverDBCluster.html
     */
    failoverDBCluster() {
        this.add('rds:FailoverDBCluster');
        return this;
    }
    /**
     * Grants permission to list all tags on an Amazon RDS resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('rds:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to modify current cluster capacity for an Amazon Aurora Severless DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyCurrentDBClusterCapacity.html
     */
    modifyCurrentDBClusterCapacity() {
        this.add('rds:ModifyCurrentDBClusterCapacity');
        return this;
    }
    /**
     * Grants permission to modify a setting for an Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBCluster.html
     */
    modifyDBCluster() {
        this.add('rds:ModifyDBCluster');
        return this;
    }
    /**
     * Grants permission to modify the properties of an endpoint in an Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBClusterEndpoint.html
     */
    modifyDBClusterEndpoint() {
        this.add('rds:ModifyDBClusterEndpoint');
        return this;
    }
    /**
     * Grants permission to modify the parameters of a DB cluster parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBClusterParameterGroup.html
     */
    modifyDBClusterParameterGroup() {
        this.add('rds:ModifyDBClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to add an attribute and values to, or removes an attribute and values from, a manual DB cluster snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBClusterSnapshotAttribute.html
     */
    modifyDBClusterSnapshotAttribute() {
        this.add('rds:ModifyDBClusterSnapshotAttribute');
        return this;
    }
    /**
     * Grants permission to modify settings for a DB instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBInstance.html
     */
    modifyDBInstance() {
        this.add('rds:ModifyDBInstance');
        return this;
    }
    /**
     * Grants permission to modify the parameters of a DB parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBParameterGroup.html
     */
    modifyDBParameterGroup() {
        this.add('rds:ModifyDBParameterGroup');
        return this;
    }
    /**
     * Grants permission to modify database proxy
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBProxy.html
     */
    modifyDBProxy() {
        this.add('rds:ModifyDBProxy');
        return this;
    }
    /**
     * Grants permission to modify target group for a database proxy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBProxyTargetGroup.html
     */
    modifyDBProxyTargetGroup() {
        this.add('rds:ModifyDBProxyTargetGroup');
        return this;
    }
    /**
     * Grants permission to update a manual DB snapshot, which can be encrypted or not encrypted, with a new engine version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSnapshot.html
     */
    modifyDBSnapshot() {
        this.add('rds:ModifyDBSnapshot');
        return this;
    }
    /**
     * Grants permission to add an attribute and values to, or removes an attribute and values from, a manual DB snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSnapshotAttribute.html
     */
    modifyDBSnapshotAttribute() {
        this.add('rds:ModifyDBSnapshotAttribute');
        return this;
    }
    /**
     * Grants permission to modify an existing DB subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSubnetGroup.html
     */
    modifyDBSubnetGroup() {
        this.add('rds:ModifyDBSubnetGroup');
        return this;
    }
    /**
     * Grants permission to modify an existing RDS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyEventSubscription.html
     */
    modifyEventSubscription() {
        this.add('rds:ModifyEventSubscription');
        return this;
    }
    /**
     * Grants permission to modify a setting for an Amazon Aurora global cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyGlobalCluster.html
     */
    modifyGlobalCluster() {
        this.add('rds:ModifyGlobalCluster');
        return this;
    }
    /**
     * Grants permission to modify an existing option group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyOptionGroup.html
     */
    modifyOptionGroup() {
        this.add('rds:ModifyOptionGroup');
        return this;
    }
    /**
     * Grants permission to promote a Read Replica DB instance to a standalone DB instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_PromoteReadReplica.html
     */
    promoteReadReplica() {
        this.add('rds:PromoteReadReplica');
        return this;
    }
    /**
     * Grants permission to promote a Read Replica DB cluster to a standalone DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_PromoteReadReplicaDBCluster.html
     */
    promoteReadReplicaDBCluster() {
        this.add('rds:PromoteReadReplicaDBCluster');
        return this;
    }
    /**
     * Grants permission to purchase a reserved DB instance offering
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_PurchaseReservedDBInstancesOffering.html
     */
    purchaseReservedDBInstancesOffering() {
        this.add('rds:PurchaseReservedDBInstancesOffering');
        return this;
    }
    /**
     * Grants permission to restart the database engine service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RebootDBInstance.html
     */
    rebootDBInstance() {
        this.add('rds:RebootDBInstance');
        return this;
    }
    /**
     * Grants permission to add targets to a database proxy target group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RegisterDBProxyTargets.html
     */
    registerDBProxyTargets() {
        this.add('rds:RegisterDBProxyTargets');
        return this;
    }
    /**
     * Grants permission to detach an Aurora secondary cluster from an Aurora global database cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveFromGlobalCluster.html
     */
    removeFromGlobalCluster() {
        this.add('rds:RemoveFromGlobalCluster');
        return this;
    }
    /**
     * Grants permission to disassociate an AWS Identity and Access Management (IAM) role from an Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveRoleFromDBCluster.html
     */
    removeRoleFromDBCluster() {
        this.add('rds:RemoveRoleFromDBCluster');
        return this;
    }
    /**
     * Grants permission to disassociate an AWS Identity and Access Management (IAM) role from a DB instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveRoleFromDBInstance.html
     */
    removeRoleFromDBInstance() {
        this.add('rds:RemoveRoleFromDBInstance');
        return this;
    }
    /**
     * Grants permission to remove a source identifier from an existing RDS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveSourceIdentifierFromSubscription.html
     */
    removeSourceIdentifierFromSubscription() {
        this.add('rds:RemoveSourceIdentifierFromSubscription');
        return this;
    }
    /**
     * Grants permission to remove metadata tags from an Amazon RDS resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveTagsFromResource.html
     */
    removeTagsFromResource() {
        this.add('rds:RemoveTagsFromResource');
        return this;
    }
    /**
     * Grants permission to modify the parameters of a DB cluster parameter group to the default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ResetDBClusterParameterGroup.html
     */
    resetDBClusterParameterGroup() {
        this.add('rds:ResetDBClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to modify the parameters of a DB parameter group to the engine/system default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ResetDBParameterGroup.html
     */
    resetDBParameterGroup() {
        this.add('rds:ResetDBParameterGroup');
        return this;
    }
    /**
     * Grants permission to create an Amazon Aurora DB cluster from data stored in an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     * - .ifDatabaseEngine()
     * - .ifDatabaseName()
     * - .ifStorageEncrypted()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBClusterFromS3.html
     */
    restoreDBClusterFromS3() {
        this.add('rds:RestoreDBClusterFromS3');
        return this;
    }
    /**
     * Grants permission to create a new DB cluster from a DB cluster snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBClusterFromSnapshot.html
     */
    restoreDBClusterFromSnapshot() {
        this.add('rds:RestoreDBClusterFromSnapshot');
        return this;
    }
    /**
     * Grants permission to restore a DB cluster to an arbitrary point in time
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBClusterToPointInTime.html
     */
    restoreDBClusterToPointInTime() {
        this.add('rds:RestoreDBClusterToPointInTime');
        return this;
    }
    /**
     * Grants permission to create a new DB instance from a DB snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceFromDBSnapshot.html
     */
    restoreDBInstanceFromDBSnapshot() {
        this.add('rds:RestoreDBInstanceFromDBSnapshot');
        return this;
    }
    /**
     * Grants permission to create a new DB instance from an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceFromS3.html
     */
    restoreDBInstanceFromS3() {
        this.add('rds:RestoreDBInstanceFromS3');
        return this;
    }
    /**
     * Grants permission to restore a DB instance to an arbitrary point in time
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceToPointInTime.html
     */
    restoreDBInstanceToPointInTime() {
        this.add('rds:RestoreDBInstanceToPointInTime');
        return this;
    }
    /**
     * Grants permission to revoke ingress from a DBSecurityGroup for previously authorized IP ranges or EC2 or VPC Security Groups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RevokeDBSecurityGroupIngress.html
     */
    revokeDBSecurityGroupIngress() {
        this.add('rds:RevokeDBSecurityGroupIngress');
        return this;
    }
    /**
     * Grants permission to start Activity Stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartActivityStream.html
     */
    startActivityStream() {
        this.add('rds:StartActivityStream');
        return this;
    }
    /**
     * Starts the DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartDBCluster.html
     */
    startDBCluster() {
        this.add('rds:StartDBCluster');
        return this;
    }
    /**
     * Grants permission to start the DB instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartDBInstance.html
     */
    startDBInstance() {
        this.add('rds:StartDBInstance');
        return this;
    }
    /**
     * Grants permission to start a new Export task for a DB snapshot
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartExportTask.html
     */
    startExportTask() {
        this.add('rds:StartExportTask');
        return this;
    }
    /**
     * Grants permission to stop Activity Stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopActivityStream.html
     */
    stopActivityStream() {
        this.add('rds:StopActivityStream');
        return this;
    }
    /**
     * Grants permission to stop the DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopDBCluster.html
     */
    stopDBCluster() {
        this.add('rds:StopDBCluster');
        return this;
    }
    /**
     * Grants permission to stop the DB instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopDBInstance.html
     */
    stopDBInstance() {
        this.add('rds:StopDBInstance');
        return this;
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Aurora.Managing.html
     *
     * @param dbClusterInstanceName - Identifier for the dbClusterInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifClusterTag()
     */
    onCluster(dbClusterInstanceName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:cluster:${DbClusterInstanceName}';
        arn = arn.replace('${DbClusterInstanceName}', dbClusterInstanceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cluster-endpoint to the statement
     *
     * @param dbClusterEndpoint - Identifier for the dbClusterEndpoint.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onClusterEndpoint(dbClusterEndpoint, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:cluster-endpoint:${DbClusterEndpoint}';
        arn = arn.replace('${DbClusterEndpoint}', dbClusterEndpoint);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cluster-pg to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html
     *
     * @param clusterParameterGroupName - Identifier for the clusterParameterGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifClusterPgTag()
     */
    onClusterPg(clusterParameterGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:cluster-pg:${ClusterParameterGroupName}';
        arn = arn.replace('${ClusterParameterGroupName}', clusterParameterGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cluster-snapshot to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html
     *
     * @param clusterSnapshotName - Identifier for the clusterSnapshotName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifClusterSnapshotTag()
     */
    onClusterSnapshot(clusterSnapshotName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:cluster-snapshot:${ClusterSnapshotName}';
        arn = arn.replace('${ClusterSnapshotName}', clusterSnapshotName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type db to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.html
     *
     * @param dbInstanceName - Identifier for the dbInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifDatabaseClass()
     * - .ifDatabaseEngine()
     * - .ifDatabaseName()
     * - .ifMultiAz()
     * - .ifPiops()
     * - .ifStorageEncrypted()
     * - .ifStorageSize()
     * - .ifVpc()
     * - .ifDbTag()
     */
    onDb(dbInstanceName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:db:${DbInstanceName}';
        arn = arn.replace('${DbInstanceName}', dbInstanceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type es to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Events.html
     *
     * @param subscriptionName - Identifier for the subscriptionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifEsTag()
     */
    onEs(subscriptionName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:es:${SubscriptionName}';
        arn = arn.replace('${SubscriptionName}', subscriptionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type global-cluster to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html
     *
     * @param globalCluster - Identifier for the globalCluster.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGlobalCluster(globalCluster, account, partition) {
        var arn = 'arn:${Partition}:rds:${Account}:global-cluster:${GlobalCluster}';
        arn = arn.replace('${GlobalCluster}', globalCluster);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type og to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithOptionGroups.html
     *
     * @param optionGroupName - Identifier for the optionGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifOgTag()
     */
    onOg(optionGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:og:${OptionGroupName}';
        arn = arn.replace('${OptionGroupName}', optionGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type pg to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html
     *
     * @param parameterGroupName - Identifier for the parameterGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifPgTag()
     */
    onPg(parameterGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:pg:${ParameterGroupName}';
        arn = arn.replace('${ParameterGroupName}', parameterGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type proxy to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBProxy.html
     *
     * @param dbProxyId - Identifier for the dbProxyId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProxy(dbProxyId, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:db-proxy:${DbProxyId}';
        arn = arn.replace('${DbProxyId}', dbProxyId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ri to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithReservedDBInstances.html
     *
     * @param reservedDbInstanceName - Identifier for the reservedDbInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifRiTag()
     */
    onRi(reservedDbInstanceName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:ri:${ReservedDbInstanceName}';
        arn = arn.replace('${ReservedDbInstanceName}', reservedDbInstanceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type secgrp to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithSecurityGroups.html
     *
     * @param securityGroupName - Identifier for the securityGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSecgrpTag()
     */
    onSecgrp(securityGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:secgrp:${SecurityGroupName}';
        arn = arn.replace('${SecurityGroupName}', securityGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html
     *
     * @param snapshotName - Identifier for the snapshotName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSnapshotTag()
     */
    onSnapshot(snapshotName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:snapshot:${SnapshotName}';
        arn = arn.replace('${SnapshotName}', snapshotName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subgrp to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.Scenarios.html#USER_VPC.Scenario1
     *
     * @param subnetGroupName - Identifier for the subnetGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSubgrpTag()
     */
    onSubgrp(subnetGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:subgrp:${SubnetGroupName}';
        arn = arn.replace('${SubnetGroupName}', subnetGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type target to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBProxy.html
     *
     * @param targetId - Identifier for the targetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTarget(targetId, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:target:${TargetId}';
        arn = arn.replace('${TargetId}', targetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type target-group to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBProxy.html
     *
     * @param targetGroupId - Identifier for the targetGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTargetGroup(targetGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:rds:${Region}:${Account}:target-group:${TargetGroupId}';
        arn = arn.replace('${TargetGroupId}', targetGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .addTagsToResource()
     * - .createDBCluster()
     * - .createDBClusterEndpoint()
     * - .createDBClusterParameterGroup()
     * - .createDBClusterSnapshot()
     * - .createDBInstance()
     * - .createDBInstanceReadReplica()
     * - .createDBParameterGroup()
     * - .createDBProxy()
     * - .createDBSecurityGroup()
     * - .createDBSnapshot()
     * - .createDBSubnetGroup()
     * - .createEventSubscription()
     * - .createOptionGroup()
     * - .purchaseReservedDBInstancesOffering()
     * - .removeTagsFromResource()
     * - .restoreDBClusterFromS3()
     * - .restoreDBClusterFromSnapshot()
     * - .restoreDBClusterToPointInTime()
     * - .restoreDBInstanceFromDBSnapshot()
     * - .restoreDBInstanceFromS3()
     * - .restoreDBInstanceToPointInTime()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - cluster
     * - cluster-endpoint
     * - cluster-pg
     * - cluster-snapshot
     * - db
     * - es
     * - og
     * - pg
     * - proxy
     * - ri
     * - secgrp
     * - snapshot
     * - subgrp
     * - target-group
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .addTagsToResource()
     * - .createDBCluster()
     * - .createDBClusterEndpoint()
     * - .createDBClusterParameterGroup()
     * - .createDBClusterSnapshot()
     * - .createDBInstance()
     * - .createDBInstanceReadReplica()
     * - .createDBParameterGroup()
     * - .createDBProxy()
     * - .createDBSecurityGroup()
     * - .createDBSnapshot()
     * - .createDBSubnetGroup()
     * - .createEventSubscription()
     * - .createOptionGroup()
     * - .purchaseReservedDBInstancesOffering()
     * - .removeTagsFromResource()
     * - .restoreDBClusterFromS3()
     * - .restoreDBClusterFromSnapshot()
     * - .restoreDBClusterToPointInTime()
     * - .restoreDBInstanceFromDBSnapshot()
     * - .restoreDBInstanceFromS3()
     * - .restoreDBInstanceToPointInTime()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the type of DB instance class
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDatabaseClass(value, operator) {
        return this.if(`rds:DatabaseClass`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the database engine. For possible values refer to the engine parameter in CreateDBInstance API
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .createDBCluster()
     * - .restoreDBClusterFromS3()
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDatabaseEngine(value, operator) {
        return this.if(`rds:DatabaseEngine`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the user-defined name of the database on the DB instance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .createDBCluster()
     * - .restoreDBClusterFromS3()
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDatabaseName(value, operator) {
        return this.if(`rds:DatabaseName`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the type of the endpoint. One of: READER, WRITER, CUSTOM
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .createDBClusterEndpoint()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEndpointType(value, operator) {
        return this.if(`rds:EndpointType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the value that specifies whether the DB instance runs in multiple Availability Zones. To indicate that the DB instance is using Multi-AZ, specify true
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - db
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifMultiAz(value) {
        return this.if(`rds:MultiAz`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the value that contains the number of Provisioned IOPS (PIOPS) that the instance supports. To indicate a DB instance that does not have PIOPS enabled, specify 0
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPiops(value, operator) {
        return this.if(`rds:Piops`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the value that specifies whether the DB instance storage should be encrypted. To enforce storage encryption, specify true
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .createDBCluster()
     * - .restoreDBClusterFromS3()
     *
     * Applies to resource types:
     * - db
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifStorageEncrypted(value) {
        return this.if(`rds:StorageEncrypted`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the storage volume size (in GB)
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifStorageSize(value, operator) {
        return this.if(`rds:StorageSize`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the value that specifies whether the DB instance runs in an Amazon Virtual Private Cloud (Amazon VPC). To indicate that the DB instance runs in an Amazon VPC, specify true
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - db
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifVpc(value) {
        return this.if(`rds:Vpc`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the tag attached to a DB cluster parameter group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - cluster-pg
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifClusterPgTag(tagKey, value, operator) {
        return this.if(`rds:cluster-pg-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB cluster snapshot
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - cluster-snapshot
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifClusterSnapshotTag(tagKey, value, operator) {
        return this.if(`rds:cluster-snapshot-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB cluster
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - cluster
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifClusterTag(tagKey, value, operator) {
        return this.if(`rds:cluster-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB instance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - db
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDbTag(tagKey, value, operator) {
        return this.if(`rds:db-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to an event subscription
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - es
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEsTag(tagKey, value, operator) {
        return this.if(`rds:es-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB option group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - og
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOgTag(tagKey, value, operator) {
        return this.if(`rds:og-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB parameter group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - pg
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPgTag(tagKey, value, operator) {
        return this.if(`rds:pg-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the set of tag keys and values that can be used to tag a resource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .addTagsToResource()
     * - .createDBCluster()
     * - .createDBClusterParameterGroup()
     * - .createDBClusterSnapshot()
     * - .createDBInstance()
     * - .createDBInstanceReadReplica()
     * - .createDBParameterGroup()
     * - .createDBSecurityGroup()
     * - .createDBSnapshot()
     * - .createDBSubnetGroup()
     * - .createEventSubscription()
     * - .createOptionGroup()
     * - .removeTagsFromResource()
     * - .restoreDBClusterFromS3()
     * - .restoreDBClusterFromSnapshot()
     * - .restoreDBClusterToPointInTime()
     * - .restoreDBInstanceFromDBSnapshot()
     * - .restoreDBInstanceFromS3()
     * - .restoreDBInstanceToPointInTime()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReqTag(tagKey, value, operator) {
        return this.if(`rds:req-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a reserved DB instance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - ri
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRiTag(tagKey, value, operator) {
        return this.if(`rds:ri-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB security group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - secgrp
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecgrpTag(tagKey, value, operator) {
        return this.if(`rds:secgrp-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB snapshot
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - snapshot
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSnapshotTag(tagKey, value, operator) {
        return this.if(`rds:snapshot-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag attached to a DB subnet group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - subgrp
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubgrpTag(tagKey, value, operator) {
        return this.if(`rds:subgrp-tag/${tagKey}`, value, operator || 'StringLike');
    }
}
exports.Rds = Rds;
//# sourceMappingURL=data:application/json;base64,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