import { Actions, PolicyStatement, ResourceTypes } from "../shared";
/**
 * Statement provider for service [redshift](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonredshift.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Redshift extends PolicyStatement {
    servicePrefix: string;
    protected actionList: Actions;
    protected resourceTypes: ResourceTypes;
    /**
     * Statement provider for service [redshift](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonredshift.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid?: string);
    /**
     * Grants permission to exchange a DC1 reserved node for a DC2 reserved node with no changes to the configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_AcceptReservedNodeExchange.html
     */
    acceptReservedNodeExchange(): this;
    /**
     * Grants permission to add an inbound (ingress) rule to an Amazon Redshift security group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_AuthorizeClusterSecurityGroupIngress.html
     */
    authorizeClusterSecurityGroupIngress(): this;
    /**
     * Grants permission to the specified AWS account to restore a snapshot
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_AuthorizeSnapshotAccess.html
     */
    authorizeSnapshotAccess(): this;
    /**
     * Grants permission to delete snapshots in a batch of size upto 100
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_BatchDeleteClusterSnapshots.html
     */
    batchDeleteClusterSnapshots(): this;
    /**
     * Grants permission to modify settings for a list of snapshots
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_BatchModifyClusterSnapshots.html
     */
    batchModifyClusterSnapshots(): this;
    /**
     * Grants permission to cancel a query through the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    cancelQuery(): this;
    /**
     * Grants permission to see queries in the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    cancelQuerySession(): this;
    /**
     * Grants permission to cancel a resize operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CancelResize.html
     */
    cancelResize(): this;
    /**
     * Grants permission to copy a cluster snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CopyClusterSnapshot.html
     */
    copyClusterSnapshot(): this;
    /**
     * Grants permission to create a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateCluster.html
     */
    createCluster(): this;
    /**
     * Grants permission to create an Amazon Redshift parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateClusterParameterGroup.html
     */
    createClusterParameterGroup(): this;
    /**
     * Grants permission to create an Amazon Redshift security group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateClusterSecurityGroup.html
     */
    createClusterSecurityGroup(): this;
    /**
     * Grants permission to create a manual snapshot of the specified cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateClusterSnapshot.html
     */
    createClusterSnapshot(): this;
    /**
     * Grants permission to create an Amazon Redshift subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateClusterSubnetGroup.html
     */
    createClusterSubnetGroup(): this;
    /**
     * Grants permission to automatically create the specified Amazon Redshift user if it does not exist
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDbUser()
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/generating-iam-credentials-role-permissions.html
     */
    createClusterUser(): this;
    /**
     * Grants permission to create an Amazon Redshift event notification subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateEventSubscription.html
     */
    createEventSubscription(): this;
    /**
     * Grants permission to create an HSM client certificate that a cluster uses to connect to an HSM
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateHsmClientCertificate.html
     */
    createHsmClientCertificate(): this;
    /**
     * Grants permission to create an HSM configuration that contains information required by a cluster to store and use database encryption keys in a hardware security module (HSM)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateHsmConfiguration.html
     */
    createHsmConfiguration(): this;
    /**
     * Grants permission to create saved SQL queries through the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    createSavedQuery(): this;
    /**
     * Grants permission to create an Amazon Redshift scheduled action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateScheduledAction.html
     */
    createScheduledAction(): this;
    /**
     * Grants permission to create a snapshot copy grant and encrypt copied snapshots in a destination AWS Region
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateSnapshotCopyGrant.html
     */
    createSnapshotCopyGrant(): this;
    /**
     * Grants permission to create a snapshot schedule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateSnapshotSchedule.html
     */
    createSnapshotSchedule(): this;
    /**
     * Grants permission to add one or more tags to a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateTags.html
     */
    createTags(): this;
    /**
     * Grants permission to delete a previously provisioned cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteCluster.html
     */
    deleteCluster(): this;
    /**
     * Grants permission to delete an Amazon Redshift parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteClusterParameterGroup.html
     */
    deleteClusterParameterGroup(): this;
    /**
     * Grants permission to delete an Amazon Redshift security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteClusterSecurityGroup.html
     */
    deleteClusterSecurityGroup(): this;
    /**
     * Grants permission to delete a manual snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteClusterSnapshot.html
     */
    deleteClusterSnapshot(): this;
    /**
     * Grants permission to delete a cluster subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteClusterSubnetGroup.html
     */
    deleteClusterSubnetGroup(): this;
    /**
     * Grants permission to delete an Amazon Redshift event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteEventSubscription.html
     */
    deleteEventSubscription(): this;
    /**
     * Grants permission to delete an HSM client certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteHsmClientCertificate.html
     */
    deleteHsmClientCertificate(): this;
    /**
     * Grants permission to delete an Amazon Redshift HSM configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteHsmConfiguration.html
     */
    deleteHsmConfiguration(): this;
    /**
     * Grants permission to delete saved SQL queries through the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    deleteSavedQueries(): this;
    /**
     * Grants permission to delete an Amazon Redshift scheduled action
     *
     * Access Level: Write
     */
    deleteScheduledAction(): this;
    /**
     * Grants permission to delete a snapshot copy grant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteSnapshotCopyGrant.html
     */
    deleteSnapshotCopyGrant(): this;
    /**
     * Grants permission to delete a snapshot schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteSnapshotSchedule.html
     */
    deleteSnapshotSchedule(): this;
    /**
     * Grants permission to delete a tag or tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteTags.html
     */
    deleteTags(): this;
    /**
     * Grants permission to describe attributes attached to the specified AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeAccountAttributes.html
     */
    describeAccountAttributes(): this;
    /**
     * Grants permission to describe database revisions for a cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterDbRevisions.html
     */
    describeClusterDbRevisions(): this;
    /**
     * Grants permission to describe Amazon Redshift parameter groups, including parameter groups you created and the default parameter group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterParameterGroups.html
     */
    describeClusterParameterGroups(): this;
    /**
     * Grants permission to describe parameters contained within an Amazon Redshift parameter group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterParameters.html
     */
    describeClusterParameters(): this;
    /**
     * Grants permission to describe Amazon Redshift security groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterSecurityGroups.html
     */
    describeClusterSecurityGroups(): this;
    /**
     * Grants permission to describe one or more snapshot objects, which contain metadata about your cluster snapshots
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterSnapshots.html
     */
    describeClusterSnapshots(): this;
    /**
     * Grants permission to describe one or more cluster subnet group objects, which contain metadata about your cluster subnet groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterSubnetGroups.html
     */
    describeClusterSubnetGroups(): this;
    /**
     * Grants permission to describe available maintenance tracks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterTracks.html
     */
    describeClusterTracks(): this;
    /**
     * Grants permission to describe available Amazon Redshift cluster versions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterVersions.html
     */
    describeClusterVersions(): this;
    /**
     * Grants permission to describe properties of provisioned clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusters.html
     */
    describeClusters(): this;
    /**
     * Grants permission to describe parameter settings for a parameter group family
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeDefaultClusterParameters.html
     */
    describeDefaultClusterParameters(): this;
    /**
     * Grants permission to describe event categories for all event source types, or for a specified source type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeEventCategories.html
     */
    describeEventCategories(): this;
    /**
     * Grants permission to describe Amazon Redshift event notification subscriptions for the specified AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeEventSubscriptions.html
     */
    describeEventSubscriptions(): this;
    /**
     * Grants permission to describe events related to clusters, security groups, snapshots, and parameter groups for the past 14 days
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeEvents.html
     */
    describeEvents(): this;
    /**
     * Grants permission to describe HSM client certificates
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeHsmClientCertificates.html
     */
    describeHsmClientCertificates(): this;
    /**
     * Grants permission to describe Amazon Redshift HSM configurations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeHsmConfigurations.html
     */
    describeHsmConfigurations(): this;
    /**
     * Grants permission to describe whether information, such as queries and connection attempts, is being logged for a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeLoggingStatus.html
     */
    describeLoggingStatus(): this;
    /**
     * Grants permission to describe properties of possible node configurations such as node type, number of nodes, and disk usage for the specified action type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeNodeConfigurationOptions.html
     */
    describeNodeConfigurationOptions(): this;
    /**
     * Grants permission to describe orderable cluster options
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeOrderableClusterOptions.html
     */
    describeOrderableClusterOptions(): this;
    /**
     * Grants permission to describe a query through the Amazon Redshift console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    describeQuery(): this;
    /**
     * Grants permission to describe available reserved node offerings by Amazon Redshift
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeReservedNodeOfferings.html
     */
    describeReservedNodeOfferings(): this;
    /**
     * Grants permission to describe the reserved nodes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeReservedNodes.html
     */
    describeReservedNodes(): this;
    /**
     * Grants permission to describe the last resize operation for a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeResize.html
     */
    describeResize(): this;
    /**
     * Grants permission to describe saved queries through the Amazon Redshift console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    describeSavedQueries(): this;
    /**
     * Grants permission to describe created Amazon Redshift scheduled actions
     *
     * Access Level: Read
     */
    describeScheduledActions(): this;
    /**
     * Grants permission to describe snapshot copy grants owned by the specified AWS account in the destination AWS Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeSnapshotCopyGrants.html
     */
    describeSnapshotCopyGrants(): this;
    /**
     * Grants permission to describe snapshot schedules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeSnapshotSchedules.html
     */
    describeSnapshotSchedules(): this;
    /**
     * Grants permission to describe account level backups storage size and provisional storage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeStorage.html
     */
    describeStorage(): this;
    /**
     * Grants permission to describe a table through the Amazon Redshift console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    describeTable(): this;
    /**
     * Grants permission to describe status of one or more table restore requests made using the RestoreTableFromClusterSnapshot API action
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeTableRestoreStatus.html
     */
    describeTableRestoreStatus(): this;
    /**
     * Grants permission to describe tags
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeTags.html
     */
    describeTags(): this;
    /**
     * Grants permission to disable logging information, such as queries and connection attempts, for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DisableLogging.html
     */
    disableLogging(): this;
    /**
     * Grants permission to disable the automatic copy of snapshots for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DisableSnapshotCopy.html
     */
    disableSnapshotCopy(): this;
    /**
     * Grants permission to enable logging information, such as queries and connection attempts, for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_EnableLogging.html
     */
    enableLogging(): this;
    /**
     * Grants permission to enable the automatic copy of snapshots for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_EnableSnapshotCopy.html
     */
    enableSnapshotCopy(): this;
    /**
     * Grants permission to execute a query through the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    executeQuery(): this;
    /**
     * Grants permission to fetch query results through the Amazon Redshift console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    fetchResults(): this;
    /**
     * Grants permission to get temporary credentials to access an Amazon Redshift database by the specified AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDbName()
     * - .ifDbUser()
     * - .ifDurationSeconds()
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_GetClusterCredentials.html
     */
    getClusterCredentials(): this;
    /**
     * Grants permission to get an array of DC2 ReservedNodeOfferings that matches the payment type, term, and usage price of the given DC1 reserved node
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_GetReservedNodeExchangeOfferings.html
     */
    getReservedNodeExchangeOfferings(): this;
    /**
     * Grants permission to join the specified Amazon Redshift group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_GetClusterCredentials.html
     */
    joinGroup(): this;
    /**
     * Grants permission to list databases through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    listDatabases(): this;
    /**
     * Grants permission to list saved queries through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    listSavedQueries(): this;
    /**
     * Grants permission to list schemas through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    listSchemas(): this;
    /**
     * Grants permission to list tables through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    listTables(): this;
    /**
     * Grants permission to modify the settings of a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyCluster.html
     */
    modifyCluster(): this;
    /**
     * Grants permission to modify the database revision of a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterDbRevision.html
     */
    modifyClusterDbRevision(): this;
    /**
     * Grants permission to modify the list of AWS Identity and Access Management (IAM) roles that can be used by a cluster to access other AWS services
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterIamRoles.html
     */
    modifyClusterIamRoles(): this;
    /**
     * Grants permission to modify the maintenance settings of a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterMaintenance.html
     */
    modifyClusterMaintenance(): this;
    /**
     * Grants permission to modify the parameters of a parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterParameterGroup.html
     */
    modifyClusterParameterGroup(): this;
    /**
     * Grants permission to modify the settings of a snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterSnapshot.html
     */
    modifyClusterSnapshot(): this;
    /**
     * Grants permission to modify a snapshot schedule for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterSnapshotSchedule.html
     */
    modifyClusterSnapshotSchedule(): this;
    /**
     * Grants permission to modify a cluster subnet group to include the specified list of VPC subnets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterSubnetGroup.html
     */
    modifyClusterSubnetGroup(): this;
    /**
     * Grants permission to modify an existing Amazon Redshift event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyEventSubscription.html
     */
    modifyEventSubscription(): this;
    /**
     * Grants permission to modify an existing saved query through the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    modifySavedQuery(): this;
    /**
     * Grants permission to modify an existing Amazon Redshift scheduled action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyScheduledAction.html
     */
    modifyScheduledAction(): this;
    /**
     * Grants permission to modify the number of days to retain snapshots in the destination AWS Region after they are copied from the source AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifySnapshotCopyRetentionPeriod.html
     */
    modifySnapshotCopyRetentionPeriod(): this;
    /**
     * Grants permission to modify a snapshot schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifySnapshotSchedule.html
     */
    modifySnapshotSchedule(): this;
    /**
     * Grants permission to pause a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_PauseCluster.html
     */
    pauseCluster(): this;
    /**
     * Grants permission to purchase a reserved node
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_PurchaseReservedNodeOffering.html
     */
    purchaseReservedNodeOffering(): this;
    /**
     * Grants permission to reboot a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RebootCluster.html
     */
    rebootCluster(): this;
    /**
     * Grants permission to set one or more parameters of a parameter group to their default values and set the source values of the parameters to "engine-default"
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ResetClusterParameterGroup.html
     */
    resetClusterParameterGroup(): this;
    /**
     * Grants permission to change the size of a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ResizeCluster.html
     */
    resizeCluster(): this;
    /**
     * Grants permission to create a cluster from a snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RestoreFromClusterSnapshot.html
     */
    restoreFromClusterSnapshot(): this;
    /**
     * Grants permission to create a table from a table in an Amazon Redshift cluster snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RestoreTableFromClusterSnapshot.html
     */
    restoreTableFromClusterSnapshot(): this;
    /**
     * Grants permission to resume a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ResumeCluster.html
     */
    resumeCluster(): this;
    /**
     * Grants permission to revoke an ingress rule in an Amazon Redshift security group for a previously authorized IP range or Amazon EC2 security group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RevokeClusterSecurityGroupIngress.html
     */
    revokeClusterSecurityGroupIngress(): this;
    /**
     * Grants permission to revoke access from the specified AWS account to restore a snapshot
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RevokeSnapshotAccess.html
     */
    revokeSnapshotAccess(): this;
    /**
     * Grants permission to rotate an encryption key for a cluster
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RotateEncryptionKey.html
     */
    rotateEncryptionKey(): this;
    /**
     * Grants permission to view query results through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    viewQueriesFromConsole(): this;
    /**
     * Grants permission to terminate running queries and loads through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    viewQueriesInConsole(): this;
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-clusters.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(clusterName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type dbgroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/dg/r_CREATE_GROUP.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param dbGroup - Identifier for the dbGroup.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDbgroup(clusterName: string, dbGroup: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type dbname to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/dg/t_creating_database.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param dbName - Identifier for the dbName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDbname(clusterName: string, dbName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type dbuser to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/dg/r_Users.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param dbUser - Identifier for the dbUser.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDbuser(clusterName: string, dbUser: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type eventsubscription to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-events.html
     *
     * @param eventSubscriptionName - Identifier for the eventSubscriptionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventsubscription(eventSubscriptionName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type hsmclientcertificate to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-db-encryption.html#working-with-HSM
     *
     * @param hSMClientCertificateId - Identifier for the hSMClientCertificateId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHsmclientcertificate(hSMClientCertificateId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type hsmconfiguration to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-db-encryption.html#working-with-HSM
     *
     * @param hSMConfigurationId - Identifier for the hSMConfigurationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHsmconfiguration(hSMConfigurationId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type parametergroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-parameter-groups.html
     *
     * @param parameterGroupName - Identifier for the parameterGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onParametergroup(parameterGroupName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type securitygroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-security-groups.html
     *
     * @param securityGroupName - Identifier for the securityGroupName.
     * @param owner - Identifier for the owner.
     * @param ec2SecurityGroupId - Identifier for the ec2SecurityGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSecuritygroup(securityGroupName: string, owner: string, ec2SecurityGroupId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type securitygroupingress-cidr to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-security-groups.html
     *
     * @param securityGroupName - Identifier for the securityGroupName.
     * @param ipRange - Identifier for the ipRange.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSecuritygroupingressCidr(securityGroupName: string, ipRange: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type securitygroupingress-ec2securitygroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-security-groups.html
     *
     * @param securityGroupName - Identifier for the securityGroupName.
     * @param owner - Identifier for the owner.
     * @param ece2SecuritygroupId - Identifier for the ece2SecuritygroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSecuritygroupingressEc2securitygroup(securityGroupName: string, owner: string, ece2SecuritygroupId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-snapshots.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param snapshotName - Identifier for the snapshotName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSnapshot(clusterName: string, snapshotName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type snapshotcopygrant to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-db-encryption.html#configure-snapshot-copy-grant
     *
     * @param snapshotCopyGrantName - Identifier for the snapshotCopyGrantName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSnapshotcopygrant(snapshotCopyGrantName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type snapshotschedule to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-snapshots.html
     *
     * @param parameterGroupName - Identifier for the parameterGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSnapshotschedule(parameterGroupName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type subnetgroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-cluster-subnet-groups.html
     *
     * @param subnetGroupName - Identifier for the subnetGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSubnetgroup(subnetGroupName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters actions based on the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-overview.html#redshift-policy-resources.conditions
     *
     * Applies to actions:
     * - .createCluster()
     * - .createClusterParameterGroup()
     * - .createClusterSecurityGroup()
     * - .createClusterSnapshot()
     * - .createClusterSubnetGroup()
     * - .createEventSubscription()
     * - .createHsmClientCertificate()
     * - .createHsmConfiguration()
     * - .createSnapshotCopyGrant()
     * - .createSnapshotSchedule()
     * - .createTags()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: string): this;
    /**
     * Filters actions based on tag-value associated with the resource
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-overview.html#redshift-policy-resources.conditions
     *
     * Applies to resource types:
     * - cluster
     * - dbgroup
     * - dbname
     * - dbuser
     * - eventsubscription
     * - hsmclientcertificate
     * - hsmconfiguration
     * - parametergroup
     * - securitygroup
     * - securitygroupingress-cidr
     * - securitygroupingress-ec2securitygroup
     * - snapshot
     * - snapshotcopygrant
     * - snapshotschedule
     * - subnetgroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: string): this;
    /**
     * Filters actions based on the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-overview.html#redshift-policy-resources.conditions
     *
     * Applies to actions:
     * - .createCluster()
     * - .createClusterParameterGroup()
     * - .createClusterSecurityGroup()
     * - .createClusterSnapshot()
     * - .createClusterSubnetGroup()
     * - .createEventSubscription()
     * - .createHsmClientCertificate()
     * - .createHsmConfiguration()
     * - .createSnapshotCopyGrant()
     * - .createSnapshotSchedule()
     * - .createTags()
     * - .deleteTags()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: string): this;
    /**
     * Filters access by the database name
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-overview.html#redshift-policy-resources.conditions
     *
     * Applies to actions:
     * - .getClusterCredentials()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDbName(value: string | string[], operator?: string): this;
    /**
     * Filters access by the database user name
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-overview.html#redshift-policy-resources.conditions
     *
     * Applies to actions:
     * - .createClusterUser()
     * - .getClusterCredentials()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDbUser(value: string | string[], operator?: string): this;
    /**
     * Filters access by the number of seconds until a temporary credential set expires
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-overview.html#redshift-policy-resources.conditions
     *
     * Applies to actions:
     * - .getClusterCredentials()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDurationSeconds(value: string | string[], operator?: string): this;
}
