import { Actions, PolicyStatement, ResourceTypes } from "../shared";
/**
 * Statement provider for service [s3](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazons3.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class S3 extends PolicyStatement {
    servicePrefix: string;
    protected actionList: Actions;
    protected resourceTypes: ResourceTypes;
    /**
     * Statement provider for service [s3](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazons3.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid?: string);
    /**
     * Grants permission to abort a multipart upload
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_AbortMultipartUpload.html
     */
    abortMultipartUpload(): this;
    /**
     * Grants permission to allow circumvention of governance-mode object retention settings
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzCopySource()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzMetadataDirective()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifXAmzStorageClass()
     * - .ifXAmzWebsiteRedirectLocation()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     * - .ifObjectLockLegalHold()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-managing.html#object-lock-managing-bypass
     */
    bypassGovernanceRetention(): this;
    /**
     * Grants permission to create a new access point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifLocationconstraint()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html
     */
    createAccessPoint(): this;
    /**
     * Grants permission to create a new bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifLocationconstraint()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateBucket.html
     */
    createBucket(): this;
    /**
     * Grants permission to create a new Amazon S3 Batch Operations job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifRequestJobPriority()
     * - .ifRequestJobOperation()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html
     */
    createJob(): this;
    /**
     * Grants permission to delete the access point named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html
     */
    deleteAccessPoint(): this;
    /**
     * Grants permission to delete the policy on a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html
     */
    deleteAccessPointPolicy(): this;
    /**
     * Grants permission to delete the bucket named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucket.html
     */
    deleteBucket(): this;
    /**
     * Grants permission to delete the policy on a specified bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketPolicy.html
     */
    deleteBucketPolicy(): this;
    /**
     * Grants permission to remove the website configuration for a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketWebsite.html
     */
    deleteBucketWebsite(): this;
    /**
     * Grants permission to remove tags from an existing Amazon S3 Batch Operations job
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteJobTagging.html
     */
    deleteJobTagging(): this;
    /**
     * Grants permission to remove the null version of an object and insert a delete marker, which becomes the current version of the object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    deleteObject(): this;
    /**
     * Grants permission to use the tagging subresource to remove the entire tag set from the specified object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    deleteObjectTagging(): this;
    /**
     * Grants permission to remove a specific version of an object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    deleteObjectVersion(): this;
    /**
     * Grants permission to remove the entire tag set for a specific version of the object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    deleteObjectVersionTagging(): this;
    /**
     * Grants permission to retrieve the configuration parameters and status for a batch operations job.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeJob.html
     */
    describeJob(): this;
    /**
     * Grants permission to uses the accelerate subresource to return the Transfer Acceleration state of a bucket, which is either Enabled or Suspended
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAccelerateConfiguration.html
     */
    getAccelerateConfiguration(): this;
    /**
     * Grants permission to return configuration information about the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html
     */
    getAccessPoint(): this;
    /**
     * Grants permission to returns the access point policy associated with the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicy.html
     */
    getAccessPointPolicy(): this;
    /**
     * Grants permission to return the policy status for a specific access point policy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicyStatus.html
     */
    getAccessPointPolicyStatus(): this;
    /**
     * Grants permission to retrieve the PublicAccessBlock configuration for an AWS account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetPublicAccessBlock.html
     */
    getAccountPublicAccessBlock(): this;
    /**
     * Grants permission to get an analytics configuration from an Amazon S3 bucket, identified by the analytics configuration ID
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAnalyticsConfiguration.html
     */
    getAnalyticsConfiguration(): this;
    /**
     * Grants permission to use the acl subresource to return the access control list (ACL) of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAcl.html
     */
    getBucketAcl(): this;
    /**
     * Grants permission to return the CORS configuration information set for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketCors.html
     */
    getBucketCORS(): this;
    /**
     * Grants permission to return the Region that an Amazon S3 bucket resides in
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLocation.html
     */
    getBucketLocation(): this;
    /**
     * Grants permission to return the logging status of an Amazon S3 bucket and the permissions users have to view or modify that status
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLogging.html
     */
    getBucketLogging(): this;
    /**
     * Grants permission to get the notification configuration of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketNotification.html
     */
    getBucketNotification(): this;
    /**
     * Grants permission to get the Object Lock configuration of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLockConfiguration.html
     */
    getBucketObjectLockConfiguration(): this;
    /**
     * Grants permission to return the policy of the specified bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicy.html
     */
    getBucketPolicy(): this;
    /**
     * Grants permission to retrieve the policy status for a specific Amazon S3 bucket, which indicates whether the bucket is public
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicyStatus.html
     */
    getBucketPolicyStatus(): this;
    /**
     * Grants permission to retrieve the PublicAccessBlock configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetPublicAccessBlock.html
     */
    getBucketPublicAccessBlock(): this;
    /**
     * Grants permission to return the request payment configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketRequestPayment.html
     */
    getBucketRequestPayment(): this;
    /**
     * Grants permission to return the tag set associated with an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketTagging.html
     */
    getBucketTagging(): this;
    /**
     * Grants permission to return the versioning state of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketVersioning.html
     */
    getBucketVersioning(): this;
    /**
     * Grants permission to return the website configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketWebsite.html
     */
    getBucketWebsite(): this;
    /**
     * Grants permission to return the default encryption configuration an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketEncryption.html
     */
    getEncryptionConfiguration(): this;
    /**
     * Grants permission to return an inventory configuration from an Amazon S3 bucket, identified by the inventory configuration ID
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketInventoryConfiguration.html
     */
    getInventoryConfiguration(): this;
    /**
     * Grants permission to return the tag set of an existing Amazon S3 Batch Operations job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetJobTagging.html
     */
    getJobTagging(): this;
    /**
     * Grants permission to return the lifecycle configuration information set on an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLifecycleConfiguration.html
     */
    getLifecycleConfiguration(): this;
    /**
     * Grants permission to get a metrics configuration from an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketMetricsConfiguration.html
     */
    getMetricsConfiguration(): this;
    /**
     * Grants permission to retrieve objects from Amazon S3
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    getObject(): this;
    /**
     * Grants permission to return the access control list (ACL) of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html
     */
    getObjectAcl(): this;
    /**
     * Grants permission to get an object's current Legal Hold status
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLegalHold.html
     */
    getObjectLegalHold(): this;
    /**
     * Grants permission to retrieve the retention settings for an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectRetention.html
     */
    getObjectRetention(): this;
    /**
     * Grants permission to return the tag set of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTagging.html
     */
    getObjectTagging(): this;
    /**
     * Grants permission to return torrent files from an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html
     */
    getObjectTorrent(): this;
    /**
     * Grants permission to retrieve a specific version of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    getObjectVersion(): this;
    /**
     * Grants permission to return the access control list (ACL) of a specific object version
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html
     */
    getObjectVersionAcl(): this;
    /**
     * Grants permission to replicate both unencrypted objects and objects encrypted with SSE-S3 or SSE-KMS
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     */
    getObjectVersionForReplication(): this;
    /**
     * Grants permission to return the tag set for a specific version of the object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     */
    getObjectVersionTagging(): this;
    /**
     * Grants permission to get Torrent files about a different version using the versionId subresource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html
     */
    getObjectVersionTorrent(): this;
    /**
     * Grants permission to get the replication configuration information set on an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketReplication.html
     */
    getReplicationConfiguration(): this;
    /**
     * Grants permission to list access points
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPoints.html
     */
    listAccessPoints(): this;
    /**
     * Grants permission to list all buckets owned by the authenticated sender of the request
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListBuckets.html
     */
    listAllMyBuckets(): this;
    /**
     * Grants permission to list some or all of the objects in an Amazon S3 bucket (up to 1000)
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifDelimiter()
     * - .ifMaxKeys()
     * - .ifPrefix()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectsV2.html
     */
    listBucket(): this;
    /**
     * Grants permission to list in-progress multipart uploads
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListMultipartUploads.html
     */
    listBucketMultipartUploads(): this;
    /**
     * Grants permission to list metadata about all the versions of objects in an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifDelimiter()
     * - .ifMaxKeys()
     * - .ifPrefix()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectVersions.html
     */
    listBucketVersions(): this;
    /**
     * Grants permission to list current jobs and jobs that have ended recently
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListJobs.html
     */
    listJobs(): this;
    /**
     * Grants permission to list the parts that have been uploaded for a specific multipart upload
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListParts.html
     */
    listMultipartUploadParts(): this;
    /**
     * Grants permission to change replica ownership
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/replication-change-owner.html#repl-ownership-add-role-permission
     */
    objectOwnerOverrideToBucketOwner(): this;
    /**
     * Grants permission to use the accelerate subresource to set the Transfer Acceleration state of an existing S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAccelerateConfiguration.html
     */
    putAccelerateConfiguration(): this;
    /**
     * Grants permission to associate an access policy with a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html
     */
    putAccessPointPolicy(): this;
    /**
     * Grants permission to create or modify the PublicAccessBlock configuration for an AWS account
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutPublicAccessBlock.html
     */
    putAccountPublicAccessBlock(): this;
    /**
     * Grants permission to set an analytics configuration for the bucket, specified by the analytics configuration ID
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAnalyticsConfiguration.html
     */
    putAnalyticsConfiguration(): this;
    /**
     * Grants permission to set the permissions on an existing bucket using access control lists (ACLs)
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAcl.html
     */
    putBucketAcl(): this;
    /**
     * Grants permission to set the CORS configuration for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketCors.html
     */
    putBucketCORS(): this;
    /**
     * Grants permission to set the logging parameters for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLogging.html
     */
    putBucketLogging(): this;
    /**
     * Grants permission to receive notifications when certain events happen in an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketNotification.html
     */
    putBucketNotification(): this;
    /**
     * Grants permission to put Object Lock configuration on a specific bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLockConfiguration.html
     */
    putBucketObjectLockConfiguration(): this;
    /**
     * Grants permission to add or replace a bucket policy on a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketPolicy.html
     */
    putBucketPolicy(): this;
    /**
     * Grants permission to create or modify the PublicAccessBlock configuration for a specific Amazon S3 bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutPublicAccessBlock.html
     */
    putBucketPublicAccessBlock(): this;
    /**
     * Grants permission to set the request payment configuration of a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketRequestPayment.html
     */
    putBucketRequestPayment(): this;
    /**
     * Grants permission to add a set of tags to an existing Amazon S3 bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketTagging.html
     */
    putBucketTagging(): this;
    /**
     * Grants permission to set the versioning state of an existing Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketVersioning.html
     */
    putBucketVersioning(): this;
    /**
     * Grants permission to set the configuration of the website that is specified in the website subresource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketWebsite.html
     */
    putBucketWebsite(): this;
    /**
     * Grants permission to set the encryption configuration for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketEncryption.html
     */
    putEncryptionConfiguration(): this;
    /**
     * Grants permission to add an inventory configuration to the bucket, identified by the inventory ID
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketInventoryConfiguration.html
     */
    putInventoryConfiguration(): this;
    /**
     * Grants permission to replace tags on an existing Amazon S3 Batch Operations job
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutJobTagging.html
     */
    putJobTagging(): this;
    /**
     * Grants permission to create a new lifecycle configuration for the bucket or replace an existing lifecycle configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLifecycleConfiguration.html
     */
    putLifecycleConfiguration(): this;
    /**
     * Grants permission to set or update a metrics configuration for the CloudWatch request metrics from an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketMetricsConfiguration.html
     */
    putMetricsConfiguration(): this;
    /**
     * Grants permission to add an object to a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzCopySource()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzMetadataDirective()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifXAmzStorageClass()
     * - .ifXAmzWebsiteRedirectLocation()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     * - .ifObjectLockLegalHold()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObject.html
     */
    putObject(): this;
    /**
     * Grants permission to set the access control list (ACL) permissions for an object that already exists in a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzStorageClass()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    putObjectAcl(): this;
    /**
     * Grants permission to apply a Legal Hold configuration to the specified object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifObjectLockLegalHold()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLegalHold.html
     */
    putObjectLegalHold(): this;
    /**
     * Grants permission to place an Object Retention configuration on an object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectRetention.html
     */
    putObjectRetention(): this;
    /**
     * Grants permission to set the supplied tag-set to an object that already exists in a bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    putObjectTagging(): this;
    /**
     * Grants permission to use the acl subresource to set the access control list (ACL) permissions for an object that already exists in a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzStorageClass()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    putObjectVersionAcl(): this;
    /**
     * Grants permission to set the supplied tag-set for a specific version of an object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    putObjectVersionTagging(): this;
    /**
     * Grants permission to create a new replication configuration or replace an existing one
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketReplication.html
     */
    putReplicationConfiguration(): this;
    /**
     * Grants permission to replicate delete markers to the destination bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html
     */
    replicateDelete(): this;
    /**
     * Grants permission to replicate objects and object tags to the destination bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html
     */
    replicateObject(): this;
    /**
     * Grants permission to replicate object tags to the destination bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html
     */
    replicateTags(): this;
    /**
     * Grants permission to restore an archived copy of an object back into Amazon S3
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_RestoreObject.html
     */
    restoreObject(): this;
    /**
     * Grants permission to update the priority of an existing job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifRequestJobPriority()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobPriority.html
     */
    updateJobPriority(): this;
    /**
     * Grants permission to update the status for the specified job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     * - .ifJobSuspendedCause()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobStatus.html
     */
    updateJobStatus(): this;
    /**
     * Adds a resource of type accesspoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/access-points.html
     *
     * @param accessPointName - Identifier for the accessPointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAccesspoint(accessPointName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type bucket to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingBucket.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBucket(bucketName: string, partition?: string): this;
    /**
     * Adds a resource of type object to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingObjects.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param objectName - Identifier for the objectName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onObject(bucketName: string, objectName: string, partition?: string): this;
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-managing-jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJob(jobId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters actions based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .createJob()
     * - .putJobTagging()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: string): this;
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: string): this;
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .createJob()
     * - .putJobTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: string): this;
    /**
     * Filters access by the network origin (Internet or VPC)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .abortMultipartUpload()
     * - .bypassGovernanceRetention()
     * - .createAccessPoint()
     * - .deleteAccessPoint()
     * - .deleteAccessPointPolicy()
     * - .deleteObject()
     * - .deleteObjectTagging()
     * - .deleteObjectVersion()
     * - .deleteObjectVersionTagging()
     * - .getAccessPoint()
     * - .getAccessPointPolicy()
     * - .getAccessPointPolicyStatus()
     * - .getObject()
     * - .getObjectAcl()
     * - .getObjectLegalHold()
     * - .getObjectRetention()
     * - .getObjectTagging()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionTagging()
     * - .listBucket()
     * - .listBucketMultipartUploads()
     * - .listBucketVersions()
     * - .listMultipartUploadParts()
     * - .putAccessPointPolicy()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectLegalHold()
     * - .putObjectRetention()
     * - .putObjectTagging()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     * - .restoreObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessPointNetworkOrigin(value: string | string[], operator?: string): this;
    /**
     * Filters access by the AWS Account ID that owns the access point
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .abortMultipartUpload()
     * - .bypassGovernanceRetention()
     * - .createAccessPoint()
     * - .deleteAccessPoint()
     * - .deleteAccessPointPolicy()
     * - .deleteObject()
     * - .deleteObjectTagging()
     * - .deleteObjectVersion()
     * - .deleteObjectVersionTagging()
     * - .getAccessPoint()
     * - .getAccessPointPolicy()
     * - .getAccessPointPolicyStatus()
     * - .getObject()
     * - .getObjectAcl()
     * - .getObjectLegalHold()
     * - .getObjectRetention()
     * - .getObjectTagging()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionTagging()
     * - .listBucket()
     * - .listBucketMultipartUploads()
     * - .listBucketVersions()
     * - .listMultipartUploadParts()
     * - .putAccessPointPolicy()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectLegalHold()
     * - .putObjectRetention()
     * - .putObjectTagging()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     * - .restoreObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointAccount(value: string | string[], operator?: string): this;
    /**
     * Filters access by an access point Amazon Resource Name (ARN)
     *
     * Applies to actions:
     * - .abortMultipartUpload()
     * - .bypassGovernanceRetention()
     * - .createAccessPoint()
     * - .deleteAccessPoint()
     * - .deleteAccessPointPolicy()
     * - .deleteObject()
     * - .deleteObjectTagging()
     * - .deleteObjectVersion()
     * - .deleteObjectVersionTagging()
     * - .getAccessPoint()
     * - .getAccessPointPolicy()
     * - .getAccessPointPolicyStatus()
     * - .getObject()
     * - .getObjectAcl()
     * - .getObjectLegalHold()
     * - .getObjectRetention()
     * - .getObjectTagging()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionTagging()
     * - .listBucket()
     * - .listBucketMultipartUploads()
     * - .listBucketVersions()
     * - .listMultipartUploadParts()
     * - .putAccessPointPolicy()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectLegalHold()
     * - .putObjectRetention()
     * - .putObjectTagging()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     * - .restoreObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointArn(value: string | string[], operator?: string): this;
    /**
     * Filters access to updating the job priority by operation
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .deleteJobTagging()
     * - .putJobTagging()
     * - .updateJobPriority()
     * - .updateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExistingJobOperation(value: string | string[], operator?: string): this;
    /**
     * Filters access to cancelling existing jobs by priority range
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .deleteJobTagging()
     * - .putJobTagging()
     * - .updateJobPriority()
     * - .updateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifExistingJobPriority(value: number | number[], operator?: string): this;
    /**
     * Requires that an existing object tag has a specific tag key and value.
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .deleteObjectTagging()
     * - .deleteObjectVersionTagging()
     * - .getObject()
     * - .getObjectAcl()
     * - .getObjectTagging()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionTagging()
     * - .putObjectAcl()
     * - .putObjectTagging()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExistingObjectTag(key: string, value: string | string[], operator?: string): this;
    /**
     * Filters access to cancelling suspended jobs by a specific job suspended cause (for example, AWAITING_CONFIRMATION)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .updateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifJobSuspendedCause(value: string | string[], operator?: string): this;
    /**
     * Filters access by a specific Region
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#condition-key-bucket-ops-1
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLocationConstraint(value: string | string[], operator?: string): this;
    /**
     * Filters access to creating jobs by operation
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .createJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestJobOperation(value: string | string[], operator?: string): this;
    /**
     * Filters access to creating new jobs by priority range
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .createJob()
     * - .updateJobPriority()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRequestJobPriority(value: number | number[], operator?: string): this;
    /**
     * Restricts the tag keys and values allowed on objects
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .putObjectTagging()
     * - .putObjectVersionTagging()
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestObjectTag(key: string, value: string | string[], operator?: string): this;
    /**
     * Restricts the tag keys allowed on objects
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .putObjectTagging()
     * - .putObjectVersionTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestObjectTagKeys(value: string | string[], operator?: string): this;
    /**
     * Filters access by a specific object version
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#getobjectversion-limit-access-to-specific-version-3
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionId(value: string | string[], operator?: string): this;
    /**
     * Restricts incoming requests to a specific authentication method
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .abortMultipartUpload()
     * - .bypassGovernanceRetention()
     * - .createAccessPoint()
     * - .createBucket()
     * - .createJob()
     * - .deleteAccessPoint()
     * - .deleteAccessPointPolicy()
     * - .deleteBucket()
     * - .deleteBucketPolicy()
     * - .deleteBucketWebsite()
     * - .deleteJobTagging()
     * - .deleteObject()
     * - .deleteObjectTagging()
     * - .deleteObjectVersion()
     * - .deleteObjectVersionTagging()
     * - .describeJob()
     * - .getAccelerateConfiguration()
     * - .getAccessPoint()
     * - .getAccessPointPolicy()
     * - .getAccessPointPolicyStatus()
     * - .getAccountPublicAccessBlock()
     * - .getAnalyticsConfiguration()
     * - .getBucketAcl()
     * - .getBucketCORS()
     * - .getBucketLogging()
     * - .getBucketNotification()
     * - .getBucketObjectLockConfiguration()
     * - .getBucketPolicy()
     * - .getBucketPolicyStatus()
     * - .getBucketPublicAccessBlock()
     * - .getBucketRequestPayment()
     * - .getBucketTagging()
     * - .getBucketVersioning()
     * - .getBucketWebsite()
     * - .getEncryptionConfiguration()
     * - .getInventoryConfiguration()
     * - .getJobTagging()
     * - .getLifecycleConfiguration()
     * - .getMetricsConfiguration()
     * - .getObject()
     * - .getObjectAcl()
     * - .getObjectLegalHold()
     * - .getObjectRetention()
     * - .getObjectTagging()
     * - .getObjectTorrent()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionForReplication()
     * - .getObjectVersionTagging()
     * - .getObjectVersionTorrent()
     * - .getReplicationConfiguration()
     * - .listAccessPoints()
     * - .listAllMyBuckets()
     * - .listBucket()
     * - .listBucketMultipartUploads()
     * - .listBucketVersions()
     * - .listJobs()
     * - .listMultipartUploadParts()
     * - .objectOwnerOverrideToBucketOwner()
     * - .putAccelerateConfiguration()
     * - .putAccessPointPolicy()
     * - .putAccountPublicAccessBlock()
     * - .putAnalyticsConfiguration()
     * - .putBucketAcl()
     * - .putBucketCORS()
     * - .putBucketLogging()
     * - .putBucketNotification()
     * - .putBucketObjectLockConfiguration()
     * - .putBucketPolicy()
     * - .putBucketPublicAccessBlock()
     * - .putBucketRequestPayment()
     * - .putBucketTagging()
     * - .putBucketVersioning()
     * - .putBucketWebsite()
     * - .putEncryptionConfiguration()
     * - .putInventoryConfiguration()
     * - .putJobTagging()
     * - .putLifecycleConfiguration()
     * - .putMetricsConfiguration()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectLegalHold()
     * - .putObjectRetention()
     * - .putObjectTagging()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     * - .putReplicationConfiguration()
     * - .replicateDelete()
     * - .replicateObject()
     * - .replicateTags()
     * - .restoreObject()
     * - .updateJobPriority()
     * - .updateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthType(value: string | string[], operator?: string): this;
    /**
     * Requires the delimiter parameter
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/walkthrough1.html
     *
     * Applies to actions:
     * - .listBucket()
     * - .listBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDelimiter(value: string | string[], operator?: string): this;
    /**
     * Filters access by a specific Region
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#condition-key-bucket-ops-1
     *
     * Applies to actions:
     * - .createAccessPoint()
     * - .createBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLocationconstraint(value: string | string[], operator?: string): this;
    /**
     * Limits the maximum number of keys returned in a ListBucket request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#example-numeric-condition-operators
     *
     * Applies to actions:
     * - .listBucket()
     * - .listBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaxKeys(value: number | number[], operator?: string): this;
    /**
     * Enables enforcement of the specified object legal hold status
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-legal-holds
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .putObjectLegalHold()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockLegalHold(value: string | string[], operator?: string): this;
    /**
     * Enables enforcement of the specified object retention mode (COMPLIANCE or GOVERNANCE)
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .putObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockMode(value: string | string[], operator?: string): this;
    /**
     * Enables enforcement of an object relative to the remaining retention days
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-managing.html#object-lock-managing-retention-limits
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .putObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockRemainingRetentionDays(value: string | string[], operator?: string): this;
    /**
     * Enables enforcement of a specific retain-until-date
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-retention-periods
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .putObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockRetainUntilDate(value: string | string[], operator?: string): this;
    /**
     * Filters access by key name prefix
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#condition-key-bucket-ops-2
     *
     * Applies to actions:
     * - .listBucket()
     * - .listBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrefix(value: string | string[], operator?: string): this;
    /**
     * Identifies the length of time, in milliseconds, that a signature is valid in an authenticated request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .abortMultipartUpload()
     * - .bypassGovernanceRetention()
     * - .createAccessPoint()
     * - .createBucket()
     * - .createJob()
     * - .deleteAccessPoint()
     * - .deleteAccessPointPolicy()
     * - .deleteBucket()
     * - .deleteBucketPolicy()
     * - .deleteBucketWebsite()
     * - .deleteJobTagging()
     * - .deleteObject()
     * - .deleteObjectTagging()
     * - .deleteObjectVersion()
     * - .deleteObjectVersionTagging()
     * - .describeJob()
     * - .getAccelerateConfiguration()
     * - .getAccessPoint()
     * - .getAccessPointPolicy()
     * - .getAccessPointPolicyStatus()
     * - .getAccountPublicAccessBlock()
     * - .getAnalyticsConfiguration()
     * - .getBucketAcl()
     * - .getBucketCORS()
     * - .getBucketLogging()
     * - .getBucketNotification()
     * - .getBucketObjectLockConfiguration()
     * - .getBucketPolicy()
     * - .getBucketPolicyStatus()
     * - .getBucketPublicAccessBlock()
     * - .getBucketRequestPayment()
     * - .getBucketTagging()
     * - .getBucketVersioning()
     * - .getBucketWebsite()
     * - .getEncryptionConfiguration()
     * - .getInventoryConfiguration()
     * - .getJobTagging()
     * - .getLifecycleConfiguration()
     * - .getMetricsConfiguration()
     * - .getObject()
     * - .getObjectAcl()
     * - .getObjectLegalHold()
     * - .getObjectRetention()
     * - .getObjectTagging()
     * - .getObjectTorrent()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionForReplication()
     * - .getObjectVersionTagging()
     * - .getObjectVersionTorrent()
     * - .getReplicationConfiguration()
     * - .listAccessPoints()
     * - .listAllMyBuckets()
     * - .listBucket()
     * - .listBucketMultipartUploads()
     * - .listBucketVersions()
     * - .listJobs()
     * - .listMultipartUploadParts()
     * - .objectOwnerOverrideToBucketOwner()
     * - .putAccelerateConfiguration()
     * - .putAccessPointPolicy()
     * - .putAccountPublicAccessBlock()
     * - .putAnalyticsConfiguration()
     * - .putBucketAcl()
     * - .putBucketCORS()
     * - .putBucketLogging()
     * - .putBucketNotification()
     * - .putBucketObjectLockConfiguration()
     * - .putBucketPolicy()
     * - .putBucketPublicAccessBlock()
     * - .putBucketRequestPayment()
     * - .putBucketTagging()
     * - .putBucketVersioning()
     * - .putBucketWebsite()
     * - .putEncryptionConfiguration()
     * - .putInventoryConfiguration()
     * - .putJobTagging()
     * - .putLifecycleConfiguration()
     * - .putMetricsConfiguration()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectLegalHold()
     * - .putObjectRetention()
     * - .putObjectTagging()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     * - .putReplicationConfiguration()
     * - .replicateDelete()
     * - .replicateObject()
     * - .replicateTags()
     * - .restoreObject()
     * - .updateJobPriority()
     * - .updateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSignatureAge(value: number | number[], operator?: string): this;
    /**
     * Identifies the version of AWS Signature that is supported for authenticated requests
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .abortMultipartUpload()
     * - .bypassGovernanceRetention()
     * - .createAccessPoint()
     * - .createBucket()
     * - .createJob()
     * - .deleteAccessPoint()
     * - .deleteAccessPointPolicy()
     * - .deleteBucket()
     * - .deleteBucketPolicy()
     * - .deleteBucketWebsite()
     * - .deleteJobTagging()
     * - .deleteObject()
     * - .deleteObjectTagging()
     * - .deleteObjectVersion()
     * - .deleteObjectVersionTagging()
     * - .describeJob()
     * - .getAccelerateConfiguration()
     * - .getAccessPoint()
     * - .getAccessPointPolicy()
     * - .getAccessPointPolicyStatus()
     * - .getAccountPublicAccessBlock()
     * - .getAnalyticsConfiguration()
     * - .getBucketAcl()
     * - .getBucketCORS()
     * - .getBucketLogging()
     * - .getBucketNotification()
     * - .getBucketObjectLockConfiguration()
     * - .getBucketPolicy()
     * - .getBucketPolicyStatus()
     * - .getBucketPublicAccessBlock()
     * - .getBucketRequestPayment()
     * - .getBucketTagging()
     * - .getBucketVersioning()
     * - .getBucketWebsite()
     * - .getEncryptionConfiguration()
     * - .getInventoryConfiguration()
     * - .getJobTagging()
     * - .getLifecycleConfiguration()
     * - .getMetricsConfiguration()
     * - .getObject()
     * - .getObjectAcl()
     * - .getObjectLegalHold()
     * - .getObjectRetention()
     * - .getObjectTagging()
     * - .getObjectTorrent()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionForReplication()
     * - .getObjectVersionTagging()
     * - .getObjectVersionTorrent()
     * - .getReplicationConfiguration()
     * - .listAccessPoints()
     * - .listAllMyBuckets()
     * - .listBucket()
     * - .listBucketMultipartUploads()
     * - .listBucketVersions()
     * - .listJobs()
     * - .listMultipartUploadParts()
     * - .objectOwnerOverrideToBucketOwner()
     * - .putAccelerateConfiguration()
     * - .putAccessPointPolicy()
     * - .putAccountPublicAccessBlock()
     * - .putAnalyticsConfiguration()
     * - .putBucketAcl()
     * - .putBucketCORS()
     * - .putBucketLogging()
     * - .putBucketNotification()
     * - .putBucketObjectLockConfiguration()
     * - .putBucketPolicy()
     * - .putBucketPublicAccessBlock()
     * - .putBucketRequestPayment()
     * - .putBucketTagging()
     * - .putBucketVersioning()
     * - .putBucketWebsite()
     * - .putEncryptionConfiguration()
     * - .putInventoryConfiguration()
     * - .putJobTagging()
     * - .putLifecycleConfiguration()
     * - .putMetricsConfiguration()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectLegalHold()
     * - .putObjectRetention()
     * - .putObjectTagging()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     * - .putReplicationConfiguration()
     * - .replicateDelete()
     * - .replicateObject()
     * - .replicateTags()
     * - .restoreObject()
     * - .updateJobPriority()
     * - .updateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSignatureversion(value: string | string[], operator?: string): this;
    /**
     * Filters access by a specific object version
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#getobjectversion-limit-access-to-specific-version-3
     *
     * Applies to actions:
     * - .deleteObjectVersion()
     * - .deleteObjectVersionTagging()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionTagging()
     * - .getObjectVersionTorrent()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionid(value: string | string[], operator?: string): this;
    /**
     * Requires the x-amz-acl header with a specific canned ACL in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .createAccessPoint()
     * - .createBucket()
     * - .putBucketAcl()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzAcl(value: string | string[], operator?: string): this;
    /**
     * Disallows unsigned content in your bucket
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .abortMultipartUpload()
     * - .bypassGovernanceRetention()
     * - .createAccessPoint()
     * - .createBucket()
     * - .createJob()
     * - .deleteAccessPoint()
     * - .deleteAccessPointPolicy()
     * - .deleteBucket()
     * - .deleteBucketPolicy()
     * - .deleteBucketWebsite()
     * - .deleteJobTagging()
     * - .deleteObject()
     * - .deleteObjectTagging()
     * - .deleteObjectVersion()
     * - .deleteObjectVersionTagging()
     * - .describeJob()
     * - .getAccelerateConfiguration()
     * - .getAccessPoint()
     * - .getAccessPointPolicy()
     * - .getAccessPointPolicyStatus()
     * - .getAccountPublicAccessBlock()
     * - .getAnalyticsConfiguration()
     * - .getBucketAcl()
     * - .getBucketCORS()
     * - .getBucketLogging()
     * - .getBucketNotification()
     * - .getBucketPolicy()
     * - .getBucketPolicyStatus()
     * - .getBucketPublicAccessBlock()
     * - .getBucketRequestPayment()
     * - .getBucketTagging()
     * - .getBucketVersioning()
     * - .getBucketWebsite()
     * - .getEncryptionConfiguration()
     * - .getInventoryConfiguration()
     * - .getJobTagging()
     * - .getLifecycleConfiguration()
     * - .getMetricsConfiguration()
     * - .getObject()
     * - .getObjectAcl()
     * - .getObjectLegalHold()
     * - .getObjectRetention()
     * - .getObjectTagging()
     * - .getObjectTorrent()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionForReplication()
     * - .getObjectVersionTagging()
     * - .getObjectVersionTorrent()
     * - .getReplicationConfiguration()
     * - .listAccessPoints()
     * - .listAllMyBuckets()
     * - .listBucket()
     * - .listBucketMultipartUploads()
     * - .listBucketVersions()
     * - .listJobs()
     * - .listMultipartUploadParts()
     * - .objectOwnerOverrideToBucketOwner()
     * - .putAccelerateConfiguration()
     * - .putAccessPointPolicy()
     * - .putAccountPublicAccessBlock()
     * - .putAnalyticsConfiguration()
     * - .putBucketAcl()
     * - .putBucketCORS()
     * - .putBucketLogging()
     * - .putBucketNotification()
     * - .putBucketPolicy()
     * - .putBucketPublicAccessBlock()
     * - .putBucketRequestPayment()
     * - .putBucketTagging()
     * - .putBucketVersioning()
     * - .putBucketWebsite()
     * - .putEncryptionConfiguration()
     * - .putInventoryConfiguration()
     * - .putJobTagging()
     * - .putLifecycleConfiguration()
     * - .putMetricsConfiguration()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectLegalHold()
     * - .putObjectRetention()
     * - .putObjectTagging()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     * - .putReplicationConfiguration()
     * - .replicateDelete()
     * - .replicateObject()
     * - .replicateTags()
     * - .restoreObject()
     * - .updateJobPriority()
     * - .updateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzContentSha256(value: string | string[], operator?: string): this;
    /**
     * Restricts the copy source to a specific bucket, prefix, or object
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#putobject-limit-copy-source-3
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzCopySource(value: string | string[], operator?: string): this;
    /**
     * Requires the x-amz-grant-full-control (full control) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .createBucket()
     * - .putBucketAcl()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantFullControl(value: string | string[], operator?: string): this;
    /**
     * Requires the x-amz-grant-read (read access) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .createBucket()
     * - .putBucketAcl()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantRead(value: string | string[], operator?: string): this;
    /**
     * Requires the x-amz-grant-read-acp (read permissions for the ACL) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .createBucket()
     * - .putBucketAcl()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantReadAcp(value: string | string[], operator?: string): this;
    /**
     * Requires the x-amz-grant-write (write access) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .createBucket()
     * - .putBucketAcl()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantWrite(value: string | string[], operator?: string): this;
    /**
     * Requires the x-amz-grant-write-acp (write permissions for the ACL) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .createBucket()
     * - .putBucketAcl()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantWriteAcp(value: string | string[], operator?: string): this;
    /**
     * Enables enforcement of object metadata behavior (COPY or REPLACE) when objects are copied
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CopyObject.html
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzMetadataDirective(value: string | string[], operator?: string): this;
    /**
     * Requires server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .replicateObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryption(value: string | string[], operator?: string): this;
    /**
     * Requires a specific AWS KMS customer managed CMK for server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html#require-sse-kms
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .replicateObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryptionAwsKmsKeyId(value: string | string[], operator?: string): this;
    /**
     * Filters access by storage class
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/storage-class-intro.html#sc-howtoset
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzStorageClass(value: string | string[], operator?: string): this;
    /**
     * Filters access by a specific website redirect location for buckets that are configured as static websites
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/how-to-page-redirect.html#page-redirect-using-rest-api
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzWebsiteRedirectLocation(value: string | string[], operator?: string): this;
}
