"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [s3](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazons3.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class S3 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [s3](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazons3.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 's3';
        this.actionList = {
            "AbortMultipartUpload": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_AbortMultipartUpload.html",
                "description": "Grants permission to abort a multipart upload",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointArn",
                    "s3:DataAccessPointAccount",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "BypassGovernanceRetention": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-managing.html#object-lock-managing-bypass",
                "description": "Grants permission to allow circumvention of governance-mode object retention settings",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:RequestObjectTag/<key>",
                    "s3:RequestObjectTagKeys",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-copy-source",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp",
                    "s3:x-amz-metadata-directive",
                    "s3:x-amz-server-side-encryption",
                    "s3:x-amz-server-side-encryption-aws-kms-key-id",
                    "s3:x-amz-storage-class",
                    "s3:x-amz-website-redirect-location",
                    "s3:object-lock-mode",
                    "s3:object-lock-retain-until-date",
                    "s3:object-lock-remaining-retention-days",
                    "s3:object-lock-legal-hold"
                ]
            },
            "CreateAccessPoint": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html",
                "description": "Grants permission to create a new access point",
                "accessLevel": "Write",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:locationconstraint",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256"
                ]
            },
            "CreateBucket": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateBucket.html",
                "description": "Grants permission to create a new bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:locationconstraint",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp"
                ]
            },
            "CreateJob": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html",
                "description": "Grants permission to create a new Amazon S3 Batch Operations job",
                "accessLevel": "Write",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:RequestJobPriority",
                    "s3:RequestJobOperation",
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "DeleteAccessPoint": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html",
                "description": "Grants permission to delete the access point named in the URI",
                "accessLevel": "Write",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointArn",
                    "s3:DataAccessPointAccount",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteAccessPointPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html",
                "description": "Grants permission to delete the policy on a specified access point",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointArn",
                    "s3:DataAccessPointAccount",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteBucket": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucket.html",
                "description": "Grants permission to delete the bucket named in the URI",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteBucketPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketPolicy.html",
                "description": "Grants permission to delete the policy on a specified bucket",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteBucketWebsite": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketWebsite.html",
                "description": "Grants permission to remove the website configuration for a bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteJobTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteJobTagging.html",
                "description": "Grants permission to remove tags from an existing Amazon S3 Batch Operations job",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:ExistingJobPriority",
                    "s3:ExistingJobOperation"
                ]
            },
            "DeleteObject": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html",
                "description": "Grants permission to remove the null version of an object and insert a delete marker, which becomes the current version of the object",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteObjectTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html",
                "description": "Grants permission to use the tagging subresource to remove the entire tag set from the specified object",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteObjectVersion": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html",
                "description": "Grants permission to remove a specific version of an object",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DeleteObjectVersionTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html",
                "description": "Grants permission to remove the entire tag set for a specific version of the object",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "DescribeJob": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeJob.html",
                "description": "Grants permission to retrieve the configuration parameters and status for a batch operations job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAccelerateConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAccelerateConfiguration.html",
                "description": "Grants permission to uses the accelerate subresource to return the Transfer Acceleration state of a bucket, which is either Enabled or Suspended",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAccessPoint": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html",
                "description": "Grants permission to return configuration information about the specified access point",
                "accessLevel": "Read",
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAccessPointPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicy.html",
                "description": "Grants permission to returns the access point policy associated with the specified access point",
                "accessLevel": "Read",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAccessPointPolicyStatus": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicyStatus.html",
                "description": "Grants permission to return the policy status for a specific access point policy",
                "accessLevel": "Read",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAccountPublicAccessBlock": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetPublicAccessBlock.html",
                "description": "Grants permission to retrieve the PublicAccessBlock configuration for an AWS account",
                "accessLevel": "Read",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetAnalyticsConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAnalyticsConfiguration.html",
                "description": "Grants permission to get an analytics configuration from an Amazon S3 bucket, identified by the analytics configuration ID",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAcl.html",
                "description": "Grants permission to use the acl subresource to return the access control list (ACL) of an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketCORS": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketCors.html",
                "description": "Grants permission to return the CORS configuration information set for an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketLocation": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLocation.html",
                "description": "Grants permission to return the Region that an Amazon S3 bucket resides in",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                }
            },
            "GetBucketLogging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLogging.html",
                "description": "Grants permission to return the logging status of an Amazon S3 bucket and the permissions users have to view or modify that status",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketNotification": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketNotification.html",
                "description": "Grants permission to get the notification configuration of an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketObjectLockConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLockConfiguration.html",
                "description": "Grants permission to get the Object Lock configuration of an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion"
                ]
            },
            "GetBucketPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicy.html",
                "description": "Grants permission to return the policy of the specified bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketPolicyStatus": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicyStatus.html",
                "description": "Grants permission to retrieve the policy status for a specific Amazon S3 bucket, which indicates whether the bucket is public",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketPublicAccessBlock": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetPublicAccessBlock.html",
                "description": "Grants permission to retrieve the PublicAccessBlock configuration for an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketRequestPayment": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketRequestPayment.html",
                "description": "Grants permission to return the request payment configuration for an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketTagging.html",
                "description": "Grants permission to return the tag set associated with an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketVersioning": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketVersioning.html",
                "description": "Grants permission to return the versioning state of an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetBucketWebsite": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketWebsite.html",
                "description": "Grants permission to return the website configuration for an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetEncryptionConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketEncryption.html",
                "description": "Grants permission to return the default encryption configuration an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetInventoryConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketInventoryConfiguration.html",
                "description": "Grants permission to return an inventory configuration from an Amazon S3 bucket, identified by the inventory configuration ID",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetJobTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetJobTagging.html",
                "description": "Grants permission to return the tag set of an existing Amazon S3 Batch Operations job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetLifecycleConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLifecycleConfiguration.html",
                "description": "Grants permission to return the lifecycle configuration information set on an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetMetricsConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketMetricsConfiguration.html",
                "description": "Grants permission to get a metrics configuration from an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObject": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html",
                "description": "Grants permission to retrieve objects from Amazon S3",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html",
                "description": "Grants permission to return the access control list (ACL) of an object",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectLegalHold": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLegalHold.html",
                "description": "Grants permission to get an object's current Legal Hold status",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectRetention": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectRetention.html",
                "description": "Grants permission to retrieve the retention settings for an object",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTagging.html",
                "description": "Grants permission to return the tag set of an object",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectTorrent": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html",
                "description": "Grants permission to return torrent files from an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectVersion": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html",
                "description": "Grants permission to retrieve a specific version of an object",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectVersionAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html",
                "description": "Grants permission to return the access control list (ACL) of a specific object version",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectVersionForReplication": {
                "url": "",
                "description": "Grants permission to replicate both unencrypted objects and objects encrypted with SSE-S3 or SSE-KMS",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectVersionTagging": {
                "url": "",
                "description": "Grants permission to return the tag set for a specific version of the object",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetObjectVersionTorrent": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html",
                "description": "Grants permission to get Torrent files about a different version using the versionId subresource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "GetReplicationConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketReplication.html",
                "description": "Grants permission to get the replication configuration information set on an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListAccessPoints": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPoints.html",
                "description": "Grants permission to list access points",
                "accessLevel": "Read",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListAllMyBuckets": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListBuckets.html",
                "description": "Grants permission to list all buckets owned by the authenticated sender of the request",
                "accessLevel": "List",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListBucket": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectsV2.html",
                "description": "Grants permission to list some or all of the objects in an Amazon S3 bucket (up to 1000)",
                "accessLevel": "List",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:delimiter",
                    "s3:max-keys",
                    "s3:prefix",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListBucketMultipartUploads": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListMultipartUploads.html",
                "description": "Grants permission to list in-progress multipart uploads",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListBucketVersions": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectVersions.html",
                "description": "Grants permission to list metadata about all the versions of objects in an Amazon S3 bucket",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:delimiter",
                    "s3:max-keys",
                    "s3:prefix",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListJobs.html",
                "description": "Grants permission to list current jobs and jobs that have ended recently",
                "accessLevel": "Read",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ListMultipartUploadParts": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListParts.html",
                "description": "Grants permission to list the parts that have been uploaded for a specific multipart upload",
                "accessLevel": "Read",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ObjectOwnerOverrideToBucketOwner": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/replication-change-owner.html#repl-ownership-add-role-permission",
                "description": "Grants permission to change replica ownership",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutAccelerateConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAccelerateConfiguration.html",
                "description": "Grants permission to use the accelerate subresource to set the Transfer Acceleration state of an existing S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutAccessPointPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html",
                "description": "Grants permission to associate an access policy with a specified access point",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "accesspoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutAccountPublicAccessBlock": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutPublicAccessBlock.html",
                "description": "Grants permission to create or modify the PublicAccessBlock configuration for an AWS account",
                "accessLevel": "Permissions management",
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutAnalyticsConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAnalyticsConfiguration.html",
                "description": "Grants permission to set an analytics configuration for the bucket, specified by the analytics configuration ID",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAcl.html",
                "description": "Grants permission to set the permissions on an existing bucket using access control lists (ACLs)",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp"
                ]
            },
            "PutBucketCORS": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketCors.html",
                "description": "Grants permission to set the CORS configuration for an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketLogging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLogging.html",
                "description": "Grants permission to set the logging parameters for an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketNotification": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketNotification.html",
                "description": "Grants permission to receive notifications when certain events happen in an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketObjectLockConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLockConfiguration.html",
                "description": "Grants permission to put Object Lock configuration on a specific bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion"
                ]
            },
            "PutBucketPolicy": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketPolicy.html",
                "description": "Grants permission to add or replace a bucket policy on a bucket",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketPublicAccessBlock": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutPublicAccessBlock.html",
                "description": "Grants permission to create or modify the PublicAccessBlock configuration for a specific Amazon S3 bucket",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketRequestPayment": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketRequestPayment.html",
                "description": "Grants permission to set the request payment configuration of a bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketTagging.html",
                "description": "Grants permission to add a set of tags to an existing Amazon S3 bucket",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketVersioning": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketVersioning.html",
                "description": "Grants permission to set the versioning state of an existing Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutBucketWebsite": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketWebsite.html",
                "description": "Grants permission to set the configuration of the website that is specified in the website subresource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutEncryptionConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketEncryption.html",
                "description": "Grants permission to set the encryption configuration for an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutInventoryConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketInventoryConfiguration.html",
                "description": "Grants permission to add an inventory configuration to the bucket, identified by the inventory ID",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutJobTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutJobTagging.html",
                "description": "Grants permission to replace tags on an existing Amazon S3 Batch Operations job",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:ExistingJobPriority",
                    "s3:ExistingJobOperation",
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "PutLifecycleConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLifecycleConfiguration.html",
                "description": "Grants permission to create a new lifecycle configuration for the bucket or replace an existing lifecycle configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutMetricsConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketMetricsConfiguration.html",
                "description": "Grants permission to set or update a metrics configuration for the CloudWatch request metrics from an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutObject": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObject.html",
                "description": "Grants permission to add an object to a bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:RequestObjectTag/<key>",
                    "s3:RequestObjectTagKeys",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-copy-source",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp",
                    "s3:x-amz-metadata-directive",
                    "s3:x-amz-server-side-encryption",
                    "s3:x-amz-server-side-encryption-aws-kms-key-id",
                    "s3:x-amz-storage-class",
                    "s3:x-amz-website-redirect-location",
                    "s3:object-lock-mode",
                    "s3:object-lock-retain-until-date",
                    "s3:object-lock-remaining-retention-days",
                    "s3:object-lock-legal-hold"
                ]
            },
            "PutObjectAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html",
                "description": "Grants permission to set the access control list (ACL) permissions for an object that already exists in a bucket",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp",
                    "s3:x-amz-storage-class"
                ]
            },
            "PutObjectLegalHold": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLegalHold.html",
                "description": "Grants permission to apply a Legal Hold configuration to the specified object",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:object-lock-legal-hold"
                ]
            },
            "PutObjectRetention": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectRetention.html",
                "description": "Grants permission to place an Object Retention configuration on an object",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:object-lock-mode",
                    "s3:object-lock-retain-until-date",
                    "s3:object-lock-remaining-retention-days"
                ]
            },
            "PutObjectTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html",
                "description": "Grants permission to set the supplied tag-set to an object that already exists in a bucket",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:RequestObjectTag/<key>",
                    "s3:RequestObjectTagKeys",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutObjectVersionAcl": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html",
                "description": "Grants permission to use the acl subresource to set the access control list (ACL) permissions for an object that already exists in a bucket",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-acl",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-grant-full-control",
                    "s3:x-amz-grant-read",
                    "s3:x-amz-grant-read-acp",
                    "s3:x-amz-grant-write",
                    "s3:x-amz-grant-write-acp",
                    "s3:x-amz-storage-class"
                ]
            },
            "PutObjectVersionTagging": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html",
                "description": "Grants permission to set the supplied tag-set for a specific version of an object",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:ExistingObjectTag/<key>",
                    "s3:RequestObjectTag/<key>",
                    "s3:RequestObjectTagKeys",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:versionid",
                    "s3:x-amz-content-sha256"
                ]
            },
            "PutReplicationConfiguration": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketReplication.html",
                "description": "Grants permission to create a new replication configuration or replace an existing one",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bucket": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ReplicateDelete": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html",
                "description": "Grants permission to replicate delete markers to the destination bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "ReplicateObject": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html",
                "description": "Grants permission to replicate objects and object tags to the destination bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:x-amz-server-side-encryption",
                    "s3:x-amz-server-side-encryption-aws-kms-key-id"
                ]
            },
            "ReplicateTags": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html",
                "description": "Grants permission to replicate object tags to the destination bucket",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "RestoreObject": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_RestoreObject.html",
                "description": "Grants permission to restore an archived copy of an object back into Amazon S3",
                "accessLevel": "Write",
                "resourceTypes": {
                    "object": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:DataAccessPointAccount",
                    "s3:DataAccessPointArn",
                    "s3:AccessPointNetworkOrigin",
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256"
                ]
            },
            "UpdateJobPriority": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobPriority.html",
                "description": "Grants permission to update the priority of an existing job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:RequestJobPriority",
                    "s3:ExistingJobPriority",
                    "s3:ExistingJobOperation"
                ]
            },
            "UpdateJobStatus": {
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobStatus.html",
                "description": "Grants permission to update the status for the specified job",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                },
                "conditions": [
                    "s3:authType",
                    "s3:signatureAge",
                    "s3:signatureversion",
                    "s3:x-amz-content-sha256",
                    "s3:ExistingJobPriority",
                    "s3:ExistingJobOperation",
                    "s3:JobSuspendedCause"
                ]
            }
        };
        this.resourceTypes = {
            "accesspoint": {
                "name": "accesspoint",
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/access-points.html",
                "arn": "arn:${Partition}:s3:${Region}:${Account}:accesspoint/${AccessPointName}",
                "conditionKeys": []
            },
            "bucket": {
                "name": "bucket",
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingBucket.html",
                "arn": "arn:${Partition}:s3:::${BucketName}",
                "conditionKeys": []
            },
            "object": {
                "name": "object",
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingObjects.html",
                "arn": "arn:${Partition}:s3:::${BucketName}/${ObjectName}",
                "conditionKeys": []
            },
            "job": {
                "name": "job",
                "url": "https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-managing-jobs.html",
                "arn": "arn:${Partition}:s3:${Region}:${Account}:job/${JobId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to abort a multipart upload
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_AbortMultipartUpload.html
     */
    abortMultipartUpload() {
        this.add('s3:AbortMultipartUpload');
        return this;
    }
    /**
     * Grants permission to allow circumvention of governance-mode object retention settings
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzCopySource()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzMetadataDirective()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifXAmzStorageClass()
     * - .ifXAmzWebsiteRedirectLocation()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     * - .ifObjectLockLegalHold()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-managing.html#object-lock-managing-bypass
     */
    bypassGovernanceRetention() {
        this.add('s3:BypassGovernanceRetention');
        return this;
    }
    /**
     * Grants permission to create a new access point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifLocationconstraint()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html
     */
    createAccessPoint() {
        this.add('s3:CreateAccessPoint');
        return this;
    }
    /**
     * Grants permission to create a new bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifLocationconstraint()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateBucket.html
     */
    createBucket() {
        this.add('s3:CreateBucket');
        return this;
    }
    /**
     * Grants permission to create a new Amazon S3 Batch Operations job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifRequestJobPriority()
     * - .ifRequestJobOperation()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html
     */
    createJob() {
        this.add('s3:CreateJob');
        return this;
    }
    /**
     * Grants permission to delete the access point named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html
     */
    deleteAccessPoint() {
        this.add('s3:DeleteAccessPoint');
        return this;
    }
    /**
     * Grants permission to delete the policy on a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html
     */
    deleteAccessPointPolicy() {
        this.add('s3:DeleteAccessPointPolicy');
        return this;
    }
    /**
     * Grants permission to delete the bucket named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucket.html
     */
    deleteBucket() {
        this.add('s3:DeleteBucket');
        return this;
    }
    /**
     * Grants permission to delete the policy on a specified bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketPolicy.html
     */
    deleteBucketPolicy() {
        this.add('s3:DeleteBucketPolicy');
        return this;
    }
    /**
     * Grants permission to remove the website configuration for a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketWebsite.html
     */
    deleteBucketWebsite() {
        this.add('s3:DeleteBucketWebsite');
        return this;
    }
    /**
     * Grants permission to remove tags from an existing Amazon S3 Batch Operations job
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteJobTagging.html
     */
    deleteJobTagging() {
        this.add('s3:DeleteJobTagging');
        return this;
    }
    /**
     * Grants permission to remove the null version of an object and insert a delete marker, which becomes the current version of the object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    deleteObject() {
        this.add('s3:DeleteObject');
        return this;
    }
    /**
     * Grants permission to use the tagging subresource to remove the entire tag set from the specified object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    deleteObjectTagging() {
        this.add('s3:DeleteObjectTagging');
        return this;
    }
    /**
     * Grants permission to remove a specific version of an object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    deleteObjectVersion() {
        this.add('s3:DeleteObjectVersion');
        return this;
    }
    /**
     * Grants permission to remove the entire tag set for a specific version of the object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    deleteObjectVersionTagging() {
        this.add('s3:DeleteObjectVersionTagging');
        return this;
    }
    /**
     * Grants permission to retrieve the configuration parameters and status for a batch operations job.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeJob.html
     */
    describeJob() {
        this.add('s3:DescribeJob');
        return this;
    }
    /**
     * Grants permission to uses the accelerate subresource to return the Transfer Acceleration state of a bucket, which is either Enabled or Suspended
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAccelerateConfiguration.html
     */
    getAccelerateConfiguration() {
        this.add('s3:GetAccelerateConfiguration');
        return this;
    }
    /**
     * Grants permission to return configuration information about the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html
     */
    getAccessPoint() {
        this.add('s3:GetAccessPoint');
        return this;
    }
    /**
     * Grants permission to returns the access point policy associated with the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicy.html
     */
    getAccessPointPolicy() {
        this.add('s3:GetAccessPointPolicy');
        return this;
    }
    /**
     * Grants permission to return the policy status for a specific access point policy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicyStatus.html
     */
    getAccessPointPolicyStatus() {
        this.add('s3:GetAccessPointPolicyStatus');
        return this;
    }
    /**
     * Grants permission to retrieve the PublicAccessBlock configuration for an AWS account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetPublicAccessBlock.html
     */
    getAccountPublicAccessBlock() {
        this.add('s3:GetAccountPublicAccessBlock');
        return this;
    }
    /**
     * Grants permission to get an analytics configuration from an Amazon S3 bucket, identified by the analytics configuration ID
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAnalyticsConfiguration.html
     */
    getAnalyticsConfiguration() {
        this.add('s3:GetAnalyticsConfiguration');
        return this;
    }
    /**
     * Grants permission to use the acl subresource to return the access control list (ACL) of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAcl.html
     */
    getBucketAcl() {
        this.add('s3:GetBucketAcl');
        return this;
    }
    /**
     * Grants permission to return the CORS configuration information set for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketCors.html
     */
    getBucketCORS() {
        this.add('s3:GetBucketCORS');
        return this;
    }
    /**
     * Grants permission to return the Region that an Amazon S3 bucket resides in
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLocation.html
     */
    getBucketLocation() {
        this.add('s3:GetBucketLocation');
        return this;
    }
    /**
     * Grants permission to return the logging status of an Amazon S3 bucket and the permissions users have to view or modify that status
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLogging.html
     */
    getBucketLogging() {
        this.add('s3:GetBucketLogging');
        return this;
    }
    /**
     * Grants permission to get the notification configuration of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketNotification.html
     */
    getBucketNotification() {
        this.add('s3:GetBucketNotification');
        return this;
    }
    /**
     * Grants permission to get the Object Lock configuration of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLockConfiguration.html
     */
    getBucketObjectLockConfiguration() {
        this.add('s3:GetBucketObjectLockConfiguration');
        return this;
    }
    /**
     * Grants permission to return the policy of the specified bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicy.html
     */
    getBucketPolicy() {
        this.add('s3:GetBucketPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the policy status for a specific Amazon S3 bucket, which indicates whether the bucket is public
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicyStatus.html
     */
    getBucketPolicyStatus() {
        this.add('s3:GetBucketPolicyStatus');
        return this;
    }
    /**
     * Grants permission to retrieve the PublicAccessBlock configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetPublicAccessBlock.html
     */
    getBucketPublicAccessBlock() {
        this.add('s3:GetBucketPublicAccessBlock');
        return this;
    }
    /**
     * Grants permission to return the request payment configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketRequestPayment.html
     */
    getBucketRequestPayment() {
        this.add('s3:GetBucketRequestPayment');
        return this;
    }
    /**
     * Grants permission to return the tag set associated with an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketTagging.html
     */
    getBucketTagging() {
        this.add('s3:GetBucketTagging');
        return this;
    }
    /**
     * Grants permission to return the versioning state of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketVersioning.html
     */
    getBucketVersioning() {
        this.add('s3:GetBucketVersioning');
        return this;
    }
    /**
     * Grants permission to return the website configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketWebsite.html
     */
    getBucketWebsite() {
        this.add('s3:GetBucketWebsite');
        return this;
    }
    /**
     * Grants permission to return the default encryption configuration an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketEncryption.html
     */
    getEncryptionConfiguration() {
        this.add('s3:GetEncryptionConfiguration');
        return this;
    }
    /**
     * Grants permission to return an inventory configuration from an Amazon S3 bucket, identified by the inventory configuration ID
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketInventoryConfiguration.html
     */
    getInventoryConfiguration() {
        this.add('s3:GetInventoryConfiguration');
        return this;
    }
    /**
     * Grants permission to return the tag set of an existing Amazon S3 Batch Operations job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetJobTagging.html
     */
    getJobTagging() {
        this.add('s3:GetJobTagging');
        return this;
    }
    /**
     * Grants permission to return the lifecycle configuration information set on an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLifecycleConfiguration.html
     */
    getLifecycleConfiguration() {
        this.add('s3:GetLifecycleConfiguration');
        return this;
    }
    /**
     * Grants permission to get a metrics configuration from an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketMetricsConfiguration.html
     */
    getMetricsConfiguration() {
        this.add('s3:GetMetricsConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve objects from Amazon S3
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    getObject() {
        this.add('s3:GetObject');
        return this;
    }
    /**
     * Grants permission to return the access control list (ACL) of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html
     */
    getObjectAcl() {
        this.add('s3:GetObjectAcl');
        return this;
    }
    /**
     * Grants permission to get an object's current Legal Hold status
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLegalHold.html
     */
    getObjectLegalHold() {
        this.add('s3:GetObjectLegalHold');
        return this;
    }
    /**
     * Grants permission to retrieve the retention settings for an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectRetention.html
     */
    getObjectRetention() {
        this.add('s3:GetObjectRetention');
        return this;
    }
    /**
     * Grants permission to return the tag set of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTagging.html
     */
    getObjectTagging() {
        this.add('s3:GetObjectTagging');
        return this;
    }
    /**
     * Grants permission to return torrent files from an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html
     */
    getObjectTorrent() {
        this.add('s3:GetObjectTorrent');
        return this;
    }
    /**
     * Grants permission to retrieve a specific version of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    getObjectVersion() {
        this.add('s3:GetObjectVersion');
        return this;
    }
    /**
     * Grants permission to return the access control list (ACL) of a specific object version
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html
     */
    getObjectVersionAcl() {
        this.add('s3:GetObjectVersionAcl');
        return this;
    }
    /**
     * Grants permission to replicate both unencrypted objects and objects encrypted with SSE-S3 or SSE-KMS
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     */
    getObjectVersionForReplication() {
        this.add('s3:GetObjectVersionForReplication');
        return this;
    }
    /**
     * Grants permission to return the tag set for a specific version of the object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     */
    getObjectVersionTagging() {
        this.add('s3:GetObjectVersionTagging');
        return this;
    }
    /**
     * Grants permission to get Torrent files about a different version using the versionId subresource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html
     */
    getObjectVersionTorrent() {
        this.add('s3:GetObjectVersionTorrent');
        return this;
    }
    /**
     * Grants permission to get the replication configuration information set on an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketReplication.html
     */
    getReplicationConfiguration() {
        this.add('s3:GetReplicationConfiguration');
        return this;
    }
    /**
     * Grants permission to list access points
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPoints.html
     */
    listAccessPoints() {
        this.add('s3:ListAccessPoints');
        return this;
    }
    /**
     * Grants permission to list all buckets owned by the authenticated sender of the request
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListBuckets.html
     */
    listAllMyBuckets() {
        this.add('s3:ListAllMyBuckets');
        return this;
    }
    /**
     * Grants permission to list some or all of the objects in an Amazon S3 bucket (up to 1000)
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifDelimiter()
     * - .ifMaxKeys()
     * - .ifPrefix()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectsV2.html
     */
    listBucket() {
        this.add('s3:ListBucket');
        return this;
    }
    /**
     * Grants permission to list in-progress multipart uploads
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListMultipartUploads.html
     */
    listBucketMultipartUploads() {
        this.add('s3:ListBucketMultipartUploads');
        return this;
    }
    /**
     * Grants permission to list metadata about all the versions of objects in an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifDelimiter()
     * - .ifMaxKeys()
     * - .ifPrefix()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectVersions.html
     */
    listBucketVersions() {
        this.add('s3:ListBucketVersions');
        return this;
    }
    /**
     * Grants permission to list current jobs and jobs that have ended recently
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListJobs.html
     */
    listJobs() {
        this.add('s3:ListJobs');
        return this;
    }
    /**
     * Grants permission to list the parts that have been uploaded for a specific multipart upload
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListParts.html
     */
    listMultipartUploadParts() {
        this.add('s3:ListMultipartUploadParts');
        return this;
    }
    /**
     * Grants permission to change replica ownership
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/replication-change-owner.html#repl-ownership-add-role-permission
     */
    objectOwnerOverrideToBucketOwner() {
        this.add('s3:ObjectOwnerOverrideToBucketOwner');
        return this;
    }
    /**
     * Grants permission to use the accelerate subresource to set the Transfer Acceleration state of an existing S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAccelerateConfiguration.html
     */
    putAccelerateConfiguration() {
        this.add('s3:PutAccelerateConfiguration');
        return this;
    }
    /**
     * Grants permission to associate an access policy with a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html
     */
    putAccessPointPolicy() {
        this.add('s3:PutAccessPointPolicy');
        return this;
    }
    /**
     * Grants permission to create or modify the PublicAccessBlock configuration for an AWS account
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutPublicAccessBlock.html
     */
    putAccountPublicAccessBlock() {
        this.add('s3:PutAccountPublicAccessBlock');
        return this;
    }
    /**
     * Grants permission to set an analytics configuration for the bucket, specified by the analytics configuration ID
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAnalyticsConfiguration.html
     */
    putAnalyticsConfiguration() {
        this.add('s3:PutAnalyticsConfiguration');
        return this;
    }
    /**
     * Grants permission to set the permissions on an existing bucket using access control lists (ACLs)
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAcl.html
     */
    putBucketAcl() {
        this.add('s3:PutBucketAcl');
        return this;
    }
    /**
     * Grants permission to set the CORS configuration for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketCors.html
     */
    putBucketCORS() {
        this.add('s3:PutBucketCORS');
        return this;
    }
    /**
     * Grants permission to set the logging parameters for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLogging.html
     */
    putBucketLogging() {
        this.add('s3:PutBucketLogging');
        return this;
    }
    /**
     * Grants permission to receive notifications when certain events happen in an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketNotification.html
     */
    putBucketNotification() {
        this.add('s3:PutBucketNotification');
        return this;
    }
    /**
     * Grants permission to put Object Lock configuration on a specific bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLockConfiguration.html
     */
    putBucketObjectLockConfiguration() {
        this.add('s3:PutBucketObjectLockConfiguration');
        return this;
    }
    /**
     * Grants permission to add or replace a bucket policy on a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketPolicy.html
     */
    putBucketPolicy() {
        this.add('s3:PutBucketPolicy');
        return this;
    }
    /**
     * Grants permission to create or modify the PublicAccessBlock configuration for a specific Amazon S3 bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutPublicAccessBlock.html
     */
    putBucketPublicAccessBlock() {
        this.add('s3:PutBucketPublicAccessBlock');
        return this;
    }
    /**
     * Grants permission to set the request payment configuration of a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketRequestPayment.html
     */
    putBucketRequestPayment() {
        this.add('s3:PutBucketRequestPayment');
        return this;
    }
    /**
     * Grants permission to add a set of tags to an existing Amazon S3 bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketTagging.html
     */
    putBucketTagging() {
        this.add('s3:PutBucketTagging');
        return this;
    }
    /**
     * Grants permission to set the versioning state of an existing Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketVersioning.html
     */
    putBucketVersioning() {
        this.add('s3:PutBucketVersioning');
        return this;
    }
    /**
     * Grants permission to set the configuration of the website that is specified in the website subresource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketWebsite.html
     */
    putBucketWebsite() {
        this.add('s3:PutBucketWebsite');
        return this;
    }
    /**
     * Grants permission to set the encryption configuration for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketEncryption.html
     */
    putEncryptionConfiguration() {
        this.add('s3:PutEncryptionConfiguration');
        return this;
    }
    /**
     * Grants permission to add an inventory configuration to the bucket, identified by the inventory ID
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketInventoryConfiguration.html
     */
    putInventoryConfiguration() {
        this.add('s3:PutInventoryConfiguration');
        return this;
    }
    /**
     * Grants permission to replace tags on an existing Amazon S3 Batch Operations job
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutJobTagging.html
     */
    putJobTagging() {
        this.add('s3:PutJobTagging');
        return this;
    }
    /**
     * Grants permission to create a new lifecycle configuration for the bucket or replace an existing lifecycle configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLifecycleConfiguration.html
     */
    putLifecycleConfiguration() {
        this.add('s3:PutLifecycleConfiguration');
        return this;
    }
    /**
     * Grants permission to set or update a metrics configuration for the CloudWatch request metrics from an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketMetricsConfiguration.html
     */
    putMetricsConfiguration() {
        this.add('s3:PutMetricsConfiguration');
        return this;
    }
    /**
     * Grants permission to add an object to a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzCopySource()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzMetadataDirective()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifXAmzStorageClass()
     * - .ifXAmzWebsiteRedirectLocation()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     * - .ifObjectLockLegalHold()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObject.html
     */
    putObject() {
        this.add('s3:PutObject');
        return this;
    }
    /**
     * Grants permission to set the access control list (ACL) permissions for an object that already exists in a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzStorageClass()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    putObjectAcl() {
        this.add('s3:PutObjectAcl');
        return this;
    }
    /**
     * Grants permission to apply a Legal Hold configuration to the specified object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifObjectLockLegalHold()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLegalHold.html
     */
    putObjectLegalHold() {
        this.add('s3:PutObjectLegalHold');
        return this;
    }
    /**
     * Grants permission to place an Object Retention configuration on an object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectRetention.html
     */
    putObjectRetention() {
        this.add('s3:PutObjectRetention');
        return this;
    }
    /**
     * Grants permission to set the supplied tag-set to an object that already exists in a bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    putObjectTagging() {
        this.add('s3:PutObjectTagging');
        return this;
    }
    /**
     * Grants permission to use the acl subresource to set the access control list (ACL) permissions for an object that already exists in a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzStorageClass()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    putObjectVersionAcl() {
        this.add('s3:PutObjectVersionAcl');
        return this;
    }
    /**
     * Grants permission to set the supplied tag-set for a specific version of an object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    putObjectVersionTagging() {
        this.add('s3:PutObjectVersionTagging');
        return this;
    }
    /**
     * Grants permission to create a new replication configuration or replace an existing one
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketReplication.html
     */
    putReplicationConfiguration() {
        this.add('s3:PutReplicationConfiguration');
        return this;
    }
    /**
     * Grants permission to replicate delete markers to the destination bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html
     */
    replicateDelete() {
        this.add('s3:ReplicateDelete');
        return this;
    }
    /**
     * Grants permission to replicate objects and object tags to the destination bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html
     */
    replicateObject() {
        this.add('s3:ReplicateObject');
        return this;
    }
    /**
     * Grants permission to replicate object tags to the destination bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html
     */
    replicateTags() {
        this.add('s3:ReplicateTags');
        return this;
    }
    /**
     * Grants permission to restore an archived copy of an object back into Amazon S3
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_RestoreObject.html
     */
    restoreObject() {
        this.add('s3:RestoreObject');
        return this;
    }
    /**
     * Grants permission to update the priority of an existing job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifRequestJobPriority()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobPriority.html
     */
    updateJobPriority() {
        this.add('s3:UpdateJobPriority');
        return this;
    }
    /**
     * Grants permission to update the status for the specified job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     * - .ifJobSuspendedCause()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobStatus.html
     */
    updateJobStatus() {
        this.add('s3:UpdateJobStatus');
        return this;
    }
    /**
     * Adds a resource of type accesspoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/access-points.html
     *
     * @param accessPointName - Identifier for the accessPointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAccesspoint(accessPointName, account, region, partition) {
        var arn = 'arn:${Partition}:s3:${Region}:${Account}:accesspoint/${AccessPointName}';
        arn = arn.replace('${AccessPointName}', accessPointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type bucket to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingBucket.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBucket(bucketName, partition) {
        var arn = 'arn:${Partition}:s3:::${BucketName}';
        arn = arn.replace('${BucketName}', bucketName);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type object to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingObjects.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param objectName - Identifier for the objectName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onObject(bucketName, objectName, partition) {
        var arn = 'arn:${Partition}:s3:::${BucketName}/${ObjectName}';
        arn = arn.replace('${BucketName}', bucketName);
        arn = arn.replace('${ObjectName}', objectName);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-managing-jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJob(jobId, account, region, partition) {
        var arn = 'arn:${Partition}:s3:${Region}:${Account}:job/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .createJob()
     * - .putJobTagging()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .createJob()
     * - .putJobTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the network origin (Internet or VPC)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .abortMultipartUpload()
     * - .bypassGovernanceRetention()
     * - .createAccessPoint()
     * - .deleteAccessPoint()
     * - .deleteAccessPointPolicy()
     * - .deleteObject()
     * - .deleteObjectTagging()
     * - .deleteObjectVersion()
     * - .deleteObjectVersionTagging()
     * - .getAccessPoint()
     * - .getAccessPointPolicy()
     * - .getAccessPointPolicyStatus()
     * - .getObject()
     * - .getObjectAcl()
     * - .getObjectLegalHold()
     * - .getObjectRetention()
     * - .getObjectTagging()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionTagging()
     * - .listBucket()
     * - .listBucketMultipartUploads()
     * - .listBucketVersions()
     * - .listMultipartUploadParts()
     * - .putAccessPointPolicy()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectLegalHold()
     * - .putObjectRetention()
     * - .putObjectTagging()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     * - .restoreObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessPointNetworkOrigin(value, operator) {
        return this.if(`s3:AccessPointNetworkOrigin`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the AWS Account ID that owns the access point
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .abortMultipartUpload()
     * - .bypassGovernanceRetention()
     * - .createAccessPoint()
     * - .deleteAccessPoint()
     * - .deleteAccessPointPolicy()
     * - .deleteObject()
     * - .deleteObjectTagging()
     * - .deleteObjectVersion()
     * - .deleteObjectVersionTagging()
     * - .getAccessPoint()
     * - .getAccessPointPolicy()
     * - .getAccessPointPolicyStatus()
     * - .getObject()
     * - .getObjectAcl()
     * - .getObjectLegalHold()
     * - .getObjectRetention()
     * - .getObjectTagging()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionTagging()
     * - .listBucket()
     * - .listBucketMultipartUploads()
     * - .listBucketVersions()
     * - .listMultipartUploadParts()
     * - .putAccessPointPolicy()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectLegalHold()
     * - .putObjectRetention()
     * - .putObjectTagging()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     * - .restoreObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointAccount(value, operator) {
        return this.if(`s3:DataAccessPointAccount`, value, operator || 'StringLike');
    }
    /**
     * Filters access by an access point Amazon Resource Name (ARN)
     *
     * Applies to actions:
     * - .abortMultipartUpload()
     * - .bypassGovernanceRetention()
     * - .createAccessPoint()
     * - .deleteAccessPoint()
     * - .deleteAccessPointPolicy()
     * - .deleteObject()
     * - .deleteObjectTagging()
     * - .deleteObjectVersion()
     * - .deleteObjectVersionTagging()
     * - .getAccessPoint()
     * - .getAccessPointPolicy()
     * - .getAccessPointPolicyStatus()
     * - .getObject()
     * - .getObjectAcl()
     * - .getObjectLegalHold()
     * - .getObjectRetention()
     * - .getObjectTagging()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionTagging()
     * - .listBucket()
     * - .listBucketMultipartUploads()
     * - .listBucketVersions()
     * - .listMultipartUploadParts()
     * - .putAccessPointPolicy()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectLegalHold()
     * - .putObjectRetention()
     * - .putObjectTagging()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     * - .restoreObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointArn(value, operator) {
        return this.if(`s3:DataAccessPointArn`, value, operator || 'StringLike');
    }
    /**
     * Filters access to updating the job priority by operation
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .deleteJobTagging()
     * - .putJobTagging()
     * - .updateJobPriority()
     * - .updateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExistingJobOperation(value, operator) {
        return this.if(`s3:ExistingJobOperation`, value, operator || 'StringLike');
    }
    /**
     * Filters access to cancelling existing jobs by priority range
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .deleteJobTagging()
     * - .putJobTagging()
     * - .updateJobPriority()
     * - .updateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifExistingJobPriority(value, operator) {
        return this.if(`s3:ExistingJobPriority`, value, operator || 'NumericEquals');
    }
    /**
     * Requires that an existing object tag has a specific tag key and value.
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .deleteObjectTagging()
     * - .deleteObjectVersionTagging()
     * - .getObject()
     * - .getObjectAcl()
     * - .getObjectTagging()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionTagging()
     * - .putObjectAcl()
     * - .putObjectTagging()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExistingObjectTag(key, value, operator) {
        return this.if(`s3:ExistingObjectTag/${key}`, value, operator || 'StringLike');
    }
    /**
     * Filters access to cancelling suspended jobs by a specific job suspended cause (for example, AWAITING_CONFIRMATION)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .updateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifJobSuspendedCause(value, operator) {
        return this.if(`s3:JobSuspendedCause`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific Region
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#condition-key-bucket-ops-1
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLocationConstraint(value, operator) {
        return this.if(`s3:LocationConstraint`, value, operator || 'StringLike');
    }
    /**
     * Filters access to creating jobs by operation
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .createJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestJobOperation(value, operator) {
        return this.if(`s3:RequestJobOperation`, value, operator || 'StringLike');
    }
    /**
     * Filters access to creating new jobs by priority range
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .createJob()
     * - .updateJobPriority()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRequestJobPriority(value, operator) {
        return this.if(`s3:RequestJobPriority`, value, operator || 'NumericEquals');
    }
    /**
     * Restricts the tag keys and values allowed on objects
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .putObjectTagging()
     * - .putObjectVersionTagging()
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestObjectTag(key, value, operator) {
        return this.if(`s3:RequestObjectTag/${key}`, value, operator || 'StringLike');
    }
    /**
     * Restricts the tag keys allowed on objects
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .putObjectTagging()
     * - .putObjectVersionTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestObjectTagKeys(value, operator) {
        return this.if(`s3:RequestObjectTagKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific object version
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#getobjectversion-limit-access-to-specific-version-3
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionId(value, operator) {
        return this.if(`s3:VersionId`, value, operator || 'StringLike');
    }
    /**
     * Restricts incoming requests to a specific authentication method
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .abortMultipartUpload()
     * - .bypassGovernanceRetention()
     * - .createAccessPoint()
     * - .createBucket()
     * - .createJob()
     * - .deleteAccessPoint()
     * - .deleteAccessPointPolicy()
     * - .deleteBucket()
     * - .deleteBucketPolicy()
     * - .deleteBucketWebsite()
     * - .deleteJobTagging()
     * - .deleteObject()
     * - .deleteObjectTagging()
     * - .deleteObjectVersion()
     * - .deleteObjectVersionTagging()
     * - .describeJob()
     * - .getAccelerateConfiguration()
     * - .getAccessPoint()
     * - .getAccessPointPolicy()
     * - .getAccessPointPolicyStatus()
     * - .getAccountPublicAccessBlock()
     * - .getAnalyticsConfiguration()
     * - .getBucketAcl()
     * - .getBucketCORS()
     * - .getBucketLogging()
     * - .getBucketNotification()
     * - .getBucketObjectLockConfiguration()
     * - .getBucketPolicy()
     * - .getBucketPolicyStatus()
     * - .getBucketPublicAccessBlock()
     * - .getBucketRequestPayment()
     * - .getBucketTagging()
     * - .getBucketVersioning()
     * - .getBucketWebsite()
     * - .getEncryptionConfiguration()
     * - .getInventoryConfiguration()
     * - .getJobTagging()
     * - .getLifecycleConfiguration()
     * - .getMetricsConfiguration()
     * - .getObject()
     * - .getObjectAcl()
     * - .getObjectLegalHold()
     * - .getObjectRetention()
     * - .getObjectTagging()
     * - .getObjectTorrent()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionForReplication()
     * - .getObjectVersionTagging()
     * - .getObjectVersionTorrent()
     * - .getReplicationConfiguration()
     * - .listAccessPoints()
     * - .listAllMyBuckets()
     * - .listBucket()
     * - .listBucketMultipartUploads()
     * - .listBucketVersions()
     * - .listJobs()
     * - .listMultipartUploadParts()
     * - .objectOwnerOverrideToBucketOwner()
     * - .putAccelerateConfiguration()
     * - .putAccessPointPolicy()
     * - .putAccountPublicAccessBlock()
     * - .putAnalyticsConfiguration()
     * - .putBucketAcl()
     * - .putBucketCORS()
     * - .putBucketLogging()
     * - .putBucketNotification()
     * - .putBucketObjectLockConfiguration()
     * - .putBucketPolicy()
     * - .putBucketPublicAccessBlock()
     * - .putBucketRequestPayment()
     * - .putBucketTagging()
     * - .putBucketVersioning()
     * - .putBucketWebsite()
     * - .putEncryptionConfiguration()
     * - .putInventoryConfiguration()
     * - .putJobTagging()
     * - .putLifecycleConfiguration()
     * - .putMetricsConfiguration()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectLegalHold()
     * - .putObjectRetention()
     * - .putObjectTagging()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     * - .putReplicationConfiguration()
     * - .replicateDelete()
     * - .replicateObject()
     * - .replicateTags()
     * - .restoreObject()
     * - .updateJobPriority()
     * - .updateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthType(value, operator) {
        return this.if(`s3:authType`, value, operator || 'StringLike');
    }
    /**
     * Requires the delimiter parameter
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/walkthrough1.html
     *
     * Applies to actions:
     * - .listBucket()
     * - .listBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDelimiter(value, operator) {
        return this.if(`s3:delimiter`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific Region
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#condition-key-bucket-ops-1
     *
     * Applies to actions:
     * - .createAccessPoint()
     * - .createBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLocationconstraint(value, operator) {
        return this.if(`s3:locationconstraint`, value, operator || 'StringLike');
    }
    /**
     * Limits the maximum number of keys returned in a ListBucket request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#example-numeric-condition-operators
     *
     * Applies to actions:
     * - .listBucket()
     * - .listBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaxKeys(value, operator) {
        return this.if(`s3:max-keys`, value, operator || 'NumericEquals');
    }
    /**
     * Enables enforcement of the specified object legal hold status
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-legal-holds
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .putObjectLegalHold()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockLegalHold(value, operator) {
        return this.if(`s3:object-lock-legal-hold`, value, operator || 'StringLike');
    }
    /**
     * Enables enforcement of the specified object retention mode (COMPLIANCE or GOVERNANCE)
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .putObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockMode(value, operator) {
        return this.if(`s3:object-lock-mode`, value, operator || 'StringLike');
    }
    /**
     * Enables enforcement of an object relative to the remaining retention days
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-managing.html#object-lock-managing-retention-limits
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .putObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockRemainingRetentionDays(value, operator) {
        return this.if(`s3:object-lock-remaining-retention-days`, value, operator || 'StringLike');
    }
    /**
     * Enables enforcement of a specific retain-until-date
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-retention-periods
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .putObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockRetainUntilDate(value, operator) {
        return this.if(`s3:object-lock-retain-until-date`, value, operator || 'StringLike');
    }
    /**
     * Filters access by key name prefix
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#condition-key-bucket-ops-2
     *
     * Applies to actions:
     * - .listBucket()
     * - .listBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrefix(value, operator) {
        return this.if(`s3:prefix`, value, operator || 'StringLike');
    }
    /**
     * Identifies the length of time, in milliseconds, that a signature is valid in an authenticated request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .abortMultipartUpload()
     * - .bypassGovernanceRetention()
     * - .createAccessPoint()
     * - .createBucket()
     * - .createJob()
     * - .deleteAccessPoint()
     * - .deleteAccessPointPolicy()
     * - .deleteBucket()
     * - .deleteBucketPolicy()
     * - .deleteBucketWebsite()
     * - .deleteJobTagging()
     * - .deleteObject()
     * - .deleteObjectTagging()
     * - .deleteObjectVersion()
     * - .deleteObjectVersionTagging()
     * - .describeJob()
     * - .getAccelerateConfiguration()
     * - .getAccessPoint()
     * - .getAccessPointPolicy()
     * - .getAccessPointPolicyStatus()
     * - .getAccountPublicAccessBlock()
     * - .getAnalyticsConfiguration()
     * - .getBucketAcl()
     * - .getBucketCORS()
     * - .getBucketLogging()
     * - .getBucketNotification()
     * - .getBucketObjectLockConfiguration()
     * - .getBucketPolicy()
     * - .getBucketPolicyStatus()
     * - .getBucketPublicAccessBlock()
     * - .getBucketRequestPayment()
     * - .getBucketTagging()
     * - .getBucketVersioning()
     * - .getBucketWebsite()
     * - .getEncryptionConfiguration()
     * - .getInventoryConfiguration()
     * - .getJobTagging()
     * - .getLifecycleConfiguration()
     * - .getMetricsConfiguration()
     * - .getObject()
     * - .getObjectAcl()
     * - .getObjectLegalHold()
     * - .getObjectRetention()
     * - .getObjectTagging()
     * - .getObjectTorrent()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionForReplication()
     * - .getObjectVersionTagging()
     * - .getObjectVersionTorrent()
     * - .getReplicationConfiguration()
     * - .listAccessPoints()
     * - .listAllMyBuckets()
     * - .listBucket()
     * - .listBucketMultipartUploads()
     * - .listBucketVersions()
     * - .listJobs()
     * - .listMultipartUploadParts()
     * - .objectOwnerOverrideToBucketOwner()
     * - .putAccelerateConfiguration()
     * - .putAccessPointPolicy()
     * - .putAccountPublicAccessBlock()
     * - .putAnalyticsConfiguration()
     * - .putBucketAcl()
     * - .putBucketCORS()
     * - .putBucketLogging()
     * - .putBucketNotification()
     * - .putBucketObjectLockConfiguration()
     * - .putBucketPolicy()
     * - .putBucketPublicAccessBlock()
     * - .putBucketRequestPayment()
     * - .putBucketTagging()
     * - .putBucketVersioning()
     * - .putBucketWebsite()
     * - .putEncryptionConfiguration()
     * - .putInventoryConfiguration()
     * - .putJobTagging()
     * - .putLifecycleConfiguration()
     * - .putMetricsConfiguration()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectLegalHold()
     * - .putObjectRetention()
     * - .putObjectTagging()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     * - .putReplicationConfiguration()
     * - .replicateDelete()
     * - .replicateObject()
     * - .replicateTags()
     * - .restoreObject()
     * - .updateJobPriority()
     * - .updateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSignatureAge(value, operator) {
        return this.if(`s3:signatureAge`, value, operator || 'NumericEquals');
    }
    /**
     * Identifies the version of AWS Signature that is supported for authenticated requests
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .abortMultipartUpload()
     * - .bypassGovernanceRetention()
     * - .createAccessPoint()
     * - .createBucket()
     * - .createJob()
     * - .deleteAccessPoint()
     * - .deleteAccessPointPolicy()
     * - .deleteBucket()
     * - .deleteBucketPolicy()
     * - .deleteBucketWebsite()
     * - .deleteJobTagging()
     * - .deleteObject()
     * - .deleteObjectTagging()
     * - .deleteObjectVersion()
     * - .deleteObjectVersionTagging()
     * - .describeJob()
     * - .getAccelerateConfiguration()
     * - .getAccessPoint()
     * - .getAccessPointPolicy()
     * - .getAccessPointPolicyStatus()
     * - .getAccountPublicAccessBlock()
     * - .getAnalyticsConfiguration()
     * - .getBucketAcl()
     * - .getBucketCORS()
     * - .getBucketLogging()
     * - .getBucketNotification()
     * - .getBucketObjectLockConfiguration()
     * - .getBucketPolicy()
     * - .getBucketPolicyStatus()
     * - .getBucketPublicAccessBlock()
     * - .getBucketRequestPayment()
     * - .getBucketTagging()
     * - .getBucketVersioning()
     * - .getBucketWebsite()
     * - .getEncryptionConfiguration()
     * - .getInventoryConfiguration()
     * - .getJobTagging()
     * - .getLifecycleConfiguration()
     * - .getMetricsConfiguration()
     * - .getObject()
     * - .getObjectAcl()
     * - .getObjectLegalHold()
     * - .getObjectRetention()
     * - .getObjectTagging()
     * - .getObjectTorrent()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionForReplication()
     * - .getObjectVersionTagging()
     * - .getObjectVersionTorrent()
     * - .getReplicationConfiguration()
     * - .listAccessPoints()
     * - .listAllMyBuckets()
     * - .listBucket()
     * - .listBucketMultipartUploads()
     * - .listBucketVersions()
     * - .listJobs()
     * - .listMultipartUploadParts()
     * - .objectOwnerOverrideToBucketOwner()
     * - .putAccelerateConfiguration()
     * - .putAccessPointPolicy()
     * - .putAccountPublicAccessBlock()
     * - .putAnalyticsConfiguration()
     * - .putBucketAcl()
     * - .putBucketCORS()
     * - .putBucketLogging()
     * - .putBucketNotification()
     * - .putBucketObjectLockConfiguration()
     * - .putBucketPolicy()
     * - .putBucketPublicAccessBlock()
     * - .putBucketRequestPayment()
     * - .putBucketTagging()
     * - .putBucketVersioning()
     * - .putBucketWebsite()
     * - .putEncryptionConfiguration()
     * - .putInventoryConfiguration()
     * - .putJobTagging()
     * - .putLifecycleConfiguration()
     * - .putMetricsConfiguration()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectLegalHold()
     * - .putObjectRetention()
     * - .putObjectTagging()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     * - .putReplicationConfiguration()
     * - .replicateDelete()
     * - .replicateObject()
     * - .replicateTags()
     * - .restoreObject()
     * - .updateJobPriority()
     * - .updateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSignatureversion(value, operator) {
        return this.if(`s3:signatureversion`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific object version
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#getobjectversion-limit-access-to-specific-version-3
     *
     * Applies to actions:
     * - .deleteObjectVersion()
     * - .deleteObjectVersionTagging()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionTagging()
     * - .getObjectVersionTorrent()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionid(value, operator) {
        return this.if(`s3:versionid`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-acl header with a specific canned ACL in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .createAccessPoint()
     * - .createBucket()
     * - .putBucketAcl()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzAcl(value, operator) {
        return this.if(`s3:x-amz-acl`, value, operator || 'StringLike');
    }
    /**
     * Disallows unsigned content in your bucket
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .abortMultipartUpload()
     * - .bypassGovernanceRetention()
     * - .createAccessPoint()
     * - .createBucket()
     * - .createJob()
     * - .deleteAccessPoint()
     * - .deleteAccessPointPolicy()
     * - .deleteBucket()
     * - .deleteBucketPolicy()
     * - .deleteBucketWebsite()
     * - .deleteJobTagging()
     * - .deleteObject()
     * - .deleteObjectTagging()
     * - .deleteObjectVersion()
     * - .deleteObjectVersionTagging()
     * - .describeJob()
     * - .getAccelerateConfiguration()
     * - .getAccessPoint()
     * - .getAccessPointPolicy()
     * - .getAccessPointPolicyStatus()
     * - .getAccountPublicAccessBlock()
     * - .getAnalyticsConfiguration()
     * - .getBucketAcl()
     * - .getBucketCORS()
     * - .getBucketLogging()
     * - .getBucketNotification()
     * - .getBucketPolicy()
     * - .getBucketPolicyStatus()
     * - .getBucketPublicAccessBlock()
     * - .getBucketRequestPayment()
     * - .getBucketTagging()
     * - .getBucketVersioning()
     * - .getBucketWebsite()
     * - .getEncryptionConfiguration()
     * - .getInventoryConfiguration()
     * - .getJobTagging()
     * - .getLifecycleConfiguration()
     * - .getMetricsConfiguration()
     * - .getObject()
     * - .getObjectAcl()
     * - .getObjectLegalHold()
     * - .getObjectRetention()
     * - .getObjectTagging()
     * - .getObjectTorrent()
     * - .getObjectVersion()
     * - .getObjectVersionAcl()
     * - .getObjectVersionForReplication()
     * - .getObjectVersionTagging()
     * - .getObjectVersionTorrent()
     * - .getReplicationConfiguration()
     * - .listAccessPoints()
     * - .listAllMyBuckets()
     * - .listBucket()
     * - .listBucketMultipartUploads()
     * - .listBucketVersions()
     * - .listJobs()
     * - .listMultipartUploadParts()
     * - .objectOwnerOverrideToBucketOwner()
     * - .putAccelerateConfiguration()
     * - .putAccessPointPolicy()
     * - .putAccountPublicAccessBlock()
     * - .putAnalyticsConfiguration()
     * - .putBucketAcl()
     * - .putBucketCORS()
     * - .putBucketLogging()
     * - .putBucketNotification()
     * - .putBucketPolicy()
     * - .putBucketPublicAccessBlock()
     * - .putBucketRequestPayment()
     * - .putBucketTagging()
     * - .putBucketVersioning()
     * - .putBucketWebsite()
     * - .putEncryptionConfiguration()
     * - .putInventoryConfiguration()
     * - .putJobTagging()
     * - .putLifecycleConfiguration()
     * - .putMetricsConfiguration()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectLegalHold()
     * - .putObjectRetention()
     * - .putObjectTagging()
     * - .putObjectVersionAcl()
     * - .putObjectVersionTagging()
     * - .putReplicationConfiguration()
     * - .replicateDelete()
     * - .replicateObject()
     * - .replicateTags()
     * - .restoreObject()
     * - .updateJobPriority()
     * - .updateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzContentSha256(value, operator) {
        return this.if(`s3:x-amz-content-sha256`, value, operator || 'StringLike');
    }
    /**
     * Restricts the copy source to a specific bucket, prefix, or object
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#putobject-limit-copy-source-3
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzCopySource(value, operator) {
        return this.if(`s3:x-amz-copy-source`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-grant-full-control (full control) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .createBucket()
     * - .putBucketAcl()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantFullControl(value, operator) {
        return this.if(`s3:x-amz-grant-full-control`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-grant-read (read access) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .createBucket()
     * - .putBucketAcl()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantRead(value, operator) {
        return this.if(`s3:x-amz-grant-read`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-grant-read-acp (read permissions for the ACL) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .createBucket()
     * - .putBucketAcl()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantReadAcp(value, operator) {
        return this.if(`s3:x-amz-grant-read-acp`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-grant-write (write access) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .createBucket()
     * - .putBucketAcl()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantWrite(value, operator) {
        return this.if(`s3:x-amz-grant-write`, value, operator || 'StringLike');
    }
    /**
     * Requires the x-amz-grant-write-acp (write permissions for the ACL) header in a request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .createBucket()
     * - .putBucketAcl()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantWriteAcp(value, operator) {
        return this.if(`s3:x-amz-grant-write-acp`, value, operator || 'StringLike');
    }
    /**
     * Enables enforcement of object metadata behavior (COPY or REPLACE) when objects are copied
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CopyObject.html
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzMetadataDirective(value, operator) {
        return this.if(`s3:x-amz-metadata-directive`, value, operator || 'StringLike');
    }
    /**
     * Requires server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .replicateObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryption(value, operator) {
        return this.if(`s3:x-amz-server-side-encryption`, value, operator || 'StringLike');
    }
    /**
     * Requires a specific AWS KMS customer managed CMK for server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html#require-sse-kms
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .replicateObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryptionAwsKmsKeyId(value, operator) {
        return this.if(`s3:x-amz-server-side-encryption-aws-kms-key-id`, value, operator || 'StringLike');
    }
    /**
     * Filters access by storage class
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/storage-class-intro.html#sc-howtoset
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     * - .putObjectAcl()
     * - .putObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzStorageClass(value, operator) {
        return this.if(`s3:x-amz-storage-class`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific website redirect location for buckets that are configured as static websites
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/how-to-page-redirect.html#page-redirect-using-rest-api
     *
     * Applies to actions:
     * - .bypassGovernanceRetention()
     * - .putObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzWebsiteRedirectLocation(value, operator) {
        return this.if(`s3:x-amz-website-redirect-location`, value, operator || 'StringLike');
    }
}
exports.S3 = S3;
//# sourceMappingURL=data:application/json;base64,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