"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ses = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ses](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ses extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ses](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ses';
        this.actionList = {
            "CloneReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CloneReceiptRuleSet.html",
                "description": "Creates a receipt rule set by cloning an existing one",
                "accessLevel": "Write"
            },
            "CreateConfigurationSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSet.html",
                "description": "Creates a new configuration set",
                "accessLevel": "Write"
            },
            "CreateConfigurationSetEventDestination": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSetEventDestination.html",
                "description": "Creates a configuration set event destination",
                "accessLevel": "Write"
            },
            "CreateConfigurationSetTrackingOptions": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSetTrackingOptions.html",
                "description": "Creates an association between a configuration set and a custom domain for open and click event tracking",
                "accessLevel": "Write"
            },
            "CreateCustomVerificationEmailTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateCustomVerificationEmailTemplate.html",
                "description": "Creates a new custom verification email template",
                "accessLevel": "Write"
            },
            "CreateReceiptFilter": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptFilter.html",
                "description": "Creates a new IP address filter",
                "accessLevel": "Write"
            },
            "CreateReceiptRule": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptRule.html",
                "description": "Creates a receipt rule",
                "accessLevel": "Write"
            },
            "CreateReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptRuleSet.html",
                "description": "Creates an empty receipt rule set",
                "accessLevel": "Write"
            },
            "CreateTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateTemplate.html",
                "description": "Creates an email template",
                "accessLevel": "Write"
            },
            "DeleteConfigurationSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSet.html",
                "description": "Deletes the configuration set",
                "accessLevel": "Write"
            },
            "DeleteConfigurationSetEventDestination": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSetEventDestination.html",
                "description": "Deletes a configuration set event destination",
                "accessLevel": "Write"
            },
            "DeleteConfigurationSetTrackingOptions": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSetTrackingOptions.html",
                "description": "Deletes an association between a configuration set and a custom domain for open and click event tracking",
                "accessLevel": "Write"
            },
            "DeleteCustomVerificationEmailTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteCustomVerificationEmailTemplate.html",
                "description": "Deletes an existing custom verification email template",
                "accessLevel": "Write"
            },
            "DeleteIdentity": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteIdentity.html",
                "description": "Deletes the specified identity (an email address or a domain) from the list of verified identities",
                "accessLevel": "Write"
            },
            "DeleteIdentityPolicy": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteIdentityPolicy.html",
                "description": "Deletes the specified identity (an email address or a domain) from the list of verified identities",
                "accessLevel": "Write"
            },
            "DeleteReceiptFilter": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptFilter.html",
                "description": "Deletes the specified IP address filter",
                "accessLevel": "Write"
            },
            "DeleteReceiptRule": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptRule.html",
                "description": "Deletes the specified receipt rule",
                "accessLevel": "Write"
            },
            "DeleteReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptRuleSet.html",
                "description": "Deletes the specified receipt rule set and all of the receipt rules it contains",
                "accessLevel": "Write"
            },
            "DeleteTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteTemplate.html",
                "description": "Deletes an email template",
                "accessLevel": "Write"
            },
            "DeleteVerifiedEmailAddress": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteVerifiedEmailAddress.html",
                "description": "Deletes the specified email address from the list of verified addresses",
                "accessLevel": "Write"
            },
            "DescribeActiveReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeActiveReceiptRuleSet.html",
                "description": "Returns the metadata and receipt rules for the receipt rule set that is currently active",
                "accessLevel": "Read"
            },
            "DescribeConfigurationSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeConfigurationSet.html",
                "description": "Returns the details of the specified configuration set",
                "accessLevel": "Read"
            },
            "DescribeReceiptRule": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeReceiptRule.html",
                "description": "Returns the details of the specified receipt rule",
                "accessLevel": "Read"
            },
            "DescribeReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeReceiptRuleSet.html",
                "description": "Returns the details of the specified receipt rule set",
                "accessLevel": "Read"
            },
            "GetAccountSendingEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetAccountSendingEnabled.html",
                "description": "Returns the email sending status of the Amazon SES account for the current region",
                "accessLevel": "Read"
            },
            "GetCustomVerificationEmailTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetCustomVerificationEmailTemplate.html",
                "description": "Returns the custom email verification template for the template name you specify",
                "accessLevel": "Read"
            },
            "GetIdentityDkimAttributes": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityDkimAttributes.html",
                "description": "Returns the current status of Easy DKIM signing for an entity",
                "accessLevel": "Read"
            },
            "GetIdentityMailFromDomainAttributes": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityMailFromDomainAttributes.html",
                "description": "Returns the custom MAIL FROM attributes for a list of identities (email addresses and/or domains)",
                "accessLevel": "Read"
            },
            "GetIdentityNotificationAttributes": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityNotificationAttributes.html",
                "description": "Given a list of verified identities (email addresses and/or domains), returns a structure describing identity notification attributes",
                "accessLevel": "Read"
            },
            "GetIdentityPolicies": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityPolicies.html",
                "description": "Returns the requested sending authorization policies for the given identity (an email address or a domain)",
                "accessLevel": "Read"
            },
            "GetIdentityVerificationAttributes": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityVerificationAttributes.html",
                "description": "Given a list of identities (email addresses and/or domains), returns the verification status and (for domain identities) the verification token for each identity",
                "accessLevel": "Read"
            },
            "GetSendQuota": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetSendQuota.html",
                "description": "Returns the user's current sending limits",
                "accessLevel": "Read"
            },
            "GetSendStatistics": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetSendStatistics.html",
                "description": "Returns the user's sending statistics. The result is a list of data points, representing the last two weeks of sending activity",
                "accessLevel": "Read"
            },
            "GetTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetTemplate.html",
                "description": "Returns the template object (which includes the Subject line, HTML part and text part) for the template you specify",
                "accessLevel": "Read"
            },
            "ListConfigurationSets": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListConfigurationSets.html",
                "description": "Returns a list of the configuration sets associated with your Amazon SES account in the current AWS Region",
                "accessLevel": "List"
            },
            "ListCustomVerificationEmailTemplates": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListCustomVerificationEmailTemplates.html",
                "description": "Lists the existing custom verification email templates for your account in the current AWS Region",
                "accessLevel": "List"
            },
            "ListIdentities": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListIdentities.html",
                "description": "Returns a list containing all of the identities (email addresses and domains) for your AWS account, regardless of verification status",
                "accessLevel": "List"
            },
            "ListIdentityPolicies": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListIdentityPolicies.html",
                "description": "Returns a list of sending authorization policies that are attached to the given identity (an email address or a domain)",
                "accessLevel": "List"
            },
            "ListReceiptFilters": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListReceiptFilters.html",
                "description": "Lists the IP address filters associated with your AWS account",
                "accessLevel": "List"
            },
            "ListReceiptRuleSets": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListReceiptRuleSets.html",
                "description": "Lists the receipt rule sets that exist under your AWS account",
                "accessLevel": "List"
            },
            "ListTemplates": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListTemplates.html",
                "description": "Lists the email templates present in your Amazon SES account in the current AWS Region",
                "accessLevel": "List"
            },
            "ListVerifiedEmailAddresses": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListVerifiedEmailAddresses.html",
                "description": "Returns a list containing all of the email addresses that have been verified",
                "accessLevel": "List"
            },
            "PutIdentityPolicy": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_PutIdentityPolicy.html",
                "description": "Adds or updates a sending authorization policy for the specified identity (an email address or a domain)",
                "accessLevel": "Write"
            },
            "ReorderReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ReorderReceiptRuleSet.html",
                "description": "Reorders the receipt rules within a receipt rule set",
                "accessLevel": "Write"
            },
            "SendBounce": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendBounce.html",
                "description": "Generates and sends a bounce message to the sender of an email you received through Amazon SES",
                "accessLevel": "Write",
                "conditions": [
                    "ses:FromAddress"
                ]
            },
            "SendBulkTemplatedEmail": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendBulkTemplatedEmail.html",
                "description": "Composes an email message to multiple destinations",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "ses:FeedbackAddress",
                    "ses:FromAddress",
                    "ses:FromDisplayName",
                    "ses:Recipients"
                ]
            },
            "SendCustomVerificationEmail": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendCustomVerificationEmail.html",
                "description": "Adds an email address to the list of identities for your Amazon SES account in the current AWS Region and attempts to verify it",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "ses:FeedbackAddress",
                    "ses:FromAddress",
                    "ses:FromDisplayName",
                    "ses:Recipients"
                ]
            },
            "SendEmail": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendEmail.html",
                "description": "Composes an email message based on input data, and then immediately queues the message for sending",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "ses:FeedbackAddress",
                    "ses:FromAddress",
                    "ses:FromDisplayName",
                    "ses:Recipients"
                ]
            },
            "SendRawEmail": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendRawEmail.html",
                "description": "Sends an email message, with header and content specified by the client",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "ses:FeedbackAddress",
                    "ses:FromAddress",
                    "ses:FromDisplayName",
                    "ses:Recipients"
                ]
            },
            "SendTemplatedEmail": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendTemplatedEmail.html",
                "description": "Composes an email message using an email template and immediately queues it for sending",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "ses:FeedbackAddress",
                    "ses:FromAddress",
                    "ses:FromDisplayName",
                    "ses:Recipients"
                ]
            },
            "SetActiveReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetActiveReceiptRuleSet.html",
                "description": "Sets the specified receipt rule set as the active receipt rule set",
                "accessLevel": "Write"
            },
            "SetIdentityDkimEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityDkimEnabled.html",
                "description": "Enables or disables Easy DKIM signing of email sent from an identity",
                "accessLevel": "Write"
            },
            "SetIdentityFeedbackForwardingEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityFeedbackForwardingEnabled.html",
                "description": "Given an identity (an email address or a domain), enables or disables whether Amazon SES forwards bounce and complaint notifications as email",
                "accessLevel": "Write"
            },
            "SetIdentityHeadersInNotificationsEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityHeadersInNotificationsEnabled.html",
                "description": "Given an identity (an email address or a domain), sets whether Amazon SES includes the original email headers in the Amazon Simple Notification Service (Amazon SNS) notifications of a specified type",
                "accessLevel": "Write"
            },
            "SetIdentityMailFromDomain": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityMailFromDomain.html",
                "description": "Enables or disables the custom MAIL FROM domain setup for a verified identity (an email address or a domain)",
                "accessLevel": "Write"
            },
            "SetIdentityNotificationTopic": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityNotificationTopic.html",
                "description": "Given an identity (an email address or a domain), sets the Amazon Simple Notification Service (Amazon SNS) topic to which Amazon SES will publish bounce, complaint, and/or delivery notifications for emails sent with that identity as the Source",
                "accessLevel": "Write"
            },
            "SetReceiptRulePosition": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetReceiptRulePosition.html",
                "description": "Sets the position of the specified receipt rule in the receipt rule set",
                "accessLevel": "Write"
            },
            "TestRenderTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_TestRenderTemplate.html",
                "description": "Creates a preview of the MIME content of an email when provided with a template and a set of replacement data",
                "accessLevel": "Write"
            },
            "UpdateAccountSendingEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateAccountSendingEnabled.html",
                "description": "Enables or disables email sending across your entire Amazon SES account in the current AWS Region",
                "accessLevel": "Write"
            },
            "UpdateConfigurationSetEventDestination": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetEventDestination.html",
                "description": "Updates the event destination of a configuration set",
                "accessLevel": "Write"
            },
            "UpdateConfigurationSetReputationMetricsEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetReputationMetricsEnabled.html",
                "description": "Enables or disables the publishing of reputation metrics for emails sent using a specific configuration set in a given AWS Region",
                "accessLevel": "Write"
            },
            "UpdateConfigurationSetSendingEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetSendingEnabled.html",
                "description": "Enables or disables email sending for messages sent using a specific configuration set in a given AWS Region",
                "accessLevel": "Write"
            },
            "UpdateConfigurationSetTrackingOptions": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetTrackingOptions.html",
                "description": "Modifies an association between a configuration set and a custom domain for open and click event tracking",
                "accessLevel": "Write"
            },
            "UpdateCustomVerificationEmailTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateCustomVerificationEmailTemplate.html",
                "description": "Updates an existing custom verification email template",
                "accessLevel": "Write"
            },
            "UpdateReceiptRule": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateReceiptRule.html",
                "description": "Updates a receipt rule",
                "accessLevel": "Write"
            },
            "UpdateTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateTemplate.html",
                "description": "Updates an email template",
                "accessLevel": "Write"
            },
            "VerifyDomainDkim": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyDomainDkim.html",
                "description": "Returns a set of DKIM tokens for a domain",
                "accessLevel": "Read"
            },
            "VerifyDomainIdentity": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyDomainIdentity.html",
                "description": "Verifies a domain",
                "accessLevel": "Read"
            },
            "VerifyEmailAddress": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyEmailAddress.html",
                "description": "Verifies an email address. This action causes a confirmation email message to be sent to the specified address. This action is throttled at one request per second",
                "accessLevel": "Read"
            },
            "VerifyEmailIdentity": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyEmailIdentity.html",
                "description": "Verifies an email address. This action causes a confirmation email message to be sent to the specified address. This action is throttled at one request per second",
                "accessLevel": "Read"
            }
        };
        this.resourceTypes = {
            "configuration-set": {
                "name": "configuration-set",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ConfigurationSet.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}",
                "conditionKeys": []
            },
            "custom-verification-email-template": {
                "name": "custom-verification-email-template",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CustomVerificationEmailTemplate.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:custom-verification-email-template/${CustomVerificationEmailTemplateName}",
                "conditionKeys": []
            },
            "event-destination": {
                "name": "event-destination",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_EventDestination.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}:event-destination/${EventDestinationName}",
                "conditionKeys": []
            },
            "identity": {
                "name": "identity",
                "url": "",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:identity/${IdentityName}",
                "conditionKeys": []
            },
            "receipt-filter": {
                "name": "receipt-filter",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptFilter.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:receipt-filter/${ReceiptFilterName}",
                "conditionKeys": []
            },
            "receipt-rule": {
                "name": "receipt-rule",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptRule.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:receipt-rule-set/${ReceiptRuleSetName}:receipt-rule/${ReceiptRuleName}",
                "conditionKeys": []
            },
            "receipt-rule-set": {
                "name": "receipt-rule-set",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptRuleSetMetadata.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:receipt-rule-set/${ReceiptRuleSetName}",
                "conditionKeys": []
            },
            "template": {
                "name": "template",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_Template.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:template/${TemplateName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Creates a receipt rule set by cloning an existing one
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CloneReceiptRuleSet.html
     */
    cloneReceiptRuleSet() {
        this.add('ses:CloneReceiptRuleSet');
        return this;
    }
    /**
     * Creates a new configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSet.html
     */
    createConfigurationSet() {
        this.add('ses:CreateConfigurationSet');
        return this;
    }
    /**
     * Creates a configuration set event destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSetEventDestination.html
     */
    createConfigurationSetEventDestination() {
        this.add('ses:CreateConfigurationSetEventDestination');
        return this;
    }
    /**
     * Creates an association between a configuration set and a custom domain for open and click event tracking
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSetTrackingOptions.html
     */
    createConfigurationSetTrackingOptions() {
        this.add('ses:CreateConfigurationSetTrackingOptions');
        return this;
    }
    /**
     * Creates a new custom verification email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateCustomVerificationEmailTemplate.html
     */
    createCustomVerificationEmailTemplate() {
        this.add('ses:CreateCustomVerificationEmailTemplate');
        return this;
    }
    /**
     * Creates a new IP address filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptFilter.html
     */
    createReceiptFilter() {
        this.add('ses:CreateReceiptFilter');
        return this;
    }
    /**
     * Creates a receipt rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptRule.html
     */
    createReceiptRule() {
        this.add('ses:CreateReceiptRule');
        return this;
    }
    /**
     * Creates an empty receipt rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptRuleSet.html
     */
    createReceiptRuleSet() {
        this.add('ses:CreateReceiptRuleSet');
        return this;
    }
    /**
     * Creates an email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateTemplate.html
     */
    createTemplate() {
        this.add('ses:CreateTemplate');
        return this;
    }
    /**
     * Deletes the configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSet.html
     */
    deleteConfigurationSet() {
        this.add('ses:DeleteConfigurationSet');
        return this;
    }
    /**
     * Deletes a configuration set event destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSetEventDestination.html
     */
    deleteConfigurationSetEventDestination() {
        this.add('ses:DeleteConfigurationSetEventDestination');
        return this;
    }
    /**
     * Deletes an association between a configuration set and a custom domain for open and click event tracking
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSetTrackingOptions.html
     */
    deleteConfigurationSetTrackingOptions() {
        this.add('ses:DeleteConfigurationSetTrackingOptions');
        return this;
    }
    /**
     * Deletes an existing custom verification email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteCustomVerificationEmailTemplate.html
     */
    deleteCustomVerificationEmailTemplate() {
        this.add('ses:DeleteCustomVerificationEmailTemplate');
        return this;
    }
    /**
     * Deletes the specified identity (an email address or a domain) from the list of verified identities
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteIdentity.html
     */
    deleteIdentity() {
        this.add('ses:DeleteIdentity');
        return this;
    }
    /**
     * Deletes the specified identity (an email address or a domain) from the list of verified identities
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteIdentityPolicy.html
     */
    deleteIdentityPolicy() {
        this.add('ses:DeleteIdentityPolicy');
        return this;
    }
    /**
     * Deletes the specified IP address filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptFilter.html
     */
    deleteReceiptFilter() {
        this.add('ses:DeleteReceiptFilter');
        return this;
    }
    /**
     * Deletes the specified receipt rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptRule.html
     */
    deleteReceiptRule() {
        this.add('ses:DeleteReceiptRule');
        return this;
    }
    /**
     * Deletes the specified receipt rule set and all of the receipt rules it contains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptRuleSet.html
     */
    deleteReceiptRuleSet() {
        this.add('ses:DeleteReceiptRuleSet');
        return this;
    }
    /**
     * Deletes an email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteTemplate.html
     */
    deleteTemplate() {
        this.add('ses:DeleteTemplate');
        return this;
    }
    /**
     * Deletes the specified email address from the list of verified addresses
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteVerifiedEmailAddress.html
     */
    deleteVerifiedEmailAddress() {
        this.add('ses:DeleteVerifiedEmailAddress');
        return this;
    }
    /**
     * Returns the metadata and receipt rules for the receipt rule set that is currently active
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeActiveReceiptRuleSet.html
     */
    describeActiveReceiptRuleSet() {
        this.add('ses:DescribeActiveReceiptRuleSet');
        return this;
    }
    /**
     * Returns the details of the specified configuration set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeConfigurationSet.html
     */
    describeConfigurationSet() {
        this.add('ses:DescribeConfigurationSet');
        return this;
    }
    /**
     * Returns the details of the specified receipt rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeReceiptRule.html
     */
    describeReceiptRule() {
        this.add('ses:DescribeReceiptRule');
        return this;
    }
    /**
     * Returns the details of the specified receipt rule set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeReceiptRuleSet.html
     */
    describeReceiptRuleSet() {
        this.add('ses:DescribeReceiptRuleSet');
        return this;
    }
    /**
     * Returns the email sending status of the Amazon SES account for the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetAccountSendingEnabled.html
     */
    getAccountSendingEnabled() {
        this.add('ses:GetAccountSendingEnabled');
        return this;
    }
    /**
     * Returns the custom email verification template for the template name you specify
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetCustomVerificationEmailTemplate.html
     */
    getCustomVerificationEmailTemplate() {
        this.add('ses:GetCustomVerificationEmailTemplate');
        return this;
    }
    /**
     * Returns the current status of Easy DKIM signing for an entity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityDkimAttributes.html
     */
    getIdentityDkimAttributes() {
        this.add('ses:GetIdentityDkimAttributes');
        return this;
    }
    /**
     * Returns the custom MAIL FROM attributes for a list of identities (email addresses and/or domains)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityMailFromDomainAttributes.html
     */
    getIdentityMailFromDomainAttributes() {
        this.add('ses:GetIdentityMailFromDomainAttributes');
        return this;
    }
    /**
     * Given a list of verified identities (email addresses and/or domains), returns a structure describing identity notification attributes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityNotificationAttributes.html
     */
    getIdentityNotificationAttributes() {
        this.add('ses:GetIdentityNotificationAttributes');
        return this;
    }
    /**
     * Returns the requested sending authorization policies for the given identity (an email address or a domain)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityPolicies.html
     */
    getIdentityPolicies() {
        this.add('ses:GetIdentityPolicies');
        return this;
    }
    /**
     * Given a list of identities (email addresses and/or domains), returns the verification status and (for domain identities) the verification token for each identity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityVerificationAttributes.html
     */
    getIdentityVerificationAttributes() {
        this.add('ses:GetIdentityVerificationAttributes');
        return this;
    }
    /**
     * Returns the user's current sending limits
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetSendQuota.html
     */
    getSendQuota() {
        this.add('ses:GetSendQuota');
        return this;
    }
    /**
     * Returns the user's sending statistics. The result is a list of data points, representing the last two weeks of sending activity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetSendStatistics.html
     */
    getSendStatistics() {
        this.add('ses:GetSendStatistics');
        return this;
    }
    /**
     * Returns the template object (which includes the Subject line, HTML part and text part) for the template you specify
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetTemplate.html
     */
    getTemplate() {
        this.add('ses:GetTemplate');
        return this;
    }
    /**
     * Returns a list of the configuration sets associated with your Amazon SES account in the current AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListConfigurationSets.html
     */
    listConfigurationSets() {
        this.add('ses:ListConfigurationSets');
        return this;
    }
    /**
     * Lists the existing custom verification email templates for your account in the current AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListCustomVerificationEmailTemplates.html
     */
    listCustomVerificationEmailTemplates() {
        this.add('ses:ListCustomVerificationEmailTemplates');
        return this;
    }
    /**
     * Returns a list containing all of the identities (email addresses and domains) for your AWS account, regardless of verification status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListIdentities.html
     */
    listIdentities() {
        this.add('ses:ListIdentities');
        return this;
    }
    /**
     * Returns a list of sending authorization policies that are attached to the given identity (an email address or a domain)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListIdentityPolicies.html
     */
    listIdentityPolicies() {
        this.add('ses:ListIdentityPolicies');
        return this;
    }
    /**
     * Lists the IP address filters associated with your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListReceiptFilters.html
     */
    listReceiptFilters() {
        this.add('ses:ListReceiptFilters');
        return this;
    }
    /**
     * Lists the receipt rule sets that exist under your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListReceiptRuleSets.html
     */
    listReceiptRuleSets() {
        this.add('ses:ListReceiptRuleSets');
        return this;
    }
    /**
     * Lists the email templates present in your Amazon SES account in the current AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListTemplates.html
     */
    listTemplates() {
        this.add('ses:ListTemplates');
        return this;
    }
    /**
     * Returns a list containing all of the email addresses that have been verified
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListVerifiedEmailAddresses.html
     */
    listVerifiedEmailAddresses() {
        this.add('ses:ListVerifiedEmailAddresses');
        return this;
    }
    /**
     * Adds or updates a sending authorization policy for the specified identity (an email address or a domain)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_PutIdentityPolicy.html
     */
    putIdentityPolicy() {
        this.add('ses:PutIdentityPolicy');
        return this;
    }
    /**
     * Reorders the receipt rules within a receipt rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReorderReceiptRuleSet.html
     */
    reorderReceiptRuleSet() {
        this.add('ses:ReorderReceiptRuleSet');
        return this;
    }
    /**
     * Generates and sends a bounce message to the sender of an email you received through Amazon SES
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFromAddress()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendBounce.html
     */
    sendBounce() {
        this.add('ses:SendBounce');
        return this;
    }
    /**
     * Composes an email message to multiple destinations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendBulkTemplatedEmail.html
     */
    sendBulkTemplatedEmail() {
        this.add('ses:SendBulkTemplatedEmail');
        return this;
    }
    /**
     * Adds an email address to the list of identities for your Amazon SES account in the current AWS Region and attempts to verify it
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendCustomVerificationEmail.html
     */
    sendCustomVerificationEmail() {
        this.add('ses:SendCustomVerificationEmail');
        return this;
    }
    /**
     * Composes an email message based on input data, and then immediately queues the message for sending
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendEmail.html
     */
    sendEmail() {
        this.add('ses:SendEmail');
        return this;
    }
    /**
     * Sends an email message, with header and content specified by the client
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendRawEmail.html
     */
    sendRawEmail() {
        this.add('ses:SendRawEmail');
        return this;
    }
    /**
     * Composes an email message using an email template and immediately queues it for sending
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendTemplatedEmail.html
     */
    sendTemplatedEmail() {
        this.add('ses:SendTemplatedEmail');
        return this;
    }
    /**
     * Sets the specified receipt rule set as the active receipt rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetActiveReceiptRuleSet.html
     */
    setActiveReceiptRuleSet() {
        this.add('ses:SetActiveReceiptRuleSet');
        return this;
    }
    /**
     * Enables or disables Easy DKIM signing of email sent from an identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityDkimEnabled.html
     */
    setIdentityDkimEnabled() {
        this.add('ses:SetIdentityDkimEnabled');
        return this;
    }
    /**
     * Given an identity (an email address or a domain), enables or disables whether Amazon SES forwards bounce and complaint notifications as email
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityFeedbackForwardingEnabled.html
     */
    setIdentityFeedbackForwardingEnabled() {
        this.add('ses:SetIdentityFeedbackForwardingEnabled');
        return this;
    }
    /**
     * Given an identity (an email address or a domain), sets whether Amazon SES includes the original email headers in the Amazon Simple Notification Service (Amazon SNS) notifications of a specified type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityHeadersInNotificationsEnabled.html
     */
    setIdentityHeadersInNotificationsEnabled() {
        this.add('ses:SetIdentityHeadersInNotificationsEnabled');
        return this;
    }
    /**
     * Enables or disables the custom MAIL FROM domain setup for a verified identity (an email address or a domain)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityMailFromDomain.html
     */
    setIdentityMailFromDomain() {
        this.add('ses:SetIdentityMailFromDomain');
        return this;
    }
    /**
     * Given an identity (an email address or a domain), sets the Amazon Simple Notification Service (Amazon SNS) topic to which Amazon SES will publish bounce, complaint, and/or delivery notifications for emails sent with that identity as the Source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityNotificationTopic.html
     */
    setIdentityNotificationTopic() {
        this.add('ses:SetIdentityNotificationTopic');
        return this;
    }
    /**
     * Sets the position of the specified receipt rule in the receipt rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetReceiptRulePosition.html
     */
    setReceiptRulePosition() {
        this.add('ses:SetReceiptRulePosition');
        return this;
    }
    /**
     * Creates a preview of the MIME content of an email when provided with a template and a set of replacement data
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_TestRenderTemplate.html
     */
    testRenderTemplate() {
        this.add('ses:TestRenderTemplate');
        return this;
    }
    /**
     * Enables or disables email sending across your entire Amazon SES account in the current AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateAccountSendingEnabled.html
     */
    updateAccountSendingEnabled() {
        this.add('ses:UpdateAccountSendingEnabled');
        return this;
    }
    /**
     * Updates the event destination of a configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetEventDestination.html
     */
    updateConfigurationSetEventDestination() {
        this.add('ses:UpdateConfigurationSetEventDestination');
        return this;
    }
    /**
     * Enables or disables the publishing of reputation metrics for emails sent using a specific configuration set in a given AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetReputationMetricsEnabled.html
     */
    updateConfigurationSetReputationMetricsEnabled() {
        this.add('ses:UpdateConfigurationSetReputationMetricsEnabled');
        return this;
    }
    /**
     * Enables or disables email sending for messages sent using a specific configuration set in a given AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetSendingEnabled.html
     */
    updateConfigurationSetSendingEnabled() {
        this.add('ses:UpdateConfigurationSetSendingEnabled');
        return this;
    }
    /**
     * Modifies an association between a configuration set and a custom domain for open and click event tracking
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetTrackingOptions.html
     */
    updateConfigurationSetTrackingOptions() {
        this.add('ses:UpdateConfigurationSetTrackingOptions');
        return this;
    }
    /**
     * Updates an existing custom verification email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateCustomVerificationEmailTemplate.html
     */
    updateCustomVerificationEmailTemplate() {
        this.add('ses:UpdateCustomVerificationEmailTemplate');
        return this;
    }
    /**
     * Updates a receipt rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateReceiptRule.html
     */
    updateReceiptRule() {
        this.add('ses:UpdateReceiptRule');
        return this;
    }
    /**
     * Updates an email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateTemplate.html
     */
    updateTemplate() {
        this.add('ses:UpdateTemplate');
        return this;
    }
    /**
     * Returns a set of DKIM tokens for a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyDomainDkim.html
     */
    verifyDomainDkim() {
        this.add('ses:VerifyDomainDkim');
        return this;
    }
    /**
     * Verifies a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyDomainIdentity.html
     */
    verifyDomainIdentity() {
        this.add('ses:VerifyDomainIdentity');
        return this;
    }
    /**
     * Verifies an email address. This action causes a confirmation email message to be sent to the specified address. This action is throttled at one request per second
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyEmailAddress.html
     */
    verifyEmailAddress() {
        this.add('ses:VerifyEmailAddress');
        return this;
    }
    /**
     * Verifies an email address. This action causes a confirmation email message to be sent to the specified address. This action is throttled at one request per second
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyEmailIdentity.html
     */
    verifyEmailIdentity() {
        this.add('ses:VerifyEmailIdentity');
        return this;
    }
    /**
     * Adds a resource of type configuration-set to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ConfigurationSet.html
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConfigurationSet(configurationSetName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}';
        arn = arn.replace('${ConfigurationSetName}', configurationSetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type custom-verification-email-template to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CustomVerificationEmailTemplate.html
     *
     * @param customVerificationEmailTemplateName - Identifier for the customVerificationEmailTemplateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCustomVerificationEmailTemplate(customVerificationEmailTemplateName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:custom-verification-email-template/${CustomVerificationEmailTemplateName}';
        arn = arn.replace('${CustomVerificationEmailTemplateName}', customVerificationEmailTemplateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type event-destination to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_EventDestination.html
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param eventDestinationName - Identifier for the eventDestinationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEventDestination(configurationSetName, eventDestinationName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}:event-destination/${EventDestinationName}';
        arn = arn.replace('${ConfigurationSetName}', configurationSetName);
        arn = arn.replace('${EventDestinationName}', eventDestinationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type identity to the statement
     *
     * @param identityName - Identifier for the identityName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIdentity(identityName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:identity/${IdentityName}';
        arn = arn.replace('${IdentityName}', identityName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type receipt-filter to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptFilter.html
     *
     * @param receiptFilterName - Identifier for the receiptFilterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReceiptFilter(receiptFilterName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:receipt-filter/${ReceiptFilterName}';
        arn = arn.replace('${ReceiptFilterName}', receiptFilterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type receipt-rule to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptRule.html
     *
     * @param receiptRuleSetName - Identifier for the receiptRuleSetName.
     * @param receiptRuleName - Identifier for the receiptRuleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReceiptRule(receiptRuleSetName, receiptRuleName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:receipt-rule-set/${ReceiptRuleSetName}:receipt-rule/${ReceiptRuleName}';
        arn = arn.replace('${ReceiptRuleSetName}', receiptRuleSetName);
        arn = arn.replace('${ReceiptRuleName}', receiptRuleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type receipt-rule-set to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptRuleSetMetadata.html
     *
     * @param receiptRuleSetName - Identifier for the receiptRuleSetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReceiptRuleSet(receiptRuleSetName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:receipt-rule-set/${ReceiptRuleSetName}';
        arn = arn.replace('${ReceiptRuleSetName}', receiptRuleSetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type template to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_Template.html
     *
     * @param templateName - Identifier for the templateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTemplate(templateName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:template/${TemplateName}';
        arn = arn.replace('${TemplateName}', templateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The "Return-Path" address, which specifies where bounces and complaints are sent by email feedback forwarding.
     *
     * https://docs.aws.amazon.com/ses/latest/DeveloperGuide/email-format.html#email-header
     *
     * Applies to actions:
     * - .sendBulkTemplatedEmail()
     * - .sendCustomVerificationEmail()
     * - .sendEmail()
     * - .sendRawEmail()
     * - .sendTemplatedEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFeedbackAddress(value, operator) {
        return this.if(`ses:FeedbackAddress`, value, operator || 'StringLike');
    }
    /**
     * The "From" address of a message.
     *
     * https://docs.aws.amazon.com/ses/latest/DeveloperGuide/email-format.html#email-header
     *
     * Applies to actions:
     * - .sendBounce()
     * - .sendBulkTemplatedEmail()
     * - .sendCustomVerificationEmail()
     * - .sendEmail()
     * - .sendRawEmail()
     * - .sendTemplatedEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromAddress(value, operator) {
        return this.if(`ses:FromAddress`, value, operator || 'StringLike');
    }
    /**
     * The "From" address that is used as the display name of a message.
     *
     * https://docs.aws.amazon.com/ses/latest/DeveloperGuide/email-format.html#email-header
     *
     * Applies to actions:
     * - .sendBulkTemplatedEmail()
     * - .sendCustomVerificationEmail()
     * - .sendEmail()
     * - .sendRawEmail()
     * - .sendTemplatedEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromDisplayName(value, operator) {
        return this.if(`ses:FromDisplayName`, value, operator || 'StringLike');
    }
    /**
     * The recipient addresses of a message, which include the "To", "CC", and "BCC" addresses.
     *
     * https://docs.aws.amazon.com/ses/latest/DeveloperGuide/email-format.html#email-header
     *
     * Applies to actions:
     * - .sendBulkTemplatedEmail()
     * - .sendCustomVerificationEmail()
     * - .sendEmail()
     * - .sendRawEmail()
     * - .sendTemplatedEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRecipients(value, operator) {
        return this.if(`ses:Recipients`, value, operator || 'StringLike');
    }
}
exports.Ses = Ses;
//# sourceMappingURL=data:application/json;base64,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