"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AcmPca = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [acm-pca](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanagerprivatecertificateauthority.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class AcmPca extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [acm-pca](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanagerprivatecertificateauthority.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'acm-pca';
        this.actionList = {
            "CreateCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html",
                "description": "Creates an ACM Private CA and its associated private key and configuration.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateCertificateAuthorityAuditReport": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthorityAuditReport.html",
                "description": "Creates an audit report for an ACM Private CA.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "CreatePermission": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreatePermission.html",
                "description": "Creates a permission for an ACM Private CA.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "DeleteCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeleteCertificateAuthority.html",
                "description": "Deletes an ACM Private CA and its associated private key and configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "DeletePermission": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePermission.html",
                "description": "Deletes a permission for an ACM Private CA.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "DeletePolicy": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePolicy.html",
                "description": "Deletes the policy for an ACM Private CA.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "DescribeCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DescribeCertificateAuthority.html",
                "description": "Returns a list of the configuration and status fields contained in the specified ACM Private CA.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "DescribeCertificateAuthorityAuditReport": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DescribeCertificateAuthorityAuditReport.html",
                "description": "Returns the status and information about an ACM Private CA audit report.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "GetCertificate": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificate.html",
                "description": "Retrieves an ACM Private CA certificate and certificate chain for the certificate authority specified by an ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "GetCertificateAuthorityCertificate": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificateAuthorityCertificate.html",
                "description": "Retrieves an ACM Private CA certificate and certificate chain for the certificate authority specified by an ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "GetCertificateAuthorityCsr": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificateAuthorityCsr.html",
                "description": "Retrieves an ACM Private CA certificate signing request (CSR) for the certificate-authority specified by an ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "GetPolicy": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetPolicy.html",
                "description": "Retrieves the policy on an ACM Private CA.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "ImportCertificateAuthorityCertificate": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ImportCertificateAuthorityCertificate.html",
                "description": "Imports an SSL/TLS certificate into ACM Private CA for use as the CA certificate of an ACM Private CA.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "IssueCertificate": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_IssueCertificate.html",
                "description": "Issues an ACM Private CA certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                },
                "conditions": [
                    "acm-pca:TemplateArn"
                ]
            },
            "ListCertificateAuthorities": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListCertificateAuthorities.html",
                "description": "Retrieves a list of the ACM Private CA certificate authority ARNs, and a summary of the status of each CA in the calling account.",
                "accessLevel": "List"
            },
            "ListPermissions": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListPermissions.html",
                "description": "Lists the permissions that have been applied to the ACM Private CA certificate authority.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListTags.html",
                "description": "Lists the tags that have been applied to the ACM Private CA certificate authority.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "PutPolicy": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_PutPolicy.html",
                "description": "Puts a policy on an ACM Private CA.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "RestoreCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RestoreCertificateAuthority.html",
                "description": "Restores an ACM Private CA from the deleted state to the state it was in when deleted.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "RevokeCertificate": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RevokeCertificate.html",
                "description": "Revokes a certificate issued by an ACM Private CA.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "TagCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_TagCertificateAuthority.html",
                "description": "Adds one or more tags to an ACM Private CA.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UntagCertificateAuthority.html",
                "description": "Remove one or more tags from an ACM Private CA.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html",
                "description": "Updates the configuration of an ACM Private CA.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "certificate-authority": {
                "name": "certificate-authority",
                "url": "https://docs.aws.amazon.com/acm-pca/latest/userguide/authen-overview.html#acm-pca-resources-operations",
                "arn": "arn:${Partition}:acm-pca:${Region}:${Account}:certificate-authority/${CertificateAuthorityId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates an ACM Private CA and its associated private key and configuration.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html
     */
    createCertificateAuthority() {
        this.add('acm-pca:CreateCertificateAuthority');
        return this;
    }
    /**
     * Creates an audit report for an ACM Private CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthorityAuditReport.html
     */
    createCertificateAuthorityAuditReport() {
        this.add('acm-pca:CreateCertificateAuthorityAuditReport');
        return this;
    }
    /**
     * Creates a permission for an ACM Private CA.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreatePermission.html
     */
    createPermission() {
        this.add('acm-pca:CreatePermission');
        return this;
    }
    /**
     * Deletes an ACM Private CA and its associated private key and configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeleteCertificateAuthority.html
     */
    deleteCertificateAuthority() {
        this.add('acm-pca:DeleteCertificateAuthority');
        return this;
    }
    /**
     * Deletes a permission for an ACM Private CA.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePermission.html
     */
    deletePermission() {
        this.add('acm-pca:DeletePermission');
        return this;
    }
    /**
     * Deletes the policy for an ACM Private CA.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePolicy.html
     */
    deletePolicy() {
        this.add('acm-pca:DeletePolicy');
        return this;
    }
    /**
     * Returns a list of the configuration and status fields contained in the specified ACM Private CA.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DescribeCertificateAuthority.html
     */
    describeCertificateAuthority() {
        this.add('acm-pca:DescribeCertificateAuthority');
        return this;
    }
    /**
     * Returns the status and information about an ACM Private CA audit report.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DescribeCertificateAuthorityAuditReport.html
     */
    describeCertificateAuthorityAuditReport() {
        this.add('acm-pca:DescribeCertificateAuthorityAuditReport');
        return this;
    }
    /**
     * Retrieves an ACM Private CA certificate and certificate chain for the certificate authority specified by an ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificate.html
     */
    getCertificate() {
        this.add('acm-pca:GetCertificate');
        return this;
    }
    /**
     * Retrieves an ACM Private CA certificate and certificate chain for the certificate authority specified by an ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificateAuthorityCertificate.html
     */
    getCertificateAuthorityCertificate() {
        this.add('acm-pca:GetCertificateAuthorityCertificate');
        return this;
    }
    /**
     * Retrieves an ACM Private CA certificate signing request (CSR) for the certificate-authority specified by an ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificateAuthorityCsr.html
     */
    getCertificateAuthorityCsr() {
        this.add('acm-pca:GetCertificateAuthorityCsr');
        return this;
    }
    /**
     * Retrieves the policy on an ACM Private CA.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetPolicy.html
     */
    getPolicy() {
        this.add('acm-pca:GetPolicy');
        return this;
    }
    /**
     * Imports an SSL/TLS certificate into ACM Private CA for use as the CA certificate of an ACM Private CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ImportCertificateAuthorityCertificate.html
     */
    importCertificateAuthorityCertificate() {
        this.add('acm-pca:ImportCertificateAuthorityCertificate');
        return this;
    }
    /**
     * Issues an ACM Private CA certificate.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTemplateArn()
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_IssueCertificate.html
     */
    issueCertificate() {
        this.add('acm-pca:IssueCertificate');
        return this;
    }
    /**
     * Retrieves a list of the ACM Private CA certificate authority ARNs, and a summary of the status of each CA in the calling account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListCertificateAuthorities.html
     */
    listCertificateAuthorities() {
        this.add('acm-pca:ListCertificateAuthorities');
        return this;
    }
    /**
     * Lists the permissions that have been applied to the ACM Private CA certificate authority.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListPermissions.html
     */
    listPermissions() {
        this.add('acm-pca:ListPermissions');
        return this;
    }
    /**
     * Lists the tags that have been applied to the ACM Private CA certificate authority.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListTags.html
     */
    listTags() {
        this.add('acm-pca:ListTags');
        return this;
    }
    /**
     * Puts a policy on an ACM Private CA.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_PutPolicy.html
     */
    putPolicy() {
        this.add('acm-pca:PutPolicy');
        return this;
    }
    /**
     * Restores an ACM Private CA from the deleted state to the state it was in when deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RestoreCertificateAuthority.html
     */
    restoreCertificateAuthority() {
        this.add('acm-pca:RestoreCertificateAuthority');
        return this;
    }
    /**
     * Revokes a certificate issued by an ACM Private CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RevokeCertificate.html
     */
    revokeCertificate() {
        this.add('acm-pca:RevokeCertificate');
        return this;
    }
    /**
     * Adds one or more tags to an ACM Private CA.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_TagCertificateAuthority.html
     */
    tagCertificateAuthority() {
        this.add('acm-pca:TagCertificateAuthority');
        return this;
    }
    /**
     * Remove one or more tags from an ACM Private CA.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UntagCertificateAuthority.html
     */
    untagCertificateAuthority() {
        this.add('acm-pca:UntagCertificateAuthority');
        return this;
    }
    /**
     * Updates the configuration of an ACM Private CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html
     */
    updateCertificateAuthority() {
        this.add('acm-pca:UpdateCertificateAuthority');
        return this;
    }
    /**
     * Adds a resource of type certificate-authority to the statement
     *
     * https://docs.aws.amazon.com/acm-pca/latest/userguide/authen-overview.html#acm-pca-resources-operations
     *
     * @param certificateAuthorityId - Identifier for the certificateAuthorityId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCertificateAuthority(certificateAuthorityId, account, region, partition) {
        var arn = 'arn:${Partition}:acm-pca:${Region}:${Account}:certificate-authority/${CertificateAuthorityId}';
        arn = arn.replace('${CertificateAuthorityId}', certificateAuthorityId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters issue certificate requests based on the presence of TemplateArn in the request.
     *
     * Applies to actions:
     * - .issueCertificate()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTemplateArn(value, operator) {
        return this.if(`acm-pca:TemplateArn`, value, operator || 'StringLike');
    }
    /**
     * Filters create requests based on the allowed set of values for each of the tags.
     *
     * Applies to actions:
     * - .createCertificateAuthority()
     * - .tagCertificateAuthority()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag-value associated with the resource.
     *
     * Applies to resource types:
     * - certificate-authority
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters create requests based on the presence of mandatory tags in the request.
     *
     * Applies to actions:
     * - .createCertificateAuthority()
     * - .tagCertificateAuthority()
     * - .untagCertificateAuthority()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.AcmPca = AcmPca;
//# sourceMappingURL=data:application/json;base64,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