"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iot = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [iot](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiot.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iot extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [iot](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiot.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'iot';
        this.actionList = {
            "AcceptCertificateTransfer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AcceptCertificateTransfer.html",
                "description": "Accepts a pending certificate transfer.",
                "accessLevel": "Write"
            },
            "AddThingToBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AddThingToBillingGroup.html",
                "description": "Adds a thing to the specified billing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    }
                }
            },
            "AddThingToThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AddThingToThingGroup.html",
                "description": "Adds a thing to the specified thing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    },
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "AssociateTargetsWithJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AssociateTargetsWithJob.html",
                "description": "Associates a group with a continuous job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    },
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "AttachPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AttachPolicy.html",
                "description": "Attaches a policy to the specified target.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    },
                    "thinggroup": {
                        "required": false
                    }
                }
            },
            "AttachPrincipalPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AttachPrincipalPolicy.html",
                "description": "Attaches the specified policy to the specified principal (certificate or other credential).",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    }
                }
            },
            "AttachSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AttachSecurityProfile.html",
                "description": "Associates a Device Defender security profile with a thing group or with this account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    },
                    "dimension": {
                        "required": false
                    }
                }
            },
            "AttachThingPrincipal": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AttachThingPrincipal.html",
                "description": "Attaches the specified principal to the specified thing.",
                "accessLevel": "Write"
            },
            "CancelAuditMitigationActionsTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CancelAuditMitigationActionsTask.html",
                "description": "Cancels a mitigation action task that is in progress.",
                "accessLevel": "Write"
            },
            "CancelAuditTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CancelAuditTask.html",
                "description": "Cancels an audit that is in progress. The audit can be either scheduled or on-demand.",
                "accessLevel": "Write"
            },
            "CancelCertificateTransfer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CancelCertificateTransfer.html",
                "description": "Cancels a pending transfer for the specified certificate.",
                "accessLevel": "Write"
            },
            "CancelJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CancelJob.html",
                "description": "Cancels a job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "CancelJobExecution": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CancelJobExecution.html",
                "description": "Cancels a job execution on a particular device.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    }
                }
            },
            "ClearDefaultAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ClearDefaultAuthorizer.html",
                "description": "Clears the default authorizer.",
                "accessLevel": "Write"
            },
            "CloseTunnel": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CloseTunnel.html",
                "description": "Closes a tunnel.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "tunnel": {
                        "required": true
                    }
                },
                "conditions": [
                    "iot:Delete"
                ]
            },
            "Connect": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Connect as the specified client",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client": {
                        "required": true
                    }
                }
            },
            "CreateAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateAuthorizer.html",
                "description": "Creates an authorizer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "CreateBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateBillingGroup.html",
                "description": "Creates a billing group.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateCertificateFromCsr": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateCertificateFromCsr.html",
                "description": "Creates an X.509 certificate using the specified certificate signing request.",
                "accessLevel": "Write"
            },
            "CreateDimension": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateDimension.html",
                "description": "Defines a dimension that can be used to to limit the scope of a metric used in a security profile.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dimension": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDynamicThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateDynamicThingGroup.html",
                "description": "Creates a Dynamic Thing Group",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dynamicthinggroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateJob.html",
                "description": "Creates a job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    },
                    "thinggroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateKeysAndCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateKeysAndCertificate.html",
                "description": "Creates a 2048 bit RSA key pair and issues an X.509 certificate using the issued public key.",
                "accessLevel": "Write"
            },
            "CreateMitigationAction": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateMitigationAction.html",
                "description": "Defines an action that can be applied to audit findings by using StartAuditMitigationActionsTask.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mitigationaction": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateOTAUpdate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateOTAUpdate.html",
                "description": "Creates an OTA update job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "otaupdate": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreatePolicy.html",
                "description": "Creates an AWS IoT policy.",
                "accessLevel": "Write"
            },
            "CreatePolicyVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreatePolicyVersion.html",
                "description": "Creates a new version of the specified AWS IoT policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "CreateProvisioningClaim": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningClaim.html",
                "description": "Creates a provisioning claim.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "CreateProvisioningTemplate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningTemplate.html",
                "description": "Creates a fleet provisioning template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "CreateProvisioningTemplateVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningTemplateVersion.html",
                "description": "Creates a new version of a fleet provisioning template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "CreateRoleAlias": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateRoleAlias.html",
                "description": "Creates a role alias.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rolealias": {
                        "required": true
                    }
                }
            },
            "CreateScheduledAudit": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateScheduledAudit.html",
                "description": "Creates a scheduled audit that is run at a specified time interval.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "scheduledaudit": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateSecurityProfile.html",
                "description": "Creates a Device Defender security profile.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    },
                    "dimension": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateStream": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateStream.html",
                "description": "Creates a new AWS IoT stream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThing.html",
                "description": "Creates a thing in the thing registry.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    },
                    "billinggroup": {
                        "required": false
                    }
                }
            },
            "CreateThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThingGroup.html",
                "description": "Creates a thing group.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "thinggroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateThingType": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThingType.html",
                "description": "Creates a new thing type.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "thingtype": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateTopicRule.html",
                "description": "Creates a rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteAccountAuditConfiguration": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteAccountAuditConfiguration.html",
                "description": "Deletes the audit configuration associated with the account.",
                "accessLevel": "Write"
            },
            "DeleteAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteAuthorizer.html",
                "description": "Deletes the specified authorizer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "DeleteBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteBillingGroup.html",
                "description": "Deletes the specified billing group.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    }
                }
            },
            "DeleteCACertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteCACertificate.html",
                "description": "Deletes a registered CA certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cacert": {
                        "required": true
                    }
                }
            },
            "DeleteCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteCertificate.html",
                "description": "Deletes the specified certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cert": {
                        "required": true
                    }
                }
            },
            "DeleteDimension": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteDimension.html",
                "description": "Removes the specified dimension from your AWS account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dimension": {
                        "required": true
                    }
                }
            },
            "DeleteDynamicThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteDynamicThingGroup.html",
                "description": "Deletes the specified Dynamic Thing Group",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dynamicthinggroup": {
                        "required": true
                    }
                }
            },
            "DeleteJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteJob.html",
                "description": "Deletes a job and its related job executions.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "DeleteJobExecution": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteJobExecution.html",
                "description": "Deletes a job execution.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    }
                }
            },
            "DeleteMitigationAction": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteMitigationAction.html",
                "description": "Deletes a defined mitigation action from your AWS account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mitigationaction": {
                        "required": true
                    }
                }
            },
            "DeleteOTAUpdate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteOTAUpdate.html",
                "description": "Deletes an OTA update job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "otaupdate": {
                        "required": true
                    }
                }
            },
            "DeletePolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeletePolicy.html",
                "description": "Deletes the specified policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "DeletePolicyVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeletePolicyVersion.html",
                "description": "Deletes the specified version of the specified policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "DeleteProvisioningTemplate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteProvisioningTemplate.html",
                "description": "Deletes a fleet provisioning template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "DeleteProvisioningTemplateVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteProvisioningTemplateVersion.html",
                "description": "Deletes a fleet provisioning template version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "DeleteRegistrationCode": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteRegistrationCode.html",
                "description": "Deletes a CA certificate registration code.",
                "accessLevel": "Write"
            },
            "DeleteRoleAlias": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteRoleAlias.html",
                "description": "Deletes the specified role alias.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rolealias": {
                        "required": true
                    }
                }
            },
            "DeleteScheduledAudit": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteScheduledAudit.html",
                "description": "Deletes a scheduled audit.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "scheduledaudit": {
                        "required": true
                    }
                }
            },
            "DeleteSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteSecurityProfile.html",
                "description": "Deletes a Device Defender security profile.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    },
                    "dimension": {
                        "required": false
                    }
                }
            },
            "DeleteStream": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteStream.html",
                "description": "Deletes a specified stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DeleteThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThing.html",
                "description": "Deletes the specified thing.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "DeleteThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThingGroup.html",
                "description": "Deletes the specified thing group.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "DeleteThingShadow": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Deletes the specified thing shadow.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "DeleteThingType": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThingType.html",
                "description": "Deletes the specified thing type.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "thingtype": {
                        "required": true
                    }
                }
            },
            "DeleteTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteTopicRule.html",
                "description": "Deletes the specified rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "DeleteV2LoggingLevel": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteV2LoggingLevel.html",
                "description": "Deletes the specified v2 logging level.",
                "accessLevel": "Write"
            },
            "DeprecateThingType": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeprecateThingType.html",
                "description": "Deprecates the specified thing type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thingtype": {
                        "required": true
                    }
                }
            },
            "DescribeAccountAuditConfiguration": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAccountAuditConfiguration.html",
                "description": "Gets information about audit configurations for the account.",
                "accessLevel": "Read"
            },
            "DescribeAuditMitigationActionsTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuditMitigationActionsTask.html",
                "description": "Gets information about an audit mitigation task that is used to apply mitigation actions to a set of audit findings.",
                "accessLevel": "Read"
            },
            "DescribeAuditTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuditTask.html",
                "description": "Gets information about a Device Defender audit.",
                "accessLevel": "Read"
            },
            "DescribeAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuthorizer.html",
                "description": "Describes an authorizer.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "DescribeBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeBillingGroup.html",
                "description": "Gets information about the specified billing group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    }
                }
            },
            "DescribeCACertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeCACertificate.html",
                "description": "Describes a registered CA certificate.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cacert": {
                        "required": true
                    }
                }
            },
            "DescribeCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeCertificate.html",
                "description": "Gets information about the specified certificate.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cert": {
                        "required": true
                    }
                }
            },
            "DescribeDefaultAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeDefaultAuthorizer.html",
                "description": "Describes the default authorizer.",
                "accessLevel": "Read"
            },
            "DescribeDimension": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeDimension.html",
                "description": "Provides details about a dimension that is defined in your AWS account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dimension": {
                        "required": true
                    }
                }
            },
            "DescribeEndpoint": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeEndpoint.html",
                "description": "Returns a unique endpoint specific to the AWS account making the call.",
                "accessLevel": "Read"
            },
            "DescribeEventConfigurations": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeEventConfigurations.html",
                "description": "Returns account event configurations.",
                "accessLevel": "Read"
            },
            "DescribeIndex": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeIndex.html",
                "description": "Gets information about the specified index.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "index": {
                        "required": true
                    }
                }
            },
            "DescribeJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeJob.html",
                "description": "Describes a job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "DescribeJobExecution": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeJobExecution.html",
                "description": "Describes a job execution.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": false
                    },
                    "thing": {
                        "required": false
                    }
                }
            },
            "DescribeMitigationAction": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeMitigationAction.html",
                "description": "Gets information about a mitigation action.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "mitigationaction": {
                        "required": true
                    }
                }
            },
            "DescribeProvisioningTemplate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeProvisioningTemplate.html",
                "description": "Returns information about a fleet provisioning template.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "DescribeProvisioningTemplateVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeProvisioningTemplateVersion.html",
                "description": "Returns information about a fleet provisioning template version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "DescribeRoleAlias": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeRoleAlias.html",
                "description": "Describes a role alias.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rolealias": {
                        "required": true
                    }
                }
            },
            "DescribeScheduledAudit": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeScheduledAudit.html",
                "description": "Gets information about a scheduled audit.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "scheduledaudit": {
                        "required": true
                    }
                }
            },
            "DescribeSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeSecurityProfile.html",
                "description": "Gets information about a Device Defender security profile.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    }
                }
            },
            "DescribeStream": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeStream.html",
                "description": "Gets information about the specified stream.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DescribeThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThing.html",
                "description": "Gets information about the specified thing.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "DescribeThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingGroup.html",
                "description": "Gets information about the specified thing group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "DescribeThingRegistrationTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingRegistrationTask.html",
                "description": "Gets information about the bulk thing registration task.",
                "accessLevel": "Read"
            },
            "DescribeThingType": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingType.html",
                "description": "Gets information about the specified thing type.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "thingtype": {
                        "required": true
                    }
                }
            },
            "DescribeTunnel": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeTunnel.html",
                "description": "Describes a tunnel.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "tunnel": {
                        "required": true
                    }
                }
            },
            "DetachPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DetachPolicy.html",
                "description": "Detaches a policy from the specified target.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    },
                    "thinggroup": {
                        "required": false
                    }
                }
            },
            "DetachPrincipalPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DetachPrincipalPolicy.html",
                "description": "Removes the specified policy from the specified certificate.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    }
                }
            },
            "DetachSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DetachSecurityProfile.html",
                "description": "Disassociates a Device Defender security profile from a thing group or from this account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    },
                    "dimension": {
                        "required": false
                    }
                }
            },
            "DetachThingPrincipal": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DetachThingPrincipal.html",
                "description": "Detaches the specified principal from the specified thing.",
                "accessLevel": "Write"
            },
            "DisableTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DisableTopicRule.html",
                "description": "Disables the specified rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "EnableTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_EnableTopicRule.html",
                "description": "Enables the specified rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "GetCardinality": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetCardinality.html",
                "description": "Get cardinality for IoT fleet index",
                "accessLevel": "Read",
                "resourceTypes": {
                    "index": {
                        "required": true
                    }
                }
            },
            "GetEffectivePolicies": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetEffectivePolicies.html",
                "description": "Gets effective policies.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    }
                }
            },
            "GetIndexingConfiguration": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetIndexingConfiguration.html",
                "description": "Gets current fleet indexing configuration",
                "accessLevel": "Read"
            },
            "GetJobDocument": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetJobDocument.html",
                "description": "Gets a job document.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "GetLoggingOptions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetLoggingOptions.html",
                "description": "Gets the logging options.",
                "accessLevel": "Read"
            },
            "GetOTAUpdate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetOTAUpdate.html",
                "description": "Gets the information about the OTA update job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "otaupdate": {
                        "required": true
                    }
                }
            },
            "GetPendingJobExecutions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetPendingJobExecutions.html",
                "description": "Gets the list of all jobs for a thing that are not in a terminal state.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "GetPercentiles": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetPercentiles.html",
                "description": "Get percentiles for IoT fleet index",
                "accessLevel": "Read",
                "resourceTypes": {
                    "index": {
                        "required": true
                    }
                }
            },
            "GetPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetPolicy.html",
                "description": "Gets information about the specified policy with the policy document of the default version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "GetPolicyVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetPolicyVersion.html",
                "description": "Gets information about the specified policy version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "GetRegistrationCode": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetRegistrationCode.html",
                "description": "Gets a registration code used to register a CA certificate with AWS IoT.",
                "accessLevel": "Read"
            },
            "GetStatistics": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetStatistics.html",
                "description": "Get statistics for IoT fleet index",
                "accessLevel": "Read",
                "resourceTypes": {
                    "index": {
                        "required": true
                    }
                }
            },
            "GetThingShadow": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Gets the thing shadow.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "GetTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetTopicRule.html",
                "description": "Gets information about the specified rule.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "GetV2LoggingOptions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetV2LoggingOptions.html",
                "description": "Gets v2 logging options.",
                "accessLevel": "Read"
            },
            "ListActiveViolations": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListActiveViolations.html",
                "description": "Lists the active violations for a given Device Defender security profile or Thing.",
                "accessLevel": "List",
                "resourceTypes": {
                    "securityprofile": {
                        "required": false
                    },
                    "thing": {
                        "required": false
                    }
                }
            },
            "ListAttachedPolicies": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAttachedPolicies.html",
                "description": "Lists the policies attached to the specified thing group.",
                "accessLevel": "List"
            },
            "ListAuditFindings": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditFindings.html",
                "description": "Lists the findings (results) of a Device Defender audit or of the audits performed during a specified time period.",
                "accessLevel": "List"
            },
            "ListAuditMitigationActionsExecutions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditMitigationActionsExecutions.html",
                "description": "Gets the status of audit mitigation action tasks that were executed.",
                "accessLevel": "List"
            },
            "ListAuditMitigationActionsTasks": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditMitigationActionsTasks.html",
                "description": "Gets a list of audit mitigation action tasks that match the specified filters.",
                "accessLevel": "List"
            },
            "ListAuditTasks": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditTasks.html",
                "description": "Lists the Device Defender audits that have been performed during a given time period.",
                "accessLevel": "List"
            },
            "ListAuthorizers": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuthorizers.html",
                "description": "Lists the authorizers registered in your account.",
                "accessLevel": "List"
            },
            "ListBillingGroups": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListBillingGroups.html",
                "description": "Lists all billing groups.",
                "accessLevel": "List"
            },
            "ListCACertificates": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListCACertificates.html",
                "description": "Lists the CA certificates registered for your AWS account.",
                "accessLevel": "List"
            },
            "ListCertificates": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListCertificates.html",
                "description": "Lists your certificates.",
                "accessLevel": "List"
            },
            "ListCertificatesByCA": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListCertificatesByCA.html",
                "description": "List the device certificates signed by the specified CA certificate.",
                "accessLevel": "List"
            },
            "ListDimensions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListDimensions.html",
                "description": "Lists the dimensions that are defined for your AWS account.",
                "accessLevel": "List"
            },
            "ListIndices": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListIndices.html",
                "description": "Lists all indices for fleet index",
                "accessLevel": "List"
            },
            "ListJobExecutionsForJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobExecutionsForJob.html",
                "description": "Lists the job executions for a job.",
                "accessLevel": "List",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "ListJobExecutionsForThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobExecutionsForThing.html",
                "description": "Lists the job executions for the specified thing.",
                "accessLevel": "List",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobs.html",
                "description": "Lists jobs.",
                "accessLevel": "List"
            },
            "ListMitigationActions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListMitigationActions.html",
                "description": "Gets a list of all mitigation actions that match the specified filter criteria.",
                "accessLevel": "List"
            },
            "ListNamedShadowsForThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListNamedShadowsForThing.html",
                "description": "Lists all named shadows for a given thing.",
                "accessLevel": "List",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "ListOTAUpdates": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListOTAUpdates.html",
                "description": "Lists OTA update jobs in the account.",
                "accessLevel": "List"
            },
            "ListOutgoingCertificates": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListOutgoingCertificates.html",
                "description": "Lists certificates that are being transfered but not yet accepted.",
                "accessLevel": "List"
            },
            "ListPolicies": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicies.html",
                "description": "Lists your policies.",
                "accessLevel": "List"
            },
            "ListPolicyPrincipals": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicyPrincipals.html",
                "description": "Lists the principals associated with the specified policy.",
                "accessLevel": "List"
            },
            "ListPolicyVersions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicyVersions.html",
                "description": "Lists the versions of the specified policy, and identifies the default version.",
                "accessLevel": "List"
            },
            "ListPrincipalPolicies": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListPrincipalPolicies.html",
                "description": "Lists the policies attached to the specified principal. If you use an Amazon Cognito identity, the ID needs to be in Amazon Cognito Identity format.",
                "accessLevel": "List"
            },
            "ListPrincipalThings": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListPrincipalThings.html",
                "description": "Lists the things associated with the specified principal.",
                "accessLevel": "List"
            },
            "ListProvisioningTemplateVersions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListProvisioningTemplateVersions.html",
                "description": "A list of fleet provisioning template versions.",
                "accessLevel": "List",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "ListProvisioningTemplates": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListProvisioningTemplates.html",
                "description": "Lists the fleet provisioning templates in your AWS account.",
                "accessLevel": "List"
            },
            "ListRoleAliases": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListRoleAliases.html",
                "description": "Lists role aliases.",
                "accessLevel": "List"
            },
            "ListScheduledAudits": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListScheduledAudits.html",
                "description": "Lists all of your scheduled audits.",
                "accessLevel": "List"
            },
            "ListSecurityProfiles": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListSecurityProfiles.html",
                "description": "Lists the Device Defender security profiles you have created.",
                "accessLevel": "List",
                "resourceTypes": {
                    "dimension": {
                        "required": false
                    }
                }
            },
            "ListSecurityProfilesForTarget": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListSecurityProfilesForTarget.html",
                "description": "Lists the Device Defender security profiles attached to a target.",
                "accessLevel": "List",
                "resourceTypes": {
                    "thinggroup": {
                        "required": false
                    }
                }
            },
            "ListStreams": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListStreams.html",
                "description": "Lists the streams in your account.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListTagsForResource.html",
                "description": "Lists all tags for a given resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "billinggroup": {
                        "required": false
                    },
                    "dimension": {
                        "required": false
                    },
                    "dynamicthinggroup": {
                        "required": false
                    },
                    "job": {
                        "required": false
                    },
                    "mitigationaction": {
                        "required": false
                    },
                    "otaupdate": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "scheduledaudit": {
                        "required": false
                    },
                    "securityprofile": {
                        "required": false
                    },
                    "stream": {
                        "required": false
                    },
                    "thinggroup": {
                        "required": false
                    },
                    "thingtype": {
                        "required": false
                    }
                }
            },
            "ListTargetsForPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListTargetsForPolicy.html",
                "description": "List targets for the specified policy.",
                "accessLevel": "List",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "ListTargetsForSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListTargetsForSecurityProfile.html",
                "description": "Lists the targets associated with a given Device Defender security profile.",
                "accessLevel": "List",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    }
                }
            },
            "ListThingGroups": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingGroups.html",
                "description": "Lists all thing groups.",
                "accessLevel": "List"
            },
            "ListThingGroupsForThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingGroupsForThing.html",
                "description": "List thing groups to which the specified thing belongs.",
                "accessLevel": "List",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "ListThingPrincipals": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingPrincipals.html",
                "description": "Lists the principals associated with the specified thing.",
                "accessLevel": "List"
            },
            "ListThingRegistrationTaskReports": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingRegistrationTaskReports.html",
                "description": "Lists information about bulk thing registration tasks.",
                "accessLevel": "List"
            },
            "ListThingRegistrationTasks": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingRegistrationTasks.html",
                "description": "Lists bulk thing registration tasks.",
                "accessLevel": "List"
            },
            "ListThingTypes": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingTypes.html",
                "description": "Lists all thing types.",
                "accessLevel": "List"
            },
            "ListThings": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThings.html",
                "description": "Lists all things.",
                "accessLevel": "List"
            },
            "ListThingsInBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingsInBillingGroup.html",
                "description": "Lists all things in the specified billing group.",
                "accessLevel": "List",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    }
                }
            },
            "ListThingsInThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingsInThingGroup.html",
                "description": "Lists all things in the specified thing group.",
                "accessLevel": "List",
                "resourceTypes": {
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "ListTopicRules": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListTopicRules.html",
                "description": "Lists the rules for the specific topic.",
                "accessLevel": "List"
            },
            "ListTunnels": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListTunnels.html",
                "description": "Lists tunnels.",
                "accessLevel": "List"
            },
            "ListV2LoggingLevels": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListV2LoggingLevels.html",
                "description": "Lists the v2 logging levels.",
                "accessLevel": "List"
            },
            "ListViolationEvents": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListViolationEvents.html",
                "description": "Lists the Device Defender security profile violations discovered during the given time period.",
                "accessLevel": "List",
                "resourceTypes": {
                    "securityprofile": {
                        "required": false
                    },
                    "thing": {
                        "required": false
                    }
                }
            },
            "OpenTunnel": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_OpenTunnel.html",
                "description": "Opens a tunnel.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "iot:ThingGroupArn",
                    "iot:TunnelDestinationService"
                ]
            },
            "Publish": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Publish to the specified topic.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "Receive": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Receive from the specified topic.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "RegisterCACertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCACertificate.html",
                "description": "Registers a CA certificate with AWS IoT.",
                "accessLevel": "Write"
            },
            "RegisterCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificate.html",
                "description": "Registers a device certificate with AWS IoT.",
                "accessLevel": "Write"
            },
            "RegisterCertificateWithoutCA": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificateWithoutCA.html",
                "description": "Registers a device certificate with AWS IoT without a registered CA (certificate authority).",
                "accessLevel": "Write"
            },
            "RegisterThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterThing.html",
                "description": "Registers your thing.",
                "accessLevel": "Write"
            },
            "RejectCertificateTransfer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RejectCertificateTransfer.html",
                "description": "Rejects a pending certificate transfer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cert": {
                        "required": true
                    }
                }
            },
            "RemoveThingFromBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RemoveThingFromBillingGroup.html",
                "description": "Removes thing from the specified billing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    }
                }
            },
            "RemoveThingFromThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RemoveThingFromThingGroup.html",
                "description": "Removes thing from the specified thing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    },
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "ReplaceTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ReplaceTopicRule.html",
                "description": "Replaces the specified rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "SearchIndex": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SearchIndex.html",
                "description": "Search IoT fleet index",
                "accessLevel": "Read",
                "resourceTypes": {
                    "index": {
                        "required": true
                    }
                }
            },
            "SetDefaultAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SetDefaultAuthorizer.html",
                "description": "Sets the default authorizer. This will be used if a websocket connection is made without specifying an authorizer.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "SetDefaultPolicyVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SetDefaultPolicyVersion.html",
                "description": "Sets the specified version of the specified policy as the policy's default (operative) version.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "SetLoggingOptions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SetLoggingOptions.html",
                "description": "Sets the logging options.",
                "accessLevel": "Write"
            },
            "SetV2LoggingLevel": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SetV2LoggingLevel.html",
                "description": "Sets the v2 logging level.",
                "accessLevel": "Write"
            },
            "SetV2LoggingOptions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SetV2LoggingOptions.html",
                "description": "Sets the v2 logging options.",
                "accessLevel": "Write"
            },
            "StartAuditMitigationActionsTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_StartAuditMitigationActionsTask.html",
                "description": "Starts a task that applies a set of mitigation actions to the specified target.",
                "accessLevel": "Write"
            },
            "StartNextPendingJobExecution": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_StartNextPendingJobExecution.html",
                "description": "Gets and starts the next pending job execution for a thing.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "StartOnDemandAuditTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_StartOnDemandAuditTask.html",
                "description": "Starts an on-demand Device Defender audit.",
                "accessLevel": "Write"
            },
            "StartThingRegistrationTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_StartThingRegistrationTask.html",
                "description": "Starts a bulk thing registration task.",
                "accessLevel": "Write"
            },
            "StopThingRegistrationTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_StopThingRegistrationTask.html",
                "description": "Stops a bulk thing registration task.",
                "accessLevel": "Write"
            },
            "Subscribe": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Subscribe to the specified TopicFilter.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topicfilter": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "",
                "description": "Tag a specified resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "billinggroup": {
                        "required": false
                    },
                    "dimension": {
                        "required": false
                    },
                    "dynamicthinggroup": {
                        "required": false
                    },
                    "job": {
                        "required": false
                    },
                    "mitigationaction": {
                        "required": false
                    },
                    "otaupdate": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "scheduledaudit": {
                        "required": false
                    },
                    "securityprofile": {
                        "required": false
                    },
                    "stream": {
                        "required": false
                    },
                    "thinggroup": {
                        "required": false
                    },
                    "thingtype": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "TestAuthorization": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_TestAuthorization.html",
                "description": "Test the policies evaluation for group policies",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    }
                }
            },
            "TestInvokeAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_TestInvokeAuthorizer.html",
                "description": "Invoke the specified custom authorizer for testing purposes.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "TransferCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_TransferCertificate.html",
                "description": "Transfers the specified certificate to the specified AWS account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cert": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "",
                "description": "Untag a specified resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "billinggroup": {
                        "required": false
                    },
                    "dimension": {
                        "required": false
                    },
                    "dynamicthinggroup": {
                        "required": false
                    },
                    "job": {
                        "required": false
                    },
                    "mitigationaction": {
                        "required": false
                    },
                    "otaupdate": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "scheduledaudit": {
                        "required": false
                    },
                    "securityprofile": {
                        "required": false
                    },
                    "stream": {
                        "required": false
                    },
                    "thinggroup": {
                        "required": false
                    },
                    "thingtype": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateAccountAuditConfiguration": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateAccountAuditConfiguration.html",
                "description": "Configures or reconfigures the Device Defender audit settings for this account.",
                "accessLevel": "Write"
            },
            "UpdateAuthorizer": {
                "url": "",
                "description": "Updates an authorizer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "UpdateBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateBillingGroup.html",
                "description": "Updates information associated with the specified billing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    }
                }
            },
            "UpdateCACertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateCACertificate.html",
                "description": "Updates a registered CA certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cacert": {
                        "required": true
                    }
                }
            },
            "UpdateCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateCertificate.html",
                "description": "Updates the status of the specified certificate. This operation is idempotent.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cert": {
                        "required": true
                    }
                }
            },
            "UpdateDimension": {
                "url": "",
                "description": "Updates the definition for a dimension.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dimension": {
                        "required": true
                    }
                }
            },
            "UpdateDynamicThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateDynamicThingGroup.html",
                "description": "Updates a Dynamic Thing Group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dynamicthinggroup": {
                        "required": true
                    }
                }
            },
            "UpdateEventConfigurations": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateEventConfigurations.html",
                "description": "Updates event configurations.",
                "accessLevel": "Write"
            },
            "UpdateIndexingConfiguration": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateIndexingConfiguration.html",
                "description": "Updates fleet indexing configuration",
                "accessLevel": "Write"
            },
            "UpdateJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateJob.html",
                "description": "Updates a job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "UpdateJobExecution": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateJobExecution.html",
                "description": "Updates a job execution.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "UpdateMitigationAction": {
                "url": "",
                "description": "Updates the definition for the specified mitigation action.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mitigationaction": {
                        "required": true
                    }
                }
            },
            "UpdateProvisioningTemplate": {
                "url": "",
                "description": "Updates a fleet provisioning template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "UpdateRoleAlias": {
                "url": "",
                "description": "Updates the role alias",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rolealias": {
                        "required": true
                    }
                }
            },
            "UpdateScheduledAudit": {
                "url": "",
                "description": "Updates a scheduled audit, including what checks are performed and how often the audit takes place.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "scheduledaudit": {
                        "required": true
                    }
                }
            },
            "UpdateSecurityProfile": {
                "url": "",
                "description": "Updates a Device Defender security profile.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    },
                    "dimension": {
                        "required": false
                    }
                }
            },
            "UpdateStream": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateStream.html",
                "description": "Updates the data for a stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "UpdateThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThing.html",
                "description": "Updates information associated with the specified thing.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "UpdateThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThingGroup.html",
                "description": "Updates information associated with the specified thing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "UpdateThingGroupsForThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThingGroupsForThing.html",
                "description": "Updates the thing groups to which the thing belongs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    },
                    "thinggroup": {
                        "required": false
                    }
                }
            },
            "UpdateThingShadow": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Updates the thing shadow.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "ValidateSecurityProfileBehaviors": {
                "url": "",
                "description": "Validates a Device Defender security profile behaviors specification.",
                "accessLevel": "Read"
            }
        };
        this.resourceTypes = {
            "client": {
                "name": "client",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-message-broker.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:client/${ClientId}",
                "conditionKeys": []
            },
            "index": {
                "name": "index",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-indexing.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:index/${IndexName}",
                "conditionKeys": []
            },
            "job": {
                "name": "job",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-jobs.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:job/${JobId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "tunnel": {
                "name": "tunnel",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-tunnels.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:tunnel/${TunnelId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "thing": {
                "name": "thing",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/thing-registry.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:thing/${ThingName}",
                "conditionKeys": []
            },
            "thinggroup": {
                "name": "thinggroup",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/thing-groups.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:thinggroup/${ThingGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "billinggroup": {
                "name": "billinggroup",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/billing-groups.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:billinggroup/${BillingGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dynamicthinggroup": {
                "name": "dynamicthinggroup",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/dynamic-thing-groups.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:thinggroup/${ThingGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "thingtype": {
                "name": "thingtype",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/thing-types.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:thingtype/${ThingTypeName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "topic": {
                "name": "topic",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-message-broker.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:topic/${TopicName}",
                "conditionKeys": []
            },
            "topicfilter": {
                "name": "topicfilter",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/topics.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:topicfilter/${TopicFilter}",
                "conditionKeys": []
            },
            "rolealias": {
                "name": "rolealias",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/authorizing-direct-aws.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:rolealias/${RoleAlias}",
                "conditionKeys": []
            },
            "authorizer": {
                "name": "authorizer",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/custom-authorizer.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:authorizer/${AuthorizerName}",
                "conditionKeys": []
            },
            "policy": {
                "name": "policy",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-policies.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:policy/${PolicyName}",
                "conditionKeys": []
            },
            "cert": {
                "name": "cert",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/x509-certs.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:cert/${Certificate}",
                "conditionKeys": []
            },
            "cacert": {
                "name": "cacert",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/x509-certs.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:cacert/${CACertificate}",
                "conditionKeys": []
            },
            "stream": {
                "name": "stream",
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ota-dev.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:stream/${StreamId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "otaupdate": {
                "name": "otaupdate",
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ota-dev.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:otaupdate/${OtaUpdateId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "scheduledaudit": {
                "name": "scheduledaudit",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-audit.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:scheduledaudit/${ScheduleName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "mitigationaction": {
                "name": "mitigationaction",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-mitigation-actions.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:mitigationaction/${MitigationActionName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "securityprofile": {
                "name": "securityprofile",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-detect.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:securityprofile/${SecurityProfileName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dimension": {
                "name": "dimension",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-detect.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:dimension/${DimensionName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "rule": {
                "name": "rule",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-rules.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:rule/${RuleName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "provisioningtemplate": {
                "name": "provisioningtemplate",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/provision-template.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:provisioningtemplate/${ProvisioningTemplate}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Accepts a pending certificate transfer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AcceptCertificateTransfer.html
     */
    acceptCertificateTransfer() {
        this.add('iot:AcceptCertificateTransfer');
        return this;
    }
    /**
     * Adds a thing to the specified billing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AddThingToBillingGroup.html
     */
    addThingToBillingGroup() {
        this.add('iot:AddThingToBillingGroup');
        return this;
    }
    /**
     * Adds a thing to the specified thing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AddThingToThingGroup.html
     */
    addThingToThingGroup() {
        this.add('iot:AddThingToThingGroup');
        return this;
    }
    /**
     * Associates a group with a continuous job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AssociateTargetsWithJob.html
     */
    associateTargetsWithJob() {
        this.add('iot:AssociateTargetsWithJob');
        return this;
    }
    /**
     * Attaches a policy to the specified target.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AttachPolicy.html
     */
    attachPolicy() {
        this.add('iot:AttachPolicy');
        return this;
    }
    /**
     * Attaches the specified policy to the specified principal (certificate or other credential).
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AttachPrincipalPolicy.html
     */
    attachPrincipalPolicy() {
        this.add('iot:AttachPrincipalPolicy');
        return this;
    }
    /**
     * Associates a Device Defender security profile with a thing group or with this account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AttachSecurityProfile.html
     */
    attachSecurityProfile() {
        this.add('iot:AttachSecurityProfile');
        return this;
    }
    /**
     * Attaches the specified principal to the specified thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AttachThingPrincipal.html
     */
    attachThingPrincipal() {
        this.add('iot:AttachThingPrincipal');
        return this;
    }
    /**
     * Cancels a mitigation action task that is in progress.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CancelAuditMitigationActionsTask.html
     */
    cancelAuditMitigationActionsTask() {
        this.add('iot:CancelAuditMitigationActionsTask');
        return this;
    }
    /**
     * Cancels an audit that is in progress. The audit can be either scheduled or on-demand.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CancelAuditTask.html
     */
    cancelAuditTask() {
        this.add('iot:CancelAuditTask');
        return this;
    }
    /**
     * Cancels a pending transfer for the specified certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CancelCertificateTransfer.html
     */
    cancelCertificateTransfer() {
        this.add('iot:CancelCertificateTransfer');
        return this;
    }
    /**
     * Cancels a job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CancelJob.html
     */
    cancelJob() {
        this.add('iot:CancelJob');
        return this;
    }
    /**
     * Cancels a job execution on a particular device.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CancelJobExecution.html
     */
    cancelJobExecution() {
        this.add('iot:CancelJobExecution');
        return this;
    }
    /**
     * Clears the default authorizer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ClearDefaultAuthorizer.html
     */
    clearDefaultAuthorizer() {
        this.add('iot:ClearDefaultAuthorizer');
        return this;
    }
    /**
     * Closes a tunnel.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDelete()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CloseTunnel.html
     */
    closeTunnel() {
        this.add('iot:CloseTunnel');
        return this;
    }
    /**
     * Connect as the specified client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    connect() {
        this.add('iot:Connect');
        return this;
    }
    /**
     * Creates an authorizer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateAuthorizer.html
     */
    createAuthorizer() {
        this.add('iot:CreateAuthorizer');
        return this;
    }
    /**
     * Creates a billing group.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateBillingGroup.html
     */
    createBillingGroup() {
        this.add('iot:CreateBillingGroup');
        return this;
    }
    /**
     * Creates an X.509 certificate using the specified certificate signing request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateCertificateFromCsr.html
     */
    createCertificateFromCsr() {
        this.add('iot:CreateCertificateFromCsr');
        return this;
    }
    /**
     * Defines a dimension that can be used to to limit the scope of a metric used in a security profile.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateDimension.html
     */
    createDimension() {
        this.add('iot:CreateDimension');
        return this;
    }
    /**
     * Creates a Dynamic Thing Group
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateDynamicThingGroup.html
     */
    createDynamicThingGroup() {
        this.add('iot:CreateDynamicThingGroup');
        return this;
    }
    /**
     * Creates a job.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateJob.html
     */
    createJob() {
        this.add('iot:CreateJob');
        return this;
    }
    /**
     * Creates a 2048 bit RSA key pair and issues an X.509 certificate using the issued public key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateKeysAndCertificate.html
     */
    createKeysAndCertificate() {
        this.add('iot:CreateKeysAndCertificate');
        return this;
    }
    /**
     * Defines an action that can be applied to audit findings by using StartAuditMitigationActionsTask.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateMitigationAction.html
     */
    createMitigationAction() {
        this.add('iot:CreateMitigationAction');
        return this;
    }
    /**
     * Creates an OTA update job.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateOTAUpdate.html
     */
    createOTAUpdate() {
        this.add('iot:CreateOTAUpdate');
        return this;
    }
    /**
     * Creates an AWS IoT policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreatePolicy.html
     */
    createPolicy() {
        this.add('iot:CreatePolicy');
        return this;
    }
    /**
     * Creates a new version of the specified AWS IoT policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreatePolicyVersion.html
     */
    createPolicyVersion() {
        this.add('iot:CreatePolicyVersion');
        return this;
    }
    /**
     * Creates a provisioning claim.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningClaim.html
     */
    createProvisioningClaim() {
        this.add('iot:CreateProvisioningClaim');
        return this;
    }
    /**
     * Creates a fleet provisioning template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningTemplate.html
     */
    createProvisioningTemplate() {
        this.add('iot:CreateProvisioningTemplate');
        return this;
    }
    /**
     * Creates a new version of a fleet provisioning template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningTemplateVersion.html
     */
    createProvisioningTemplateVersion() {
        this.add('iot:CreateProvisioningTemplateVersion');
        return this;
    }
    /**
     * Creates a role alias.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateRoleAlias.html
     */
    createRoleAlias() {
        this.add('iot:CreateRoleAlias');
        return this;
    }
    /**
     * Creates a scheduled audit that is run at a specified time interval.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateScheduledAudit.html
     */
    createScheduledAudit() {
        this.add('iot:CreateScheduledAudit');
        return this;
    }
    /**
     * Creates a Device Defender security profile.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateSecurityProfile.html
     */
    createSecurityProfile() {
        this.add('iot:CreateSecurityProfile');
        return this;
    }
    /**
     * Creates a new AWS IoT stream
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateStream.html
     */
    createStream() {
        this.add('iot:CreateStream');
        return this;
    }
    /**
     * Creates a thing in the thing registry.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThing.html
     */
    createThing() {
        this.add('iot:CreateThing');
        return this;
    }
    /**
     * Creates a thing group.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThingGroup.html
     */
    createThingGroup() {
        this.add('iot:CreateThingGroup');
        return this;
    }
    /**
     * Creates a new thing type.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThingType.html
     */
    createThingType() {
        this.add('iot:CreateThingType');
        return this;
    }
    /**
     * Creates a rule.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateTopicRule.html
     */
    createTopicRule() {
        this.add('iot:CreateTopicRule');
        return this;
    }
    /**
     * Deletes the audit configuration associated with the account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteAccountAuditConfiguration.html
     */
    deleteAccountAuditConfiguration() {
        this.add('iot:DeleteAccountAuditConfiguration');
        return this;
    }
    /**
     * Deletes the specified authorizer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteAuthorizer.html
     */
    deleteAuthorizer() {
        this.add('iot:DeleteAuthorizer');
        return this;
    }
    /**
     * Deletes the specified billing group.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteBillingGroup.html
     */
    deleteBillingGroup() {
        this.add('iot:DeleteBillingGroup');
        return this;
    }
    /**
     * Deletes a registered CA certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteCACertificate.html
     */
    deleteCACertificate() {
        this.add('iot:DeleteCACertificate');
        return this;
    }
    /**
     * Deletes the specified certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteCertificate.html
     */
    deleteCertificate() {
        this.add('iot:DeleteCertificate');
        return this;
    }
    /**
     * Removes the specified dimension from your AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteDimension.html
     */
    deleteDimension() {
        this.add('iot:DeleteDimension');
        return this;
    }
    /**
     * Deletes the specified Dynamic Thing Group
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteDynamicThingGroup.html
     */
    deleteDynamicThingGroup() {
        this.add('iot:DeleteDynamicThingGroup');
        return this;
    }
    /**
     * Deletes a job and its related job executions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteJob.html
     */
    deleteJob() {
        this.add('iot:DeleteJob');
        return this;
    }
    /**
     * Deletes a job execution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteJobExecution.html
     */
    deleteJobExecution() {
        this.add('iot:DeleteJobExecution');
        return this;
    }
    /**
     * Deletes a defined mitigation action from your AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteMitigationAction.html
     */
    deleteMitigationAction() {
        this.add('iot:DeleteMitigationAction');
        return this;
    }
    /**
     * Deletes an OTA update job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteOTAUpdate.html
     */
    deleteOTAUpdate() {
        this.add('iot:DeleteOTAUpdate');
        return this;
    }
    /**
     * Deletes the specified policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeletePolicy.html
     */
    deletePolicy() {
        this.add('iot:DeletePolicy');
        return this;
    }
    /**
     * Deletes the specified version of the specified policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeletePolicyVersion.html
     */
    deletePolicyVersion() {
        this.add('iot:DeletePolicyVersion');
        return this;
    }
    /**
     * Deletes a fleet provisioning template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteProvisioningTemplate.html
     */
    deleteProvisioningTemplate() {
        this.add('iot:DeleteProvisioningTemplate');
        return this;
    }
    /**
     * Deletes a fleet provisioning template version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteProvisioningTemplateVersion.html
     */
    deleteProvisioningTemplateVersion() {
        this.add('iot:DeleteProvisioningTemplateVersion');
        return this;
    }
    /**
     * Deletes a CA certificate registration code.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteRegistrationCode.html
     */
    deleteRegistrationCode() {
        this.add('iot:DeleteRegistrationCode');
        return this;
    }
    /**
     * Deletes the specified role alias.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteRoleAlias.html
     */
    deleteRoleAlias() {
        this.add('iot:DeleteRoleAlias');
        return this;
    }
    /**
     * Deletes a scheduled audit.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteScheduledAudit.html
     */
    deleteScheduledAudit() {
        this.add('iot:DeleteScheduledAudit');
        return this;
    }
    /**
     * Deletes a Device Defender security profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteSecurityProfile.html
     */
    deleteSecurityProfile() {
        this.add('iot:DeleteSecurityProfile');
        return this;
    }
    /**
     * Deletes a specified stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteStream.html
     */
    deleteStream() {
        this.add('iot:DeleteStream');
        return this;
    }
    /**
     * Deletes the specified thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThing.html
     */
    deleteThing() {
        this.add('iot:DeleteThing');
        return this;
    }
    /**
     * Deletes the specified thing group.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThingGroup.html
     */
    deleteThingGroup() {
        this.add('iot:DeleteThingGroup');
        return this;
    }
    /**
     * Deletes the specified thing shadow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    deleteThingShadow() {
        this.add('iot:DeleteThingShadow');
        return this;
    }
    /**
     * Deletes the specified thing type.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThingType.html
     */
    deleteThingType() {
        this.add('iot:DeleteThingType');
        return this;
    }
    /**
     * Deletes the specified rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteTopicRule.html
     */
    deleteTopicRule() {
        this.add('iot:DeleteTopicRule');
        return this;
    }
    /**
     * Deletes the specified v2 logging level.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteV2LoggingLevel.html
     */
    deleteV2LoggingLevel() {
        this.add('iot:DeleteV2LoggingLevel');
        return this;
    }
    /**
     * Deprecates the specified thing type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeprecateThingType.html
     */
    deprecateThingType() {
        this.add('iot:DeprecateThingType');
        return this;
    }
    /**
     * Gets information about audit configurations for the account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAccountAuditConfiguration.html
     */
    describeAccountAuditConfiguration() {
        this.add('iot:DescribeAccountAuditConfiguration');
        return this;
    }
    /**
     * Gets information about an audit mitigation task that is used to apply mitigation actions to a set of audit findings.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuditMitigationActionsTask.html
     */
    describeAuditMitigationActionsTask() {
        this.add('iot:DescribeAuditMitigationActionsTask');
        return this;
    }
    /**
     * Gets information about a Device Defender audit.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuditTask.html
     */
    describeAuditTask() {
        this.add('iot:DescribeAuditTask');
        return this;
    }
    /**
     * Describes an authorizer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuthorizer.html
     */
    describeAuthorizer() {
        this.add('iot:DescribeAuthorizer');
        return this;
    }
    /**
     * Gets information about the specified billing group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeBillingGroup.html
     */
    describeBillingGroup() {
        this.add('iot:DescribeBillingGroup');
        return this;
    }
    /**
     * Describes a registered CA certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeCACertificate.html
     */
    describeCACertificate() {
        this.add('iot:DescribeCACertificate');
        return this;
    }
    /**
     * Gets information about the specified certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeCertificate.html
     */
    describeCertificate() {
        this.add('iot:DescribeCertificate');
        return this;
    }
    /**
     * Describes the default authorizer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeDefaultAuthorizer.html
     */
    describeDefaultAuthorizer() {
        this.add('iot:DescribeDefaultAuthorizer');
        return this;
    }
    /**
     * Provides details about a dimension that is defined in your AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeDimension.html
     */
    describeDimension() {
        this.add('iot:DescribeDimension');
        return this;
    }
    /**
     * Returns a unique endpoint specific to the AWS account making the call.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeEndpoint.html
     */
    describeEndpoint() {
        this.add('iot:DescribeEndpoint');
        return this;
    }
    /**
     * Returns account event configurations.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeEventConfigurations.html
     */
    describeEventConfigurations() {
        this.add('iot:DescribeEventConfigurations');
        return this;
    }
    /**
     * Gets information about the specified index.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeIndex.html
     */
    describeIndex() {
        this.add('iot:DescribeIndex');
        return this;
    }
    /**
     * Describes a job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeJob.html
     */
    describeJob() {
        this.add('iot:DescribeJob');
        return this;
    }
    /**
     * Describes a job execution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeJobExecution.html
     */
    describeJobExecution() {
        this.add('iot:DescribeJobExecution');
        return this;
    }
    /**
     * Gets information about a mitigation action.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeMitigationAction.html
     */
    describeMitigationAction() {
        this.add('iot:DescribeMitigationAction');
        return this;
    }
    /**
     * Returns information about a fleet provisioning template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeProvisioningTemplate.html
     */
    describeProvisioningTemplate() {
        this.add('iot:DescribeProvisioningTemplate');
        return this;
    }
    /**
     * Returns information about a fleet provisioning template version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeProvisioningTemplateVersion.html
     */
    describeProvisioningTemplateVersion() {
        this.add('iot:DescribeProvisioningTemplateVersion');
        return this;
    }
    /**
     * Describes a role alias.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeRoleAlias.html
     */
    describeRoleAlias() {
        this.add('iot:DescribeRoleAlias');
        return this;
    }
    /**
     * Gets information about a scheduled audit.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeScheduledAudit.html
     */
    describeScheduledAudit() {
        this.add('iot:DescribeScheduledAudit');
        return this;
    }
    /**
     * Gets information about a Device Defender security profile.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeSecurityProfile.html
     */
    describeSecurityProfile() {
        this.add('iot:DescribeSecurityProfile');
        return this;
    }
    /**
     * Gets information about the specified stream.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeStream.html
     */
    describeStream() {
        this.add('iot:DescribeStream');
        return this;
    }
    /**
     * Gets information about the specified thing.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThing.html
     */
    describeThing() {
        this.add('iot:DescribeThing');
        return this;
    }
    /**
     * Gets information about the specified thing group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingGroup.html
     */
    describeThingGroup() {
        this.add('iot:DescribeThingGroup');
        return this;
    }
    /**
     * Gets information about the bulk thing registration task.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingRegistrationTask.html
     */
    describeThingRegistrationTask() {
        this.add('iot:DescribeThingRegistrationTask');
        return this;
    }
    /**
     * Gets information about the specified thing type.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingType.html
     */
    describeThingType() {
        this.add('iot:DescribeThingType');
        return this;
    }
    /**
     * Describes a tunnel.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeTunnel.html
     */
    describeTunnel() {
        this.add('iot:DescribeTunnel');
        return this;
    }
    /**
     * Detaches a policy from the specified target.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DetachPolicy.html
     */
    detachPolicy() {
        this.add('iot:DetachPolicy');
        return this;
    }
    /**
     * Removes the specified policy from the specified certificate.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DetachPrincipalPolicy.html
     */
    detachPrincipalPolicy() {
        this.add('iot:DetachPrincipalPolicy');
        return this;
    }
    /**
     * Disassociates a Device Defender security profile from a thing group or from this account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DetachSecurityProfile.html
     */
    detachSecurityProfile() {
        this.add('iot:DetachSecurityProfile');
        return this;
    }
    /**
     * Detaches the specified principal from the specified thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DetachThingPrincipal.html
     */
    detachThingPrincipal() {
        this.add('iot:DetachThingPrincipal');
        return this;
    }
    /**
     * Disables the specified rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DisableTopicRule.html
     */
    disableTopicRule() {
        this.add('iot:DisableTopicRule');
        return this;
    }
    /**
     * Enables the specified rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_EnableTopicRule.html
     */
    enableTopicRule() {
        this.add('iot:EnableTopicRule');
        return this;
    }
    /**
     * Get cardinality for IoT fleet index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetCardinality.html
     */
    getCardinality() {
        this.add('iot:GetCardinality');
        return this;
    }
    /**
     * Gets effective policies.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetEffectivePolicies.html
     */
    getEffectivePolicies() {
        this.add('iot:GetEffectivePolicies');
        return this;
    }
    /**
     * Gets current fleet indexing configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetIndexingConfiguration.html
     */
    getIndexingConfiguration() {
        this.add('iot:GetIndexingConfiguration');
        return this;
    }
    /**
     * Gets a job document.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetJobDocument.html
     */
    getJobDocument() {
        this.add('iot:GetJobDocument');
        return this;
    }
    /**
     * Gets the logging options.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetLoggingOptions.html
     */
    getLoggingOptions() {
        this.add('iot:GetLoggingOptions');
        return this;
    }
    /**
     * Gets the information about the OTA update job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetOTAUpdate.html
     */
    getOTAUpdate() {
        this.add('iot:GetOTAUpdate');
        return this;
    }
    /**
     * Gets the list of all jobs for a thing that are not in a terminal state.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetPendingJobExecutions.html
     */
    getPendingJobExecutions() {
        this.add('iot:GetPendingJobExecutions');
        return this;
    }
    /**
     * Get percentiles for IoT fleet index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetPercentiles.html
     */
    getPercentiles() {
        this.add('iot:GetPercentiles');
        return this;
    }
    /**
     * Gets information about the specified policy with the policy document of the default version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetPolicy.html
     */
    getPolicy() {
        this.add('iot:GetPolicy');
        return this;
    }
    /**
     * Gets information about the specified policy version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetPolicyVersion.html
     */
    getPolicyVersion() {
        this.add('iot:GetPolicyVersion');
        return this;
    }
    /**
     * Gets a registration code used to register a CA certificate with AWS IoT.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetRegistrationCode.html
     */
    getRegistrationCode() {
        this.add('iot:GetRegistrationCode');
        return this;
    }
    /**
     * Get statistics for IoT fleet index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetStatistics.html
     */
    getStatistics() {
        this.add('iot:GetStatistics');
        return this;
    }
    /**
     * Gets the thing shadow.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    getThingShadow() {
        this.add('iot:GetThingShadow');
        return this;
    }
    /**
     * Gets information about the specified rule.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetTopicRule.html
     */
    getTopicRule() {
        this.add('iot:GetTopicRule');
        return this;
    }
    /**
     * Gets v2 logging options.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetV2LoggingOptions.html
     */
    getV2LoggingOptions() {
        this.add('iot:GetV2LoggingOptions');
        return this;
    }
    /**
     * Lists the active violations for a given Device Defender security profile or Thing.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListActiveViolations.html
     */
    listActiveViolations() {
        this.add('iot:ListActiveViolations');
        return this;
    }
    /**
     * Lists the policies attached to the specified thing group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAttachedPolicies.html
     */
    listAttachedPolicies() {
        this.add('iot:ListAttachedPolicies');
        return this;
    }
    /**
     * Lists the findings (results) of a Device Defender audit or of the audits performed during a specified time period.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditFindings.html
     */
    listAuditFindings() {
        this.add('iot:ListAuditFindings');
        return this;
    }
    /**
     * Gets the status of audit mitigation action tasks that were executed.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditMitigationActionsExecutions.html
     */
    listAuditMitigationActionsExecutions() {
        this.add('iot:ListAuditMitigationActionsExecutions');
        return this;
    }
    /**
     * Gets a list of audit mitigation action tasks that match the specified filters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditMitigationActionsTasks.html
     */
    listAuditMitigationActionsTasks() {
        this.add('iot:ListAuditMitigationActionsTasks');
        return this;
    }
    /**
     * Lists the Device Defender audits that have been performed during a given time period.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditTasks.html
     */
    listAuditTasks() {
        this.add('iot:ListAuditTasks');
        return this;
    }
    /**
     * Lists the authorizers registered in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuthorizers.html
     */
    listAuthorizers() {
        this.add('iot:ListAuthorizers');
        return this;
    }
    /**
     * Lists all billing groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListBillingGroups.html
     */
    listBillingGroups() {
        this.add('iot:ListBillingGroups');
        return this;
    }
    /**
     * Lists the CA certificates registered for your AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListCACertificates.html
     */
    listCACertificates() {
        this.add('iot:ListCACertificates');
        return this;
    }
    /**
     * Lists your certificates.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListCertificates.html
     */
    listCertificates() {
        this.add('iot:ListCertificates');
        return this;
    }
    /**
     * List the device certificates signed by the specified CA certificate.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListCertificatesByCA.html
     */
    listCertificatesByCA() {
        this.add('iot:ListCertificatesByCA');
        return this;
    }
    /**
     * Lists the dimensions that are defined for your AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListDimensions.html
     */
    listDimensions() {
        this.add('iot:ListDimensions');
        return this;
    }
    /**
     * Lists all indices for fleet index
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListIndices.html
     */
    listIndices() {
        this.add('iot:ListIndices');
        return this;
    }
    /**
     * Lists the job executions for a job.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobExecutionsForJob.html
     */
    listJobExecutionsForJob() {
        this.add('iot:ListJobExecutionsForJob');
        return this;
    }
    /**
     * Lists the job executions for the specified thing.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobExecutionsForThing.html
     */
    listJobExecutionsForThing() {
        this.add('iot:ListJobExecutionsForThing');
        return this;
    }
    /**
     * Lists jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobs.html
     */
    listJobs() {
        this.add('iot:ListJobs');
        return this;
    }
    /**
     * Gets a list of all mitigation actions that match the specified filter criteria.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListMitigationActions.html
     */
    listMitigationActions() {
        this.add('iot:ListMitigationActions');
        return this;
    }
    /**
     * Lists all named shadows for a given thing.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListNamedShadowsForThing.html
     */
    listNamedShadowsForThing() {
        this.add('iot:ListNamedShadowsForThing');
        return this;
    }
    /**
     * Lists OTA update jobs in the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListOTAUpdates.html
     */
    listOTAUpdates() {
        this.add('iot:ListOTAUpdates');
        return this;
    }
    /**
     * Lists certificates that are being transfered but not yet accepted.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListOutgoingCertificates.html
     */
    listOutgoingCertificates() {
        this.add('iot:ListOutgoingCertificates');
        return this;
    }
    /**
     * Lists your policies.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicies.html
     */
    listPolicies() {
        this.add('iot:ListPolicies');
        return this;
    }
    /**
     * Lists the principals associated with the specified policy.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicyPrincipals.html
     */
    listPolicyPrincipals() {
        this.add('iot:ListPolicyPrincipals');
        return this;
    }
    /**
     * Lists the versions of the specified policy, and identifies the default version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicyVersions.html
     */
    listPolicyVersions() {
        this.add('iot:ListPolicyVersions');
        return this;
    }
    /**
     * Lists the policies attached to the specified principal. If you use an Amazon Cognito identity, the ID needs to be in Amazon Cognito Identity format.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListPrincipalPolicies.html
     */
    listPrincipalPolicies() {
        this.add('iot:ListPrincipalPolicies');
        return this;
    }
    /**
     * Lists the things associated with the specified principal.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListPrincipalThings.html
     */
    listPrincipalThings() {
        this.add('iot:ListPrincipalThings');
        return this;
    }
    /**
     * A list of fleet provisioning template versions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListProvisioningTemplateVersions.html
     */
    listProvisioningTemplateVersions() {
        this.add('iot:ListProvisioningTemplateVersions');
        return this;
    }
    /**
     * Lists the fleet provisioning templates in your AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListProvisioningTemplates.html
     */
    listProvisioningTemplates() {
        this.add('iot:ListProvisioningTemplates');
        return this;
    }
    /**
     * Lists role aliases.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListRoleAliases.html
     */
    listRoleAliases() {
        this.add('iot:ListRoleAliases');
        return this;
    }
    /**
     * Lists all of your scheduled audits.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListScheduledAudits.html
     */
    listScheduledAudits() {
        this.add('iot:ListScheduledAudits');
        return this;
    }
    /**
     * Lists the Device Defender security profiles you have created.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListSecurityProfiles.html
     */
    listSecurityProfiles() {
        this.add('iot:ListSecurityProfiles');
        return this;
    }
    /**
     * Lists the Device Defender security profiles attached to a target.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListSecurityProfilesForTarget.html
     */
    listSecurityProfilesForTarget() {
        this.add('iot:ListSecurityProfilesForTarget');
        return this;
    }
    /**
     * Lists the streams in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListStreams.html
     */
    listStreams() {
        this.add('iot:ListStreams');
        return this;
    }
    /**
     * Lists all tags for a given resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('iot:ListTagsForResource');
        return this;
    }
    /**
     * List targets for the specified policy.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListTargetsForPolicy.html
     */
    listTargetsForPolicy() {
        this.add('iot:ListTargetsForPolicy');
        return this;
    }
    /**
     * Lists the targets associated with a given Device Defender security profile.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListTargetsForSecurityProfile.html
     */
    listTargetsForSecurityProfile() {
        this.add('iot:ListTargetsForSecurityProfile');
        return this;
    }
    /**
     * Lists all thing groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingGroups.html
     */
    listThingGroups() {
        this.add('iot:ListThingGroups');
        return this;
    }
    /**
     * List thing groups to which the specified thing belongs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingGroupsForThing.html
     */
    listThingGroupsForThing() {
        this.add('iot:ListThingGroupsForThing');
        return this;
    }
    /**
     * Lists the principals associated with the specified thing.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingPrincipals.html
     */
    listThingPrincipals() {
        this.add('iot:ListThingPrincipals');
        return this;
    }
    /**
     * Lists information about bulk thing registration tasks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingRegistrationTaskReports.html
     */
    listThingRegistrationTaskReports() {
        this.add('iot:ListThingRegistrationTaskReports');
        return this;
    }
    /**
     * Lists bulk thing registration tasks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingRegistrationTasks.html
     */
    listThingRegistrationTasks() {
        this.add('iot:ListThingRegistrationTasks');
        return this;
    }
    /**
     * Lists all thing types.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingTypes.html
     */
    listThingTypes() {
        this.add('iot:ListThingTypes');
        return this;
    }
    /**
     * Lists all things.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThings.html
     */
    listThings() {
        this.add('iot:ListThings');
        return this;
    }
    /**
     * Lists all things in the specified billing group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingsInBillingGroup.html
     */
    listThingsInBillingGroup() {
        this.add('iot:ListThingsInBillingGroup');
        return this;
    }
    /**
     * Lists all things in the specified thing group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingsInThingGroup.html
     */
    listThingsInThingGroup() {
        this.add('iot:ListThingsInThingGroup');
        return this;
    }
    /**
     * Lists the rules for the specific topic.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListTopicRules.html
     */
    listTopicRules() {
        this.add('iot:ListTopicRules');
        return this;
    }
    /**
     * Lists tunnels.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListTunnels.html
     */
    listTunnels() {
        this.add('iot:ListTunnels');
        return this;
    }
    /**
     * Lists the v2 logging levels.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListV2LoggingLevels.html
     */
    listV2LoggingLevels() {
        this.add('iot:ListV2LoggingLevels');
        return this;
    }
    /**
     * Lists the Device Defender security profile violations discovered during the given time period.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListViolationEvents.html
     */
    listViolationEvents() {
        this.add('iot:ListViolationEvents');
        return this;
    }
    /**
     * Opens a tunnel.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifThingGroupArn()
     * - .ifTunnelDestinationService()
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_OpenTunnel.html
     */
    openTunnel() {
        this.add('iot:OpenTunnel');
        return this;
    }
    /**
     * Publish to the specified topic.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    publish() {
        this.add('iot:Publish');
        return this;
    }
    /**
     * Receive from the specified topic.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    receive() {
        this.add('iot:Receive');
        return this;
    }
    /**
     * Registers a CA certificate with AWS IoT.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCACertificate.html
     */
    registerCACertificate() {
        this.add('iot:RegisterCACertificate');
        return this;
    }
    /**
     * Registers a device certificate with AWS IoT.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificate.html
     */
    registerCertificate() {
        this.add('iot:RegisterCertificate');
        return this;
    }
    /**
     * Registers a device certificate with AWS IoT without a registered CA (certificate authority).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificateWithoutCA.html
     */
    registerCertificateWithoutCA() {
        this.add('iot:RegisterCertificateWithoutCA');
        return this;
    }
    /**
     * Registers your thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterThing.html
     */
    registerThing() {
        this.add('iot:RegisterThing');
        return this;
    }
    /**
     * Rejects a pending certificate transfer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RejectCertificateTransfer.html
     */
    rejectCertificateTransfer() {
        this.add('iot:RejectCertificateTransfer');
        return this;
    }
    /**
     * Removes thing from the specified billing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RemoveThingFromBillingGroup.html
     */
    removeThingFromBillingGroup() {
        this.add('iot:RemoveThingFromBillingGroup');
        return this;
    }
    /**
     * Removes thing from the specified thing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RemoveThingFromThingGroup.html
     */
    removeThingFromThingGroup() {
        this.add('iot:RemoveThingFromThingGroup');
        return this;
    }
    /**
     * Replaces the specified rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ReplaceTopicRule.html
     */
    replaceTopicRule() {
        this.add('iot:ReplaceTopicRule');
        return this;
    }
    /**
     * Search IoT fleet index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SearchIndex.html
     */
    searchIndex() {
        this.add('iot:SearchIndex');
        return this;
    }
    /**
     * Sets the default authorizer. This will be used if a websocket connection is made without specifying an authorizer.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SetDefaultAuthorizer.html
     */
    setDefaultAuthorizer() {
        this.add('iot:SetDefaultAuthorizer');
        return this;
    }
    /**
     * Sets the specified version of the specified policy as the policy's default (operative) version.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SetDefaultPolicyVersion.html
     */
    setDefaultPolicyVersion() {
        this.add('iot:SetDefaultPolicyVersion');
        return this;
    }
    /**
     * Sets the logging options.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SetLoggingOptions.html
     */
    setLoggingOptions() {
        this.add('iot:SetLoggingOptions');
        return this;
    }
    /**
     * Sets the v2 logging level.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SetV2LoggingLevel.html
     */
    setV2LoggingLevel() {
        this.add('iot:SetV2LoggingLevel');
        return this;
    }
    /**
     * Sets the v2 logging options.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SetV2LoggingOptions.html
     */
    setV2LoggingOptions() {
        this.add('iot:SetV2LoggingOptions');
        return this;
    }
    /**
     * Starts a task that applies a set of mitigation actions to the specified target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_StartAuditMitigationActionsTask.html
     */
    startAuditMitigationActionsTask() {
        this.add('iot:StartAuditMitigationActionsTask');
        return this;
    }
    /**
     * Gets and starts the next pending job execution for a thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_StartNextPendingJobExecution.html
     */
    startNextPendingJobExecution() {
        this.add('iot:StartNextPendingJobExecution');
        return this;
    }
    /**
     * Starts an on-demand Device Defender audit.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_StartOnDemandAuditTask.html
     */
    startOnDemandAuditTask() {
        this.add('iot:StartOnDemandAuditTask');
        return this;
    }
    /**
     * Starts a bulk thing registration task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_StartThingRegistrationTask.html
     */
    startThingRegistrationTask() {
        this.add('iot:StartThingRegistrationTask');
        return this;
    }
    /**
     * Stops a bulk thing registration task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_StopThingRegistrationTask.html
     */
    stopThingRegistrationTask() {
        this.add('iot:StopThingRegistrationTask');
        return this;
    }
    /**
     * Subscribe to the specified TopicFilter.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    subscribe() {
        this.add('iot:Subscribe');
        return this;
    }
    /**
     * Tag a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     */
    tagResource() {
        this.add('iot:TagResource');
        return this;
    }
    /**
     * Test the policies evaluation for group policies
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_TestAuthorization.html
     */
    testAuthorization() {
        this.add('iot:TestAuthorization');
        return this;
    }
    /**
     * Invoke the specified custom authorizer for testing purposes.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_TestInvokeAuthorizer.html
     */
    testInvokeAuthorizer() {
        this.add('iot:TestInvokeAuthorizer');
        return this;
    }
    /**
     * Transfers the specified certificate to the specified AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_TransferCertificate.html
     */
    transferCertificate() {
        this.add('iot:TransferCertificate');
        return this;
    }
    /**
     * Untag a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     */
    untagResource() {
        this.add('iot:UntagResource');
        return this;
    }
    /**
     * Configures or reconfigures the Device Defender audit settings for this account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateAccountAuditConfiguration.html
     */
    updateAccountAuditConfiguration() {
        this.add('iot:UpdateAccountAuditConfiguration');
        return this;
    }
    /**
     * Updates an authorizer
     *
     * Access Level: Write
     */
    updateAuthorizer() {
        this.add('iot:UpdateAuthorizer');
        return this;
    }
    /**
     * Updates information associated with the specified billing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateBillingGroup.html
     */
    updateBillingGroup() {
        this.add('iot:UpdateBillingGroup');
        return this;
    }
    /**
     * Updates a registered CA certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateCACertificate.html
     */
    updateCACertificate() {
        this.add('iot:UpdateCACertificate');
        return this;
    }
    /**
     * Updates the status of the specified certificate. This operation is idempotent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateCertificate.html
     */
    updateCertificate() {
        this.add('iot:UpdateCertificate');
        return this;
    }
    /**
     * Updates the definition for a dimension.
     *
     * Access Level: Write
     */
    updateDimension() {
        this.add('iot:UpdateDimension');
        return this;
    }
    /**
     * Updates a Dynamic Thing Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateDynamicThingGroup.html
     */
    updateDynamicThingGroup() {
        this.add('iot:UpdateDynamicThingGroup');
        return this;
    }
    /**
     * Updates event configurations.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateEventConfigurations.html
     */
    updateEventConfigurations() {
        this.add('iot:UpdateEventConfigurations');
        return this;
    }
    /**
     * Updates fleet indexing configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateIndexingConfiguration.html
     */
    updateIndexingConfiguration() {
        this.add('iot:UpdateIndexingConfiguration');
        return this;
    }
    /**
     * Updates a job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateJob.html
     */
    updateJob() {
        this.add('iot:UpdateJob');
        return this;
    }
    /**
     * Updates a job execution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateJobExecution.html
     */
    updateJobExecution() {
        this.add('iot:UpdateJobExecution');
        return this;
    }
    /**
     * Updates the definition for the specified mitigation action.
     *
     * Access Level: Write
     */
    updateMitigationAction() {
        this.add('iot:UpdateMitigationAction');
        return this;
    }
    /**
     * Updates a fleet provisioning template.
     *
     * Access Level: Write
     */
    updateProvisioningTemplate() {
        this.add('iot:UpdateProvisioningTemplate');
        return this;
    }
    /**
     * Updates the role alias
     *
     * Access Level: Write
     */
    updateRoleAlias() {
        this.add('iot:UpdateRoleAlias');
        return this;
    }
    /**
     * Updates a scheduled audit, including what checks are performed and how often the audit takes place.
     *
     * Access Level: Write
     */
    updateScheduledAudit() {
        this.add('iot:UpdateScheduledAudit');
        return this;
    }
    /**
     * Updates a Device Defender security profile.
     *
     * Access Level: Write
     */
    updateSecurityProfile() {
        this.add('iot:UpdateSecurityProfile');
        return this;
    }
    /**
     * Updates the data for a stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateStream.html
     */
    updateStream() {
        this.add('iot:UpdateStream');
        return this;
    }
    /**
     * Updates information associated with the specified thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThing.html
     */
    updateThing() {
        this.add('iot:UpdateThing');
        return this;
    }
    /**
     * Updates information associated with the specified thing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThingGroup.html
     */
    updateThingGroup() {
        this.add('iot:UpdateThingGroup');
        return this;
    }
    /**
     * Updates the thing groups to which the thing belongs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThingGroupsForThing.html
     */
    updateThingGroupsForThing() {
        this.add('iot:UpdateThingGroupsForThing');
        return this;
    }
    /**
     * Updates the thing shadow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    updateThingShadow() {
        this.add('iot:UpdateThingShadow');
        return this;
    }
    /**
     * Validates a Device Defender security profile behaviors specification.
     *
     * Access Level: Read
     */
    validateSecurityProfileBehaviors() {
        this.add('iot:ValidateSecurityProfileBehaviors');
        return this;
    }
    /**
     * Adds a resource of type client to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-message-broker.html
     *
     * @param clientId - Identifier for the clientId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onClient(clientId, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:client/${ClientId}';
        arn = arn.replace('${ClientId}', clientId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type index to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-indexing.html
     *
     * @param indexName - Identifier for the indexName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIndex(indexName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:index/${IndexName}';
        arn = arn.replace('${IndexName}', indexName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJob(jobId, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:job/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type tunnel to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-tunnels.html
     *
     * @param tunnelId - Identifier for the tunnelId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTunnel(tunnelId, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:tunnel/${TunnelId}';
        arn = arn.replace('${TunnelId}', tunnelId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type thing to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/thing-registry.html
     *
     * @param thingName - Identifier for the thingName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onThing(thingName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:thing/${ThingName}';
        arn = arn.replace('${ThingName}', thingName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type thinggroup to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/thing-groups.html
     *
     * @param thingGroupName - Identifier for the thingGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onThinggroup(thingGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:thinggroup/${ThingGroupName}';
        arn = arn.replace('${ThingGroupName}', thingGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type billinggroup to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/billing-groups.html
     *
     * @param billingGroupName - Identifier for the billingGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBillinggroup(billingGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:billinggroup/${BillingGroupName}';
        arn = arn.replace('${BillingGroupName}', billingGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dynamicthinggroup to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/dynamic-thing-groups.html
     *
     * @param thingGroupName - Identifier for the thingGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDynamicthinggroup(thingGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:thinggroup/${ThingGroupName}';
        arn = arn.replace('${ThingGroupName}', thingGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type thingtype to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/thing-types.html
     *
     * @param thingTypeName - Identifier for the thingTypeName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onThingtype(thingTypeName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:thingtype/${ThingTypeName}';
        arn = arn.replace('${ThingTypeName}', thingTypeName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type topic to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-message-broker.html
     *
     * @param topicName - Identifier for the topicName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTopic(topicName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:topic/${TopicName}';
        arn = arn.replace('${TopicName}', topicName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type topicfilter to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/topics.html
     *
     * @param topicFilter - Identifier for the topicFilter.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTopicfilter(topicFilter, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:topicfilter/${TopicFilter}';
        arn = arn.replace('${TopicFilter}', topicFilter);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rolealias to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/authorizing-direct-aws.html
     *
     * @param roleAlias - Identifier for the roleAlias.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRolealias(roleAlias, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:rolealias/${RoleAlias}';
        arn = arn.replace('${RoleAlias}', roleAlias);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type authorizer to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/custom-authorizer.html
     *
     * @param authorizerName - Identifier for the authorizerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAuthorizer(authorizerName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:authorizer/${AuthorizerName}';
        arn = arn.replace('${AuthorizerName}', authorizerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type policy to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-policies.html
     *
     * @param policyName - Identifier for the policyName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPolicy(policyName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:policy/${PolicyName}';
        arn = arn.replace('${PolicyName}', policyName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cert to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/x509-certs.html
     *
     * @param certificate - Identifier for the certificate.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCert(certificate, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:cert/${Certificate}';
        arn = arn.replace('${Certificate}', certificate);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cacert to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/x509-certs.html
     *
     * @param cACertificate - Identifier for the cACertificate.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCacert(cACertificate, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:cacert/${CACertificate}';
        arn = arn.replace('${CACertificate}', cACertificate);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ota-dev.html
     *
     * @param streamId - Identifier for the streamId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStream(streamId, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:stream/${StreamId}';
        arn = arn.replace('${StreamId}', streamId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type otaupdate to the statement
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ota-dev.html
     *
     * @param otaUpdateId - Identifier for the otaUpdateId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOtaupdate(otaUpdateId, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:otaupdate/${OtaUpdateId}';
        arn = arn.replace('${OtaUpdateId}', otaUpdateId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type scheduledaudit to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-audit.html
     *
     * @param scheduleName - Identifier for the scheduleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onScheduledaudit(scheduleName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:scheduledaudit/${ScheduleName}';
        arn = arn.replace('${ScheduleName}', scheduleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type mitigationaction to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-mitigation-actions.html
     *
     * @param mitigationActionName - Identifier for the mitigationActionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMitigationaction(mitigationActionName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:mitigationaction/${MitigationActionName}';
        arn = arn.replace('${MitigationActionName}', mitigationActionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type securityprofile to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-detect.html
     *
     * @param securityProfileName - Identifier for the securityProfileName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSecurityprofile(securityProfileName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:securityprofile/${SecurityProfileName}';
        arn = arn.replace('${SecurityProfileName}', securityProfileName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dimension to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-detect.html
     *
     * @param dimensionName - Identifier for the dimensionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDimension(dimensionName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:dimension/${DimensionName}';
        arn = arn.replace('${DimensionName}', dimensionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rule to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-rules.html
     *
     * @param ruleName - Identifier for the ruleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRule(ruleName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:rule/${RuleName}';
        arn = arn.replace('${RuleName}', ruleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type provisioningtemplate to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/provision-template.html
     *
     * @param provisioningTemplate - Identifier for the provisioningTemplate.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProvisioningtemplate(provisioningTemplate, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:provisioningtemplate/${ProvisioningTemplate}';
        arn = arn.replace('${ProvisioningTemplate}', provisioningTemplate);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * A tag key that is present in the request that the user makes to IoT.
     *
     * Applies to actions:
     * - .createBillingGroup()
     * - .createDimension()
     * - .createDynamicThingGroup()
     * - .createJob()
     * - .createMitigationAction()
     * - .createOTAUpdate()
     * - .createScheduledAudit()
     * - .createSecurityProfile()
     * - .createStream()
     * - .createThingGroup()
     * - .createThingType()
     * - .createTopicRule()
     * - .openTunnel()
     * - .tagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * The tag key component of a tag attached to an IoT resource.
     *
     * Applies to resource types:
     * - job
     * - tunnel
     * - thinggroup
     * - billinggroup
     * - dynamicthinggroup
     * - thingtype
     * - stream
     * - otaupdate
     * - scheduledaudit
     * - mitigationaction
     * - securityprofile
     * - dimension
     * - rule
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * The list of all the tag key names associated with the resource in the request.
     *
     * Applies to actions:
     * - .createBillingGroup()
     * - .createDimension()
     * - .createDynamicThingGroup()
     * - .createJob()
     * - .createMitigationAction()
     * - .createOTAUpdate()
     * - .createScheduledAudit()
     * - .createSecurityProfile()
     * - .createStream()
     * - .createThingGroup()
     * - .createThingType()
     * - .createTopicRule()
     * - .openTunnel()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * The flag indicating whether or not to also delete an IoT Tunnel immediately
     *
     * Applies to actions:
     * - .closeTunnel()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifDelete(value) {
        return this.if(`iot:Delete`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * The list of all IoT Thing Group ARNs that the destination IoT Thing belongs to for an IoT Tunnel
     *
     * Applies to actions:
     * - .openTunnel()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifThingGroupArn(value, operator) {
        return this.if(`iot:ThingGroupArn`, value, operator || 'StringLike');
    }
    /**
     * The list of all destination services for an IoT Tunnel
     *
     * Applies to actions:
     * - .openTunnel()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTunnelDestinationService(value, operator) {
        return this.if(`iot:TunnelDestinationService`, value, operator || 'StringLike');
    }
}
exports.Iot = Iot;
//# sourceMappingURL=data:application/json;base64,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