"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iotthingsgraph = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [iotthingsgraph](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotthingsgraph.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iotthingsgraph extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [iotthingsgraph](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotthingsgraph.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'iotthingsgraph';
        this.actionList = {
            "AssociateEntityToThing": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_AssociateEntityToThing.html",
                "description": "Associates a device with a concrete thing that is in the user's registry. A thing can be associated with only one device at a time. If you associate a thing with a new device id, its previous association will be removed.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iot:DescribeThing",
                    "iot:DescribeThingGroup"
                ]
            },
            "CreateFlowTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_CreateFlowTemplate.html",
                "description": "Creates a workflow template. Workflows can be created only in the user's namespace. (The public namespace contains only entities.) The workflow can contain only entities in the specified namespace. The workflow is validated against the entities in the latest version of the user's namespace unless another namespace version is specified in the request.",
                "accessLevel": "Write"
            },
            "CreateSystemInstance": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_CreateSystemInstance.html",
                "description": "Creates an instance of a system with specified configurations and Things.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSystemTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_CreateSystemTemplate.html",
                "description": "Creates a system. The system is validated against the entities in the latest version of the user's namespace unless another namespace version is specified in the request.",
                "accessLevel": "Write"
            },
            "DeleteFlowTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteFlowTemplate.html",
                "description": "Deletes a workflow. Any new system or system instance that contains this workflow will fail to update or deploy. Existing system instances that contain the workflow will continue to run (since they use a snapshot of the workflow taken at the time of deploying the system instance).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Workflow": {
                        "required": true
                    }
                }
            },
            "DeleteNamespace": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteNamespace.html",
                "description": "Deletes the specified namespace. This action deletes all of the entities in the namespace. Delete the systems and flows in the namespace before performing this action.",
                "accessLevel": "Write"
            },
            "DeleteSystemInstance": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteSystemInstance.html",
                "description": "Deletes a system instance. Only instances that have never been deployed, or that have been undeployed from the target can be deleted. Users can create a new system instance that has the same ID as a deleted system instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "SystemInstance": {
                        "required": true
                    }
                }
            },
            "DeleteSystemTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteSystemTemplate.html",
                "description": "Deletes a system. New system instances can't contain the system after its deletion. Existing system instances that contain the system will continue to work because they use a snapshot of the system that is taken when it is deployed.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "System": {
                        "required": true
                    }
                }
            },
            "DeploySystemInstance": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeploySystemInstance.html",
                "description": "Deploys the system instance to the target specified in CreateSystemInstance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "SystemInstance": {
                        "required": true
                    }
                }
            },
            "DeprecateFlowTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeprecateFlowTemplate.html",
                "description": "Deprecates the specified workflow. This action marks the workflow for deletion. Deprecated flows can't be deployed, but existing system instances that use the flow will continue to run.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Workflow": {
                        "required": true
                    }
                }
            },
            "DeprecateSystemTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeprecateSystemTemplate.html",
                "description": "Deprecates the specified system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "System": {
                        "required": true
                    }
                }
            },
            "DescribeNamespace": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DescribeNamespace.html",
                "description": "Gets the latest version of the user's namespace and the public version that it is tracking.",
                "accessLevel": "Read"
            },
            "DissociateEntityFromThing": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DissociateEntityFromThing.html",
                "description": "Dissociates a device entity from a concrete thing. The action takes only the type of the entity that you need to dissociate because only one entity of a particular type can be associated with a thing.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iot:DescribeThing",
                    "iot:DescribeThingGroup"
                ]
            },
            "GetEntities": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetEntities.html",
                "description": "Gets descriptions of the specified entities. Uses the latest version of the user's namespace by default.",
                "accessLevel": "Read"
            },
            "GetFlowTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetFlowTemplate.html",
                "description": "Gets the latest version of the DefinitionDocument and FlowTemplateSummary for the specified workflow.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Workflow": {
                        "required": true
                    }
                }
            },
            "GetFlowTemplateRevisions": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetFlowTemplateRevisions.html",
                "description": "Gets revisions of the specified workflow. Only the last 100 revisions are stored. If the workflow has been deprecated, this action will return revisions that occurred before the deprecation. This action won't work for workflows that have been deleted.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Workflow": {
                        "required": true
                    }
                }
            },
            "GetNamespaceDeletionStatus": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetNamespaceDeletionStatus.html",
                "description": "Gets the status of a namespace deletion task.",
                "accessLevel": "Read"
            },
            "GetSystemInstance": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetSystemInstance.html",
                "description": "Gets a system instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "SystemInstance": {
                        "required": true
                    }
                }
            },
            "GetSystemTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetSystemTemplate.html",
                "description": "Gets a system.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "System": {
                        "required": true
                    }
                }
            },
            "GetSystemTemplateRevisions": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetSystemTemplateRevisions.html",
                "description": "Gets revisions made to the specified system template. Only the previous 100 revisions are stored. If the system has been deprecated, this action will return the revisions that occurred before its deprecation. This action won't work with systems that have been deleted.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "System": {
                        "required": true
                    }
                }
            },
            "GetUploadStatus": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetUploadStatus.html",
                "description": "Gets the status of the specified upload.",
                "accessLevel": "Read"
            },
            "ListFlowExecutionMessages": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_ListFlowExecutionMessages.html",
                "description": "Lists details of a single workflow execution",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_ListTagsForResource.html",
                "description": "Lists all tags for a given resource",
                "accessLevel": "List",
                "resourceTypes": {
                    "SystemInstance": {
                        "required": false
                    }
                }
            },
            "SearchEntities": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchEntities.html",
                "description": "Searches for entities of the specified type. You can search for entities in your namespace and the public namespace that you're tracking.",
                "accessLevel": "Read"
            },
            "SearchFlowExecutions": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchFlowExecutions.html",
                "description": "Searches for workflow executions of a system instance",
                "accessLevel": "Read",
                "resourceTypes": {
                    "SystemInstance": {
                        "required": true
                    }
                }
            },
            "SearchFlowTemplates": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchFlowTemplates.html",
                "description": "Searches for summary information about workflows.",
                "accessLevel": "Read"
            },
            "SearchSystemInstances": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchSystemInstances.html",
                "description": "Searches for system instances in the user's account.",
                "accessLevel": "Read"
            },
            "SearchSystemTemplates": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchSystemTemplates.html",
                "description": "Searches for summary information about systems in the user's account. You can filter by the ID of a workflow to return only systems that use the specified workflow.",
                "accessLevel": "Read"
            },
            "SearchThings": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchThings.html",
                "description": "Searches for things associated with the specified entity. You can search by both device and device model.",
                "accessLevel": "Read"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_TagResource.html",
                "description": "Tag a specified resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "SystemInstance": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UndeploySystemInstance": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UndeploySystemInstance.html",
                "description": "Removes the system instance and associated triggers from the target.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "SystemInstance": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UntagResource.html",
                "description": "Untag a specified resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "SystemInstance": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateFlowTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UpdateFlowTemplate.html",
                "description": "Updates the specified workflow. All deployed systems and system instances that use the workflow will see the changes in the flow when it is redeployed. The workflow can contain only entities in the specified namespace.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Workflow": {
                        "required": true
                    }
                }
            },
            "UpdateSystemTemplate": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UpdateSystemTemplate.html",
                "description": "Updates the specified system. You don't need to run this action after updating a workflow. Any system instance that uses the system will see the changes in the system when it is redeployed.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "System": {
                        "required": true
                    }
                }
            },
            "UploadEntityDefinitions": {
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UploadEntityDefinitions.html",
                "description": "Asynchronously uploads one or more entity definitions to the user's namespace.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "Workflow": {
                "name": "Workflow",
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/ug/iot-tg-models-tdm-iot-workflow.html",
                "arn": "arn:${Partition}:iotthingsgraph:${Region}:${Account}:Workflow/${NamespacePath}",
                "conditionKeys": []
            },
            "System": {
                "name": "System",
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/ug/iot-tg-models-tdm-iot-system.html",
                "arn": "arn:${Partition}:iotthingsgraph:${Region}:${Account}:System/${NamespacePath}",
                "conditionKeys": []
            },
            "SystemInstance": {
                "name": "SystemInstance",
                "url": "https://docs.aws.amazon.com/thingsgraph/latest/ug/iot-tg-models-tdm-iot-sdc-deployconfig.html",
                "arn": "arn:${Partition}:iotthingsgraph:${Region}:${Account}:Deployment/${NamespacePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Associates a device with a concrete thing that is in the user's registry. A thing can be associated with only one device at a time. If you associate a thing with a new device id, its previous association will be removed.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iot:DescribeThing
     * - iot:DescribeThingGroup
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_AssociateEntityToThing.html
     */
    associateEntityToThing() {
        this.add('iotthingsgraph:AssociateEntityToThing');
        return this;
    }
    /**
     * Creates a workflow template. Workflows can be created only in the user's namespace. (The public namespace contains only entities.) The workflow can contain only entities in the specified namespace. The workflow is validated against the entities in the latest version of the user's namespace unless another namespace version is specified in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_CreateFlowTemplate.html
     */
    createFlowTemplate() {
        this.add('iotthingsgraph:CreateFlowTemplate');
        return this;
    }
    /**
     * Creates an instance of a system with specified configurations and Things.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_CreateSystemInstance.html
     */
    createSystemInstance() {
        this.add('iotthingsgraph:CreateSystemInstance');
        return this;
    }
    /**
     * Creates a system. The system is validated against the entities in the latest version of the user's namespace unless another namespace version is specified in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_CreateSystemTemplate.html
     */
    createSystemTemplate() {
        this.add('iotthingsgraph:CreateSystemTemplate');
        return this;
    }
    /**
     * Deletes a workflow. Any new system or system instance that contains this workflow will fail to update or deploy. Existing system instances that contain the workflow will continue to run (since they use a snapshot of the workflow taken at the time of deploying the system instance).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteFlowTemplate.html
     */
    deleteFlowTemplate() {
        this.add('iotthingsgraph:DeleteFlowTemplate');
        return this;
    }
    /**
     * Deletes the specified namespace. This action deletes all of the entities in the namespace. Delete the systems and flows in the namespace before performing this action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteNamespace.html
     */
    deleteNamespace() {
        this.add('iotthingsgraph:DeleteNamespace');
        return this;
    }
    /**
     * Deletes a system instance. Only instances that have never been deployed, or that have been undeployed from the target can be deleted. Users can create a new system instance that has the same ID as a deleted system instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteSystemInstance.html
     */
    deleteSystemInstance() {
        this.add('iotthingsgraph:DeleteSystemInstance');
        return this;
    }
    /**
     * Deletes a system. New system instances can't contain the system after its deletion. Existing system instances that contain the system will continue to work because they use a snapshot of the system that is taken when it is deployed.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeleteSystemTemplate.html
     */
    deleteSystemTemplate() {
        this.add('iotthingsgraph:DeleteSystemTemplate');
        return this;
    }
    /**
     * Deploys the system instance to the target specified in CreateSystemInstance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeploySystemInstance.html
     */
    deploySystemInstance() {
        this.add('iotthingsgraph:DeploySystemInstance');
        return this;
    }
    /**
     * Deprecates the specified workflow. This action marks the workflow for deletion. Deprecated flows can't be deployed, but existing system instances that use the flow will continue to run.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeprecateFlowTemplate.html
     */
    deprecateFlowTemplate() {
        this.add('iotthingsgraph:DeprecateFlowTemplate');
        return this;
    }
    /**
     * Deprecates the specified system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DeprecateSystemTemplate.html
     */
    deprecateSystemTemplate() {
        this.add('iotthingsgraph:DeprecateSystemTemplate');
        return this;
    }
    /**
     * Gets the latest version of the user's namespace and the public version that it is tracking.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DescribeNamespace.html
     */
    describeNamespace() {
        this.add('iotthingsgraph:DescribeNamespace');
        return this;
    }
    /**
     * Dissociates a device entity from a concrete thing. The action takes only the type of the entity that you need to dissociate because only one entity of a particular type can be associated with a thing.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iot:DescribeThing
     * - iot:DescribeThingGroup
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_DissociateEntityFromThing.html
     */
    dissociateEntityFromThing() {
        this.add('iotthingsgraph:DissociateEntityFromThing');
        return this;
    }
    /**
     * Gets descriptions of the specified entities. Uses the latest version of the user's namespace by default.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetEntities.html
     */
    getEntities() {
        this.add('iotthingsgraph:GetEntities');
        return this;
    }
    /**
     * Gets the latest version of the DefinitionDocument and FlowTemplateSummary for the specified workflow.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetFlowTemplate.html
     */
    getFlowTemplate() {
        this.add('iotthingsgraph:GetFlowTemplate');
        return this;
    }
    /**
     * Gets revisions of the specified workflow. Only the last 100 revisions are stored. If the workflow has been deprecated, this action will return revisions that occurred before the deprecation. This action won't work for workflows that have been deleted.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetFlowTemplateRevisions.html
     */
    getFlowTemplateRevisions() {
        this.add('iotthingsgraph:GetFlowTemplateRevisions');
        return this;
    }
    /**
     * Gets the status of a namespace deletion task.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetNamespaceDeletionStatus.html
     */
    getNamespaceDeletionStatus() {
        this.add('iotthingsgraph:GetNamespaceDeletionStatus');
        return this;
    }
    /**
     * Gets a system instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetSystemInstance.html
     */
    getSystemInstance() {
        this.add('iotthingsgraph:GetSystemInstance');
        return this;
    }
    /**
     * Gets a system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetSystemTemplate.html
     */
    getSystemTemplate() {
        this.add('iotthingsgraph:GetSystemTemplate');
        return this;
    }
    /**
     * Gets revisions made to the specified system template. Only the previous 100 revisions are stored. If the system has been deprecated, this action will return the revisions that occurred before its deprecation. This action won't work with systems that have been deleted.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetSystemTemplateRevisions.html
     */
    getSystemTemplateRevisions() {
        this.add('iotthingsgraph:GetSystemTemplateRevisions');
        return this;
    }
    /**
     * Gets the status of the specified upload.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_GetUploadStatus.html
     */
    getUploadStatus() {
        this.add('iotthingsgraph:GetUploadStatus');
        return this;
    }
    /**
     * Lists details of a single workflow execution
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_ListFlowExecutionMessages.html
     */
    listFlowExecutionMessages() {
        this.add('iotthingsgraph:ListFlowExecutionMessages');
        return this;
    }
    /**
     * Lists all tags for a given resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('iotthingsgraph:ListTagsForResource');
        return this;
    }
    /**
     * Searches for entities of the specified type. You can search for entities in your namespace and the public namespace that you're tracking.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchEntities.html
     */
    searchEntities() {
        this.add('iotthingsgraph:SearchEntities');
        return this;
    }
    /**
     * Searches for workflow executions of a system instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchFlowExecutions.html
     */
    searchFlowExecutions() {
        this.add('iotthingsgraph:SearchFlowExecutions');
        return this;
    }
    /**
     * Searches for summary information about workflows.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchFlowTemplates.html
     */
    searchFlowTemplates() {
        this.add('iotthingsgraph:SearchFlowTemplates');
        return this;
    }
    /**
     * Searches for system instances in the user's account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchSystemInstances.html
     */
    searchSystemInstances() {
        this.add('iotthingsgraph:SearchSystemInstances');
        return this;
    }
    /**
     * Searches for summary information about systems in the user's account. You can filter by the ID of a workflow to return only systems that use the specified workflow.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchSystemTemplates.html
     */
    searchSystemTemplates() {
        this.add('iotthingsgraph:SearchSystemTemplates');
        return this;
    }
    /**
     * Searches for things associated with the specified entity. You can search by both device and device model.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_SearchThings.html
     */
    searchThings() {
        this.add('iotthingsgraph:SearchThings');
        return this;
    }
    /**
     * Tag a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_TagResource.html
     */
    tagResource() {
        this.add('iotthingsgraph:TagResource');
        return this;
    }
    /**
     * Removes the system instance and associated triggers from the target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UndeploySystemInstance.html
     */
    undeploySystemInstance() {
        this.add('iotthingsgraph:UndeploySystemInstance');
        return this;
    }
    /**
     * Untag a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UntagResource.html
     */
    untagResource() {
        this.add('iotthingsgraph:UntagResource');
        return this;
    }
    /**
     * Updates the specified workflow. All deployed systems and system instances that use the workflow will see the changes in the flow when it is redeployed. The workflow can contain only entities in the specified namespace.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UpdateFlowTemplate.html
     */
    updateFlowTemplate() {
        this.add('iotthingsgraph:UpdateFlowTemplate');
        return this;
    }
    /**
     * Updates the specified system. You don't need to run this action after updating a workflow. Any system instance that uses the system will see the changes in the system when it is redeployed.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UpdateSystemTemplate.html
     */
    updateSystemTemplate() {
        this.add('iotthingsgraph:UpdateSystemTemplate');
        return this;
    }
    /**
     * Asynchronously uploads one or more entity definitions to the user's namespace.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/APIReference/{APIReferenceDocPage}API_UploadEntityDefinitions.html
     */
    uploadEntityDefinitions() {
        this.add('iotthingsgraph:UploadEntityDefinitions');
        return this;
    }
    /**
     * Adds a resource of type Workflow to the statement
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/ug/iot-tg-models-tdm-iot-workflow.html
     *
     * @param namespacePath - Identifier for the namespacePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWorkflow(namespacePath, account, region, partition) {
        var arn = 'arn:${Partition}:iotthingsgraph:${Region}:${Account}:Workflow/${NamespacePath}';
        arn = arn.replace('${NamespacePath}', namespacePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type System to the statement
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/ug/iot-tg-models-tdm-iot-system.html
     *
     * @param namespacePath - Identifier for the namespacePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSystem(namespacePath, account, region, partition) {
        var arn = 'arn:${Partition}:iotthingsgraph:${Region}:${Account}:System/${NamespacePath}';
        arn = arn.replace('${NamespacePath}', namespacePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type SystemInstance to the statement
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/ug/iot-tg-models-tdm-iot-sdc-deployconfig.html
     *
     * @param namespacePath - Identifier for the namespacePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSystemInstance(namespacePath, account, region, partition) {
        var arn = 'arn:${Partition}:iotthingsgraph:${Region}:${Account}:Deployment/${NamespacePath}';
        arn = arn.replace('${NamespacePath}', namespacePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by a key that is present in the request the user makes to the thingsgraph service.
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/ug/tagging-tg.html/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .createSystemInstance()
     * - .tagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair.
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/ug/tagging-tg.html/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to resource types:
     * - SystemInstance
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the list of all the tag key names present in the request the user makes to the thingsgraph service.
     *
     * https://docs.aws.amazon.com/thingsgraph/latest/ug/tagging-tg.html/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .createSystemInstance()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Iotthingsgraph = Iotthingsgraph;
//# sourceMappingURL=data:application/json;base64,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