"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Opsworks = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [opsworks](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsopsworks.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Opsworks extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [opsworks](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsopsworks.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'opsworks';
        this.actionList = {
            "AssignInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AssignInstance.html",
                "description": "Assign a registered instance to a layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "AssignVolume": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AssignVolume.html",
                "description": "Assigns one of the stack's registered Amazon EBS volumes to a specified instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "AssociateElasticIp": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AssociateElasticIp.html",
                "description": "Associates one of the stack's registered Elastic IP addresses with a specified instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "AttachElasticLoadBalancer": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AttachElasticLoadBalancer.html",
                "description": "Attaches an Elastic Load Balancing load balancer to a specified layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "CloneStack": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CloneStack.html",
                "description": "Creates a clone of a specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "CreateApp": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateApp.html",
                "description": "Creates an app for a specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "CreateDeployment": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateDeployment.html",
                "description": "Runs deployment or stack commands",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "CreateInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateInstance.html",
                "description": "Creates an instance in a specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "CreateLayer": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateLayer.html",
                "description": "Creates a layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "CreateStack": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateStack.html",
                "description": "Creates a new stack",
                "accessLevel": "Write"
            },
            "CreateUserProfile": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateUserProfile.html",
                "description": "Creates a new user profile",
                "accessLevel": "Write"
            },
            "DeleteApp": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteApp.html",
                "description": "Deletes a specified app",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DeleteInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteInstance.html",
                "description": "Deletes a specified instance, which terminates the associated Amazon EC2 instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DeleteLayer": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteLayer.html",
                "description": "Deletes a specified layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DeleteStack": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteStack.html",
                "description": "Deletes a specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DeleteUserProfile": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteUserProfile.html",
                "description": "Deletes a user profile",
                "accessLevel": "Write"
            },
            "DeregisterEcsCluster": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterEcsCluster.html",
                "description": "Deletes a user profile",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DeregisterElasticIp": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterElasticIp.html",
                "description": "Deregisters a specified Elastic IP address",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DeregisterInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterInstance.html",
                "description": "Deregister a registered Amazon EC2 or on-premises instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DeregisterRdsDbInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterRdsDbInstance.html",
                "description": "Deregisters an Amazon RDS instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DeregisterVolume": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterVolume.html",
                "description": "Deregisters an Amazon EBS volume",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeAgentVersions": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeAgentVersions.html",
                "description": "Describes the available AWS OpsWorks agent versions",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeApps": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeApps.html",
                "description": "Requests a description of a specified set of apps",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeCommands": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeCommands.html",
                "description": "Describes the results of specified commands",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeDeployments": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeDeployments.html",
                "description": "Requests a description of a specified set of deployments",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeEcsClusters": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeEcsClusters.html",
                "description": "Describes Amazon ECS clusters that are registered with a stack",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeElasticIps": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeElasticIps.html",
                "description": "Describes Elastic IP addresses",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeElasticLoadBalancers": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeElasticLoadBalancers.html",
                "description": "Describes a stack's Elastic Load Balancing instances",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeInstances": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeInstances.html",
                "description": "Requests a description of a set of instances",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeLayers": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeLayers.html",
                "description": "Requests a description of one or more layers in a specified stack",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeLoadBasedAutoScaling": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeLoadBasedAutoScaling.html",
                "description": "Describes load-based auto scaling configurations for specified layers",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeMyUserProfile": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeMyUserProfile.html",
                "description": "Describes a user's SSH information",
                "accessLevel": "List"
            },
            "DescribePermissions": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribePermissions.html",
                "description": "Describes the permissions for a specified stack",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeRaidArrays": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeRaidArrays.html",
                "description": "Describe an instance's RAID arrays",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeRdsDbInstances": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeRdsDbInstances.html",
                "description": "Describes Amazon RDS instances",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeServiceErrors": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeServiceErrors.html",
                "description": "Describes AWS OpsWorks service errors",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeStackProvisioningParameters": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeStackProvisioningParameters.html",
                "description": "Requests a description of a stack's provisioning parameters",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeStackSummary": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeStackSummary.html",
                "description": "Describes the number of layers and apps in a specified stack, and the number of instances in each state, such as running_setup or online",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeStacks": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeStacks.html",
                "description": "Requests a description of one or more stacks",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeTimeBasedAutoScaling": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeTimeBasedAutoScaling.html",
                "description": "Describes time-based auto scaling configurations for specified instances",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeUserProfiles": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeUserProfiles.html",
                "description": "Describe specified users",
                "accessLevel": "List"
            },
            "DescribeVolumes": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeVolumes.html",
                "description": "Describes an instance's Amazon EBS volumes",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DetachElasticLoadBalancer": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DetachElasticLoadBalancer.html",
                "description": "Detaches a specified Elastic Load Balancing instance from its layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DisassociateElasticIp": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DisassociateElasticIp.html",
                "description": "Disassociates an Elastic IP address from its instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "GetHostnameSuggestion": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_GetHostnameSuggestion.html",
                "description": "Gets a generated host name for the specified layer, based on the current host name theme",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "GrantAccess": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RebootInstance.html",
                "description": "Grants RDP access to a Windows instance for a specified time period",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_ListTags.html",
                "description": "Returns a list of tags that are applied to the specified stack or layer",
                "accessLevel": "List",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "RebootInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RebootInstance.html",
                "description": "Reboots a specified instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "RegisterEcsCluster": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterEcsCluster.html",
                "description": "Registers a specified Amazon ECS cluster with a stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "RegisterElasticIp": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterElasticIp.html",
                "description": "Registers an Elastic IP address with a specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "RegisterInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterInstance.html",
                "description": "Registers instances with a specified stack that were created outside of AWS OpsWorks",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "RegisterRdsDbInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterRdsDbInstance.html",
                "description": "Registers an Amazon RDS instance with a stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "RegisterVolume": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterVolume.html",
                "description": "Registers an Amazon EBS volume with a specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "SetLoadBasedAutoScaling": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_SetLoadBasedAutoScaling.html",
                "description": "Specify the load-based auto scaling configuration for a specified layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "SetPermission": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_SetPermission.html",
                "description": "Specifies a user's permissions",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "SetTimeBasedAutoScaling": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_SetTimeBasedAutoScaling.html",
                "description": "Specify the time-based auto scaling configuration for a specified instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "StartInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StartInstance.html",
                "description": "Starts a specified instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "StartStack": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StartStack.html",
                "description": "Starts a stack's instances",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "StopInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StopInstance.html",
                "description": "Stops a specified instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "StopStack": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StopStack.html",
                "description": "Stops a specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_TagResource.html",
                "description": "Apply tags to a specified stack or layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UnassignInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UnassignInstance.html",
                "description": "Unassigns a registered instance from all of it's layers",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UnassignVolume": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UnassignVolume.html",
                "description": "Unassigns an assigned Amazon EBS volume",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UntagResource.html",
                "description": "Removes tags from a specified stack or layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UpdateApp": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateApp.html",
                "description": "Updates a specified app",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UpdateElasticIp": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateElasticIp.html",
                "description": "Updates a registered Elastic IP address's name",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UpdateInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateInstance.html",
                "description": "Updates a specified instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UpdateLayer": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateLayer.html",
                "description": "Updates a specified layer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UpdateMyUserProfile": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateMyUserProfile.html",
                "description": "Updates a user's SSH public key",
                "accessLevel": "Write"
            },
            "UpdateRdsDbInstance": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateRdsDbInstance.html",
                "description": "Updates an Amazon RDS instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UpdateStack": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateStack.html",
                "description": "Updates a specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "UpdateUserProfile": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateUserProfile.html",
                "description": "Updates a specified user profile",
                "accessLevel": "Permissions management"
            },
            "UpdateVolume": {
                "url": "https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateVolume.html",
                "description": "Updates an Amazon EBS volume's name or mount point",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            }
        };
        this.resourceTypes = {
            "stack": {
                "name": "stack",
                "url": "https://docs.aws.amazon.com/opsworks/latest/userguide/workingstacks.html",
                "arn": "arn:${Partition}:opsworks:${Region}:${Account}:stack/${StackId}/",
                "conditionKeys": []
            }
        };
    }
    /**
     * Assign a registered instance to a layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AssignInstance.html
     */
    assignInstance() {
        this.add('opsworks:AssignInstance');
        return this;
    }
    /**
     * Assigns one of the stack's registered Amazon EBS volumes to a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AssignVolume.html
     */
    assignVolume() {
        this.add('opsworks:AssignVolume');
        return this;
    }
    /**
     * Associates one of the stack's registered Elastic IP addresses with a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AssociateElasticIp.html
     */
    associateElasticIp() {
        this.add('opsworks:AssociateElasticIp');
        return this;
    }
    /**
     * Attaches an Elastic Load Balancing load balancer to a specified layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_AttachElasticLoadBalancer.html
     */
    attachElasticLoadBalancer() {
        this.add('opsworks:AttachElasticLoadBalancer');
        return this;
    }
    /**
     * Creates a clone of a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CloneStack.html
     */
    cloneStack() {
        this.add('opsworks:CloneStack');
        return this;
    }
    /**
     * Creates an app for a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateApp.html
     */
    createApp() {
        this.add('opsworks:CreateApp');
        return this;
    }
    /**
     * Runs deployment or stack commands
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateDeployment.html
     */
    createDeployment() {
        this.add('opsworks:CreateDeployment');
        return this;
    }
    /**
     * Creates an instance in a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateInstance.html
     */
    createInstance() {
        this.add('opsworks:CreateInstance');
        return this;
    }
    /**
     * Creates a layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateLayer.html
     */
    createLayer() {
        this.add('opsworks:CreateLayer');
        return this;
    }
    /**
     * Creates a new stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateStack.html
     */
    createStack() {
        this.add('opsworks:CreateStack');
        return this;
    }
    /**
     * Creates a new user profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_CreateUserProfile.html
     */
    createUserProfile() {
        this.add('opsworks:CreateUserProfile');
        return this;
    }
    /**
     * Deletes a specified app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteApp.html
     */
    deleteApp() {
        this.add('opsworks:DeleteApp');
        return this;
    }
    /**
     * Deletes a specified instance, which terminates the associated Amazon EC2 instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteInstance.html
     */
    deleteInstance() {
        this.add('opsworks:DeleteInstance');
        return this;
    }
    /**
     * Deletes a specified layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteLayer.html
     */
    deleteLayer() {
        this.add('opsworks:DeleteLayer');
        return this;
    }
    /**
     * Deletes a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteStack.html
     */
    deleteStack() {
        this.add('opsworks:DeleteStack');
        return this;
    }
    /**
     * Deletes a user profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeleteUserProfile.html
     */
    deleteUserProfile() {
        this.add('opsworks:DeleteUserProfile');
        return this;
    }
    /**
     * Deletes a user profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterEcsCluster.html
     */
    deregisterEcsCluster() {
        this.add('opsworks:DeregisterEcsCluster');
        return this;
    }
    /**
     * Deregisters a specified Elastic IP address
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterElasticIp.html
     */
    deregisterElasticIp() {
        this.add('opsworks:DeregisterElasticIp');
        return this;
    }
    /**
     * Deregister a registered Amazon EC2 or on-premises instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterInstance.html
     */
    deregisterInstance() {
        this.add('opsworks:DeregisterInstance');
        return this;
    }
    /**
     * Deregisters an Amazon RDS instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterRdsDbInstance.html
     */
    deregisterRdsDbInstance() {
        this.add('opsworks:DeregisterRdsDbInstance');
        return this;
    }
    /**
     * Deregisters an Amazon EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DeregisterVolume.html
     */
    deregisterVolume() {
        this.add('opsworks:DeregisterVolume');
        return this;
    }
    /**
     * Describes the available AWS OpsWorks agent versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeAgentVersions.html
     */
    describeAgentVersions() {
        this.add('opsworks:DescribeAgentVersions');
        return this;
    }
    /**
     * Requests a description of a specified set of apps
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeApps.html
     */
    describeApps() {
        this.add('opsworks:DescribeApps');
        return this;
    }
    /**
     * Describes the results of specified commands
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeCommands.html
     */
    describeCommands() {
        this.add('opsworks:DescribeCommands');
        return this;
    }
    /**
     * Requests a description of a specified set of deployments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeDeployments.html
     */
    describeDeployments() {
        this.add('opsworks:DescribeDeployments');
        return this;
    }
    /**
     * Describes Amazon ECS clusters that are registered with a stack
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeEcsClusters.html
     */
    describeEcsClusters() {
        this.add('opsworks:DescribeEcsClusters');
        return this;
    }
    /**
     * Describes Elastic IP addresses
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeElasticIps.html
     */
    describeElasticIps() {
        this.add('opsworks:DescribeElasticIps');
        return this;
    }
    /**
     * Describes a stack's Elastic Load Balancing instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeElasticLoadBalancers.html
     */
    describeElasticLoadBalancers() {
        this.add('opsworks:DescribeElasticLoadBalancers');
        return this;
    }
    /**
     * Requests a description of a set of instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeInstances.html
     */
    describeInstances() {
        this.add('opsworks:DescribeInstances');
        return this;
    }
    /**
     * Requests a description of one or more layers in a specified stack
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeLayers.html
     */
    describeLayers() {
        this.add('opsworks:DescribeLayers');
        return this;
    }
    /**
     * Describes load-based auto scaling configurations for specified layers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeLoadBasedAutoScaling.html
     */
    describeLoadBasedAutoScaling() {
        this.add('opsworks:DescribeLoadBasedAutoScaling');
        return this;
    }
    /**
     * Describes a user's SSH information
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeMyUserProfile.html
     */
    describeMyUserProfile() {
        this.add('opsworks:DescribeMyUserProfile');
        return this;
    }
    /**
     * Describes the permissions for a specified stack
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribePermissions.html
     */
    describePermissions() {
        this.add('opsworks:DescribePermissions');
        return this;
    }
    /**
     * Describe an instance's RAID arrays
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeRaidArrays.html
     */
    describeRaidArrays() {
        this.add('opsworks:DescribeRaidArrays');
        return this;
    }
    /**
     * Describes Amazon RDS instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeRdsDbInstances.html
     */
    describeRdsDbInstances() {
        this.add('opsworks:DescribeRdsDbInstances');
        return this;
    }
    /**
     * Describes AWS OpsWorks service errors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeServiceErrors.html
     */
    describeServiceErrors() {
        this.add('opsworks:DescribeServiceErrors');
        return this;
    }
    /**
     * Requests a description of a stack's provisioning parameters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeStackProvisioningParameters.html
     */
    describeStackProvisioningParameters() {
        this.add('opsworks:DescribeStackProvisioningParameters');
        return this;
    }
    /**
     * Describes the number of layers and apps in a specified stack, and the number of instances in each state, such as running_setup or online
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeStackSummary.html
     */
    describeStackSummary() {
        this.add('opsworks:DescribeStackSummary');
        return this;
    }
    /**
     * Requests a description of one or more stacks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeStacks.html
     */
    describeStacks() {
        this.add('opsworks:DescribeStacks');
        return this;
    }
    /**
     * Describes time-based auto scaling configurations for specified instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeTimeBasedAutoScaling.html
     */
    describeTimeBasedAutoScaling() {
        this.add('opsworks:DescribeTimeBasedAutoScaling');
        return this;
    }
    /**
     * Describe specified users
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeUserProfiles.html
     */
    describeUserProfiles() {
        this.add('opsworks:DescribeUserProfiles');
        return this;
    }
    /**
     * Describes an instance's Amazon EBS volumes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DescribeVolumes.html
     */
    describeVolumes() {
        this.add('opsworks:DescribeVolumes');
        return this;
    }
    /**
     * Detaches a specified Elastic Load Balancing instance from its layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DetachElasticLoadBalancer.html
     */
    detachElasticLoadBalancer() {
        this.add('opsworks:DetachElasticLoadBalancer');
        return this;
    }
    /**
     * Disassociates an Elastic IP address from its instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_DisassociateElasticIp.html
     */
    disassociateElasticIp() {
        this.add('opsworks:DisassociateElasticIp');
        return this;
    }
    /**
     * Gets a generated host name for the specified layer, based on the current host name theme
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_GetHostnameSuggestion.html
     */
    getHostnameSuggestion() {
        this.add('opsworks:GetHostnameSuggestion');
        return this;
    }
    /**
     * Grants RDP access to a Windows instance for a specified time period
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RebootInstance.html
     */
    grantAccess() {
        this.add('opsworks:GrantAccess');
        return this;
    }
    /**
     * Returns a list of tags that are applied to the specified stack or layer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_ListTags.html
     */
    listTags() {
        this.add('opsworks:ListTags');
        return this;
    }
    /**
     * Reboots a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RebootInstance.html
     */
    rebootInstance() {
        this.add('opsworks:RebootInstance');
        return this;
    }
    /**
     * Registers a specified Amazon ECS cluster with a stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterEcsCluster.html
     */
    registerEcsCluster() {
        this.add('opsworks:RegisterEcsCluster');
        return this;
    }
    /**
     * Registers an Elastic IP address with a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterElasticIp.html
     */
    registerElasticIp() {
        this.add('opsworks:RegisterElasticIp');
        return this;
    }
    /**
     * Registers instances with a specified stack that were created outside of AWS OpsWorks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterInstance.html
     */
    registerInstance() {
        this.add('opsworks:RegisterInstance');
        return this;
    }
    /**
     * Registers an Amazon RDS instance with a stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterRdsDbInstance.html
     */
    registerRdsDbInstance() {
        this.add('opsworks:RegisterRdsDbInstance');
        return this;
    }
    /**
     * Registers an Amazon EBS volume with a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_RegisterVolume.html
     */
    registerVolume() {
        this.add('opsworks:RegisterVolume');
        return this;
    }
    /**
     * Specify the load-based auto scaling configuration for a specified layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_SetLoadBasedAutoScaling.html
     */
    setLoadBasedAutoScaling() {
        this.add('opsworks:SetLoadBasedAutoScaling');
        return this;
    }
    /**
     * Specifies a user's permissions
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_SetPermission.html
     */
    setPermission() {
        this.add('opsworks:SetPermission');
        return this;
    }
    /**
     * Specify the time-based auto scaling configuration for a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_SetTimeBasedAutoScaling.html
     */
    setTimeBasedAutoScaling() {
        this.add('opsworks:SetTimeBasedAutoScaling');
        return this;
    }
    /**
     * Starts a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StartInstance.html
     */
    startInstance() {
        this.add('opsworks:StartInstance');
        return this;
    }
    /**
     * Starts a stack's instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StartStack.html
     */
    startStack() {
        this.add('opsworks:StartStack');
        return this;
    }
    /**
     * Stops a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StopInstance.html
     */
    stopInstance() {
        this.add('opsworks:StopInstance');
        return this;
    }
    /**
     * Stops a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_StopStack.html
     */
    stopStack() {
        this.add('opsworks:StopStack');
        return this;
    }
    /**
     * Apply tags to a specified stack or layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('opsworks:TagResource');
        return this;
    }
    /**
     * Unassigns a registered instance from all of it's layers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UnassignInstance.html
     */
    unassignInstance() {
        this.add('opsworks:UnassignInstance');
        return this;
    }
    /**
     * Unassigns an assigned Amazon EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UnassignVolume.html
     */
    unassignVolume() {
        this.add('opsworks:UnassignVolume');
        return this;
    }
    /**
     * Removes tags from a specified stack or layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('opsworks:UntagResource');
        return this;
    }
    /**
     * Updates a specified app
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateApp.html
     */
    updateApp() {
        this.add('opsworks:UpdateApp');
        return this;
    }
    /**
     * Updates a registered Elastic IP address's name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateElasticIp.html
     */
    updateElasticIp() {
        this.add('opsworks:UpdateElasticIp');
        return this;
    }
    /**
     * Updates a specified instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateInstance.html
     */
    updateInstance() {
        this.add('opsworks:UpdateInstance');
        return this;
    }
    /**
     * Updates a specified layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateLayer.html
     */
    updateLayer() {
        this.add('opsworks:UpdateLayer');
        return this;
    }
    /**
     * Updates a user's SSH public key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateMyUserProfile.html
     */
    updateMyUserProfile() {
        this.add('opsworks:UpdateMyUserProfile');
        return this;
    }
    /**
     * Updates an Amazon RDS instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateRdsDbInstance.html
     */
    updateRdsDbInstance() {
        this.add('opsworks:UpdateRdsDbInstance');
        return this;
    }
    /**
     * Updates a specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateStack.html
     */
    updateStack() {
        this.add('opsworks:UpdateStack');
        return this;
    }
    /**
     * Updates a specified user profile
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateUserProfile.html
     */
    updateUserProfile() {
        this.add('opsworks:UpdateUserProfile');
        return this;
    }
    /**
     * Updates an Amazon EBS volume's name or mount point
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opsworks/latest/APIReference/API_UpdateVolume.html
     */
    updateVolume() {
        this.add('opsworks:UpdateVolume');
        return this;
    }
    /**
     * Adds a resource of type stack to the statement
     *
     * https://docs.aws.amazon.com/opsworks/latest/userguide/workingstacks.html
     *
     * @param stackId - Identifier for the stackId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onStack(stackId, account, region, partition) {
        var arn = 'arn:${Partition}:opsworks:${Region}:${Account}:stack/${StackId}/';
        arn = arn.replace('${StackId}', stackId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Opsworks = Opsworks;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzb3Bzd29ya3MuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhd3NvcHN3b3Jrcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxzQ0FBb0U7QUFFcEU7Ozs7R0FJRztBQUNILE1BQWEsUUFBUyxTQUFRLHdCQUFlO0lBb3NCM0M7Ozs7T0FJRztJQUNILFlBQWEsR0FBWTtRQUN2QixLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUF6c0JOLGtCQUFhLEdBQUcsVUFBVSxDQUFDO1FBQ3hCLGVBQVUsR0FBWTtZQUM5QixnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLHlDQUF5QztnQkFDeEQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLGtGQUFrRjtnQkFDakcsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLHlGQUF5RjtnQkFDeEcsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCwyQkFBMkIsRUFBRTtnQkFDM0IsS0FBSyxFQUFFLDZGQUE2RjtnQkFDcEcsYUFBYSxFQUFFLHVFQUF1RTtnQkFDdEYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsYUFBYSxFQUFFLHNDQUFzQztnQkFDckQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsS0FBSyxFQUFFLDZFQUE2RTtnQkFDcEYsYUFBYSxFQUFFLHNDQUFzQztnQkFDckQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLG1DQUFtQztnQkFDbEQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLDBDQUEwQztnQkFDekQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsYUFBYSxFQUFFLGlCQUFpQjtnQkFDaEMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsYUFBYSxFQUFFLHFCQUFxQjtnQkFDcEMsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLDRCQUE0QjtnQkFDM0MsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsS0FBSyxFQUFFLDZFQUE2RTtnQkFDcEYsYUFBYSxFQUFFLHlCQUF5QjtnQkFDeEMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLG1GQUFtRjtnQkFDbEcsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsYUFBYSxFQUFFLDJCQUEyQjtnQkFDMUMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLCtFQUErRTtnQkFDdEYsYUFBYSxFQUFFLDJCQUEyQjtnQkFDMUMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLHdCQUF3QjtnQkFDdkMsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLHdCQUF3QjtnQkFDdkMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLDRDQUE0QztnQkFDM0QsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLDREQUE0RDtnQkFDM0UsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLDJGQUEyRjtnQkFDbEcsYUFBYSxFQUFFLG9DQUFvQztnQkFDbkQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLGtDQUFrQztnQkFDakQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLHFEQUFxRDtnQkFDcEUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLG1EQUFtRDtnQkFDbEUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLDZDQUE2QztnQkFDNUQsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLDBEQUEwRDtnQkFDekUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLGdFQUFnRTtnQkFDL0UsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLGdDQUFnQztnQkFDL0MsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCw4QkFBOEIsRUFBRTtnQkFDOUIsS0FBSyxFQUFFLGdHQUFnRztnQkFDdkcsYUFBYSxFQUFFLHNEQUFzRDtnQkFDckUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLDhDQUE4QztnQkFDN0QsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLG1FQUFtRTtnQkFDbEYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCw4QkFBOEIsRUFBRTtnQkFDOUIsS0FBSyxFQUFFLGdHQUFnRztnQkFDdkcsYUFBYSxFQUFFLHVFQUF1RTtnQkFDdEYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLG9DQUFvQztnQkFDbkQsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLGlEQUFpRDtnQkFDaEUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLG9DQUFvQztnQkFDbkQsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx3QkFBd0IsRUFBRTtnQkFDeEIsS0FBSyxFQUFFLDBGQUEwRjtnQkFDakcsYUFBYSxFQUFFLGdDQUFnQztnQkFDL0MsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLHVDQUF1QztnQkFDdEQsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxxQ0FBcUMsRUFBRTtnQkFDckMsS0FBSyxFQUFFLHVHQUF1RztnQkFDOUcsYUFBYSxFQUFFLDZEQUE2RDtnQkFDNUUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLDBJQUEwSTtnQkFDekosYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLDhDQUE4QztnQkFDN0QsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCw4QkFBOEIsRUFBRTtnQkFDOUIsS0FBSyxFQUFFLGdHQUFnRztnQkFDdkcsYUFBYSxFQUFFLDBFQUEwRTtnQkFDekYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLDBCQUEwQjtnQkFDekMsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxpQkFBaUIsRUFBRTtnQkFDakIsS0FBSyxFQUFFLG1GQUFtRjtnQkFDMUYsYUFBYSxFQUFFLDRDQUE0QztnQkFDM0QsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCwyQkFBMkIsRUFBRTtnQkFDM0IsS0FBSyxFQUFFLDZGQUE2RjtnQkFDcEcsYUFBYSxFQUFFLHFFQUFxRTtnQkFDcEYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLHVEQUF1RDtnQkFDdEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLDBGQUEwRjtnQkFDekcsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLHFFQUFxRTtnQkFDcEYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxVQUFVLEVBQUU7Z0JBQ1YsS0FBSyxFQUFFLDRFQUE0RTtnQkFDbkYsYUFBYSxFQUFFLHlFQUF5RTtnQkFDeEYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLDhCQUE4QjtnQkFDN0MsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLHVEQUF1RDtnQkFDdEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLHdEQUF3RDtnQkFDdkUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLHNGQUFzRjtnQkFDckcsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLCtDQUErQztnQkFDOUQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLHVEQUF1RDtnQkFDdEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLDJGQUEyRjtnQkFDbEcsYUFBYSxFQUFFLHlFQUF5RTtnQkFDeEYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLGlGQUFpRjtnQkFDeEYsYUFBYSxFQUFFLGdDQUFnQztnQkFDL0MsYUFBYSxFQUFFLHdCQUF3QjtnQkFDdkMsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELHlCQUF5QixFQUFFO2dCQUN6QixLQUFLLEVBQUUsMkZBQTJGO2dCQUNsRyxhQUFhLEVBQUUsNEVBQTRFO2dCQUMzRixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGVBQWUsRUFBRTtnQkFDZixLQUFLLEVBQUUsaUZBQWlGO2dCQUN4RixhQUFhLEVBQUUsNkJBQTZCO2dCQUM1QyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELFlBQVksRUFBRTtnQkFDWixLQUFLLEVBQUUsOEVBQThFO2dCQUNyRixhQUFhLEVBQUUsNEJBQTRCO2dCQUMzQyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGNBQWMsRUFBRTtnQkFDZCxLQUFLLEVBQUUsZ0ZBQWdGO2dCQUN2RixhQUFhLEVBQUUsNEJBQTRCO2dCQUMzQyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELFdBQVcsRUFBRTtnQkFDWCxLQUFLLEVBQUUsNkVBQTZFO2dCQUNwRixhQUFhLEVBQUUseUJBQXlCO2dCQUN4QyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGFBQWEsRUFBRTtnQkFDYixLQUFLLEVBQUUsK0VBQStFO2dCQUN0RixhQUFhLEVBQUUsMENBQTBDO2dCQUN6RCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGtCQUFrQixFQUFFO2dCQUNsQixLQUFLLEVBQUUsb0ZBQW9GO2dCQUMzRixhQUFhLEVBQUUseURBQXlEO2dCQUN4RSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGdCQUFnQixFQUFFO2dCQUNoQixLQUFLLEVBQUUsa0ZBQWtGO2dCQUN6RixhQUFhLEVBQUUseUNBQXlDO2dCQUN4RCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGVBQWUsRUFBRTtnQkFDZixLQUFLLEVBQUUsaUZBQWlGO2dCQUN4RixhQUFhLEVBQUUsOENBQThDO2dCQUM3RCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELFdBQVcsRUFBRTtnQkFDWCxLQUFLLEVBQUUsNkVBQTZFO2dCQUNwRixhQUFhLEVBQUUseUJBQXlCO2dCQUN4QyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGlCQUFpQixFQUFFO2dCQUNqQixLQUFLLEVBQUUsbUZBQW1GO2dCQUMxRixhQUFhLEVBQUUsZ0RBQWdEO2dCQUMvRCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGdCQUFnQixFQUFFO2dCQUNoQixLQUFLLEVBQUUsa0ZBQWtGO2dCQUN6RixhQUFhLEVBQUUsOEJBQThCO2dCQUM3QyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGFBQWEsRUFBRTtnQkFDYixLQUFLLEVBQUUsK0VBQStFO2dCQUN0RixhQUFhLEVBQUUsMkJBQTJCO2dCQUMxQyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELHFCQUFxQixFQUFFO2dCQUNyQixLQUFLLEVBQUUsdUZBQXVGO2dCQUM5RixhQUFhLEVBQUUsaUNBQWlDO2dCQUNoRCxhQUFhLEVBQUUsT0FBTzthQUN2QjtZQUNELHFCQUFxQixFQUFFO2dCQUNyQixLQUFLLEVBQUUsdUZBQXVGO2dCQUM5RixhQUFhLEVBQUUsZ0NBQWdDO2dCQUMvQyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELGFBQWEsRUFBRTtnQkFDYixLQUFLLEVBQUUsK0VBQStFO2dCQUN0RixhQUFhLEVBQUUsMkJBQTJCO2dCQUMxQyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7YUFDRjtZQUNELG1CQUFtQixFQUFFO2dCQUNuQixLQUFLLEVBQUUscUZBQXFGO2dCQUM1RixhQUFhLEVBQUUsa0NBQWtDO2dCQUNqRCxhQUFhLEVBQUUsd0JBQXdCO2FBQ3hDO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSxnRkFBZ0Y7Z0JBQ3ZGLGFBQWEsRUFBRSxvREFBb0Q7Z0JBQ25FLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjthQUNGO1NBQ0YsQ0FBQztRQUNRLGtCQUFhLEdBQWtCO1lBQ3ZDLE9BQU8sRUFBRTtnQkFDUCxNQUFNLEVBQUUsT0FBTztnQkFDZixLQUFLLEVBQUUsMEVBQTBFO2dCQUNqRixLQUFLLEVBQUUsa0VBQWtFO2dCQUN6RSxlQUFlLEVBQUUsRUFBRTthQUNwQjtTQUNGLENBQUM7SUFTRixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO1FBQy9DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFVBQVU7UUFDZixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksU0FBUztRQUNkLElBQUksQ0FBQyxHQUFHLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUMvQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksU0FBUztRQUNkLElBQUksQ0FBQyxHQUFHLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUMvQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNEJBQTRCO1FBQ2pDLElBQUksQ0FBQyxHQUFHLENBQUMsdUNBQXVDLENBQUMsQ0FBQztRQUNsRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxJQUFJLENBQUMsR0FBRyxDQUFDLHVDQUF1QyxDQUFDLENBQUM7UUFDbEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxHQUFHLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLElBQUksQ0FBQyxHQUFHLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUM1QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1DQUFtQztRQUN4QyxJQUFJLENBQUMsR0FBRyxDQUFDLDhDQUE4QyxDQUFDLENBQUM7UUFDekQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO1FBQ2xELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxHQUFHLENBQUMsb0NBQW9DLENBQUMsQ0FBQztRQUMvQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsR0FBRyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksUUFBUTtRQUNiLElBQUksQ0FBQyxHQUFHLENBQUMsbUJBQW1CLENBQUMsQ0FBQztRQUM5QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxHQUFHLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUM3QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxVQUFVO1FBQ2YsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFNBQVM7UUFDZCxJQUFJLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxTQUFTO1FBQ2QsSUFBSSxDQUFDLEdBQUcsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEdBQUcsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLE9BQU8sQ0FBQyxPQUFlLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDbkYsSUFBSSxHQUFHLEdBQUcsa0VBQWtFLENBQUM7UUFDN0UsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ3pDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0NBQ0Y7QUEza0RELDRCQTJrREMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY3Rpb25zLCBQb2xpY3lTdGF0ZW1lbnQsIFJlc291cmNlVHlwZXMgfSBmcm9tIFwiLi4vc2hhcmVkXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtvcHN3b3Jrc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYXdzb3Bzd29ya3MuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBPcHN3b3JrcyBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ29wc3dvcmtzJztcbiAgcHJvdGVjdGVkIGFjdGlvbkxpc3Q6IEFjdGlvbnMgPSB7XG4gICAgXCJBc3NpZ25JbnN0YW5jZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Bc3NpZ25JbnN0YW5jZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQXNzaWduIGEgcmVnaXN0ZXJlZCBpbnN0YW5jZSB0byBhIGxheWVyXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJBc3NpZ25Wb2x1bWVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQXNzaWduVm9sdW1lLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJBc3NpZ25zIG9uZSBvZiB0aGUgc3RhY2sncyByZWdpc3RlcmVkIEFtYXpvbiBFQlMgdm9sdW1lcyB0byBhIHNwZWNpZmllZCBpbnN0YW5jZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQXNzb2NpYXRlRWxhc3RpY0lwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Fzc29jaWF0ZUVsYXN0aWNJcC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQXNzb2NpYXRlcyBvbmUgb2YgdGhlIHN0YWNrJ3MgcmVnaXN0ZXJlZCBFbGFzdGljIElQIGFkZHJlc3NlcyB3aXRoIGEgc3BlY2lmaWVkIGluc3RhbmNlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJBdHRhY2hFbGFzdGljTG9hZEJhbGFuY2VyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0F0dGFjaEVsYXN0aWNMb2FkQmFsYW5jZXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkF0dGFjaGVzIGFuIEVsYXN0aWMgTG9hZCBCYWxhbmNpbmcgbG9hZCBiYWxhbmNlciB0byBhIHNwZWNpZmllZCBsYXllclwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQ2xvbmVTdGFja1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DbG9uZVN0YWNrLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGEgY2xvbmUgb2YgYSBzcGVjaWZpZWQgc3RhY2tcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZUFwcFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVBcHAuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYW4gYXBwIGZvciBhIHNwZWNpZmllZCBzdGFja1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQ3JlYXRlRGVwbG95bWVudFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVEZXBsb3ltZW50Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSdW5zIGRlcGxveW1lbnQgb3Igc3RhY2sgY29tbWFuZHNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZUluc3RhbmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZUluc3RhbmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGFuIGluc3RhbmNlIGluIGEgc3BlY2lmaWVkIHN0YWNrXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJDcmVhdGVMYXllclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVMYXllci5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIGxheWVyXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJDcmVhdGVTdGFja1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVTdGFjay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIG5ldyBzdGFja1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiQ3JlYXRlVXNlclByb2ZpbGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlVXNlclByb2ZpbGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYSBuZXcgdXNlciBwcm9maWxlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJEZWxldGVBcHBcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlQXBwLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIGEgc3BlY2lmaWVkIGFwcFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlSW5zdGFuY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlSW5zdGFuY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgYSBzcGVjaWZpZWQgaW5zdGFuY2UsIHdoaWNoIHRlcm1pbmF0ZXMgdGhlIGFzc29jaWF0ZWQgQW1hem9uIEVDMiBpbnN0YW5jZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlTGF5ZXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlTGF5ZXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgYSBzcGVjaWZpZWQgbGF5ZXJcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZVN0YWNrXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVN0YWNrLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIGEgc3BlY2lmaWVkIHN0YWNrXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVVc2VyUHJvZmlsZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVVc2VyUHJvZmlsZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyBhIHVzZXIgcHJvZmlsZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiRGVyZWdpc3RlckVjc0NsdXN0ZXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVyZWdpc3RlckVjc0NsdXN0ZXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgYSB1c2VyIHByb2ZpbGVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlcmVnaXN0ZXJFbGFzdGljSXBcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVyZWdpc3RlckVsYXN0aWNJcC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVyZWdpc3RlcnMgYSBzcGVjaWZpZWQgRWxhc3RpYyBJUCBhZGRyZXNzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXJlZ2lzdGVySW5zdGFuY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVyZWdpc3Rlckluc3RhbmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZXJlZ2lzdGVyIGEgcmVnaXN0ZXJlZCBBbWF6b24gRUMyIG9yIG9uLXByZW1pc2VzIGluc3RhbmNlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXJlZ2lzdGVyUmRzRGJJbnN0YW5jZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXJlZ2lzdGVyUmRzRGJJbnN0YW5jZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVyZWdpc3RlcnMgYW4gQW1hem9uIFJEUyBpbnN0YW5jZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVyZWdpc3RlclZvbHVtZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXJlZ2lzdGVyVm9sdW1lLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZXJlZ2lzdGVycyBhbiBBbWF6b24gRUJTIHZvbHVtZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVBZ2VudFZlcnNpb25zXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQWdlbnRWZXJzaW9ucy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmVzIHRoZSBhdmFpbGFibGUgQVdTIE9wc1dvcmtzIGFnZW50IHZlcnNpb25zXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlQXBwc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUFwcHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJlcXVlc3RzIGEgZGVzY3JpcHRpb24gb2YgYSBzcGVjaWZpZWQgc2V0IG9mIGFwcHNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVDb21tYW5kc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUNvbW1hbmRzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZXNjcmliZXMgdGhlIHJlc3VsdHMgb2Ygc3BlY2lmaWVkIGNvbW1hbmRzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlRGVwbG95bWVudHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVEZXBsb3ltZW50cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmVxdWVzdHMgYSBkZXNjcmlwdGlvbiBvZiBhIHNwZWNpZmllZCBzZXQgb2YgZGVwbG95bWVudHNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVFY3NDbHVzdGVyc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUVjc0NsdXN0ZXJzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZXNjcmliZXMgQW1hem9uIEVDUyBjbHVzdGVycyB0aGF0IGFyZSByZWdpc3RlcmVkIHdpdGggYSBzdGFja1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZUVsYXN0aWNJcHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVFbGFzdGljSXBzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZXNjcmliZXMgRWxhc3RpYyBJUCBhZGRyZXNzZXNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVFbGFzdGljTG9hZEJhbGFuY2Vyc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUVsYXN0aWNMb2FkQmFsYW5jZXJzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZXNjcmliZXMgYSBzdGFjaydzIEVsYXN0aWMgTG9hZCBCYWxhbmNpbmcgaW5zdGFuY2VzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlSW5zdGFuY2VzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlSW5zdGFuY2VzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXF1ZXN0cyBhIGRlc2NyaXB0aW9uIG9mIGEgc2V0IG9mIGluc3RhbmNlc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZUxheWVyc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUxheWVycy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmVxdWVzdHMgYSBkZXNjcmlwdGlvbiBvZiBvbmUgb3IgbW9yZSBsYXllcnMgaW4gYSBzcGVjaWZpZWQgc3RhY2tcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVMb2FkQmFzZWRBdXRvU2NhbGluZ1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUxvYWRCYXNlZEF1dG9TY2FsaW5nLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZXNjcmliZXMgbG9hZC1iYXNlZCBhdXRvIHNjYWxpbmcgY29uZmlndXJhdGlvbnMgZm9yIHNwZWNpZmllZCBsYXllcnNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVNeVVzZXJQcm9maWxlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlTXlVc2VyUHJvZmlsZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmVzIGEgdXNlcidzIFNTSCBpbmZvcm1hdGlvblwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJEZXNjcmliZVBlcm1pc3Npb25zXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlUGVybWlzc2lvbnMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlc2NyaWJlcyB0aGUgcGVybWlzc2lvbnMgZm9yIGEgc3BlY2lmaWVkIHN0YWNrXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlUmFpZEFycmF5c1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVJhaWRBcnJheXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlc2NyaWJlIGFuIGluc3RhbmNlJ3MgUkFJRCBhcnJheXNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVSZHNEYkluc3RhbmNlc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVJkc0RiSW5zdGFuY2VzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZXNjcmliZXMgQW1hem9uIFJEUyBpbnN0YW5jZXNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVTZXJ2aWNlRXJyb3JzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlU2VydmljZUVycm9ycy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmVzIEFXUyBPcHNXb3JrcyBzZXJ2aWNlIGVycm9yc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZVN0YWNrUHJvdmlzaW9uaW5nUGFyYW1ldGVyc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVN0YWNrUHJvdmlzaW9uaW5nUGFyYW1ldGVycy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmVxdWVzdHMgYSBkZXNjcmlwdGlvbiBvZiBhIHN0YWNrJ3MgcHJvdmlzaW9uaW5nIHBhcmFtZXRlcnNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVTdGFja1N1bW1hcnlcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVTdGFja1N1bW1hcnkuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlc2NyaWJlcyB0aGUgbnVtYmVyIG9mIGxheWVycyBhbmQgYXBwcyBpbiBhIHNwZWNpZmllZCBzdGFjaywgYW5kIHRoZSBudW1iZXIgb2YgaW5zdGFuY2VzIGluIGVhY2ggc3RhdGUsIHN1Y2ggYXMgcnVubmluZ19zZXR1cCBvciBvbmxpbmVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVTdGFja3NcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVTdGFja3MuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJlcXVlc3RzIGEgZGVzY3JpcHRpb24gb2Ygb25lIG9yIG1vcmUgc3RhY2tzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlVGltZUJhc2VkQXV0b1NjYWxpbmdcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVUaW1lQmFzZWRBdXRvU2NhbGluZy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmVzIHRpbWUtYmFzZWQgYXV0byBzY2FsaW5nIGNvbmZpZ3VyYXRpb25zIGZvciBzcGVjaWZpZWQgaW5zdGFuY2VzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlVXNlclByb2ZpbGVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlVXNlclByb2ZpbGVzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZXNjcmliZSBzcGVjaWZpZWQgdXNlcnNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiRGVzY3JpYmVWb2x1bWVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlVm9sdW1lcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmVzIGFuIGluc3RhbmNlJ3MgQW1hem9uIEVCUyB2b2x1bWVzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRldGFjaEVsYXN0aWNMb2FkQmFsYW5jZXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGV0YWNoRWxhc3RpY0xvYWRCYWxhbmNlci5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGV0YWNoZXMgYSBzcGVjaWZpZWQgRWxhc3RpYyBMb2FkIEJhbGFuY2luZyBpbnN0YW5jZSBmcm9tIGl0cyBsYXllclwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGlzYXNzb2NpYXRlRWxhc3RpY0lwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rpc2Fzc29jaWF0ZUVsYXN0aWNJcC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGlzYXNzb2NpYXRlcyBhbiBFbGFzdGljIElQIGFkZHJlc3MgZnJvbSBpdHMgaW5zdGFuY2VcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldEhvc3RuYW1lU3VnZ2VzdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRIb3N0bmFtZVN1Z2dlc3Rpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdldHMgYSBnZW5lcmF0ZWQgaG9zdCBuYW1lIGZvciB0aGUgc3BlY2lmaWVkIGxheWVyLCBiYXNlZCBvbiB0aGUgY3VycmVudCBob3N0IG5hbWUgdGhlbWVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR3JhbnRBY2Nlc3NcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVib290SW5zdGFuY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBSRFAgYWNjZXNzIHRvIGEgV2luZG93cyBpbnN0YW5jZSBmb3IgYSBzcGVjaWZpZWQgdGltZSBwZXJpb2RcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkxpc3RUYWdzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RUYWdzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGEgbGlzdCBvZiB0YWdzIHRoYXQgYXJlIGFwcGxpZWQgdG8gdGhlIHNwZWNpZmllZCBzdGFjayBvciBsYXllclwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJSZWJvb3RJbnN0YW5jZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZWJvb3RJbnN0YW5jZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmVib290cyBhIHNwZWNpZmllZCBpbnN0YW5jZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiUmVnaXN0ZXJFY3NDbHVzdGVyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlZ2lzdGVyRWNzQ2x1c3Rlci5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmVnaXN0ZXJzIGEgc3BlY2lmaWVkIEFtYXpvbiBFQ1MgY2x1c3RlciB3aXRoIGEgc3RhY2tcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlJlZ2lzdGVyRWxhc3RpY0lwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlZ2lzdGVyRWxhc3RpY0lwLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZWdpc3RlcnMgYW4gRWxhc3RpYyBJUCBhZGRyZXNzIHdpdGggYSBzcGVjaWZpZWQgc3RhY2tcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlJlZ2lzdGVySW5zdGFuY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVnaXN0ZXJJbnN0YW5jZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmVnaXN0ZXJzIGluc3RhbmNlcyB3aXRoIGEgc3BlY2lmaWVkIHN0YWNrIHRoYXQgd2VyZSBjcmVhdGVkIG91dHNpZGUgb2YgQVdTIE9wc1dvcmtzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJSZWdpc3RlclJkc0RiSW5zdGFuY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVnaXN0ZXJSZHNEYkluc3RhbmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZWdpc3RlcnMgYW4gQW1hem9uIFJEUyBpbnN0YW5jZSB3aXRoIGEgc3RhY2tcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlJlZ2lzdGVyVm9sdW1lXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlZ2lzdGVyVm9sdW1lLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZWdpc3RlcnMgYW4gQW1hem9uIEVCUyB2b2x1bWUgd2l0aCBhIHNwZWNpZmllZCBzdGFja1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiU2V0TG9hZEJhc2VkQXV0b1NjYWxpbmdcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2V0TG9hZEJhc2VkQXV0b1NjYWxpbmcuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlNwZWNpZnkgdGhlIGxvYWQtYmFzZWQgYXV0byBzY2FsaW5nIGNvbmZpZ3VyYXRpb24gZm9yIGEgc3BlY2lmaWVkIGxheWVyXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJTZXRQZXJtaXNzaW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1NldFBlcm1pc3Npb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlNwZWNpZmllcyBhIHVzZXIncyBwZXJtaXNzaW9uc1wiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlBlcm1pc3Npb25zIG1hbmFnZW1lbnRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJTZXRUaW1lQmFzZWRBdXRvU2NhbGluZ1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TZXRUaW1lQmFzZWRBdXRvU2NhbGluZy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiU3BlY2lmeSB0aGUgdGltZS1iYXNlZCBhdXRvIHNjYWxpbmcgY29uZmlndXJhdGlvbiBmb3IgYSBzcGVjaWZpZWQgaW5zdGFuY2VcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlN0YXJ0SW5zdGFuY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RhcnRJbnN0YW5jZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiU3RhcnRzIGEgc3BlY2lmaWVkIGluc3RhbmNlXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJTdGFydFN0YWNrXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0YXJ0U3RhY2suaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlN0YXJ0cyBhIHN0YWNrJ3MgaW5zdGFuY2VzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJTdG9wSW5zdGFuY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RvcEluc3RhbmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJTdG9wcyBhIHNwZWNpZmllZCBpbnN0YW5jZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiU3RvcFN0YWNrXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0b3BTdGFjay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiU3RvcHMgYSBzcGVjaWZpZWQgc3RhY2tcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlRhZ1Jlc291cmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1RhZ1Jlc291cmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJBcHBseSB0YWdzIHRvIGEgc3BlY2lmaWVkIHN0YWNrIG9yIGxheWVyXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVbmFzc2lnbkluc3RhbmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VuYXNzaWduSW5zdGFuY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVuYXNzaWducyBhIHJlZ2lzdGVyZWQgaW5zdGFuY2UgZnJvbSBhbGwgb2YgaXQncyBsYXllcnNcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVuYXNzaWduVm9sdW1lXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VuYXNzaWduVm9sdW1lLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJVbmFzc2lnbnMgYW4gYXNzaWduZWQgQW1hem9uIEVCUyB2b2x1bWVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVudGFnUmVzb3VyY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVW50YWdSZXNvdXJjZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmVtb3ZlcyB0YWdzIGZyb20gYSBzcGVjaWZpZWQgc3RhY2sgb3IgbGF5ZXJcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZUFwcFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVBcHAuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVwZGF0ZXMgYSBzcGVjaWZpZWQgYXBwXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVFbGFzdGljSXBcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlRWxhc3RpY0lwLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJVcGRhdGVzIGEgcmVnaXN0ZXJlZCBFbGFzdGljIElQIGFkZHJlc3MncyBuYW1lXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVJbnN0YW5jZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVJbnN0YW5jZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyBhIHNwZWNpZmllZCBpbnN0YW5jZVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInN0YWNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlTGF5ZXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlTGF5ZXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVwZGF0ZXMgYSBzcGVjaWZpZWQgbGF5ZXJcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZU15VXNlclByb2ZpbGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlTXlVc2VyUHJvZmlsZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyBhIHVzZXIncyBTU0ggcHVibGljIGtleVwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiVXBkYXRlUmRzRGJJbnN0YW5jZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVSZHNEYkluc3RhbmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJVcGRhdGVzIGFuIEFtYXpvbiBSRFMgaW5zdGFuY2VcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZVN0YWNrXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZVN0YWNrLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJVcGRhdGVzIGEgc3BlY2lmaWVkIHN0YWNrXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3RhY2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVVc2VyUHJvZmlsZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVVc2VyUHJvZmlsZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyBhIHNwZWNpZmllZCB1c2VyIHByb2ZpbGVcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XCJcbiAgICB9LFxuICAgIFwiVXBkYXRlVm9sdW1lXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZVZvbHVtZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyBhbiBBbWF6b24gRUJTIHZvbHVtZSdzIG5hbWUgb3IgbW91bnQgcG9pbnRcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzdGFja1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICB9O1xuICBwcm90ZWN0ZWQgcmVzb3VyY2VUeXBlczogUmVzb3VyY2VUeXBlcyA9IHtcbiAgICBcInN0YWNrXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcInN0YWNrXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvdXNlcmd1aWRlL3dvcmtpbmdzdGFja3MuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259Om9wc3dvcmtzOiR7UmVnaW9ufToke0FjY291bnR9OnN0YWNrLyR7U3RhY2tJZH0vXCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW11cbiAgICB9XG4gIH07XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbb3Bzd29ya3NdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2F3c29wc3dvcmtzLmh0bWwpLlxuICAgKlxuICAgKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAgICovXG4gIGNvbnN0cnVjdG9yIChzaWQ/OiBzdHJpbmcpIHtcbiAgICBzdXBlcihzaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFzc2lnbiBhIHJlZ2lzdGVyZWQgaW5zdGFuY2UgdG8gYSBsYXllclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Bc3NpZ25JbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgYXNzaWduSW5zdGFuY2UoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkFzc2lnbkluc3RhbmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQXNzaWducyBvbmUgb2YgdGhlIHN0YWNrJ3MgcmVnaXN0ZXJlZCBBbWF6b24gRUJTIHZvbHVtZXMgdG8gYSBzcGVjaWZpZWQgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQXNzaWduVm9sdW1lLmh0bWxcbiAgICovXG4gIHB1YmxpYyBhc3NpZ25Wb2x1bWUoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkFzc2lnblZvbHVtZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFzc29jaWF0ZXMgb25lIG9mIHRoZSBzdGFjaydzIHJlZ2lzdGVyZWQgRWxhc3RpYyBJUCBhZGRyZXNzZXMgd2l0aCBhIHNwZWNpZmllZCBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Bc3NvY2lhdGVFbGFzdGljSXAuaHRtbFxuICAgKi9cbiAgcHVibGljIGFzc29jaWF0ZUVsYXN0aWNJcCgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6QXNzb2NpYXRlRWxhc3RpY0lwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQXR0YWNoZXMgYW4gRWxhc3RpYyBMb2FkIEJhbGFuY2luZyBsb2FkIGJhbGFuY2VyIHRvIGEgc3BlY2lmaWVkIGxheWVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0F0dGFjaEVsYXN0aWNMb2FkQmFsYW5jZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIGF0dGFjaEVsYXN0aWNMb2FkQmFsYW5jZXIoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkF0dGFjaEVsYXN0aWNMb2FkQmFsYW5jZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgY2xvbmUgb2YgYSBzcGVjaWZpZWQgc3RhY2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ2xvbmVTdGFjay5odG1sXG4gICAqL1xuICBwdWJsaWMgY2xvbmVTdGFjaygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6Q2xvbmVTdGFjaycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYW4gYXBwIGZvciBhIHNwZWNpZmllZCBzdGFja1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVBcHAuaHRtbFxuICAgKi9cbiAgcHVibGljIGNyZWF0ZUFwcCgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6Q3JlYXRlQXBwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUnVucyBkZXBsb3ltZW50IG9yIHN0YWNrIGNvbW1hbmRzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZURlcGxveW1lbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIGNyZWF0ZURlcGxveW1lbnQoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkNyZWF0ZURlcGxveW1lbnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGFuIGluc3RhbmNlIGluIGEgc3BlY2lmaWVkIHN0YWNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZUluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBjcmVhdGVJbnN0YW5jZSgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6Q3JlYXRlSW5zdGFuY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgbGF5ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlTGF5ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIGNyZWF0ZUxheWVyKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpDcmVhdGVMYXllcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBuZXcgc3RhY2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlU3RhY2suaHRtbFxuICAgKi9cbiAgcHVibGljIGNyZWF0ZVN0YWNrKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpDcmVhdGVTdGFjaycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBuZXcgdXNlciBwcm9maWxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVVzZXJQcm9maWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBjcmVhdGVVc2VyUHJvZmlsZSgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6Q3JlYXRlVXNlclByb2ZpbGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgc3BlY2lmaWVkIGFwcFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVBcHAuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZUFwcCgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVsZXRlQXBwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIHNwZWNpZmllZCBpbnN0YW5jZSwgd2hpY2ggdGVybWluYXRlcyB0aGUgYXNzb2NpYXRlZCBBbWF6b24gRUMyIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVJbnN0YW5jZSgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVsZXRlSW5zdGFuY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgc3BlY2lmaWVkIGxheWVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUxheWVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVMYXllcigpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVsZXRlTGF5ZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgc3BlY2lmaWVkIHN0YWNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVN0YWNrLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVTdGFjaygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVsZXRlU3RhY2snKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgdXNlciBwcm9maWxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVVzZXJQcm9maWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVVc2VyUHJvZmlsZSgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVsZXRlVXNlclByb2ZpbGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgdXNlciBwcm9maWxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlcmVnaXN0ZXJFY3NDbHVzdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZXJlZ2lzdGVyRWNzQ2x1c3RlcigpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVyZWdpc3RlckVjc0NsdXN0ZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXJlZ2lzdGVycyBhIHNwZWNpZmllZCBFbGFzdGljIElQIGFkZHJlc3NcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVyZWdpc3RlckVsYXN0aWNJcC5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVyZWdpc3RlckVsYXN0aWNJcCgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVyZWdpc3RlckVsYXN0aWNJcCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlcmVnaXN0ZXIgYSByZWdpc3RlcmVkIEFtYXpvbiBFQzIgb3Igb24tcHJlbWlzZXMgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVyZWdpc3Rlckluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZXJlZ2lzdGVySW5zdGFuY2UoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlcmVnaXN0ZXJJbnN0YW5jZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlcmVnaXN0ZXJzIGFuIEFtYXpvbiBSRFMgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVyZWdpc3RlclJkc0RiSW5zdGFuY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlcmVnaXN0ZXJSZHNEYkluc3RhbmNlKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpEZXJlZ2lzdGVyUmRzRGJJbnN0YW5jZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlcmVnaXN0ZXJzIGFuIEFtYXpvbiBFQlMgdm9sdW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlcmVnaXN0ZXJWb2x1bWUuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlcmVnaXN0ZXJWb2x1bWUoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlcmVnaXN0ZXJWb2x1bWUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgdGhlIGF2YWlsYWJsZSBBV1MgT3BzV29ya3MgYWdlbnQgdmVyc2lvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUFnZW50VmVyc2lvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlQWdlbnRWZXJzaW9ucygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVBZ2VudFZlcnNpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWVzdHMgYSBkZXNjcmlwdGlvbiBvZiBhIHNwZWNpZmllZCBzZXQgb2YgYXBwc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQXBwcy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVBcHBzKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpEZXNjcmliZUFwcHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgdGhlIHJlc3VsdHMgb2Ygc3BlY2lmaWVkIGNvbW1hbmRzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVDb21tYW5kcy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVDb21tYW5kcygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVDb21tYW5kcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlcXVlc3RzIGEgZGVzY3JpcHRpb24gb2YgYSBzcGVjaWZpZWQgc2V0IG9mIGRlcGxveW1lbnRzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVEZXBsb3ltZW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVEZXBsb3ltZW50cygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVEZXBsb3ltZW50cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlcyBBbWF6b24gRUNTIGNsdXN0ZXJzIHRoYXQgYXJlIHJlZ2lzdGVyZWQgd2l0aCBhIHN0YWNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVFY3NDbHVzdGVycy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVFY3NDbHVzdGVycygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVFY3NDbHVzdGVycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlcyBFbGFzdGljIElQIGFkZHJlc3Nlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRWxhc3RpY0lwcy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVFbGFzdGljSXBzKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpEZXNjcmliZUVsYXN0aWNJcHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgYSBzdGFjaydzIEVsYXN0aWMgTG9hZCBCYWxhbmNpbmcgaW5zdGFuY2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVFbGFzdGljTG9hZEJhbGFuY2Vycy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVFbGFzdGljTG9hZEJhbGFuY2VycygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVFbGFzdGljTG9hZEJhbGFuY2VycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlcXVlc3RzIGEgZGVzY3JpcHRpb24gb2YgYSBzZXQgb2YgaW5zdGFuY2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVJbnN0YW5jZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlSW5zdGFuY2VzKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpEZXNjcmliZUluc3RhbmNlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlcXVlc3RzIGEgZGVzY3JpcHRpb24gb2Ygb25lIG9yIG1vcmUgbGF5ZXJzIGluIGEgc3BlY2lmaWVkIHN0YWNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVMYXllcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlTGF5ZXJzKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpEZXNjcmliZUxheWVycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlcyBsb2FkLWJhc2VkIGF1dG8gc2NhbGluZyBjb25maWd1cmF0aW9ucyBmb3Igc3BlY2lmaWVkIGxheWVyc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlTG9hZEJhc2VkQXV0b1NjYWxpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlTG9hZEJhc2VkQXV0b1NjYWxpbmcoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlc2NyaWJlTG9hZEJhc2VkQXV0b1NjYWxpbmcnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgYSB1c2VyJ3MgU1NIIGluZm9ybWF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVNeVVzZXJQcm9maWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZXNjcmliZU15VXNlclByb2ZpbGUoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlc2NyaWJlTXlVc2VyUHJvZmlsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlcyB0aGUgcGVybWlzc2lvbnMgZm9yIGEgc3BlY2lmaWVkIHN0YWNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVQZXJtaXNzaW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVQZXJtaXNzaW9ucygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVQZXJtaXNzaW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlIGFuIGluc3RhbmNlJ3MgUkFJRCBhcnJheXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVJhaWRBcnJheXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlUmFpZEFycmF5cygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVSYWlkQXJyYXlzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIEFtYXpvbiBSRFMgaW5zdGFuY2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVSZHNEYkluc3RhbmNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVSZHNEYkluc3RhbmNlcygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVSZHNEYkluc3RhbmNlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlcyBBV1MgT3BzV29ya3Mgc2VydmljZSBlcnJvcnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVNlcnZpY2VFcnJvcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlU2VydmljZUVycm9ycygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVTZXJ2aWNlRXJyb3JzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWVzdHMgYSBkZXNjcmlwdGlvbiBvZiBhIHN0YWNrJ3MgcHJvdmlzaW9uaW5nIHBhcmFtZXRlcnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVN0YWNrUHJvdmlzaW9uaW5nUGFyYW1ldGVycy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVTdGFja1Byb3Zpc2lvbmluZ1BhcmFtZXRlcnMoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRlc2NyaWJlU3RhY2tQcm92aXNpb25pbmdQYXJhbWV0ZXJzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIHRoZSBudW1iZXIgb2YgbGF5ZXJzIGFuZCBhcHBzIGluIGEgc3BlY2lmaWVkIHN0YWNrLCBhbmQgdGhlIG51bWJlciBvZiBpbnN0YW5jZXMgaW4gZWFjaCBzdGF0ZSwgc3VjaCBhcyBydW5uaW5nX3NldHVwIG9yIG9ubGluZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlU3RhY2tTdW1tYXJ5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZXNjcmliZVN0YWNrU3VtbWFyeSgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVTdGFja1N1bW1hcnknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXF1ZXN0cyBhIGRlc2NyaXB0aW9uIG9mIG9uZSBvciBtb3JlIHN0YWNrc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlU3RhY2tzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZXNjcmliZVN0YWNrcygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGVzY3JpYmVTdGFja3MnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgdGltZS1iYXNlZCBhdXRvIHNjYWxpbmcgY29uZmlndXJhdGlvbnMgZm9yIHNwZWNpZmllZCBpbnN0YW5jZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVRpbWVCYXNlZEF1dG9TY2FsaW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZXNjcmliZVRpbWVCYXNlZEF1dG9TY2FsaW5nKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpEZXNjcmliZVRpbWVCYXNlZEF1dG9TY2FsaW5nJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmUgc3BlY2lmaWVkIHVzZXJzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVVc2VyUHJvZmlsZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlVXNlclByb2ZpbGVzKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpEZXNjcmliZVVzZXJQcm9maWxlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlcyBhbiBpbnN0YW5jZSdzIEFtYXpvbiBFQlMgdm9sdW1lc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlVm9sdW1lcy5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVWb2x1bWVzKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpEZXNjcmliZVZvbHVtZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXRhY2hlcyBhIHNwZWNpZmllZCBFbGFzdGljIExvYWQgQmFsYW5jaW5nIGluc3RhbmNlIGZyb20gaXRzIGxheWVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RldGFjaEVsYXN0aWNMb2FkQmFsYW5jZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIGRldGFjaEVsYXN0aWNMb2FkQmFsYW5jZXIoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOkRldGFjaEVsYXN0aWNMb2FkQmFsYW5jZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEaXNhc3NvY2lhdGVzIGFuIEVsYXN0aWMgSVAgYWRkcmVzcyBmcm9tIGl0cyBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EaXNhc3NvY2lhdGVFbGFzdGljSXAuaHRtbFxuICAgKi9cbiAgcHVibGljIGRpc2Fzc29jaWF0ZUVsYXN0aWNJcCgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6RGlzYXNzb2NpYXRlRWxhc3RpY0lwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR2V0cyBhIGdlbmVyYXRlZCBob3N0IG5hbWUgZm9yIHRoZSBzcGVjaWZpZWQgbGF5ZXIsIGJhc2VkIG9uIHRoZSBjdXJyZW50IGhvc3QgbmFtZSB0aGVtZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldEhvc3RuYW1lU3VnZ2VzdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgZ2V0SG9zdG5hbWVTdWdnZXN0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpHZXRIb3N0bmFtZVN1Z2dlc3Rpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgUkRQIGFjY2VzcyB0byBhIFdpbmRvd3MgaW5zdGFuY2UgZm9yIGEgc3BlY2lmaWVkIHRpbWUgcGVyaW9kXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlYm9vdEluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBncmFudEFjY2VzcygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6R3JhbnRBY2Nlc3MnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGEgbGlzdCBvZiB0YWdzIHRoYXQgYXJlIGFwcGxpZWQgdG8gdGhlIHNwZWNpZmllZCBzdGFjayBvciBsYXllclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RUYWdzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0VGFncygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6TGlzdFRhZ3MnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZWJvb3RzIGEgc3BlY2lmaWVkIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlYm9vdEluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyByZWJvb3RJbnN0YW5jZSgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6UmVib290SW5zdGFuY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZWdpc3RlcnMgYSBzcGVjaWZpZWQgQW1hem9uIEVDUyBjbHVzdGVyIHdpdGggYSBzdGFja1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZWdpc3RlckVjc0NsdXN0ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHJlZ2lzdGVyRWNzQ2x1c3RlcigpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6UmVnaXN0ZXJFY3NDbHVzdGVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVnaXN0ZXJzIGFuIEVsYXN0aWMgSVAgYWRkcmVzcyB3aXRoIGEgc3BlY2lmaWVkIHN0YWNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlZ2lzdGVyRWxhc3RpY0lwLmh0bWxcbiAgICovXG4gIHB1YmxpYyByZWdpc3RlckVsYXN0aWNJcCgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6UmVnaXN0ZXJFbGFzdGljSXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZWdpc3RlcnMgaW5zdGFuY2VzIHdpdGggYSBzcGVjaWZpZWQgc3RhY2sgdGhhdCB3ZXJlIGNyZWF0ZWQgb3V0c2lkZSBvZiBBV1MgT3BzV29ya3NcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVnaXN0ZXJJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgcmVnaXN0ZXJJbnN0YW5jZSgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6UmVnaXN0ZXJJbnN0YW5jZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlZ2lzdGVycyBhbiBBbWF6b24gUkRTIGluc3RhbmNlIHdpdGggYSBzdGFja1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZWdpc3RlclJkc0RiSW5zdGFuY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHJlZ2lzdGVyUmRzRGJJbnN0YW5jZSgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6UmVnaXN0ZXJSZHNEYkluc3RhbmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVnaXN0ZXJzIGFuIEFtYXpvbiBFQlMgdm9sdW1lIHdpdGggYSBzcGVjaWZpZWQgc3RhY2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVnaXN0ZXJWb2x1bWUuaHRtbFxuICAgKi9cbiAgcHVibGljIHJlZ2lzdGVyVm9sdW1lKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpSZWdpc3RlclZvbHVtZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFNwZWNpZnkgdGhlIGxvYWQtYmFzZWQgYXV0byBzY2FsaW5nIGNvbmZpZ3VyYXRpb24gZm9yIGEgc3BlY2lmaWVkIGxheWVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1NldExvYWRCYXNlZEF1dG9TY2FsaW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyBzZXRMb2FkQmFzZWRBdXRvU2NhbGluZygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6U2V0TG9hZEJhc2VkQXV0b1NjYWxpbmcnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgYSB1c2VyJ3MgcGVybWlzc2lvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TZXRQZXJtaXNzaW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyBzZXRQZXJtaXNzaW9uKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpTZXRQZXJtaXNzaW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU3BlY2lmeSB0aGUgdGltZS1iYXNlZCBhdXRvIHNjYWxpbmcgY29uZmlndXJhdGlvbiBmb3IgYSBzcGVjaWZpZWQgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2V0VGltZUJhc2VkQXV0b1NjYWxpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHNldFRpbWVCYXNlZEF1dG9TY2FsaW5nKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpTZXRUaW1lQmFzZWRBdXRvU2NhbGluZycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXJ0cyBhIHNwZWNpZmllZCBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydEluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBzdGFydEluc3RhbmNlKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpTdGFydEluc3RhbmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU3RhcnRzIGEgc3RhY2sncyBpbnN0YW5jZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RhcnRTdGFjay5odG1sXG4gICAqL1xuICBwdWJsaWMgc3RhcnRTdGFjaygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6U3RhcnRTdGFjaycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0b3BzIGEgc3BlY2lmaWVkIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0b3BJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgc3RvcEluc3RhbmNlKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpTdG9wSW5zdGFuY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTdG9wcyBhIHNwZWNpZmllZCBzdGFja1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdG9wU3RhY2suaHRtbFxuICAgKi9cbiAgcHVibGljIHN0b3BTdGFjaygpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6U3RvcFN0YWNrJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQXBwbHkgdGFncyB0byBhIHNwZWNpZmllZCBzdGFjayBvciBsYXllclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9UYWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdGFnUmVzb3VyY2UoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVW5hc3NpZ25zIGEgcmVnaXN0ZXJlZCBpbnN0YW5jZSBmcm9tIGFsbCBvZiBpdCdzIGxheWVyc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbmFzc2lnbkluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB1bmFzc2lnbkluc3RhbmNlKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpVbmFzc2lnbkluc3RhbmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVW5hc3NpZ25zIGFuIGFzc2lnbmVkIEFtYXpvbiBFQlMgdm9sdW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VuYXNzaWduVm9sdW1lLmh0bWxcbiAgICovXG4gIHB1YmxpYyB1bmFzc2lnblZvbHVtZSgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6VW5hc3NpZ25Wb2x1bWUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW1vdmVzIHRhZ3MgZnJvbSBhIHNwZWNpZmllZCBzdGFjayBvciBsYXllclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB1bnRhZ1Jlc291cmNlKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpVbnRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhIHNwZWNpZmllZCBhcHBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlQXBwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB1cGRhdGVBcHAoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlVwZGF0ZUFwcCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgYSByZWdpc3RlcmVkIEVsYXN0aWMgSVAgYWRkcmVzcydzIG5hbWVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlRWxhc3RpY0lwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB1cGRhdGVFbGFzdGljSXAoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlVwZGF0ZUVsYXN0aWNJcCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgYSBzcGVjaWZpZWQgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlSW5zdGFuY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHVwZGF0ZUluc3RhbmNlKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpVcGRhdGVJbnN0YW5jZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgYSBzcGVjaWZpZWQgbGF5ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlTGF5ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHVwZGF0ZUxheWVyKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpVcGRhdGVMYXllcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgYSB1c2VyJ3MgU1NIIHB1YmxpYyBrZXlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlTXlVc2VyUHJvZmlsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlTXlVc2VyUHJvZmlsZSgpIHtcbiAgICB0aGlzLmFkZCgnb3Bzd29ya3M6VXBkYXRlTXlVc2VyUHJvZmlsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgYW4gQW1hem9uIFJEUyBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVSZHNEYkluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB1cGRhdGVSZHNEYkluc3RhbmNlKCkge1xuICAgIHRoaXMuYWRkKCdvcHN3b3JrczpVcGRhdGVSZHNEYkluc3RhbmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhIHNwZWNpZmllZCBzdGFja1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVTdGFjay5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlU3RhY2soKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlVwZGF0ZVN0YWNrJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhIHNwZWNpZmllZCB1c2VyIHByb2ZpbGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcHN3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVVc2VyUHJvZmlsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlVXNlclByb2ZpbGUoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlVwZGF0ZVVzZXJQcm9maWxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhbiBBbWF6b24gRUJTIHZvbHVtZSdzIG5hbWUgb3IgbW91bnQgcG9pbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3Bzd29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlVm9sdW1lLmh0bWxcbiAgICovXG4gIHB1YmxpYyB1cGRhdGVWb2x1bWUoKSB7XG4gICAgdGhpcy5hZGQoJ29wc3dvcmtzOlVwZGF0ZVZvbHVtZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHN0YWNrIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wc3dvcmtzL2xhdGVzdC91c2VyZ3VpZGUvd29ya2luZ3N0YWNrcy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBzdGFja0lkIC0gSWRlbnRpZmllciBmb3IgdGhlIHN0YWNrSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25TdGFjayhzdGFja0lkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259Om9wc3dvcmtzOiR7UmVnaW9ufToke0FjY291bnR9OnN0YWNrLyR7U3RhY2tJZH0vJztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtTdGFja0lkfScsIHN0YWNrSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG59XG4iXX0=