"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Secretsmanager = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [secretsmanager](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssecretsmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Secretsmanager extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [secretsmanager](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssecretsmanager.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'secretsmanager';
        this.actionList = {
            "CancelRotateSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to cancel an in-progress secret rotation.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "CreateSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to create a secret that stores encrypted data that can be queried and rotated.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:Name",
                    "secretsmanager:Description",
                    "secretsmanager:KmsKeyId",
                    "aws:RequestTag/tag-key",
                    "aws:TagKeys",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "DeleteResourcePolicy": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to delete the resource policy attached to a secret.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "DeleteSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to delete a secret.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:RecoveryWindowInDays",
                    "secretsmanager:ForceDeleteWithoutRecovery",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "DescribeSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to retrieve the metadata about a secret, but not the encrypted data.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "GetRandomPassword": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to generate a random string for use in password creation.",
                "accessLevel": "Read"
            },
            "GetResourcePolicy": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to get the resource policy attached to a secret.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "GetSecretValue": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to retrieve and decrypt the encrypted data.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:VersionId",
                    "secretsmanager:VersionStage",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "ListSecretVersionIds": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to list the available versions of a secret.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "ListSecrets": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to list the available secrets.",
                "accessLevel": "List"
            },
            "PutResourcePolicy": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to attach a resource policy to a secret.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key",
                    "secretsmanager:BlockPublicPolicy"
                ]
            },
            "PutSecretValue": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to create a new version of the secret with new encrypted data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "RestoreSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to cancel deletion of a secret.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "RotateSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to start rotation of a secret.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:RotationLambdaARN",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to add tags to a secret.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "aws:RequestTag/tag-key",
                    "aws:TagKeys",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to remove tags from a secret.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "aws:TagKeys",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "UpdateSecret": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to update a secret with new metadata or with a new version of the encrypted data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:Description",
                    "secretsmanager:KmsKeyId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "UpdateSecretVersionStage": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to move a stage from one secret to another.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:VersionStage",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            },
            "ValidateResourcePolicy": {
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions",
                "description": "Enables the user to validate a resource policy before attaching policy.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "Secret": {
                        "required": true
                    }
                },
                "conditions": [
                    "secretsmanager:SecretId",
                    "secretsmanager:resource/AllowRotationLambdaArn",
                    "secretsmanager:ResourceTag/tag-key"
                ]
            }
        };
        this.resourceTypes = {
            "Secret": {
                "name": "Secret",
                "url": "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-resources",
                "arn": "arn:${Partition}:secretsmanager:${Region}:${Account}:secret:${SecretId}",
                "conditionKeys": [
                    "aws:RequestTag/tag-key",
                    "aws:TagKeys",
                    "secretsmanager:ResourceTag/tag-key",
                    "secretsmanager:resource/AllowRotationLambdaArn"
                ]
            }
        };
    }
    /**
     * Enables the user to cancel an in-progress secret rotation.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    cancelRotateSecret() {
        this.add('secretsmanager:CancelRotateSecret');
        return this;
    }
    /**
     * Enables the user to create a secret that stores encrypted data that can be queried and rotated.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifName()
     * - .ifDescription()
     * - .ifKmsKeyId()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    createSecret() {
        this.add('secretsmanager:CreateSecret');
        return this;
    }
    /**
     * Enables the user to delete the resource policy attached to a secret.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    deleteResourcePolicy() {
        this.add('secretsmanager:DeleteResourcePolicy');
        return this;
    }
    /**
     * Enables the user to delete a secret.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifRecoveryWindowInDays()
     * - .ifForceDeleteWithoutRecovery()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    deleteSecret() {
        this.add('secretsmanager:DeleteSecret');
        return this;
    }
    /**
     * Enables the user to retrieve the metadata about a secret, but not the encrypted data.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    describeSecret() {
        this.add('secretsmanager:DescribeSecret');
        return this;
    }
    /**
     * Enables the user to generate a random string for use in password creation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    getRandomPassword() {
        this.add('secretsmanager:GetRandomPassword');
        return this;
    }
    /**
     * Enables the user to get the resource policy attached to a secret.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    getResourcePolicy() {
        this.add('secretsmanager:GetResourcePolicy');
        return this;
    }
    /**
     * Enables the user to retrieve and decrypt the encrypted data.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifVersionId()
     * - .ifVersionStage()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    getSecretValue() {
        this.add('secretsmanager:GetSecretValue');
        return this;
    }
    /**
     * Enables the user to list the available versions of a secret.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    listSecretVersionIds() {
        this.add('secretsmanager:ListSecretVersionIds');
        return this;
    }
    /**
     * Enables the user to list the available secrets.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    listSecrets() {
        this.add('secretsmanager:ListSecrets');
        return this;
    }
    /**
     * Enables the user to attach a resource policy to a secret.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     * - .ifBlockPublicPolicy()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    putResourcePolicy() {
        this.add('secretsmanager:PutResourcePolicy');
        return this;
    }
    /**
     * Enables the user to create a new version of the secret with new encrypted data.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    putSecretValue() {
        this.add('secretsmanager:PutSecretValue');
        return this;
    }
    /**
     * Enables the user to cancel deletion of a secret.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    restoreSecret() {
        this.add('secretsmanager:RestoreSecret');
        return this;
    }
    /**
     * Enables the user to start rotation of a secret.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifRotationLambdaARN()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    rotateSecret() {
        this.add('secretsmanager:RotateSecret');
        return this;
    }
    /**
     * Enables the user to add tags to a secret.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    tagResource() {
        this.add('secretsmanager:TagResource');
        return this;
    }
    /**
     * Enables the user to remove tags from a secret.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifAwsTagKeys()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    untagResource() {
        this.add('secretsmanager:UntagResource');
        return this;
    }
    /**
     * Enables the user to update a secret with new metadata or with a new version of the encrypted data.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifDescription()
     * - .ifKmsKeyId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    updateSecret() {
        this.add('secretsmanager:UpdateSecret');
        return this;
    }
    /**
     * Enables the user to move a stage from one secret to another.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifVersionStage()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    updateSecretVersionStage() {
        this.add('secretsmanager:UpdateSecretVersionStage');
        return this;
    }
    /**
     * Enables the user to validate a resource policy before attaching policy.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifSecretId()
     * - .ifResource()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-actions
     */
    validateResourcePolicy() {
        this.add('secretsmanager:ValidateResourcePolicy');
        return this;
    }
    /**
     * Adds a resource of type Secret to the statement
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-resources
     *
     * @param secretId - Identifier for the secretId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTag()
     * - .ifResource()
     */
    onSecret(secretId, account, region, partition) {
        var arn = 'arn:${Partition}:secretsmanager:${Region}:${Account}:secret:${SecretId}';
        arn = arn.replace('${SecretId}', secretId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by a key that is present in the request the user makes to the Secrets Manager service.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .createSecret()
     * - .tagResource()
     *
     * Applies to resource types:
     * - Secret
     *
     * @param tagkey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagkey, value, operator) {
        return this.if(`aws:RequestTag/${tagkey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the list of all the tag key namespresent in the request the user makes to the Secrets Manager service.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .createSecret()
     * - .tagResource()
     * - .untagResource()
     *
     * Applies to resource types:
     * - Secret
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether the resource policy blocks broad AWS account access.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .putResourcePolicy()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifBlockPublicPolicy(value) {
        return this.if(`secretsmanager:BlockPublicPolicy`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the description text in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .createSecret()
     * - .updateSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDescription(value, operator) {
        return this.if(`secretsmanager:Description`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether the secret is to be deleted immediately without any recovery window.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .deleteSecret()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifForceDeleteWithoutRecovery(value) {
        return this.if(`secretsmanager:ForceDeleteWithoutRecovery`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the ARN of the KMS key in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .createSecret()
     * - .updateSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKmsKeyId(value, operator) {
        return this.if(`secretsmanager:KmsKeyId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the friendly name of the secret in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .createSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifName(value, operator) {
        return this.if(`secretsmanager:Name`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the number of days that Secrets Manager waits before it can delete the secret.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .deleteSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRecoveryWindowInDays(value, operator) {
        return this.if(`secretsmanager:RecoveryWindowInDays`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by a tag key and value pair.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .cancelRotateSecret()
     * - .createSecret()
     * - .deleteResourcePolicy()
     * - .deleteSecret()
     * - .describeSecret()
     * - .getResourcePolicy()
     * - .getSecretValue()
     * - .listSecretVersionIds()
     * - .putResourcePolicy()
     * - .putSecretValue()
     * - .restoreSecret()
     * - .rotateSecret()
     * - .tagResource()
     * - .untagResource()
     * - .updateSecret()
     * - .updateSecretVersionStage()
     * - .validateResourcePolicy()
     *
     * Applies to resource types:
     * - Secret
     *
     * @param tagkey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagkey, value, operator) {
        return this.if(`secretsmanager:ResourceTag/${tagkey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of the rotation Lambda function in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .rotateSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifRotationLambdaARN(value, operator) {
        return this.if(`secretsmanager:RotationLambdaARN`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the SecretID value in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .cancelRotateSecret()
     * - .deleteResourcePolicy()
     * - .deleteSecret()
     * - .describeSecret()
     * - .getResourcePolicy()
     * - .getSecretValue()
     * - .listSecretVersionIds()
     * - .putResourcePolicy()
     * - .putSecretValue()
     * - .restoreSecret()
     * - .rotateSecret()
     * - .tagResource()
     * - .untagResource()
     * - .updateSecret()
     * - .updateSecretVersionStage()
     * - .validateResourcePolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifSecretId(value, operator) {
        return this.if(`secretsmanager:SecretId`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the unique identifier of the version of the secret in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .getSecretValue()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionId(value, operator) {
        return this.if(`secretsmanager:VersionId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the list of version stages in the request.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .getSecretValue()
     * - .updateSecretVersionStage()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionStage(value, operator) {
        return this.if(`secretsmanager:VersionStage`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of the rotation Lambda function associated with the secret.
     *
     * https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .cancelRotateSecret()
     * - .deleteResourcePolicy()
     * - .deleteSecret()
     * - .describeSecret()
     * - .getResourcePolicy()
     * - .getSecretValue()
     * - .listSecretVersionIds()
     * - .putResourcePolicy()
     * - .putSecretValue()
     * - .restoreSecret()
     * - .rotateSecret()
     * - .tagResource()
     * - .untagResource()
     * - .updateSecret()
     * - .updateSecretVersionStage()
     * - .validateResourcePolicy()
     *
     * Applies to resource types:
     * - Secret
     *
     * @param allowRotationLambdaArn The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifResource(allowRotationLambdaArn, value, operator) {
        return this.if(`secretsmanager:resource/${allowRotationLambdaArn}`, value, operator || 'ArnEquals');
    }
}
exports.Secretsmanager = Secretsmanager;
//# sourceMappingURL=data:application/json;base64,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