"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Waf = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [waf](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awswaf.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Waf extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [waf](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awswaf.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'waf';
        this.actionList = {
            "CreateByteMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateByteMatchSet.html",
                "description": "Creates a ByteMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bytematchset": {
                        "required": true
                    }
                }
            },
            "CreateGeoMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateGeoMatchSet.html",
                "description": "Creates a GeoMatchSet, which you use to specify which web requests you want to allow or block based on the country that the requests originate from.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "geomatchset": {
                        "required": true
                    }
                }
            },
            "CreateIPSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateIPSet.html",
                "description": "Creates an IPSet, which you use to specify which web requests you want to allow or block based on the IP addresses that the requests originate from.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ipset": {
                        "required": true
                    }
                }
            },
            "CreateRateBasedRule": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRateBasedRule.html",
                "description": "Creates a RateBasedRule, which contains a RateLimit specifying the maximum number of requests that AWS WAF allows from a specified IP address in a five-minute period.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ratebasedrule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateRegexMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRegexMatchSet.html",
                "description": "Creates a RegexMatchSet, which you use to specify which web requests you want to allow or block based on the regex patterns you specified in a RegexPatternSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "regexmatchset": {
                        "required": true
                    }
                }
            },
            "CreateRegexPatternSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRegexPatternSet.html",
                "description": "Creates a RegexPatternSet, which you use to specify the regular expression (regex) pattern that you want AWS WAF to search for.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "regexpatternset": {
                        "required": true
                    }
                }
            },
            "CreateRule": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRule.html",
                "description": "Creates a Rule, which contains the IPSet objects, ByteMatchSet objects, and other predicates that identify the requests that you want to block.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateRuleGroup": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRuleGroup.html",
                "description": "Creates a RuleGroup. A rule group is a collection of predefined rules that you add to a WebACL.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSizeConstraintSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateSizeConstraintSet.html",
                "description": "Creates a SizeConstraintSet, which you use to identify the part of a web request that you want to check for length.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sizeconstraintset": {
                        "required": true
                    }
                }
            },
            "CreateSqlInjectionMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateSqlInjectionMatchSet.html",
                "description": "Creates a SqlInjectionMatchSet, which you use to allow, block, or count requests that contain snippets of SQL code in a specified part of web requests.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sqlinjectionmatchset": {
                        "required": true
                    }
                }
            },
            "CreateWebACL": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateWebACL.html",
                "description": "Creates a WebACL, which contains the Rules that identify the CloudFront web requests that you want to allow, block, or count.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateWebACLMigrationStack": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateWebACLMigrationStack.html",
                "description": "Create and store a CloudFormation tempalte that creates an equivalent WAF v2 WebACL from the given WAF Classic WebACL in the given S3 bucket.",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:PutObject"
                ],
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "CreateXssMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateXssMatchSet.html",
                "description": "Creates an XssMatchSet, which you use to allow, block, or count requests that contain cross-site scripting attacks in the specified part of web requests.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "xssmatchset": {
                        "required": true
                    }
                }
            },
            "DeleteByteMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteByteMatchSet.html",
                "description": "Permanently deletes a ByteMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bytematchset": {
                        "required": true
                    }
                }
            },
            "DeleteGeoMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteGeoMatchSet.html",
                "description": "Permanently deletes an GeoMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "geomatchset": {
                        "required": true
                    }
                }
            },
            "DeleteIPSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteIPSet.html",
                "description": "Permanently deletes an IPSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ipset": {
                        "required": true
                    }
                }
            },
            "DeleteLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteLoggingConfiguration.html",
                "description": "Permanently deletes the LoggingConfiguration from the specified web ACL.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "DeletePermissionPolicy": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeletePermissionPolicy.html",
                "description": "Permanently deletes an IAM policy from the specified RuleGroup.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                }
            },
            "DeleteRateBasedRule": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRateBasedRule.html",
                "description": "Permanently deletes a RateBasedRule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ratebasedrule": {
                        "required": true
                    }
                }
            },
            "DeleteRegexMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRegexMatchSet.html",
                "description": "Permanently deletes an RegexMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "regexmatchset": {
                        "required": true
                    }
                }
            },
            "DeleteRegexPatternSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRegexPatternSet.html",
                "description": "Permanently deletes an RegexPatternSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "regexpatternset": {
                        "required": true
                    }
                }
            },
            "DeleteRule": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRule.html",
                "description": "Permanently deletes a Rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "DeleteRuleGroup": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRuleGroup.html",
                "description": "Permanently deletes a RuleGroup.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                }
            },
            "DeleteSizeConstraintSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteSizeConstraintSet.html",
                "description": "Permanently deletes a SizeConstraintSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sizeconstraintset": {
                        "required": true
                    }
                }
            },
            "DeleteSqlInjectionMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteSqlInjectionMatchSet.html",
                "description": "Permanently deletes a SqlInjectionMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sqlinjectionmatchset": {
                        "required": true
                    }
                }
            },
            "DeleteWebACL": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteWebACL.html",
                "description": "Permanently deletes a WebACL.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "DeleteXssMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteXssMatchSet.html",
                "description": "Permanently deletes an XssMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "xssmatchset": {
                        "required": true
                    }
                }
            },
            "GetByteMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetByteMatchSet.html",
                "description": "Returns the ByteMatchSet specified by ByteMatchSetId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bytematchset": {
                        "required": true
                    }
                }
            },
            "GetChangeToken": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetChangeToken.html",
                "description": "When you want to create, update, or delete AWS WAF objects, get a change token and include the change token in the create, update, or delete request.",
                "accessLevel": "Read"
            },
            "GetChangeTokenStatus": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetChangeTokenStatus.html",
                "description": "Returns the status of a ChangeToken that you got by calling GetChangeToken.",
                "accessLevel": "Read"
            },
            "GetGeoMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetGeoMatchSet.html",
                "description": "Returns the GeoMatchSet specified by GeoMatchSetId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "geomatchset": {
                        "required": true
                    }
                }
            },
            "GetIPSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetIPSet.html",
                "description": "Returns the IPSet that is specified by IPSetId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ipset": {
                        "required": true
                    }
                }
            },
            "GetLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetLoggingConfiguration.html",
                "description": "Returns the LoggingConfiguration for the specified web ACL.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "GetPermissionPolicy": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetPermissionPolicy.html",
                "description": "Returns the IAM policy attached to the RuleGroup.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                }
            },
            "GetRateBasedRule": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRateBasedRule.html",
                "description": "Returns the RateBasedRule that is specified by the RuleId that you included in the GetRateBasedRule request.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ratebasedrule": {
                        "required": true
                    }
                }
            },
            "GetRateBasedRuleManagedKeys": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRateBasedRuleManagedKeys.html",
                "description": "Returns an array of IP addresses currently being blocked by the RateBasedRule that is specified by the RuleId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ratebasedrule": {
                        "required": true
                    }
                }
            },
            "GetRegexMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRegexMatchSet.html",
                "description": "Returns the RegexMatchSet specified by RegexMatchSetId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "regexmatchset": {
                        "required": true
                    }
                }
            },
            "GetRegexPatternSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRegexPatternSet.html",
                "description": "Returns the RegexPatternSet specified by RegexPatternSetId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "regexpatternset": {
                        "required": true
                    }
                }
            },
            "GetRule": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRule.html",
                "description": "Returns the Rule that is specified by the RuleId that you included in the GetRule request.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "GetRuleGroup": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRuleGroup.html",
                "description": "Returns the RuleGroup that is specified by the RuleGroupId that you included in the GetRuleGroup request.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                }
            },
            "GetSampledRequests": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetSampledRequests.html",
                "description": "Gets detailed information about a specified number of requests--a sample--that AWS WAF randomly selects from among the first 5,000 requests that your AWS resource received during a time range that you choose.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rule": {
                        "required": false
                    },
                    "webacl": {
                        "required": false
                    }
                }
            },
            "GetSizeConstraintSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetSizeConstraintSet.html",
                "description": "Returns the SizeConstraintSet specified by SizeConstraintSetId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "sizeconstraintset": {
                        "required": true
                    }
                }
            },
            "GetSqlInjectionMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetSqlInjectionMatchSet.html",
                "description": "Returns the SqlInjectionMatchSet that is specified by SqlInjectionMatchSetId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "sqlinjectionmatchset": {
                        "required": true
                    }
                }
            },
            "GetWebACL": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetWebACL.html",
                "description": "Returns the WebACL that is specified by WebACLId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "GetXssMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetXssMatchSet.html",
                "description": "Returns the XssMatchSet that is specified by XssMatchSetId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "xssmatchset": {
                        "required": true
                    }
                }
            },
            "ListActivatedRulesInRuleGroup": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListActivatedRulesInRuleGroup.html",
                "description": "Returns an array of ActivatedRule objects.",
                "accessLevel": "List"
            },
            "ListByteMatchSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListByteMatchSets.html",
                "description": "Returns an array of ByteMatchSetSummary objects.",
                "accessLevel": "List"
            },
            "ListGeoMatchSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListGeoMatchSets.html",
                "description": "Returns an array of GeoMatchSetSummary objects.",
                "accessLevel": "List"
            },
            "ListIPSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListIPSets.html",
                "description": "Returns an array of IPSetSummary objects in the response.",
                "accessLevel": "List"
            },
            "ListLoggingConfigurations": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListLoggingConfigurations.html",
                "description": "Returns an array of LoggingConfiguration objects.",
                "accessLevel": "List"
            },
            "ListRateBasedRules": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRateBasedRules.html",
                "description": "Returns an array of RuleSummary objects.",
                "accessLevel": "List"
            },
            "ListRegexMatchSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRegexMatchSets.html",
                "description": "Returns an array of RegexMatchSetSummary objects.",
                "accessLevel": "List"
            },
            "ListRegexPatternSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRegexPatternSets.html",
                "description": "Returns an array of RegexPatternSetSummary objects.",
                "accessLevel": "List"
            },
            "ListRuleGroups": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRuleGroups.html",
                "description": "Returns an array of RuleGroup objects.",
                "accessLevel": "List"
            },
            "ListRules": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRules.html",
                "description": "Returns an array of RuleSummary objects.",
                "accessLevel": "List"
            },
            "ListSizeConstraintSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListSizeConstraintSets.html",
                "description": "Returns an array of SizeConstraintSetSummary objects.",
                "accessLevel": "List"
            },
            "ListSqlInjectionMatchSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListSqlInjectionMatchSets.html",
                "description": "Returns an array of SqlInjectionMatchSet objects.",
                "accessLevel": "List"
            },
            "ListSubscribedRuleGroups": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListSubscribedRuleGroups.html",
                "description": "Returns an array of RuleGroup objects that you are subscribed to.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListTagsForResource.html",
                "description": "Lists the Tags for a given resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ratebasedrule": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "rulegroup": {
                        "required": false
                    },
                    "webacl": {
                        "required": false
                    }
                }
            },
            "ListWebACLs": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListWebACLs.html",
                "description": "Returns an array of WebACLSummary objects in the response.",
                "accessLevel": "List"
            },
            "ListXssMatchSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListXssMatchSets.html",
                "description": "Returns an array of XssMatchSet objects.",
                "accessLevel": "List"
            },
            "PutLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_PutLoggingConfiguration.html",
                "description": "Associates a LoggingConfiguration with a specified web ACL.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:CreateServiceLinkedRole"
                ],
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "PutPermissionPolicy": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_PutPermissionPolicy.html",
                "description": "Attaches a IAM policy to the specified resource. The only supported use for this action is to share a RuleGroup across accounts.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_TagResource.html",
                "description": "Adds a Tag to a given resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "ratebasedrule": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "rulegroup": {
                        "required": false
                    },
                    "webacl": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UntagResource.html",
                "description": "Removes a Tag from a given resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "ratebasedrule": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "rulegroup": {
                        "required": false
                    },
                    "webacl": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateByteMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateByteMatchSet.html",
                "description": "Inserts or deletes ByteMatchTuple objects (filters) in a ByteMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bytematchset": {
                        "required": true
                    }
                }
            },
            "UpdateGeoMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateGeoMatchSet.html",
                "description": "Inserts or deletes GeoMatchConstraint objects in a GeoMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "geomatchset": {
                        "required": true
                    }
                }
            },
            "UpdateIPSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateIPSet.html",
                "description": "Inserts or deletes IPSetDescriptor objects in an IPSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ipset": {
                        "required": true
                    }
                }
            },
            "UpdateRateBasedRule": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRateBasedRule.html",
                "description": "Inserts or deletes Predicate objects in a rule and updates the RateLimit in the rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ratebasedrule": {
                        "required": true
                    }
                }
            },
            "UpdateRegexMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRegexMatchSet.html",
                "description": "Inserts or deletes RegexMatchTuple objects (filters) in a RegexMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "regexmatchset": {
                        "required": true
                    }
                }
            },
            "UpdateRegexPatternSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRegexPatternSet.html",
                "description": "Inserts or deletes RegexPatternStrings in a RegexPatternSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "regexpatternset": {
                        "required": true
                    }
                }
            },
            "UpdateRule": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRule.html",
                "description": "Inserts or deletes Predicate objects in a Rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "UpdateRuleGroup": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRuleGroup.html",
                "description": "Inserts or deletes ActivatedRule objects in a RuleGroup.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                }
            },
            "UpdateSizeConstraintSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateSizeConstraintSet.html",
                "description": "Inserts or deletes SizeConstraint objects (filters) in a SizeConstraintSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sizeconstraintset": {
                        "required": true
                    }
                }
            },
            "UpdateSqlInjectionMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateSqlInjectionMatchSet.html",
                "description": "Inserts or deletes SqlInjectionMatchTuple objects (filters) in a SqlInjectionMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sqlinjectionmatchset": {
                        "required": true
                    }
                }
            },
            "UpdateWebACL": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateWebACL.html",
                "description": "Inserts or deletes ActivatedRule objects in a WebACL.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "UpdateXssMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateXssMatchSet.html",
                "description": "Inserts or deletes XssMatchTuple objects (filters) in an XssMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "xssmatchset": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "bytematchset": {
                "name": "bytematchset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ByteMatchSet.html",
                "arn": "arn:${Partition}:waf::${Account}:bytematchset/${Id}",
                "conditionKeys": []
            },
            "ipset": {
                "name": "ipset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_IPSet.html",
                "arn": "arn:${Partition}:waf::${Account}:ipset/${Id}",
                "conditionKeys": []
            },
            "ratebasedrule": {
                "name": "ratebasedrule",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_RateBasedRule.html",
                "arn": "arn:${Partition}:waf::${Account}:ratebasedrule/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "rule": {
                "name": "rule",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_Rule.html",
                "arn": "arn:${Partition}:waf::${Account}:rule/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "sizeconstraintset": {
                "name": "sizeconstraintset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_SizeConstraintSet.html",
                "arn": "arn:${Partition}:waf::${Account}:sizeconstraintset/${Id}",
                "conditionKeys": []
            },
            "sqlinjectionmatchset": {
                "name": "sqlinjectionmatchset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_SqlInjectionMatchSet.html",
                "arn": "arn:${Partition}:waf::${Account}:sqlinjectionset/${Id}",
                "conditionKeys": []
            },
            "webacl": {
                "name": "webacl",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_WebACL.html",
                "arn": "arn:${Partition}:waf::${Account}:webacl/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "xssmatchset": {
                "name": "xssmatchset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_XssMatchSet.html",
                "arn": "arn:${Partition}:waf::${Account}:xssmatchset/${Id}",
                "conditionKeys": []
            },
            "regexmatchset": {
                "name": "regexmatchset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_RegexMatchSet.html",
                "arn": "arn:${Partition}:waf::${Account}:regexmatch/${Id}",
                "conditionKeys": []
            },
            "regexpatternset": {
                "name": "regexpatternset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_RegexPatternSet.html",
                "arn": "arn:${Partition}:waf::${Account}:regexpatternset/${Id}",
                "conditionKeys": []
            },
            "geomatchset": {
                "name": "geomatchset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GeoMatchSet.html",
                "arn": "arn:${Partition}:waf::${Account}:geomatchset/${Id}",
                "conditionKeys": []
            },
            "rulegroup": {
                "name": "rulegroup",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_RuleGroup.html",
                "arn": "arn:${Partition}:waf::${Account}:rulegroup/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a ByteMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateByteMatchSet.html
     */
    createByteMatchSet() {
        this.add('waf:CreateByteMatchSet');
        return this;
    }
    /**
     * Creates a GeoMatchSet, which you use to specify which web requests you want to allow or block based on the country that the requests originate from.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateGeoMatchSet.html
     */
    createGeoMatchSet() {
        this.add('waf:CreateGeoMatchSet');
        return this;
    }
    /**
     * Creates an IPSet, which you use to specify which web requests you want to allow or block based on the IP addresses that the requests originate from.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateIPSet.html
     */
    createIPSet() {
        this.add('waf:CreateIPSet');
        return this;
    }
    /**
     * Creates a RateBasedRule, which contains a RateLimit specifying the maximum number of requests that AWS WAF allows from a specified IP address in a five-minute period.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRateBasedRule.html
     */
    createRateBasedRule() {
        this.add('waf:CreateRateBasedRule');
        return this;
    }
    /**
     * Creates a RegexMatchSet, which you use to specify which web requests you want to allow or block based on the regex patterns you specified in a RegexPatternSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRegexMatchSet.html
     */
    createRegexMatchSet() {
        this.add('waf:CreateRegexMatchSet');
        return this;
    }
    /**
     * Creates a RegexPatternSet, which you use to specify the regular expression (regex) pattern that you want AWS WAF to search for.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRegexPatternSet.html
     */
    createRegexPatternSet() {
        this.add('waf:CreateRegexPatternSet');
        return this;
    }
    /**
     * Creates a Rule, which contains the IPSet objects, ByteMatchSet objects, and other predicates that identify the requests that you want to block.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRule.html
     */
    createRule() {
        this.add('waf:CreateRule');
        return this;
    }
    /**
     * Creates a RuleGroup. A rule group is a collection of predefined rules that you add to a WebACL.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRuleGroup.html
     */
    createRuleGroup() {
        this.add('waf:CreateRuleGroup');
        return this;
    }
    /**
     * Creates a SizeConstraintSet, which you use to identify the part of a web request that you want to check for length.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateSizeConstraintSet.html
     */
    createSizeConstraintSet() {
        this.add('waf:CreateSizeConstraintSet');
        return this;
    }
    /**
     * Creates a SqlInjectionMatchSet, which you use to allow, block, or count requests that contain snippets of SQL code in a specified part of web requests.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateSqlInjectionMatchSet.html
     */
    createSqlInjectionMatchSet() {
        this.add('waf:CreateSqlInjectionMatchSet');
        return this;
    }
    /**
     * Creates a WebACL, which contains the Rules that identify the CloudFront web requests that you want to allow, block, or count.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateWebACL.html
     */
    createWebACL() {
        this.add('waf:CreateWebACL');
        return this;
    }
    /**
     * Create and store a CloudFormation tempalte that creates an equivalent WAF v2 WebACL from the given WAF Classic WebACL in the given S3 bucket.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateWebACLMigrationStack.html
     */
    createWebACLMigrationStack() {
        this.add('waf:CreateWebACLMigrationStack');
        return this;
    }
    /**
     * Creates an XssMatchSet, which you use to allow, block, or count requests that contain cross-site scripting attacks in the specified part of web requests.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateXssMatchSet.html
     */
    createXssMatchSet() {
        this.add('waf:CreateXssMatchSet');
        return this;
    }
    /**
     * Permanently deletes a ByteMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteByteMatchSet.html
     */
    deleteByteMatchSet() {
        this.add('waf:DeleteByteMatchSet');
        return this;
    }
    /**
     * Permanently deletes an GeoMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteGeoMatchSet.html
     */
    deleteGeoMatchSet() {
        this.add('waf:DeleteGeoMatchSet');
        return this;
    }
    /**
     * Permanently deletes an IPSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteIPSet.html
     */
    deleteIPSet() {
        this.add('waf:DeleteIPSet');
        return this;
    }
    /**
     * Permanently deletes the LoggingConfiguration from the specified web ACL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteLoggingConfiguration.html
     */
    deleteLoggingConfiguration() {
        this.add('waf:DeleteLoggingConfiguration');
        return this;
    }
    /**
     * Permanently deletes an IAM policy from the specified RuleGroup.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeletePermissionPolicy.html
     */
    deletePermissionPolicy() {
        this.add('waf:DeletePermissionPolicy');
        return this;
    }
    /**
     * Permanently deletes a RateBasedRule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRateBasedRule.html
     */
    deleteRateBasedRule() {
        this.add('waf:DeleteRateBasedRule');
        return this;
    }
    /**
     * Permanently deletes an RegexMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRegexMatchSet.html
     */
    deleteRegexMatchSet() {
        this.add('waf:DeleteRegexMatchSet');
        return this;
    }
    /**
     * Permanently deletes an RegexPatternSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRegexPatternSet.html
     */
    deleteRegexPatternSet() {
        this.add('waf:DeleteRegexPatternSet');
        return this;
    }
    /**
     * Permanently deletes a Rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRule.html
     */
    deleteRule() {
        this.add('waf:DeleteRule');
        return this;
    }
    /**
     * Permanently deletes a RuleGroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRuleGroup.html
     */
    deleteRuleGroup() {
        this.add('waf:DeleteRuleGroup');
        return this;
    }
    /**
     * Permanently deletes a SizeConstraintSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteSizeConstraintSet.html
     */
    deleteSizeConstraintSet() {
        this.add('waf:DeleteSizeConstraintSet');
        return this;
    }
    /**
     * Permanently deletes a SqlInjectionMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteSqlInjectionMatchSet.html
     */
    deleteSqlInjectionMatchSet() {
        this.add('waf:DeleteSqlInjectionMatchSet');
        return this;
    }
    /**
     * Permanently deletes a WebACL.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteWebACL.html
     */
    deleteWebACL() {
        this.add('waf:DeleteWebACL');
        return this;
    }
    /**
     * Permanently deletes an XssMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteXssMatchSet.html
     */
    deleteXssMatchSet() {
        this.add('waf:DeleteXssMatchSet');
        return this;
    }
    /**
     * Returns the ByteMatchSet specified by ByteMatchSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetByteMatchSet.html
     */
    getByteMatchSet() {
        this.add('waf:GetByteMatchSet');
        return this;
    }
    /**
     * When you want to create, update, or delete AWS WAF objects, get a change token and include the change token in the create, update, or delete request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetChangeToken.html
     */
    getChangeToken() {
        this.add('waf:GetChangeToken');
        return this;
    }
    /**
     * Returns the status of a ChangeToken that you got by calling GetChangeToken.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetChangeTokenStatus.html
     */
    getChangeTokenStatus() {
        this.add('waf:GetChangeTokenStatus');
        return this;
    }
    /**
     * Returns the GeoMatchSet specified by GeoMatchSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetGeoMatchSet.html
     */
    getGeoMatchSet() {
        this.add('waf:GetGeoMatchSet');
        return this;
    }
    /**
     * Returns the IPSet that is specified by IPSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetIPSet.html
     */
    getIPSet() {
        this.add('waf:GetIPSet');
        return this;
    }
    /**
     * Returns the LoggingConfiguration for the specified web ACL.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetLoggingConfiguration.html
     */
    getLoggingConfiguration() {
        this.add('waf:GetLoggingConfiguration');
        return this;
    }
    /**
     * Returns the IAM policy attached to the RuleGroup.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetPermissionPolicy.html
     */
    getPermissionPolicy() {
        this.add('waf:GetPermissionPolicy');
        return this;
    }
    /**
     * Returns the RateBasedRule that is specified by the RuleId that you included in the GetRateBasedRule request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRateBasedRule.html
     */
    getRateBasedRule() {
        this.add('waf:GetRateBasedRule');
        return this;
    }
    /**
     * Returns an array of IP addresses currently being blocked by the RateBasedRule that is specified by the RuleId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRateBasedRuleManagedKeys.html
     */
    getRateBasedRuleManagedKeys() {
        this.add('waf:GetRateBasedRuleManagedKeys');
        return this;
    }
    /**
     * Returns the RegexMatchSet specified by RegexMatchSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRegexMatchSet.html
     */
    getRegexMatchSet() {
        this.add('waf:GetRegexMatchSet');
        return this;
    }
    /**
     * Returns the RegexPatternSet specified by RegexPatternSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRegexPatternSet.html
     */
    getRegexPatternSet() {
        this.add('waf:GetRegexPatternSet');
        return this;
    }
    /**
     * Returns the Rule that is specified by the RuleId that you included in the GetRule request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRule.html
     */
    getRule() {
        this.add('waf:GetRule');
        return this;
    }
    /**
     * Returns the RuleGroup that is specified by the RuleGroupId that you included in the GetRuleGroup request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRuleGroup.html
     */
    getRuleGroup() {
        this.add('waf:GetRuleGroup');
        return this;
    }
    /**
     * Gets detailed information about a specified number of requests--a sample--that AWS WAF randomly selects from among the first 5,000 requests that your AWS resource received during a time range that you choose.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetSampledRequests.html
     */
    getSampledRequests() {
        this.add('waf:GetSampledRequests');
        return this;
    }
    /**
     * Returns the SizeConstraintSet specified by SizeConstraintSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetSizeConstraintSet.html
     */
    getSizeConstraintSet() {
        this.add('waf:GetSizeConstraintSet');
        return this;
    }
    /**
     * Returns the SqlInjectionMatchSet that is specified by SqlInjectionMatchSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetSqlInjectionMatchSet.html
     */
    getSqlInjectionMatchSet() {
        this.add('waf:GetSqlInjectionMatchSet');
        return this;
    }
    /**
     * Returns the WebACL that is specified by WebACLId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetWebACL.html
     */
    getWebACL() {
        this.add('waf:GetWebACL');
        return this;
    }
    /**
     * Returns the XssMatchSet that is specified by XssMatchSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetXssMatchSet.html
     */
    getXssMatchSet() {
        this.add('waf:GetXssMatchSet');
        return this;
    }
    /**
     * Returns an array of ActivatedRule objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListActivatedRulesInRuleGroup.html
     */
    listActivatedRulesInRuleGroup() {
        this.add('waf:ListActivatedRulesInRuleGroup');
        return this;
    }
    /**
     * Returns an array of ByteMatchSetSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListByteMatchSets.html
     */
    listByteMatchSets() {
        this.add('waf:ListByteMatchSets');
        return this;
    }
    /**
     * Returns an array of GeoMatchSetSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListGeoMatchSets.html
     */
    listGeoMatchSets() {
        this.add('waf:ListGeoMatchSets');
        return this;
    }
    /**
     * Returns an array of IPSetSummary objects in the response.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListIPSets.html
     */
    listIPSets() {
        this.add('waf:ListIPSets');
        return this;
    }
    /**
     * Returns an array of LoggingConfiguration objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListLoggingConfigurations.html
     */
    listLoggingConfigurations() {
        this.add('waf:ListLoggingConfigurations');
        return this;
    }
    /**
     * Returns an array of RuleSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRateBasedRules.html
     */
    listRateBasedRules() {
        this.add('waf:ListRateBasedRules');
        return this;
    }
    /**
     * Returns an array of RegexMatchSetSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRegexMatchSets.html
     */
    listRegexMatchSets() {
        this.add('waf:ListRegexMatchSets');
        return this;
    }
    /**
     * Returns an array of RegexPatternSetSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRegexPatternSets.html
     */
    listRegexPatternSets() {
        this.add('waf:ListRegexPatternSets');
        return this;
    }
    /**
     * Returns an array of RuleGroup objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRuleGroups.html
     */
    listRuleGroups() {
        this.add('waf:ListRuleGroups');
        return this;
    }
    /**
     * Returns an array of RuleSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRules.html
     */
    listRules() {
        this.add('waf:ListRules');
        return this;
    }
    /**
     * Returns an array of SizeConstraintSetSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListSizeConstraintSets.html
     */
    listSizeConstraintSets() {
        this.add('waf:ListSizeConstraintSets');
        return this;
    }
    /**
     * Returns an array of SqlInjectionMatchSet objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListSqlInjectionMatchSets.html
     */
    listSqlInjectionMatchSets() {
        this.add('waf:ListSqlInjectionMatchSets');
        return this;
    }
    /**
     * Returns an array of RuleGroup objects that you are subscribed to.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListSubscribedRuleGroups.html
     */
    listSubscribedRuleGroups() {
        this.add('waf:ListSubscribedRuleGroups');
        return this;
    }
    /**
     * Lists the Tags for a given resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('waf:ListTagsForResource');
        return this;
    }
    /**
     * Returns an array of WebACLSummary objects in the response.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListWebACLs.html
     */
    listWebACLs() {
        this.add('waf:ListWebACLs');
        return this;
    }
    /**
     * Returns an array of XssMatchSet objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListXssMatchSets.html
     */
    listXssMatchSets() {
        this.add('waf:ListXssMatchSets');
        return this;
    }
    /**
     * Associates a LoggingConfiguration with a specified web ACL.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_PutLoggingConfiguration.html
     */
    putLoggingConfiguration() {
        this.add('waf:PutLoggingConfiguration');
        return this;
    }
    /**
     * Attaches a IAM policy to the specified resource. The only supported use for this action is to share a RuleGroup across accounts.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_PutPermissionPolicy.html
     */
    putPermissionPolicy() {
        this.add('waf:PutPermissionPolicy');
        return this;
    }
    /**
     * Adds a Tag to a given resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_TagResource.html
     */
    tagResource() {
        this.add('waf:TagResource');
        return this;
    }
    /**
     * Removes a Tag from a given resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UntagResource.html
     */
    untagResource() {
        this.add('waf:UntagResource');
        return this;
    }
    /**
     * Inserts or deletes ByteMatchTuple objects (filters) in a ByteMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateByteMatchSet.html
     */
    updateByteMatchSet() {
        this.add('waf:UpdateByteMatchSet');
        return this;
    }
    /**
     * Inserts or deletes GeoMatchConstraint objects in a GeoMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateGeoMatchSet.html
     */
    updateGeoMatchSet() {
        this.add('waf:UpdateGeoMatchSet');
        return this;
    }
    /**
     * Inserts or deletes IPSetDescriptor objects in an IPSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateIPSet.html
     */
    updateIPSet() {
        this.add('waf:UpdateIPSet');
        return this;
    }
    /**
     * Inserts or deletes Predicate objects in a rule and updates the RateLimit in the rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRateBasedRule.html
     */
    updateRateBasedRule() {
        this.add('waf:UpdateRateBasedRule');
        return this;
    }
    /**
     * Inserts or deletes RegexMatchTuple objects (filters) in a RegexMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRegexMatchSet.html
     */
    updateRegexMatchSet() {
        this.add('waf:UpdateRegexMatchSet');
        return this;
    }
    /**
     * Inserts or deletes RegexPatternStrings in a RegexPatternSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRegexPatternSet.html
     */
    updateRegexPatternSet() {
        this.add('waf:UpdateRegexPatternSet');
        return this;
    }
    /**
     * Inserts or deletes Predicate objects in a Rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRule.html
     */
    updateRule() {
        this.add('waf:UpdateRule');
        return this;
    }
    /**
     * Inserts or deletes ActivatedRule objects in a RuleGroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRuleGroup.html
     */
    updateRuleGroup() {
        this.add('waf:UpdateRuleGroup');
        return this;
    }
    /**
     * Inserts or deletes SizeConstraint objects (filters) in a SizeConstraintSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateSizeConstraintSet.html
     */
    updateSizeConstraintSet() {
        this.add('waf:UpdateSizeConstraintSet');
        return this;
    }
    /**
     * Inserts or deletes SqlInjectionMatchTuple objects (filters) in a SqlInjectionMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateSqlInjectionMatchSet.html
     */
    updateSqlInjectionMatchSet() {
        this.add('waf:UpdateSqlInjectionMatchSet');
        return this;
    }
    /**
     * Inserts or deletes ActivatedRule objects in a WebACL.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateWebACL.html
     */
    updateWebACL() {
        this.add('waf:UpdateWebACL');
        return this;
    }
    /**
     * Inserts or deletes XssMatchTuple objects (filters) in an XssMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateXssMatchSet.html
     */
    updateXssMatchSet() {
        this.add('waf:UpdateXssMatchSet');
        return this;
    }
    /**
     * Adds a resource of type bytematchset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ByteMatchSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBytematchset(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:bytematchset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ipset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_IPSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIpset(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:ipset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ratebasedrule to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_RateBasedRule.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRatebasedrule(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:ratebasedrule/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rule to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_Rule.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRule(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:rule/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type sizeconstraintset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_SizeConstraintSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSizeconstraintset(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:sizeconstraintset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type sqlinjectionmatchset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_SqlInjectionMatchSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSqlinjectionmatchset(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:sqlinjectionset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type webacl to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_WebACL.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWebacl(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:webacl/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type xssmatchset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_XssMatchSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onXssmatchset(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:xssmatchset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type regexmatchset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_RegexMatchSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRegexmatchset(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:regexmatch/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type regexpatternset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_RegexPatternSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRegexpatternset(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:regexpatternset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type geomatchset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GeoMatchSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGeomatchset(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:geomatchset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rulegroup to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_RuleGroup.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRulegroup(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:rulegroup/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .createRateBasedRule()
     * - .createRule()
     * - .createRuleGroup()
     * - .createWebACL()
     * - .tagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag-value assoicated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - ratebasedrule
     * - rule
     * - webacl
     * - rulegroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .createRateBasedRule()
     * - .createRule()
     * - .createRuleGroup()
     * - .createWebACL()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
}
exports.Waf = Waf;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzd2FmLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYXdzd2FmLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHNDQUFvRTtBQUVwRTs7OztHQUlHO0FBQ0gsTUFBYSxHQUFJLFNBQVEsd0JBQWU7SUErekJ0Qzs7OztPQUlHO0lBQ0gsWUFBYSxHQUFZO1FBQ3ZCLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQXAwQk4sa0JBQWEsR0FBRyxLQUFLLENBQUM7UUFDbkIsZUFBVSxHQUFZO1lBQzlCLG9CQUFvQixFQUFFO2dCQUNwQixLQUFLLEVBQUUscUZBQXFGO2dCQUM1RixhQUFhLEVBQUUseUJBQXlCO2dCQUN4QyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLGNBQWMsRUFBRTt3QkFDZCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELG1CQUFtQixFQUFFO2dCQUNuQixLQUFLLEVBQUUsb0ZBQW9GO2dCQUMzRixhQUFhLEVBQUUsc0pBQXNKO2dCQUNySyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLGFBQWEsRUFBRTt3QkFDYixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELGFBQWEsRUFBRTtnQkFDYixLQUFLLEVBQUUsOEVBQThFO2dCQUNyRixhQUFhLEVBQUUsc0pBQXNKO2dCQUNySyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELHFCQUFxQixFQUFFO2dCQUNyQixLQUFLLEVBQUUsc0ZBQXNGO2dCQUM3RixhQUFhLEVBQUUsd0tBQXdLO2dCQUN2TCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLGVBQWUsRUFBRTt3QkFDZixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSxzRkFBc0Y7Z0JBQzdGLGFBQWEsRUFBRSxpS0FBaUs7Z0JBQ2hMLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsZUFBZSxFQUFFO3dCQUNmLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsdUJBQXVCLEVBQUU7Z0JBQ3ZCLEtBQUssRUFBRSx3RkFBd0Y7Z0JBQy9GLGFBQWEsRUFBRSxpSUFBaUk7Z0JBQ2hKLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsaUJBQWlCLEVBQUU7d0JBQ2pCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsWUFBWSxFQUFFO2dCQUNaLEtBQUssRUFBRSw2RUFBNkU7Z0JBQ3BGLGFBQWEsRUFBRSxpSkFBaUo7Z0JBQ2hLLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsTUFBTSxFQUFFO3dCQUNOLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxpQkFBaUIsRUFBRTtnQkFDakIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsYUFBYSxFQUFFLGlHQUFpRztnQkFDaEgsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixXQUFXLEVBQUU7d0JBQ1gsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELHlCQUF5QixFQUFFO2dCQUN6QixLQUFLLEVBQUUsMEZBQTBGO2dCQUNqRyxhQUFhLEVBQUUscUhBQXFIO2dCQUNwSSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLG1CQUFtQixFQUFFO3dCQUNuQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELDRCQUE0QixFQUFFO2dCQUM1QixLQUFLLEVBQUUsNkZBQTZGO2dCQUNwRyxhQUFhLEVBQUUseUpBQXlKO2dCQUN4SyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLHNCQUFzQixFQUFFO3dCQUN0QixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELGNBQWMsRUFBRTtnQkFDZCxLQUFLLEVBQUUsK0VBQStFO2dCQUN0RixhQUFhLEVBQUUsK0hBQStIO2dCQUM5SSxhQUFhLEVBQUUsd0JBQXdCO2dCQUN2QyxlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCw0QkFBNEIsRUFBRTtnQkFDNUIsS0FBSyxFQUFFLDZGQUE2RjtnQkFDcEcsYUFBYSxFQUFFLCtJQUErSTtnQkFDOUosYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGtCQUFrQixFQUFFO29CQUNsQixjQUFjO2lCQUNmO2dCQUNELGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLDJKQUEySjtnQkFDMUssYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixhQUFhLEVBQUU7d0JBQ2IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLHFDQUFxQztnQkFDcEQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixjQUFjLEVBQUU7d0JBQ2QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLHFDQUFxQztnQkFDcEQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixhQUFhLEVBQUU7d0JBQ2IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsYUFBYSxFQUFFLCtCQUErQjtnQkFDOUMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCw0QkFBNEIsRUFBRTtnQkFDNUIsS0FBSyxFQUFFLDZGQUE2RjtnQkFDcEcsYUFBYSxFQUFFLDBFQUEwRTtnQkFDekYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCx3QkFBd0IsRUFBRTtnQkFDeEIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLGlFQUFpRTtnQkFDaEYsYUFBYSxFQUFFLHdCQUF3QjtnQkFDdkMsZUFBZSxFQUFFO29CQUNmLFdBQVcsRUFBRTt3QkFDWCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELHFCQUFxQixFQUFFO2dCQUNyQixLQUFLLEVBQUUsc0ZBQXNGO2dCQUM3RixhQUFhLEVBQUUsc0NBQXNDO2dCQUNyRCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLGVBQWUsRUFBRTt3QkFDZixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELHFCQUFxQixFQUFFO2dCQUNyQixLQUFLLEVBQUUsc0ZBQXNGO2dCQUM3RixhQUFhLEVBQUUsdUNBQXVDO2dCQUN0RCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLGVBQWUsRUFBRTt3QkFDZixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELHVCQUF1QixFQUFFO2dCQUN2QixLQUFLLEVBQUUsd0ZBQXdGO2dCQUMvRixhQUFhLEVBQUUseUNBQXlDO2dCQUN4RCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLGlCQUFpQixFQUFFO3dCQUNqQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELFlBQVksRUFBRTtnQkFDWixLQUFLLEVBQUUsNkVBQTZFO2dCQUNwRixhQUFhLEVBQUUsNkJBQTZCO2dCQUM1QyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE1BQU0sRUFBRTt3QkFDTixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELGlCQUFpQixFQUFFO2dCQUNqQixLQUFLLEVBQUUsa0ZBQWtGO2dCQUN6RixhQUFhLEVBQUUsa0NBQWtDO2dCQUNqRCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFdBQVcsRUFBRTt3QkFDWCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELHlCQUF5QixFQUFFO2dCQUN6QixLQUFLLEVBQUUsMEZBQTBGO2dCQUNqRyxhQUFhLEVBQUUsMENBQTBDO2dCQUN6RCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLG1CQUFtQixFQUFFO3dCQUNuQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELDRCQUE0QixFQUFFO2dCQUM1QixLQUFLLEVBQUUsNkZBQTZGO2dCQUNwRyxhQUFhLEVBQUUsNkNBQTZDO2dCQUM1RCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLHNCQUFzQixFQUFFO3dCQUN0QixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELGNBQWMsRUFBRTtnQkFDZCxLQUFLLEVBQUUsK0VBQStFO2dCQUN0RixhQUFhLEVBQUUsK0JBQStCO2dCQUM5QyxhQUFhLEVBQUUsd0JBQXdCO2dCQUN2QyxlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGFBQWEsRUFBRSxxQ0FBcUM7Z0JBQ3BELGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsYUFBYSxFQUFFO3dCQUNiLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSxrRkFBa0Y7Z0JBQ3pGLGFBQWEsRUFBRSx1REFBdUQ7Z0JBQ3RFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsY0FBYyxFQUFFO3dCQUNkLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSxpRkFBaUY7Z0JBQ3hGLGFBQWEsRUFBRSx1SkFBdUo7Z0JBQ3RLLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0Qsc0JBQXNCLEVBQUU7Z0JBQ3RCLEtBQUssRUFBRSx1RkFBdUY7Z0JBQzlGLGFBQWEsRUFBRSw2RUFBNkU7Z0JBQzVGLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSxpRkFBaUY7Z0JBQ3hGLGFBQWEsRUFBRSxxREFBcUQ7Z0JBQ3BFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsYUFBYSxFQUFFO3dCQUNiLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsVUFBVSxFQUFFO2dCQUNWLEtBQUssRUFBRSwyRUFBMkU7Z0JBQ2xGLGFBQWEsRUFBRSxpREFBaUQ7Z0JBQ2hFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QseUJBQXlCLEVBQUU7Z0JBQ3pCLEtBQUssRUFBRSwwRkFBMEY7Z0JBQ2pHLGFBQWEsRUFBRSw2REFBNkQ7Z0JBQzVFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSxzRkFBc0Y7Z0JBQzdGLGFBQWEsRUFBRSxtREFBbUQ7Z0JBQ2xFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSxtRkFBbUY7Z0JBQzFGLGFBQWEsRUFBRSw4R0FBOEc7Z0JBQzdILGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsZUFBZSxFQUFFO3dCQUNmLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsNkJBQTZCLEVBQUU7Z0JBQzdCLEtBQUssRUFBRSw4RkFBOEY7Z0JBQ3JHLGFBQWEsRUFBRSxnSEFBZ0g7Z0JBQy9ILGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsZUFBZSxFQUFFO3dCQUNmLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSxtRkFBbUY7Z0JBQzFGLGFBQWEsRUFBRSx5REFBeUQ7Z0JBQ3hFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsZUFBZSxFQUFFO3dCQUNmLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSxxRkFBcUY7Z0JBQzVGLGFBQWEsRUFBRSw2REFBNkQ7Z0JBQzVFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsaUJBQWlCLEVBQUU7d0JBQ2pCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsU0FBUyxFQUFFO2dCQUNULEtBQUssRUFBRSwwRUFBMEU7Z0JBQ2pGLGFBQWEsRUFBRSw0RkFBNEY7Z0JBQzNHLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsTUFBTSxFQUFFO3dCQUNOLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSwrRUFBK0U7Z0JBQ3RGLGFBQWEsRUFBRSwyR0FBMkc7Z0JBQzFILGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSxxRkFBcUY7Z0JBQzVGLGFBQWEsRUFBRSxrTkFBa047Z0JBQ2pPLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsTUFBTSxFQUFFO3dCQUNOLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLGlFQUFpRTtnQkFDaEYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixtQkFBbUIsRUFBRTt3QkFDbkIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLDBGQUEwRjtnQkFDakcsYUFBYSxFQUFFLCtFQUErRTtnQkFDOUYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixzQkFBc0IsRUFBRTt3QkFDdEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsS0FBSyxFQUFFLDRFQUE0RTtnQkFDbkYsYUFBYSxFQUFFLG1EQUFtRDtnQkFDbEUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGlGQUFpRjtnQkFDeEYsYUFBYSxFQUFFLDZEQUE2RDtnQkFDNUUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixhQUFhLEVBQUU7d0JBQ2IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCwrQkFBK0IsRUFBRTtnQkFDL0IsS0FBSyxFQUFFLGdHQUFnRztnQkFDdkcsYUFBYSxFQUFFLDRDQUE0QztnQkFDM0QsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLGtEQUFrRDtnQkFDakUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLG1GQUFtRjtnQkFDMUYsYUFBYSxFQUFFLGlEQUFpRDtnQkFDaEUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osS0FBSyxFQUFFLDZFQUE2RTtnQkFDcEYsYUFBYSxFQUFFLDJEQUEyRDtnQkFDMUUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCwyQkFBMkIsRUFBRTtnQkFDM0IsS0FBSyxFQUFFLDRGQUE0RjtnQkFDbkcsYUFBYSxFQUFFLG1EQUFtRDtnQkFDbEUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLDBDQUEwQztnQkFDekQsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLG1EQUFtRDtnQkFDbEUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLHFEQUFxRDtnQkFDcEUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGlGQUFpRjtnQkFDeEYsYUFBYSxFQUFFLHdDQUF3QztnQkFDdkQsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsS0FBSyxFQUFFLDRFQUE0RTtnQkFDbkYsYUFBYSxFQUFFLDBDQUEwQztnQkFDekQsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCx3QkFBd0IsRUFBRTtnQkFDeEIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLHVEQUF1RDtnQkFDdEUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCwyQkFBMkIsRUFBRTtnQkFDM0IsS0FBSyxFQUFFLDRGQUE0RjtnQkFDbkcsYUFBYSxFQUFFLG1EQUFtRDtnQkFDbEUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCwwQkFBMEIsRUFBRTtnQkFDMUIsS0FBSyxFQUFFLDJGQUEyRjtnQkFDbEcsYUFBYSxFQUFFLG1FQUFtRTtnQkFDbEYsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLHNDQUFzQztnQkFDckQsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixlQUFlLEVBQUU7d0JBQ2YsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELE1BQU0sRUFBRTt3QkFDTixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsYUFBYSxFQUFFLDREQUE0RDtnQkFDM0UsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLG1GQUFtRjtnQkFDMUYsYUFBYSxFQUFFLDBDQUEwQztnQkFDekQsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLDBGQUEwRjtnQkFDakcsYUFBYSxFQUFFLDZEQUE2RDtnQkFDNUUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGtCQUFrQixFQUFFO29CQUNsQiw2QkFBNkI7aUJBQzlCO2dCQUNELGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLGtJQUFrSTtnQkFDakosYUFBYSxFQUFFLHdCQUF3QjtnQkFDdkMsZUFBZSxFQUFFO29CQUNmLFdBQVcsRUFBRTt3QkFDWCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELGFBQWEsRUFBRTtnQkFDYixLQUFLLEVBQUUsOEVBQThFO2dCQUNyRixhQUFhLEVBQUUsaUNBQWlDO2dCQUNoRCxhQUFhLEVBQUUsU0FBUztnQkFDeEIsZUFBZSxFQUFFO29CQUNmLGVBQWUsRUFBRTt3QkFDZixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsTUFBTSxFQUFFO3dCQUNOLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxXQUFXLEVBQUU7d0JBQ1gsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSxnRkFBZ0Y7Z0JBQ3ZGLGFBQWEsRUFBRSxzQ0FBc0M7Z0JBQ3JELGFBQWEsRUFBRSxTQUFTO2dCQUN4QixlQUFlLEVBQUU7b0JBQ2YsZUFBZSxFQUFFO3dCQUNmLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxNQUFNLEVBQUU7d0JBQ04sVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELFdBQVcsRUFBRTt3QkFDWCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osYUFBYTtpQkFDZDthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSxxRkFBcUY7Z0JBQzVGLGFBQWEsRUFBRSx3RUFBd0U7Z0JBQ3ZGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsY0FBYyxFQUFFO3dCQUNkLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGFBQWEsRUFBRSxpRUFBaUU7Z0JBQ2hGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsYUFBYSxFQUFFO3dCQUNiLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSw4RUFBOEU7Z0JBQ3JGLGFBQWEsRUFBRSx5REFBeUQ7Z0JBQ3hFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSxzRkFBc0Y7Z0JBQzdGLGFBQWEsRUFBRSx1RkFBdUY7Z0JBQ3RHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsZUFBZSxFQUFFO3dCQUNmLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSxzRkFBc0Y7Z0JBQzdGLGFBQWEsRUFBRSwwRUFBMEU7Z0JBQ3pGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsZUFBZSxFQUFFO3dCQUNmLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsdUJBQXVCLEVBQUU7Z0JBQ3ZCLEtBQUssRUFBRSx3RkFBd0Y7Z0JBQy9GLGFBQWEsRUFBRSw4REFBOEQ7Z0JBQzdFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsaUJBQWlCLEVBQUU7d0JBQ2pCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsWUFBWSxFQUFFO2dCQUNaLEtBQUssRUFBRSw2RUFBNkU7Z0JBQ3BGLGFBQWEsRUFBRSxpREFBaUQ7Z0JBQ2hFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsTUFBTSxFQUFFO3dCQUNOLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSxrRkFBa0Y7Z0JBQ3pGLGFBQWEsRUFBRSwwREFBMEQ7Z0JBQ3pFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QseUJBQXlCLEVBQUU7Z0JBQ3pCLEtBQUssRUFBRSwwRkFBMEY7Z0JBQ2pHLGFBQWEsRUFBRSw2RUFBNkU7Z0JBQzVGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsbUJBQW1CLEVBQUU7d0JBQ25CLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsNEJBQTRCLEVBQUU7Z0JBQzVCLEtBQUssRUFBRSw2RkFBNkY7Z0JBQ3BHLGFBQWEsRUFBRSx3RkFBd0Y7Z0JBQ3ZHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2Ysc0JBQXNCLEVBQUU7d0JBQ3RCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSwrRUFBK0U7Z0JBQ3RGLGFBQWEsRUFBRSx1REFBdUQ7Z0JBQ3RFLGFBQWEsRUFBRSx3QkFBd0I7Z0JBQ3ZDLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLHVFQUF1RTtnQkFDdEYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixhQUFhLEVBQUU7d0JBQ2IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7U0FDRixDQUFDO1FBQ1Esa0JBQWEsR0FBa0I7WUFDdkMsY0FBYyxFQUFFO2dCQUNkLE1BQU0sRUFBRSxjQUFjO2dCQUN0QixLQUFLLEVBQUUsK0VBQStFO2dCQUN0RixLQUFLLEVBQUUscURBQXFEO2dCQUM1RCxlQUFlLEVBQUUsRUFBRTthQUNwQjtZQUNELE9BQU8sRUFBRTtnQkFDUCxNQUFNLEVBQUUsT0FBTztnQkFDZixLQUFLLEVBQUUsd0VBQXdFO2dCQUMvRSxLQUFLLEVBQUUsOENBQThDO2dCQUNyRCxlQUFlLEVBQUUsRUFBRTthQUNwQjtZQUNELGVBQWUsRUFBRTtnQkFDZixNQUFNLEVBQUUsZUFBZTtnQkFDdkIsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsS0FBSyxFQUFFLHNEQUFzRDtnQkFDN0QsZUFBZSxFQUFFO29CQUNmLDJCQUEyQjtpQkFDNUI7YUFDRjtZQUNELE1BQU0sRUFBRTtnQkFDTixNQUFNLEVBQUUsTUFBTTtnQkFDZCxLQUFLLEVBQUUsdUVBQXVFO2dCQUM5RSxLQUFLLEVBQUUsNkNBQTZDO2dCQUNwRCxlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLE1BQU0sRUFBRSxtQkFBbUI7Z0JBQzNCLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLEtBQUssRUFBRSwwREFBMEQ7Z0JBQ2pFLGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1lBQ0Qsc0JBQXNCLEVBQUU7Z0JBQ3RCLE1BQU0sRUFBRSxzQkFBc0I7Z0JBQzlCLEtBQUssRUFBRSx1RkFBdUY7Z0JBQzlGLEtBQUssRUFBRSx3REFBd0Q7Z0JBQy9ELGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1lBQ0QsUUFBUSxFQUFFO2dCQUNSLE1BQU0sRUFBRSxRQUFRO2dCQUNoQixLQUFLLEVBQUUseUVBQXlFO2dCQUNoRixLQUFLLEVBQUUsK0NBQStDO2dCQUN0RCxlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLE1BQU0sRUFBRSxhQUFhO2dCQUNyQixLQUFLLEVBQUUsOEVBQThFO2dCQUNyRixLQUFLLEVBQUUsb0RBQW9EO2dCQUMzRCxlQUFlLEVBQUUsRUFBRTthQUNwQjtZQUNELGVBQWUsRUFBRTtnQkFDZixNQUFNLEVBQUUsZUFBZTtnQkFDdkIsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsS0FBSyxFQUFFLG1EQUFtRDtnQkFDMUQsZUFBZSxFQUFFLEVBQUU7YUFDcEI7WUFDRCxpQkFBaUIsRUFBRTtnQkFDakIsTUFBTSxFQUFFLGlCQUFpQjtnQkFDekIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsS0FBSyxFQUFFLHdEQUF3RDtnQkFDL0QsZUFBZSxFQUFFLEVBQUU7YUFDcEI7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsTUFBTSxFQUFFLGFBQWE7Z0JBQ3JCLEtBQUssRUFBRSw4RUFBOEU7Z0JBQ3JGLEtBQUssRUFBRSxvREFBb0Q7Z0JBQzNELGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLE1BQU0sRUFBRSxXQUFXO2dCQUNuQixLQUFLLEVBQUUsNEVBQTRFO2dCQUNuRixLQUFLLEVBQUUsa0RBQWtEO2dCQUN6RCxlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO2lCQUM1QjthQUNGO1NBQ0YsQ0FBQztJQVNGLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDNUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxVQUFVO1FBQ2YsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQzNCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxHQUFHLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixJQUFJLENBQUMsR0FBRyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzdCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDBCQUEwQjtRQUMvQixJQUFJLENBQUMsR0FBRyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDNUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxHQUFHLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFVBQVU7UUFDZixJQUFJLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDM0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxHQUFHLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzdCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUMvQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxRQUFRO1FBQ2IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUN6QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO1FBQzVDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxPQUFPO1FBQ1osSUFBSSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUN4QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxHQUFHLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUM3QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxHQUFHLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxTQUFTO1FBQ2QsSUFBSSxDQUFDLEdBQUcsQ0FBQyxlQUFlLENBQUMsQ0FBQztRQUMxQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUMvQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO1FBQzlDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxVQUFVO1FBQ2YsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQzNCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksU0FBUztRQUNkLElBQUksQ0FBQyxHQUFHLENBQUMsZUFBZSxDQUFDLENBQUM7UUFDMUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsSUFBSSxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixJQUFJLENBQUMsR0FBRyxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUM1QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDNUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDOUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQzVCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFVBQVU7UUFDZixJQUFJLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDM0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxHQUFHLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzdCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSxjQUFjLENBQUMsRUFBVSxFQUFFLE9BQWdCLEVBQUUsU0FBa0I7UUFDcEUsSUFBSSxHQUFHLEdBQUcscURBQXFELENBQUM7UUFDaEUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQy9CLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksT0FBTyxDQUFDLEVBQVUsRUFBRSxPQUFnQixFQUFFLFNBQWtCO1FBQzdELElBQUksR0FBRyxHQUFHLDhDQUE4QyxDQUFDO1FBQ3pELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUMvQixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGVBQWUsQ0FBQyxFQUFVLEVBQUUsT0FBZ0IsRUFBRSxTQUFrQjtRQUNyRSxJQUFJLEdBQUcsR0FBRyxzREFBc0QsQ0FBQztRQUNqRSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDL0IsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxNQUFNLENBQUMsRUFBVSxFQUFFLE9BQWdCLEVBQUUsU0FBa0I7UUFDNUQsSUFBSSxHQUFHLEdBQUcsNkNBQTZDLENBQUM7UUFDeEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQy9CLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksbUJBQW1CLENBQUMsRUFBVSxFQUFFLE9BQWdCLEVBQUUsU0FBa0I7UUFDekUsSUFBSSxHQUFHLEdBQUcsMERBQTBELENBQUM7UUFDckUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQy9CLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksc0JBQXNCLENBQUMsRUFBVSxFQUFFLE9BQWdCLEVBQUUsU0FBa0I7UUFDNUUsSUFBSSxHQUFHLEdBQUcsd0RBQXdELENBQUM7UUFDbkUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQy9CLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksUUFBUSxDQUFDLEVBQVUsRUFBRSxPQUFnQixFQUFFLFNBQWtCO1FBQzlELElBQUksR0FBRyxHQUFHLCtDQUErQyxDQUFDO1FBQzFELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUMvQixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNJLGFBQWEsQ0FBQyxFQUFVLEVBQUUsT0FBZ0IsRUFBRSxTQUFrQjtRQUNuRSxJQUFJLEdBQUcsR0FBRyxvREFBb0QsQ0FBQztRQUMvRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDL0IsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSxlQUFlLENBQUMsRUFBVSxFQUFFLE9BQWdCLEVBQUUsU0FBa0I7UUFDckUsSUFBSSxHQUFHLEdBQUcsbURBQW1ELENBQUM7UUFDOUQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQy9CLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksaUJBQWlCLENBQUMsRUFBVSxFQUFFLE9BQWdCLEVBQUUsU0FBa0I7UUFDdkUsSUFBSSxHQUFHLEdBQUcsd0RBQXdELENBQUM7UUFDbkUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQy9CLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksYUFBYSxDQUFDLEVBQVUsRUFBRSxPQUFnQixFQUFFLFNBQWtCO1FBQ25FLElBQUksR0FBRyxHQUFHLG9EQUFvRCxDQUFDO1FBQy9ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUMvQixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLFdBQVcsQ0FBQyxFQUFVLEVBQUUsT0FBZ0IsRUFBRSxTQUFrQjtRQUNqRSxJQUFJLEdBQUcsR0FBRyxrREFBa0QsQ0FBQztRQUM3RCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDL0IsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksZUFBZSxDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQWlCO1FBQ2hGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBbUIsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNoRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxnQkFBZ0IsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUFpQjtRQUNqRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW9CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQWlCO1FBQzdELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0NBQ0Y7QUFuaEVELGtCQW1oRUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY3Rpb25zLCBQb2xpY3lTdGF0ZW1lbnQsIFJlc291cmNlVHlwZXMgfSBmcm9tIFwiLi4vc2hhcmVkXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFt3YWZdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2F3c3dhZi5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIFdhZiBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ3dhZic7XG4gIHByb3RlY3RlZCBhY3Rpb25MaXN0OiBBY3Rpb25zID0ge1xuICAgIFwiQ3JlYXRlQnl0ZU1hdGNoU2V0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfQ3JlYXRlQnl0ZU1hdGNoU2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGEgQnl0ZU1hdGNoU2V0LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ5dGVtYXRjaHNldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQ3JlYXRlR2VvTWF0Y2hTZXRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9DcmVhdGVHZW9NYXRjaFNldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIEdlb01hdGNoU2V0LCB3aGljaCB5b3UgdXNlIHRvIHNwZWNpZnkgd2hpY2ggd2ViIHJlcXVlc3RzIHlvdSB3YW50IHRvIGFsbG93IG9yIGJsb2NrIGJhc2VkIG9uIHRoZSBjb3VudHJ5IHRoYXQgdGhlIHJlcXVlc3RzIG9yaWdpbmF0ZSBmcm9tLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImdlb21hdGNoc2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJDcmVhdGVJUFNldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0NyZWF0ZUlQU2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGFuIElQU2V0LCB3aGljaCB5b3UgdXNlIHRvIHNwZWNpZnkgd2hpY2ggd2ViIHJlcXVlc3RzIHlvdSB3YW50IHRvIGFsbG93IG9yIGJsb2NrIGJhc2VkIG9uIHRoZSBJUCBhZGRyZXNzZXMgdGhhdCB0aGUgcmVxdWVzdHMgb3JpZ2luYXRlIGZyb20uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiaXBzZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZVJhdGVCYXNlZFJ1bGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9DcmVhdGVSYXRlQmFzZWRSdWxlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGEgUmF0ZUJhc2VkUnVsZSwgd2hpY2ggY29udGFpbnMgYSBSYXRlTGltaXQgc3BlY2lmeWluZyB0aGUgbWF4aW11bSBudW1iZXIgb2YgcmVxdWVzdHMgdGhhdCBBV1MgV0FGIGFsbG93cyBmcm9tIGEgc3BlY2lmaWVkIElQIGFkZHJlc3MgaW4gYSBmaXZlLW1pbnV0ZSBwZXJpb2QuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicmF0ZWJhc2VkcnVsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiQ3JlYXRlUmVnZXhNYXRjaFNldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0NyZWF0ZVJlZ2V4TWF0Y2hTZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYSBSZWdleE1hdGNoU2V0LCB3aGljaCB5b3UgdXNlIHRvIHNwZWNpZnkgd2hpY2ggd2ViIHJlcXVlc3RzIHlvdSB3YW50IHRvIGFsbG93IG9yIGJsb2NrIGJhc2VkIG9uIHRoZSByZWdleCBwYXR0ZXJucyB5b3Ugc3BlY2lmaWVkIGluIGEgUmVnZXhQYXR0ZXJuU2V0LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJlZ2V4bWF0Y2hzZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZVJlZ2V4UGF0dGVyblNldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0NyZWF0ZVJlZ2V4UGF0dGVyblNldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIFJlZ2V4UGF0dGVyblNldCwgd2hpY2ggeW91IHVzZSB0byBzcGVjaWZ5IHRoZSByZWd1bGFyIGV4cHJlc3Npb24gKHJlZ2V4KSBwYXR0ZXJuIHRoYXQgeW91IHdhbnQgQVdTIFdBRiB0byBzZWFyY2ggZm9yLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJlZ2V4cGF0dGVybnNldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQ3JlYXRlUnVsZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0NyZWF0ZVJ1bGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYSBSdWxlLCB3aGljaCBjb250YWlucyB0aGUgSVBTZXQgb2JqZWN0cywgQnl0ZU1hdGNoU2V0IG9iamVjdHMsIGFuZCBvdGhlciBwcmVkaWNhdGVzIHRoYXQgaWRlbnRpZnkgdGhlIHJlcXVlc3RzIHRoYXQgeW91IHdhbnQgdG8gYmxvY2suXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicnVsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiQ3JlYXRlUnVsZUdyb3VwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfQ3JlYXRlUnVsZUdyb3VwLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGEgUnVsZUdyb3VwLiBBIHJ1bGUgZ3JvdXAgaXMgYSBjb2xsZWN0aW9uIG9mIHByZWRlZmluZWQgcnVsZXMgdGhhdCB5b3UgYWRkIHRvIGEgV2ViQUNMLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJ1bGVncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiQ3JlYXRlU2l6ZUNvbnN0cmFpbnRTZXRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9DcmVhdGVTaXplQ29uc3RyYWludFNldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIFNpemVDb25zdHJhaW50U2V0LCB3aGljaCB5b3UgdXNlIHRvIGlkZW50aWZ5IHRoZSBwYXJ0IG9mIGEgd2ViIHJlcXVlc3QgdGhhdCB5b3Ugd2FudCB0byBjaGVjayBmb3IgbGVuZ3RoLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInNpemVjb25zdHJhaW50c2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJDcmVhdGVTcWxJbmplY3Rpb25NYXRjaFNldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0NyZWF0ZVNxbEluamVjdGlvbk1hdGNoU2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGEgU3FsSW5qZWN0aW9uTWF0Y2hTZXQsIHdoaWNoIHlvdSB1c2UgdG8gYWxsb3csIGJsb2NrLCBvciBjb3VudCByZXF1ZXN0cyB0aGF0IGNvbnRhaW4gc25pcHBldHMgb2YgU1FMIGNvZGUgaW4gYSBzcGVjaWZpZWQgcGFydCBvZiB3ZWIgcmVxdWVzdHMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic3FsaW5qZWN0aW9ubWF0Y2hzZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZVdlYkFDTFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0NyZWF0ZVdlYkFDTC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIFdlYkFDTCwgd2hpY2ggY29udGFpbnMgdGhlIFJ1bGVzIHRoYXQgaWRlbnRpZnkgdGhlIENsb3VkRnJvbnQgd2ViIHJlcXVlc3RzIHRoYXQgeW91IHdhbnQgdG8gYWxsb3csIGJsb2NrLCBvciBjb3VudC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIndlYmFjbFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiQ3JlYXRlV2ViQUNMTWlncmF0aW9uU3RhY2tcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9DcmVhdGVXZWJBQ0xNaWdyYXRpb25TdGFjay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlIGFuZCBzdG9yZSBhIENsb3VkRm9ybWF0aW9uIHRlbXBhbHRlIHRoYXQgY3JlYXRlcyBhbiBlcXVpdmFsZW50IFdBRiB2MiBXZWJBQ0wgZnJvbSB0aGUgZ2l2ZW4gV0FGIENsYXNzaWMgV2ViQUNMIGluIHRoZSBnaXZlbiBTMyBidWNrZXQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwiZGVwZW5kZW50QWN0aW9uc1wiOiBbXG4gICAgICAgIFwiczM6UHV0T2JqZWN0XCJcbiAgICAgIF0sXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIndlYmFjbFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQ3JlYXRlWHNzTWF0Y2hTZXRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9DcmVhdGVYc3NNYXRjaFNldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhbiBYc3NNYXRjaFNldCwgd2hpY2ggeW91IHVzZSB0byBhbGxvdywgYmxvY2ssIG9yIGNvdW50IHJlcXVlc3RzIHRoYXQgY29udGFpbiBjcm9zcy1zaXRlIHNjcmlwdGluZyBhdHRhY2tzIGluIHRoZSBzcGVjaWZpZWQgcGFydCBvZiB3ZWIgcmVxdWVzdHMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwieHNzbWF0Y2hzZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZUJ5dGVNYXRjaFNldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0RlbGV0ZUJ5dGVNYXRjaFNldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUGVybWFuZW50bHkgZGVsZXRlcyBhIEJ5dGVNYXRjaFNldC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJieXRlbWF0Y2hzZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZUdlb01hdGNoU2V0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfRGVsZXRlR2VvTWF0Y2hTZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlBlcm1hbmVudGx5IGRlbGV0ZXMgYW4gR2VvTWF0Y2hTZXQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZ2VvbWF0Y2hzZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZUlQU2V0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfRGVsZXRlSVBTZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlBlcm1hbmVudGx5IGRlbGV0ZXMgYW4gSVBTZXQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiaXBzZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZUxvZ2dpbmdDb25maWd1cmF0aW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfRGVsZXRlTG9nZ2luZ0NvbmZpZ3VyYXRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlBlcm1hbmVudGx5IGRlbGV0ZXMgdGhlIExvZ2dpbmdDb25maWd1cmF0aW9uIGZyb20gdGhlIHNwZWNpZmllZCB3ZWIgQUNMLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIndlYmFjbFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlUGVybWlzc2lvblBvbGljeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0RlbGV0ZVBlcm1pc3Npb25Qb2xpY3kuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlBlcm1hbmVudGx5IGRlbGV0ZXMgYW4gSUFNIHBvbGljeSBmcm9tIHRoZSBzcGVjaWZpZWQgUnVsZUdyb3VwLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlBlcm1pc3Npb25zIG1hbmFnZW1lbnRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicnVsZWdyb3VwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVSYXRlQmFzZWRSdWxlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfRGVsZXRlUmF0ZUJhc2VkUnVsZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUGVybWFuZW50bHkgZGVsZXRlcyBhIFJhdGVCYXNlZFJ1bGUuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicmF0ZWJhc2VkcnVsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlUmVnZXhNYXRjaFNldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0RlbGV0ZVJlZ2V4TWF0Y2hTZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlBlcm1hbmVudGx5IGRlbGV0ZXMgYW4gUmVnZXhNYXRjaFNldC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJyZWdleG1hdGNoc2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVSZWdleFBhdHRlcm5TZXRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9EZWxldGVSZWdleFBhdHRlcm5TZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlBlcm1hbmVudGx5IGRlbGV0ZXMgYW4gUmVnZXhQYXR0ZXJuU2V0LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJlZ2V4cGF0dGVybnNldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlUnVsZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0RlbGV0ZVJ1bGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlBlcm1hbmVudGx5IGRlbGV0ZXMgYSBSdWxlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJ1bGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZVJ1bGVHcm91cFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0RlbGV0ZVJ1bGVHcm91cC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUGVybWFuZW50bHkgZGVsZXRlcyBhIFJ1bGVHcm91cC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJydWxlZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZVNpemVDb25zdHJhaW50U2V0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfRGVsZXRlU2l6ZUNvbnN0cmFpbnRTZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlBlcm1hbmVudGx5IGRlbGV0ZXMgYSBTaXplQ29uc3RyYWludFNldC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzaXplY29uc3RyYWludHNldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlU3FsSW5qZWN0aW9uTWF0Y2hTZXRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9EZWxldGVTcWxJbmplY3Rpb25NYXRjaFNldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUGVybWFuZW50bHkgZGVsZXRlcyBhIFNxbEluamVjdGlvbk1hdGNoU2V0LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInNxbGluamVjdGlvbm1hdGNoc2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVXZWJBQ0xcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9EZWxldGVXZWJBQ0wuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlBlcm1hbmVudGx5IGRlbGV0ZXMgYSBXZWJBQ0wuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUGVybWlzc2lvbnMgbWFuYWdlbWVudFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ3ZWJhY2xcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZVhzc01hdGNoU2V0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfRGVsZXRlWHNzTWF0Y2hTZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlBlcm1hbmVudGx5IGRlbGV0ZXMgYW4gWHNzTWF0Y2hTZXQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwieHNzbWF0Y2hzZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldEJ5dGVNYXRjaFNldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0dldEJ5dGVNYXRjaFNldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyB0aGUgQnl0ZU1hdGNoU2V0IHNwZWNpZmllZCBieSBCeXRlTWF0Y2hTZXRJZC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ5dGVtYXRjaHNldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0Q2hhbmdlVG9rZW5cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9HZXRDaGFuZ2VUb2tlbi5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiV2hlbiB5b3Ugd2FudCB0byBjcmVhdGUsIHVwZGF0ZSwgb3IgZGVsZXRlIEFXUyBXQUYgb2JqZWN0cywgZ2V0IGEgY2hhbmdlIHRva2VuIGFuZCBpbmNsdWRlIHRoZSBjaGFuZ2UgdG9rZW4gaW4gdGhlIGNyZWF0ZSwgdXBkYXRlLCBvciBkZWxldGUgcmVxdWVzdC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiR2V0Q2hhbmdlVG9rZW5TdGF0dXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9HZXRDaGFuZ2VUb2tlblN0YXR1cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyB0aGUgc3RhdHVzIG9mIGEgQ2hhbmdlVG9rZW4gdGhhdCB5b3UgZ290IGJ5IGNhbGxpbmcgR2V0Q2hhbmdlVG9rZW4uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkdldEdlb01hdGNoU2V0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfR2V0R2VvTWF0Y2hTZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgdGhlIEdlb01hdGNoU2V0IHNwZWNpZmllZCBieSBHZW9NYXRjaFNldElkLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZ2VvbWF0Y2hzZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldElQU2V0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfR2V0SVBTZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgdGhlIElQU2V0IHRoYXQgaXMgc3BlY2lmaWVkIGJ5IElQU2V0SWQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJpcHNldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0TG9nZ2luZ0NvbmZpZ3VyYXRpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9HZXRMb2dnaW5nQ29uZmlndXJhdGlvbi5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyB0aGUgTG9nZ2luZ0NvbmZpZ3VyYXRpb24gZm9yIHRoZSBzcGVjaWZpZWQgd2ViIEFDTC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIndlYmFjbFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0UGVybWlzc2lvblBvbGljeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0dldFBlcm1pc3Npb25Qb2xpY3kuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgdGhlIElBTSBwb2xpY3kgYXR0YWNoZWQgdG8gdGhlIFJ1bGVHcm91cC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJ1bGVncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0UmF0ZUJhc2VkUnVsZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0dldFJhdGVCYXNlZFJ1bGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgdGhlIFJhdGVCYXNlZFJ1bGUgdGhhdCBpcyBzcGVjaWZpZWQgYnkgdGhlIFJ1bGVJZCB0aGF0IHlvdSBpbmNsdWRlZCBpbiB0aGUgR2V0UmF0ZUJhc2VkUnVsZSByZXF1ZXN0LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicmF0ZWJhc2VkcnVsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0UmF0ZUJhc2VkUnVsZU1hbmFnZWRLZXlzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfR2V0UmF0ZUJhc2VkUnVsZU1hbmFnZWRLZXlzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGFuIGFycmF5IG9mIElQIGFkZHJlc3NlcyBjdXJyZW50bHkgYmVpbmcgYmxvY2tlZCBieSB0aGUgUmF0ZUJhc2VkUnVsZSB0aGF0IGlzIHNwZWNpZmllZCBieSB0aGUgUnVsZUlkLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicmF0ZWJhc2VkcnVsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0UmVnZXhNYXRjaFNldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0dldFJlZ2V4TWF0Y2hTZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgdGhlIFJlZ2V4TWF0Y2hTZXQgc3BlY2lmaWVkIGJ5IFJlZ2V4TWF0Y2hTZXRJZC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJlZ2V4bWF0Y2hzZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldFJlZ2V4UGF0dGVyblNldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0dldFJlZ2V4UGF0dGVyblNldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyB0aGUgUmVnZXhQYXR0ZXJuU2V0IHNwZWNpZmllZCBieSBSZWdleFBhdHRlcm5TZXRJZC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJlZ2V4cGF0dGVybnNldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0UnVsZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0dldFJ1bGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgdGhlIFJ1bGUgdGhhdCBpcyBzcGVjaWZpZWQgYnkgdGhlIFJ1bGVJZCB0aGF0IHlvdSBpbmNsdWRlZCBpbiB0aGUgR2V0UnVsZSByZXF1ZXN0LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicnVsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0UnVsZUdyb3VwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfR2V0UnVsZUdyb3VwLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIHRoZSBSdWxlR3JvdXAgdGhhdCBpcyBzcGVjaWZpZWQgYnkgdGhlIFJ1bGVHcm91cElkIHRoYXQgeW91IGluY2x1ZGVkIGluIHRoZSBHZXRSdWxlR3JvdXAgcmVxdWVzdC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJ1bGVncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0U2FtcGxlZFJlcXVlc3RzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfR2V0U2FtcGxlZFJlcXVlc3RzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHZXRzIGRldGFpbGVkIGluZm9ybWF0aW9uIGFib3V0IGEgc3BlY2lmaWVkIG51bWJlciBvZiByZXF1ZXN0cy0tYSBzYW1wbGUtLXRoYXQgQVdTIFdBRiByYW5kb21seSBzZWxlY3RzIGZyb20gYW1vbmcgdGhlIGZpcnN0IDUsMDAwIHJlcXVlc3RzIHRoYXQgeW91ciBBV1MgcmVzb3VyY2UgcmVjZWl2ZWQgZHVyaW5nIGEgdGltZSByYW5nZSB0aGF0IHlvdSBjaG9vc2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJydWxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwid2ViYWNsXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0U2l6ZUNvbnN0cmFpbnRTZXRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9HZXRTaXplQ29uc3RyYWludFNldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyB0aGUgU2l6ZUNvbnN0cmFpbnRTZXQgc3BlY2lmaWVkIGJ5IFNpemVDb25zdHJhaW50U2V0SWQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzaXplY29uc3RyYWludHNldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0U3FsSW5qZWN0aW9uTWF0Y2hTZXRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9HZXRTcWxJbmplY3Rpb25NYXRjaFNldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyB0aGUgU3FsSW5qZWN0aW9uTWF0Y2hTZXQgdGhhdCBpcyBzcGVjaWZpZWQgYnkgU3FsSW5qZWN0aW9uTWF0Y2hTZXRJZC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInNxbGluamVjdGlvbm1hdGNoc2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRXZWJBQ0xcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9HZXRXZWJBQ0wuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgdGhlIFdlYkFDTCB0aGF0IGlzIHNwZWNpZmllZCBieSBXZWJBQ0xJZC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIndlYmFjbFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0WHNzTWF0Y2hTZXRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9HZXRYc3NNYXRjaFNldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyB0aGUgWHNzTWF0Y2hTZXQgdGhhdCBpcyBzcGVjaWZpZWQgYnkgWHNzTWF0Y2hTZXRJZC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInhzc21hdGNoc2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJMaXN0QWN0aXZhdGVkUnVsZXNJblJ1bGVHcm91cFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0xpc3RBY3RpdmF0ZWRSdWxlc0luUnVsZUdyb3VwLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGFuIGFycmF5IG9mIEFjdGl2YXRlZFJ1bGUgb2JqZWN0cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiTGlzdEJ5dGVNYXRjaFNldHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9MaXN0Qnl0ZU1hdGNoU2V0cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhbiBhcnJheSBvZiBCeXRlTWF0Y2hTZXRTdW1tYXJ5IG9iamVjdHMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RHZW9NYXRjaFNldHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9MaXN0R2VvTWF0Y2hTZXRzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGFuIGFycmF5IG9mIEdlb01hdGNoU2V0U3VtbWFyeSBvYmplY3RzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0SVBTZXRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfTGlzdElQU2V0cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhbiBhcnJheSBvZiBJUFNldFN1bW1hcnkgb2JqZWN0cyBpbiB0aGUgcmVzcG9uc2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RMb2dnaW5nQ29uZmlndXJhdGlvbnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9MaXN0TG9nZ2luZ0NvbmZpZ3VyYXRpb25zLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGFuIGFycmF5IG9mIExvZ2dpbmdDb25maWd1cmF0aW9uIG9iamVjdHMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RSYXRlQmFzZWRSdWxlc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0xpc3RSYXRlQmFzZWRSdWxlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhbiBhcnJheSBvZiBSdWxlU3VtbWFyeSBvYmplY3RzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0UmVnZXhNYXRjaFNldHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9MaXN0UmVnZXhNYXRjaFNldHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYW4gYXJyYXkgb2YgUmVnZXhNYXRjaFNldFN1bW1hcnkgb2JqZWN0cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiTGlzdFJlZ2V4UGF0dGVyblNldHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9MaXN0UmVnZXhQYXR0ZXJuU2V0cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhbiBhcnJheSBvZiBSZWdleFBhdHRlcm5TZXRTdW1tYXJ5IG9iamVjdHMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RSdWxlR3JvdXBzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfTGlzdFJ1bGVHcm91cHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYW4gYXJyYXkgb2YgUnVsZUdyb3VwIG9iamVjdHMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkxpc3RSdWxlc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0xpc3RSdWxlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhbiBhcnJheSBvZiBSdWxlU3VtbWFyeSBvYmplY3RzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0U2l6ZUNvbnN0cmFpbnRTZXRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfTGlzdFNpemVDb25zdHJhaW50U2V0cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhbiBhcnJheSBvZiBTaXplQ29uc3RyYWludFNldFN1bW1hcnkgb2JqZWN0cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiTGlzdFNxbEluamVjdGlvbk1hdGNoU2V0c1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0xpc3RTcWxJbmplY3Rpb25NYXRjaFNldHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYW4gYXJyYXkgb2YgU3FsSW5qZWN0aW9uTWF0Y2hTZXQgb2JqZWN0cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiTGlzdFN1YnNjcmliZWRSdWxlR3JvdXBzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfTGlzdFN1YnNjcmliZWRSdWxlR3JvdXBzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGFuIGFycmF5IG9mIFJ1bGVHcm91cCBvYmplY3RzIHRoYXQgeW91IGFyZSBzdWJzY3JpYmVkIHRvLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0VGFnc0ZvclJlc291cmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfTGlzdFRhZ3NGb3JSZXNvdXJjZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiTGlzdHMgdGhlIFRhZ3MgZm9yIGEgZ2l2ZW4gcmVzb3VyY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJyYXRlYmFzZWRydWxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwicnVsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcInJ1bGVncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcIndlYmFjbFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkxpc3RXZWJBQ0xzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfTGlzdFdlYkFDTHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYW4gYXJyYXkgb2YgV2ViQUNMU3VtbWFyeSBvYmplY3RzIGluIHRoZSByZXNwb25zZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiTGlzdFhzc01hdGNoU2V0c1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0xpc3RYc3NNYXRjaFNldHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYW4gYXJyYXkgb2YgWHNzTWF0Y2hTZXQgb2JqZWN0cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiUHV0TG9nZ2luZ0NvbmZpZ3VyYXRpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9QdXRMb2dnaW5nQ29uZmlndXJhdGlvbi5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQXNzb2NpYXRlcyBhIExvZ2dpbmdDb25maWd1cmF0aW9uIHdpdGggYSBzcGVjaWZpZWQgd2ViIEFDTC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJkZXBlbmRlbnRBY3Rpb25zXCI6IFtcbiAgICAgICAgXCJpYW06Q3JlYXRlU2VydmljZUxpbmtlZFJvbGVcIlxuICAgICAgXSxcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwid2ViYWNsXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJQdXRQZXJtaXNzaW9uUG9saWN5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfUHV0UGVybWlzc2lvblBvbGljeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQXR0YWNoZXMgYSBJQU0gcG9saWN5IHRvIHRoZSBzcGVjaWZpZWQgcmVzb3VyY2UuIFRoZSBvbmx5IHN1cHBvcnRlZCB1c2UgZm9yIHRoaXMgYWN0aW9uIGlzIHRvIHNoYXJlIGEgUnVsZUdyb3VwIGFjcm9zcyBhY2NvdW50cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJ1bGVncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVGFnUmVzb3VyY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9UYWdSZXNvdXJjZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQWRkcyBhIFRhZyB0byBhIGdpdmVuIHJlc291cmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlRhZ2dpbmdcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicmF0ZWJhc2VkcnVsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcInJ1bGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJydWxlZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJ3ZWJhY2xcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJVbnRhZ1Jlc291cmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfVW50YWdSZXNvdXJjZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmVtb3ZlcyBhIFRhZyBmcm9tIGEgZ2l2ZW4gcmVzb3VyY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiVGFnZ2luZ1wiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJyYXRlYmFzZWRydWxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwicnVsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcInJ1bGVncm91cFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcIndlYmFjbFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlVwZGF0ZUJ5dGVNYXRjaFNldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX1VwZGF0ZUJ5dGVNYXRjaFNldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiSW5zZXJ0cyBvciBkZWxldGVzIEJ5dGVNYXRjaFR1cGxlIG9iamVjdHMgKGZpbHRlcnMpIGluIGEgQnl0ZU1hdGNoU2V0LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImJ5dGVtYXRjaHNldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlR2VvTWF0Y2hTZXRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9VcGRhdGVHZW9NYXRjaFNldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiSW5zZXJ0cyBvciBkZWxldGVzIEdlb01hdGNoQ29uc3RyYWludCBvYmplY3RzIGluIGEgR2VvTWF0Y2hTZXQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZ2VvbWF0Y2hzZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZUlQU2V0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfVXBkYXRlSVBTZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkluc2VydHMgb3IgZGVsZXRlcyBJUFNldERlc2NyaXB0b3Igb2JqZWN0cyBpbiBhbiBJUFNldC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJpcHNldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlUmF0ZUJhc2VkUnVsZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX1VwZGF0ZVJhdGVCYXNlZFJ1bGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkluc2VydHMgb3IgZGVsZXRlcyBQcmVkaWNhdGUgb2JqZWN0cyBpbiBhIHJ1bGUgYW5kIHVwZGF0ZXMgdGhlIFJhdGVMaW1pdCBpbiB0aGUgcnVsZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJyYXRlYmFzZWRydWxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVSZWdleE1hdGNoU2V0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfVXBkYXRlUmVnZXhNYXRjaFNldC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiSW5zZXJ0cyBvciBkZWxldGVzIFJlZ2V4TWF0Y2hUdXBsZSBvYmplY3RzIChmaWx0ZXJzKSBpbiBhIFJlZ2V4TWF0Y2hTZXQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicmVnZXhtYXRjaHNldFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlUmVnZXhQYXR0ZXJuU2V0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfVXBkYXRlUmVnZXhQYXR0ZXJuU2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJJbnNlcnRzIG9yIGRlbGV0ZXMgUmVnZXhQYXR0ZXJuU3RyaW5ncyBpbiBhIFJlZ2V4UGF0dGVyblNldC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJyZWdleHBhdHRlcm5zZXRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZVJ1bGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9VcGRhdGVSdWxlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJJbnNlcnRzIG9yIGRlbGV0ZXMgUHJlZGljYXRlIG9iamVjdHMgaW4gYSBSdWxlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJ1bGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZVJ1bGVHcm91cFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX1VwZGF0ZVJ1bGVHcm91cC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiSW5zZXJ0cyBvciBkZWxldGVzIEFjdGl2YXRlZFJ1bGUgb2JqZWN0cyBpbiBhIFJ1bGVHcm91cC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJydWxlZ3JvdXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZVNpemVDb25zdHJhaW50U2V0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfVXBkYXRlU2l6ZUNvbnN0cmFpbnRTZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkluc2VydHMgb3IgZGVsZXRlcyBTaXplQ29uc3RyYWludCBvYmplY3RzIChmaWx0ZXJzKSBpbiBhIFNpemVDb25zdHJhaW50U2V0LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInNpemVjb25zdHJhaW50c2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVTcWxJbmplY3Rpb25NYXRjaFNldFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX1VwZGF0ZVNxbEluamVjdGlvbk1hdGNoU2V0Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJJbnNlcnRzIG9yIGRlbGV0ZXMgU3FsSW5qZWN0aW9uTWF0Y2hUdXBsZSBvYmplY3RzIChmaWx0ZXJzKSBpbiBhIFNxbEluamVjdGlvbk1hdGNoU2V0LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInNxbGluamVjdGlvbm1hdGNoc2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVXZWJBQ0xcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9VcGRhdGVXZWJBQ0wuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkluc2VydHMgb3IgZGVsZXRlcyBBY3RpdmF0ZWRSdWxlIG9iamVjdHMgaW4gYSBXZWJBQ0wuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUGVybWlzc2lvbnMgbWFuYWdlbWVudFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ3ZWJhY2xcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZVhzc01hdGNoU2V0XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfVXBkYXRlWHNzTWF0Y2hTZXQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkluc2VydHMgb3IgZGVsZXRlcyBYc3NNYXRjaFR1cGxlIG9iamVjdHMgKGZpbHRlcnMpIGluIGFuIFhzc01hdGNoU2V0LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInhzc21hdGNoc2V0XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgfTtcbiAgcHJvdGVjdGVkIHJlc291cmNlVHlwZXM6IFJlc291cmNlVHlwZXMgPSB7XG4gICAgXCJieXRlbWF0Y2hzZXRcIjoge1xuICAgICAgXCJuYW1lXCI6IFwiYnl0ZW1hdGNoc2V0XCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0J5dGVNYXRjaFNldC5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06d2FmOjoke0FjY291bnR9OmJ5dGVtYXRjaHNldC8ke0lkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfSxcbiAgICBcImlwc2V0XCI6IHtcbiAgICAgIFwibmFtZVwiOiBcImlwc2V0XCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0lQU2V0Lmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTp3YWY6OiR7QWNjb3VudH06aXBzZXQvJHtJZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH0sXG4gICAgXCJyYXRlYmFzZWRydWxlXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcInJhdGViYXNlZHJ1bGVcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfUmF0ZUJhc2VkUnVsZS5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06d2FmOjoke0FjY291bnR9OnJhdGViYXNlZHJ1bGUvJHtJZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcInJ1bGVcIjoge1xuICAgICAgXCJuYW1lXCI6IFwicnVsZVwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9SdWxlLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTp3YWY6OiR7QWNjb3VudH06cnVsZS8ke0lkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwic2l6ZWNvbnN0cmFpbnRzZXRcIjoge1xuICAgICAgXCJuYW1lXCI6IFwic2l6ZWNvbnN0cmFpbnRzZXRcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfU2l6ZUNvbnN0cmFpbnRTZXQuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OndhZjo6JHtBY2NvdW50fTpzaXplY29uc3RyYWludHNldC8ke0lkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfSxcbiAgICBcInNxbGluamVjdGlvbm1hdGNoc2V0XCI6IHtcbiAgICAgIFwibmFtZVwiOiBcInNxbGluamVjdGlvbm1hdGNoc2V0XCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX1NxbEluamVjdGlvbk1hdGNoU2V0Lmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTp3YWY6OiR7QWNjb3VudH06c3FsaW5qZWN0aW9uc2V0LyR7SWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW11cbiAgICB9LFxuICAgIFwid2ViYWNsXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcIndlYmFjbFwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9XZWJBQ0wuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OndhZjo6JHtBY2NvdW50fTp3ZWJhY2wvJHtJZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcInhzc21hdGNoc2V0XCI6IHtcbiAgICAgIFwibmFtZVwiOiBcInhzc21hdGNoc2V0XCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX1hzc01hdGNoU2V0Lmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTp3YWY6OiR7QWNjb3VudH06eHNzbWF0Y2hzZXQvJHtJZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH0sXG4gICAgXCJyZWdleG1hdGNoc2V0XCI6IHtcbiAgICAgIFwibmFtZVwiOiBcInJlZ2V4bWF0Y2hzZXRcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfUmVnZXhNYXRjaFNldC5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06d2FmOjoke0FjY291bnR9OnJlZ2V4bWF0Y2gvJHtJZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH0sXG4gICAgXCJyZWdleHBhdHRlcm5zZXRcIjoge1xuICAgICAgXCJuYW1lXCI6IFwicmVnZXhwYXR0ZXJuc2V0XCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX1JlZ2V4UGF0dGVyblNldC5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06d2FmOjoke0FjY291bnR9OnJlZ2V4cGF0dGVybnNldC8ke0lkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfSxcbiAgICBcImdlb21hdGNoc2V0XCI6IHtcbiAgICAgIFwibmFtZVwiOiBcImdlb21hdGNoc2V0XCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0dlb01hdGNoU2V0Lmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTp3YWY6OiR7QWNjb3VudH06Z2VvbWF0Y2hzZXQvJHtJZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXVxuICAgIH0sXG4gICAgXCJydWxlZ3JvdXBcIjoge1xuICAgICAgXCJuYW1lXCI6IFwicnVsZWdyb3VwXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX1J1bGVHcm91cC5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06d2FmOjoke0FjY291bnR9OnJ1bGVncm91cC8ke0lkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9XG4gIH07XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbd2FmXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hd3N3YWYuaHRtbCkuXG4gICAqXG4gICAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICAgKi9cbiAgY29uc3RydWN0b3IgKHNpZD86IHN0cmluZykge1xuICAgIHN1cGVyKHNpZCk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIEJ5dGVNYXRjaFNldC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9DcmVhdGVCeXRlTWF0Y2hTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIGNyZWF0ZUJ5dGVNYXRjaFNldCgpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkNyZWF0ZUJ5dGVNYXRjaFNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBHZW9NYXRjaFNldCwgd2hpY2ggeW91IHVzZSB0byBzcGVjaWZ5IHdoaWNoIHdlYiByZXF1ZXN0cyB5b3Ugd2FudCB0byBhbGxvdyBvciBibG9jayBiYXNlZCBvbiB0aGUgY291bnRyeSB0aGF0IHRoZSByZXF1ZXN0cyBvcmlnaW5hdGUgZnJvbS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9DcmVhdGVHZW9NYXRjaFNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgY3JlYXRlR2VvTWF0Y2hTZXQoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpDcmVhdGVHZW9NYXRjaFNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYW4gSVBTZXQsIHdoaWNoIHlvdSB1c2UgdG8gc3BlY2lmeSB3aGljaCB3ZWIgcmVxdWVzdHMgeW91IHdhbnQgdG8gYWxsb3cgb3IgYmxvY2sgYmFzZWQgb24gdGhlIElQIGFkZHJlc3NlcyB0aGF0IHRoZSByZXF1ZXN0cyBvcmlnaW5hdGUgZnJvbS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9DcmVhdGVJUFNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgY3JlYXRlSVBTZXQoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpDcmVhdGVJUFNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBSYXRlQmFzZWRSdWxlLCB3aGljaCBjb250YWlucyBhIFJhdGVMaW1pdCBzcGVjaWZ5aW5nIHRoZSBtYXhpbXVtIG51bWJlciBvZiByZXF1ZXN0cyB0aGF0IEFXUyBXQUYgYWxsb3dzIGZyb20gYSBzcGVjaWZpZWQgSVAgYWRkcmVzcyBpbiBhIGZpdmUtbWludXRlIHBlcmlvZC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9DcmVhdGVSYXRlQmFzZWRSdWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBjcmVhdGVSYXRlQmFzZWRSdWxlKCkge1xuICAgIHRoaXMuYWRkKCd3YWY6Q3JlYXRlUmF0ZUJhc2VkUnVsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBSZWdleE1hdGNoU2V0LCB3aGljaCB5b3UgdXNlIHRvIHNwZWNpZnkgd2hpY2ggd2ViIHJlcXVlc3RzIHlvdSB3YW50IHRvIGFsbG93IG9yIGJsb2NrIGJhc2VkIG9uIHRoZSByZWdleCBwYXR0ZXJucyB5b3Ugc3BlY2lmaWVkIGluIGEgUmVnZXhQYXR0ZXJuU2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0NyZWF0ZVJlZ2V4TWF0Y2hTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIGNyZWF0ZVJlZ2V4TWF0Y2hTZXQoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpDcmVhdGVSZWdleE1hdGNoU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIFJlZ2V4UGF0dGVyblNldCwgd2hpY2ggeW91IHVzZSB0byBzcGVjaWZ5IHRoZSByZWd1bGFyIGV4cHJlc3Npb24gKHJlZ2V4KSBwYXR0ZXJuIHRoYXQgeW91IHdhbnQgQVdTIFdBRiB0byBzZWFyY2ggZm9yLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0NyZWF0ZVJlZ2V4UGF0dGVyblNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgY3JlYXRlUmVnZXhQYXR0ZXJuU2V0KCkge1xuICAgIHRoaXMuYWRkKCd3YWY6Q3JlYXRlUmVnZXhQYXR0ZXJuU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIFJ1bGUsIHdoaWNoIGNvbnRhaW5zIHRoZSBJUFNldCBvYmplY3RzLCBCeXRlTWF0Y2hTZXQgb2JqZWN0cywgYW5kIG90aGVyIHByZWRpY2F0ZXMgdGhhdCBpZGVudGlmeSB0aGUgcmVxdWVzdHMgdGhhdCB5b3Ugd2FudCB0byBibG9jay5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9DcmVhdGVSdWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBjcmVhdGVSdWxlKCkge1xuICAgIHRoaXMuYWRkKCd3YWY6Q3JlYXRlUnVsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBSdWxlR3JvdXAuIEEgcnVsZSBncm91cCBpcyBhIGNvbGxlY3Rpb24gb2YgcHJlZGVmaW5lZCBydWxlcyB0aGF0IHlvdSBhZGQgdG8gYSBXZWJBQ0wuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfQ3JlYXRlUnVsZUdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyBjcmVhdGVSdWxlR3JvdXAoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpDcmVhdGVSdWxlR3JvdXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgU2l6ZUNvbnN0cmFpbnRTZXQsIHdoaWNoIHlvdSB1c2UgdG8gaWRlbnRpZnkgdGhlIHBhcnQgb2YgYSB3ZWIgcmVxdWVzdCB0aGF0IHlvdSB3YW50IHRvIGNoZWNrIGZvciBsZW5ndGguXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfQ3JlYXRlU2l6ZUNvbnN0cmFpbnRTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIGNyZWF0ZVNpemVDb25zdHJhaW50U2V0KCkge1xuICAgIHRoaXMuYWRkKCd3YWY6Q3JlYXRlU2l6ZUNvbnN0cmFpbnRTZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgU3FsSW5qZWN0aW9uTWF0Y2hTZXQsIHdoaWNoIHlvdSB1c2UgdG8gYWxsb3csIGJsb2NrLCBvciBjb3VudCByZXF1ZXN0cyB0aGF0IGNvbnRhaW4gc25pcHBldHMgb2YgU1FMIGNvZGUgaW4gYSBzcGVjaWZpZWQgcGFydCBvZiB3ZWIgcmVxdWVzdHMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfQ3JlYXRlU3FsSW5qZWN0aW9uTWF0Y2hTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIGNyZWF0ZVNxbEluamVjdGlvbk1hdGNoU2V0KCkge1xuICAgIHRoaXMuYWRkKCd3YWY6Q3JlYXRlU3FsSW5qZWN0aW9uTWF0Y2hTZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgV2ViQUNMLCB3aGljaCBjb250YWlucyB0aGUgUnVsZXMgdGhhdCBpZGVudGlmeSB0aGUgQ2xvdWRGcm9udCB3ZWIgcmVxdWVzdHMgdGhhdCB5b3Ugd2FudCB0byBhbGxvdywgYmxvY2ssIG9yIGNvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfQ3JlYXRlV2ViQUNMLmh0bWxcbiAgICovXG4gIHB1YmxpYyBjcmVhdGVXZWJBQ0woKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpDcmVhdGVXZWJBQ0wnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYW5kIHN0b3JlIGEgQ2xvdWRGb3JtYXRpb24gdGVtcGFsdGUgdGhhdCBjcmVhdGVzIGFuIGVxdWl2YWxlbnQgV0FGIHYyIFdlYkFDTCBmcm9tIHRoZSBnaXZlbiBXQUYgQ2xhc3NpYyBXZWJBQ0wgaW4gdGhlIGdpdmVuIFMzIGJ1Y2tldC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBzMzpQdXRPYmplY3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfQ3JlYXRlV2ViQUNMTWlncmF0aW9uU3RhY2suaHRtbFxuICAgKi9cbiAgcHVibGljIGNyZWF0ZVdlYkFDTE1pZ3JhdGlvblN0YWNrKCkge1xuICAgIHRoaXMuYWRkKCd3YWY6Q3JlYXRlV2ViQUNMTWlncmF0aW9uU3RhY2snKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGFuIFhzc01hdGNoU2V0LCB3aGljaCB5b3UgdXNlIHRvIGFsbG93LCBibG9jaywgb3IgY291bnQgcmVxdWVzdHMgdGhhdCBjb250YWluIGNyb3NzLXNpdGUgc2NyaXB0aW5nIGF0dGFja3MgaW4gdGhlIHNwZWNpZmllZCBwYXJ0IG9mIHdlYiByZXF1ZXN0cy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9DcmVhdGVYc3NNYXRjaFNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgY3JlYXRlWHNzTWF0Y2hTZXQoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpDcmVhdGVYc3NNYXRjaFNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFBlcm1hbmVudGx5IGRlbGV0ZXMgYSBCeXRlTWF0Y2hTZXQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfRGVsZXRlQnl0ZU1hdGNoU2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVCeXRlTWF0Y2hTZXQoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpEZWxldGVCeXRlTWF0Y2hTZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBQZXJtYW5lbnRseSBkZWxldGVzIGFuIEdlb01hdGNoU2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0RlbGV0ZUdlb01hdGNoU2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVHZW9NYXRjaFNldCgpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkRlbGV0ZUdlb01hdGNoU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUGVybWFuZW50bHkgZGVsZXRlcyBhbiBJUFNldC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9EZWxldGVJUFNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVsZXRlSVBTZXQoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpEZWxldGVJUFNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFBlcm1hbmVudGx5IGRlbGV0ZXMgdGhlIExvZ2dpbmdDb25maWd1cmF0aW9uIGZyb20gdGhlIHNwZWNpZmllZCB3ZWIgQUNMLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0RlbGV0ZUxvZ2dpbmdDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVMb2dnaW5nQ29uZmlndXJhdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkRlbGV0ZUxvZ2dpbmdDb25maWd1cmF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUGVybWFuZW50bHkgZGVsZXRlcyBhbiBJQU0gcG9saWN5IGZyb20gdGhlIHNwZWNpZmllZCBSdWxlR3JvdXAuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9EZWxldGVQZXJtaXNzaW9uUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVQZXJtaXNzaW9uUG9saWN5KCkge1xuICAgIHRoaXMuYWRkKCd3YWY6RGVsZXRlUGVybWlzc2lvblBvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFBlcm1hbmVudGx5IGRlbGV0ZXMgYSBSYXRlQmFzZWRSdWxlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0RlbGV0ZVJhdGVCYXNlZFJ1bGUuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZVJhdGVCYXNlZFJ1bGUoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpEZWxldGVSYXRlQmFzZWRSdWxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUGVybWFuZW50bHkgZGVsZXRlcyBhbiBSZWdleE1hdGNoU2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0RlbGV0ZVJlZ2V4TWF0Y2hTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZVJlZ2V4TWF0Y2hTZXQoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpEZWxldGVSZWdleE1hdGNoU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUGVybWFuZW50bHkgZGVsZXRlcyBhbiBSZWdleFBhdHRlcm5TZXQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfRGVsZXRlUmVnZXhQYXR0ZXJuU2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVSZWdleFBhdHRlcm5TZXQoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpEZWxldGVSZWdleFBhdHRlcm5TZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBQZXJtYW5lbnRseSBkZWxldGVzIGEgUnVsZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9EZWxldGVSdWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVSdWxlKCkge1xuICAgIHRoaXMuYWRkKCd3YWY6RGVsZXRlUnVsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFBlcm1hbmVudGx5IGRlbGV0ZXMgYSBSdWxlR3JvdXAuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfRGVsZXRlUnVsZUdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVSdWxlR3JvdXAoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpEZWxldGVSdWxlR3JvdXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBQZXJtYW5lbnRseSBkZWxldGVzIGEgU2l6ZUNvbnN0cmFpbnRTZXQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfRGVsZXRlU2l6ZUNvbnN0cmFpbnRTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZVNpemVDb25zdHJhaW50U2V0KCkge1xuICAgIHRoaXMuYWRkKCd3YWY6RGVsZXRlU2l6ZUNvbnN0cmFpbnRTZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBQZXJtYW5lbnRseSBkZWxldGVzIGEgU3FsSW5qZWN0aW9uTWF0Y2hTZXQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfRGVsZXRlU3FsSW5qZWN0aW9uTWF0Y2hTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZVNxbEluamVjdGlvbk1hdGNoU2V0KCkge1xuICAgIHRoaXMuYWRkKCd3YWY6RGVsZXRlU3FsSW5qZWN0aW9uTWF0Y2hTZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBQZXJtYW5lbnRseSBkZWxldGVzIGEgV2ViQUNMLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfRGVsZXRlV2ViQUNMLmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVXZWJBQ0woKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpEZWxldGVXZWJBQ0wnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBQZXJtYW5lbnRseSBkZWxldGVzIGFuIFhzc01hdGNoU2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0RlbGV0ZVhzc01hdGNoU2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVYc3NNYXRjaFNldCgpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkRlbGV0ZVhzc01hdGNoU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgQnl0ZU1hdGNoU2V0IHNwZWNpZmllZCBieSBCeXRlTWF0Y2hTZXRJZC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0dldEJ5dGVNYXRjaFNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgZ2V0Qnl0ZU1hdGNoU2V0KCkge1xuICAgIHRoaXMuYWRkKCd3YWY6R2V0Qnl0ZU1hdGNoU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogV2hlbiB5b3Ugd2FudCB0byBjcmVhdGUsIHVwZGF0ZSwgb3IgZGVsZXRlIEFXUyBXQUYgb2JqZWN0cywgZ2V0IGEgY2hhbmdlIHRva2VuIGFuZCBpbmNsdWRlIHRoZSBjaGFuZ2UgdG9rZW4gaW4gdGhlIGNyZWF0ZSwgdXBkYXRlLCBvciBkZWxldGUgcmVxdWVzdC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0dldENoYW5nZVRva2VuLmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRDaGFuZ2VUb2tlbigpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkdldENoYW5nZVRva2VuJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgc3RhdHVzIG9mIGEgQ2hhbmdlVG9rZW4gdGhhdCB5b3UgZ290IGJ5IGNhbGxpbmcgR2V0Q2hhbmdlVG9rZW4uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9HZXRDaGFuZ2VUb2tlblN0YXR1cy5odG1sXG4gICAqL1xuICBwdWJsaWMgZ2V0Q2hhbmdlVG9rZW5TdGF0dXMoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpHZXRDaGFuZ2VUb2tlblN0YXR1cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIEdlb01hdGNoU2V0IHNwZWNpZmllZCBieSBHZW9NYXRjaFNldElkLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfR2V0R2VvTWF0Y2hTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIGdldEdlb01hdGNoU2V0KCkge1xuICAgIHRoaXMuYWRkKCd3YWY6R2V0R2VvTWF0Y2hTZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBJUFNldCB0aGF0IGlzIHNwZWNpZmllZCBieSBJUFNldElkLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfR2V0SVBTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIGdldElQU2V0KCkge1xuICAgIHRoaXMuYWRkKCd3YWY6R2V0SVBTZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBMb2dnaW5nQ29uZmlndXJhdGlvbiBmb3IgdGhlIHNwZWNpZmllZCB3ZWIgQUNMLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfR2V0TG9nZ2luZ0NvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIGdldExvZ2dpbmdDb25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMuYWRkKCd3YWY6R2V0TG9nZ2luZ0NvbmZpZ3VyYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBJQU0gcG9saWN5IGF0dGFjaGVkIHRvIHRoZSBSdWxlR3JvdXAuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9HZXRQZXJtaXNzaW9uUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRQZXJtaXNzaW9uUG9saWN5KCkge1xuICAgIHRoaXMuYWRkKCd3YWY6R2V0UGVybWlzc2lvblBvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIFJhdGVCYXNlZFJ1bGUgdGhhdCBpcyBzcGVjaWZpZWQgYnkgdGhlIFJ1bGVJZCB0aGF0IHlvdSBpbmNsdWRlZCBpbiB0aGUgR2V0UmF0ZUJhc2VkUnVsZSByZXF1ZXN0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfR2V0UmF0ZUJhc2VkUnVsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgZ2V0UmF0ZUJhc2VkUnVsZSgpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkdldFJhdGVCYXNlZFJ1bGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFuIGFycmF5IG9mIElQIGFkZHJlc3NlcyBjdXJyZW50bHkgYmVpbmcgYmxvY2tlZCBieSB0aGUgUmF0ZUJhc2VkUnVsZSB0aGF0IGlzIHNwZWNpZmllZCBieSB0aGUgUnVsZUlkLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfR2V0UmF0ZUJhc2VkUnVsZU1hbmFnZWRLZXlzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRSYXRlQmFzZWRSdWxlTWFuYWdlZEtleXMoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpHZXRSYXRlQmFzZWRSdWxlTWFuYWdlZEtleXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBSZWdleE1hdGNoU2V0IHNwZWNpZmllZCBieSBSZWdleE1hdGNoU2V0SWQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9HZXRSZWdleE1hdGNoU2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRSZWdleE1hdGNoU2V0KCkge1xuICAgIHRoaXMuYWRkKCd3YWY6R2V0UmVnZXhNYXRjaFNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIFJlZ2V4UGF0dGVyblNldCBzcGVjaWZpZWQgYnkgUmVnZXhQYXR0ZXJuU2V0SWQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9HZXRSZWdleFBhdHRlcm5TZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIGdldFJlZ2V4UGF0dGVyblNldCgpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkdldFJlZ2V4UGF0dGVyblNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIFJ1bGUgdGhhdCBpcyBzcGVjaWZpZWQgYnkgdGhlIFJ1bGVJZCB0aGF0IHlvdSBpbmNsdWRlZCBpbiB0aGUgR2V0UnVsZSByZXF1ZXN0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfR2V0UnVsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgZ2V0UnVsZSgpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkdldFJ1bGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBSdWxlR3JvdXAgdGhhdCBpcyBzcGVjaWZpZWQgYnkgdGhlIFJ1bGVHcm91cElkIHRoYXQgeW91IGluY2x1ZGVkIGluIHRoZSBHZXRSdWxlR3JvdXAgcmVxdWVzdC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0dldFJ1bGVHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgZ2V0UnVsZUdyb3VwKCkge1xuICAgIHRoaXMuYWRkKCd3YWY6R2V0UnVsZUdyb3VwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR2V0cyBkZXRhaWxlZCBpbmZvcm1hdGlvbiBhYm91dCBhIHNwZWNpZmllZCBudW1iZXIgb2YgcmVxdWVzdHMtLWEgc2FtcGxlLS10aGF0IEFXUyBXQUYgcmFuZG9tbHkgc2VsZWN0cyBmcm9tIGFtb25nIHRoZSBmaXJzdCA1LDAwMCByZXF1ZXN0cyB0aGF0IHlvdXIgQVdTIHJlc291cmNlIHJlY2VpdmVkIGR1cmluZyBhIHRpbWUgcmFuZ2UgdGhhdCB5b3UgY2hvb3NlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfR2V0U2FtcGxlZFJlcXVlc3RzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRTYW1wbGVkUmVxdWVzdHMoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpHZXRTYW1wbGVkUmVxdWVzdHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBTaXplQ29uc3RyYWludFNldCBzcGVjaWZpZWQgYnkgU2l6ZUNvbnN0cmFpbnRTZXRJZC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0dldFNpemVDb25zdHJhaW50U2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRTaXplQ29uc3RyYWludFNldCgpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkdldFNpemVDb25zdHJhaW50U2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgU3FsSW5qZWN0aW9uTWF0Y2hTZXQgdGhhdCBpcyBzcGVjaWZpZWQgYnkgU3FsSW5qZWN0aW9uTWF0Y2hTZXRJZC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0dldFNxbEluamVjdGlvbk1hdGNoU2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRTcWxJbmplY3Rpb25NYXRjaFNldCgpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkdldFNxbEluamVjdGlvbk1hdGNoU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgV2ViQUNMIHRoYXQgaXMgc3BlY2lmaWVkIGJ5IFdlYkFDTElkLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfR2V0V2ViQUNMLmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRXZWJBQ0woKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpHZXRXZWJBQ0wnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBYc3NNYXRjaFNldCB0aGF0IGlzIHNwZWNpZmllZCBieSBYc3NNYXRjaFNldElkLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfR2V0WHNzTWF0Y2hTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIGdldFhzc01hdGNoU2V0KCkge1xuICAgIHRoaXMuYWRkKCd3YWY6R2V0WHNzTWF0Y2hTZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFuIGFycmF5IG9mIEFjdGl2YXRlZFJ1bGUgb2JqZWN0cy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0xpc3RBY3RpdmF0ZWRSdWxlc0luUnVsZUdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0QWN0aXZhdGVkUnVsZXNJblJ1bGVHcm91cCgpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkxpc3RBY3RpdmF0ZWRSdWxlc0luUnVsZUdyb3VwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhbiBhcnJheSBvZiBCeXRlTWF0Y2hTZXRTdW1tYXJ5IG9iamVjdHMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9MaXN0Qnl0ZU1hdGNoU2V0cy5odG1sXG4gICAqL1xuICBwdWJsaWMgbGlzdEJ5dGVNYXRjaFNldHMoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpMaXN0Qnl0ZU1hdGNoU2V0cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYW4gYXJyYXkgb2YgR2VvTWF0Y2hTZXRTdW1tYXJ5IG9iamVjdHMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9MaXN0R2VvTWF0Y2hTZXRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0R2VvTWF0Y2hTZXRzKCkge1xuICAgIHRoaXMuYWRkKCd3YWY6TGlzdEdlb01hdGNoU2V0cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYW4gYXJyYXkgb2YgSVBTZXRTdW1tYXJ5IG9iamVjdHMgaW4gdGhlIHJlc3BvbnNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfTGlzdElQU2V0cy5odG1sXG4gICAqL1xuICBwdWJsaWMgbGlzdElQU2V0cygpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkxpc3RJUFNldHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFuIGFycmF5IG9mIExvZ2dpbmdDb25maWd1cmF0aW9uIG9iamVjdHMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9MaXN0TG9nZ2luZ0NvbmZpZ3VyYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0TG9nZ2luZ0NvbmZpZ3VyYXRpb25zKCkge1xuICAgIHRoaXMuYWRkKCd3YWY6TGlzdExvZ2dpbmdDb25maWd1cmF0aW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYW4gYXJyYXkgb2YgUnVsZVN1bW1hcnkgb2JqZWN0cy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0xpc3RSYXRlQmFzZWRSdWxlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgbGlzdFJhdGVCYXNlZFJ1bGVzKCkge1xuICAgIHRoaXMuYWRkKCd3YWY6TGlzdFJhdGVCYXNlZFJ1bGVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhbiBhcnJheSBvZiBSZWdleE1hdGNoU2V0U3VtbWFyeSBvYmplY3RzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfTGlzdFJlZ2V4TWF0Y2hTZXRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0UmVnZXhNYXRjaFNldHMoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpMaXN0UmVnZXhNYXRjaFNldHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFuIGFycmF5IG9mIFJlZ2V4UGF0dGVyblNldFN1bW1hcnkgb2JqZWN0cy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0xpc3RSZWdleFBhdHRlcm5TZXRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0UmVnZXhQYXR0ZXJuU2V0cygpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkxpc3RSZWdleFBhdHRlcm5TZXRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhbiBhcnJheSBvZiBSdWxlR3JvdXAgb2JqZWN0cy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0xpc3RSdWxlR3JvdXBzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0UnVsZUdyb3VwcygpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkxpc3RSdWxlR3JvdXBzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhbiBhcnJheSBvZiBSdWxlU3VtbWFyeSBvYmplY3RzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfTGlzdFJ1bGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0UnVsZXMoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpMaXN0UnVsZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFuIGFycmF5IG9mIFNpemVDb25zdHJhaW50U2V0U3VtbWFyeSBvYmplY3RzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfTGlzdFNpemVDb25zdHJhaW50U2V0cy5odG1sXG4gICAqL1xuICBwdWJsaWMgbGlzdFNpemVDb25zdHJhaW50U2V0cygpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkxpc3RTaXplQ29uc3RyYWludFNldHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFuIGFycmF5IG9mIFNxbEluamVjdGlvbk1hdGNoU2V0IG9iamVjdHMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9MaXN0U3FsSW5qZWN0aW9uTWF0Y2hTZXRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0U3FsSW5qZWN0aW9uTWF0Y2hTZXRzKCkge1xuICAgIHRoaXMuYWRkKCd3YWY6TGlzdFNxbEluamVjdGlvbk1hdGNoU2V0cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYW4gYXJyYXkgb2YgUnVsZUdyb3VwIG9iamVjdHMgdGhhdCB5b3UgYXJlIHN1YnNjcmliZWQgdG8uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9MaXN0U3Vic2NyaWJlZFJ1bGVHcm91cHMuaHRtbFxuICAgKi9cbiAgcHVibGljIGxpc3RTdWJzY3JpYmVkUnVsZUdyb3VwcygpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkxpc3RTdWJzY3JpYmVkUnVsZUdyb3VwcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIHRoZSBUYWdzIGZvciBhIGdpdmVuIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfTGlzdFRhZ3NGb3JSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgbGlzdFRhZ3NGb3JSZXNvdXJjZSgpIHtcbiAgICB0aGlzLmFkZCgnd2FmOkxpc3RUYWdzRm9yUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFuIGFycmF5IG9mIFdlYkFDTFN1bW1hcnkgb2JqZWN0cyBpbiB0aGUgcmVzcG9uc2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9MaXN0V2ViQUNMcy5odG1sXG4gICAqL1xuICBwdWJsaWMgbGlzdFdlYkFDTHMoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpMaXN0V2ViQUNMcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYW4gYXJyYXkgb2YgWHNzTWF0Y2hTZXQgb2JqZWN0cy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX0xpc3RYc3NNYXRjaFNldHMuaHRtbFxuICAgKi9cbiAgcHVibGljIGxpc3RYc3NNYXRjaFNldHMoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpMaXN0WHNzTWF0Y2hTZXRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQXNzb2NpYXRlcyBhIExvZ2dpbmdDb25maWd1cmF0aW9uIHdpdGggYSBzcGVjaWZpZWQgd2ViIEFDTC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06Q3JlYXRlU2VydmljZUxpbmtlZFJvbGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfUHV0TG9nZ2luZ0NvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHB1dExvZ2dpbmdDb25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMuYWRkKCd3YWY6UHV0TG9nZ2luZ0NvbmZpZ3VyYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBdHRhY2hlcyBhIElBTSBwb2xpY3kgdG8gdGhlIHNwZWNpZmllZCByZXNvdXJjZS4gVGhlIG9ubHkgc3VwcG9ydGVkIHVzZSBmb3IgdGhpcyBhY3Rpb24gaXMgdG8gc2hhcmUgYSBSdWxlR3JvdXAgYWNyb3NzIGFjY291bnRzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfUHV0UGVybWlzc2lvblBvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgcHV0UGVybWlzc2lvblBvbGljeSgpIHtcbiAgICB0aGlzLmFkZCgnd2FmOlB1dFBlcm1pc3Npb25Qb2xpY3knKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgVGFnIHRvIGEgZ2l2ZW4gcmVzb3VyY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9UYWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdGFnUmVzb3VyY2UoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpUYWdSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbW92ZXMgYSBUYWcgZnJvbSBhIGdpdmVuIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfVW50YWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdW50YWdSZXNvdXJjZSgpIHtcbiAgICB0aGlzLmFkZCgnd2FmOlVudGFnUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBJbnNlcnRzIG9yIGRlbGV0ZXMgQnl0ZU1hdGNoVHVwbGUgb2JqZWN0cyAoZmlsdGVycykgaW4gYSBCeXRlTWF0Y2hTZXQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfVXBkYXRlQnl0ZU1hdGNoU2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB1cGRhdGVCeXRlTWF0Y2hTZXQoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpVcGRhdGVCeXRlTWF0Y2hTZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBJbnNlcnRzIG9yIGRlbGV0ZXMgR2VvTWF0Y2hDb25zdHJhaW50IG9iamVjdHMgaW4gYSBHZW9NYXRjaFNldC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9VcGRhdGVHZW9NYXRjaFNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlR2VvTWF0Y2hTZXQoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpVcGRhdGVHZW9NYXRjaFNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEluc2VydHMgb3IgZGVsZXRlcyBJUFNldERlc2NyaXB0b3Igb2JqZWN0cyBpbiBhbiBJUFNldC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9VcGRhdGVJUFNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlSVBTZXQoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpVcGRhdGVJUFNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEluc2VydHMgb3IgZGVsZXRlcyBQcmVkaWNhdGUgb2JqZWN0cyBpbiBhIHJ1bGUgYW5kIHVwZGF0ZXMgdGhlIFJhdGVMaW1pdCBpbiB0aGUgcnVsZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9VcGRhdGVSYXRlQmFzZWRSdWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB1cGRhdGVSYXRlQmFzZWRSdWxlKCkge1xuICAgIHRoaXMuYWRkKCd3YWY6VXBkYXRlUmF0ZUJhc2VkUnVsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEluc2VydHMgb3IgZGVsZXRlcyBSZWdleE1hdGNoVHVwbGUgb2JqZWN0cyAoZmlsdGVycykgaW4gYSBSZWdleE1hdGNoU2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX1VwZGF0ZVJlZ2V4TWF0Y2hTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHVwZGF0ZVJlZ2V4TWF0Y2hTZXQoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpVcGRhdGVSZWdleE1hdGNoU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogSW5zZXJ0cyBvciBkZWxldGVzIFJlZ2V4UGF0dGVyblN0cmluZ3MgaW4gYSBSZWdleFBhdHRlcm5TZXQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfVXBkYXRlUmVnZXhQYXR0ZXJuU2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB1cGRhdGVSZWdleFBhdHRlcm5TZXQoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpVcGRhdGVSZWdleFBhdHRlcm5TZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBJbnNlcnRzIG9yIGRlbGV0ZXMgUHJlZGljYXRlIG9iamVjdHMgaW4gYSBSdWxlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX1VwZGF0ZVJ1bGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHVwZGF0ZVJ1bGUoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpVcGRhdGVSdWxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogSW5zZXJ0cyBvciBkZWxldGVzIEFjdGl2YXRlZFJ1bGUgb2JqZWN0cyBpbiBhIFJ1bGVHcm91cC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9VcGRhdGVSdWxlR3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHVwZGF0ZVJ1bGVHcm91cCgpIHtcbiAgICB0aGlzLmFkZCgnd2FmOlVwZGF0ZVJ1bGVHcm91cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEluc2VydHMgb3IgZGVsZXRlcyBTaXplQ29uc3RyYWludCBvYmplY3RzIChmaWx0ZXJzKSBpbiBhIFNpemVDb25zdHJhaW50U2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX1VwZGF0ZVNpemVDb25zdHJhaW50U2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB1cGRhdGVTaXplQ29uc3RyYWludFNldCgpIHtcbiAgICB0aGlzLmFkZCgnd2FmOlVwZGF0ZVNpemVDb25zdHJhaW50U2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogSW5zZXJ0cyBvciBkZWxldGVzIFNxbEluamVjdGlvbk1hdGNoVHVwbGUgb2JqZWN0cyAoZmlsdGVycykgaW4gYSBTcWxJbmplY3Rpb25NYXRjaFNldC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9VcGRhdGVTcWxJbmplY3Rpb25NYXRjaFNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlU3FsSW5qZWN0aW9uTWF0Y2hTZXQoKSB7XG4gICAgdGhpcy5hZGQoJ3dhZjpVcGRhdGVTcWxJbmplY3Rpb25NYXRjaFNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEluc2VydHMgb3IgZGVsZXRlcyBBY3RpdmF0ZWRSdWxlIG9iamVjdHMgaW4gYSBXZWJBQ0wuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9VcGRhdGVXZWJBQ0wuaHRtbFxuICAgKi9cbiAgcHVibGljIHVwZGF0ZVdlYkFDTCgpIHtcbiAgICB0aGlzLmFkZCgnd2FmOlVwZGF0ZVdlYkFDTCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEluc2VydHMgb3IgZGVsZXRlcyBYc3NNYXRjaFR1cGxlIG9iamVjdHMgKGZpbHRlcnMpIGluIGFuIFhzc01hdGNoU2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX1VwZGF0ZVhzc01hdGNoU2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB1cGRhdGVYc3NNYXRjaFNldCgpIHtcbiAgICB0aGlzLmFkZCgnd2FmOlVwZGF0ZVhzc01hdGNoU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgYnl0ZW1hdGNoc2V0IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfQnl0ZU1hdGNoU2V0Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25CeXRlbWF0Y2hzZXQoaWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OndhZjo6JHtBY2NvdW50fTpieXRlbWF0Y2hzZXQvJHtJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0lkfScsIGlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGlwc2V0IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfSVBTZXQuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gaWQgLSBJZGVudGlmaWVyIGZvciB0aGUgaWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvbklwc2V0KGlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTp3YWY6OiR7QWNjb3VudH06aXBzZXQvJHtJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0lkfScsIGlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHJhdGViYXNlZHJ1bGUgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9SYXRlQmFzZWRSdWxlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uUmF0ZWJhc2VkcnVsZShpZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06d2FmOjoke0FjY291bnR9OnJhdGViYXNlZHJ1bGUvJHtJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0lkfScsIGlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHJ1bGUgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9SdWxlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uUnVsZShpZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06d2FmOjoke0FjY291bnR9OnJ1bGUvJHtJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0lkfScsIGlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHNpemVjb25zdHJhaW50c2V0IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfU2l6ZUNvbnN0cmFpbnRTZXQuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gaWQgLSBJZGVudGlmaWVyIGZvciB0aGUgaWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvblNpemVjb25zdHJhaW50c2V0KGlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTp3YWY6OiR7QWNjb3VudH06c2l6ZWNvbnN0cmFpbnRzZXQvJHtJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0lkfScsIGlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHNxbGluamVjdGlvbm1hdGNoc2V0IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfU3FsSW5qZWN0aW9uTWF0Y2hTZXQuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gaWQgLSBJZGVudGlmaWVyIGZvciB0aGUgaWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvblNxbGluamVjdGlvbm1hdGNoc2V0KGlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTp3YWY6OiR7QWNjb3VudH06c3FsaW5qZWN0aW9uc2V0LyR7SWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtJZH0nLCBpZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSB3ZWJhY2wgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9XZWJBQ0wuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gaWQgLSBJZGVudGlmaWVyIGZvciB0aGUgaWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25XZWJhY2woaWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OndhZjo6JHtBY2NvdW50fTp3ZWJhY2wvJHtJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0lkfScsIGlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHhzc21hdGNoc2V0IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfWHNzTWF0Y2hTZXQuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gaWQgLSBJZGVudGlmaWVyIGZvciB0aGUgaWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvblhzc21hdGNoc2V0KGlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTp3YWY6OiR7QWNjb3VudH06eHNzbWF0Y2hzZXQvJHtJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0lkfScsIGlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHJlZ2V4bWF0Y2hzZXQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX3dhZl9SZWdleE1hdGNoU2V0Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25SZWdleG1hdGNoc2V0KGlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTp3YWY6OiR7QWNjb3VudH06cmVnZXhtYXRjaC8ke0lkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7SWR9JywgaWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgcmVnZXhwYXR0ZXJuc2V0IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfUmVnZXhQYXR0ZXJuU2V0Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25SZWdleHBhdHRlcm5zZXQoaWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OndhZjo6JHtBY2NvdW50fTpyZWdleHBhdHRlcm5zZXQvJHtJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0lkfScsIGlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGdlb21hdGNoc2V0IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV93YWZfR2VvTWF0Y2hTZXQuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gaWQgLSBJZGVudGlmaWVyIGZvciB0aGUgaWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvbkdlb21hdGNoc2V0KGlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTp3YWY6OiR7QWNjb3VudH06Z2VvbWF0Y2hzZXQvJHtJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0lkfScsIGlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHJ1bGVncm91cCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfd2FmX1J1bGVHcm91cC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBpZCAtIElkZW50aWZpZXIgZm9yIHRoZSBpZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblJ1bGVncm91cChpZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06d2FmOjoke0FjY291bnR9OnJ1bGVncm91cC8ke0lkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7SWR9JywgaWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY3Rpb25zIGJhc2VkIG9uIHRoZSBhbGxvd2VkIHNldCBvZiB2YWx1ZXMgZm9yIGVhY2ggb2YgdGhlIHRhZ3NcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlcXVlc3R0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC5jcmVhdGVSYXRlQmFzZWRSdWxlKClcbiAgICogLSAuY3JlYXRlUnVsZSgpXG4gICAqIC0gLmNyZWF0ZVJ1bGVHcm91cCgpXG4gICAqIC0gLmNyZWF0ZVdlYkFDTCgpXG4gICAqIC0gLnRhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1JlcXVlc3RUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlcXVlc3RUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjdGlvbnMgYmFzZWQgb24gdGFnLXZhbHVlIGFzc29pY2F0ZWQgd2l0aCB0aGUgcmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlc291cmNldGFnXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gcmF0ZWJhc2VkcnVsZVxuICAgKiAtIHJ1bGVcbiAgICogLSB3ZWJhY2xcbiAgICogLSBydWxlZ3JvdXBcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1Jlc291cmNlVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXNvdXJjZVRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWN0aW9ucyBiYXNlZCBvbiB0aGUgcHJlc2VuY2Ugb2YgbWFuZGF0b3J5IHRhZ3MgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXRhZ2tleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC5jcmVhdGVSYXRlQmFzZWRSdWxlKClcbiAgICogLSAuY3JlYXRlUnVsZSgpXG4gICAqIC0gLmNyZWF0ZVJ1bGVHcm91cCgpXG4gICAqIC0gLmNyZWF0ZVdlYkFDTCgpXG4gICAqIC0gLnRhZ1Jlc291cmNlKClcbiAgICogLSAudW50YWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzVGFnS2V5cyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpUYWdLZXlzYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cbn1cbiJdfQ==