"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iam = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [iam](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_identityandaccessmanagement.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iam extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [iam](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_identityandaccessmanagement.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'iam';
        this.actionList = {
            "AddClientIDToOpenIDConnectProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_AddClientIDToOpenIDConnectProvider.html",
                "description": "Grants permission to add a new client ID (audience) to the list of registered IDs for the specified IAM OpenID Connect (OIDC) provider resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "oidc-provider": {
                        "required": true
                    }
                }
            },
            "AddRoleToInstanceProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_AddRoleToInstanceProfile.html",
                "description": "Grants permission to add an IAM role to the specified instance profile",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "instance-profile": {
                        "required": true
                    }
                }
            },
            "AddUserToGroup": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_AddUserToGroup.html",
                "description": "Grants permission to add an IAM user to the specified IAM group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "AttachGroupPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_AttachGroupPolicy.html",
                "description": "Grants permission to attach a managed policy to the specified IAM group",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PolicyARN"
                ]
            },
            "AttachRolePolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_AttachRolePolicy.html",
                "description": "Grants permission to attach a managed policy to the specified IAM role",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PolicyARN",
                    "iam:PermissionsBoundary"
                ]
            },
            "AttachUserPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_AttachUserPolicy.html",
                "description": "Grants permission to attach a managed policy to the specified IAM user",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PolicyARN",
                    "iam:PermissionsBoundary"
                ]
            },
            "ChangePassword": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ChangePassword.html",
                "description": "Grants permission for an IAM user to to change their own password",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "CreateAccessKey": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateAccessKey.html",
                "description": "Grants permission to create access key and secret access key for the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "CreateAccountAlias": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateAccountAlias.html",
                "description": "Grants permission to create an alias for your AWS account",
                "accessLevel": "Write"
            },
            "CreateGroup": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateGroup.html",
                "description": "Grants permission to create a new group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "CreateInstanceProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateInstanceProfile.html",
                "description": "Grants permission to create a new instance profile",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance-profile": {
                        "required": true
                    }
                }
            },
            "CreateLoginProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateLoginProfile.html",
                "description": "Grants permission to create a password for the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "CreateOpenIDConnectProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateOpenIDConnectProvider.html",
                "description": "Grants permission to create an IAM resource that describes an identity provider (IdP) that supports OpenID Connect (OIDC)",
                "accessLevel": "Write",
                "resourceTypes": {
                    "oidc-provider": {
                        "required": true
                    }
                }
            },
            "CreatePolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreatePolicy.html",
                "description": "Grants permission to create a new managed policy",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "CreatePolicyVersion": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreatePolicyVersion.html",
                "description": "Grants permission to create a new version of the specified managed policy",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "CreateRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateRole.html",
                "description": "Grants permission to create a new role",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "CreateSAMLProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateSAMLProvider.html",
                "description": "Grants permission to create an IAM resource that describes an identity provider (IdP) that supports SAML 2.0",
                "accessLevel": "Write",
                "resourceTypes": {
                    "saml-provider": {
                        "required": true
                    }
                }
            },
            "CreateServiceLinkedRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateServiceLinkedRole.html",
                "description": "Grants permission to create an IAM role that allows an AWS service to perform actions on your behalf",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:AWSServiceName"
                ]
            },
            "CreateServiceSpecificCredential": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateServiceSpecificCredential.html",
                "description": "Grants permission to create a new service-specific credential for an IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "CreateUser": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateUser.html",
                "description": "Grants permission to create a new IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "CreateVirtualMFADevice": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateVirtualMFADevice.html",
                "description": "Grants permission to create a new virtual MFA device",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mfa": {
                        "required": true
                    }
                }
            },
            "DeactivateMFADevice": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeactivateMFADevice.html",
                "description": "Grants permission to deactivate the specified MFA device and remove its association with the IAM user for which it was originally enabled",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteAccessKey": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteAccessKey.html",
                "description": "Grants permission to delete the access key pair that is associated with the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteAccountAlias": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteAccountAlias.html",
                "description": "Grants permission to delete the specified AWS account alias",
                "accessLevel": "Write"
            },
            "DeleteAccountPasswordPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteAccountPasswordPolicy.html",
                "description": "Grants permission to delete the password policy for the AWS account",
                "accessLevel": "Permissions management"
            },
            "DeleteGroup": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteGroup.html",
                "description": "Grants permission to delete the specified IAM group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "DeleteGroupPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteGroupPolicy.html",
                "description": "Grants permission to delete the specified inline policy from its group",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "DeleteInstanceProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteInstanceProfile.html",
                "description": "Grants permission to delete the specified instance profile",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance-profile": {
                        "required": true
                    }
                }
            },
            "DeleteLoginProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteLoginProfile.html",
                "description": "Grants permission to delete the password for the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteOpenIDConnectProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteOpenIDConnectProvider.html",
                "description": "Grants permission to delete an OpenID Connect identity provider (IdP) resource object in IAM",
                "accessLevel": "Write",
                "resourceTypes": {
                    "oidc-provider": {
                        "required": true
                    }
                }
            },
            "DeletePolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeletePolicy.html",
                "description": "Grants permission to delete the specified managed policy and remove it from any IAM entities (users, groups, or roles) to which it is attached",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "DeletePolicyVersion": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeletePolicyVersion.html",
                "description": "Grants permission to delete a version from the specified managed policy",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "DeleteRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteRole.html",
                "description": "Grants permission to delete the specified role",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "DeleteRolePermissionsBoundary": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteRolePermissionsBoundary.html",
                "description": "Grants permission to remove the permissions boundary from a role",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "DeleteRolePolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteRolePolicy.html",
                "description": "Grants permission to delete the specified inline policy from the specified role",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "DeleteSAMLProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteSAMLProvider.html",
                "description": "Grants permission to delete a SAML provider resource in IAM",
                "accessLevel": "Write",
                "resourceTypes": {
                    "saml-provider": {
                        "required": true
                    }
                }
            },
            "DeleteSSHPublicKey": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteSSHPublicKey.html",
                "description": "Grants permission to delete the specified SSH public key",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteServerCertificate": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteServerCertificate.html",
                "description": "Grants permission to delete the specified server certificate",
                "accessLevel": "Write",
                "resourceTypes": {
                    "server-certificate": {
                        "required": true
                    }
                }
            },
            "DeleteServiceLinkedRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteServiceLinkedRole.html",
                "description": "Grants permission to delete an IAM role that is linked to a specific AWS service, if the service is no longer using it",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "DeleteServiceSpecificCredential": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteServiceSpecificCredential.html",
                "description": "Grants permission to delete the specified service-specific credential for an IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteSigningCertificate": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteSigningCertificate.html",
                "description": "Grants permission to delete a signing certificate that is associated with the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteUser": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteUser.html",
                "description": "Grants permission to delete the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "DeleteUserPermissionsBoundary": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteUserPermissionsBoundary.html",
                "description": "Grants permission to remove the permissions boundary from the specified IAM user",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "DeleteUserPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteUserPolicy.html",
                "description": "Grants permission to delete the specified inline policy from an IAM user",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "DeleteVirtualMFADevice": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteVirtualMFADevice.html",
                "description": "Grants permission to delete a virtual MFA device",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mfa": {
                        "required": false
                    },
                    "sms-mfa": {
                        "required": false
                    }
                }
            },
            "DetachGroupPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DetachGroupPolicy.html",
                "description": "Grants permission to detach a managed policy from the specified IAM group",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PolicyARN"
                ]
            },
            "DetachRolePolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DetachRolePolicy.html",
                "description": "Grants permission to detach a managed policy from the specified role",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PolicyARN",
                    "iam:PermissionsBoundary"
                ]
            },
            "DetachUserPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_DetachUserPolicy.html",
                "description": "Grants permission to detach a managed policy from the specified IAM user",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PolicyARN",
                    "iam:PermissionsBoundary"
                ]
            },
            "EnableMFADevice": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_EnableMFADevice.html",
                "description": "Grants permission to enable an MFA device and associate it with the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "GenerateCredentialReport": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GenerateCredentialReport.html",
                "description": "Grants permission to generate a credential report for the AWS account",
                "accessLevel": "Read"
            },
            "GenerateOrganizationsAccessReport": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GenerateOrganizationsAccessReport.html",
                "description": "Grants permission to generate an access report for an AWS Organizations entity",
                "accessLevel": "Read",
                "dependentActions": [
                    "organizations:DescribePolicy",
                    "organizations:ListChildren",
                    "organizations:ListParents",
                    "organizations:ListPoliciesForTarget",
                    "organizations:ListRoots",
                    "organizations:ListTargetsForPolicy"
                ],
                "resourceTypes": {
                    "access-report": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:OrganizationsPolicyId"
                ]
            },
            "GenerateServiceLastAccessedDetails": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GenerateServiceLastAccessedDetails.html",
                "description": "Grants permission to generate a service last accessed data report for an IAM resource",
                "accessLevel": "Read"
            },
            "GetAccessKeyLastUsed": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccessKeyLastUsed.html",
                "description": "Grants permission to retrieve information about when the specified access key was last used",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "GetAccountAuthorizationDetails": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccountAuthorizationDetails.html",
                "description": "Grants permission to retrieve information about all IAM users, groups, roles, and policies in your AWS account, including their relationships to one another",
                "accessLevel": "Read"
            },
            "GetAccountPasswordPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccountPasswordPolicy.html",
                "description": "Grants permission to retrieve the password policy for the AWS account",
                "accessLevel": "Read"
            },
            "GetAccountSummary": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccountSummary.html",
                "description": "Grants permission to retrieve information about IAM entity usage and IAM quotas in the AWS account",
                "accessLevel": "List"
            },
            "GetContextKeysForCustomPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetContextKeysForCustomPolicy.html",
                "description": "Grants permission to retrieve a list of all of the context keys that are referenced in the specified policy",
                "accessLevel": "Read"
            },
            "GetContextKeysForPrincipalPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetContextKeysForPrincipalPolicy.html",
                "description": "Grants permission to retrieve a list of all context keys that are referenced in all IAM policies that are attached to the specified IAM identity (user, group, or role)",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": false
                    },
                    "role": {
                        "required": false
                    },
                    "user": {
                        "required": false
                    }
                }
            },
            "GetCredentialReport": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetCredentialReport.html",
                "description": "Grants permission to retrieve a credential report for the AWS account",
                "accessLevel": "Read"
            },
            "GetGroup": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetGroup.html",
                "description": "Grants permission to retrieve a list of IAM users in the specified IAM group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "GetGroupPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetGroupPolicy.html",
                "description": "Grants permission to retrieve an inline policy document that is embedded in the specified IAM group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "GetInstanceProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetInstanceProfile.html",
                "description": "Grants permission to retrieve information about the specified instance profile, including the instance profile's path, GUID, ARN, and role",
                "accessLevel": "Read",
                "resourceTypes": {
                    "instance-profile": {
                        "required": true
                    }
                }
            },
            "GetLoginProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetLoginProfile.html",
                "description": "Grants permission to retrieve the user name and password creation date for the specified IAM user",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "GetOpenIDConnectProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetOpenIDConnectProvider.html",
                "description": "Grants permission to retrieve information about the specified OpenID Connect (OIDC) provider resource in IAM",
                "accessLevel": "Read",
                "resourceTypes": {
                    "oidc-provider": {
                        "required": true
                    }
                }
            },
            "GetOrganizationsAccessReport": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetOrganizationsAccessReport.html",
                "description": "Grants permission to retrieve an AWS Organizations access report",
                "accessLevel": "Read"
            },
            "GetPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetPolicy.html",
                "description": "Grants permission to retrieve information about the specified managed policy, including the policy's default version and the total number of identities to which the policy is attached",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "GetPolicyVersion": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetPolicyVersion.html",
                "description": "Grants permission to retrieve information about a version of the specified managed policy, including the policy document",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "GetRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetRole.html",
                "description": "Grants permission to retrieve information about the specified role, including the role's path, GUID, ARN, and the role's trust policy",
                "accessLevel": "Read",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "GetRolePolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetRolePolicy.html",
                "description": "Grants permission to retrieve an inline policy document that is embedded with the specified IAM role",
                "accessLevel": "Read",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "GetSAMLProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetSAMLProvider.html",
                "description": "Grants permission to retrieve the SAML provider metadocument that was uploaded when the IAM SAML provider resource was created or updated",
                "accessLevel": "Read",
                "resourceTypes": {
                    "saml-provider": {
                        "required": true
                    }
                }
            },
            "GetSSHPublicKey": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetSSHPublicKey.html",
                "description": "Grants permission to retrieve the specified SSH public key, including metadata about the key",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "GetServerCertificate": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServerCertificate.html",
                "description": "Grants permission to retrieve information about the specified server certificate stored in IAM",
                "accessLevel": "Read",
                "resourceTypes": {
                    "server-certificate": {
                        "required": true
                    }
                }
            },
            "GetServiceLastAccessedDetails": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServiceLastAccessedDetails.html",
                "description": "Grants permission to retrieve information about the service last accessed data report",
                "accessLevel": "Read"
            },
            "GetServiceLastAccessedDetailsWithEntities": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServiceLastAccessedDetailsWithEntities.html",
                "description": "Grants permission to retrieve information about the entities from the service last accessed data report",
                "accessLevel": "Read"
            },
            "GetServiceLinkedRoleDeletionStatus": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServiceLinkedRoleDeletionStatus.html",
                "description": "Grants permission to retrieve an IAM service-linked role deletion status",
                "accessLevel": "Read",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "GetUser": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetUser.html",
                "description": "Grants permission to retrieve information about the specified IAM user, including the user's creation date, path, unique ID, and ARN",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "GetUserPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetUserPolicy.html",
                "description": "Grants permission to retrieve an inline policy document that is embedded in the specified IAM user",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListAccessKeys": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAccessKeys.html",
                "description": "Grants permission to list information about the access key IDs that are associated with the specified IAM user",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListAccountAliases": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAccountAliases.html",
                "description": "Grants permission to list the account alias that is associated with the AWS account",
                "accessLevel": "List"
            },
            "ListAttachedGroupPolicies": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAttachedGroupPolicies.html",
                "description": "Grants permission to list all managed policies that are attached to the specified IAM group",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ListAttachedRolePolicies": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAttachedRolePolicies.html",
                "description": "Grants permission to list all managed policies that are attached to the specified IAM role",
                "accessLevel": "List",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "ListAttachedUserPolicies": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAttachedUserPolicies.html",
                "description": "Grants permission to list all managed policies that are attached to the specified IAM user",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListEntitiesForPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListEntitiesForPolicy.html",
                "description": "Grants permission to list all IAM identities to which the specified managed policy is attached",
                "accessLevel": "List",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "ListGroupPolicies": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListGroupPolicies.html",
                "description": "Grants permission to list the names of the inline policies that are embedded in the specified IAM group",
                "accessLevel": "List",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ListGroups": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListGroups.html",
                "description": "Grants permission to list the IAM groups that have the specified path prefix",
                "accessLevel": "List"
            },
            "ListGroupsForUser": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListGroupsForUser.html",
                "description": "Grants permission to list the IAM groups that the specified IAM user belongs to",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListInstanceProfiles": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListInstanceProfiles.html",
                "description": "Grants permission to list the instance profiles that have the specified path prefix",
                "accessLevel": "List",
                "resourceTypes": {
                    "instance-profile": {
                        "required": true
                    }
                }
            },
            "ListInstanceProfilesForRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListInstanceProfilesForRole.html",
                "description": "Grants permission to list the instance profiles that have the specified associated IAM role",
                "accessLevel": "List",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "ListMFADevices": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListMFADevices.html",
                "description": "Grants permission to list the MFA devices for an IAM user",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": false
                    }
                }
            },
            "ListOpenIDConnectProviders": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListOpenIDConnectProviders.html",
                "description": "Grants permission to list information about the IAM OpenID Connect (OIDC) provider resource objects that are defined in the AWS account",
                "accessLevel": "List"
            },
            "ListPolicies": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListPolicies.html",
                "description": "Grants permission to list all managed policies",
                "accessLevel": "List"
            },
            "ListPoliciesGrantingServiceAccess": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListPoliciesGrantingServiceAccess.html",
                "description": "Grants permission to list information about the policies that grant an entity access to a specific service",
                "accessLevel": "List"
            },
            "ListPolicyVersions": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListPolicyVersions.html",
                "description": "Grants permission to list information about the versions of the specified managed policy, including the version that is currently set as the policy's default version",
                "accessLevel": "List",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "ListRolePolicies": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListRolePolicies.html",
                "description": "Grants permission to list the names of the inline policies that are embedded in the specified IAM role",
                "accessLevel": "List",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "ListRoleTags": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListRoleTags.html",
                "description": "Grants permission to list the tags that are attached to the specified IAM role.",
                "accessLevel": "List",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "ListRoles": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListRoles.html",
                "description": "Grants permission to list the IAM roles that have the specified path prefix",
                "accessLevel": "List"
            },
            "ListSAMLProviders": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListSAMLProviders.html",
                "description": "Grants permission to list the SAML provider resources in IAM",
                "accessLevel": "List"
            },
            "ListSSHPublicKeys": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListSSHPublicKeys.html",
                "description": "Grants permission to list information about the SSH public keys that are associated with the specified IAM user",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListServerCertificates": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListServerCertificates.html",
                "description": "Grants permission to list the server certificates that have the specified path prefix",
                "accessLevel": "List"
            },
            "ListServiceSpecificCredentials": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListServiceSpecificCredentials.html",
                "description": "Grants permission to list the service-specific credentials that are associated with the specified IAM user",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListSigningCertificates": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListSigningCertificates.html",
                "description": "Grants permission to list information about the signing certificates that are associated with the specified IAM user",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListUserPolicies": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListUserPolicies.html",
                "description": "Grants permission to list the names of the inline policies that are embedded in the specified IAM user",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListUserTags": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListUserTags.html",
                "description": "Grants permission to list the tags that are attached to the specified IAM user.",
                "accessLevel": "List",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ListUsers": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListUsers.html",
                "description": "Grants permission to list the IAM users that have the specified path prefix",
                "accessLevel": "List"
            },
            "ListVirtualMFADevices": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListVirtualMFADevices.html",
                "description": "Grants permission to list virtual MFA devices by assignment status",
                "accessLevel": "List"
            },
            "PassRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html",
                "description": "Grants permission to pass a role to a service",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:AssociatedResourceArn",
                    "iam:PassedToService"
                ]
            },
            "PutGroupPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutGroupPolicy.html",
                "description": "Grants permission to create or update an inline policy document that is embedded in the specified IAM group",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "PutRolePermissionsBoundary": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutRolePermissionsBoundary.html",
                "description": "Grants permission to set a managed policy as a permissions boundary for a role",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "PutRolePolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutRolePolicy.html",
                "description": "Grants permission to create or update an inline policy document that is embedded in the specified IAM role",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "PutUserPermissionsBoundary": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutUserPermissionsBoundary.html",
                "description": "Grants permission to set a managed policy as a permissions boundary for an IAM user",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "PutUserPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutUserPolicy.html",
                "description": "Grants permission to create or update an inline policy document that is embedded in the specified IAM user",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                },
                "conditions": [
                    "iam:PermissionsBoundary"
                ]
            },
            "RemoveClientIDFromOpenIDConnectProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_RemoveClientIDFromOpenIDConnectProvider.html",
                "description": "Grants permission to remove the client ID (audience) from the list of client IDs in the specified IAM OpenID Connect (OIDC) provider resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "oidc-provider": {
                        "required": true
                    }
                }
            },
            "RemoveRoleFromInstanceProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_RemoveRoleFromInstanceProfile.html",
                "description": "Grants permission to remove an IAM role from the specified EC2 instance profile",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance-profile": {
                        "required": true
                    }
                }
            },
            "RemoveUserFromGroup": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_RemoveUserFromGroup.html",
                "description": "Grants permission to remove an IAM user from the specified group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "ResetServiceSpecificCredential": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ResetServiceSpecificCredential.html",
                "description": "Grants permission to reset the password for an existing service-specific credential for an IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "ResyncMFADevice": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_ResyncMFADevice.html",
                "description": "Grants permission to synchronize the specified MFA device with its IAM entity (user or role)",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "SetDefaultPolicyVersion": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_SetDefaultPolicyVersion.html",
                "description": "Grants permission to set the version of the specified policy as the policy's default version",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "SetSecurityTokenServicePreferences": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_SetSecurityTokenServicePreferences.html",
                "description": "Grants permission to set the STS global endpoint token version",
                "accessLevel": "Write"
            },
            "SimulateCustomPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_SimulateCustomPolicy.html",
                "description": "Grants permission to simulate whether an identity-based policy or resource-based policy provides permissions for specific API operations and resources",
                "accessLevel": "Read"
            },
            "SimulatePrincipalPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_SimulatePrincipalPolicy.html",
                "description": "Grants permission to simulate whether an identity-based policy that is attached to a specified IAM entity (user or role) provides permissions for specific API operations and resources",
                "accessLevel": "Read",
                "resourceTypes": {
                    "group": {
                        "required": false
                    },
                    "role": {
                        "required": false
                    },
                    "user": {
                        "required": false
                    }
                }
            },
            "TagRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_TagRole.html",
                "description": "Grants permission to add tags to an IAM role.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "TagUser": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_TagUser.html",
                "description": "Grants permission to add tags to an IAM user.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UntagRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UntagRole.html",
                "description": "Grants permission to remove the specified tags from the role.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "UntagUser": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UntagUser.html",
                "description": "Grants permission to remove the specified tags from the user.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UpdateAccessKey": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateAccessKey.html",
                "description": "Grants permission to update the status of the specified access key as Active or Inactive",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UpdateAccountPasswordPolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateAccountPasswordPolicy.html",
                "description": "Grants permission to update the password policy settings for the AWS account",
                "accessLevel": "Write"
            },
            "UpdateAssumeRolePolicy": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateAssumeRolePolicy.html",
                "description": "Grants permission to update the policy that grants an IAM entity permission to assume a role",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "UpdateGroup": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateGroup.html",
                "description": "Grants permission to update the name or path of the specified IAM group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "group": {
                        "required": true
                    }
                }
            },
            "UpdateLoginProfile": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateLoginProfile.html",
                "description": "Grants permission to change the password for the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UpdateOpenIDConnectProviderThumbprint": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateOpenIDConnectProviderThumbprint.html",
                "description": "Grants permission to update the entire list of server certificate thumbprints that are associated with an OpenID Connect (OIDC) provider resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "oidc-provider": {
                        "required": true
                    }
                }
            },
            "UpdateRole": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateRole.html",
                "description": "Grants permission to update the description or maximum session duration setting of a role",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "UpdateRoleDescription": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateRoleDescription.html",
                "description": "Grants permission to update only the description of a role",
                "accessLevel": "Write",
                "resourceTypes": {
                    "role": {
                        "required": true
                    }
                }
            },
            "UpdateSAMLProvider": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateSAMLProvider.html",
                "description": "Grants permission to update the metadata document for an existing SAML provider resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "saml-provider": {
                        "required": true
                    }
                }
            },
            "UpdateSSHPublicKey": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateSSHPublicKey.html",
                "description": "Grants permission to update the status of an IAM user's SSH public key to active or inactive",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UpdateServerCertificate": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateServerCertificate.html",
                "description": "Grants permission to update the name or the path of the specified server certificate stored in IAM",
                "accessLevel": "Write",
                "resourceTypes": {
                    "server-certificate": {
                        "required": true
                    }
                }
            },
            "UpdateServiceSpecificCredential": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateServiceSpecificCredential.html",
                "description": "Grants permission to update the status of a service-specific credential to active or inactive for an IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UpdateSigningCertificate": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateSigningCertificate.html",
                "description": "Grants permission to update the status of the specified user signing certificate to active or disabled",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UpdateUser": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateUser.html",
                "description": "Grants permission to update the name or the path of the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UploadSSHPublicKey": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UploadSSHPublicKey.html",
                "description": "Grants permission to upload an SSH public key and associate it with the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            },
            "UploadServerCertificate": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UploadServerCertificate.html",
                "description": "Grants permission to upload a server certificate entity for the AWS account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "server-certificate": {
                        "required": true
                    }
                }
            },
            "UploadSigningCertificate": {
                "url": "https://docs.aws.amazon.com/IAM/latest/APIReference/API_UploadSigningCertificate.html",
                "description": "Grants permission to upload an X.509 signing certificate and associate it with the specified IAM user",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "access-report": {
                "name": "access-report",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor-view-data-orgs.html",
                "arn": "arn:${Partition}:iam::${Account}:access-report/${EntityPath}",
                "conditionKeys": []
            },
            "assumed-role": {
                "name": "assumed-role",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_temp_use-resources.html",
                "arn": "arn:${Partition}:iam::${Account}:assumed-role/${RoleName}/${RoleSessionName}",
                "conditionKeys": []
            },
            "federated-user": {
                "name": "federated-user",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_saml.html",
                "arn": "arn:${Partition}:iam::${Account}:federated-user/${UserName}",
                "conditionKeys": []
            },
            "group": {
                "name": "group",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_groups.html",
                "arn": "arn:${Partition}:iam::${Account}:group/${GroupNameWithPath}",
                "conditionKeys": []
            },
            "instance-profile": {
                "name": "instance-profile",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_switch-role-ec2_instance-profiles.html",
                "arn": "arn:${Partition}:iam::${Account}:instance-profile/${InstanceProfileNameWithPath}",
                "conditionKeys": []
            },
            "mfa": {
                "name": "mfa",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa.html",
                "arn": "arn:${Partition}:iam::${Account}:mfa/${MfaTokenIdWithPath}",
                "conditionKeys": []
            },
            "oidc-provider": {
                "name": "oidc-provider",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html",
                "arn": "arn:${Partition}:iam::${Account}:oidc-provider/${OidcProviderName}",
                "conditionKeys": []
            },
            "policy": {
                "name": "policy",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_managed-vs-inline.html",
                "arn": "arn:${Partition}:iam::${Account}:policy/${PolicyNameWithPath}",
                "conditionKeys": []
            },
            "role": {
                "name": "role",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles.html",
                "arn": "arn:${Partition}:iam::${Account}:role/${RoleNameWithPath}",
                "conditionKeys": [
                    "iam:ResourceTag/${TagKey}"
                ]
            },
            "saml-provider": {
                "name": "saml-provider",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_saml.html",
                "arn": "arn:${Partition}:iam::${Account}:saml-provider/${SamlProviderName}",
                "conditionKeys": []
            },
            "server-certificate": {
                "name": "server-certificate",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_server-certs.html",
                "arn": "arn:${Partition}:iam::${Account}:server-certificate/${CertificateNameWithPath}",
                "conditionKeys": []
            },
            "sms-mfa": {
                "name": "sms-mfa",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa_enable_sms.html",
                "arn": "arn:${Partition}:iam::${Account}:sms-mfa/${MfaTokenIdWithPath}",
                "conditionKeys": []
            },
            "user": {
                "name": "user",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html",
                "arn": "arn:${Partition}:iam::${Account}:user/${UserNameWithPath}",
                "conditionKeys": [
                    "iam:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to add a new client ID (audience) to the list of registered IDs for the specified IAM OpenID Connect (OIDC) provider resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AddClientIDToOpenIDConnectProvider.html
     */
    addClientIDToOpenIDConnectProvider() {
        this.add('iam:AddClientIDToOpenIDConnectProvider');
        return this;
    }
    /**
     * Grants permission to add an IAM role to the specified instance profile
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AddRoleToInstanceProfile.html
     */
    addRoleToInstanceProfile() {
        this.add('iam:AddRoleToInstanceProfile');
        return this;
    }
    /**
     * Grants permission to add an IAM user to the specified IAM group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AddUserToGroup.html
     */
    addUserToGroup() {
        this.add('iam:AddUserToGroup');
        return this;
    }
    /**
     * Grants permission to attach a managed policy to the specified IAM group
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AttachGroupPolicy.html
     */
    attachGroupPolicy() {
        this.add('iam:AttachGroupPolicy');
        return this;
    }
    /**
     * Grants permission to attach a managed policy to the specified IAM role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AttachRolePolicy.html
     */
    attachRolePolicy() {
        this.add('iam:AttachRolePolicy');
        return this;
    }
    /**
     * Grants permission to attach a managed policy to the specified IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_AttachUserPolicy.html
     */
    attachUserPolicy() {
        this.add('iam:AttachUserPolicy');
        return this;
    }
    /**
     * Grants permission for an IAM user to to change their own password
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ChangePassword.html
     */
    changePassword() {
        this.add('iam:ChangePassword');
        return this;
    }
    /**
     * Grants permission to create access key and secret access key for the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateAccessKey.html
     */
    createAccessKey() {
        this.add('iam:CreateAccessKey');
        return this;
    }
    /**
     * Grants permission to create an alias for your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateAccountAlias.html
     */
    createAccountAlias() {
        this.add('iam:CreateAccountAlias');
        return this;
    }
    /**
     * Grants permission to create a new group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateGroup.html
     */
    createGroup() {
        this.add('iam:CreateGroup');
        return this;
    }
    /**
     * Grants permission to create a new instance profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateInstanceProfile.html
     */
    createInstanceProfile() {
        this.add('iam:CreateInstanceProfile');
        return this;
    }
    /**
     * Grants permission to create a password for the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateLoginProfile.html
     */
    createLoginProfile() {
        this.add('iam:CreateLoginProfile');
        return this;
    }
    /**
     * Grants permission to create an IAM resource that describes an identity provider (IdP) that supports OpenID Connect (OIDC)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateOpenIDConnectProvider.html
     */
    createOpenIDConnectProvider() {
        this.add('iam:CreateOpenIDConnectProvider');
        return this;
    }
    /**
     * Grants permission to create a new managed policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreatePolicy.html
     */
    createPolicy() {
        this.add('iam:CreatePolicy');
        return this;
    }
    /**
     * Grants permission to create a new version of the specified managed policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreatePolicyVersion.html
     */
    createPolicyVersion() {
        this.add('iam:CreatePolicyVersion');
        return this;
    }
    /**
     * Grants permission to create a new role
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateRole.html
     */
    createRole() {
        this.add('iam:CreateRole');
        return this;
    }
    /**
     * Grants permission to create an IAM resource that describes an identity provider (IdP) that supports SAML 2.0
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateSAMLProvider.html
     */
    createSAMLProvider() {
        this.add('iam:CreateSAMLProvider');
        return this;
    }
    /**
     * Grants permission to create an IAM role that allows an AWS service to perform actions on your behalf
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAWSServiceName()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateServiceLinkedRole.html
     */
    createServiceLinkedRole() {
        this.add('iam:CreateServiceLinkedRole');
        return this;
    }
    /**
     * Grants permission to create a new service-specific credential for an IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateServiceSpecificCredential.html
     */
    createServiceSpecificCredential() {
        this.add('iam:CreateServiceSpecificCredential');
        return this;
    }
    /**
     * Grants permission to create a new IAM user
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateUser.html
     */
    createUser() {
        this.add('iam:CreateUser');
        return this;
    }
    /**
     * Grants permission to create a new virtual MFA device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateVirtualMFADevice.html
     */
    createVirtualMFADevice() {
        this.add('iam:CreateVirtualMFADevice');
        return this;
    }
    /**
     * Grants permission to deactivate the specified MFA device and remove its association with the IAM user for which it was originally enabled
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeactivateMFADevice.html
     */
    deactivateMFADevice() {
        this.add('iam:DeactivateMFADevice');
        return this;
    }
    /**
     * Grants permission to delete the access key pair that is associated with the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteAccessKey.html
     */
    deleteAccessKey() {
        this.add('iam:DeleteAccessKey');
        return this;
    }
    /**
     * Grants permission to delete the specified AWS account alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteAccountAlias.html
     */
    deleteAccountAlias() {
        this.add('iam:DeleteAccountAlias');
        return this;
    }
    /**
     * Grants permission to delete the password policy for the AWS account
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteAccountPasswordPolicy.html
     */
    deleteAccountPasswordPolicy() {
        this.add('iam:DeleteAccountPasswordPolicy');
        return this;
    }
    /**
     * Grants permission to delete the specified IAM group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteGroup.html
     */
    deleteGroup() {
        this.add('iam:DeleteGroup');
        return this;
    }
    /**
     * Grants permission to delete the specified inline policy from its group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteGroupPolicy.html
     */
    deleteGroupPolicy() {
        this.add('iam:DeleteGroupPolicy');
        return this;
    }
    /**
     * Grants permission to delete the specified instance profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteInstanceProfile.html
     */
    deleteInstanceProfile() {
        this.add('iam:DeleteInstanceProfile');
        return this;
    }
    /**
     * Grants permission to delete the password for the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteLoginProfile.html
     */
    deleteLoginProfile() {
        this.add('iam:DeleteLoginProfile');
        return this;
    }
    /**
     * Grants permission to delete an OpenID Connect identity provider (IdP) resource object in IAM
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteOpenIDConnectProvider.html
     */
    deleteOpenIDConnectProvider() {
        this.add('iam:DeleteOpenIDConnectProvider');
        return this;
    }
    /**
     * Grants permission to delete the specified managed policy and remove it from any IAM entities (users, groups, or roles) to which it is attached
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeletePolicy.html
     */
    deletePolicy() {
        this.add('iam:DeletePolicy');
        return this;
    }
    /**
     * Grants permission to delete a version from the specified managed policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeletePolicyVersion.html
     */
    deletePolicyVersion() {
        this.add('iam:DeletePolicyVersion');
        return this;
    }
    /**
     * Grants permission to delete the specified role
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteRole.html
     */
    deleteRole() {
        this.add('iam:DeleteRole');
        return this;
    }
    /**
     * Grants permission to remove the permissions boundary from a role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteRolePermissionsBoundary.html
     */
    deleteRolePermissionsBoundary() {
        this.add('iam:DeleteRolePermissionsBoundary');
        return this;
    }
    /**
     * Grants permission to delete the specified inline policy from the specified role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteRolePolicy.html
     */
    deleteRolePolicy() {
        this.add('iam:DeleteRolePolicy');
        return this;
    }
    /**
     * Grants permission to delete a SAML provider resource in IAM
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteSAMLProvider.html
     */
    deleteSAMLProvider() {
        this.add('iam:DeleteSAMLProvider');
        return this;
    }
    /**
     * Grants permission to delete the specified SSH public key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteSSHPublicKey.html
     */
    deleteSSHPublicKey() {
        this.add('iam:DeleteSSHPublicKey');
        return this;
    }
    /**
     * Grants permission to delete the specified server certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteServerCertificate.html
     */
    deleteServerCertificate() {
        this.add('iam:DeleteServerCertificate');
        return this;
    }
    /**
     * Grants permission to delete an IAM role that is linked to a specific AWS service, if the service is no longer using it
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteServiceLinkedRole.html
     */
    deleteServiceLinkedRole() {
        this.add('iam:DeleteServiceLinkedRole');
        return this;
    }
    /**
     * Grants permission to delete the specified service-specific credential for an IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteServiceSpecificCredential.html
     */
    deleteServiceSpecificCredential() {
        this.add('iam:DeleteServiceSpecificCredential');
        return this;
    }
    /**
     * Grants permission to delete a signing certificate that is associated with the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteSigningCertificate.html
     */
    deleteSigningCertificate() {
        this.add('iam:DeleteSigningCertificate');
        return this;
    }
    /**
     * Grants permission to delete the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteUser.html
     */
    deleteUser() {
        this.add('iam:DeleteUser');
        return this;
    }
    /**
     * Grants permission to remove the permissions boundary from the specified IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteUserPermissionsBoundary.html
     */
    deleteUserPermissionsBoundary() {
        this.add('iam:DeleteUserPermissionsBoundary');
        return this;
    }
    /**
     * Grants permission to delete the specified inline policy from an IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteUserPolicy.html
     */
    deleteUserPolicy() {
        this.add('iam:DeleteUserPolicy');
        return this;
    }
    /**
     * Grants permission to delete a virtual MFA device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DeleteVirtualMFADevice.html
     */
    deleteVirtualMFADevice() {
        this.add('iam:DeleteVirtualMFADevice');
        return this;
    }
    /**
     * Grants permission to detach a managed policy from the specified IAM group
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DetachGroupPolicy.html
     */
    detachGroupPolicy() {
        this.add('iam:DetachGroupPolicy');
        return this;
    }
    /**
     * Grants permission to detach a managed policy from the specified role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DetachRolePolicy.html
     */
    detachRolePolicy() {
        this.add('iam:DetachRolePolicy');
        return this;
    }
    /**
     * Grants permission to detach a managed policy from the specified IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPolicyARN()
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_DetachUserPolicy.html
     */
    detachUserPolicy() {
        this.add('iam:DetachUserPolicy');
        return this;
    }
    /**
     * Grants permission to enable an MFA device and associate it with the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_EnableMFADevice.html
     */
    enableMFADevice() {
        this.add('iam:EnableMFADevice');
        return this;
    }
    /**
     * Grants permission to generate a credential report for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GenerateCredentialReport.html
     */
    generateCredentialReport() {
        this.add('iam:GenerateCredentialReport');
        return this;
    }
    /**
     * Grants permission to generate an access report for an AWS Organizations entity
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifOrganizationsPolicyId()
     *
     * Dependent actions:
     * - organizations:DescribePolicy
     * - organizations:ListChildren
     * - organizations:ListParents
     * - organizations:ListPoliciesForTarget
     * - organizations:ListRoots
     * - organizations:ListTargetsForPolicy
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GenerateOrganizationsAccessReport.html
     */
    generateOrganizationsAccessReport() {
        this.add('iam:GenerateOrganizationsAccessReport');
        return this;
    }
    /**
     * Grants permission to generate a service last accessed data report for an IAM resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GenerateServiceLastAccessedDetails.html
     */
    generateServiceLastAccessedDetails() {
        this.add('iam:GenerateServiceLastAccessedDetails');
        return this;
    }
    /**
     * Grants permission to retrieve information about when the specified access key was last used
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccessKeyLastUsed.html
     */
    getAccessKeyLastUsed() {
        this.add('iam:GetAccessKeyLastUsed');
        return this;
    }
    /**
     * Grants permission to retrieve information about all IAM users, groups, roles, and policies in your AWS account, including their relationships to one another
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccountAuthorizationDetails.html
     */
    getAccountAuthorizationDetails() {
        this.add('iam:GetAccountAuthorizationDetails');
        return this;
    }
    /**
     * Grants permission to retrieve the password policy for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccountPasswordPolicy.html
     */
    getAccountPasswordPolicy() {
        this.add('iam:GetAccountPasswordPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve information about IAM entity usage and IAM quotas in the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetAccountSummary.html
     */
    getAccountSummary() {
        this.add('iam:GetAccountSummary');
        return this;
    }
    /**
     * Grants permission to retrieve a list of all of the context keys that are referenced in the specified policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetContextKeysForCustomPolicy.html
     */
    getContextKeysForCustomPolicy() {
        this.add('iam:GetContextKeysForCustomPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve a list of all context keys that are referenced in all IAM policies that are attached to the specified IAM identity (user, group, or role)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetContextKeysForPrincipalPolicy.html
     */
    getContextKeysForPrincipalPolicy() {
        this.add('iam:GetContextKeysForPrincipalPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve a credential report for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetCredentialReport.html
     */
    getCredentialReport() {
        this.add('iam:GetCredentialReport');
        return this;
    }
    /**
     * Grants permission to retrieve a list of IAM users in the specified IAM group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetGroup.html
     */
    getGroup() {
        this.add('iam:GetGroup');
        return this;
    }
    /**
     * Grants permission to retrieve an inline policy document that is embedded in the specified IAM group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetGroupPolicy.html
     */
    getGroupPolicy() {
        this.add('iam:GetGroupPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified instance profile, including the instance profile's path, GUID, ARN, and role
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetInstanceProfile.html
     */
    getInstanceProfile() {
        this.add('iam:GetInstanceProfile');
        return this;
    }
    /**
     * Grants permission to retrieve the user name and password creation date for the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetLoginProfile.html
     */
    getLoginProfile() {
        this.add('iam:GetLoginProfile');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified OpenID Connect (OIDC) provider resource in IAM
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetOpenIDConnectProvider.html
     */
    getOpenIDConnectProvider() {
        this.add('iam:GetOpenIDConnectProvider');
        return this;
    }
    /**
     * Grants permission to retrieve an AWS Organizations access report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetOrganizationsAccessReport.html
     */
    getOrganizationsAccessReport() {
        this.add('iam:GetOrganizationsAccessReport');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified managed policy, including the policy's default version and the total number of identities to which the policy is attached
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetPolicy.html
     */
    getPolicy() {
        this.add('iam:GetPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve information about a version of the specified managed policy, including the policy document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetPolicyVersion.html
     */
    getPolicyVersion() {
        this.add('iam:GetPolicyVersion');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified role, including the role's path, GUID, ARN, and the role's trust policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetRole.html
     */
    getRole() {
        this.add('iam:GetRole');
        return this;
    }
    /**
     * Grants permission to retrieve an inline policy document that is embedded with the specified IAM role
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetRolePolicy.html
     */
    getRolePolicy() {
        this.add('iam:GetRolePolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the SAML provider metadocument that was uploaded when the IAM SAML provider resource was created or updated
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetSAMLProvider.html
     */
    getSAMLProvider() {
        this.add('iam:GetSAMLProvider');
        return this;
    }
    /**
     * Grants permission to retrieve the specified SSH public key, including metadata about the key
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetSSHPublicKey.html
     */
    getSSHPublicKey() {
        this.add('iam:GetSSHPublicKey');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified server certificate stored in IAM
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServerCertificate.html
     */
    getServerCertificate() {
        this.add('iam:GetServerCertificate');
        return this;
    }
    /**
     * Grants permission to retrieve information about the service last accessed data report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServiceLastAccessedDetails.html
     */
    getServiceLastAccessedDetails() {
        this.add('iam:GetServiceLastAccessedDetails');
        return this;
    }
    /**
     * Grants permission to retrieve information about the entities from the service last accessed data report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServiceLastAccessedDetailsWithEntities.html
     */
    getServiceLastAccessedDetailsWithEntities() {
        this.add('iam:GetServiceLastAccessedDetailsWithEntities');
        return this;
    }
    /**
     * Grants permission to retrieve an IAM service-linked role deletion status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetServiceLinkedRoleDeletionStatus.html
     */
    getServiceLinkedRoleDeletionStatus() {
        this.add('iam:GetServiceLinkedRoleDeletionStatus');
        return this;
    }
    /**
     * Grants permission to retrieve information about the specified IAM user, including the user's creation date, path, unique ID, and ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetUser.html
     */
    getUser() {
        this.add('iam:GetUser');
        return this;
    }
    /**
     * Grants permission to retrieve an inline policy document that is embedded in the specified IAM user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_GetUserPolicy.html
     */
    getUserPolicy() {
        this.add('iam:GetUserPolicy');
        return this;
    }
    /**
     * Grants permission to list information about the access key IDs that are associated with the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAccessKeys.html
     */
    listAccessKeys() {
        this.add('iam:ListAccessKeys');
        return this;
    }
    /**
     * Grants permission to list the account alias that is associated with the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAccountAliases.html
     */
    listAccountAliases() {
        this.add('iam:ListAccountAliases');
        return this;
    }
    /**
     * Grants permission to list all managed policies that are attached to the specified IAM group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAttachedGroupPolicies.html
     */
    listAttachedGroupPolicies() {
        this.add('iam:ListAttachedGroupPolicies');
        return this;
    }
    /**
     * Grants permission to list all managed policies that are attached to the specified IAM role
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAttachedRolePolicies.html
     */
    listAttachedRolePolicies() {
        this.add('iam:ListAttachedRolePolicies');
        return this;
    }
    /**
     * Grants permission to list all managed policies that are attached to the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListAttachedUserPolicies.html
     */
    listAttachedUserPolicies() {
        this.add('iam:ListAttachedUserPolicies');
        return this;
    }
    /**
     * Grants permission to list all IAM identities to which the specified managed policy is attached
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListEntitiesForPolicy.html
     */
    listEntitiesForPolicy() {
        this.add('iam:ListEntitiesForPolicy');
        return this;
    }
    /**
     * Grants permission to list the names of the inline policies that are embedded in the specified IAM group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListGroupPolicies.html
     */
    listGroupPolicies() {
        this.add('iam:ListGroupPolicies');
        return this;
    }
    /**
     * Grants permission to list the IAM groups that have the specified path prefix
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListGroups.html
     */
    listGroups() {
        this.add('iam:ListGroups');
        return this;
    }
    /**
     * Grants permission to list the IAM groups that the specified IAM user belongs to
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListGroupsForUser.html
     */
    listGroupsForUser() {
        this.add('iam:ListGroupsForUser');
        return this;
    }
    /**
     * Grants permission to list the instance profiles that have the specified path prefix
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListInstanceProfiles.html
     */
    listInstanceProfiles() {
        this.add('iam:ListInstanceProfiles');
        return this;
    }
    /**
     * Grants permission to list the instance profiles that have the specified associated IAM role
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListInstanceProfilesForRole.html
     */
    listInstanceProfilesForRole() {
        this.add('iam:ListInstanceProfilesForRole');
        return this;
    }
    /**
     * Grants permission to list the MFA devices for an IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListMFADevices.html
     */
    listMFADevices() {
        this.add('iam:ListMFADevices');
        return this;
    }
    /**
     * Grants permission to list information about the IAM OpenID Connect (OIDC) provider resource objects that are defined in the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListOpenIDConnectProviders.html
     */
    listOpenIDConnectProviders() {
        this.add('iam:ListOpenIDConnectProviders');
        return this;
    }
    /**
     * Grants permission to list all managed policies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListPolicies.html
     */
    listPolicies() {
        this.add('iam:ListPolicies');
        return this;
    }
    /**
     * Grants permission to list information about the policies that grant an entity access to a specific service
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListPoliciesGrantingServiceAccess.html
     */
    listPoliciesGrantingServiceAccess() {
        this.add('iam:ListPoliciesGrantingServiceAccess');
        return this;
    }
    /**
     * Grants permission to list information about the versions of the specified managed policy, including the version that is currently set as the policy's default version
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListPolicyVersions.html
     */
    listPolicyVersions() {
        this.add('iam:ListPolicyVersions');
        return this;
    }
    /**
     * Grants permission to list the names of the inline policies that are embedded in the specified IAM role
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListRolePolicies.html
     */
    listRolePolicies() {
        this.add('iam:ListRolePolicies');
        return this;
    }
    /**
     * Grants permission to list the tags that are attached to the specified IAM role.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListRoleTags.html
     */
    listRoleTags() {
        this.add('iam:ListRoleTags');
        return this;
    }
    /**
     * Grants permission to list the IAM roles that have the specified path prefix
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListRoles.html
     */
    listRoles() {
        this.add('iam:ListRoles');
        return this;
    }
    /**
     * Grants permission to list the SAML provider resources in IAM
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListSAMLProviders.html
     */
    listSAMLProviders() {
        this.add('iam:ListSAMLProviders');
        return this;
    }
    /**
     * Grants permission to list information about the SSH public keys that are associated with the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListSSHPublicKeys.html
     */
    listSSHPublicKeys() {
        this.add('iam:ListSSHPublicKeys');
        return this;
    }
    /**
     * Grants permission to list the server certificates that have the specified path prefix
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListServerCertificates.html
     */
    listServerCertificates() {
        this.add('iam:ListServerCertificates');
        return this;
    }
    /**
     * Grants permission to list the service-specific credentials that are associated with the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListServiceSpecificCredentials.html
     */
    listServiceSpecificCredentials() {
        this.add('iam:ListServiceSpecificCredentials');
        return this;
    }
    /**
     * Grants permission to list information about the signing certificates that are associated with the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListSigningCertificates.html
     */
    listSigningCertificates() {
        this.add('iam:ListSigningCertificates');
        return this;
    }
    /**
     * Grants permission to list the names of the inline policies that are embedded in the specified IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListUserPolicies.html
     */
    listUserPolicies() {
        this.add('iam:ListUserPolicies');
        return this;
    }
    /**
     * Grants permission to list the tags that are attached to the specified IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListUserTags.html
     */
    listUserTags() {
        this.add('iam:ListUserTags');
        return this;
    }
    /**
     * Grants permission to list the IAM users that have the specified path prefix
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListUsers.html
     */
    listUsers() {
        this.add('iam:ListUsers');
        return this;
    }
    /**
     * Grants permission to list virtual MFA devices by assignment status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ListVirtualMFADevices.html
     */
    listVirtualMFADevices() {
        this.add('iam:ListVirtualMFADevices');
        return this;
    }
    /**
     * Grants permission to pass a role to a service
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAssociatedResourceArn()
     * - .ifPassedToService()
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html
     */
    passRole() {
        this.add('iam:PassRole');
        return this;
    }
    /**
     * Grants permission to create or update an inline policy document that is embedded in the specified IAM group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutGroupPolicy.html
     */
    putGroupPolicy() {
        this.add('iam:PutGroupPolicy');
        return this;
    }
    /**
     * Grants permission to set a managed policy as a permissions boundary for a role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutRolePermissionsBoundary.html
     */
    putRolePermissionsBoundary() {
        this.add('iam:PutRolePermissionsBoundary');
        return this;
    }
    /**
     * Grants permission to create or update an inline policy document that is embedded in the specified IAM role
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutRolePolicy.html
     */
    putRolePolicy() {
        this.add('iam:PutRolePolicy');
        return this;
    }
    /**
     * Grants permission to set a managed policy as a permissions boundary for an IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutUserPermissionsBoundary.html
     */
    putUserPermissionsBoundary() {
        this.add('iam:PutUserPermissionsBoundary');
        return this;
    }
    /**
     * Grants permission to create or update an inline policy document that is embedded in the specified IAM user
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifPermissionsBoundary()
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_PutUserPolicy.html
     */
    putUserPolicy() {
        this.add('iam:PutUserPolicy');
        return this;
    }
    /**
     * Grants permission to remove the client ID (audience) from the list of client IDs in the specified IAM OpenID Connect (OIDC) provider resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_RemoveClientIDFromOpenIDConnectProvider.html
     */
    removeClientIDFromOpenIDConnectProvider() {
        this.add('iam:RemoveClientIDFromOpenIDConnectProvider');
        return this;
    }
    /**
     * Grants permission to remove an IAM role from the specified EC2 instance profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_RemoveRoleFromInstanceProfile.html
     */
    removeRoleFromInstanceProfile() {
        this.add('iam:RemoveRoleFromInstanceProfile');
        return this;
    }
    /**
     * Grants permission to remove an IAM user from the specified group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_RemoveUserFromGroup.html
     */
    removeUserFromGroup() {
        this.add('iam:RemoveUserFromGroup');
        return this;
    }
    /**
     * Grants permission to reset the password for an existing service-specific credential for an IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ResetServiceSpecificCredential.html
     */
    resetServiceSpecificCredential() {
        this.add('iam:ResetServiceSpecificCredential');
        return this;
    }
    /**
     * Grants permission to synchronize the specified MFA device with its IAM entity (user or role)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_ResyncMFADevice.html
     */
    resyncMFADevice() {
        this.add('iam:ResyncMFADevice');
        return this;
    }
    /**
     * Grants permission to set the version of the specified policy as the policy's default version
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_SetDefaultPolicyVersion.html
     */
    setDefaultPolicyVersion() {
        this.add('iam:SetDefaultPolicyVersion');
        return this;
    }
    /**
     * Grants permission to set the STS global endpoint token version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_SetSecurityTokenServicePreferences.html
     */
    setSecurityTokenServicePreferences() {
        this.add('iam:SetSecurityTokenServicePreferences');
        return this;
    }
    /**
     * Grants permission to simulate whether an identity-based policy or resource-based policy provides permissions for specific API operations and resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_SimulateCustomPolicy.html
     */
    simulateCustomPolicy() {
        this.add('iam:SimulateCustomPolicy');
        return this;
    }
    /**
     * Grants permission to simulate whether an identity-based policy that is attached to a specified IAM entity (user or role) provides permissions for specific API operations and resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_SimulatePrincipalPolicy.html
     */
    simulatePrincipalPolicy() {
        this.add('iam:SimulatePrincipalPolicy');
        return this;
    }
    /**
     * Grants permission to add tags to an IAM role.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_TagRole.html
     */
    tagRole() {
        this.add('iam:TagRole');
        return this;
    }
    /**
     * Grants permission to add tags to an IAM user.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_TagUser.html
     */
    tagUser() {
        this.add('iam:TagUser');
        return this;
    }
    /**
     * Grants permission to remove the specified tags from the role.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UntagRole.html
     */
    untagRole() {
        this.add('iam:UntagRole');
        return this;
    }
    /**
     * Grants permission to remove the specified tags from the user.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UntagUser.html
     */
    untagUser() {
        this.add('iam:UntagUser');
        return this;
    }
    /**
     * Grants permission to update the status of the specified access key as Active or Inactive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateAccessKey.html
     */
    updateAccessKey() {
        this.add('iam:UpdateAccessKey');
        return this;
    }
    /**
     * Grants permission to update the password policy settings for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateAccountPasswordPolicy.html
     */
    updateAccountPasswordPolicy() {
        this.add('iam:UpdateAccountPasswordPolicy');
        return this;
    }
    /**
     * Grants permission to update the policy that grants an IAM entity permission to assume a role
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateAssumeRolePolicy.html
     */
    updateAssumeRolePolicy() {
        this.add('iam:UpdateAssumeRolePolicy');
        return this;
    }
    /**
     * Grants permission to update the name or path of the specified IAM group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateGroup.html
     */
    updateGroup() {
        this.add('iam:UpdateGroup');
        return this;
    }
    /**
     * Grants permission to change the password for the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateLoginProfile.html
     */
    updateLoginProfile() {
        this.add('iam:UpdateLoginProfile');
        return this;
    }
    /**
     * Grants permission to update the entire list of server certificate thumbprints that are associated with an OpenID Connect (OIDC) provider resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateOpenIDConnectProviderThumbprint.html
     */
    updateOpenIDConnectProviderThumbprint() {
        this.add('iam:UpdateOpenIDConnectProviderThumbprint');
        return this;
    }
    /**
     * Grants permission to update the description or maximum session duration setting of a role
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateRole.html
     */
    updateRole() {
        this.add('iam:UpdateRole');
        return this;
    }
    /**
     * Grants permission to update only the description of a role
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateRoleDescription.html
     */
    updateRoleDescription() {
        this.add('iam:UpdateRoleDescription');
        return this;
    }
    /**
     * Grants permission to update the metadata document for an existing SAML provider resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateSAMLProvider.html
     */
    updateSAMLProvider() {
        this.add('iam:UpdateSAMLProvider');
        return this;
    }
    /**
     * Grants permission to update the status of an IAM user's SSH public key to active or inactive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateSSHPublicKey.html
     */
    updateSSHPublicKey() {
        this.add('iam:UpdateSSHPublicKey');
        return this;
    }
    /**
     * Grants permission to update the name or the path of the specified server certificate stored in IAM
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateServerCertificate.html
     */
    updateServerCertificate() {
        this.add('iam:UpdateServerCertificate');
        return this;
    }
    /**
     * Grants permission to update the status of a service-specific credential to active or inactive for an IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateServiceSpecificCredential.html
     */
    updateServiceSpecificCredential() {
        this.add('iam:UpdateServiceSpecificCredential');
        return this;
    }
    /**
     * Grants permission to update the status of the specified user signing certificate to active or disabled
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateSigningCertificate.html
     */
    updateSigningCertificate() {
        this.add('iam:UpdateSigningCertificate');
        return this;
    }
    /**
     * Grants permission to update the name or the path of the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UpdateUser.html
     */
    updateUser() {
        this.add('iam:UpdateUser');
        return this;
    }
    /**
     * Grants permission to upload an SSH public key and associate it with the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UploadSSHPublicKey.html
     */
    uploadSSHPublicKey() {
        this.add('iam:UploadSSHPublicKey');
        return this;
    }
    /**
     * Grants permission to upload a server certificate entity for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UploadServerCertificate.html
     */
    uploadServerCertificate() {
        this.add('iam:UploadServerCertificate');
        return this;
    }
    /**
     * Grants permission to upload an X.509 signing certificate and associate it with the specified IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/IAM/latest/APIReference/API_UploadSigningCertificate.html
     */
    uploadSigningCertificate() {
        this.add('iam:UploadSigningCertificate');
        return this;
    }
    /**
     * Adds a resource of type access-report to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor-view-data-orgs.html
     *
     * @param entityPath - Identifier for the entityPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAccessReport(entityPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:access-report/${EntityPath}';
        arn = arn.replace('${EntityPath}', entityPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type assumed-role to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_temp_use-resources.html
     *
     * @param roleName - Identifier for the roleName.
     * @param roleSessionName - Identifier for the roleSessionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAssumedRole(roleName, roleSessionName, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:assumed-role/${RoleName}/${RoleSessionName}';
        arn = arn.replace('${RoleName}', roleName);
        arn = arn.replace('${RoleSessionName}', roleSessionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type federated-user to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_saml.html
     *
     * @param userName - Identifier for the userName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFederatedUser(userName, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:federated-user/${UserName}';
        arn = arn.replace('${UserName}', userName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_groups.html
     *
     * @param groupNameWithPath - Identifier for the groupNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGroup(groupNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:group/${GroupNameWithPath}';
        arn = arn.replace('${GroupNameWithPath}', groupNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type instance-profile to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_switch-role-ec2_instance-profiles.html
     *
     * @param instanceProfileNameWithPath - Identifier for the instanceProfileNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onInstanceProfile(instanceProfileNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:instance-profile/${InstanceProfileNameWithPath}';
        arn = arn.replace('${InstanceProfileNameWithPath}', instanceProfileNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type mfa to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa.html
     *
     * @param mfaTokenIdWithPath - Identifier for the mfaTokenIdWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onMfa(mfaTokenIdWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:mfa/${MfaTokenIdWithPath}';
        arn = arn.replace('${MfaTokenIdWithPath}', mfaTokenIdWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type oidc-provider to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html
     *
     * @param oidcProviderName - Identifier for the oidcProviderName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOidcProvider(oidcProviderName, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:oidc-provider/${OidcProviderName}';
        arn = arn.replace('${OidcProviderName}', oidcProviderName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type policy to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_managed-vs-inline.html
     *
     * @param policyNameWithPath - Identifier for the policyNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPolicy(policyNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:policy/${PolicyNameWithPath}';
        arn = arn.replace('${PolicyNameWithPath}', policyNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type role to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles.html
     *
     * @param roleNameWithPath - Identifier for the roleNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifResourceTag()
     */
    onRole(roleNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:role/${RoleNameWithPath}';
        arn = arn.replace('${RoleNameWithPath}', roleNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type saml-provider to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_saml.html
     *
     * @param samlProviderName - Identifier for the samlProviderName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSamlProvider(samlProviderName, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:saml-provider/${SamlProviderName}';
        arn = arn.replace('${SamlProviderName}', samlProviderName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type server-certificate to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_server-certs.html
     *
     * @param certificateNameWithPath - Identifier for the certificateNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onServerCertificate(certificateNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:server-certificate/${CertificateNameWithPath}';
        arn = arn.replace('${CertificateNameWithPath}', certificateNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type sms-mfa to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa_enable_sms.html
     *
     * @param mfaTokenIdWithPath - Identifier for the mfaTokenIdWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSmsMfa(mfaTokenIdWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:sms-mfa/${MfaTokenIdWithPath}';
        arn = arn.replace('${MfaTokenIdWithPath}', mfaTokenIdWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html
     *
     * @param userNameWithPath - Identifier for the userNameWithPath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifResourceTag()
     */
    onUser(userNameWithPath, account, partition) {
        var arn = 'arn:${Partition}:iam::${Account}:user/${UserNameWithPath}';
        arn = arn.replace('${UserNameWithPath}', userNameWithPath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the AWS service to which this role is attached
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_AWSServiceName
     *
     * Applies to actions:
     * - .createServiceLinkedRole()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAWSServiceName(value, operator) {
        return this.if(`iam:AWSServiceName`, value, operator || 'StringLike');
    }
    /**
     * Filters by the resource that the role will be used on behalf of
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_AssociatedResourceArn
     *
     * Applies to actions:
     * - .passRole()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifAssociatedResourceArn(value, operator) {
        return this.if(`iam:AssociatedResourceArn`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the ID of an AWS Organizations policy
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_OrganizationsPolicyId
     *
     * Applies to actions:
     * - .generateOrganizationsAccessReport()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOrganizationsPolicyId(value, operator) {
        return this.if(`iam:OrganizationsPolicyId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the AWS service to which this role is passed
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_PassedToService
     *
     * Applies to actions:
     * - .passRole()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPassedToService(value, operator) {
        return this.if(`iam:PassedToService`, value, operator || 'StringLike');
    }
    /**
     * Filters access if the specified policy is set as the permissions boundary on the IAM entity (user or role)
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_PermissionsBoundary
     *
     * Applies to actions:
     * - .attachRolePolicy()
     * - .attachUserPolicy()
     * - .createRole()
     * - .createUser()
     * - .deleteRolePermissionsBoundary()
     * - .deleteRolePolicy()
     * - .deleteUserPermissionsBoundary()
     * - .deleteUserPolicy()
     * - .detachRolePolicy()
     * - .detachUserPolicy()
     * - .putRolePermissionsBoundary()
     * - .putRolePolicy()
     * - .putUserPermissionsBoundary()
     * - .putUserPolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPermissionsBoundary(value, operator) {
        return this.if(`iam:PermissionsBoundary`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of an IAM policy
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_PolicyARN
     *
     * Applies to actions:
     * - .attachGroupPolicy()
     * - .attachRolePolicy()
     * - .attachUserPolicy()
     * - .detachGroupPolicy()
     * - .detachRolePolicy()
     * - .detachUserPolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifPolicyARN(value, operator) {
        return this.if(`iam:PolicyARN`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the tags attached to an IAM entity (user or role).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_ResourceTag
     *
     * Applies to resource types:
     * - role
     * - user
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`iam:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
}
exports.Iam = Iam;
//# sourceMappingURL=data:application/json;base64,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