"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Networkmanager = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [networkmanager](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_networkmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Networkmanager extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [networkmanager](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_networkmanager.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'networkmanager';
        this.actionList = {
            "AssociateCustomerGateway": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_AssociateCustomerGateway.html",
                "description": "Grants permission to associate a customer gateway to a device",
                "accessLevel": "Write",
                "resourceTypes": {
                    "device": {
                        "required": true
                    },
                    "global-network": {
                        "required": true
                    },
                    "link": {
                        "required": false
                    }
                },
                "conditions": [
                    "networkmanager:cgwArn"
                ]
            },
            "AssociateLink": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_AssociateLink.html",
                "description": "Grants permission to associate a link to a device",
                "accessLevel": "Write",
                "resourceTypes": {
                    "device": {
                        "required": true
                    },
                    "global-network": {
                        "required": true
                    },
                    "link": {
                        "required": true
                    }
                }
            },
            "CreateDevice": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateDevice.html",
                "description": "Grants permission to create a new device",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateGlobalNetwork": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateGlobalNetwork.html",
                "description": "Grants permission to create a new global network",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:CreateServiceLinkedRole"
                ],
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLink": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateLink.html",
                "description": "Grants permission to create a new link",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    },
                    "site": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSite": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateSite.html",
                "description": "Grants permission to create a new site",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteDevice": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteDevice.html",
                "description": "Grants permission to delete a device",
                "accessLevel": "Write",
                "resourceTypes": {
                    "device": {
                        "required": true
                    },
                    "global-network": {
                        "required": true
                    }
                }
            },
            "DeleteGlobalNetwork": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteGlobalNetwork.html",
                "description": "Grants permission to delete a global network",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    }
                }
            },
            "DeleteLink": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteLink.html",
                "description": "Grants permission to delete a link",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    },
                    "link": {
                        "required": true
                    }
                }
            },
            "DeleteSite": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteSite.html",
                "description": "Grants permission to delete a site",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    },
                    "site": {
                        "required": true
                    }
                }
            },
            "DeregisterTransitGateway": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeregisterTransitGateway.html",
                "description": "Grants permission to deregister a transit gateway from a global network",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    }
                },
                "conditions": [
                    "networkmanager:tgwArn"
                ]
            },
            "DescribeGlobalNetworks": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DescribeGlobalNetworks.html",
                "description": "Grants permission to describe global networks",
                "accessLevel": "List",
                "resourceTypes": {
                    "global-network": {
                        "required": false
                    }
                }
            },
            "DisassociateCustomerGateway": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DisassociateCustomerGateway.html",
                "description": "Grants permission to disassociate a customer gateway from a device",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    }
                },
                "conditions": [
                    "networkmanager:cgwArn"
                ]
            },
            "DisassociateLink": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DisassociateLink.html",
                "description": "Grants permission to disassociate a link from a device",
                "accessLevel": "Write",
                "resourceTypes": {
                    "device": {
                        "required": true
                    },
                    "global-network": {
                        "required": true
                    },
                    "link": {
                        "required": true
                    }
                }
            },
            "GetCustomerGatewayAssociations": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetCustomerGatewayAssociations.html",
                "description": "Grants permission to describe customer gateway associations",
                "accessLevel": "List",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    }
                }
            },
            "GetDevices": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetDevices.html",
                "description": "Grants permission to describe devices",
                "accessLevel": "List",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    },
                    "device": {
                        "required": false
                    }
                }
            },
            "GetLinkAssociations": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetLinkAssociations.html",
                "description": "Grants permission to describe link associations",
                "accessLevel": "List",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    },
                    "device": {
                        "required": false
                    },
                    "link": {
                        "required": false
                    }
                }
            },
            "GetLinks": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetLinks.html",
                "description": "Grants permission to describe links",
                "accessLevel": "List",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    },
                    "link": {
                        "required": false
                    }
                }
            },
            "GetSites": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetSites.html",
                "description": "Grants permission to describe global networks",
                "accessLevel": "List",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    },
                    "site": {
                        "required": false
                    }
                }
            },
            "GetTransitGatewayRegistrations": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetTransitGatewayRegistrations.html",
                "description": "Grants permission to describe transit gateway registrations",
                "accessLevel": "List",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to lists tag for a Network Manager resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "device": {
                        "required": false
                    },
                    "global-network": {
                        "required": false
                    },
                    "link": {
                        "required": false
                    },
                    "site": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "RegisterTransitGateway": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_RegisterTransitGateway.html",
                "description": "Grants permission to register a transit gateway to a global network",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    }
                },
                "conditions": [
                    "networkmanager:tgwArn"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to tag a Network Manager resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "device": {
                        "required": false
                    },
                    "global-network": {
                        "required": false
                    },
                    "link": {
                        "required": false
                    },
                    "site": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to untag a Network Manager resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "device": {
                        "required": false
                    },
                    "global-network": {
                        "required": false
                    },
                    "link": {
                        "required": false
                    },
                    "site": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDevice": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateDevice.html",
                "description": "Grants permission to update a device",
                "accessLevel": "Write",
                "resourceTypes": {
                    "device": {
                        "required": true
                    },
                    "global-network": {
                        "required": true
                    }
                }
            },
            "UpdateGlobalNetwork": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateGlobalNetwork.html",
                "description": "Grants permission to update a global network",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    }
                }
            },
            "UpdateLink": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateLink.html",
                "description": "Grants permission to update a link",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    },
                    "link": {
                        "required": true
                    }
                }
            },
            "UpdateSite": {
                "url": "https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateSite.html",
                "description": "Grants permission to update a site",
                "accessLevel": "Write",
                "resourceTypes": {
                    "global-network": {
                        "required": true
                    },
                    "site": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "global-network": {
                "name": "global-network",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html/",
                "arn": "arn:${Partition}:networkmanager::${Account}:global-network/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "site": {
                "name": "site",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html/",
                "arn": "arn:${Partition}:networkmanager::${Account}:site/${GlobalNetworkId}/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "link": {
                "name": "link",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html/",
                "arn": "arn:${Partition}:networkmanager::${Account}:link/${GlobalNetworkId}/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "device": {
                "name": "device",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html/",
                "arn": "arn:${Partition}:networkmanager::${Account}:device/${GlobalNetworkId}/${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to associate a customer gateway to a device
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCgwArn()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_AssociateCustomerGateway.html
     */
    associateCustomerGateway() {
        this.add('networkmanager:AssociateCustomerGateway');
        return this;
    }
    /**
     * Grants permission to associate a link to a device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_AssociateLink.html
     */
    associateLink() {
        this.add('networkmanager:AssociateLink');
        return this;
    }
    /**
     * Grants permission to create a new device
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateDevice.html
     */
    createDevice() {
        this.add('networkmanager:CreateDevice');
        return this;
    }
    /**
     * Grants permission to create a new global network
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateGlobalNetwork.html
     */
    createGlobalNetwork() {
        this.add('networkmanager:CreateGlobalNetwork');
        return this;
    }
    /**
     * Grants permission to create a new link
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateLink.html
     */
    createLink() {
        this.add('networkmanager:CreateLink');
        return this;
    }
    /**
     * Grants permission to create a new site
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_CreateSite.html
     */
    createSite() {
        this.add('networkmanager:CreateSite');
        return this;
    }
    /**
     * Grants permission to delete a device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteDevice.html
     */
    deleteDevice() {
        this.add('networkmanager:DeleteDevice');
        return this;
    }
    /**
     * Grants permission to delete a global network
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteGlobalNetwork.html
     */
    deleteGlobalNetwork() {
        this.add('networkmanager:DeleteGlobalNetwork');
        return this;
    }
    /**
     * Grants permission to delete a link
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteLink.html
     */
    deleteLink() {
        this.add('networkmanager:DeleteLink');
        return this;
    }
    /**
     * Grants permission to delete a site
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeleteSite.html
     */
    deleteSite() {
        this.add('networkmanager:DeleteSite');
        return this;
    }
    /**
     * Grants permission to deregister a transit gateway from a global network
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTgwArn()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DeregisterTransitGateway.html
     */
    deregisterTransitGateway() {
        this.add('networkmanager:DeregisterTransitGateway');
        return this;
    }
    /**
     * Grants permission to describe global networks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DescribeGlobalNetworks.html
     */
    describeGlobalNetworks() {
        this.add('networkmanager:DescribeGlobalNetworks');
        return this;
    }
    /**
     * Grants permission to disassociate a customer gateway from a device
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCgwArn()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DisassociateCustomerGateway.html
     */
    disassociateCustomerGateway() {
        this.add('networkmanager:DisassociateCustomerGateway');
        return this;
    }
    /**
     * Grants permission to disassociate a link from a device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_DisassociateLink.html
     */
    disassociateLink() {
        this.add('networkmanager:DisassociateLink');
        return this;
    }
    /**
     * Grants permission to describe customer gateway associations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetCustomerGatewayAssociations.html
     */
    getCustomerGatewayAssociations() {
        this.add('networkmanager:GetCustomerGatewayAssociations');
        return this;
    }
    /**
     * Grants permission to describe devices
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetDevices.html
     */
    getDevices() {
        this.add('networkmanager:GetDevices');
        return this;
    }
    /**
     * Grants permission to describe link associations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetLinkAssociations.html
     */
    getLinkAssociations() {
        this.add('networkmanager:GetLinkAssociations');
        return this;
    }
    /**
     * Grants permission to describe links
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetLinks.html
     */
    getLinks() {
        this.add('networkmanager:GetLinks');
        return this;
    }
    /**
     * Grants permission to describe global networks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetSites.html
     */
    getSites() {
        this.add('networkmanager:GetSites');
        return this;
    }
    /**
     * Grants permission to describe transit gateway registrations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_GetTransitGatewayRegistrations.html
     */
    getTransitGatewayRegistrations() {
        this.add('networkmanager:GetTransitGatewayRegistrations');
        return this;
    }
    /**
     * Grants permission to lists tag for a Network Manager resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('networkmanager:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to register a transit gateway to a global network
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTgwArn()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_RegisterTransitGateway.html
     */
    registerTransitGateway() {
        this.add('networkmanager:RegisterTransitGateway');
        return this;
    }
    /**
     * Grants permission to tag a Network Manager resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('networkmanager:TagResource');
        return this;
    }
    /**
     * Grants permission to untag a Network Manager resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('networkmanager:UntagResource');
        return this;
    }
    /**
     * Grants permission to update a device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateDevice.html
     */
    updateDevice() {
        this.add('networkmanager:UpdateDevice');
        return this;
    }
    /**
     * Grants permission to update a global network
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateGlobalNetwork.html
     */
    updateGlobalNetwork() {
        this.add('networkmanager:UpdateGlobalNetwork');
        return this;
    }
    /**
     * Grants permission to update a link
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateLink.html
     */
    updateLink() {
        this.add('networkmanager:UpdateLink');
        return this;
    }
    /**
     * Grants permission to update a site
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/networkmanager/latest/APIReference/API_UpdateSite.html
     */
    updateSite() {
        this.add('networkmanager:UpdateSite');
        return this;
    }
    /**
     * Adds a resource of type global-network to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html/
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGlobalNetwork(resourceId, account, partition) {
        var arn = 'arn:${Partition}:networkmanager::${Account}:global-network/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type site to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html/
     *
     * @param globalNetworkId - Identifier for the globalNetworkId.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSite(globalNetworkId, resourceId, account, partition) {
        var arn = 'arn:${Partition}:networkmanager::${Account}:site/${GlobalNetworkId}/${ResourceId}';
        arn = arn.replace('${GlobalNetworkId}', globalNetworkId);
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type link to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html/
     *
     * @param globalNetworkId - Identifier for the globalNetworkId.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLink(globalNetworkId, resourceId, account, partition) {
        var arn = 'arn:${Partition}:networkmanager::${Account}:link/${GlobalNetworkId}/${ResourceId}';
        arn = arn.replace('${GlobalNetworkId}', globalNetworkId);
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type device to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/what-is-network-manager.html/
     *
     * @param globalNetworkId - Identifier for the globalNetworkId.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDevice(globalNetworkId, resourceId, account, partition) {
        var arn = 'arn:${Partition}:networkmanager::${Account}:device/${GlobalNetworkId}/${ResourceId}';
        arn = arn.replace('${GlobalNetworkId}', globalNetworkId);
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .createDevice()
     * - .createGlobalNetwork()
     * - .createLink()
     * - .createSite()
     * - .tagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .listTagsForResource()
     * - .tagResource()
     *
     * Applies to resource types:
     * - global-network
     * - site
     * - link
     * - device
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .createDevice()
     * - .createGlobalNetwork()
     * - .createLink()
     * - .createSite()
     * - .tagResource()
     * - .untagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator || 'StringLike');
    }
    /**
     * Controls which customer gateways can be associated or disassociated
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/nm-security-iam.html
     *
     * Applies to actions:
     * - .associateCustomerGateway()
     * - .disassociateCustomerGateway()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCgwArn(value, operator) {
        return this.if(`networkmanager:cgwArn`, value, operator || 'StringLike');
    }
    /**
     * Controls which transit gateways can be registered or deregistered
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/nm-security-iam.html
     *
     * Applies to actions:
     * - .deregisterTransitGateway()
     * - .registerTransitGateway()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTgwArn(value, operator) {
        return this.if(`networkmanager:tgwArn`, value, operator || 'StringLike');
    }
}
exports.Networkmanager = Networkmanager;
//# sourceMappingURL=data:application/json;base64,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