"""Check if a host is in the Google Chrome HSTS Preload list"""

import functools
import os
import typing

__version__ = "2020.7.15"
__checksum__ = "9bb28e9ff612aee5978ff256975de734e59fe94ba68c614f0d433115c19d9bc3"
__all__ = ["in_hsts_preload"]

# fmt: off
_GTLD_INCLUDE_SUBDOMAINS = {b'android', b'app', b'bank', b'chrome', b'dev', b'foo', b'gle', b'gmail', b'google', b'hangout', b'insurance', b'meet', b'new', b'page', b'play', b'search', b'youtube'}  # noqa: E501
_JUMPTABLE = [[(0, 11), (11, 5), None, (16, 57), (73, 26), (99, 12), None, (111, 19), (130, 11), (141, 7), (148, 20), (168, 18), None, (186, 22), (208, 45), (253, 7), (260, 9), (269, 36), (305, 10), (315, 10), (325, 21), None, (346, 50), (396, 8), (404, 9), (413, 19), (432, 13), (445, 14), (459, 14), None, None, (473, 29), (502, 16), (518, 35), (553, 14), (567, 24), (591, 9), None, (600, 25), (625, 20), (645, 8), (653, 13), (666, 10), None, (676, 11), (687, 6), (693, 26), (719, 5), (724, 5), (729, 10), (739, 10), (749, 11), (760, 12), (772, 27), None, (799, 11), (810, 11), (821, 7), (828, 29), (857, 18), (875, 27), (902, 46), (948, 25), (973, 16), (989, 8), (997, 5), (1002, 22), (1024, 18), None, (1042, 36), (1078, 15), (1093, 8), (1101, 5), None, (1106, 5), (1111, 16), (1127, 14), (1141, 18), None, (1159, 14), (1173, 18), (1191, 48), (1239, 19), (1258, 5), (1263, 46), (1309, 14), (1323, 14), (1337, 20), None, (1357, 10), (1367, 13), (1380, 10), (1390, 19), None, (1409, 13), (1422, 19), (1441, 5), (1446, 4), (1450, 22), (1472, 10), (1482, 7), (1489, 14), (1503, 21), (1524, 11), (1535, 10), (1545, 12), (1557, 32), None, (1589, 10), (1599, 14), (1613, 12), (1625, 45), (1670, 15), None, (1685, 11), (1696, 23), (1719, 21), (1740, 26), (1766, 6), (1772, 6), (1778, 7), (1785, 5), (1790, 20), (1810, 23), (1833, 24), (1857, 13), (1870, 15), (1885, 19), (1904, 6), (1910, 61), (1971, 44), (2015, 12), (2027, 23), (2050, 16), (2066, 38), (2104, 6), (2110, 12), (2122, 44), (2166, 6), (2172, 41), (2213, 13), (2226, 23), (2249, 30), (2279, 16), (2295, 8), (2303, 6), (2309, 12), (2321, 19), (2340, 21), (2361, 15), None, (2376, 35), (2411, 21), (2432, 17), (2449, 19), (2468, 26), (2494, 5), (2499, 37), (2536, 30), (2566, 16), (2582, 10), (2592, 17), (2609, 23), (2632, 14), (2646, 17), (2663, 8), (2671, 4), (2675, 7), (2682, 29), (2711, 6), (2717, 18), (2735, 27), (2762, 20), (2782, 17), (2799, 19), (2818, 12), (2830, 40), (2870, 40), (2910, 12), (2922, 48), (2970, 25), (2995, 12), None, (3007, 8), (3015, 20), (3035, 19), (3054, 6), (3060, 23), None, (3083, 23), (3106, 33), (3139, 14), (3153, 12), (3165, 27), None, (3192, 26), (3218, 31), (3249, 50), (3299, 15), (3314, 20), (3334, 15), (3349, 21), (3370, 32), (3402, 24), (3426, 20), (3446, 13), (3459, 60), (3519, 19), (3538, 9), (3547, 12), (3559, 12), (3571, 11), (3582, 10), (3592, 48), (3640, 32), None, (3672, 25), (3697, 12), None, (3709, 8), (3717, 8), (3725, 7), None, (3732, 25), (3757, 17), None, (3774, 21), (3795, 35), (3830, 12), (3842, 10), (3852, 36), (3888, 20), (3908, 22), (3930, 23), (3953, 19), (3972, 12), (3984, 5), (3989, 30), (4019, 24), (4043, 14), (4057, 14), (4071, 47), (4118, 46), None, None, (4164, 51), (4215, 42), None, (4257, 14), None, (4271, 15), (4286, 8), (4294, 21), (4315, 6), (4321, 16), (4337, 17)], [(4354, 6037), (10391, 6485), (16876, 6911), (23787, 5772), (29559, 6134), (35693, 5824), (41517, 6824), (48341, 5968), (54309, 6607), (60916, 5939), (66855, 6938), (73793, 6221), (80014, 6584), (86598, 6955), (93553, 6434), (99987, 6297), (106284, 6926), (113210, 5854), (119064, 6168), (125232, 6466), (131698, 6722), (138420, 6445), (144865, 6754), (151619, 5955), (157574, 6072), (163646, 6543), (170189, 6426), (176615, 6698), (183313, 6160), (189473, 6395), (195868, 6712), (202580, 6304), (208884, 6284), (215168, 6893), (222061, 5989), (228050, 6689), (234739, 6177), (240916, 6865), (247781, 6579), (254360, 6876), (261236, 7284), (268520, 6233), (274753, 6206), (280959, 5992), (286951, 6214), (293165, 5971), (299136, 6149), (305285, 6895), (312180, 6186), (318366, 5590), (323956, 6365), (330321, 6456), (336777, 6397), (343174, 6627), (349801, 6670), (356471, 6547), (363018, 6671), (369689, 5700), (375389, 6680), (382069, 5674), (387743, 6436), (394179, 6294), (400473, 6212), (406685, 6688), (413373, 6492), (419865, 6428), (426293, 5960), (432253, 6839), (439092, 6551), (445643, 6688), (452331, 6281), (458612, 6328), (464940, 5609), (470549, 6879), (477428, 6888), (484316, 6712), (491028, 5977), (497005, 7049), (504054, 6885), (510939, 5959), (516898, 6548), (523446, 5649), (529095, 6242), (535337, 6457), (541794, 6355), (548149, 6274), (554423, 6459), (560882, 6570), (567452, 6564), (574016, 6312), (580328, 7067), (587395, 5938), (593333, 6064), (599397, 6478), (605875, 6372), (612247, 6910), (619157, 6698), (625855, 6253), (632108, 6140), (638248, 6026), (644274, 6099), (650373, 6671), (657044, 6117), (663161, 6350), (669511, 5990), (675501, 6696), (682197, 6478), (688675, 6833), (695508, 7860), (703368, 6963), (710331, 6753), (717084, 6332), (723416, 6139), (729555, 6576), (736131, 6785), (742916, 6341), (749257, 6133), (755390, 6276), (761666, 6266), (767932, 6734), (774666, 6630), (781296, 6637), (787933, 7005), (794938, 6698), (801636, 7665), (809301, 6292), (815593, 5654), (821247, 6766), (828013, 6499), (834512, 7831), (842343, 6797), (849140, 6087), (855227, 6595), (861822, 6739), (868561, 6237), (874798, 6631), (881429, 5900), (887329, 6654), (893983, 6281), (900264, 6400), (906664, 6381), (913045, 7118), (920163, 6091), (926254, 6143), (932397, 6448), (938845, 6370), (945215, 6330), (951545, 6714), (958259, 6048), (964307, 6942), (971249, 6490), (977739, 6611), (984350, 6692), (991042, 6378), (997420, 6377), (1003797, 6317), (1010114, 6235), (1016349, 6248), (1022597, 6081), (1028678, 5607), (1034285, 5989), (1040274, 6424), (1046698, 7081), (1053779, 5997), (1059776, 6423), (1066199, 6831), (1073030, 6225), (1079255, 6018), (1085273, 6643), (1091916, 6342), (1098258, 5808), (1104066, 6392), (1110458, 7431), (1117889, 5884), (1123773, 6052), (1129825, 6536), (1136361, 6130), (1142491, 6569), (1149060, 6222), (1155282, 5907), (1161189, 7230), (1168419, 6556), (1174975, 6231), (1181206, 6839), (1188045, 7151), (1195196, 7202), (1202398, 6092), (1208490, 6834), (1215324, 6157), (1221481, 6478), (1227959, 6614), (1234573, 5993), (1240566, 6721), (1247287, 6830), (1254117, 6453), (1260570, 6319), (1266889, 6168), (1273057, 6290), (1279347, 6613), (1285960, 6139), (1292099, 6486), (1298585, 5760), (1304345, 6990), (1311335, 6806), (1318141, 6530), (1324671, 6778), (1331449, 5596), (1337045, 6538), (1343583, 6326), (1349909, 6506), (1356415, 6659), (1363074, 6982), (1370056, 6423), (1376479, 6521), (1383000, 6602), (1389602, 6306), (1395908, 6308), (1402216, 6361), (1408577, 6453), (1415030, 6127), (1421157, 6327), (1427484, 5886), (1433370, 7293), (1440663, 6506), (1447169, 6233), (1453402, 6624), (1460026, 6488), (1466514, 5692), (1472206, 6548), (1478754, 6391), (1485145, 7471), (1492616, 6346), (1498962, 5718), (1504680, 6886), (1511566, 6299), (1517865, 6889), (1524754, 6012), (1530766, 6071), (1536837, 5759), (1542596, 6550), (1549146, 6365), (1555511, 6747), (1562258, 6087), (1568345, 6469), (1574814, 6260), (1581074, 6909), (1587983, 6270), (1594253, 5695), (1599948, 6422), (1606370, 6101), (1612471, 6509), (1618980, 6660), (1625640, 7024), (1632664, 6139), (1638803, 6135), (1644938, 6560)], [(1651498, 703), (1652201, 625), (1652826, 628), (1653454, 663), (1654117, 523), (1654640, 611), (1655251, 644), (1655895, 808), (1656703, 640), (1657343, 627), (1657970, 509), (1658479, 544), (1659023, 758), (1659781, 886), (1660667, 936), (1661603, 714), (1662317, 1207), (1663524, 589), (1664113, 846), (1664959, 640), (1665599, 733), (1666332, 708), (1667040, 802), (1667842, 709), (1668551, 684), (1669235, 631), (1669866, 940), (1670806, 1033), (1671839, 767), (1672606, 657), (1673263, 896), (1674159, 728), (1674887, 557), (1675444, 671), (1676115, 732), (1676847, 761), (1677608, 619), (1678227, 674), (1678901, 692), (1679593, 929), (1680522, 683), (1681205, 823), (1682028, 660), (1682688, 671), (1683359, 728), (1684087, 362), (1684449, 763), (1685212, 857), (1686069, 673), (1686742, 525), (1687267, 789), (1688056, 633), (1688689, 763), (1689452, 935), (1690387, 918), (1691305, 464), (1691769, 661), (1692430, 486), (1692916, 578), (1693494, 701), (1694195, 734), (1694929, 753), (1695682, 1008), (1696690, 836), (1697526, 602), (1698128, 692), (1698820, 753), (1699573, 444), (1700017, 561), (1700578, 487), (1701065, 692), (1701757, 786), (1702543, 540), (1703083, 725), (1703808, 634), (1704442, 671), (1705113, 552), (1705665, 680), (1706345, 768), (1707113, 428), (1707541, 672), (1708213, 607), (1708820, 828), (1709648, 623), (1710271, 588), (1710859, 282), (1711141, 597), (1711738, 703), (1712441, 756), (1713197, 663), (1713860, 816), (1714676, 1074), (1715750, 788), (1716538, 773), (1717311, 674), (1717985, 436), (1718421, 938), (1719359, 858), (1720217, 564), (1720781, 580), (1721361, 666), (1722027, 843), (1722870, 839), (1723709, 541), (1724250, 632), (1724882, 697), (1725579, 363), (1725942, 467), (1726409, 924), (1727333, 863), (1728196, 792), (1728988, 774), (1729762, 610), (1730372, 771), (1731143, 643), (1731786, 683), (1732469, 689), (1733158, 433), (1733591, 639), (1734230, 588), (1734818, 914), (1735732, 653), (1736385, 789), (1737174, 404), (1737578, 703), (1738281, 656), (1738937, 835), (1739772, 883), (1740655, 738), (1741393, 904), (1742297, 747), (1743044, 524), (1743568, 757), (1744325, 583), (1744908, 737), (1745645, 715), (1746360, 639), (1746999, 682), (1747681, 589), (1748270, 640), (1748910, 594), (1749504, 691), (1750195, 691), (1750886, 637), (1751523, 437), (1751960, 549), (1752509, 641), (1753150, 556), (1753706, 683), (1754389, 594), (1754983, 761), (1755744, 520), (1756264, 490), (1756754, 656), (1757410, 551), (1757961, 607), (1758568, 639), (1759207, 796), (1760003, 594), (1760597, 609), (1761206, 841), (1762047, 826), (1762873, 522), (1763395, 695), (1764090, 796), (1764886, 587), (1765473, 659), (1766132, 480), (1766612, 580), (1767192, 621), (1767813, 714), (1768527, 598), (1769125, 898), (1770023, 676), (1770699, 794), (1771493, 683), (1772176, 649), (1772825, 580), (1773405, 646), (1774051, 699), (1774750, 1277), (1776027, 514), (1776541, 641), (1777182, 608), (1777790, 944), (1778734, 754), (1779488, 729), (1780217, 546), (1780763, 565), (1781328, 808), (1782136, 553), (1782689, 569), (1783258, 826), (1784084, 650), (1784734, 871), (1785605, 772), (1786377, 668), (1787045, 671), (1787716, 783), (1788499, 620), (1789119, 889), (1790008, 631), (1790639, 722), (1791361, 558), (1791919, 689), (1792608, 449), (1793057, 753), (1793810, 779), (1794589, 648), (1795237, 901), (1796138, 775), (1796913, 759), (1797672, 903), (1798575, 1077), (1799652, 815), (1800467, 586), (1801053, 839), (1801892, 662), (1802554, 483), (1803037, 443), (1803480, 692), (1804172, 749), (1804921, 406), (1805327, 974), (1806301, 462), (1806763, 758), (1807521, 844), (1808365, 712), (1809077, 769), (1809846, 626), (1810472, 743), (1811215, 665), (1811880, 753), (1812633, 566), (1813199, 566), (1813765, 408), (1814173, 601), (1814774, 437), (1815211, 740), (1815951, 815), (1816766, 764), (1817530, 718), (1818248, 621), (1818869, 568), (1819437, 808), (1820245, 422), (1820667, 573), (1821240, 780), (1822020, 513), (1822533, 839), (1823372, 2071), (1825443, 507), (1825950, 602), (1826552, 875), (1827427, 833), (1828260, 510)], [(1828770, 48), None, (1828818, 35), (1828853, 42), None, None, None, None, None, None, None, None, None, None, None, None, None, (1828895, 42), None, (1828937, 25), (1828962, 16), None, None, None, None, None, None, (1828978, 26), None, None, None, None, (1829004, 21), (1829025, 25), None, None, (1829050, 26), None, None, None, None, (1829076, 44), (1829120, 21), (1829141, 23), None, None, None, None, (1829164, 48), None, None, None, None, None, (1829212, 31), None, None, None, None, (1829243, 42), None, (1829285, 22), None, (1829307, 21), None, (1829328, 26), (1829354, 42), None, None, (1829396, 77), None, None, None, None, None, (1829473, 21), (1829494, 21), None, None, (1829515, 34), (1829549, 42), None, None, None, (1829591, 25), None, None, (1829616, 21), None, None, None, None, None, (1829637, 24), (1829661, 21), None, None, (1829682, 26), None, (1829708, 18), None, (1829726, 54), None, None, None, None, None, None, (1829780, 26), None, (1829806, 19), None, (1829825, 20), None, None, (1829845, 42), (1829887, 42), (1829929, 17), None, (1829946, 26), None, (1829972, 26), None, None, None, (1829998, 26), (1830024, 20), (1830044, 26), None, (1830070, 42), (1830112, 63), None, None, None, (1830175, 40), (1830215, 48), None, None, None, (1830263, 47), None, None, None, None, None, None, None, (1830310, 42), None, (1830352, 55), None, (1830407, 9), None, (1830416, 21), (1830437, 42), None, None, (1830479, 42), (1830521, 82), None, None, (1830603, 42), None, None, None, None, None, None, None, None, None, (1830645, 42), (1830687, 21), (1830708, 21), None, (1830729, 42), (1830771, 25), None, None, (1830796, 21), (1830817, 42), None, None, (1830859, 21), (1830880, 19), (1830899, 26), None, None, None, (1830925, 21), None, None, (1830946, 38), None, (1830984, 22), (1831006, 21), (1831027, 21), None, None, (1831048, 63), None, (1831111, 21), (1831132, 42), None, (1831174, 17), None, None, None, None, (1831191, 21), (1831212, 21), None, None, (1831233, 21), None, None, (1831254, 21), None, (1831275, 26), None, (1831301, 50), None, None, None, (1831351, 50), (1831401, 26), (1831427, 21), (1831448, 21), (1831469, 19), None, (1831488, 35), (1831523, 26), (1831549, 23), (1831572, 21), (1831593, 42), None, None, None, None, None, None, (1831635, 21), None, None, None, (1831656, 21), None, None, (1831677, 90), None, (1831767, 239), (1832006, 38), None, None, None, None]]  # noqa: E501
_CRC8_TABLE = [
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
]
# fmt: on

_IS_LEAF = 0x80
_INCLUDE_SUBDOMAINS = 0x40


try:
    from importlib.resources import open_binary

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open_binary("hstspreload", path)


except ImportError:

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open(
            os.path.join(os.path.dirname(os.path.abspath(__file__)), path), "rb",
        )


@functools.lru_cache(maxsize=1024)
def in_hsts_preload(host: typing.AnyStr) -> bool:
    """Determines if an IDNA-encoded host is on the HSTS preload list"""

    if isinstance(host, str):
        host = host.encode("ascii")
    labels = host.lower().split(b".")

    # Fast-branch for gTLDs that are registered to preload all sub-domains.
    if labels[-1] in _GTLD_INCLUDE_SUBDOMAINS:
        return True

    with open_pkg_binary("hstspreload.bin") as f:
        for layer, label in enumerate(labels[::-1]):
            # None of our layers are greater than 4 deep.
            if layer > 3:
                return False

            # Read the jump table for the layer and label
            jump_info = _JUMPTABLE[layer][_crc8(label)]
            if jump_info is None:
                # No entry: host is not preloaded
                return False

            # Read the set of entries for that layer and label
            f.seek(jump_info[0])
            data = bytearray(jump_info[1])
            f.readinto(data)

            for is_leaf, include_subdomains, ent_label in _iter_entries(data):
                # We found a potential leaf
                if is_leaf:
                    if ent_label == host:
                        return True
                    if include_subdomains and host.endswith(b"." + ent_label):
                        return True

                # Continue traversing as we're not at a leaf.
                elif label == ent_label:
                    break
            else:
                return False
    return False


def _iter_entries(data: bytes) -> typing.Iterable[typing.Tuple[int, int, bytes]]:
    while data:
        flags = data[0]
        size = data[1]
        label = bytes(data[2 : 2 + size])
        yield (flags & _IS_LEAF, flags & _INCLUDE_SUBDOMAINS, label)
        data = data[2 + size :]


def _crc8(value: bytes) -> int:
    # CRC8 reference implementation: https://github.com/niccokunzmann/crc8
    checksum = 0x00
    for byte in value:
        checksum = _CRC8_TABLE[checksum ^ byte]
    return checksum
