"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const apigateway = require("../lib");
const RESOURCE_TYPE = 'AWS::ApiGateway::UsagePlan';
module.exports = {
    'default setup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'my-api', { cloudWatchRole: false, deploy: false });
        api.root.addMethod('GET'); // Need at least one method on the api
        const usagePlanName = 'Pro';
        const usagePlanDescription = 'Pro Usage Plan with no throttling limits';
        // WHEN
        new apigateway.UsagePlan(stack, 'my-usage-plan', {
            name: usagePlanName,
            description: usagePlanDescription
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(RESOURCE_TYPE, {
            UsagePlanName: usagePlanName,
            Description: usagePlanDescription
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'usage plan with throttling limits'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'my-api', { cloudWatchRole: false, deploy: true, deployOptions: { stageName: 'test' } });
        const method = api.root.addMethod('GET'); // Need at least one method on the api
        const usagePlanName = 'Basic';
        const usagePlanDescription = 'Basic Usage Plan with no throttling limits';
        // WHEN
        new apigateway.UsagePlan(stack, 'my-usage-plan', {
            name: usagePlanName,
            description: usagePlanDescription,
            apiStages: [
                {
                    stage: api.deploymentStage,
                    throttle: [
                        {
                            method,
                            throttle: {
                                burstLimit: 20,
                                rateLimit: 10
                            }
                        }
                    ]
                }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(RESOURCE_TYPE, {
            UsagePlanName: usagePlanName,
            Description: usagePlanDescription,
            ApiStages: [
                {
                    ApiId: {
                        Ref: 'myapi4C7BF186'
                    },
                    Stage: {
                        Ref: 'myapiDeploymentStagetest4A4AB65E'
                    },
                    Throttle: {
                        '//GET': {
                            BurstLimit: 20,
                            RateLimit: 10
                        }
                    }
                }
            ]
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'usage plan with quota limits'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new apigateway.UsagePlan(stack, 'my-usage-plan', {
            quota: {
                limit: 10000,
                period: apigateway.Period.MONTH
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(RESOURCE_TYPE, {
            Quota: {
                Limit: 10000,
                Period: 'MONTH'
            }
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'UsagePlanKey'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const usagePlan = new apigateway.UsagePlan(stack, 'my-usage-plan', {
            name: 'Basic',
        });
        const apiKey = new apigateway.ApiKey(stack, 'my-api-key');
        // WHEN
        usagePlan.addApiKey(apiKey);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::UsagePlanKey', {
            KeyId: {
                Ref: 'myapikey1B052F70'
            },
            KeyType: 'API_KEY',
            UsagePlanId: {
                Ref: 'myusageplan23AA1E32'
            }
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'UsagePlan can have multiple keys'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const usagePlan = new apigateway.UsagePlan(stack, 'my-usage-plan');
        const apiKey1 = new apigateway.ApiKey(stack, 'my-api-key-1', {
            apiKeyName: 'my-api-key-1'
        });
        const apiKey2 = new apigateway.ApiKey(stack, 'my-api-key-2', {
            apiKeyName: 'my-api-key-2'
        });
        // WHEN
        usagePlan.addApiKey(apiKey1);
        usagePlan.addApiKey(apiKey2);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::ApiKey', {
            Name: 'my-api-key-1'
        }, assert_1.ResourcePart.Properties));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::ApiKey', {
            Name: 'my-api-key-2'
        }, assert_1.ResourcePart.Properties));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::UsagePlanKey', {
            KeyId: {
                Ref: 'myapikey11F723FC7'
            }
        }, assert_1.ResourcePart.Properties));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::UsagePlanKey', {
            KeyId: {
                Ref: 'myapikey2ABDEF012'
            }
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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