# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['splunk_otel', 'splunk_otel.cmd']

package_data = \
{'': ['*']}

install_requires = \
['opentelemetry-api>=1.2.0,<2.0.0',
 'opentelemetry-instrumentation==0.21b0',
 'opentelemetry-sdk>=1.2.0,<2.0.0']

extras_require = \
{'all': ['opentelemetry-propagator-b3>=1.2.0,<2.0.0',
         'opentelemetry-exporter-jaeger-thrift==1.2.0',
         'opentelemetry-exporter-otlp-proto-grpc==1.2.0'],
 'b3': ['opentelemetry-propagator-b3>=1.2.0,<2.0.0'],
 'jaeger': ['opentelemetry-exporter-jaeger-thrift==1.2.0'],
 'otlp': ['opentelemetry-exporter-otlp-proto-grpc==1.2.0']}

entry_points = \
{'console_scripts': ['splk-py-trace = splunk_otel.cmd.trace:run',
                     'splk-py-trace-bootstrap = splunk_otel.cmd.bootstrap:run'],
 'opentelemetry_distro': ['splunk_distro = splunk_otel.distro:SplunkDistro']}

setup_kwargs = {
    'name': 'splunk-opentelemetry',
    'version': '0.13.0',
    'description': 'The Splunk distribution of OpenTelemetry Python Instrumentation provides a Python agent that automatically instruments your Python application to capture and report distributed traces to SignalFx APM.',
    'long_description': '\n# Splunk distribution of OpenTelemetry Python\n\n[![OpenTelemetry Python Version](https://img.shields.io/badge/otel-1.2.0-blueviolet?style=for-the-badge)](https://github.com/open-telemetry/opentelemetry-python/releases/tag/v1.2.0)\n[![GitHub release (latest SemVer)](https://img.shields.io/github/v/release/signalfx/splunk-otel-python?style=for-the-badge)](https://github.com/signalfx/splunk-otel-python/releases)\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/splunk-opentelemetry?style=for-the-badge)](https://pypi.org/project/splunk-opentelemetry/)\n[![CircleCI](https://img.shields.io/circleci/build/github/signalfx/splunk-otel-python/main?style=for-the-badge)](https://circleci.com/gh/signalfx/splunk-otel-python)\n[![Codecov](https://img.shields.io/codecov/c/github/signalfx/splunk-otel-python?style=for-the-badge&token=XKXjEQKGaK)](https://codecov.io/gh/signalfx/splunk-otel-python)\n\n-------\n\nThe documentation below refers to the in development version of this package. Docs for the latest version ([v0.13.0](https://github.com/signalfx/splunk-otel-python/releases/tag/v0.13.0)) can be found [here](https://github.com/signalfx/splunk-otel-python/blob/v0.13.0/README.md).\n\n-------\n\nThe Splunk distribution of [OpenTelemetry\nPython](https://github.com/open-telemetry/opentelemetry-python) provides\nmultiple installable packages that automatically instruments your Python\napplication to capture and report distributed traces to Splunk APM.\n\nThis Splunk distribution comes with the following defaults:\n\n- [W3C tracecontext](https://www.w3.org/TR/trace-context/) and [W3C baggage](https://www.w3.org/TR/baggage/) context propagation.\n- [OTLP gRPC exporter](https://opentelemetry-python.readthedocs.io/en/latest/exporter/otlp/otlp.html)\n  configured to send spans to a locally running [OpenTelemetry Collector](https://docs.signalfx.com/en/latest/apm/apm-getting-started/apm-opentelemetry-collector.html)\n  (`http://localhost:4317`).\n- Unlimited default limits for [configuration options](#trace-configuration) to\n  support full-fidelity traces.\n\nIf you\'re currently using the SignalFx Tracing Library for Python and want to\nmigrate to the Splunk Distribution of OpenTelemetry Python, see [Migrate from\nthe SignalFx Tracing Library for Python](migration.md).\n\n> :construction: This project is currently in **BETA**. It is **officially supported** by Splunk. However, breaking changes **MAY** be introduced.\n\n## Requirements\n\nThis Splunk Distribution of OpenTelemetry requires Python 3.6 or later.\nIf you\'re still using Python 2, continue using the SignalFx Tracing Library\nfor Python.\n\n## Getting Started\n\nThe instrumentation works with Python version 3.6+. Supported libraries are\nlisted\n[here](https://github.com/open-telemetry/opentelemetry-python-contrib/tree/master/instrumentation).\n\nTo get started, install the `splunk-opentelemetry[all]` package, run the bootstrap\nscript and wrap your run command with `splk-py-trace`.\n\nFor example, if the runtime parameters were:\n\n```\npython main.py --port=8000\n```\n\nThen the runtime parameters should be updated to:\n\n```\n$ pip install splunk-opentelemetry[all]\n$ splk-py-trace-bootstrap\n$ OTEL_SERVICE_NAME=my-python-app \\\n    splk-py-trace python main.py --port=8000\n```\n\n\nThe service name is the only configuration option that typically needs to be\nspecified. A couple other configuration options that may need to be changed or\nset are:\n\n- Endpoint if not sending to a locally running OpenTelemetry Collector.\n- Environment attribute (example:\n  `OTEL_RESOURCE_ATTRIBUTES=deployment.environment=production`) to specify what\n  environment the span originated from.\n\nInstrumentation works by patching supported libraries at runtime with an\nOpenTelemetry-compatible tracer to capture and export trace spans. \n\nTo see the Python instrumentation in action with sample applications, see our\n[examples](https://github.com/signalfx/tracing-examples/tree/main/opentelemetry-tracing/opentelemetry-python-tracing).\n\n## All configuration options\n\n| Environment variable          | Config Option                        | Default value                        | Notes                                                                  |\n| ----------------------------- | ------------------------------------ | ------------------------------------ | ---------------------------------------------------------------------- |\n| OTEL_SERVICE_NAME             | service_name                         | `unnamed-python-service`     | The service name of this Python application.   |\n| OTEL_TRACES_EXPORTER | exporter_factories | `otlp`     | The exporter(s) that should be used to export tracing data. |\n| OTEL_EXPORTER_OTLP_ENDPOINT |  | `http://localhost:4317`     | The OTLP gRPC endpoint to connect to. Used when `OTEL_TRACES_EXPORTER` is set to `otlp` |\n| OTEL_EXPORTER_JAEGER_ENDPOINT |  | `http://localhost:9080/v1/trace`     | The Jaeger Thrift endpoint to connect to. Used when `OTEL_TRACES_EXPORTER` is set to `jaeger-thrift-splunk` |\n| SPLUNK_ACCESS_TOKEN          | access_token |      | The optional organization access token for trace submission requests.  |\n| SPLUNK_MAX_ATTR_LENGTH       | max_attr_length | `1200`            | Maximum length of string attribute value in characters. Longer values are truncated.                                                                                                                                                                                                                                                                                                                      |\n| SPLUNK_TRACE_RESPONSE_HEADER_ENABLED | trace_response_header_enabled | True | Enables adding server trace information to HTTP response headers. |\n| OTEL_RESOURCE_ATTRIBUTES      | resource_attributes | unset          | Comma-separated list of resource attributes added to every reported span. <details><summary>Example</summary>`service.name=my-python-service,service.version=3.1,deployment.environment=production`</details>\n| OTEL_PROPAGATORS              |            | `tracecontext,baggage`   | Comma-separated list of propagator names to be used. See[Configuring Propagators](#configuring-propagators) for more details.\n| OTEL_TRACE_ENABLED            |            | `true`         | Globally enables tracer creation and auto-instrumentation. |\n\n\n## Exporting telemetry data\n\nThis package can export spans in the OTLP format over gRPRC or Jaeger Thrift format over HTTP.\nThis allows you to export data to wide range of destinations such as OpenTelemetry Collector,\nSignalFx Smart Agent or even Splunk APM ingest. \n\n### To OpenTelemetry Collector\n\nThis is the default option. You do not need to set any config options if you want to exporter\nto the OpenTelemetry collector, the collector has OTLP gRPC receiver enabled with default settings\nand can be reached by `localhost` as by default everything by be exported to `http://localhost:4317`\nin OTLP over gRPC.\n\nIf your collector cannot be reached at `http://localhost:4317`, you\'ll need to set the `OTEL_EXPORTER_OTLP_ENDPOINT`\nto `http://<otel-collector-address>:<port>`. Replace `<otel-collector-address>` and `<port>` with the address and\nport of your OpenTelemetry Collector deployment.\n\nNote: You\'ll make sure that the OTLP gRPC exporter is installed. This can be done by running `pip install splunk-opentelemetry[all]`\nor `splunk-opentelemetry[otlp]`.\n\n### To SignalFx Smart Agent\n\n1. Set `OTEL_TRACES_EXPORTER` environment variable to `jaeger-thrift-splunk`.\n   If you are running the SignalFx Smart Agent locally (reachable via `localhost`) and it is listening \non the default port (`9080`), you do not need to perform any additional steps. Otherwise, follow the next step. \n2. Set the `OTEL_EXPORTER_JAEGER_ENDPOINT` environment variable to `http://<address>:<port>/v1/trace`. Replace `<address>` and `<port>`\nwith the address and port of your Smart Agent deployment.\n\nNote: You\'ll make sure that the Jaeger Thrift exporter is installed. This can be done by running `pip install splunk-opentelemetry[all]`\nor `splunk-opentelemetry[jaeger]`.\n\n### To Splunk Observability Cloud\n\nIn order to send traces directly to SignalFx ingest API, you need to:\n\n1. Set `OTEL_TRACES_EXPORTER` to `jaeger-thrift-splunk`.\n2. Set `OTEL_EXPORTER_JAEGER_ENDPOINT` to\n   `https://ingest.<realm>.signalfx.com/v2/trace` where `realm` is your\n   SignalFx realm e.g, `https://ingest.us0.signalfx.com/v2/trace`.\n3. Set `SPLUNK_ACCESS_TOKEN` to one of your Splunk APM access tokens.\n\nNote: You\'ll make sure that the Jaeger Thrift exporter is installed. This can be done by running `pip install splunk-opentelemetry[all]`\nor `splunk-opentelemetry[jaeger]`.\n\n## Configuring Propagators <a name="configuring-propagators"></a>\n\nThis package uses W3C trace context and W3C baggage propagators by default. You can override\nthis by setting the `OTEL_PROPAGATORS` environment variable to a comma separated list of one\nmore propagators. The SDK will use Python\'s entry points mechanism to load the specified\npropagator implementation(s) and use it.\n\nFor example, to only use W3C trace context without baggage, you can set the environment variable\n`OTEL_PROPAGATORS` environment variable to `tracecontext`.\n\nYou can specify any propagator name as long as the propagator implementation can be found via\nentry points by that name.\n\n### Configuring B3 propagator\n\n\nIf you\'d like to use `b3` instead of or in addition to the default propagators, you can set `OTEL_PROPAGATORS` to `b3`\nfor [B3 single header](https://github.com/openzipkin/b3-propagation#single-header) or `b3multi` for\n[B3 multi header](https://github.com/openzipkin/b3-propagation#multiple-headers) implementation. For example, to configure\nyour service to use B3 multi header and W3C baggage, set the environment variable as\n\n```\nOTEL_PROPAGATORS=b3multi,baggage\n```\n\nYou can specify any combination of supported propagators. Choices are `tracecontext`, `baggae`, `b3` and `b3multi`. Note that\n`b3` and `b3multi` are only available when the `opentelemetry-propagator-b3` package is installed. This is installed automatically\nby installing `splunk-opentelemetry[all]` or `splunk-opentelemetry[b3]`.\n\n\n## Advanced Getting Started\n\n### Instrument and configure with code\n\nIf you cannot use `splk-py-trace` command, you can also add a couple of lines\nof code to your Python application to achieve the same result.\n\n```python\nfrom splunk_otel.tracing import start_tracing\n\nstart_tracing()\n\n# Also accepts optional config options:\n# start_tracing(\n#   service_name=\'my-python-service\',\n#   exporter_factories=[OTLPSpanExporter]\n#   access_token=\'\',\n#   max_attr_length=1200,\n#   trace_response_header_enabled=True,\n#   resource_attributes={\n#    \'service.version\': \'3.1\',\n#    \'deployment.environment\': \'production\',\n#  })\n\n# rest of your python application\'s entrypoint script\n```\n\n### Using a different exporter\n\nThe `splunk-opentelemetry` Python package does not install any exporters by default. You can install it with the OTLP or Jaeger Thrift exporter by\nusing the `otlp` or `jaeger` extra options. For example, installing `splunk-opentelemetry[otlp]` will also pull in the OTLP gRPC exporter. Similarly,\ninstalling `splunk-opentelemetry[jaeger]` will install the Jaeger Thrift exporter. You can also install both exporters by mentioning them\nboth like `splunk-opentelemetry[jaeger,otlp]`\n\nThe distributions uses OTLP by default so we recommend installing `splunk-opentelemetry[otlp]` unless you want to use another exporter.\n\nOnce you install the exporter package you want to use, you can tell the distribution to use a different exporter by setting the `OTEL_TRACES_EXPORTER`\nenvironment variables.\n\nFor example, to use the Jaeger exporter, set it as follows:\n\n```\nOTEL_TRACES_EXPORTER=jaeger-thrift-splunk\n```\n\n#### Using multiple exporters\n\nThe environment variable accepts multiple comma-separated values. If multiple exporters are specified, all of them will be used. This can be used to export\nto multiple destinations or to debug with the console exporter while still exporting to another destination. For example, the following configuration will\nexport all spans using both the OTLP exporter and the Console exporter.\n\n```\nOTEL_TRACES_EXPORTER=otlp,console_span\n```\n\n#### Accepted values for OTEL_TRACES_EXPORTER\n\nThis package uses Python\'s entry points mechanism to look up the requested exporters. As a result, you can install any thrid party or custom exporter package and\nas long as it specifies a `opentelemetry_exporter` entry point to the exporter implementation, you can specify it as a value in `OTEL_TRACES_EXPORTER`.\n\nKnown values and the Python packages they ship in are listed below\n\n\n| Exporter name | Python package | Additional comments |\n| ------------- | --------------- | --------------------- | \n| otlp | opentelemetry-exporter-otlp-proto-grpc | Can be installed with `pip install splunk-opentelemetry[otlp]` | \n| jaeger-thrift-splunk | opentelemetry-exporter-jaeger-thrift  | Can be installed with `pip install splunk-opentelemetry[jaeger]` | \n| console_span | opentelemetry-sdk | Always installed with `splunk-opentelemetry` | \n\n\n### Bootstrap: List requirements instead of installing them\n\nThe `splk-py-trace-bootstrap` command can optionally print out the list of\npackages it would install if you chose. In order to do so, pass\n`-a=requirements` CLI argument to it. For example,\n\n```\nsplk-py-trace-bootstrap -a requirements\n```\n\nWill output something like the following:\n\n```\nopentelemetry-instrumentation-falcon>=0.15b0\nopentelemetry-instrumentation-jinja2>=0.15b0\nopentelemetry-instrumentation-requests>=0.15b0\nopentelemetry-instrumentation-sqlite3>=0.15b0\n```\n\nYou can pipe the output of this command to append the new packages to your\nrequirements.txt file or to something like `poetry add`.\n\n\n### Installing only a subset of dependencies\n\nInstalling `splunk-opentelemetry[all]` automatically pulls in all of the optional dependencies. These include the OTLP gRPC exporter, the Jaeger Thrift exporter\nand the B3 propagator. If you\'d like to install only the packages you need, you can use any combination of `oltp`, `jaeger` and `b3`. For example, in order\nto install only `otlp` exporter, you can run\n\n```\npip install splunk-opentelemetry[otlp]\n```\n\nTo install the Jaeger Thrift exporter and the B3 propagator, you can run\n\n```\npip install splunk-opentelemetry[jaeger,b3]\n```\n\n## Special Cases\n\n### Celery\n\nTracing Celery workers works out of the box when you use the `splk-py-trace`\ncommand to start your Python application. However, if you are instrumenting\nyour celery workers with code, you\'ll need to make sure you setup tracing for\neach worker by using Celery\'s `celery.signals.worker_process_init` signal.\n\nFor example:\n\n```python\nfrom splunk_otel.tracing import start_tracing\nfrom celery.signals import worker_process_init\n\n@worker_process_init.connect(weak=False)\ndef on_worker_process_init(*args, **kwargs):\n    start_tracing()\n\n# rest of your python application\'s entrypoint script\n```\n\n### Django\n\nAutomatically instrumenting Django requires `DJANGO_SETTINGS_MODULE`\nenvironment variable to be set. The value should be the same as set in your\n`manage.py` or `wsgi.py` modules. For example, if your manage.py file sets this\nenvironment variable to `mydjangoproject.settings` and you start your Django\nproject as `./manage.py runserver`, then you can automatically instrument your\nDjango project as follows:\n\n```\nexport DJANGO_SETTINGS_MODULE=mydjangoproject.settings\nsplk-py-trace ./manage.py runserver\n```\n\n### Gunicorn\n\nLike Celery, we\'ll also need to setup tracing per Gunicorn worker. This can be\ndone by setting up tracing inside Gunicorn\'s `post_fork()` handler.\n\nFor example:\n\n```python\n# gunicorn.config.py\nfrom splunk_otel.tracing import start_tracing\n\ndef post_fork(server, worker):\n    start_tracing()\n```\n\nThen add `-c gunicorn.config.py` CLI flag to your gunicorn command.\n\n### UWSGI \n\nWhen using UWSGI, tracing must be setup as a response to the `post_fork` signal. \n\nFor example:\n\n```python\nimport uwsgidecorators\nfrom splunk_otel.tracing import start_tracing\n\n@uwsgidecorators.postfork\ndef setup_tracing():\n    start_tracing()\n\n```\n\n##### Running with uwsgi\n```\nuwsgi --http :9090 --wsgi-file <your_app.py> --callable <your_wsgi_callable> --master --enable-threads\n```\n\nThe above snippet should be placed in the main python script that uwsgi imports and loads.\n\n#### UWSGI and Flask\n\nUsing USWGI with Flask requires one additional little step. Calling `start_tracing()` does not auto-instrument pre-existing flask app instances but only flask instances created after. When running flask with uwsgi, we need to create a new flask app instance before the post_fork signal is emitted. This means your flask app will not be auto-instrumented. However, you can still auto-instrument an existing flask app explicitly by importing and calling the flask instrumentor. \n\nFor example:\n```python\n# app.py\nimport uwsgidecorators\nfrom splunk_otel.tracing import start_tracing\nfrom opentelemetry.instrumentation.flask import FlaskInstrumentor\nfrom flask import Flask\n\napp = Flask(__name__)\n\n@uwsgidecorators.postfork\ndef setup_tracing():\n    start_tracing()\n    # instrument our flask app instance eplicitly\n    FlaskInstrumentor().instrument_app(app)\n\n@app.route(\'/\')\ndef hello_world():\n    return \'Hello, World!\'\n```\n\n##### Running with uWSGI:\n```\nuwsgi --http :9090 --wsgi-file app.py --callable app --master --enable-threads\n```\n\n\n## Manually instrument an application\n\nDocumentation on how to manually instrument a Python application is available\n[here](https://opentelemetry-python.readthedocs.io/en/stable/getting-started.html).\n\n## Troubleshooting\n\n- Depending on the default python version on your system, you might want to use `pip3` and `python3` instead. \n- To be able to run `splk-py-trace` and `splk-py-trace-bootstrap`, the directory pip installs scripts to will\n  have to be on your system\'s PATH environment variable. Generally, this works out of the box when using\n  virtual environments, installing packages system-wide or in container images. In some cases, pip may install\n  packages into your user local environment. In that case you\'ll need to add your Python user base\'s bin directory\n  to your system path. You can find out your Python user base as follows by running `python -m site --user-base`.\n\n  For example, if `python -m site --user-base` reports that `/Users/splunk/.local` as the Python user base, then\n  you can add the directory to your path on Unix like system as follows:\n\n  ```\n  export PATH="/Users/splunk/.local/bin:$PATH"\n  ```\n- Enable debug logging like you would for any Python application.\n\n  ```python\n  import logging\n\n  logging.basicConfig(level=logging.DEBUG)\n  ```\n\n  > :warning: Debug logging is extremely verbose and resource intensive. Enable\n  > debug logging only when needed and disable when done.\n\n# License and versioning\n\nThe Splunk distribution of OpenTelemetry Python Instrumentation is a\ndistribution of the [OpenTelemetry Python\nproject](https://github.com/open-telemetry/opentelemetry-python). It is\nreleased under the terms of the Apache Software License version 2.0. See [the\nlicense file](./LICENSE) for more details.\n',
    'author': 'Splunk',
    'author_email': 'splunk-oss@splunk.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
