"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const cdk = require("@aws-cdk/core");
const sns = require("../lib");
module.exports = {
    'create a subscription'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription', {
            Endpoint: 'endpoint',
            Protocol: 'lambda',
            TopicArn: {
                Ref: 'TopicBFC7AF6E',
            },
        }));
        test.done();
    },
    'create a subscription with DLQ when client provides DLQ'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const dlQueue = new aws_sqs_1.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MySubscription_DLQ',
            retentionPeriod: cdk.Duration.days(14),
        });
        // WHEN
        new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
            deadLetterQueue: dlQueue,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription', {
            Endpoint: 'endpoint',
            Protocol: 'lambda',
            TopicArn: {
                Ref: 'TopicBFC7AF6E',
            },
            RedrivePolicy: {
                deadLetterTargetArn: {
                    'Fn::GetAtt': [
                        'DeadLetterQueue9F481546',
                        'Arn',
                    ],
                },
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::Queue', {
            QueueName: 'MySubscription_DLQ',
            MessageRetentionPeriod: 1209600,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::QueuePolicy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'sqs:SendMessage',
                        Condition: {
                            ArnEquals: {
                                'aws:SourceArn': {
                                    Ref: 'TopicBFC7AF6E',
                                },
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'sns.amazonaws.com',
                        },
                        Resource: {
                            'Fn::GetAtt': [
                                'DeadLetterQueue9F481546',
                                'Arn',
                            ],
                        },
                    },
                ],
                Version: '2012-10-17',
            },
            Queues: [
                {
                    Ref: 'DeadLetterQueue9F481546',
                },
            ],
        }));
        test.done();
    },
    'with filter policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            filterPolicy: {
                color: sns.SubscriptionFilter.stringFilter({
                    whitelist: ['red', 'green'],
                    blacklist: ['white', 'orange'],
                    matchPrefixes: ['bl', 'ye'],
                }),
                price: sns.SubscriptionFilter.numericFilter({
                    whitelist: [100, 200],
                    between: { start: 300, stop: 350 },
                    greaterThan: 500,
                    lessThan: 1000,
                    betweenStrict: { start: 2000, stop: 3000 },
                    greaterThanOrEqualTo: 1000,
                    lessThanOrEqualTo: -2,
                }),
            },
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription', {
            FilterPolicy: {
                color: [
                    'red',
                    'green',
                    { 'anything-but': ['white', 'orange'] },
                    { prefix: 'bl' },
                    { prefix: 'ye' },
                ],
                price: [
                    { numeric: ['=', 100] },
                    { numeric: ['=', 200] },
                    { numeric: ['>', 500] },
                    { numeric: ['>=', 1000] },
                    { numeric: ['<', 1000] },
                    { numeric: ['<=', -2] },
                    { numeric: ['>=', 300, '<=', 350] },
                    { numeric: ['>', 2000, '<', 3000] },
                ],
            },
        }));
        test.done();
    },
    'with existsFilter'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            filterPolicy: {
                size: sns.SubscriptionFilter.existsFilter(),
            },
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription', {
            FilterPolicy: {
                size: [{ exists: true }],
            },
        }));
        test.done();
    },
    'throws with raw delivery for protocol other than http, https or sqs'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // THEN
        test.throws(() => new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
            rawMessageDelivery: true,
        }), /Raw message delivery/);
        test.done();
    },
    'throws with more than 5 attributes in a filter policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const cond = { conditions: [] };
        // THEN
        test.throws(() => new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
            filterPolicy: {
                a: cond,
                b: cond,
                c: cond,
                d: cond,
                e: cond,
                f: cond,
            },
        }), /5 attribute names/);
        test.done();
    },
    'throws with more than 100 conditions in a filter policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // THEN
        test.throws(() => new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
            filterPolicy: {
                a: { conditions: [...Array.from(Array(2).keys())] },
                b: { conditions: [...Array.from(Array(10).keys())] },
                c: { conditions: [...Array.from(Array(6).keys())] },
            },
        }), /\(120\) must not exceed 100/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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