#    @Author: Justin C Presley
#    @Author-Email: justincpresley@gmail.com
#    @Project: NDN State Vector Sync Protocol
#    @Source-Code: https://github.com/justincpresley/ndn-python-svs
#    @Pip-Library: https://pypi.org/project/ndn-svs/

# Basic Libraries
import asyncio as aio
from enum import Enum
from random import uniform
from typing import Callable, Optional, Tuple
# NDN Imports
from ndn.app import NDNApp
from ndn.encoding import Name, InterestParam, BinaryStr, FormalName, SignaturePtrs
from ndn.types import InterestNack, InterestTimeout, InterestCanceled, ValidationFailure
# Custom Imports
from .balancer import SVSyncBalancer
from .state_table import StateTable
from .meta_data import MetaData
from .state_vector import StateVector
from .scheduler import AsyncScheduler
from .security import SecurityOptions
from .logger import SVSyncLogger

# Class Type: an enumeration struct
# Class Purpose:
#   to differ core states.
class SVSyncCoreState(Enum):
    STEADY     = 0
    SUPRESSION = 1

# Class Type: a ndn class
# Class Purpose:
#   manage sync interests that are sent out.
#   to hear about other sync interests
#   to find out about new data from other nodes.
class SVSyncCore:
    def __init__(self, app:NDNApp, syncPrefix:Name, groupPrefix:Name, nid:Name, updateCallback:Callable, secOptions:SecurityOptions) -> None:
        SVSyncLogger.info(f'SVSyncCore: started svsync core')
        self.state = SVSyncCoreState.STEADY
        self.app = app
        self.nid = nid
        self.updateCallback = updateCallback
        self.syncPrefix = syncPrefix
        self.groupPrefix = groupPrefix
        self.secOptions = secOptions
        self.table = StateTable(self.nid)
        self.balancer = SVSyncBalancer(self.app, self.groupPrefix, self.nid, self.table, self.updateCallback, self.secOptions)
        self.seqNum = 0
        self.interval = 30000 # time in milliseconds
        self.randomPercent = 0.1
        self.briefInterval = 200 # time in milliseconds
        self.briefRandomPercent = 0.5
        self.app.route(self.syncPrefix, need_sig_ptrs=True)(self.onSyncInterest)
        SVSyncLogger.info(f'SVSyncCore: started listening to {Name.to_str(self.syncPrefix)}')
        self.scheduler = AsyncScheduler(self.sendSyncInterest, self.interval, self.randomPercent)
        self.scheduler.skip_interval()
    async def asyncSendSyncInterest(self) -> None:
        name = self.syncPrefix + [self.table.getMetaData().encode()] + [ self.table.getPart(0) ]
        SVSyncLogger.info(f'SVSyncCore: sync {Name.to_str(name)}')
        try:
            data_name, meta_info, content = await self.app.express_interest(
                name, signer=self.secOptions.syncSig.signer, must_be_fresh=True, can_be_prefix=True, lifetime=1000)
        except (InterestNack, InterestTimeout, InterestCanceled, ValidationFailure) as e:
            pass
    def sendSyncInterest(self) -> None:
        aio.get_event_loop().create_task(self.asyncSendSyncInterest())
    def onSyncInterest(self, int_name:FormalName, int_param:InterestParam, _app_param:Optional[BinaryStr], sig_ptrs:SignaturePtrs) -> None:
        aio.get_event_loop().create_task(self.onSyncInterestHelper(int_name, int_param, _app_param, sig_ptrs))
    async def onSyncInterestHelper(self, int_name:FormalName, int_param:InterestParam, _app_param:Optional[BinaryStr], sig_ptrs:SignaturePtrs) -> None:
        isValidated = await self.secOptions.syncVal.validate(int_name, sig_ptrs)
        if not isValidated:
            return

        incomingVector = StateVector(int_name[-2])
        incomingMetadata = MetaData(int_name[-3])
        SVSyncLogger.info(f'SVSyncCore: >> I: received sync')
        SVSyncLogger.info(f'SVSyncCore:       rmeta {bytes(incomingMetadata.source).decode()} - {incomingMetadata.tseqno} total, {incomingMetadata.nopcks} pcks')
        SVSyncLogger.info(f'SVSyncCore:       {incomingVector.to_str()}')

        missingList = self.table.processStateVector(incomingVector, oldData=False)
        SVSyncLogger.info(f'SVSyncCore:       {missingList}')
        self.table.updateMetaData()
        if missingList:
            self.updateCallback(missingList)

        supress, equalize = self.compareMetaData(incomingMetadata)
        self.state = SVSyncCoreState.SUPRESSION if supress else SVSyncCoreState.STEADY

        # reset the sync timer if STEADY
        # supress the timer if SUPPRESION
        if self.state == SVSyncCoreState.STEADY:
            self.scheduler.set_cycle()
        else:
            delay = self.briefInterval + round( uniform(-self.briefRandomPercent,self.briefRandomPercent)*self.briefInterval )
            if self.scheduler.get_time_left() > delay:
                self.scheduler.set_cycle(delay)
        SVSyncLogger.info(f'SVSyncCore: state {self.state.name}')
        SVSyncLogger.info(f'SVSyncCore: parts-{self.table.getPartCuts()} | 1stlength-{len(self.table.getPart(0))}')
        SVSyncLogger.info(f'SVSyncCore: table {self.table.getCompleteStateVector().to_str()}')

        if equalize and not self.balancer.isBusy():
            await self.balancer.equalize(incomingMetadata)
    def compareMetaData(self, incoming_md:MetaData) -> Tuple[bool, bool]:
        table_md = self.table.getMetaData()
        supress, equalize = False, False
        if table_md.tseqno > incoming_md.tseqno:
            supress = True
        elif table_md.tseqno < incoming_md.tseqno:
            equalize = True
        return (supress, equalize)
    def updateMyState(self, seqno:int) -> None:
        self.table.updateMyState(seqno)
        self.table.updateMetaData()
        self.scheduler.skip_interval()
        self.seqNum = self.table.getSeqNum(self.nid)
    def getSeqNum(self) -> int:
        return self.seqNum
    def getStateTable(self) -> StateTable:
        return self.table