"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const aws_cloudformation_1 = require("@aws-cdk/aws-cloudformation");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const dynamodb_generated_1 = require("./dynamodb.generated");
const replica_provider_1 = require("./replica-provider");
const scalable_table_attribute_1 = require("./scalable-table-attribute");
const HASH_KEY_TYPE = 'HASH';
const RANGE_KEY_TYPE = 'RANGE';
// https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
const MAX_LOCAL_SECONDARY_INDEX_COUNT = 5;
const READ_DATA_ACTIONS = [
    'dynamodb:BatchGetItem',
    'dynamodb:GetRecords',
    'dynamodb:GetShardIterator',
    'dynamodb:Query',
    'dynamodb:GetItem',
    'dynamodb:Scan'
];
const READ_STREAM_DATA_ACTIONS = [
    "dynamodb:DescribeStream",
    "dynamodb:GetRecords",
    "dynamodb:GetShardIterator",
];
const WRITE_DATA_ACTIONS = [
    'dynamodb:BatchWriteItem',
    'dynamodb:PutItem',
    'dynamodb:UpdateItem',
    'dynamodb:DeleteItem'
];
class TableBase extends core_1.Resource {
    /**
     * Adds an IAM policy statement associated with this table to an IAM
     * principal's policy.
     * @param grantee The principal (no-op if undefined)
     * @param actions The set of actions to allow (i.e. "dynamodb:PutItem", "dynamodb:GetItem", ...)
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [
                this.tableArn,
                core_1.Lazy.stringValue({ produce: () => this.hasIndex ? `${this.tableArn}/index/*` : core_1.Aws.NO_VALUE })
            ],
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement associated with this table's stream to an
     * IAM principal's policy.
     * @param grantee The principal (no-op if undefined)
     * @param actions The set of actions to allow (i.e. "dynamodb:DescribeStream", "dynamodb:GetRecords", ...)
     */
    grantStream(grantee, ...actions) {
        if (!this.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.tableStreamArn],
            scope: this,
        });
    }
    /**
     * Permits an IAM principal all data read operations from this table:
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan.
     * @param grantee The principal to grant access to
     */
    grantReadData(grantee) {
        return this.grant(grantee, ...READ_DATA_ACTIONS);
    }
    /**
     * Permits an IAM Principal to list streams attached to current dynamodb table.
     *
     * @param grantee The principal (no-op if undefined)
     */
    grantTableListStreams(grantee) {
        if (!this.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['dynamodb:ListStreams'],
            resourceArns: [
                core_1.Lazy.stringValue({ produce: () => `${this.tableArn}/stream/*` })
            ],
        });
    }
    /**
     * Permits an IAM principal all stream data read operations for this
     * table's stream:
     * DescribeStream, GetRecords, GetShardIterator, ListStreams.
     * @param grantee The principal to grant access to
     */
    grantStreamRead(grantee) {
        this.grantTableListStreams(grantee);
        return this.grantStream(grantee, ...READ_STREAM_DATA_ACTIONS);
    }
    /**
     * Permits an IAM principal all data write operations to this table:
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem.
     * @param grantee The principal to grant access to
     */
    grantWriteData(grantee) {
        return this.grant(grantee, ...WRITE_DATA_ACTIONS);
    }
    /**
     * Permits an IAM principal to all data read/write operations to this table.
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan,
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem
     * @param grantee The principal to grant access to
     */
    grantReadWriteData(grantee) {
        return this.grant(grantee, ...READ_DATA_ACTIONS, ...WRITE_DATA_ACTIONS);
    }
    /**
     * Permits all DynamoDB operations ("dynamodb:*") to an IAM principal.
     * @param grantee The principal to grant access to
     */
    grantFullAccess(grantee) {
        return this.grant(grantee, 'dynamodb:*');
    }
    /**
     * Return the given named metric for this Table
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/DynamoDB',
            metricName,
            dimensions: {
                TableName: this.tableName,
            },
            ...props
        });
    }
    /**
     * Metric for the consumed read capacity units this table
     *
     * @default sum over a minute
     */
    metricConsumedReadCapacityUnits(props) {
        return this.metric('ConsumedReadCapacityUnits', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the consumed write capacity units this table
     *
     * @default sum over a minute
     */
    metricConsumedWriteCapacityUnits(props) {
        return this.metric('ConsumedWriteCapacityUnits', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the system errors this table
     *
     * @default sum over a minute
     */
    metricSystemErrors(props) {
        return this.metric('SystemErrors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the user errors this table
     *
     * @default sum over a minute
     */
    metricUserErrors(props) {
        return this.metric('UserErrors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the conditional check failed requests this table
     *
     * @default sum over a minute
     */
    metricConditionalCheckFailedRequests(props) {
        return this.metric('ConditionalCheckFailedRequests', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the successful request latency this table
     *
     * @default avg over a minute
     */
    metricSuccessfulRequestLatency(props) {
        return this.metric('SuccessfulRequestLatency', { statistic: 'avg', ...props });
    }
}
/**
 * Provides a DynamoDB table.
 */
class Table extends TableBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.tableName,
        });
        this.keySchema = new Array();
        this.attributeDefinitions = new Array();
        this.globalSecondaryIndexes = new Array();
        this.localSecondaryIndexes = new Array();
        this.secondaryIndexNames = [];
        this.nonKeyAttributes = [];
        this.tableScaling = {};
        this.indexScaling = new Map();
        this.billingMode = props.billingMode || BillingMode.PROVISIONED;
        this.validateProvisioning(props);
        let streamSpecification;
        if (props.replicationRegions) {
            if (props.stream && props.stream !== StreamViewType.NEW_AND_OLD_IMAGES) {
                throw new Error('`stream` must be set to `NEW_AND_OLD_IMAGES` when specifying `replicationRegions`');
            }
            streamSpecification = { streamViewType: StreamViewType.NEW_AND_OLD_IMAGES };
            if (props.billingMode && props.billingMode !== BillingMode.PAY_PER_REQUEST) {
                throw new Error('The `PAY_PER_REQUEST` billing mode must be used when specifying `replicationRegions`');
            }
            this.billingMode = BillingMode.PAY_PER_REQUEST;
        }
        else if (props.stream) {
            streamSpecification = { streamViewType: props.stream };
        }
        this.table = new dynamodb_generated_1.CfnTable(this, 'Resource', {
            tableName: this.physicalName,
            keySchema: this.keySchema,
            attributeDefinitions: this.attributeDefinitions,
            globalSecondaryIndexes: core_1.Lazy.anyValue({ produce: () => this.globalSecondaryIndexes }, { omitEmptyArray: true }),
            localSecondaryIndexes: core_1.Lazy.anyValue({ produce: () => this.localSecondaryIndexes }, { omitEmptyArray: true }),
            pointInTimeRecoverySpecification: props.pointInTimeRecovery ? { pointInTimeRecoveryEnabled: props.pointInTimeRecovery } : undefined,
            billingMode: this.billingMode === BillingMode.PAY_PER_REQUEST ? this.billingMode : undefined,
            provisionedThroughput: this.billingMode === BillingMode.PAY_PER_REQUEST ? undefined : {
                readCapacityUnits: props.readCapacity || 5,
                writeCapacityUnits: props.writeCapacity || 5
            },
            sseSpecification: props.serverSideEncryption ? { sseEnabled: props.serverSideEncryption } : undefined,
            streamSpecification,
            timeToLiveSpecification: props.timeToLiveAttribute ? { attributeName: props.timeToLiveAttribute, enabled: true } : undefined
        });
        this.table.applyRemovalPolicy(props.removalPolicy);
        if (props.tableName) {
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.tableName);
        }
        this.tableArn = this.getResourceArnAttribute(this.table.attrArn, {
            service: 'dynamodb',
            resource: 'table',
            resourceName: this.physicalName,
        });
        this.tableName = this.getResourceNameAttribute(this.table.ref);
        this.tableStreamArn = streamSpecification ? this.table.attrStreamArn : undefined;
        this.scalingRole = this.makeScalingRole();
        this.addKey(props.partitionKey, HASH_KEY_TYPE);
        this.tablePartitionKey = props.partitionKey;
        if (props.sortKey) {
            this.addKey(props.sortKey, RANGE_KEY_TYPE);
            this.tableSortKey = props.sortKey;
        }
        if (props.replicationRegions) {
            this.createReplicaTables(props.replicationRegions);
        }
    }
    /**
     * Permits an IAM Principal to list all DynamoDB Streams.
     * @deprecated Use {@link #grantTableListStreams} for more granular permission
     * @param grantee The principal (no-op if undefined)
     */
    static grantListStreams(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['dynamodb:ListStreams'],
            resourceArns: ['*'],
        });
    }
    /**
     * Creates a Table construct that represents an external table via table name.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableName The table's name.
     */
    static fromTableName(scope, id, tableName) {
        return Table.fromTableAttributes(scope, id, { tableName });
    }
    /**
     * Creates a Table construct that represents an external table via table arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableArn The table's ARN.
     */
    static fromTableArn(scope, id, tableArn) {
        return Table.fromTableAttributes(scope, id, { tableArn });
    }
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `TableAttributes` object.
     */
    static fromTableAttributes(scope, id, attrs) {
        class Import extends TableBase {
            constructor(_tableArn, tableName, tableStreamArn) {
                super(scope, id);
                this.tableArn = _tableArn;
                this.tableName = tableName;
                this.tableStreamArn = tableStreamArn;
            }
            get hasIndex() {
                return false;
            }
        }
        let name;
        let arn;
        const stack = core_1.Stack.of(scope);
        if (!attrs.tableName) {
            if (!attrs.tableArn) {
                throw new Error('One of tableName or tableArn is required!');
            }
            arn = attrs.tableArn;
            const maybeTableName = stack.parseArn(attrs.tableArn).resourceName;
            if (!maybeTableName) {
                throw new Error('ARN for DynamoDB table must be in the form: ...');
            }
            name = maybeTableName;
        }
        else {
            if (attrs.tableArn) {
                throw new Error("Only one of tableArn or tableName can be provided");
            }
            name = attrs.tableName;
            arn = stack.formatArn({
                service: 'dynamodb',
                resource: 'table',
                resourceName: attrs.tableName,
            });
        }
        return new Import(arn, name, attrs.tableStreamArn);
    }
    /**
     * Add a global secondary index of table.
     *
     * @param props the property of global secondary index
     */
    addGlobalSecondaryIndex(props) {
        this.validateProvisioning(props);
        this.validateIndexName(props.indexName);
        // build key schema and projection for index
        const gsiKeySchema = this.buildIndexKeySchema(props.partitionKey, props.sortKey);
        const gsiProjection = this.buildIndexProjection(props);
        this.secondaryIndexNames.push(props.indexName);
        this.globalSecondaryIndexes.push({
            indexName: props.indexName,
            keySchema: gsiKeySchema,
            projection: gsiProjection,
            provisionedThroughput: this.billingMode === BillingMode.PAY_PER_REQUEST ? undefined : {
                readCapacityUnits: props.readCapacity || 5,
                writeCapacityUnits: props.writeCapacity || 5
            }
        });
        this.indexScaling.set(props.indexName, {});
    }
    /**
     * Add a local secondary index of table.
     *
     * @param props the property of local secondary index
     */
    addLocalSecondaryIndex(props) {
        // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
        if (this.localSecondaryIndexes.length >= MAX_LOCAL_SECONDARY_INDEX_COUNT) {
            throw new RangeError(`a maximum number of local secondary index per table is ${MAX_LOCAL_SECONDARY_INDEX_COUNT}`);
        }
        this.validateIndexName(props.indexName);
        // build key schema and projection for index
        const lsiKeySchema = this.buildIndexKeySchema(this.tablePartitionKey, props.sortKey);
        const lsiProjection = this.buildIndexProjection(props);
        this.secondaryIndexNames.push(props.indexName);
        this.localSecondaryIndexes.push({
            indexName: props.indexName,
            keySchema: lsiKeySchema,
            projection: lsiProjection
        });
    }
    /**
     * Enable read capacity scaling for this table
     *
     * @returns An object to configure additional AutoScaling settings
     */
    autoScaleReadCapacity(props) {
        if (this.tableScaling.scalableReadAttribute) {
            throw new Error('Read AutoScaling already enabled for this table');
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        return this.tableScaling.scalableReadAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, 'ReadScaling', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}`,
            dimension: 'dynamodb:table:ReadCapacityUnits',
            role: this.scalingRole,
            ...props
        });
    }
    /**
     * Enable write capacity scaling for this table
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleWriteCapacity(props) {
        if (this.tableScaling.scalableWriteAttribute) {
            throw new Error('Write AutoScaling already enabled for this table');
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        return this.tableScaling.scalableWriteAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, 'WriteScaling', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}`,
            dimension: 'dynamodb:table:WriteCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable read capacity scaling for the given GSI
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleGlobalSecondaryIndexReadCapacity(indexName, props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        const attributePair = this.indexScaling.get(indexName);
        if (!attributePair) {
            throw new Error(`No global secondary index with name ${indexName}`);
        }
        if (attributePair.scalableReadAttribute) {
            throw new Error('Read AutoScaling already enabled for this index');
        }
        return attributePair.scalableReadAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, `${indexName}ReadScaling`, {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}/index/${indexName}`,
            dimension: 'dynamodb:index:ReadCapacityUnits',
            role: this.scalingRole,
            ...props
        });
    }
    /**
     * Enable write capacity scaling for the given GSI
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleGlobalSecondaryIndexWriteCapacity(indexName, props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        const attributePair = this.indexScaling.get(indexName);
        if (!attributePair) {
            throw new Error(`No global secondary index with name ${indexName}`);
        }
        if (attributePair.scalableWriteAttribute) {
            throw new Error('Write AutoScaling already enabled for this index');
        }
        return attributePair.scalableWriteAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, `${indexName}WriteScaling`, {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}/index/${indexName}`,
            dimension: 'dynamodb:index:WriteCapacityUnits',
            role: this.scalingRole,
            ...props
        });
    }
    /**
     * Validate the table construct.
     *
     * @returns an array of validation error message
     */
    validate() {
        const errors = new Array();
        if (!this.tablePartitionKey) {
            errors.push('a partition key must be specified');
        }
        if (this.localSecondaryIndexes.length > 0 && !this.tableSortKey) {
            errors.push('a sort key of the table must be specified to add local secondary indexes');
        }
        return errors;
    }
    /**
     * Validate read and write capacity are not specified for on-demand tables (billing mode PAY_PER_REQUEST).
     *
     * @param props read and write capacity properties
     */
    validateProvisioning(props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            if (props.readCapacity !== undefined || props.writeCapacity !== undefined) {
                throw new Error('you cannot provision read and write capacity for a table with PAY_PER_REQUEST billing mode');
            }
        }
    }
    /**
     * Validate index name to check if a duplicate name already exists.
     *
     * @param indexName a name of global or local secondary index
     */
    validateIndexName(indexName) {
        if (this.secondaryIndexNames.includes(indexName)) {
            // a duplicate index name causes validation exception, status code 400, while trying to create CFN stack
            throw new Error(`a duplicate index name, ${indexName}, is not allowed`);
        }
        this.secondaryIndexNames.push(indexName);
    }
    /**
     * Validate non-key attributes by checking limits within secondary index, which may vary in future.
     *
     * @param nonKeyAttributes a list of non-key attribute names
     */
    validateNonKeyAttributes(nonKeyAttributes) {
        if (this.nonKeyAttributes.length + nonKeyAttributes.length > 20) {
            // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
            throw new RangeError('a maximum number of nonKeyAttributes across all of secondary indexes is 20');
        }
        // store all non-key attributes
        this.nonKeyAttributes.push(...nonKeyAttributes);
        // throw error if key attribute is part of non-key attributes
        this.attributeDefinitions.forEach(keyAttribute => {
            if (typeof keyAttribute.attributeName === 'string' && this.nonKeyAttributes.includes(keyAttribute.attributeName)) {
                throw new Error(`a key attribute, ${keyAttribute.attributeName}, is part of a list of non-key attributes, ${this.nonKeyAttributes}` +
                    ', which is not allowed since all key attributes are added automatically and this configuration causes stack creation failure');
            }
        });
    }
    buildIndexKeySchema(partitionKey, sortKey) {
        this.registerAttribute(partitionKey);
        const indexKeySchema = [
            { attributeName: partitionKey.name, keyType: HASH_KEY_TYPE }
        ];
        if (sortKey) {
            this.registerAttribute(sortKey);
            indexKeySchema.push({ attributeName: sortKey.name, keyType: RANGE_KEY_TYPE });
        }
        return indexKeySchema;
    }
    buildIndexProjection(props) {
        if (props.projectionType === ProjectionType.INCLUDE && !props.nonKeyAttributes) {
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-dynamodb-projectionobject.html
            throw new Error(`non-key attributes should be specified when using ${ProjectionType.INCLUDE} projection type`);
        }
        if (props.projectionType !== ProjectionType.INCLUDE && props.nonKeyAttributes) {
            // this combination causes validation exception, status code 400, while trying to create CFN stack
            throw new Error(`non-key attributes should not be specified when not using ${ProjectionType.INCLUDE} projection type`);
        }
        if (props.nonKeyAttributes) {
            this.validateNonKeyAttributes(props.nonKeyAttributes);
        }
        return {
            projectionType: props.projectionType ? props.projectionType : ProjectionType.ALL,
            nonKeyAttributes: props.nonKeyAttributes ? props.nonKeyAttributes : undefined
        };
    }
    findKey(keyType) {
        return this.keySchema.find(prop => prop.keyType === keyType);
    }
    addKey(attribute, keyType) {
        const existingProp = this.findKey(keyType);
        if (existingProp) {
            throw new Error(`Unable to set ${attribute.name} as a ${keyType} key, because ${existingProp.attributeName} is a ${keyType} key`);
        }
        this.registerAttribute(attribute);
        this.keySchema.push({
            attributeName: attribute.name,
            keyType
        });
        return this;
    }
    /**
     * Register the key attribute of table or secondary index to assemble attribute definitions of TableResourceProps.
     *
     * @param attribute the key attribute of table or secondary index
     */
    registerAttribute(attribute) {
        const name = attribute.name;
        const type = attribute.type;
        const existingDef = this.attributeDefinitions.find(def => def.attributeName === name);
        if (existingDef && existingDef.attributeType !== type) {
            throw new Error(`Unable to specify ${name} as ${type} because it was already defined as ${existingDef.attributeType}`);
        }
        if (!existingDef) {
            this.attributeDefinitions.push({
                attributeName: name,
                attributeType: type
            });
        }
    }
    /**
     * Return the role that will be used for AutoScaling
     */
    makeScalingRole() {
        // Use a Service Linked Role.
        // https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-service-linked-roles.html
        return iam.Role.fromRoleArn(this, 'ScalingRole', core_1.Stack.of(this).formatArn({
            service: 'iam',
            region: '',
            resource: 'role/aws-service-role/dynamodb.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_DynamoDBTable'
        }));
    }
    /**
     * Creates replica tables
     *
     * @param regions regions where to create tables
     */
    createReplicaTables(regions) {
        const stack = core_1.Stack.of(this);
        if (!core_1.Token.isUnresolved(stack.region) && regions.includes(stack.region)) {
            throw new Error('`replicationRegions` cannot include the region where this stack is deployed.');
        }
        const provider = replica_provider_1.ReplicaProvider.getOrCreate(this);
        // Documentation at https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2gt_IAM.html
        // is currently incorrect. AWS Support recommends `dynamodb:*` in both source and destination regions
        // Permissions in the source region
        this.grant(provider.onEventHandler, 'dynamodb:*');
        this.grant(provider.isCompleteHandler, 'dynamodb:DescribeTable');
        // Permissions in the destination regions
        provider.onEventHandler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['dynamodb:*'],
            resources: regions.map(region => stack.formatArn({
                region,
                service: 'dynamodb',
                resource: 'table',
                resourceName: this.tableName
            }))
        }));
        let previousRegion;
        for (const region of new Set(regions)) { // Remove duplicates
            // Use multiple custom resources because multiple create/delete
            // updates cannot be combined in a single API call.
            const currentRegion = new aws_cloudformation_1.CustomResource(this, `Replica${region}`, {
                provider: provider.provider,
                resourceType: 'Custom::DynamoDBReplica',
                properties: {
                    TableName: this.tableName,
                    Region: region,
                }
            });
            // Deploy time check to prevent from creating a replica in the region
            // where this stack is deployed. Only needed for environment agnostic
            // stacks.
            if (core_1.Token.isUnresolved(stack.region)) {
                const createReplica = new core_1.CfnCondition(this, `StackRegionNotEquals${region}`, {
                    expression: core_1.Fn.conditionNot(core_1.Fn.conditionEquals(region, core_1.Aws.REGION))
                });
                const cfnCustomResource = currentRegion.node.defaultChild;
                cfnCustomResource.cfnOptions.condition = createReplica;
            }
            // We need to create/delete regions sequentially because we cannot
            // have multiple table updates at the same time. The `isCompleteHandler`
            // of the provider waits until the replica is an ACTIVE state.
            if (previousRegion) {
                currentRegion.node.addDependency(previousRegion);
            }
            previousRegion = currentRegion;
        }
    }
    /**
     * Whether this table has indexes
     */
    get hasIndex() {
        return this.globalSecondaryIndexes.length + this.localSecondaryIndexes.length > 0;
    }
}
exports.Table = Table;
var AttributeType;
(function (AttributeType) {
    /** Up to 400KiB of binary data (which must be encoded as base64 before sending to DynamoDB) */
    AttributeType["BINARY"] = "B";
    /** Numeric values made of up to 38 digits (positive, negative or zero) */
    AttributeType["NUMBER"] = "N";
    /** Up to 400KiB of UTF-8 encoded text */
    AttributeType["STRING"] = "S";
})(AttributeType = exports.AttributeType || (exports.AttributeType = {}));
/**
 * DyanmoDB's Read/Write capacity modes.
 */
var BillingMode;
(function (BillingMode) {
    /**
     * Pay only for what you use. You don't configure Read/Write capacity units.
     */
    BillingMode["PAY_PER_REQUEST"] = "PAY_PER_REQUEST";
    /**
     * Explicitly specified Read/Write capacity units.
     */
    BillingMode["PROVISIONED"] = "PROVISIONED";
})(BillingMode = exports.BillingMode || (exports.BillingMode = {}));
var ProjectionType;
(function (ProjectionType) {
    /** Only the index and primary keys are projected into the index. */
    ProjectionType["KEYS_ONLY"] = "KEYS_ONLY";
    /** Only the specified table attributes are projected into the index. The list of projected attributes is in `nonKeyAttributes`. */
    ProjectionType["INCLUDE"] = "INCLUDE";
    /** All of the table attributes are projected into the index. */
    ProjectionType["ALL"] = "ALL";
})(ProjectionType = exports.ProjectionType || (exports.ProjectionType = {}));
/**
 * When an item in the table is modified, StreamViewType determines what information
 * is written to the stream for this table.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_StreamSpecification.html
 */
var StreamViewType;
(function (StreamViewType) {
    /** The entire item, as it appears after it was modified, is written to the stream. */
    StreamViewType["NEW_IMAGE"] = "NEW_IMAGE";
    /** The entire item, as it appeared before it was modified, is written to the stream. */
    StreamViewType["OLD_IMAGE"] = "OLD_IMAGE";
    /** Both the new and the old item images of the item are written to the stream. */
    StreamViewType["NEW_AND_OLD_IMAGES"] = "NEW_AND_OLD_IMAGES";
    /** Only the key attributes of the modified item are written to the stream. */
    StreamViewType["KEYS_ONLY"] = "KEYS_ONLY";
})(StreamViewType = exports.StreamViewType || (exports.StreamViewType = {}));
//# sourceMappingURL=data:application/json;base64,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