"use strict";
const AWS = require("aws-sdk-mock");
const sinon = require("sinon");
const replica_handler_1 = require("../lib/replica-handler");
AWS.setSDK(require.resolve('aws-sdk'));
const createEvent = {
    RequestType: 'Create',
    ResourceProperties: {
        TableName: 'my-table',
        Region: 'eu-west-2',
        ServiceToken: 'token'
    },
    ServiceToken: 'token',
    ResponseURL: 'url',
    LogicalResourceId: 'logical-id',
    RequestId: 'request-id',
    StackId: 'stack-id',
    ResourceType: 'resource-type'
};
module.exports = {
    'setUp'(callback) {
        process.env.USE_NORMAL_SDK = 'true';
        callback();
    },
    'tearDown'(callback) {
        delete process.env.USE_NORMAL_SDK;
        AWS.restore();
        callback();
    },
    async 'on event'(test) {
        const updateTableMock = sinon.fake.resolves({});
        AWS.mock('DynamoDB', 'updateTable', updateTableMock);
        const data = await replica_handler_1.onEventHandler(createEvent);
        sinon.assert.calledWith(updateTableMock, {
            TableName: 'my-table',
            ReplicaUpdates: [
                {
                    Create: {
                        RegionName: 'eu-west-2'
                    }
                },
            ]
        });
        test.deepEqual(data, {
            PhysicalResourceId: 'eu-west-2'
        });
        test.done();
    },
    async 'on event does not call updateTable for Update requests'(test) {
        const updateTableMock = sinon.fake.resolves({});
        AWS.mock('DynamoDB', 'updateTable', updateTableMock);
        const data = await replica_handler_1.onEventHandler({
            ...createEvent,
            RequestType: 'Update',
        });
        sinon.assert.notCalled(updateTableMock);
        test.deepEqual(data, {
            PhysicalResourceId: 'eu-west-2'
        });
        test.done();
    },
    async 'is complete for create returns false without replicas'(test) {
        const describeTableMock = sinon.fake.resolves({
            Table: {}
        });
        AWS.mock('DynamoDB', 'describeTable', describeTableMock);
        const data = await replica_handler_1.isCompleteHandler(createEvent);
        test.deepEqual(data, { IsComplete: false });
        test.done();
    },
    async 'is complete for create returns false when replica is not active'(test) {
        const describeTableMock = sinon.fake.resolves({
            Table: {
                Replicas: [
                    {
                        RegionName: 'eu-west-2',
                        ReplicaStatus: 'CREATING'
                    }
                ]
            }
        });
        AWS.mock('DynamoDB', 'describeTable', describeTableMock);
        const data = await replica_handler_1.isCompleteHandler(createEvent);
        test.deepEqual(data, { IsComplete: false });
        test.done();
    },
    async 'is complete for create returns false when table is not active'(test) {
        const describeTableMock = sinon.fake.resolves({
            Table: {
                Replicas: [
                    {
                        RegionName: 'eu-west-2',
                        ReplicaStatus: 'ACTIVE'
                    }
                ],
                TableStatus: 'UPDATING',
            }
        });
        AWS.mock('DynamoDB', 'describeTable', describeTableMock);
        const data = await replica_handler_1.isCompleteHandler(createEvent);
        test.deepEqual(data, { IsComplete: false });
        test.done();
    },
    async 'is complete for create returns true when replica is active'(test) {
        const describeTableMock = sinon.fake.resolves({
            Table: {
                Replicas: [
                    {
                        RegionName: 'eu-west-2',
                        ReplicaStatus: 'ACTIVE'
                    }
                ],
                TableStatus: 'ACTIVE',
            }
        });
        AWS.mock('DynamoDB', 'describeTable', describeTableMock);
        const data = await replica_handler_1.isCompleteHandler(createEvent);
        test.deepEqual(data, { IsComplete: true });
        test.done();
    },
    async 'is complete for delete returns true when replica is gone'(test) {
        const describeTableMock = sinon.fake.resolves({
            Table: {
                Replicas: [
                    {
                        RegionName: 'eu-west-1',
                        ReplicaStatus: 'ACTIVE'
                    }
                ],
                TableStatus: 'ACTIVE',
            }
        });
        AWS.mock('DynamoDB', 'describeTable', describeTableMock);
        const data = await replica_handler_1.isCompleteHandler({
            ...createEvent,
            RequestType: 'Delete'
        });
        test.deepEqual(data, { IsComplete: true });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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