import csv
import logging
import hashlib

#######
# Functions for making manifest
#
logger = logging.getLogger(__name__)
IO_BUFFER = 2 ** 10


class ConfigError(ValueError):
    """Indicate the config dictionary isn't correct"""
    pass


def compute_md5sums(filenames):
    """Compute md5sums of a list of files

    Parameters
    ----------
    filenames : list(string)
       List of filenames that we should compute the md5sum of

    Returns
    -------
    List of tuples containing the filename and md5 hex digest
    """
    results = []
    for f in filenames:
        with open(f, "rb") as instream:
            md5 = compute_stream_md5sum(instream)
        results.append((f, md5.hexdigest()))
    return results


def compute_stream_md5sum(instream):
    """Compute the md5sum of a file-like object

    Parameters
    ----------
    instream : file-like object

    Returns
    -------

    md5 digest object
    """
    md5 = hashlib.md5()
    readable = instream.readable()
    while readable:
        read_block = instream.read(IO_BUFFER)
        if len(read_block) == 0:
            readable = False
        else:
            md5.update(read_block)
    return md5


def create_metadata(config, md5s):
    """Compute metadata for a manifest file

    Parameters
    ----------
    config : dict
        Dictionary of metadata attributes that include experiment_accession,
        description, and library_accession.
    md5s : list((filename, md5 hexdigest))
        List of filename and hexdigest tuples that will be added to the manifest
        dictionary.
    """
    metadata = {
        "type": "MexGeneArchive_v1",
        "output_type": config['output_type'],
        "software_version": config["software_version"],
        "arguments": config["arguments"],
    }
    for key in ["experiment_accession", "description", "library_accession"]:
        if key in config:
            metadata[key] = config[key]

    for filename, md5 in md5s:
        metadata[filename] = "md5sum:{}".format(md5)

    return metadata


def validate_config_metadata(config):
    """Validate config dictionary

    Parameters
    ----------
    config : dict
        Dictionary of metadata attributes that include experiment_accession,
        description, and library_accession.
        It must also not include type, output_type, software_version, or arguments

    Raises
    ------
    ConfigError if something is wrong
    """
    not_user = ["type", "output_type", "software_version", "arguments"]
    user = ["experiment_accession", "description", "library_accession"]
    has_errors = False

    for key in not_user:
        if key in config:
            logger.error("{} will be generated by the pipeline")
            has_errors = True

    for key in user:
        if key not in config:
            logger.error("{} needs to be provided in the config file")
            has_errors = True

    if has_errors:
        raise ConfigError("Invalid attributes in config file")


def write_manifest(outstream, config):
    """Write manifest information to file-like object

    Parameters
    ----------
    outstream : writable file like object
        destination for the tab delimited contents
    config : dict
        Contains dictionary of manifest metadata values.
    """
    writer = csv.writer(outstream, delimiter="\t")
    writer.writerow(["name", "value"])
    for key in config:
        writer.writerow([key, config[key]])
    return outstream


def read_manifest(instream):
    """Read manifest file into a dictionary

    Parameters
    ----------
    instream : readable file like object
    """
    reader = csv.reader(instream, delimiter="\t")
    header = None
    metadata = {}
    for row in reader:
        if header is None:
            header = row
        else:
            metadata[row[0]] = row[1]
    return metadata
