"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SpaceDelimitedTextPattern = exports.FilterPattern = exports.JsonPattern = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Base class for patterns that only match JSON log events.
 *
 * @stability stable
 */
class JsonPattern {
    // This is a separate class so we have some type safety where users can't
    // combine text patterns and JSON patterns with an 'and' operation.
    /**
     * @stability stable
     */
    constructor(jsonPatternString) {
        this.jsonPatternString = jsonPatternString;
    }
    /**
     * @stability stable
     */
    get logPatternString() {
        return '{ ' + this.jsonPatternString + ' }';
    }
}
exports.JsonPattern = JsonPattern;
_a = JSII_RTTI_SYMBOL_1;
JsonPattern[_a] = { fqn: "@aws-cdk/aws-logs.JsonPattern", version: "1.137.0" };
/**
 * A collection of static methods to generate appropriate ILogPatterns.
 *
 * @stability stable
 */
class FilterPattern {
    /**
     * Use the given string as log pattern.
     *
     * See https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/FilterAndPatternSyntax.html
     * for information on writing log patterns.
     *
     * @param logPatternString The pattern string to use.
     * @stability stable
     */
    static literal(logPatternString) {
        return new LiteralLogPattern(logPatternString);
    }
    /**
     * A log pattern that matches all events.
     *
     * @stability stable
     */
    static allEvents() {
        return new LiteralLogPattern('');
    }
    /**
     * A log pattern that matches if all the strings given appear in the event.
     *
     * @param terms The words to search for.
     * @stability stable
     */
    static allTerms(...terms) {
        return new TextLogPattern([terms]);
    }
    /**
     * A log pattern that matches if any of the strings given appear in the event.
     *
     * @param terms The words to search for.
     * @stability stable
     */
    static anyTerm(...terms) {
        return new TextLogPattern(terms.map(t => [t]));
    }
    /**
     * A log pattern that matches if any of the given term groups matches the event.
     *
     * A term group matches an event if all the terms in it appear in the event string.
     *
     * @param termGroups A list of term groups to search for.
     * @stability stable
     */
    static anyTermGroup(...termGroups) {
        return new TextLogPattern(termGroups);
    }
    /**
     * A JSON log pattern that compares string values.
     *
     * This pattern only matches if the event is a JSON event, and the indicated field inside
     * compares with the string value.
     *
     * Use '$' to indicate the root of the JSON structure. The comparison operator can only
     * compare equality or inequality. The '*' wildcard may appear in the value may at the
     * start or at the end.
     *
     * For more information, see:
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/FilterAndPatternSyntax.html
     *
     * @param jsonField Field inside JSON.
     * @param comparison Comparison to carry out.
     * @param value The string value to compare to.
     * @stability stable
     */
    static stringValue(jsonField, comparison, value) {
        return new JSONStringPattern(jsonField, comparison, value);
    }
    /**
     * A JSON log pattern that compares numerical values.
     *
     * This pattern only matches if the event is a JSON event, and the indicated field inside
     * compares with the value in the indicated way.
     *
     * Use '$' to indicate the root of the JSON structure. The comparison operator can only
     * compare equality or inequality. The '*' wildcard may appear in the value may at the
     * start or at the end.
     *
     * For more information, see:
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/FilterAndPatternSyntax.html
     *
     * @param jsonField Field inside JSON.
     * @param comparison Comparison to carry out.
     * @param value The numerical value to compare to.
     * @stability stable
     */
    static numberValue(jsonField, comparison, value) {
        return new JSONNumberPattern(jsonField, comparison, value);
    }
    /**
     * A JSON log pattern that matches if the field exists and has the special value 'null'.
     *
     * @param jsonField Field inside JSON.
     * @stability stable
     */
    static isNull(jsonField) {
        return new JSONPostfixPattern(jsonField, 'IS NULL');
    }
    /**
     * A JSON log pattern that matches if the field does not exist.
     *
     * @param jsonField Field inside JSON.
     * @stability stable
     */
    static notExists(jsonField) {
        return new JSONPostfixPattern(jsonField, 'NOT EXISTS');
    }
    /**
     * A JSON log patter that matches if the field exists.
     *
     * This is a readable convenience wrapper over 'field = *'
     *
     * @param jsonField Field inside JSON.
     * @stability stable
     */
    static exists(jsonField) {
        return new JSONStringPattern(jsonField, '=', '*');
    }
    /**
     * A JSON log pattern that matches if the field exists and equals the boolean value.
     *
     * @param jsonField Field inside JSON.
     * @param value The value to match.
     * @stability stable
     */
    static booleanValue(jsonField, value) {
        return new JSONPostfixPattern(jsonField, value ? 'IS TRUE' : 'IS FALSE');
    }
    /**
     * A JSON log pattern that matches if all given JSON log patterns match.
     *
     * @stability stable
     */
    static all(...patterns) {
        jsiiDeprecationWarnings._aws_cdk_aws_logs_JsonPattern(patterns);
        if (patterns.length === 0) {
            throw new Error('Must supply at least one pattern, or use allEvents() to match all events.');
        }
        if (patterns.length === 1) {
            return patterns[0];
        }
        return new JSONAggregatePattern('&&', patterns);
    }
    /**
     * A JSON log pattern that matches if any of the given JSON log patterns match.
     *
     * @stability stable
     */
    static any(...patterns) {
        jsiiDeprecationWarnings._aws_cdk_aws_logs_JsonPattern(patterns);
        if (patterns.length === 0) {
            throw new Error('Must supply at least one pattern');
        }
        if (patterns.length === 1) {
            return patterns[0];
        }
        return new JSONAggregatePattern('||', patterns);
    }
    /**
     * A space delimited log pattern matcher.
     *
     * The log event is divided into space-delimited columns (optionally
     * enclosed by "" or [] to capture spaces into column values), and names
     * are given to each column.
     *
     * '...' may be specified once to match any number of columns.
     *
     * Afterwards, conditions may be added to individual columns.
     *
     * @param columns The columns in the space-delimited log stream.
     * @stability stable
     */
    static spaceDelimited(...columns) {
        return SpaceDelimitedTextPattern.construct(columns);
    }
}
exports.FilterPattern = FilterPattern;
_b = JSII_RTTI_SYMBOL_1;
FilterPattern[_b] = { fqn: "@aws-cdk/aws-logs.FilterPattern", version: "1.137.0" };
/**
 * Use a string literal as a log pattern
 */
class LiteralLogPattern {
    constructor(logPatternString) {
        this.logPatternString = logPatternString;
    }
}
/**
 * Search for a set of set of terms
 */
class TextLogPattern {
    constructor(clauses) {
        const quotedClauses = clauses.map(terms => terms.map(quoteTerm).join(' '));
        if (quotedClauses.length === 1) {
            this.logPatternString = quotedClauses[0];
        }
        else {
            this.logPatternString = quotedClauses.map(alt => '?' + alt).join(' ');
        }
    }
}
/**
 * A string comparison for JSON values
 */
class JSONStringPattern extends JsonPattern {
    constructor(jsonField, comparison, value) {
        comparison = validateStringOperator(comparison);
        super(`${jsonField} ${comparison} ${quoteTerm(value)}`);
    }
}
/**
 * A number comparison for JSON values
 */
class JSONNumberPattern extends JsonPattern {
    constructor(jsonField, comparison, value) {
        comparison = validateNumericalOperator(comparison);
        super(`${jsonField} ${comparison} ${value}`);
    }
}
/**
 * A postfix operator for JSON patterns
 */
class JSONPostfixPattern extends JsonPattern {
    constructor(jsonField, postfix) {
        // No validation, we assume these are generated by trusted factory functions
        super(`${jsonField} ${postfix}`);
    }
}
/**
 * Combines multiple other JSON patterns with an operator
 */
class JSONAggregatePattern extends JsonPattern {
    constructor(operator, patterns) {
        if (operator !== '&&' && operator !== '||') {
            throw new Error('Operator must be one of && or ||');
        }
        const clauses = patterns.map(p => '(' + p.jsonPatternString + ')');
        super(clauses.join(` ${operator} `));
    }
}
const COL_ELLIPSIS = '...';
/**
 * Space delimited text pattern.
 *
 * @stability stable
 */
class SpaceDelimitedTextPattern {
    // TODO: Temporarily changed from private to protected to unblock build. We need to think
    //     about how to handle jsii types with private constructors.
    /**
     * @stability stable
     */
    constructor(columns, restrictions) {
        this.columns = columns;
        this.restrictions = restrictions;
    }
    /**
     * Construct a new instance of a space delimited text pattern.
     *
     * Since this class must be public, we can't rely on the user only creating it through
     * the `LogPattern.spaceDelimited()` factory function. We must therefore validate the
     * argument in the constructor. Since we're returning a copy on every mutation, and we
     * don't want to re-validate the same things on every construction, we provide a limited
     * set of mutator functions and only validate the new data every time.
     *
     * @stability stable
     */
    static construct(columns) {
        // Validation happens here because a user could instantiate this object directly without
        // going through the factory
        for (const column of columns) {
            if (!validColumnName(column)) {
                throw new Error(`Invalid column name: ${column}`);
            }
        }
        if (sum(columns.map(c => c === COL_ELLIPSIS ? 1 : 0)) > 1) {
            throw new Error("Can use at most one '...' column");
        }
        return new SpaceDelimitedTextPattern(columns, {});
    }
    /**
     * Restrict where the pattern applies.
     *
     * @stability stable
     */
    whereString(columnName, comparison, value) {
        if (columnName === COL_ELLIPSIS) {
            throw new Error("Can't use '...' in a restriction");
        }
        if (this.columns.indexOf(columnName) === -1) {
            throw new Error(`Column in restrictions that is not in columns: ${columnName}`);
        }
        comparison = validateStringOperator(comparison);
        return new SpaceDelimitedTextPattern(this.columns, this.addRestriction(columnName, {
            comparison,
            stringValue: value,
        }));
    }
    /**
     * Restrict where the pattern applies.
     *
     * @stability stable
     */
    whereNumber(columnName, comparison, value) {
        if (columnName === COL_ELLIPSIS) {
            throw new Error("Can't use '...' in a restriction");
        }
        if (this.columns.indexOf(columnName) === -1) {
            throw new Error(`Column in restrictions that is not in columns: ${columnName}`);
        }
        comparison = validateNumericalOperator(comparison);
        return new SpaceDelimitedTextPattern(this.columns, this.addRestriction(columnName, {
            comparison,
            numberValue: value,
        }));
    }
    /**
     * @stability stable
     */
    get logPatternString() {
        return '[' + this.columns.map(this.columnExpression.bind(this)).join(', ') + ']';
    }
    /**
     * Return the column expression for the given column
     */
    columnExpression(column) {
        const restrictions = this.restrictions[column];
        if (!restrictions) {
            return column;
        }
        return restrictions.map(r => renderRestriction(column, r)).join(' && ');
    }
    /**
     * Make a copy of the current restrictions and add one
     */
    addRestriction(columnName, restriction) {
        const ret = {};
        for (const key of Object.keys(this.restrictions)) {
            ret[key] = this.restrictions[key].slice();
        }
        if (!(columnName in ret)) {
            ret[columnName] = [];
        }
        ret[columnName].push(restriction);
        return ret;
    }
}
exports.SpaceDelimitedTextPattern = SpaceDelimitedTextPattern;
_c = JSII_RTTI_SYMBOL_1;
SpaceDelimitedTextPattern[_c] = { fqn: "@aws-cdk/aws-logs.SpaceDelimitedTextPattern", version: "1.137.0" };
/**
 * Quote a term for use in a pattern expression
 *
 * It's never wrong to quote a string term, and required if the term
 * contains non-alphanumerical characters, so we just always do it.
 *
 * Inner double quotes are escaped using a backslash.
 */
function quoteTerm(term) {
    return '"' + term.replace(/\\/g, '\\\\').replace(/"/g, '\\"') + '"';
}
/**
 * Return whether the given column name is valid in a space-delimited table
 */
function validColumnName(column) {
    return column === COL_ELLIPSIS || /^[a-zA-Z0-9_-]+$/.exec(column);
}
/**
 * Validate and normalize the string comparison operator
 *
 * Correct for a common typo/confusion, treat '==' as '='
 */
function validateStringOperator(operator) {
    if (operator === '==') {
        operator = '=';
    }
    if (operator !== '=' && operator !== '!=') {
        throw new Error(`Invalid comparison operator ('${operator}'), must be either '=' or '!='`);
    }
    return operator;
}
const VALID_OPERATORS = ['=', '!=', '<', '<=', '>', '>='];
/**
 * Validate and normalize numerical comparison operators
 *
 * Correct for a common typo/confusion, treat '==' as '='
 */
function validateNumericalOperator(operator) {
    // Correct for a common typo, treat '==' as '='
    if (operator === '==') {
        operator = '=';
    }
    if (VALID_OPERATORS.indexOf(operator) === -1) {
        throw new Error(`Invalid comparison operator ('${operator}'), must be one of ${VALID_OPERATORS.join(', ')}`);
    }
    return operator;
}
/**
 * Render a table restriction
 */
function renderRestriction(column, restriction) {
    if (restriction.numberValue !== undefined) {
        return `${column} ${restriction.comparison} ${restriction.numberValue}`;
    }
    else if (restriction.stringValue) {
        return `${column} ${restriction.comparison} ${quoteTerm(restriction.stringValue)}`;
    }
    else {
        throw new Error('Invalid restriction');
    }
}
function sum(xs) {
    return xs.reduce((a, c) => a + c, 0);
}
//# sourceMappingURL=data:application/json;base64,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