"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClusterConfiguration = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const shard_1 = require("./shard");
class ClusterConfiguration {
    constructor(configuration) {
        this.shards = [];
        this.basicOptions = {
            availabilityZones: configuration.basicOptions?.availabilityZones ?? [],
            replicas: configuration.basicOptions?.replicas ?? 0,
        };
        this.sharding = configuration.sharding;
        if (this.sharding) {
            configuration.shards?.forEach((x) => {
                this.addShard(x);
            });
        }
    }
    static basic(options) {
        return new ClusterConfiguration({
            basicOptions: options,
            sharding: false,
        });
    }
    static complexSharding(...shards) {
        const config = new ClusterConfiguration({
            sharding: true,
        });
        shards.forEach((x) => {
            config.addShard(x);
        });
        return config;
    }
    static simpleSharding(options) {
        const config = new ClusterConfiguration({
            sharding: true,
        });
        for (let i = 0; i < (options?.shards ?? 1); i++) {
            config.addShard(new shard_1.Shard({
                replicaCount: options?.replicas,
            }));
        }
        return config;
    }
    addAvailabilityZone(zone) {
        if (this.sharding) {
            throw new Error([
                'Cannot add an availability zone to a sharded configuration.',
                'Toconfigure availability zones for sharded cluster, specify',
                'them on a per shard basis.',
            ].join(' '));
        }
        this.basicOptions?.availabilityZones?.push(zone);
        return this;
    }
    addShard(shard) {
        if (!this.sharding) {
            throw new Error('Cannot add a shard to basic configuration.');
        }
        this.shards.push(shard);
    }
    bind(scope) {
        if (!this.sharding) {
            return {
                numCacheClusters: 1 + (this.basicOptions?.replicas ?? 0),
                preferredCacheClusterAZs: aws_cdk_lib_1.Lazy.list({
                    produce: () => {
                        return this.basicOptions?.availabilityZones;
                    },
                }, {
                    omitEmpty: true,
                }),
            };
        }
        else {
            return {
                nodeGroupConfiguration: aws_cdk_lib_1.Lazy.any({
                    produce: () => {
                        return !this.hasComplexSharding() ? undefined : this.shards.map((x) => {
                            return x.bind(scope);
                        });
                    },
                }, {
                    omitEmptyArray: true,
                }),
                numNodeGroups: aws_cdk_lib_1.Lazy.number({
                    produce: () => {
                        return !this.hasComplexSharding() ? Math.max(this.shards.length, 1) : undefined;
                    },
                }),
                replicasPerNodeGroup: aws_cdk_lib_1.Lazy.number({
                    produce: () => {
                        return !this.hasComplexSharding() ? this.shards[0].replicaCount : undefined;
                    },
                }),
            };
        }
    }
    hasComplexSharding() {
        if (!this.sharding) {
            return false;
        }
        else if (this.shards.length === 0) {
            return false;
        }
        else if (!this.shards.every((x) => {
            return x.collapsible;
        })) {
            return false;
        }
        else if (!this.shards.every((x) => {
            return x.replicaCount == this.shards[0].replicaCount;
        })) {
            return false;
        }
        else {
            return true;
        }
    }
}
exports.ClusterConfiguration = ClusterConfiguration;
//# sourceMappingURL=data:application/json;base64,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