"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobExecutable = exports.JobLanguage = exports.PythonVersion = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const glue_version_1 = require("./glue-version");
const job_type_1 = require("./job-type");
var PythonVersion;
(function (PythonVersion) {
    /**
       * Python 3 (the exact version depends on GlueVersion and JobCommand used)
       */
    PythonVersion["THREE"] = "3";
    /**
       * Python 2 (the exact version depends on GlueVersion and JobCommand used)
       */
    PythonVersion["TWO"] = "2";
})(PythonVersion = exports.PythonVersion || (exports.PythonVersion = {}));
var JobLanguage;
(function (JobLanguage) {
    /**
       * Python
       */
    JobLanguage["PYTHON"] = "python";
    /**
       * Scala
       */
    JobLanguage["SCALA"] = "scala";
})(JobLanguage = exports.JobLanguage || (exports.JobLanguage = {}));
/**
 * The executable properties related to the Glue job's GlueVersion, JobType and code
 */
class JobExecutable {
    constructor(config) {
        if (job_type_1.JobType.PYTHON_SHELL === config.type) {
            if (config.language !== JobLanguage.PYTHON) {
                throw new Error('Python shell requires the language to be set to Python');
            }
            if ([glue_version_1.GlueVersion.V0_9, glue_version_1.GlueVersion.V2_0, glue_version_1.GlueVersion.V3_0].includes(config.glueVersion)) {
                throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support Python Shell`);
            }
        }
        if (config.extraJarsFirst && [glue_version_1.GlueVersion.V0_9, glue_version_1.GlueVersion.V1_0].includes(config.glueVersion)) {
            throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support extraJarsFirst`);
        }
        if (config.pythonVersion === PythonVersion.TWO && ![glue_version_1.GlueVersion.V0_9, glue_version_1.GlueVersion.V1_0].includes(config.glueVersion)) {
            throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support PythonVersion ${config.pythonVersion}`);
        }
        if (JobLanguage.PYTHON !== config.language && config.extraPythonFiles) {
            throw new Error('extraPythonFiles is not supported for languages other than JobLanguage.PYTHON');
        }
        this.config = config;
    }
    /**
       * Create Scala executable props for Apache Spark ETL job.
       *
       * @param props Scala Apache Spark Job props
       */
    static scalaEtl(props) {
        return new JobExecutable({
            ...props,
            language: JobLanguage.SCALA,
            type: job_type_1.JobType.ETL,
        });
    }
    /**
       * Create Scala executable props for Apache Spark Streaming job.
       *
       * @param props Scala Apache Spark Job props
       */
    static scalaStreaming(props) {
        return new JobExecutable({
            ...props,
            type: job_type_1.JobType.STREAMING,
            language: JobLanguage.SCALA,
        });
    }
    /**
       * Create Python executable props for Apache Spark ETL job.
       *
       * @param props Python Apache Spark Job props
       */
    static pythonEtl(props) {
        return new JobExecutable({
            ...props,
            type: job_type_1.JobType.ETL,
            language: JobLanguage.PYTHON,
        });
    }
    /**
       * Create Python executable props for Apache Spark Streaming job.
       *
       * @param props Python Apache Spark Job props
       */
    static pythonStreaming(props) {
        return new JobExecutable({
            ...props,
            type: job_type_1.JobType.STREAMING,
            language: JobLanguage.PYTHON,
        });
    }
    /**
       * Create Python executable props for python shell jobs.
       *
       * @param props Python Shell Job props.
       */
    static pythonShell(props) {
        return new JobExecutable({
            ...props,
            type: job_type_1.JobType.PYTHON_SHELL,
            language: JobLanguage.PYTHON,
        });
    }
    /**
       * Create a custom JobExecutable.
       *
       * @param config custom job executable configuration.
       */
    static of(config) {
        return new JobExecutable(config);
    }
    /**
       * Called during Job initialization to get JobExecutableConfig.
       */
    bind() {
        return this.config;
    }
}
exports.JobExecutable = JobExecutable;
_a = JSII_RTTI_SYMBOL_1;
JobExecutable[_a] = { fqn: "cdk-extensions.glue.JobExecutable", version: "0.0.12" };
//# sourceMappingURL=data:application/json;base64,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