import { Resource, ResourceProps } from 'aws-cdk-lib';
import { CfnSecurityConfiguration } from 'aws-cdk-lib/aws-glue';
import { IKey, Key } from 'aws-cdk-lib/aws-kms';
import { Construct } from 'constructs';
/**
 * Encryption mode for S3.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_S3Encryption.html#Glue-Type-S3Encryption-S3EncryptionMode
 */
export declare enum S3EncryptionMode {
    /**
       * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
       *
       * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
       */
    KMS = "SSE-KMS",
    /**
       * Server side encryption (SSE) with an Amazon S3-managed key.
       *
       * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
       */
    S3_MANAGED = "SSE-S3"
}
/**
 * Encryption mode for CloudWatch Logs.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_CloudWatchEncryption.html#Glue-Type-CloudWatchEncryption-CloudWatchEncryptionMode
 */
export declare enum CloudWatchEncryptionMode {
    /**
       * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
       *
       * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
       */
    KMS = "SSE-KMS"
}
/**
 * Encryption mode for Job Bookmarks.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_JobBookmarksEncryption.html#Glue-Type-JobBookmarksEncryption-JobBookmarksEncryptionMode
 */
export declare enum JobBookmarksEncryptionMode {
    /**
       * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
       *
       * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
       */
    CLIENT_SIDE_KMS = "CSE-KMS"
}
/**
 * CloudWatch Logs encryption configuration.
 */
export interface CloudWatchEncryption {
    /**
       * The KMS key to be used to encrypt the data.
       * @default A key will be created if one is not provided.
       */
    readonly kmsKey?: IKey;
    /**
       * Encryption mode
       */
    readonly mode: CloudWatchEncryptionMode;
}
/**
 * Job bookmarks encryption configuration.
 */
export interface JobBookmarksEncryption {
    /**
       * The KMS key to be used to encrypt the data.
       * @default A key will be created if one is not provided.
       */
    readonly kmsKey?: IKey;
    /**
       * Encryption mode.
       */
    readonly mode: JobBookmarksEncryptionMode;
}
/**
 * S3 encryption configuration.
 */
export interface S3Encryption {
    /**
       * The KMS key to be used to encrypt the data.
       * @default no kms key if mode = S3_MANAGED. A key will be created if one is not provided and mode = KMS.
       */
    readonly kmsKey?: IKey;
    /**
       * Encryption mode.
       */
    readonly mode: S3EncryptionMode;
}
/**
 * Configuration for the Glue SecurityConfiguration resource.
 */
export interface SecurityConfigurationProps extends ResourceProps {
    readonly cloudWatchEncryption?: CloudWatchEncryption;
    readonly jobBookmarksEncryption?: JobBookmarksEncryption;
    readonly name?: string;
    readonly s3Encryption?: S3Encryption;
}
export declare class SecurityConfiguration extends Resource {
    readonly cloudWatchEncryption?: CloudWatchEncryption;
    readonly jobBookmarksEncryption?: JobBookmarksEncryption;
    readonly name?: string;
    readonly s3Encryption?: S3Encryption;
    readonly key: Key;
    readonly resource: CfnSecurityConfiguration;
    readonly securityConfigurationName: string;
    constructor(scope: Construct, id: string, props: SecurityConfigurationProps);
}
