"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpEndpointDestination = exports.ContentEncoding = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const formatting_1 = require("../../../utils/formatting");
const backup_configuration_1 = require("../backup-configuration");
const delivery_stream_destination_1 = require("./delivery-stream-destination");
const s3_destination_1 = require("./s3-destination");
var ContentEncoding;
(function (ContentEncoding) {
    ContentEncoding["GZIP"] = "GZIP";
    ContentEncoding["NONE"] = "NONE";
})(ContentEncoding = exports.ContentEncoding || (exports.ContentEncoding = {}));
class HttpEndpointDestination extends delivery_stream_destination_1.DeliveryStreamDestination {
    constructor(url, options = {}) {
        super();
        // Internal properties
        this._commonAttributes = {};
        this._processors = [];
        this.accessKey = options.accessKey;
        this.backupConfiguration = options.backupConfiguration;
        this.buffering = options.buffering;
        this.cloudwatchLoggingConfiguration = options.cloudwatchLoggingConfiguration;
        this.contentEncoding = options.contentEncoding;
        this.endpointName = options.endpointName;
        this.endpointUrl = url;
        this.processorConfiguration = options.processorConfiguration;
        this.retryDuration = options.retryDuration;
        if (options.commonAttributes) {
            const commonAttributes = options.commonAttributes;
            Object.keys(commonAttributes).forEach((x) => {
                this.addCommonAttribute(x, commonAttributes[x]);
            });
        }
    }
    // Internal accessors
    get processingEnabled() {
        return this._processingEnabled ?? !!this._processors?.length;
    }
    get role() {
        return this._role;
    }
    addCommonAttribute(name, value) {
        this._commonAttributes[name] = value;
        return this;
    }
    addProcessor(processor) {
        this._processors.push(processor);
        return this;
    }
    bind(scope) {
        const processorConfiguration = this.processorConfiguration?.bind(scope);
        this._processingEnabled = processorConfiguration?.enabled;
        processorConfiguration?.processors.forEach((x) => {
            this.addProcessor(x);
        });
        if (this.cloudwatchLoggingConfiguration?.enabled) {
            const role = this.getOrCreateRole(scope);
            const logGroupScope = this.cloudwatchLoggingConfiguration.logGroup?.logGroupName ?? '*';
            const logStreamScope = this.cloudwatchLoggingConfiguration.logStream?.logStreamName ?? '*';
            role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                actions: [
                    'logs:PutLogEvents',
                ],
                effect: aws_iam_1.Effect.ALLOW,
                resources: [
                    aws_cdk_lib_1.Stack.of(this.cloudwatchLoggingConfiguration.logGroup ?? scope).formatArn({
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resource: 'log-group',
                        resourceName: `${logGroupScope}:log-stream:${logStreamScope}`,
                        service: 'logs',
                    }),
                ],
            }));
            this._role = role;
        }
        const backupConfiguration = this.buildBackupConfiguration(scope).bind(scope);
        return {
            httpEndpointDestinationConfiguration: {
                ...backupConfiguration,
                bufferingHints: this.buffering?.bind(scope),
                cloudWatchLoggingOptions: this.cloudwatchLoggingConfiguration?.bind(scope),
                endpointConfiguration: {
                    accessKey: this.accessKey?.toString(),
                    name: this.endpointName,
                    url: this.endpointUrl,
                },
                processingConfiguration: aws_cdk_lib_1.Lazy.any({
                    produce: () => {
                        return this.renderProcessorConfiguration(scope);
                    },
                }),
                requestConfiguration: formatting_1.definedFieldsOrUndefined({
                    commonAttributes: aws_cdk_lib_1.Lazy.any({
                        produce: () => {
                            return !Object.keys(this._commonAttributes).map((x) => {
                                return {
                                    attributeName: x,
                                    attributeValue: this._commonAttributes[x],
                                };
                            });
                        },
                    }),
                    contentEncoding: this.contentEncoding,
                }),
                retryOptions: formatting_1.undefinedIfNoKeys({
                    durationInSeconds: this.retryDuration?.toSeconds(),
                }),
                roleArn: this.role?.roleArn,
                s3BackupMode: backupConfiguration.s3BackupMode,
                s3Configuration: backupConfiguration.s3BackupConfiguration,
            },
        };
    }
    buildBackupConfiguration(scope) {
        if (this.backupConfiguration) {
            return this.backupConfiguration;
        }
        else {
            const bucket = new aws_s3_1.Bucket(scope, 'http-endpoint-backup-bucket', {
                blockPublicAccess: {
                    blockPublicAcls: true,
                    blockPublicPolicy: true,
                    ignorePublicAcls: true,
                    restrictPublicBuckets: true,
                },
                versioned: true,
            });
            return new backup_configuration_1.BackupConfiguration({
                destination: new s3_destination_1.S3Destination(bucket, {
                    cloudwatchLoggingConfiguration: this.cloudwatchLoggingConfiguration,
                    compressionFormat: s3_destination_1.S3CompressionFormat.GZIP,
                    role: this.getOrCreateRole(scope),
                }),
            });
        }
    }
    getOrCreateRole(scope) {
        if (this.role) {
            return this.role;
        }
        else {
            this._role = new aws_iam_1.Role(scope, 'http-endpoint-delivery-role', {
                assumedBy: new aws_iam_1.PrincipalWithConditions(new aws_iam_1.ServicePrincipal('firehose.amazonaws.com'), {
                    StringEquals: {
                        'sts:ExternalId': aws_cdk_lib_1.Stack.of(scope).account,
                    },
                }),
            });
            return this._role;
        }
    }
    renderProcessorConfiguration(scope) {
        return formatting_1.definedFieldsOrUndefined({
            enabled: this.processingEnabled,
            processors: this._processors.map((x) => {
                return x.bind(scope);
            }),
        });
    }
}
exports.HttpEndpointDestination = HttpEndpointDestination;
_a = JSII_RTTI_SYMBOL_1;
HttpEndpointDestination[_a] = { fqn: "cdk-extensions.kinesis_firehose.HttpEndpointDestination", version: "0.0.12" };
//# sourceMappingURL=data:application/json;base64,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