"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OutboundResolver = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ram_1 = require("aws-cdk-lib/aws-ram");
const forward_resolver_rule_1 = require("./forward-resolver-rule");
const resolver_endpoint_base_1 = require("./resolver-endpoint-base");
class OutboundResolver extends resolver_endpoint_base_1.ResolverEndpointBase {
    /**
       * Creates a new instance of the OutboundResolver class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            direction: OutboundResolver.DIRECTION,
            subnetOffset: OutboundResolver.SUBNET_OFFSET,
        });
        // Resource properties
        this.resolverRules = [];
        this.organizationArn = props.organizationArn;
        this.outboundCidrs = props.outboundCidrs ?? OutboundResolver.DEFAULT_OUTBOUND_CIDRS;
        for (let cidr of this.outboundCidrs) {
            this.securityGroup.addEgressRule(aws_ec2_1.Peer.ipv4(cidr), aws_ec2_1.Port.tcp(53), `TCP DNS to ${cidr}`);
            this.securityGroup.addEgressRule(aws_ec2_1.Peer.ipv4(cidr), aws_ec2_1.Port.udp(53), `UDP DNS to ${cidr}`);
        }
    }
    /**
       * Adds a new Resolver Rule for a given domain to the Outbound Resolver. Also adds the created
       * rule to a RAM Resource Share if an organization ARN was specified when the Outbound Resolver
       * was created.
       *
       * @param domain The domain you would like to add the rule for.
       * @param targets The IP addresses of the external resolver that should be used to resolve the
       * domain.
       */
    addRule(domain, targets) {
        // Create the resolver rule, forwarding requests for the domain (and all subdomains) to the
        // specified IP addressed (on port 53).
        const rule = new forward_resolver_rule_1.ForwardResolverRule(this, `resolver-rule-${domain}`, {
            domainName: domain,
            resolverEndpoint: this,
            targetIps: targets,
        });
        this.resolverRules.push(rule);
        // If this is the first rule that was added and an organization ARN was given, create a Resource
        // Share to share the created rules with the organization. The list of rules in the resource share is
        // Lazy evaluated so any subsequent rules that get added will be automatically included without
        // and further action needing to be taken.
        if (this.organizationArn && !this.resourceShare) {
            this.resourceShare = new aws_ram_1.CfnResourceShare(this, 'resource-share', {
                allowExternalPrincipals: false,
                name: aws_cdk_lib_1.Lazy.uncachedString({
                    produce: (_) => {
                        return aws_cdk_lib_1.Names.uniqueId(this.resourceShare);
                    },
                }),
                principals: [
                    this.stack.formatArn(this.organizationArn),
                ],
                resourceArns: aws_cdk_lib_1.Lazy.uncachedList({
                    produce: () => {
                        return this.resolverRules.map((x) => {
                            return x.resolverRuleArn;
                        });
                    },
                }),
            });
        }
        return rule;
    }
}
exports.OutboundResolver = OutboundResolver;
// Static properties
OutboundResolver.DEFAULT_OUTBOUND_CIDRS = [
    '10.0.0.0/8',
    '172.16.0.0/12',
    '192.168.0.0/16',
];
OutboundResolver.DIRECTION = 'OUTBOUND';
OutboundResolver.SUBNET_OFFSET = 5;
//# sourceMappingURL=data:application/json;base64,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