"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessControlAttribute = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * Represents and ABAC attribute in IAM Identity Center.
 *
 * These are IAM Identity Center identity store attributes that you can
 * configure for use in attributes-based access control (ABAC). You can create
 * permissions policies that determine who can access your AWS resources based
 * upon the configured attribute values. When you enable ABAC and specify
 * `AccessControlAttributes`, IAM Identity Center passes the attribute values
 * of the authenticated user into IAM for use in policy evaluation.
 */
class AccessControlAttribute {
    /**
       * Creates a new instance of the AccessControlAttribute class.
       *
       * @param options The configuration settings to use when configuring the
       * attribute.
       */
    constructor(options) {
        // Internal properties
        this._sources = [];
        this.name = options.name;
    }
    // Standard accessors
    /**
       * A list of identity sources to use when mapping a specified attribute to
       * IAM Identity Center. Note that the array is readonly and changes made
       * to it will not be reflected when generating ABAC attribute
       * configuration. To add a source to the attribute use the {@link addSource}
       * method.
       */
    get sources() {
        return [...this._sources];
    }
    /**
       * Adds an identity source to use when mapping the attribute to IAM
       * Identity Center.
       *
       * @param source The source to add.
       * @returns The ABAC attribute the source was associated with.
       */
    addSource(source) {
        this._sources.push(source);
        return this;
    }
    /**
       * Generates the raw CloudFormation configuration that this attribute
       * represents within the context of a given scope.
       *
       * @param scope The construct managing the access control attribute
       * configuration that will consume details of this attribute.
       * @returns The raw CloudFormation configuration that this attribute
       * represents.
       */
    bind(scope) {
        return {
            key: this.name,
            value: {
                source: aws_cdk_lib_1.Lazy.list({
                    produce: () => {
                        if (this._sources.length === 0) {
                            aws_cdk_lib_1.Annotations.of(scope).addError([
                                `AccessControlAttribute for ${this.name} has no sources.`,
                                'At least one source is required.',
                            ].join(' '));
                        }
                        return this._sources;
                    },
                }),
            },
        };
    }
}
exports.AccessControlAttribute = AccessControlAttribute;
_a = JSII_RTTI_SYMBOL_1;
AccessControlAttribute[_a] = { fqn: "cdk-extensions.sso.AccessControlAttribute", version: "0.0.12" };
//# sourceMappingURL=data:application/json;base64,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