"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PermissionSet = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_sso_1 = require("aws-cdk-lib/aws-sso");
const referenced_managed_policy_1 = require("./lib/referenced-managed-policy");
class PermissionSetBase extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
    }
}
/**
 * Specifies a permission set within a specified IAM Identity Center instance.
 */
class PermissionSet extends PermissionSetBase {
    /**
       * Creates a new instance of the PermissionSet class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in
       * the construct tree.
       * @param id A name to be associated with the stack and used in resource
       * naming. Must be unique within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope, id, props) {
        super(scope, id, props);
        // Internal properties
        this._customerManagedPolicies = {};
        this._inlinePolicies = {};
        this._managedPolicies = [];
        this.description = props.description;
        this.instance = props.instance;
        this.name = props.name ?? aws_cdk_lib_1.Names.uniqueId(this);
        this.permissionsBoundary = props.permissionsBoundary;
        this.relayState = props.relayState;
        this.sessionDuration = props.sessionDuration;
        this._inlinePolicies = props.inlinePolicies ?? {};
        this.resource = new aws_sso_1.CfnPermissionSet(this, 'Resource', {
            customerManagedPolicyReferences: aws_cdk_lib_1.Lazy.any({
                produce: () => {
                    return this.stack.toJsonString(Object.keys(this._customerManagedPolicies).map((x) => {
                        const item = this._customerManagedPolicies[x];
                        return {
                            name: item.policyName,
                            path: item.policyPath,
                        };
                    }));
                },
            }, {
                omitEmptyArray: true,
            }),
            description: this.description,
            instanceArn: this.instance.instanceArn,
            managedPolicies: aws_cdk_lib_1.Lazy.list({
                produce: () => {
                    return this._managedPolicies.map((x) => {
                        return x.managedPolicyArn;
                    });
                },
            }, {
                omitEmpty: true,
            }),
            inlinePolicy: aws_cdk_lib_1.Lazy.string({
                produce: () => {
                    if (Object.keys(this._inlinePolicies).length === 0) {
                        return undefined;
                    }
                    return this.stack.toJsonString(Object.keys(this._inlinePolicies).map((x) => {
                        return {
                            PolicyDocument: this._inlinePolicies[x].toJSON(),
                            PolicyName: x,
                        };
                    }));
                },
            }),
            name: this.name,
            permissionsBoundary: this.permissionsBoundary?.bind(this),
            relayStateType: this.relayState,
            sessionDuration: this.sessionDuration ? this.sessionDuration.toIsoString() : undefined,
        });
        this.permissionSetArn = this.resource.attrPermissionSetArn;
        props.managedPolicies?.forEach((x) => {
            this.addManagedPolicy(x);
        });
    }
    static fromArn(scope, id, arn) {
        class Import extends PermissionSetBase {
            constructor() {
                super(scope, id);
                this.permissionSetArn = arn;
            }
        }
        return new Import();
    }
    /**
       * Adds a custom managed policy to the permission set. When using customer
       * managed policies it is required that a managed policy with a matching
       * name and path exist in any AWS account for which the permission set
       * will be assigned.
       *
       * @param options The configuration for the customer managed policy.
       * @returns A dynamically generated ManagedPolicy class that can be used
       * to create compatible managed policies in other accounts.
       */
    addCustomerManagedPolicy(options) {
        const fullPath = `${options.path ?? ''}/${options.name}`;
        const result = this._customerManagedPolicies[fullPath] ?? referenced_managed_policy_1.ReferencedManagedPolicy.of(options);
        this._customerManagedPolicies[fullPath] = result;
        return result;
    }
    /**
       * Adds a new Managed Policy to the permission set. Only Managed Policies
       * created and maintained by AWS are supported. To add a custom Managed
       * Policy that you control use the {@link addCustomerManagedPolicy} method.
       *
       * @param policy The AWS Managed Policy to associate with the Permission
       * Set.
       * @returns The Permission Set resource the Managed Policy was added to.
       */
    addManagedPolicy(policy) {
        this._managedPolicies.push(policy);
        return this;
    }
    /**
       * Adds a permission to the permission set's default policy document.
       *
       * If there is no default policy attached to this permission set, it will be created.
       *
       * @param statement The permission statement to add to the policy document.
       * @returns An [AddToPrincipalPolicyResult](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_iam.AddToPrincipalPolicyResult.html) object that provides details of
       * the result of the operation.
       */
    addToPrincipalPolicy(statement) {
        const defaultPolicyName = `default-policy-${this.node.addr}`;
        const document = this._inlinePolicies[defaultPolicyName] ?? new aws_iam_1.PolicyDocument();
        this._inlinePolicies[defaultPolicyName] = document;
        document.addStatements(statement);
        return {
            statementAdded: true,
            policyDependable: document,
        };
    }
}
exports.PermissionSet = PermissionSet;
_a = JSII_RTTI_SYMBOL_1;
PermissionSet[_a] = { fqn: "cdk-extensions.sso.PermissionSet", version: "0.0.12" };
//# sourceMappingURL=data:application/json;base64,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