"""Command line argument parser for pipen"""
from __future__ import annotations

from typing import TYPE_CHECKING, Any, Type, Mapping

from argx import ArgumentParser
from diot import Diot
from pipen_annotate import annotate

from .defaults import PIPELINE_ARGS_GROUP, FLATTEN_PROC_ARGS, PIPEN_ARGS

if TYPE_CHECKING:  # pragma: no cover
    from pipen import Pipen, Proc
    from argx import Namespace


class ParserMeta(type):
    """Meta class for Proc"""

    _INST = None

    def __call__(cls, *args, **kwds) -> Parser:
        """Make sure Parser class is singleton

        Args:
            *args: and
            **kwds: Arguments for the constructor

        Returns:
            The Parser instance
        """
        if cls._INST is None:
            cls._INST = super().__call__(*args, **kwds)

        return cls._INST


class Parser(ArgumentParser, metaclass=ParserMeta):
    """Subclass of Params to fit for pipen

    Args:
        pipeline_args_group: The group name to gather all the parameters on
            help page
        flatten_proc_args: Flatten process arguments to the top level
    """
    def __init__(
        self,
        *args,
        pipeline_args_group: str = PIPELINE_ARGS_GROUP,
        flatten_proc_args: bool | str = FLATTEN_PROC_ARGS,
        **kwargs,
    ) -> None:
        """Constructor"""
        kwargs["add_help"] = "+"
        kwargs["fromfile_prefix_chars"] = "@"
        kwargs["usage"] = "%(prog)s [-h | -h+] [options]"
        super().__init__(*args, **kwargs)

        self.flatten_proc_args = flatten_proc_args
        self._cli_args = None
        self._pipeline_args_group = self.add_argument_group(
            pipeline_args_group,
            order=-99,
        )
        self._parsed = None

    def set_cli_args(self, args: Any) -> None:
        """Set cli arguments, allows externals to set arguments to parse"""
        self._cli_args = args

    def parse_args(self, args: Any = None, namespace: Any = None) -> Namespace:
        """Parse arguments"""
        if not self._parsed:
            # This should be called only once at `on_init` hook
            # If you have additional arguments, before `on_init`
            # You should call `parse_known_args` instead
            if self._cli_args is not None:
                args = self._cli_args

            self._parsed = super().parse_args(args, namespace)
        return self._parsed

    def init(self, pipen: Pipen) -> None:
        """Define arguments"""
        pipen.build_proc_relationships()
        if len(pipen.procs) > 1 and self.flatten_proc_args is True:
            raise ValueError(  # pragma: no cover
                "Cannot flatten process arguments for multiprocess pipeline."
            )

        if self.flatten_proc_args == "auto":
            self.flatten_proc_args = (
                len(pipen.procs) == 1
                and not getattr(pipen.procs[0], "__procgroup__", False)
            )

        for arg, argopt in PIPEN_ARGS.items():
            if arg == "order":
                continue

            if arg == "outdir":
                argopt["default"] = pipen.outdir
            elif arg == "name":
                argopt["default"] = pipen.name

            if arg in ("scheduler_opts", "plugin_opts"):
                if self.flatten_proc_args:
                    argopt["default"] = (
                        Diot(pipen.config.get(arg, None) or {})
                        | (getattr(pipen.procs[0], arg, None) or {})
                    )
                else:
                    argopt["default"] = pipen.config.get(arg, None) or {}

            self._pipeline_args_group.add_argument(f"--{arg}", **argopt)

        if self.flatten_proc_args is True:
            self._add_proc_args(
                pipen.procs[0],
                is_start=True,
                hide=False,
                flatten=True,
            )
        else:
            for i, proc in enumerate(pipen.procs):
                in_procgroup = bool(getattr(proc, "__procgroup__", None))
                self._add_proc_args(
                    proc,
                    is_start=proc in pipen.starts and not in_procgroup,
                    hide=(
                        in_procgroup
                        if not proc.plugin_opts
                        else proc.plugin_opts.get("args_hide", in_procgroup)
                    ),
                    flatten=False,
                    order=i,
                )
            self.description = (
                f"{self.description or ''}\n"  # type: ignore[has-type]
                "Use `@configfile` to load default values for the options."
            )

    def _get_arg_attrs_from_anno(
        self,
        anno_attrs: Mapping[str, Any],
        terms: Mapping[str, Any] | None = None,
    ) -> Mapping[str, Any]:
        """Get argument attributes from annotation"""
        out = {
            k: v
            for k, v in anno_attrs.items()
            if k in (
                "help",
                "show",
                "action",
                "nargs",
                "default",
                "dest",
                "required",
                "metavar",
                "choices",
            )
        }
        if "atype" in anno_attrs:
            out["type"] = anno_attrs["atype"]
        elif "type" in anno_attrs:
            out["type"] = anno_attrs["type"]

        typefun = None
        if out.get("type"):
            typefun = self._registry_get("type", out["type"], out["type"])

        choices = out.get("choices", None)
        if choices is True:
            out["choices"] = list(terms)
        elif isinstance(choices, str):
            out["choices"] = choices.split(",")

        if out.get("choices") and typefun:
            out["choices"] = [typefun(c) for c in out["choices"]]

        return out

    def _add_proc_args(
        self,
        proc: Type[Proc],
        is_start: bool,
        hide: bool,
        flatten: bool,
        order: int = 0,
    ) -> None:
        """Add process arguments"""
        if is_start:
            hide = False

        anno = annotate(proc)

        if not flatten:
            name = (
                f"{proc.__procgroup__.name}/{proc.name}"
                if getattr(proc, "__procgroup__", None)
                else proc.name
            )
            # add a namespace argumemnt for this proc
            self.add_namespace(
                proc.name,
                title=f"Process <{name}>",
                show=not hide,
                order=order + 1,  # avoid 0 conflicting default
            )
        else:
            self.description = (
                f"{anno.Summary.short}\n{anno.Summary.long}\n"
                "Use `@configfile` to load default values for the options."
            )

        if is_start:
            for inkey, inval in anno.Input.items():
                self.add_argument(
                    f"--in.{inkey}"
                    if flatten
                    else f"--{proc.name}.in.{inkey}",
                    help=inval.help or "",
                    **self._get_arg_attrs_from_anno(inval.attrs),
                )

        if not proc.nexts:
            for key, val in anno.Output.items():
                self.add_argument(
                    f"--out.{key}" if flatten else f"--{proc.name}.out.{key}",
                    help=val.help or "",
                    **self._get_arg_attrs_from_anno(val.attrs),
                )

        if proc.envs:
            self.add_argument(
                "--envs" if flatten else f"--{proc.name}.envs",
                action="ns",
                help="Environment variables for the process",
                default=Diot(proc.envs)
            )

        self._add_envs_arguments(
            self,
            anno.Envs,
            proc.envs or {},
            flatten,
            proc.name,
        )

        if not flatten:
            for key in (
                "cache",
                "dirsig",
                "lang",
                "error_strategy",
                "num_retries",
                "forks",
                "order",
            ):
                attrs = PIPEN_ARGS[key]
                default = getattr(proc, key)
                if default is not None:
                    attrs["default"] = default

                self.add_argument(f"--{proc.name}.{key}", **attrs)

            for key in ("plugin_opts", "scheduler_opts"):
                self.add_argument(f"--{proc.name}.{key}", **PIPEN_ARGS[key])

    def _add_envs_arguments(
        self,
        ns: Namespace,
        anno: Mapping[str, Any],
        values: Mapping[str, Any],
        flatten: bool,
        proc_name: str,
        key: str = "envs",
    ) -> None:
        """Add the envs argument to the namespace"""
        for kk, vv in anno.items():
            if kk not in values:
                continue

            default = values[kk]
            if default is not None:
                vv.attrs["default"] = default

            ns.add_argument(
                f"--{key}.{kk}" if flatten else f"--{proc_name}.{key}.{kk}",
                help=vv.help or "",
                **self._get_arg_attrs_from_anno(vv.attrs, vv.terms),
            )

            # add sub-namespace
            if vv.attrs.get("action", None) in ("namespace", "ns"):
                self._add_envs_arguments(
                    ns=ns,
                    anno=vv.terms,
                    values=default,
                    flatten=flatten,
                    proc_name=proc_name,
                    key=f"{key}.{kk}",
                )
