/* *****************************************************************************
 * A.L.E (Arcade Learning Environment)
 * Copyright (c) 2009-2013 by Yavar Naddaf, Joel Veness, Marc G. Bellemare and
 *   the Reinforcement Learning and Artificial Intelligence Laboratory
 * Released under the GNU General Public License; see License.txt for details.
 *
 * Based on: Stella  --  "An Atari 2600 VCS Emulator"
 * Copyright (c) 1995-2007 by Bradford W. Mott and the Stella team
 *
 * *****************************************************************************
 */

#include "DoubleDunk.hpp"

#include "../RomUtils.hpp"

namespace ale {

DoubleDunkSettings::DoubleDunkSettings() { reset(); }

/* create a new instance of the rom */
RomSettings* DoubleDunkSettings::clone() const {
  return new DoubleDunkSettings(*this);
}

/* process the latest information from ALE */
void DoubleDunkSettings::step(const System& system) {
  // update the reward
  int my_score = getDecimalScore(0xF6, &system);
  int oppt_score = getDecimalScore(0xF7, &system);
  int score = my_score - oppt_score;
  m_reward_p1 = score - m_score;
  m_reward_p2 = -m_reward_p1;
  m_score = score;

  // update terminal status
  int some_value = readRam(&system, 0xFE);
  m_terminal = (my_score >= 24 || oppt_score >= 24) && some_value == 0xE7;

  if(is_two_player){
    int choice_value = readRam(&system, 0x89);
    if (!(choice_value == 0 || choice_value&0x80 || choice_value&0x40)){
      no_choice_counter = 0;
    }
    no_choice_counter++;
    if(max_turn_time > 0 && no_choice_counter > max_turn_time){
      if(choice_value == 0){
        m_reward_p1 = -1;
        m_reward_p2 = -1;
      }
      else if(choice_value&0x40){
        m_reward_p1 = 0;
        m_reward_p2 = -1;
      }
      else if(choice_value&0x80){
        m_reward_p1 = -1;
        m_reward_p2 = 0;
      }
      no_choice_counter = 0;
    }
  }
}

/* is end of game */
bool DoubleDunkSettings::isTerminal() const { return m_terminal; };

/* get the most recently observed reward */
reward_t DoubleDunkSettings::getReward() const { return m_reward_p1; }
reward_t DoubleDunkSettings::getRewardP2() const { return m_reward_p2; }

/* is an action part of the minimal set? */
bool DoubleDunkSettings::isMinimal(const Action& a) const {
  switch (a) {
    case PLAYER_A_NOOP:
    case PLAYER_A_FIRE:
    case PLAYER_A_UP:
    case PLAYER_A_RIGHT:
    case PLAYER_A_LEFT:
    case PLAYER_A_DOWN:
    case PLAYER_A_UPRIGHT:
    case PLAYER_A_UPLEFT:
    case PLAYER_A_DOWNRIGHT:
    case PLAYER_A_DOWNLEFT:
    case PLAYER_A_UPFIRE:
    case PLAYER_A_RIGHTFIRE:
    case PLAYER_A_LEFTFIRE:
    case PLAYER_A_DOWNFIRE:
    case PLAYER_A_UPRIGHTFIRE:
    case PLAYER_A_UPLEFTFIRE:
    case PLAYER_A_DOWNRIGHTFIRE:
    case PLAYER_A_DOWNLEFTFIRE:
      return true;
    default:
      return false;
  }
}

/* reset the state of the game */
void DoubleDunkSettings::reset() {
  m_reward_p1 = 0;
  m_reward_p2 = 0;
  m_score = 0;
  no_choice_counter = 0;
  m_terminal = false;
}

/* saves the state of the rom settings */
void DoubleDunkSettings::saveState(Serializer& ser) {
  ser.putInt(m_reward_p1);
  ser.putInt(m_reward_p2);
  ser.putInt(m_score);
  ser.putInt(max_turn_time);
  ser.putInt(no_choice_counter);
  ser.putBool(m_terminal);
}

// loads the state of the rom settings
void DoubleDunkSettings::loadState(Deserializer& ser) {
  m_reward_p1 = ser.getInt();
  m_reward_p2 = ser.getInt();
  m_score = ser.getInt();
  max_turn_time = ser.getInt();
  no_choice_counter = ser.getInt();
  m_terminal = ser.getBool();
}

ActionVect DoubleDunkSettings::getStartingActions() {
  return {PLAYER_A_UPFIRE};
}

// returns a list of mode that the game can be played in
ModeVect DoubleDunkSettings::getAvailableModes() {
  // this game has a menu that allows to define various yes/no options
  // setting these options define in a way a different mode
  // there are 4 relevant options, which makes 2^4=16 available modes
  int single_player_modes = 16;
  ModeVect modes(single_player_modes);
  for (unsigned int i = 0; i < modes.size(); i++) {
    modes[i] = i;
  }
  return modes;
}
ModeVect DoubleDunkSettings::get2PlayerModes() {
  //the same modes as player 1, but + 16
  ModeVect modes(16);
  for (unsigned int i = 16; i < 32; i++) {
    modes[i-16] = i;
  }
  return modes;
}

void DoubleDunkSettings::goDown(
    System& system, std::unique_ptr<StellaEnvironmentWrapper>& environment) {
  // this game has a menu that allows to define various yes/no options
  // this function goes to the next option in the menu
  int previousSelection = readRam(&system, 0xB0);
  while (previousSelection == readRam(&system, 0xB0)) {
    environment->act(PLAYER_A_DOWN, PLAYER_B_NOOP);
    environment->act(PLAYER_A_NOOP, PLAYER_B_NOOP);
  }
}

void DoubleDunkSettings::activateOption(
    System& system, unsigned int bitOfInterest,
    std::unique_ptr<StellaEnvironmentWrapper>& environment) {
  // once we are at the proper option in the menu,
  // if we want to enable it all we have to do is to go right
  while ((readRam(&system, 0x80) & bitOfInterest) != bitOfInterest) {
    environment->act(PLAYER_A_RIGHT, PLAYER_B_NOOP);
    environment->act(PLAYER_A_NOOP, PLAYER_B_NOOP);
  }
}

void DoubleDunkSettings::deactivateOption(
    System& system, unsigned int bitOfInterest,
    std::unique_ptr<StellaEnvironmentWrapper>& environment) {
  // once we are at the proper optio in the menu,
  // if we want to disable it all we have to do is to go left
  while ((readRam(&system, 0x80) & bitOfInterest) == bitOfInterest) {
    environment->act(PLAYER_A_LEFT, PLAYER_B_NOOP);
    environment->act(PLAYER_A_NOOP, PLAYER_B_NOOP);
  }
}

// set the mode of the game
// the given mode must be one returned by the previous function
void DoubleDunkSettings::setMode(
    game_mode_t m, System& system,
    std::unique_ptr<StellaEnvironmentWrapper> environment) {

    environment->pressSelect();

    if(m & 16){
      is_two_player = true;
      activateOption(system, 0x40, environment);
    } else {
      is_two_player = false;
      deactivateOption(system, 0x40, environment);
    }
    goDown(system, environment);
    //discard the second entry (irrelevant)
    goDown(system, environment);

    //deal with the 3 points option
    if (m & 1) {
      activateOption(system, 0x08, environment);
    } else {
      deactivateOption(system, 0x08, environment);
    }

    //deal with the 10 seconds option
    goDown(system, environment);
    if (m & 2) {
      activateOption(system, 0x10, environment);
    } else {
      deactivateOption(system, 0x10, environment);
    }

    //deal with the 3 seconds option
    goDown(system, environment);
    if (m & 4) {
      activateOption(system, 0x04, environment);
    } else {
      deactivateOption(system, 0x04, environment);
    }

    //deal with the foul option
    goDown(system, environment);
    if (m & 8) {
      activateOption(system, 0x20, environment);
    } else {
      deactivateOption(system, 0x20, environment);
    }

    //reset the environment to apply changes.
    environment->softReset();
    //apply starting action

}


void DoubleDunkSettings::modifyEnvironmentSettings(Settings& settings) {
  int default_setting = -1;
  max_turn_time = settings.getInt("max_turn_time");
  if(max_turn_time == default_setting){
    const int DEFAULT_STALL_LIMIT = 60*2;
    max_turn_time = DEFAULT_STALL_LIMIT;
  }
}

}  // namespace ale
