# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['nyan']

package_data = \
{'': ['*']}

install_requires = \
['pygame>=1.9.6,<2.0.0', 'pyinstaller>=3.6,<4.0', 'pytest>=5.2,<6.0']

extras_require = \
{':sys_platform == "win32"': ['pefile>=2019.4.18,<2020.0.0',
                              'pywin32>=227,<228']}

entry_points = \
{'console_scripts': ['nyan-packager = nyan_packager:run']}

setup_kwargs = {
    'name': 'nyan',
    'version': '0.9.1',
    'description': 'The easiest way to start coding games and graphics projects in Python',
    'long_description': '# Nyan\n\n[![PyPI version](https://badge.fury.io/py/nyan.svg)](https://badge.fury.io/py/nyan)\n\nNyan - a fork of [Python Play](https://github.com/replit/play) - is an open-source code library for the Python programming language that makes it as easy as possible to start making games. Here\'s the code to make a simple game using Nyan:\n\n```python\nimport nyan\n\ncat = nyan.new_text(\'=^.^=\', font_size=70)\n\n@nyan.repeat_forever\nasync def move_cat():\n    cat.x = nyan.random_number(-200, 200)\n    cat.y = nyan.random_number(-200, 200)\n    cat.color = nyan.random_color()\n    \n    cat.show()\n\n    await nyan.sleep(seconds=0.4)\n\n    cat.hide()\n\n    await nyan.sleep(seconds=0.4)\n\n@cat.when_clicked\ndef win_function():\n    cat.show()\n    cat.text = \'You won!\'\n\nnyan.start_program()\n```\n\nThe code above makes a game where you have to click the cat to win:\n\n![Clicking a cat game](example.gif)\n\nNyan is an excellent choice for beginner programmers to get started with graphics programming. It was designed to have similar commands and simplicity to [MIT\'s Scratch](https://scratch.mit.edu) and is distinguished from such projects as Pygame, Arcade, or Pygame Zero because of its lack of boiler plate code, its easy-to-understand plain-english commands, and intuitive API.\n\n# How to install Nyan\n\nMake sure that you have Python 3.7 or greater. Run the following command in your terminal:\n```\npip install nyan\n```\n\n# How to use Nyan\n\nAll Nyan programs start with `import nyan` and end with `nyan.start_program()`, like this:\n\n```python\nimport nyan # this is the first line in the program\n\n\n\nnyan.start_program() # this is the last line in the program\n```\n\nAll other commands go between those two commands.\n\n## Commands\n\nThe rest of this document is divided into the following sections:\n\n- [Basic Commands](#basic-commands) - Getting graphics, shapes, and text on the screen. Also changing the backdrop.\n- [Animation and Control Commands](#animation-and-control-commands) - Animating and controlling graphics, shapes, and text.\n- [Sprite Commands](#sprite-commands) - Controlling sprites.\n- [Mouse Commands](#mouse-commands) - Detecting mouse actions (clicks, movement).\n- [Keyboard Commands](#keyboard-commands) - Detecting keyboard actions.\n- [Audio Commands](#audio-commands) - Playing sounds and music.\n- [Other Useful Commands](#other-useful-commands) - General commands.\n- [Packaging Nyan Programs](#packaging-nyan-programs) - Creating an executable that can be shared with other people.\n\n## Basic Commands\n\nTo get images or text on the screen, use the following commands. (Copy and paste the code below to try it out.)\n\n#### `nyan.new_rect()`\n```python\nbox = nyan.new_rect(\n        color=\'black\',\n        x=0,\n        y=0,\n        width=100,\n        height=200,\n        border_color="light blue",\n        border_width=10\n    )\n```\n\nThis will put a tall, black Rectangle in the middle of the screen.\n\nIf you want to change where the image is on the screen, try changing `x=0` (horizontal position) and `y=0` (vertical position). Just like Scratch, the middle of the screen is x=0, y=0. Increasing x moves the image right and decreasing x moves the image left. Likeswise, increasing y moves the image up and decreasing y moves the image down. You can also change the color by changing `\'black\'` to another color name, like `\'orange\'`.\n\n#### `nyan.new_image()`\n```python\ncharacter = nyan.new_image(\n        image=\'character.png\', \n        x=0, \n        y=0, \n        size=100\n    )\n```\n\nThis will place an image in the middle of the screen. Make sure you have a file named `character.png` in a folder named `assets` inside your project for the code above to work. You can find images online at sites like http://icons.iconarchive.com/icons/icojam/animals/64/01-bull-icon.png, or you can make your own using an online sprite editor like this one https://www.piskelapp.com/.\n\n#### `nyan.new_text()`\n```python\ngreeting = nyan.new_text(\n        text=\'hi there\', \n        x=0, \n        y=0, \n        font=None, \n        font_size=50, \n        color=\'black\'\n    )\n```\n\nThis will put some text on the screen.\n\nIf you want to change the font, you\'ll need a font file (usually named something like `Arial.ttf`) in your assets folder. Then you can change `font=None` to `font=\'Arial.ttf\'`. You can find font files at sites like [DaFont](https://www.dafont.com).\n\n#### `nyan.new_circle()`\n```python\nball = nyan.new_circle(\n        color=\'black\', \n        x=0, \n        y=0, \n        radius=100, \n        border_color="light blue", \n        border_width=10\n    )\n```\n\nThis will put a black circle in the middle of the screen.\n\n#### `nyan.set_backdrop()`\nYou can change the background color with the `nyan.set_backdrop()` command:\n\n```python\nnyan.set_backdrop(\'light blue\')\n```\n\nThere are [lots of named colors to choose from](https://upload.wikimedia.org/wikipedia/commons/2/2b/SVG_Recognized_color_keyword_names.svg). Additionally, if you want to set colors by RGB (Red Green Blue) values, you can do that like this:\n\n```python\n# Sets the background to white. Each number can go from 0 to 255\nnyan.set_backdrop((255, 255, 255))\n```\n\nAnywhere you can set a color in Nyan, you can do it using a named color like `\'red\'` or an RGB value above like `(255, 255, 255)` or even an RGBA value like `(0, 0, 0, 127)` (the fourth number is transparency from 0 to 255).\n\n## Animation and Control Commands\n\n#### `@nyan.repeat_forever`\nTo make things move around, you can start by using `@nyan.repeat_forever`, like this:\n\n```python\ncat = nyan.new_text(\'=^.^=\')\n\n@nyan.repeat_forever\ndef do():\n    cat.turn(10)  \n```    \n\nThe above code will make the cat turn around forever. Sprites have other commands that you can see in the next section called Sprite Commands.\n\n#### `@nyan.when_program_starts`\nTo make some code run just at the beginning of your project, use `@nyan.when_program_starts`, like this:\n\n```python\ncat = nyan.new_text(\'=^.^=\')\n\n@nyan.when_program_starts\ndef do():\n    cat.turn(180)  \n```\n\nThis will make the cat turn upside down instantly when the program starts.\n\n#### `await nyan.sleep(seconds=1)`\nTo run code after a waiting period, you can use the `await nyan.sleep()` command like this:\n\n```python\ncat = nyan.new_text(\'=^.^=\')\n\n@nyan.when_program_starts\nasync def do():\n    cat.turn(180)  \n    await nyan.sleep(seconds=2)\n    cat.turn(180)  \n```\n\nThis will make the cat turn upside down instantly when the program starts, wait 2 seconds, then turn back up again.\n\n#### `nyan.broadcast()`\nYou can use `nyan.broadcast()` to broadcast an event that can be picked up by an event listener.\n```python\n@nyan.repeat_forever\ndef do():\n    nyan.broadcast(\'marco\')\n```\n\n#### `@nyan.when_event_recieved()`\nTo run code in response to a broadcasted event, use `@nyan.when_event_recieved()`, like this\n```python\n@nyan.when_event_recieved(\'marco\')\ndef do():\n    print(\'polo\')\n```\n\n#### `@nyan.foreach_sprite()`\nattaches a script to each sprite passed or in the case a tag is passed, for each sprite that has the given tag. Should be used in conjunction with `@nyan.repeat_forever` and `@nyan.when_program_starts` decorators. Example:\n\n```python\n@nyan.repeat_forever\n@nyan.foreach_sprite(player1, player2, player3, player4)\nasync def animate_player(player):\n    next_frame(player)\n    await nyan.sleep(player.frameTime)\n\n@nyan.repeat_forever\n@nyan.foreach_sprite(tag=\'player-missile\')\nasync def propel_missile(missile):\n    missile.move(25)\n```\n\n## Sprite Commands\n\n#### Simple commands\n\nSprites (images and text) have a few simple commands:\n\n- **`sprite.move(10)`** — moves the sprite 10 pixels in the direction it\'s facing (starts facing right). Use negative numbers (-10) to go backward.\n- **`sprite.move(10, direction=45)`** — moves the sprite 10 pixels at an angle specified by the direction.\n- **`sprite.turn(20)`** — Turns the sprite 20 degrees counter-clockwise. Use negative numbers (-20) to turn the other way.\n- **`sprite.go_to(other_sprite)`** — Makes `sprite` jump to another sprite named `other_sprite`\'s position on the screen. Can also be used to make the sprite follow the mouse: `sprite.go_to(nyan.mouse)`.\n- **`sprite.go_to(x=100, y=50)`** — Makes `sprite` jump to x=100, y=50 (right and up a little).\n- **`sprite.point_towards(other_sprite)`** — Turns `sprite` so it points at another sprite called `other_sprite`.\n- **`sprite.point_towards(x=100, y=50)`** — Turns `sprite` so it points toward x=100, y=50 (right and up a little).\n- **`sprite.hide()`** — Hides `sprite`. It can\'t be clicked when it\'s hidden.\n- **`sprite.show()`** — Shows `sprite` if it\'s hidden.\n- **`sprite.add_tag(\'car\')`** — Adds a tag named `car` to the sprite. Tagging a sprite adds it to a group that can be fetched later by other functions.\n- **`sprite.remove_tag(\'car\')`** — removes the `car` tag from the sprite\n- **`sprite.clone()`** — Makes a copy or clone of the sprite and returns it.\n- **`sprite.remove()`** — Removes a sprite from the screen permanently. Calling sprite commands on a removed sprite won\'t do anything.\n\n#### Properties\n\nSprites also have properties that can be changed to change how the sprite looks. Here they are:\n\n- **`sprite.x`** — The sprite\'s horizontal position on the screen. Positive numbers are right, negative numbers are left. The default is 0.\n- **`sprite.y`** — The sprite\'s vertical position on the screen. Positive numbers are up, negative numbers are down. The default is 0.\n- **`sprite.size`** — How big the sprite is. The default is 100, but it can be made bigger or smaller.\n- **`sprite.angle`** — How much the sprite is turned. Positive numbers are counter-clockwise. The default is 0 degrees (pointed to the right).\n- **`sprite.transparency`** — How see-through the sprite is from 0 to 100. 0 is completely see-through, 100 is not see-through at all. The default is 100.\n- **`sprite.brightness`** - How bright or dim the sprite is from -100 to 100. -100 is completely dark, 100 is white. The default is 0.\n- **`sprite.is_hidden`** — `True` if the sprite has been hidden with the `sprite.hide()` command. Otherwise `False`.\n- **`sprite.is_shown`** — `True` if the sprite has not been hidden with the `sprite.hide()` command. Otherwise `False`.\n- **`sprite.left`** — The x position of the left-most part of the sprite.\n- **`sprite.right`** — The x position of the right-most part of the sprite.\n- **`sprite.top`** — The y position of the top-most part of the sprite.\n- **`sprite.bottom`** — The y position of the bottom-most part of the sprite.\n\nImage-sprite-only properties:\n\n- **`sprite.image`** — The filename of the image shown.\n\nText-sprite-only properties:\n\n- **`text.text`** — The displayed text content.\n- **`text.font`** — The filename of the font e.g. \'Arial.ttf\'. The default is `None`, which will use a built-in font.\n- **`text.font_size`** — The text\'s size. The default is `50` (pt).\n- **`text.color`** — The text\'s color. The default is black.\n\nBox-sprite-only properties:\n- **`box.color`** — The color filling the box. The default is `black`.\n- **`box.width`** — The width of the box. The default is `100` pixels.\n- **`box.height`** — The height of the box. The default is `200` pixels.\n- **`box.border_width`** — The width of the box\'s border, the line around it. The default is `0`.\n- **`box.border_color`** — The color of the box\'s border. The default is `\'light blue\'`.\n\nIf the box has a border, the box\'s total width, including the border, will be the width defined by the `width` property.\n\nCircle-sprite-only properties:\n- **`circle.color`** — The color filling the circle. The default is `black`.\n- **`circle.radius`** — How big the circle is, measured from the middle to the outside. The default is `100` pixels, making a 200-pixel-wide circle.\n- **`circle.border_width`** — The width of the circle\'s border, the line around it. The default is `0`.\n- **`circle.border_color`** — The color of the circle\'s border. The default is `\'light blue\'`.\n\nIf the circle has a border, the circle\'s total width, including the border, will be the width defined by the `radius` property.\n\nThese properties can changed to do the same things as the sprite commands above. For example,\n\n```python\nsprite.go_to(other_sprite)\n\n# the line above is the same as the two lines below\nsprite.x = other_sprite.x\nsprite.y = other_sprite.y\n```\n\nYou can change the properties to animate the sprites. The code below makes the cat turn around.\n\n```python\ncat = nyan.new_text(\'=^.^=\')\n\n@nyan.repeat_forever\ndef do():\n    cat.angle += 1\n    # the line above is the same as cat.turn(1)\n```\n\n#### Other info\n\nSprites also have some other useful info:\n\n- **`sprite.width`** — Gets how wide the sprite is in pixels.\n- **`sprite.height`** — Gets how tall the sprite is in pixels.\n- **`sprite.distance_to(other_sprite)`** — Gets the distance in pixels to `other_sprite`.\n- **`sprite.distance_to(x=100, y=100)`** — Gets the distance to the point x=100, y=100.\n- **`sprite.is_touching(other_sprite)`** — Returns True if `sprite` is touching the `other_sprite`. Otherwise `False`.\n- **`sprite.is_touching(point)`** — Returns True if the sprite is touching the point (anything with an `x` and `y` coordinate). For example: `sprite.is_touching(nyan.mouse)`\n\n## Mouse Commands\n\nWorking with the mouse in Nyan is easy. Here\'s a simple program that points a sprite at the mouse:\n\n```python\narrow = nyan.new_text(\'-->\', font_size=100)\n\n@nyan.repeat_forever\ndef do():\n    arrow.point_towards(nyan.mouse)\n```\n\n`nyan.mouse` has the following properties:\n\n- **`nyan.mouse.x`** — The horizontal x position of the mouse.\n- **`nyan.mouse.y`** — The vertical y position of the mouse.\n- **`nyan.mouse.is_clicked`** — `True` if the mouse is clicked down, or `False` if it\'s not.\n- **`nyan.mouse.is_touching(sprite)`** — Returns `True` if the mouse is touching a sprite, or `False` if it\'s not.\n\n#### `@sprite.when_clicked`\n\nProbably the easiest way to detect clicks is to use `@sprite.when_clicked`.\n\nIn the program below, when the face is clicked it changes for 1 second then turns back to normal:\n\n```python\nface = nyan.new_text(\'^.^\', font_size=100)\n\n@face.when_clicked\nasync def do():\n    face.text = \'*o*\'\n    await nyan.sleep(seconds=1)\n    face.text = \'^.^\'\n```\n\n#### `@nyan.when_sprite_clicked()`\n\nIf you wanted to run the same code when multiple sprites are clicked, you can use `@nyan.when_sprite_clicked()`:\n\n```python\nface1 = nyan.new_text(\'^.^\', x=-100, font_size=100)\nface2 = nyan.new_text(\'^_^\', x=100, font_size=100)\n\n@nyan.when_sprite_clicked(face1, face2) # takes as many sprites as you want\nasync def do(sprite):\n    starting_words = sprite.text\n    sprite.text = \'*o*\'\n    await play.sleep(seconds=1)\n    sprite.text = starting_words\n```\n\n#### `@nyan.mouse.when_clicked` or `@nyan.when_mouse_clicked`\n\nTo run code when the mouse is clicked anywhere, use `@nyan.mouse.when_clicked` or `@nyan.when_mouse_clicked` (they do the same exact thing).\n\nIn the code below, when a click is detected, the text will move to the click location and the coordinates will be shown:\n\n```python\ntext = nyan.new_text(\'0, 0\')\n\n@nyan.mouse.when_clicked\ndef do():\n    text.text = f\'{nyan.mouse.x}, {nyan.mouse.y}\'\n    text.go_to(nyan.mouse)\n```\n\n#### `@nyan.mouse.when_click_released` or `@nyan.when_mouse_click_released`\n\nTo run code when the mouse button is released, use `@nyan.mouse.when_click_released` `@nyan.when_mouse_click_released` (they do the same exact thing).\n\nIn the code below, the cat can be dragged around when it\'s clicked by the mouse:\n\n```python\ncat = nyan.new_text(\'=^.^= drag me!\')\ncat.is_being_dragged = False\n\n@cat.when_clicked\ndef do():\n    cat.is_being_dragged = True\n\n@nyan.mouse.when_click_released\ndef do():\n    cat.is_being_dragged = False\n\n@nyan.repeat_forever\ndef do():\n    if cat.is_being_dragged:\n        cat.go_to(nyan.mouse)\n```\n\n## Keyboard Commands\n\n#### `nyan.key_is_pressed()`\n\nYou can use `nyan.key_is_pressed()` to detect keypresses.\n\nIn the code below, pressing the `arrow` keys or `w/a/s/d` will make the cat go in the desired direction.\n\n```python\ncat = nyan.new_text(\'=^.^=\')\n\n@nyan.repeat_forever\ndef do():\n    if nyan.key_is_pressed(\'up\', \'w\'):\n        cat.y += 15\n    if nyan.key_is_pressed(\'down\', \'s\'):\n        cat.y -= 15\n\n    if nyan.key_is_pressed(\'right\', \'d\'):\n        cat.x += 15\n    if nyan.key_is_pressed(\'left\', \'a\'):\n        cat.x -= 15\n```\n\n#### `@nyan.when_key_pressed()`\n\nYou can use `@nyan.when_key_pressed()` to run code when specific keys are pressed.\n\nIn the code below, pressing the `space` key will change the cat\'s face, and pressing the `enter` key will change it to a different face.\n\n```python\ncat = nyan.new_text(\'=^.^=\')\n\n@nyan.when_key_pressed(\'space\', \'enter\') # if either the space key or enter key are pressed...\ndef do(key):\n    if key == \'enter\':\n        cat.text = \'=-.-=\'\n    if key == \'space\':\n        cat.text = \'=*_*=\'\n```\n\n#### `@nyan.when_any_key_pressed`\n\nIf you just want to detect when any key is pressed, you can use `@nyan.when_any_key_pressed`.\n\nIn the code below, any key you press will be displayed on the screen:\n\n```python\ntext = nyan.new_text(\'\')\n\n@nyan.when_any_key_pressed\ndef do(key):\n    text.text = f\'{key} pressed!\'\n```\n\n#### `@nyan.when_key_released()`\n\nExactly like `@nyan.when_key_pressed()` but runs the code when specific keys are released.\n\nIn the code below, text will appear on screen only if the `up` arrow is pressed.\n\n```python\ntext = nyan.new_text(\'\')\n\n@nyan.when_key_released(\'up\')\nasync def do(key):\n    text.text = \'up arrow released!\'\n    await nyan.sleep(seconds=1)\n    text.text = \'\'\n```\n\n#### `@nyan.when_any_key_released`\n\nExactly like `@nyan.when_any_key_pressed` but runs the code when any key is released.\n\nIn the code below, the name of the most recently released key will show up on screen.\n\n```python\ntext = nyan.new_text(\'\')\n\n@nyan.when_any_key_pressed\ndef do(key):\n    text.text = f\'{key} key released!\'\'\n```\n\n## Audio Commands\n\n#### `nyan.new_sound()`\n```python\nmeow = nyan.new_sound(sound=\'meow.wav\')\n```\nThis will load a sound file with `.wav` extension and makes it ready to be played. This function should be mainly used for sound effects. You can get sounds from this site http://soundbible.com/ or you can generate your own using this tool http://www.bfxr.net/.\n\n#### `meow.play()`\nThis will play the loaded sound file.\n```python\nmeow = nyan.new_sound(sound=\'meow.wav\')\n\n@nyan.when_key_pressed(\'space\')\ndef do():\n    meow.play()\n```\n\n#### `nyan.music.play()`\nTo play an mp3 file in the background, use `nyan.music.play()`, like this\n```python\nnyan.music.play(\'William Tell Overture Finale.mp3\', loop=True)\n```\n\n#### `await nyan.music.play_until_done()`\nThis is the same as `nyan.music.play()`, but waits for the music to finish before continuing\n```python\n@nyan.when_program_starts\nasync def do():\n    await nyan.music.play_until_done(\'William Tell Overture Finale.mp3\')\n    print(\'music has ended\')\n```\n\n#### `nyan.music.pause()`\nPauses the currently playing music\n```python\n@nyan.when_program_starts\nasync def do():\n    nyan.music.play(\'William Tell Overture Finale.mp3\')\n    await nyan.sleep(seconds=5)\n    nyan.music.pause()\n```\n\n#### `nyan.music.unpause()`\nResumes the paused music\n```python\n@nyan.when_program_starts\nasync def do():\n    nyan.music.play(\'William Tell Overture Finale.mp3\')\n    await nyan.sleep(seconds=5)\n    nyan.music.pause()\n    await nyan.sleep(seconds=5)\n    nyan.music.unpause()\n```\n\n#### `nyan.music.stop()`\nStops the currently playing music\n```python\n@nyan.when_program_starts\nasync def do():\n    nyan.music.play(\'William Tell Overture Finale.mp3\')\n    await nyan.sleep(seconds=5)\n    nyan.music.stop()\n```\n\n#### `nyan.music.volume`\nYou can use `nyan.music.volume` to get or set the current volume which ranges from 0 to 100. The default is 100\n```python\n@nyan.when_program_starts\nasync def do():\n    nyan.music.play(\'William Tell Overture Finale.mp3\')\n    await nyan.sleep(seconds=5)\n    nyan.music.volume = 70\n```\n\n## Other Useful Commands\n\n#### `nyan.screen`\n\nThe way to get information about the screen. `nyan.screen` has these properties:\n\n- `nyan.screen.width` - Defaults to 800 (pixels total). Changing this will change the screen\'s size.\n- `nyan.screen.height` - Defaults to 600 (pixels total). Changing this will change the screen\'s size.\n- `nyan.screen.left` - The `x` coordinate for the left edge of the screen.\n- `nyan.screen.right` - The `x` coordinate for the right edge of the screen.\n- `nyan.screen.top` - The `y` coordinate for the top of the screen.\n- `nyan.screen.bottom` - The `y` coordinate for the bottom of the screen.\n\n#### `nyan.get_sprites()`\n\nReturns a list of all the sprites (images, shapes, text) in the program. Takes an optional tag parameter that can be used to get all sprites that have a give tag.\n\n#### `nyan.random_number()`\n\nA function that makes random numbers.\n\nIf two whole numbers are given, `nyan.random_number()` will give a whole number back:\n\n```python\nnyan.random_number(lowest=0, highest=100)\n\n# example return value: 42\n```\n(You can also do `nyan.random_number(0, 100)` without `lowest` and `highest`.)\n\nIf non-whole numbers are given, non-whole numbers are given back:\n\n```python\nnyan.random_number(0, 1.0)\n# example return value: 0.84\n```\n\n`nyan.random_number()` is also inclusive, which means `nyan.random_number(0,1)` will return `0` and `1`.\n\n#### `nyan.random_color()`\n\nReturns a random RGB color, including white and black.\n\n```python\nnyan.random_color()\n# example return value: (201, 17, 142)\n```\n\nEach value varies from 0 to 255.\n\n#### `nyan.random_position()`\n\nReturns a random position on the screen. A position object has an `x` and `y` component.\n\n```python\ntext = nyan.text(\'WOO\')\n@nyan.repeat_forever\ndef do():\n    text.go_to(nyan.random_position())\n\n    # the above is equivalent to:\n    position = nyan.random_position()\n    text.x = position.x\n    text.y = position.y\n```\n\n#### `nyan.random_item()`\n\nPicks a random item from a list. The passed list should contain at least one item.\n\n```python\ntext = nyan.text(\'WOO\')\n@nyan.repeat_forever\ndef do():\n    random_angle = nyan.random_item([0, 90, 180, 270])\n    text.angle = random_angle\n```\n\n#### `nyan.new_timer()`\nCreates a timer. Useful for keeping track of time and for doing animations.\n\n```python\ntimer = new_timer()\n```\n\nOnce a timer is created, it will have the following commands and properties:\n\n- **`timer.reset()`** — resets the timer\n- **`timer.seconds`** — time elapsed in seconds since creation or last reset\n- **`timer.milliseconds`** — time elapsed in milliseconds since creation or last reset\n\n## Packaging Nyan Programs\nNyan comes with a packager that can be used to create stand-alone executables that can be shared with other people without them having to install Python on their computers. To use it, run the following in your terminal\n\n```\nnyan-packager python_file\n```\n\nThis will create an executable file with other stuff in a dist folder inside your project. You can change the icon of the executable with `-icon icon_file` option.\n\n## What\'s with all this `async`/`await` stuff? Is this Python?\n\nYes, this is Python! Python added `async` and `await` as special keywords in Python 3.7. It\'s part of the [asyncio module](https://docs.python.org/3/library/asyncio.html).\n\nUsing async functions means we can use the `await nyan.sleep()` function, which makes some code a lot simpler and appear to run in-parallel, which new programmers find intuitive.\n\n```python\nimport nyan\n\ncat = nyan.new_text(\'=^.^=\')\n\n# this code block uses async so it can use the \'await nyan.sleep()\' function\n@nyan.repeat_forever\nasync def change_bg():\n    nyan.set_backdrop(\'pink\')\n    await nyan.sleep(seconds=1)\n\n    nyan.set_backdrop(\'purple\')\n    await nyan.sleep(seconds=1)\n\n    nyan.set_backdrop(\'light blue\')\n    await nyan.sleep(seconds=1)\n\n# this code block doesn\'t need async because it doesn\'t have `await nyan.sleep()`\n@nyan.repeat_forever\ndef do():\n    cat.turn(1)\n\nnyan.start_program()\n```\n\nIn the above program, the backdrop will change and the cat will appear to turn at the same time even though the code is running single-threaded.\n\nThe `async` keyword isn\'t necessary to write unless you want to use `await` functions. If you try to use an `await` command inside a non-async function, Python will show you an error like this:\n\n```  \n  File "example.py", line 31\n    await nyan.sleep(seconds=1)\n    ^\nSyntaxError: \'await\' outside async function\n```\nTo fix that error, just put `async` before `def`.\n\nIf you don\'t understand any of this, it\'s generally safe to just include `async` before `def`.\n',
    'author': 'ducaale',
    'author_email': 'sharaf.13@hotmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/ducaale/nyan',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
