import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import { IResource, RemovalPolicy, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { LogStream } from './log-stream';
import { MetricFilter } from './metric-filter';
import { IFilterPattern } from './pattern';
import { ILogSubscriptionDestination, SubscriptionFilter } from './subscription-filter';
/**
 * @stability stable
 */
export interface ILogGroup extends IResource {
    /**
     * The ARN of this log group, with ':*' appended.
     *
     * @stability stable
     * @attribute true
     */
    readonly logGroupArn: string;
    /**
     * The name of this log group.
     *
     * @stability stable
     * @attribute true
     */
    readonly logGroupName: string;
    /**
     * Create a new Log Stream for this Log Group.
     *
     * @param id Unique identifier for the construct in its parent.
     * @param props Properties for creating the LogStream.
     * @stability stable
     */
    addStream(id: string, props?: StreamOptions): LogStream;
    /**
     * Create a new Subscription Filter on this Log Group.
     *
     * @param id Unique identifier for the construct in its parent.
     * @param props Properties for creating the SubscriptionFilter.
     * @stability stable
     */
    addSubscriptionFilter(id: string, props: SubscriptionFilterOptions): SubscriptionFilter;
    /**
     * Create a new Metric Filter on this Log Group.
     *
     * @param id Unique identifier for the construct in its parent.
     * @param props Properties for creating the MetricFilter.
     * @stability stable
     */
    addMetricFilter(id: string, props: MetricFilterOptions): MetricFilter;
    /**
     * Extract a metric from structured log events in the LogGroup.
     *
     * Creates a MetricFilter on this LogGroup that will extract the value
     * of the indicated JSON field in all records where it occurs.
     *
     * The metric will be available in CloudWatch Metrics under the
     * indicated namespace and name.
     *
     * @param jsonField JSON field to extract (example: '$.myfield').
     * @param metricNamespace Namespace to emit the metric under.
     * @param metricName Name to emit the metric under.
     * @returns A Metric object representing the extracted metric
     * @stability stable
     */
    extractMetric(jsonField: string, metricNamespace: string, metricName: string): cloudwatch.Metric;
    /**
     * Give permissions to write to create and write to streams in this log group.
     *
     * @stability stable
     */
    grantWrite(grantee: iam.IGrantable): iam.Grant;
    /**
     * Give the indicated permissions on this log group and all streams.
     *
     * @stability stable
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
}
/**
 * An CloudWatch Log Group
 */
declare abstract class LogGroupBase extends Resource implements ILogGroup {
    /**
     * The ARN of this log group, with ':*' appended
     */
    abstract readonly logGroupArn: string;
    /**
     * The name of this log group
     */
    abstract readonly logGroupName: string;
    /**
     * Create a new Log Stream for this Log Group.
     *
     * @param id Unique identifier for the construct in its parent.
     * @param props Properties for creating the LogStream.
     * @stability stable
     */
    addStream(id: string, props?: StreamOptions): LogStream;
    /**
     * Create a new Subscription Filter on this Log Group.
     *
     * @param id Unique identifier for the construct in its parent.
     * @param props Properties for creating the SubscriptionFilter.
     * @stability stable
     */
    addSubscriptionFilter(id: string, props: SubscriptionFilterOptions): SubscriptionFilter;
    /**
     * Create a new Metric Filter on this Log Group.
     *
     * @param id Unique identifier for the construct in its parent.
     * @param props Properties for creating the MetricFilter.
     * @stability stable
     */
    addMetricFilter(id: string, props: MetricFilterOptions): MetricFilter;
    /**
     * Extract a metric from structured log events in the LogGroup.
     *
     * Creates a MetricFilter on this LogGroup that will extract the value
     * of the indicated JSON field in all records where it occurs.
     *
     * The metric will be available in CloudWatch Metrics under the
     * indicated namespace and name.
     *
     * @param jsonField JSON field to extract (example: '$.myfield').
     * @param metricNamespace Namespace to emit the metric under.
     * @param metricName Name to emit the metric under.
     * @returns A Metric object representing the extracted metric
     * @stability stable
     */
    extractMetric(jsonField: string, metricNamespace: string, metricName: string): cloudwatch.Metric;
    /**
     * Give permissions to create and write to streams in this log group.
     *
     * @stability stable
     */
    grantWrite(grantee: iam.IGrantable): iam.Grant;
    /**
     * Give the indicated permissions on this log group and all streams.
     *
     * @stability stable
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
}
/**
 * How long, in days, the log contents will be retained.
 *
 * @stability stable
 */
export declare enum RetentionDays {
    /**
     * 1 day.
     *
     * @stability stable
     */
    ONE_DAY = 1,
    /**
     * 3 days.
     *
     * @stability stable
     */
    THREE_DAYS = 3,
    /**
     * 5 days.
     *
     * @stability stable
     */
    FIVE_DAYS = 5,
    /**
     * 1 week.
     *
     * @stability stable
     */
    ONE_WEEK = 7,
    /**
     * 2 weeks.
     *
     * @stability stable
     */
    TWO_WEEKS = 14,
    /**
     * 1 month.
     *
     * @stability stable
     */
    ONE_MONTH = 30,
    /**
     * 2 months.
     *
     * @stability stable
     */
    TWO_MONTHS = 60,
    /**
     * 3 months.
     *
     * @stability stable
     */
    THREE_MONTHS = 90,
    /**
     * 4 months.
     *
     * @stability stable
     */
    FOUR_MONTHS = 120,
    /**
     * 5 months.
     *
     * @stability stable
     */
    FIVE_MONTHS = 150,
    /**
     * 6 months.
     *
     * @stability stable
     */
    SIX_MONTHS = 180,
    /**
     * 1 year.
     *
     * @stability stable
     */
    ONE_YEAR = 365,
    /**
     * 13 months.
     *
     * @stability stable
     */
    THIRTEEN_MONTHS = 400,
    /**
     * 18 months.
     *
     * @stability stable
     */
    EIGHTEEN_MONTHS = 545,
    /**
     * 2 years.
     *
     * @stability stable
     */
    TWO_YEARS = 731,
    /**
     * 5 years.
     *
     * @stability stable
     */
    FIVE_YEARS = 1827,
    /**
     * 10 years.
     *
     * @stability stable
     */
    TEN_YEARS = 3653,
    /**
     * Retain logs forever.
     *
     * @stability stable
     */
    INFINITE = 9999
}
/**
 * Properties for a LogGroup.
 *
 * @stability stable
 */
export interface LogGroupProps {
    /**
     * The KMS Key to encrypt the log group with.
     *
     * @default - log group is encrypted with the default master key
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * Name of the log group.
     *
     * @default Automatically generated
     * @stability stable
     */
    readonly logGroupName?: string;
    /**
     * How long, in days, the log contents will be retained.
     *
     * To retain all logs, set this value to RetentionDays.INFINITE.
     *
     * @default RetentionDays.TWO_YEARS
     * @stability stable
     */
    readonly retention?: RetentionDays;
    /**
     * Determine the removal policy of this log group.
     *
     * Normally you want to retain the log group so you can diagnose issues
     * from logs even after a deployment that no longer includes the log group.
     * In that case, use the normal date-based retention policy to age out your
     * logs.
     *
     * @default RemovalPolicy.Retain
     * @stability stable
     */
    readonly removalPolicy?: RemovalPolicy;
}
/**
 * Define a CloudWatch Log Group.
 *
 * @stability stable
 */
export declare class LogGroup extends LogGroupBase {
    /**
     * Import an existing LogGroup given its ARN.
     *
     * @stability stable
     */
    static fromLogGroupArn(scope: Construct, id: string, logGroupArn: string): ILogGroup;
    /**
     * Import an existing LogGroup given its name.
     *
     * @stability stable
     */
    static fromLogGroupName(scope: Construct, id: string, logGroupName: string): ILogGroup;
    /**
     * The ARN of this log group.
     *
     * @stability stable
     */
    readonly logGroupArn: string;
    /**
     * The name of this log group.
     *
     * @stability stable
     */
    readonly logGroupName: string;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: LogGroupProps);
}
/**
 * Properties for a new LogStream created from a LogGroup.
 *
 * @stability stable
 */
export interface StreamOptions {
    /**
     * The name of the log stream to create.
     *
     * The name must be unique within the log group.
     *
     * @default Automatically generated
     * @stability stable
     */
    readonly logStreamName?: string;
}
/**
 * Properties for a new SubscriptionFilter created from a LogGroup.
 *
 * @stability stable
 */
export interface SubscriptionFilterOptions {
    /**
     * The destination to send the filtered events to.
     *
     * For example, a Kinesis stream or a Lambda function.
     *
     * @stability stable
     */
    readonly destination: ILogSubscriptionDestination;
    /**
     * Log events matching this pattern will be sent to the destination.
     *
     * @stability stable
     */
    readonly filterPattern: IFilterPattern;
}
/**
 * Properties for a MetricFilter created from a LogGroup.
 *
 * @stability stable
 */
export interface MetricFilterOptions {
    /**
     * Pattern to search for log events.
     *
     * @stability stable
     */
    readonly filterPattern: IFilterPattern;
    /**
     * The namespace of the metric to emit.
     *
     * @stability stable
     */
    readonly metricNamespace: string;
    /**
     * The name of the metric to emit.
     *
     * @stability stable
     */
    readonly metricName: string;
    /**
     * The value to emit for the metric.
     *
     * Can either be a literal number (typically "1"), or the name of a field in the structure
     * to take the value from the matched event. If you are using a field value, the field
     * value must have been matched using the pattern.
     *
     * If you want to specify a field from a matched JSON structure, use '$.fieldName',
     * and make sure the field is in the pattern (if only as '$.fieldName = *').
     *
     * If you want to specify a field from a matched space-delimited structure,
     * use '$fieldName'.
     *
     * @default "1"
     * @stability stable
     */
    readonly metricValue?: string;
    /**
     * The value to emit if the pattern does not match a particular event.
     *
     * @default No metric emitted.
     * @stability stable
     */
    readonly defaultValue?: number;
}
export {};
