"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogGroup = exports.RetentionDays = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const log_stream_1 = require("./log-stream");
const logs_generated_1 = require("./logs.generated");
const metric_filter_1 = require("./metric-filter");
const pattern_1 = require("./pattern");
const subscription_filter_1 = require("./subscription-filter");
/**
 * An CloudWatch Log Group
 */
class LogGroupBase extends core_1.Resource {
    /**
     * Create a new Log Stream for this Log Group.
     *
     * @param id Unique identifier for the construct in its parent.
     * @param props Properties for creating the LogStream.
     * @stability stable
     */
    addStream(id, props = {}) {
        return new log_stream_1.LogStream(this, id, {
            logGroup: this,
            ...props,
        });
    }
    /**
     * Create a new Subscription Filter on this Log Group.
     *
     * @param id Unique identifier for the construct in its parent.
     * @param props Properties for creating the SubscriptionFilter.
     * @stability stable
     */
    addSubscriptionFilter(id, props) {
        return new subscription_filter_1.SubscriptionFilter(this, id, {
            logGroup: this,
            ...props,
        });
    }
    /**
     * Create a new Metric Filter on this Log Group.
     *
     * @param id Unique identifier for the construct in its parent.
     * @param props Properties for creating the MetricFilter.
     * @stability stable
     */
    addMetricFilter(id, props) {
        return new metric_filter_1.MetricFilter(this, id, {
            logGroup: this,
            ...props,
        });
    }
    /**
     * Extract a metric from structured log events in the LogGroup.
     *
     * Creates a MetricFilter on this LogGroup that will extract the value
     * of the indicated JSON field in all records where it occurs.
     *
     * The metric will be available in CloudWatch Metrics under the
     * indicated namespace and name.
     *
     * @param jsonField JSON field to extract (example: '$.myfield').
     * @param metricNamespace Namespace to emit the metric under.
     * @param metricName Name to emit the metric under.
     * @returns A Metric object representing the extracted metric
     * @stability stable
     */
    extractMetric(jsonField, metricNamespace, metricName) {
        new metric_filter_1.MetricFilter(this, `${metricNamespace}_${metricName}`, {
            logGroup: this,
            metricNamespace,
            metricName,
            filterPattern: pattern_1.FilterPattern.exists(jsonField),
            metricValue: jsonField,
        });
        return new cloudwatch.Metric({ metricName, namespace: metricNamespace }).attachTo(this);
    }
    /**
     * Give permissions to create and write to streams in this log group.
     *
     * @stability stable
     */
    grantWrite(grantee) {
        return this.grant(grantee, 'logs:CreateLogStream', 'logs:PutLogEvents');
    }
    /**
     * Give the indicated permissions on this log group and all streams.
     *
     * @stability stable
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            // A LogGroup ARN out of CloudFormation already includes a ':*' at the end to include the log streams under the group.
            // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-logs-loggroup.html#w2ab1c21c10c63c43c11
            resourceArns: [this.logGroupArn],
            scope: this,
        });
    }
}
/**
 * How long, in days, the log contents will be retained.
 *
 * @stability stable
 */
var RetentionDays;
(function (RetentionDays) {
    RetentionDays[RetentionDays["ONE_DAY"] = 1] = "ONE_DAY";
    RetentionDays[RetentionDays["THREE_DAYS"] = 3] = "THREE_DAYS";
    RetentionDays[RetentionDays["FIVE_DAYS"] = 5] = "FIVE_DAYS";
    RetentionDays[RetentionDays["ONE_WEEK"] = 7] = "ONE_WEEK";
    RetentionDays[RetentionDays["TWO_WEEKS"] = 14] = "TWO_WEEKS";
    RetentionDays[RetentionDays["ONE_MONTH"] = 30] = "ONE_MONTH";
    RetentionDays[RetentionDays["TWO_MONTHS"] = 60] = "TWO_MONTHS";
    RetentionDays[RetentionDays["THREE_MONTHS"] = 90] = "THREE_MONTHS";
    RetentionDays[RetentionDays["FOUR_MONTHS"] = 120] = "FOUR_MONTHS";
    RetentionDays[RetentionDays["FIVE_MONTHS"] = 150] = "FIVE_MONTHS";
    RetentionDays[RetentionDays["SIX_MONTHS"] = 180] = "SIX_MONTHS";
    RetentionDays[RetentionDays["ONE_YEAR"] = 365] = "ONE_YEAR";
    RetentionDays[RetentionDays["THIRTEEN_MONTHS"] = 400] = "THIRTEEN_MONTHS";
    RetentionDays[RetentionDays["EIGHTEEN_MONTHS"] = 545] = "EIGHTEEN_MONTHS";
    RetentionDays[RetentionDays["TWO_YEARS"] = 731] = "TWO_YEARS";
    RetentionDays[RetentionDays["FIVE_YEARS"] = 1827] = "FIVE_YEARS";
    RetentionDays[RetentionDays["TEN_YEARS"] = 3653] = "TEN_YEARS";
    RetentionDays[RetentionDays["INFINITE"] = 9999] = "INFINITE";
})(RetentionDays = exports.RetentionDays || (exports.RetentionDays = {}));
/**
 * Define a CloudWatch Log Group.
 *
 * @stability stable
 */
class LogGroup extends LogGroupBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b;
        super(scope, id, {
            physicalName: props.logGroupName,
        });
        let retentionInDays = props.retention;
        if (retentionInDays === undefined) {
            retentionInDays = RetentionDays.TWO_YEARS;
        }
        if (retentionInDays === Infinity || retentionInDays === RetentionDays.INFINITE) {
            retentionInDays = undefined;
        }
        if (retentionInDays !== undefined && !core_1.Token.isUnresolved(retentionInDays) && retentionInDays <= 0) {
            throw new Error(`retentionInDays must be positive, got ${retentionInDays}`);
        }
        const resource = new logs_generated_1.CfnLogGroup(this, 'Resource', {
            kmsKeyId: (_b = props.encryptionKey) === null || _b === void 0 ? void 0 : _b.keyArn,
            logGroupName: this.physicalName,
            retentionInDays,
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.logGroupArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'logs',
            resource: 'log-group',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.logGroupName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Import an existing LogGroup given its ARN.
     *
     * @stability stable
     */
    static fromLogGroupArn(scope, id, logGroupArn) {
        const baseLogGroupArn = logGroupArn.replace(/:\*$/, '');
        class Import extends LogGroupBase {
            constructor() {
                super(...arguments);
                this.logGroupArn = `${baseLogGroupArn}:*`;
                this.logGroupName = core_1.Stack.of(scope).parseArn(baseLogGroupArn, ':').resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing LogGroup given its name.
     *
     * @stability stable
     */
    static fromLogGroupName(scope, id, logGroupName) {
        const baseLogGroupName = logGroupName.replace(/:\*$/, '');
        class Import extends LogGroupBase {
            constructor() {
                super(...arguments);
                this.logGroupName = baseLogGroupName;
                this.logGroupArn = core_1.Stack.of(scope).formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    sep: ':',
                    resourceName: baseLogGroupName + ':*',
                });
            }
        }
        return new Import(scope, id);
    }
}
exports.LogGroup = LogGroup;
_a = JSII_RTTI_SYMBOL_1;
LogGroup[_a] = { fqn: "@aws-cdk/aws-logs.LogGroup", version: "1.90.0" };
//# sourceMappingURL=data:application/json;base64,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