"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SpaceDelimitedTextPattern = exports.FilterPattern = exports.JsonPattern = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Base class for patterns that only match JSON log events.
 *
 * @stability stable
 */
class JsonPattern {
    // This is a separate class so we have some type safety where users can't
    // combine text patterns and JSON patterns with an 'and' operation.
    /**
     * @stability stable
     */
    constructor(jsonPatternString) {
        this.jsonPatternString = jsonPatternString;
    }
    /**
     * @stability stable
     */
    get logPatternString() {
        return '{ ' + this.jsonPatternString + ' }';
    }
}
exports.JsonPattern = JsonPattern;
_a = JSII_RTTI_SYMBOL_1;
JsonPattern[_a] = { fqn: "@aws-cdk/aws-logs.JsonPattern", version: "1.90.0" };
/**
 * A collection of static methods to generate appropriate ILogPatterns.
 *
 * @stability stable
 */
class FilterPattern {
    /**
     * Use the given string as log pattern.
     *
     * See https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/FilterAndPatternSyntax.html
     * for information on writing log patterns.
     *
     * @param logPatternString The pattern string to use.
     * @stability stable
     */
    static literal(logPatternString) {
        return new LiteralLogPattern(logPatternString);
    }
    /**
     * A log pattern that matches all events.
     *
     * @stability stable
     */
    static allEvents() {
        return new LiteralLogPattern('');
    }
    /**
     * A log pattern that matches if all the strings given appear in the event.
     *
     * @param terms The words to search for.
     * @stability stable
     */
    static allTerms(...terms) {
        return new TextLogPattern([terms]);
    }
    /**
     * A log pattern that matches if any of the strings given appear in the event.
     *
     * @param terms The words to search for.
     * @stability stable
     */
    static anyTerm(...terms) {
        return new TextLogPattern(terms.map(t => [t]));
    }
    /**
     * A log pattern that matches if any of the given term groups matches the event.
     *
     * A term group matches an event if all the terms in it appear in the event string.
     *
     * @param termGroups A list of term groups to search for.
     * @stability stable
     */
    static anyTermGroup(...termGroups) {
        return new TextLogPattern(termGroups);
    }
    /**
     * A JSON log pattern that compares string values.
     *
     * This pattern only matches if the event is a JSON event, and the indicated field inside
     * compares with the string value.
     *
     * Use '$' to indicate the root of the JSON structure. The comparison operator can only
     * compare equality or inequality. The '*' wildcard may appear in the value may at the
     * start or at the end.
     *
     * For more information, see:
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/FilterAndPatternSyntax.html
     *
     * @param jsonField Field inside JSON.
     * @param comparison Comparison to carry out.
     * @param value The string value to compare to.
     * @stability stable
     */
    static stringValue(jsonField, comparison, value) {
        return new JSONStringPattern(jsonField, comparison, value);
    }
    /**
     * A JSON log pattern that compares numerical values.
     *
     * This pattern only matches if the event is a JSON event, and the indicated field inside
     * compares with the value in the indicated way.
     *
     * Use '$' to indicate the root of the JSON structure. The comparison operator can only
     * compare equality or inequality. The '*' wildcard may appear in the value may at the
     * start or at the end.
     *
     * For more information, see:
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/FilterAndPatternSyntax.html
     *
     * @param jsonField Field inside JSON.
     * @param comparison Comparison to carry out.
     * @param value The numerical value to compare to.
     * @stability stable
     */
    static numberValue(jsonField, comparison, value) {
        return new JSONNumberPattern(jsonField, comparison, value);
    }
    /**
     * A JSON log pattern that matches if the field exists and has the special value 'null'.
     *
     * @param jsonField Field inside JSON.
     * @stability stable
     */
    static isNull(jsonField) {
        return new JSONPostfixPattern(jsonField, 'IS NULL');
    }
    /**
     * A JSON log pattern that matches if the field does not exist.
     *
     * @param jsonField Field inside JSON.
     * @stability stable
     */
    static notExists(jsonField) {
        return new JSONPostfixPattern(jsonField, 'NOT EXISTS');
    }
    /**
     * A JSON log patter that matches if the field exists.
     *
     * This is a readable convenience wrapper over 'field = *'
     *
     * @param jsonField Field inside JSON.
     * @stability stable
     */
    static exists(jsonField) {
        return new JSONStringPattern(jsonField, '=', '*');
    }
    /**
     * A JSON log pattern that matches if the field exists and equals the boolean value.
     *
     * @param jsonField Field inside JSON.
     * @param value The value to match.
     * @stability stable
     */
    static booleanValue(jsonField, value) {
        return new JSONPostfixPattern(jsonField, value ? 'IS TRUE' : 'IS FALSE');
    }
    /**
     * A JSON log pattern that matches if all given JSON log patterns match.
     *
     * @stability stable
     */
    static all(...patterns) {
        if (patterns.length === 0) {
            throw new Error('Must supply at least one pattern, or use allEvents() to match all events.');
        }
        if (patterns.length === 1) {
            return patterns[0];
        }
        return new JSONAggregatePattern('&&', patterns);
    }
    /**
     * A JSON log pattern that matches if any of the given JSON log patterns match.
     *
     * @stability stable
     */
    static any(...patterns) {
        if (patterns.length === 0) {
            throw new Error('Must supply at least one pattern');
        }
        if (patterns.length === 1) {
            return patterns[0];
        }
        return new JSONAggregatePattern('||', patterns);
    }
    /**
     * A space delimited log pattern matcher.
     *
     * The log event is divided into space-delimited columns (optionally
     * enclosed by "" or [] to capture spaces into column values), and names
     * are given to each column.
     *
     * '...' may be specified once to match any number of columns.
     *
     * Afterwards, conditions may be added to individual columns.
     *
     * @param columns The columns in the space-delimited log stream.
     * @stability stable
     */
    static spaceDelimited(...columns) {
        return SpaceDelimitedTextPattern.construct(columns);
    }
}
exports.FilterPattern = FilterPattern;
_b = JSII_RTTI_SYMBOL_1;
FilterPattern[_b] = { fqn: "@aws-cdk/aws-logs.FilterPattern", version: "1.90.0" };
/**
 * Use a string literal as a log pattern
 */
class LiteralLogPattern {
    constructor(logPatternString) {
        this.logPatternString = logPatternString;
    }
}
/**
 * Search for a set of set of terms
 */
class TextLogPattern {
    constructor(clauses) {
        const quotedClauses = clauses.map(terms => terms.map(quoteTerm).join(' '));
        if (quotedClauses.length === 1) {
            this.logPatternString = quotedClauses[0];
        }
        else {
            this.logPatternString = quotedClauses.map(alt => '?' + alt).join(' ');
        }
    }
}
/**
 * A string comparison for JSON values
 */
class JSONStringPattern extends JsonPattern {
    constructor(jsonField, comparison, value) {
        comparison = validateStringOperator(comparison);
        super(`${jsonField} ${comparison} ${quoteTerm(value)}`);
    }
}
/**
 * A number comparison for JSON values
 */
class JSONNumberPattern extends JsonPattern {
    constructor(jsonField, comparison, value) {
        comparison = validateNumericalOperator(comparison);
        super(`${jsonField} ${comparison} ${value}`);
    }
}
/**
 * A postfix operator for JSON patterns
 */
class JSONPostfixPattern extends JsonPattern {
    constructor(jsonField, postfix) {
        // No validation, we assume these are generated by trusted factory functions
        super(`${jsonField} ${postfix}`);
    }
}
/**
 * Combines multiple other JSON patterns with an operator
 */
class JSONAggregatePattern extends JsonPattern {
    constructor(operator, patterns) {
        if (operator !== '&&' && operator !== '||') {
            throw new Error('Operator must be one of && or ||');
        }
        const clauses = patterns.map(p => '(' + p.jsonPatternString + ')');
        super(clauses.join(` ${operator} `));
    }
}
const COL_ELLIPSIS = '...';
/**
 * Space delimited text pattern.
 *
 * @stability stable
 */
class SpaceDelimitedTextPattern {
    // TODO: Temporarily changed from private to protected to unblock build. We need to think
    //     about how to handle jsii types with private constructors.
    /**
     * @stability stable
     */
    constructor(columns, restrictions) {
        this.columns = columns;
        this.restrictions = restrictions;
        // Private constructor so we validate in the .construct() factory function
    }
    /**
     * Construct a new instance of a space delimited text pattern.
     *
     * Since this class must be public, we can't rely on the user only creating it through
     * the `LogPattern.spaceDelimited()` factory function. We must therefore validate the
     * argument in the constructor. Since we're returning a copy on every mutation, and we
     * don't want to re-validate the same things on every construction, we provide a limited
     * set of mutator functions and only validate the new data every time.
     *
     * @stability stable
     */
    static construct(columns) {
        // Validation happens here because a user could instantiate this object directly without
        // going through the factory
        for (const column of columns) {
            if (!validColumnName(column)) {
                throw new Error(`Invalid column name: ${column}`);
            }
        }
        if (sum(columns.map(c => c === COL_ELLIPSIS ? 1 : 0)) > 1) {
            throw new Error("Can use at most one '...' column");
        }
        return new SpaceDelimitedTextPattern(columns, {});
    }
    /**
     * Restrict where the pattern applies.
     *
     * @stability stable
     */
    whereString(columnName, comparison, value) {
        if (columnName === COL_ELLIPSIS) {
            throw new Error("Can't use '...' in a restriction");
        }
        if (this.columns.indexOf(columnName) === -1) {
            throw new Error(`Column in restrictions that is not in columns: ${columnName}`);
        }
        comparison = validateStringOperator(comparison);
        return new SpaceDelimitedTextPattern(this.columns, this.addRestriction(columnName, {
            comparison,
            stringValue: value,
        }));
    }
    /**
     * Restrict where the pattern applies.
     *
     * @stability stable
     */
    whereNumber(columnName, comparison, value) {
        if (columnName === COL_ELLIPSIS) {
            throw new Error("Can't use '...' in a restriction");
        }
        if (this.columns.indexOf(columnName) === -1) {
            throw new Error(`Column in restrictions that is not in columns: ${columnName}`);
        }
        comparison = validateNumericalOperator(comparison);
        return new SpaceDelimitedTextPattern(this.columns, this.addRestriction(columnName, {
            comparison,
            numberValue: value,
        }));
    }
    /**
     * @stability stable
     */
    get logPatternString() {
        return '[' + this.columns.map(this.columnExpression.bind(this)).join(', ') + ']';
    }
    /**
     * Return the column expression for the given column
     */
    columnExpression(column) {
        const restrictions = this.restrictions[column];
        if (!restrictions) {
            return column;
        }
        return restrictions.map(r => renderRestriction(column, r)).join(' && ');
    }
    /**
     * Make a copy of the current restrictions and add one
     */
    addRestriction(columnName, restriction) {
        const ret = {};
        for (const key of Object.keys(this.restrictions)) {
            ret[key] = this.restrictions[key].slice();
        }
        if (!(columnName in ret)) {
            ret[columnName] = [];
        }
        ret[columnName].push(restriction);
        return ret;
    }
}
exports.SpaceDelimitedTextPattern = SpaceDelimitedTextPattern;
_c = JSII_RTTI_SYMBOL_1;
SpaceDelimitedTextPattern[_c] = { fqn: "@aws-cdk/aws-logs.SpaceDelimitedTextPattern", version: "1.90.0" };
/**
 * Quote a term for use in a pattern expression
 *
 * It's never wrong to quote a string term, and required if the term
 * contains non-alphanumerical characters, so we just always do it.
 *
 * Inner double quotes are escaped using a backslash.
 */
function quoteTerm(term) {
    return '"' + term.replace(/\\/g, '\\\\').replace(/"/g, '\\"') + '"';
}
/**
 * Return whether the given column name is valid in a space-delimited table
 */
function validColumnName(column) {
    return column === COL_ELLIPSIS || /^[a-zA-Z0-9_-]+$/.exec(column);
}
/**
 * Validate and normalize the string comparison operator
 *
 * Correct for a common typo/confusion, treat '==' as '='
 */
function validateStringOperator(operator) {
    if (operator === '==') {
        operator = '=';
    }
    if (operator !== '=' && operator !== '!=') {
        throw new Error(`Invalid comparison operator ('${operator}'), must be either '=' or '!='`);
    }
    return operator;
}
const VALID_OPERATORS = ['=', '!=', '<', '<=', '>', '>='];
/**
 * Validate and normalize numerical comparison operators
 *
 * Correct for a common typo/confusion, treat '==' as '='
 */
function validateNumericalOperator(operator) {
    // Correct for a common typo, treat '==' as '='
    if (operator === '==') {
        operator = '=';
    }
    if (VALID_OPERATORS.indexOf(operator) === -1) {
        throw new Error(`Invalid comparison operator ('${operator}'), must be one of ${VALID_OPERATORS.join(', ')}`);
    }
    return operator;
}
/**
 * Render a table restriction
 */
function renderRestriction(column, restriction) {
    if (restriction.numberValue !== undefined) {
        return `${column} ${restriction.comparison} ${restriction.numberValue}`;
    }
    else if (restriction.stringValue) {
        return `${column} ${restriction.comparison} ${quoteTerm(restriction.stringValue)}`;
    }
    else {
        throw new Error('Invalid restriction');
    }
}
function sum(xs) {
    return xs.reduce((a, c) => a + c, 0);
}
//# sourceMappingURL=data:application/json;base64,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