"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExecuteApi = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [execute-api](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonapigateway.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class ExecuteApi extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [execute-api](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonapigateway.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'execute-api';
        this.actionList = {
            "InvalidateCache": {
                "url": "https://docs.aws.amazon.com/apigateway/api-reference/api-gateway-caching.html",
                "description": "Used to invalidate API cache upon a client request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "execute-api-general": {
                        "required": true
                    }
                }
            },
            "Invoke": {
                "url": "https://docs.aws.amazon.com/apigateway/api-reference/how-to-call-api.html",
                "description": "Used to invoke an API upon a client request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "execute-api-general": {
                        "required": true
                    }
                }
            },
            "ManageConnections": {
                "url": "https://docs.aws.amazon.com/apigateway/api-reference/apigateway-websocket-control-access-iam.html",
                "description": "ManageConnections controls access to the @connections API",
                "accessLevel": "Write",
                "resourceTypes": {
                    "execute-api-general": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "execute-api-general": {
                "name": "execute-api-general",
                "url": "",
                "arn": "arn:${Partition}:execute-api:${Region}:${Account}:${ApiId}/${Stage}/${Method}/${ApiSpecificResourcePath}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Used to invalidate API cache upon a client request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apigateway/api-reference/api-gateway-caching.html
     */
    invalidateCache() {
        this.add('execute-api:InvalidateCache');
        return this;
    }
    /**
     * Used to invoke an API upon a client request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apigateway/api-reference/how-to-call-api.html
     */
    invoke() {
        this.add('execute-api:Invoke');
        return this;
    }
    /**
     * ManageConnections controls access to the @connections API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apigateway/api-reference/apigateway-websocket-control-access-iam.html
     */
    manageConnections() {
        this.add('execute-api:ManageConnections');
        return this;
    }
    /**
     * Adds a resource of type execute-api-general to the statement
     *
     * @param apiId - Identifier for the apiId.
     * @param stage - Identifier for the stage.
     * @param method - Identifier for the method.
     * @param apiSpecificResourcePath - Identifier for the apiSpecificResourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onExecuteApiGeneral(apiId, stage, method, apiSpecificResourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:execute-api:${Region}:${Account}:${ApiId}/${Stage}/${Method}/${ApiSpecificResourcePath}';
        arn = arn.replace('${ApiId}', apiId);
        arn = arn.replace('${Stage}', stage);
        arn = arn.replace('${Method}', method);
        arn = arn.replace('${ApiSpecificResourcePath}', apiSpecificResourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.ExecuteApi = ExecuteApi;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1hem9uYXBpZ2F0ZXdheS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFtYXpvbmFwaWdhdGV3YXkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsc0NBQW9FO0FBRXBFOzs7O0dBSUc7QUFDSCxNQUFhLFVBQVcsU0FBUSx3QkFBZTtJQTJDN0M7Ozs7T0FJRztJQUNILFlBQWEsR0FBWTtRQUN2QixLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFoRE4sa0JBQWEsR0FBRyxhQUFhLENBQUM7UUFDM0IsZUFBVSxHQUFZO1lBQzlCLGlCQUFpQixFQUFFO2dCQUNqQixLQUFLLEVBQUUsK0VBQStFO2dCQUN0RixhQUFhLEVBQUUsb0RBQW9EO2dCQUNuRSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLHFCQUFxQixFQUFFO3dCQUNyQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELFFBQVEsRUFBRTtnQkFDUixLQUFLLEVBQUUsMkVBQTJFO2dCQUNsRixhQUFhLEVBQUUsNkNBQTZDO2dCQUM1RCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLHFCQUFxQixFQUFFO3dCQUNyQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELG1CQUFtQixFQUFFO2dCQUNuQixLQUFLLEVBQUUsbUdBQW1HO2dCQUMxRyxhQUFhLEVBQUUsMkRBQTJEO2dCQUMxRSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLHFCQUFxQixFQUFFO3dCQUNyQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtTQUNGLENBQUM7UUFDSyxrQkFBYSxHQUFrQjtZQUNwQyxxQkFBcUIsRUFBRTtnQkFDckIsTUFBTSxFQUFFLHFCQUFxQjtnQkFDN0IsS0FBSyxFQUFFLEVBQUU7Z0JBQ1QsS0FBSyxFQUFFLDBHQUEwRztnQkFDakgsZUFBZSxFQUFFLEVBQUU7YUFDcEI7U0FDRixDQUFDO0lBU0YsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLE1BQU07UUFDWCxJQUFJLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxHQUFHLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksbUJBQW1CLENBQUMsS0FBYSxFQUFFLEtBQWEsRUFBRSxNQUFjLEVBQUUsdUJBQStCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDN0osSUFBSSxHQUFHLEdBQUcsMEdBQTBHLENBQUM7UUFDckgsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3JDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUNyQyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDdkMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsNEJBQTRCLEVBQUUsdUJBQXVCLENBQUMsQ0FBQztRQUN6RSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztDQUNGO0FBOUdELGdDQThHQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjdGlvbnMsIFBvbGljeVN0YXRlbWVudCwgUmVzb3VyY2VUeXBlcyB9IGZyb20gXCIuLi9zaGFyZWRcIjtcblxuLyoqXG4gKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2V4ZWN1dGUtYXBpXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25hcGlnYXRld2F5Lmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgRXhlY3V0ZUFwaSBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ2V4ZWN1dGUtYXBpJztcbiAgcHJvdGVjdGVkIGFjdGlvbkxpc3Q6IEFjdGlvbnMgPSB7XG4gICAgXCJJbnZhbGlkYXRlQ2FjaGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9hcGktcmVmZXJlbmNlL2FwaS1nYXRld2F5LWNhY2hpbmcuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVzZWQgdG8gaW52YWxpZGF0ZSBBUEkgY2FjaGUgdXBvbiBhIGNsaWVudCByZXF1ZXN0XCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZXhlY3V0ZS1hcGktZ2VuZXJhbFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiSW52b2tlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvYXBpLXJlZmVyZW5jZS9ob3ctdG8tY2FsbC1hcGkuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVzZWQgdG8gaW52b2tlIGFuIEFQSSB1cG9uIGEgY2xpZW50IHJlcXVlc3RcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJleGVjdXRlLWFwaS1nZW5lcmFsXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJNYW5hZ2VDb25uZWN0aW9uc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2FwaS1yZWZlcmVuY2UvYXBpZ2F0ZXdheS13ZWJzb2NrZXQtY29udHJvbC1hY2Nlc3MtaWFtLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJNYW5hZ2VDb25uZWN0aW9ucyBjb250cm9scyBhY2Nlc3MgdG8gdGhlIEBjb25uZWN0aW9ucyBBUElcIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJleGVjdXRlLWFwaS1nZW5lcmFsXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgfTtcbiAgcHVibGljIHJlc291cmNlVHlwZXM6IFJlc291cmNlVHlwZXMgPSB7XG4gICAgXCJleGVjdXRlLWFwaS1nZW5lcmFsXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcImV4ZWN1dGUtYXBpLWdlbmVyYWxcIixcbiAgICAgIFwidXJsXCI6IFwiXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06ZXhlY3V0ZS1hcGk6JHtSZWdpb259OiR7QWNjb3VudH06JHtBcGlJZH0vJHtTdGFnZX0vJHtNZXRob2R9LyR7QXBpU3BlY2lmaWNSZXNvdXJjZVBhdGh9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW11cbiAgICB9XG4gIH07XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbZXhlY3V0ZS1hcGldKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2FtYXpvbmFwaWdhdGV3YXkuaHRtbCkuXG4gICAqXG4gICAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICAgKi9cbiAgY29uc3RydWN0b3IgKHNpZD86IHN0cmluZykge1xuICAgIHN1cGVyKHNpZCk7XG4gIH1cblxuICAvKipcbiAgICogVXNlZCB0byBpbnZhbGlkYXRlIEFQSSBjYWNoZSB1cG9uIGEgY2xpZW50IHJlcXVlc3RcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9hcGktcmVmZXJlbmNlL2FwaS1nYXRld2F5LWNhY2hpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIGludmFsaWRhdGVDYWNoZSgpIHtcbiAgICB0aGlzLmFkZCgnZXhlY3V0ZS1hcGk6SW52YWxpZGF0ZUNhY2hlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXNlZCB0byBpbnZva2UgYW4gQVBJIHVwb24gYSBjbGllbnQgcmVxdWVzdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2FwaS1yZWZlcmVuY2UvaG93LXRvLWNhbGwtYXBpLmh0bWxcbiAgICovXG4gIHB1YmxpYyBpbnZva2UoKSB7XG4gICAgdGhpcy5hZGQoJ2V4ZWN1dGUtYXBpOkludm9rZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hbmFnZUNvbm5lY3Rpb25zIGNvbnRyb2xzIGFjY2VzcyB0byB0aGUgQGNvbm5lY3Rpb25zIEFQSVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2FwaS1yZWZlcmVuY2UvYXBpZ2F0ZXdheS13ZWJzb2NrZXQtY29udHJvbC1hY2Nlc3MtaWFtLmh0bWxcbiAgICovXG4gIHB1YmxpYyBtYW5hZ2VDb25uZWN0aW9ucygpIHtcbiAgICB0aGlzLmFkZCgnZXhlY3V0ZS1hcGk6TWFuYWdlQ29ubmVjdGlvbnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBleGVjdXRlLWFwaS1nZW5lcmFsIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQHBhcmFtIGFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGFwaUlkLlxuICAgKiBAcGFyYW0gc3RhZ2UgLSBJZGVudGlmaWVyIGZvciB0aGUgc3RhZ2UuXG4gICAqIEBwYXJhbSBtZXRob2QgLSBJZGVudGlmaWVyIGZvciB0aGUgbWV0aG9kLlxuICAgKiBAcGFyYW0gYXBpU3BlY2lmaWNSZXNvdXJjZVBhdGggLSBJZGVudGlmaWVyIGZvciB0aGUgYXBpU3BlY2lmaWNSZXNvdXJjZVBhdGguXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25FeGVjdXRlQXBpR2VuZXJhbChhcGlJZDogc3RyaW5nLCBzdGFnZTogc3RyaW5nLCBtZXRob2Q6IHN0cmluZywgYXBpU3BlY2lmaWNSZXNvdXJjZVBhdGg6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06ZXhlY3V0ZS1hcGk6JHtSZWdpb259OiR7QWNjb3VudH06JHtBcGlJZH0vJHtTdGFnZX0vJHtNZXRob2R9LyR7QXBpU3BlY2lmaWNSZXNvdXJjZVBhdGh9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBcGlJZH0nLCBhcGlJZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7U3RhZ2V9Jywgc3RhZ2UpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke01ldGhvZH0nLCBtZXRob2QpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FwaVNwZWNpZmljUmVzb3VyY2VQYXRofScsIGFwaVNwZWNpZmljUmVzb3VyY2VQYXRoKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxufVxuIl19