"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Appstream = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [appstream](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonappstream2.0.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Appstream extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [appstream](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonappstream2.0.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'appstream';
        this.actionList = {
            "AssociateFleet": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_AssociateFleet.html",
                "description": "Grants permission to associate the specified fleet with the specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    },
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "BatchAssociateUserStack": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_BatchAssociateUserStack.html",
                "description": "Grants permission to associate the specified users with the specified stacks. Users in a user pool cannot be assigned to stacks with fleets that are joined to an Active Directory domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "BatchDisassociateUserStack": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_BatchDisassociateUserStack.html",
                "description": "Grants permission to disassociate the specified users from the specified stacks",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CopyImage": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CopyImage.html",
                "description": "Grants permission to copy the specified image within the same Region or to a new Region within the same AWS account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "image": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreateDirectoryConfig": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateDirectoryConfig.html",
                "description": "Grants permission to create a Directory Config object in AppStream 2.0. This object includes the configuration information required to join fleets and image builders to Microsoft Active Directory domains",
                "accessLevel": "Write"
            },
            "CreateFleet": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateFleet.html",
                "description": "Grants permission to create a fleet. A fleet is a group of streaming instances from which applications are launched and streamed to users",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    },
                    "image": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateImageBuilder": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateImageBuilder.html",
                "description": "Grants permission to create an image builder. An image builder is a virtual machine that is used to create an image",
                "accessLevel": "Write",
                "resourceTypes": {
                    "image": {
                        "required": true
                    },
                    "image-builder": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateImageBuilderStreamingURL": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateImageBuilderStreamingURL.html",
                "description": "Grants permission to create a URL to start an image builder streaming session",
                "accessLevel": "Write",
                "resourceTypes": {
                    "image-builder": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreateStack": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateStack.html",
                "description": "Grants permission to create a stack to start streaming applications to users. A stack consists of an associated fleet, user access policies, and storage configurations",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateStreamingURL": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateStreamingURL.html",
                "description": "Grants permission to create a temporary URL to start an AppStream 2.0 streaming session for the specified user. A streaming URL enables application streaming to be tested without user setup",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    },
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreateUsageReportSubscription": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateUsageReportSubscription.html",
                "description": "Grants permission to create a usage report subscription. Usage reports are generated daily",
                "accessLevel": "Write"
            },
            "CreateUser": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateUser.html",
                "description": "Grants permission to create a new user in the user pool",
                "accessLevel": "Write"
            },
            "DeleteDirectoryConfig": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteDirectoryConfig.html",
                "description": "Grants permission to delete the specified Directory Config object from AppStream 2.0. This object includes the configuration information required to join fleets and image builders to Microsoft Active Directory domains",
                "accessLevel": "Write"
            },
            "DeleteFleet": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteFleet.html",
                "description": "Grants permission to delete the specified fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DeleteImage": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteImage.html",
                "description": "Grants permission to delete the specified image. An image cannot be deleted when it is in use",
                "accessLevel": "Write",
                "resourceTypes": {
                    "image": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DeleteImageBuilder": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteImageBuilder.html",
                "description": "Grants permission to delete the specified image builder and release capacity",
                "accessLevel": "Write",
                "resourceTypes": {
                    "image-builder": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DeleteImagePermissions": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteImagePermissions.html",
                "description": "Grants permission to delete permissions for the specified private image",
                "accessLevel": "Write",
                "resourceTypes": {
                    "image": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DeleteStack": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteStack.html",
                "description": "Grants permission to delete the specified stack. After the stack is deleted, the application streaming environment provided by the stack is no longer available to users. Also, any reservations made for application streaming sessions for the stack are released",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DeleteUsageReportSubscription": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteUsageReportSubscription.html",
                "description": "Grants permission to disable usage report generation",
                "accessLevel": "Write"
            },
            "DeleteUser": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteUser.html",
                "description": "Grants permission to delete a user from the user pool",
                "accessLevel": "Write"
            },
            "DescribeDirectoryConfigs": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeDirectoryConfigs.html",
                "description": "Grants permission to retrieve a list that describes one or more specified Directory Config objects for AppStream 2.0, if the names for these objects are provided. Otherwise, all Directory Config objects in the account are described. This object includes the configuration information required to join fleets and image builders to Microsoft Active Directory domains",
                "accessLevel": "Read"
            },
            "DescribeFleets": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeFleets.html",
                "description": "Grants permission to retrieve a list that describes one or more specified fleets, if the fleet names are provided. Otherwise, all fleets in the account are described",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": false
                    }
                }
            },
            "DescribeImageBuilders": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeImageBuilders.html",
                "description": "Grants permission to retrieve a list that describes one or more specified image builders, if the image builder names are provided. Otherwise, all image builders in the account are described",
                "accessLevel": "Read",
                "resourceTypes": {
                    "image-builder": {
                        "required": false
                    }
                }
            },
            "DescribeImagePermissions": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeImagePermissions.html",
                "description": "Grants permission to retrieve a list that describes the permissions for shared AWS account IDs on a private image that you own",
                "accessLevel": "Read",
                "resourceTypes": {
                    "image": {
                        "required": true
                    }
                }
            },
            "DescribeImages": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeImages.html",
                "description": "Grants permission to retrieve a list that describes one or more specified images, if the image names or image ARNs are provided. Otherwise, all images in the account are described",
                "accessLevel": "Read",
                "resourceTypes": {
                    "image": {
                        "required": false
                    }
                }
            },
            "DescribeSessions": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeSessions.html",
                "description": "Grants permission to retrieve a list that describes the streaming sessions for the specified stack and fleet. If a user ID is provided for the stack and fleet, only the streaming sessions for that user are described",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    },
                    "stack": {
                        "required": true
                    }
                }
            },
            "DescribeStacks": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeStacks.html",
                "description": "Grants permission to retrieve a list that describes one or more specified stacks, if the stack names are provided. Otherwise, all stacks in the account are described",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeUsageReportSubscriptions": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeUsageReportSubscriptions.html",
                "description": "Grants permission to retrieve a list that describes one or more usage report subscriptions",
                "accessLevel": "Read"
            },
            "DescribeUserStackAssociations": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeUserStackAssociations.html",
                "description": "Grants permission to retrieve a list that describes the UserStackAssociation objects",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stack": {
                        "required": false
                    }
                }
            },
            "DescribeUsers": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeUsers.html",
                "description": "Grants permission to retrieve a list that describes users in the user pool",
                "accessLevel": "Read"
            },
            "DisableUser": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DisableUser.html",
                "description": "Grants permission to disable the specified user in the user pool. This action does not delete the user",
                "accessLevel": "Write"
            },
            "DisassociateFleet": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DisassociateFleet.html",
                "description": "Grants permission to disassociate the specified fleet from the specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    },
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "EnableUser": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_EnableUser.html",
                "description": "Grants permission to enable a user in the user pool",
                "accessLevel": "Write"
            },
            "ExpireSession": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ExpireSession.html",
                "description": "Grants permission to immediately stop the specified streaming session",
                "accessLevel": "Write"
            },
            "GetImageBuilders": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/developerguide/controlling-administrator-access-with-policies-service-linked-roles.html#managed-policies-required-to-access-appstream-resources",
                "description": "Grants permission to retrieve a list that describes one or more specified image builders, if the image builder names are provided. Otherwise, all image builders in the account are described",
                "accessLevel": "Read"
            },
            "GetParametersForThemeAssetUpload": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/developerguide/controlling-administrator-access-with-policies-service-linked-roles.html#managed-policies-required-to-access-appstream-resources",
                "description": "Grants permission to upload theme assets",
                "accessLevel": "Write"
            },
            "ListAssociatedFleets": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ListAssociatedFleets.html",
                "description": "Grants permission to retrieve the name of the fleet that is associated with the specified stack",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                }
            },
            "ListAssociatedStacks": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ListAssociatedStacks.html",
                "description": "Grants permission to retrieve the name of the stack with which the specified fleet is associated",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to retrieve a list of all tags for the specified AppStream 2.0 resource. The following resources can be tagged: Image builders, images, fleets, and stacks",
                "accessLevel": "Read"
            },
            "StartFleet": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StartFleet.html",
                "description": "Grants permission to start the specified fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "StartImageBuilder": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StartImageBuilder.html",
                "description": "Grants permission to start the specified image builder",
                "accessLevel": "Write",
                "resourceTypes": {
                    "image-builder": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "StopFleet": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StopFleet.html",
                "description": "Grants permission to stop the specified fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "StopImageBuilder": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StopImageBuilder.html",
                "description": "Grants permission to stop the specified image builder",
                "accessLevel": "Write",
                "resourceTypes": {
                    "image-builder": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Stream": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/developerguide/external-identity-providers-setting-up-saml.html#external-identity-providers-embed-inline-policy-for-IAM-role",
                "description": "Grants permission to federated users to sign in by using their existing credentials and stream applications from the specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "appstream:userId"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to add or overwrite one or more tags for the specified AppStream 2.0 resource. The following resources can be tagged: Image builders, images, fleets, and stacks",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "fleet": {
                        "required": false
                    },
                    "image": {
                        "required": false
                    },
                    "image-builder": {
                        "required": false
                    },
                    "stack": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to disassociate one or more tags from the specified AppStream 2.0 resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "fleet": {
                        "required": false
                    },
                    "image": {
                        "required": false
                    },
                    "image-builder": {
                        "required": false
                    },
                    "stack": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDirectoryConfig": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateDirectoryConfig.html",
                "description": "Grants permission to update the specified Directory Config object in AppStream 2.0. This object includes the configuration information required to join fleets and image builders to Microsoft Active Directory domains",
                "accessLevel": "Write"
            },
            "UpdateFleet": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateFleet.html",
                "description": "Grants permission to update the specified fleet. All attributes except the fleet name can be updated when the fleet is in the STOPPED state",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    },
                    "image": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UpdateImagePermissions": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateImagePermissions.html",
                "description": "Grants permission to add or update permissions for the specified private image",
                "accessLevel": "Write",
                "resourceTypes": {
                    "image": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UpdateStack": {
                "url": "https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateStack.html",
                "description": "Grants permission to update the specified fields for the specified stack",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stack": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
        this.resourceTypes = {
            "fleet": {
                "name": "fleet",
                "url": "https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts",
                "arn": "arn:${Partition}:appstream:${Region}:${Account}:fleet/${FleetName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "image": {
                "name": "image",
                "url": "https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts",
                "arn": "arn:${Partition}:appstream:${Region}:${Account}:image/${ImageName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "image-builder": {
                "name": "image-builder",
                "url": "https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts",
                "arn": "arn:${Partition}:appstream:${Region}:${Account}:image-builder/${ImageBuilderName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "stack": {
                "name": "stack",
                "url": "https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts",
                "arn": "arn:${Partition}:appstream:${Region}:${Account}:stack/${StackName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to associate the specified fleet with the specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_AssociateFleet.html
     */
    associateFleet() {
        this.add('appstream:AssociateFleet');
        return this;
    }
    /**
     * Grants permission to associate the specified users with the specified stacks. Users in a user pool cannot be assigned to stacks with fleets that are joined to an Active Directory domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_BatchAssociateUserStack.html
     */
    batchAssociateUserStack() {
        this.add('appstream:BatchAssociateUserStack');
        return this;
    }
    /**
     * Grants permission to disassociate the specified users from the specified stacks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_BatchDisassociateUserStack.html
     */
    batchDisassociateUserStack() {
        this.add('appstream:BatchDisassociateUserStack');
        return this;
    }
    /**
     * Grants permission to copy the specified image within the same Region or to a new Region within the same AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CopyImage.html
     */
    copyImage() {
        this.add('appstream:CopyImage');
        return this;
    }
    /**
     * Grants permission to create a Directory Config object in AppStream 2.0. This object includes the configuration information required to join fleets and image builders to Microsoft Active Directory domains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateDirectoryConfig.html
     */
    createDirectoryConfig() {
        this.add('appstream:CreateDirectoryConfig');
        return this;
    }
    /**
     * Grants permission to create a fleet. A fleet is a group of streaming instances from which applications are launched and streamed to users
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateFleet.html
     */
    createFleet() {
        this.add('appstream:CreateFleet');
        return this;
    }
    /**
     * Grants permission to create an image builder. An image builder is a virtual machine that is used to create an image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateImageBuilder.html
     */
    createImageBuilder() {
        this.add('appstream:CreateImageBuilder');
        return this;
    }
    /**
     * Grants permission to create a URL to start an image builder streaming session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateImageBuilderStreamingURL.html
     */
    createImageBuilderStreamingURL() {
        this.add('appstream:CreateImageBuilderStreamingURL');
        return this;
    }
    /**
     * Grants permission to create a stack to start streaming applications to users. A stack consists of an associated fleet, user access policies, and storage configurations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateStack.html
     */
    createStack() {
        this.add('appstream:CreateStack');
        return this;
    }
    /**
     * Grants permission to create a temporary URL to start an AppStream 2.0 streaming session for the specified user. A streaming URL enables application streaming to be tested without user setup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateStreamingURL.html
     */
    createStreamingURL() {
        this.add('appstream:CreateStreamingURL');
        return this;
    }
    /**
     * Grants permission to create a usage report subscription. Usage reports are generated daily
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateUsageReportSubscription.html
     */
    createUsageReportSubscription() {
        this.add('appstream:CreateUsageReportSubscription');
        return this;
    }
    /**
     * Grants permission to create a new user in the user pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateUser.html
     */
    createUser() {
        this.add('appstream:CreateUser');
        return this;
    }
    /**
     * Grants permission to delete the specified Directory Config object from AppStream 2.0. This object includes the configuration information required to join fleets and image builders to Microsoft Active Directory domains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteDirectoryConfig.html
     */
    deleteDirectoryConfig() {
        this.add('appstream:DeleteDirectoryConfig');
        return this;
    }
    /**
     * Grants permission to delete the specified fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteFleet.html
     */
    deleteFleet() {
        this.add('appstream:DeleteFleet');
        return this;
    }
    /**
     * Grants permission to delete the specified image. An image cannot be deleted when it is in use
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteImage.html
     */
    deleteImage() {
        this.add('appstream:DeleteImage');
        return this;
    }
    /**
     * Grants permission to delete the specified image builder and release capacity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteImageBuilder.html
     */
    deleteImageBuilder() {
        this.add('appstream:DeleteImageBuilder');
        return this;
    }
    /**
     * Grants permission to delete permissions for the specified private image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteImagePermissions.html
     */
    deleteImagePermissions() {
        this.add('appstream:DeleteImagePermissions');
        return this;
    }
    /**
     * Grants permission to delete the specified stack. After the stack is deleted, the application streaming environment provided by the stack is no longer available to users. Also, any reservations made for application streaming sessions for the stack are released
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteStack.html
     */
    deleteStack() {
        this.add('appstream:DeleteStack');
        return this;
    }
    /**
     * Grants permission to disable usage report generation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteUsageReportSubscription.html
     */
    deleteUsageReportSubscription() {
        this.add('appstream:DeleteUsageReportSubscription');
        return this;
    }
    /**
     * Grants permission to delete a user from the user pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteUser.html
     */
    deleteUser() {
        this.add('appstream:DeleteUser');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified Directory Config objects for AppStream 2.0, if the names for these objects are provided. Otherwise, all Directory Config objects in the account are described. This object includes the configuration information required to join fleets and image builders to Microsoft Active Directory domains
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeDirectoryConfigs.html
     */
    describeDirectoryConfigs() {
        this.add('appstream:DescribeDirectoryConfigs');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified fleets, if the fleet names are provided. Otherwise, all fleets in the account are described
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeFleets.html
     */
    describeFleets() {
        this.add('appstream:DescribeFleets');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified image builders, if the image builder names are provided. Otherwise, all image builders in the account are described
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeImageBuilders.html
     */
    describeImageBuilders() {
        this.add('appstream:DescribeImageBuilders');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes the permissions for shared AWS account IDs on a private image that you own
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeImagePermissions.html
     */
    describeImagePermissions() {
        this.add('appstream:DescribeImagePermissions');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified images, if the image names or image ARNs are provided. Otherwise, all images in the account are described
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeImages.html
     */
    describeImages() {
        this.add('appstream:DescribeImages');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes the streaming sessions for the specified stack and fleet. If a user ID is provided for the stack and fleet, only the streaming sessions for that user are described
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeSessions.html
     */
    describeSessions() {
        this.add('appstream:DescribeSessions');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified stacks, if the stack names are provided. Otherwise, all stacks in the account are described
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeStacks.html
     */
    describeStacks() {
        this.add('appstream:DescribeStacks');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes one or more usage report subscriptions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeUsageReportSubscriptions.html
     */
    describeUsageReportSubscriptions() {
        this.add('appstream:DescribeUsageReportSubscriptions');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes the UserStackAssociation objects
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeUserStackAssociations.html
     */
    describeUserStackAssociations() {
        this.add('appstream:DescribeUserStackAssociations');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes users in the user pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeUsers.html
     */
    describeUsers() {
        this.add('appstream:DescribeUsers');
        return this;
    }
    /**
     * Grants permission to disable the specified user in the user pool. This action does not delete the user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DisableUser.html
     */
    disableUser() {
        this.add('appstream:DisableUser');
        return this;
    }
    /**
     * Grants permission to disassociate the specified fleet from the specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DisassociateFleet.html
     */
    disassociateFleet() {
        this.add('appstream:DisassociateFleet');
        return this;
    }
    /**
     * Grants permission to enable a user in the user pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_EnableUser.html
     */
    enableUser() {
        this.add('appstream:EnableUser');
        return this;
    }
    /**
     * Grants permission to immediately stop the specified streaming session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ExpireSession.html
     */
    expireSession() {
        this.add('appstream:ExpireSession');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified image builders, if the image builder names are provided. Otherwise, all image builders in the account are described
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/controlling-administrator-access-with-policies-service-linked-roles.html#managed-policies-required-to-access-appstream-resources
     */
    getImageBuilders() {
        this.add('appstream:GetImageBuilders');
        return this;
    }
    /**
     * Grants permission to upload theme assets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/controlling-administrator-access-with-policies-service-linked-roles.html#managed-policies-required-to-access-appstream-resources
     */
    getParametersForThemeAssetUpload() {
        this.add('appstream:GetParametersForThemeAssetUpload');
        return this;
    }
    /**
     * Grants permission to retrieve the name of the fleet that is associated with the specified stack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ListAssociatedFleets.html
     */
    listAssociatedFleets() {
        this.add('appstream:ListAssociatedFleets');
        return this;
    }
    /**
     * Grants permission to retrieve the name of the stack with which the specified fleet is associated
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ListAssociatedStacks.html
     */
    listAssociatedStacks() {
        this.add('appstream:ListAssociatedStacks');
        return this;
    }
    /**
     * Grants permission to retrieve a list of all tags for the specified AppStream 2.0 resource. The following resources can be tagged: Image builders, images, fleets, and stacks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('appstream:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to start the specified fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StartFleet.html
     */
    startFleet() {
        this.add('appstream:StartFleet');
        return this;
    }
    /**
     * Grants permission to start the specified image builder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StartImageBuilder.html
     */
    startImageBuilder() {
        this.add('appstream:StartImageBuilder');
        return this;
    }
    /**
     * Grants permission to stop the specified fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StopFleet.html
     */
    stopFleet() {
        this.add('appstream:StopFleet');
        return this;
    }
    /**
     * Grants permission to stop the specified image builder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StopImageBuilder.html
     */
    stopImageBuilder() {
        this.add('appstream:StopImageBuilder');
        return this;
    }
    /**
     * Grants permission to federated users to sign in by using their existing credentials and stream applications from the specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/external-identity-providers-setting-up-saml.html#external-identity-providers-embed-inline-policy-for-IAM-role
     */
    stream() {
        this.add('appstream:Stream');
        return this;
    }
    /**
     * Grants permission to add or overwrite one or more tags for the specified AppStream 2.0 resource. The following resources can be tagged: Image builders, images, fleets, and stacks
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('appstream:TagResource');
        return this;
    }
    /**
     * Grants permission to disassociate one or more tags from the specified AppStream 2.0 resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('appstream:UntagResource');
        return this;
    }
    /**
     * Grants permission to update the specified Directory Config object in AppStream 2.0. This object includes the configuration information required to join fleets and image builders to Microsoft Active Directory domains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateDirectoryConfig.html
     */
    updateDirectoryConfig() {
        this.add('appstream:UpdateDirectoryConfig');
        return this;
    }
    /**
     * Grants permission to update the specified fleet. All attributes except the fleet name can be updated when the fleet is in the STOPPED state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateFleet.html
     */
    updateFleet() {
        this.add('appstream:UpdateFleet');
        return this;
    }
    /**
     * Grants permission to add or update permissions for the specified private image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateImagePermissions.html
     */
    updateImagePermissions() {
        this.add('appstream:UpdateImagePermissions');
        return this;
    }
    /**
     * Grants permission to update the specified fields for the specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateStack.html
     */
    updateStack() {
        this.add('appstream:UpdateStack');
        return this;
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts
     *
     * @param fleetName - Identifier for the fleetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onFleet(fleetName, account, region, partition) {
        var arn = 'arn:${Partition}:appstream:${Region}:${Account}:fleet/${FleetName}';
        arn = arn.replace('${FleetName}', fleetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type image to the statement
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts
     *
     * @param imageName - Identifier for the imageName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onImage(imageName, account, region, partition) {
        var arn = 'arn:${Partition}:appstream:${Region}:${Account}:image/${ImageName}';
        arn = arn.replace('${ImageName}', imageName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type image-builder to the statement
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts
     *
     * @param imageBuilderName - Identifier for the imageBuilderName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onImageBuilder(imageBuilderName, account, region, partition) {
        var arn = 'arn:${Partition}:appstream:${Region}:${Account}:image-builder/${ImageBuilderName}';
        arn = arn.replace('${ImageBuilderName}', imageBuilderName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type stack to the statement
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts
     *
     * @param stackName - Identifier for the stackName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onStack(stackName, account, region, partition) {
        var arn = 'arn:${Partition}:appstream:${Region}:${Account}:stack/${StackName}';
        arn = arn.replace('${StackName}', stackName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the ID of the AppStream 2.0 user
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/external-identity-providers-setting-up-saml.html#external-identity-providers-embed-inline-policy-for-IAM-role
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserId(value, operator) {
        return this.if(`appstream:userId`, value, operator || 'StringLike');
    }
}
exports.Appstream = Appstream;
//# sourceMappingURL=data:application/json;base64,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