"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeguruProfiler = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codeguru-profiler](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncodeguruprofiler.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CodeguruProfiler extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codeguru-profiler](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncodeguruprofiler.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codeguru-profiler';
        this.actionList = {
            "AddNotificationChannels": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_AddNotificationChannels.html",
                "description": "Grants permission to add up to 2 topic ARNs of existing AWS SNS topics to publish notifications",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "BatchGetFrameMetricData": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_BatchGetFrameMetricData.html",
                "description": "Grants permission to get the frame metric data for a Profiling Group",
                "accessLevel": "List",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "ConfigureAgent": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ConfigureAgent.html",
                "description": "Grants permission for an agent to register with the orchestration service and retrieve profiling configuration information",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "CreateProfilingGroup": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_CreateProfilingGroup.html",
                "description": "Grants permission to create a profiling group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "DeleteProfilingGroup": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_DeleteProfilingGroup.html",
                "description": "Grants permission to delete a profiling group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "DescribeProfilingGroup": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_DescribeProfilingGroup.html",
                "description": "Grants permission to describe a profiling group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "GetFindingsReportAccountSummary": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetFindingsReportAccountSummary.html",
                "description": "Grants permission to get a summary of recent recommendations for each profiling group in the account",
                "accessLevel": "Read"
            },
            "GetNotificationConfiguration": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetNotificationConfiguration.html",
                "description": "Grants permission to get the notification configuration",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "GetPolicy": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetPolicy.html",
                "description": "Grants permission to get the resource policy associated with the specified Profiling Group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "GetProfile": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetProfile.html",
                "description": "Grants permission to get aggregated profiles for a specific profiling group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "GetRecommendations": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetRecommendations.html",
                "description": "Grants permission to get recommendations",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "ListFindingsReports": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListFindingsReports.html",
                "description": "Grants permission to list the available recommendations reports for a specific profiling group",
                "accessLevel": "List",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "ListProfileTimes": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListProfileTimes.html",
                "description": "Grants permission to list the start times of the available aggregated profiles for a specific profiling group",
                "accessLevel": "List",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "ListProfilingGroups": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListProfilingGroups.html",
                "description": "Grants permission to list profiling groups in the account",
                "accessLevel": "List"
            },
            "PostAgentProfile": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_PostAgentProfile.html",
                "description": "Grants permission to submit a profile collected by an agent belonging to a specific profiling group for aggregation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "PutPermission": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_PutPermission.html",
                "description": "Grants permission to update the list of principals allowed for an action group in the resource policy associated with the specified Profiling Group.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "RemoveNotificationChannel": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_RemoveNotificationChannel.html",
                "description": "Grants permission to delete an already configured SNStopic arn from the notification configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "RemovePermission": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_RemovePermission.html",
                "description": "Grants permission to remove the permission of specified Action Group from the resource policy associated with the specified Profiling Group.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "SubmitFeedback": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_SubmitFeedback.html",
                "description": "Grants permission to submit user feedback for useful or non useful anomaly",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "UpdateProfilingGroup": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_UpdateProfilingGroup.html",
                "description": "Grants permission to update a specific profiling group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "ProfilingGroup": {
                "name": "ProfilingGroup",
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-ug/working-with-profiling-groups.html",
                "arn": "arn:${Partition}:codeguru-profiler:${Region}:${Account}:profilingGroup/${ProfilingGroupName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to add up to 2 topic ARNs of existing AWS SNS topics to publish notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_AddNotificationChannels.html
     */
    addNotificationChannels() {
        this.add('codeguru-profiler:AddNotificationChannels');
        return this;
    }
    /**
     * Grants permission to get the frame metric data for a Profiling Group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_BatchGetFrameMetricData.html
     */
    batchGetFrameMetricData() {
        this.add('codeguru-profiler:BatchGetFrameMetricData');
        return this;
    }
    /**
     * Grants permission for an agent to register with the orchestration service and retrieve profiling configuration information
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ConfigureAgent.html
     */
    configureAgent() {
        this.add('codeguru-profiler:ConfigureAgent');
        return this;
    }
    /**
     * Grants permission to create a profiling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_CreateProfilingGroup.html
     */
    createProfilingGroup() {
        this.add('codeguru-profiler:CreateProfilingGroup');
        return this;
    }
    /**
     * Grants permission to delete a profiling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_DeleteProfilingGroup.html
     */
    deleteProfilingGroup() {
        this.add('codeguru-profiler:DeleteProfilingGroup');
        return this;
    }
    /**
     * Grants permission to describe a profiling group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_DescribeProfilingGroup.html
     */
    describeProfilingGroup() {
        this.add('codeguru-profiler:DescribeProfilingGroup');
        return this;
    }
    /**
     * Grants permission to get a summary of recent recommendations for each profiling group in the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetFindingsReportAccountSummary.html
     */
    getFindingsReportAccountSummary() {
        this.add('codeguru-profiler:GetFindingsReportAccountSummary');
        return this;
    }
    /**
     * Grants permission to get the notification configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetNotificationConfiguration.html
     */
    getNotificationConfiguration() {
        this.add('codeguru-profiler:GetNotificationConfiguration');
        return this;
    }
    /**
     * Grants permission to get the resource policy associated with the specified Profiling Group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetPolicy.html
     */
    getPolicy() {
        this.add('codeguru-profiler:GetPolicy');
        return this;
    }
    /**
     * Grants permission to get aggregated profiles for a specific profiling group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetProfile.html
     */
    getProfile() {
        this.add('codeguru-profiler:GetProfile');
        return this;
    }
    /**
     * Grants permission to get recommendations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetRecommendations.html
     */
    getRecommendations() {
        this.add('codeguru-profiler:GetRecommendations');
        return this;
    }
    /**
     * Grants permission to list the available recommendations reports for a specific profiling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListFindingsReports.html
     */
    listFindingsReports() {
        this.add('codeguru-profiler:ListFindingsReports');
        return this;
    }
    /**
     * Grants permission to list the start times of the available aggregated profiles for a specific profiling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListProfileTimes.html
     */
    listProfileTimes() {
        this.add('codeguru-profiler:ListProfileTimes');
        return this;
    }
    /**
     * Grants permission to list profiling groups in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListProfilingGroups.html
     */
    listProfilingGroups() {
        this.add('codeguru-profiler:ListProfilingGroups');
        return this;
    }
    /**
     * Grants permission to submit a profile collected by an agent belonging to a specific profiling group for aggregation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_PostAgentProfile.html
     */
    postAgentProfile() {
        this.add('codeguru-profiler:PostAgentProfile');
        return this;
    }
    /**
     * Grants permission to update the list of principals allowed for an action group in the resource policy associated with the specified Profiling Group.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_PutPermission.html
     */
    putPermission() {
        this.add('codeguru-profiler:PutPermission');
        return this;
    }
    /**
     * Grants permission to delete an already configured SNStopic arn from the notification configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_RemoveNotificationChannel.html
     */
    removeNotificationChannel() {
        this.add('codeguru-profiler:RemoveNotificationChannel');
        return this;
    }
    /**
     * Grants permission to remove the permission of specified Action Group from the resource policy associated with the specified Profiling Group.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_RemovePermission.html
     */
    removePermission() {
        this.add('codeguru-profiler:RemovePermission');
        return this;
    }
    /**
     * Grants permission to submit user feedback for useful or non useful anomaly
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_SubmitFeedback.html
     */
    submitFeedback() {
        this.add('codeguru-profiler:SubmitFeedback');
        return this;
    }
    /**
     * Grants permission to update a specific profiling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_UpdateProfilingGroup.html
     */
    updateProfilingGroup() {
        this.add('codeguru-profiler:UpdateProfilingGroup');
        return this;
    }
    /**
     * Adds a resource of type ProfilingGroup to the statement
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-ug/working-with-profiling-groups.html
     *
     * @param profilingGroupName - Identifier for the profilingGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProfilingGroup(profilingGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:codeguru-profiler:${Region}:${Account}:profilingGroup/${ProfilingGroupName}';
        arn = arn.replace('${ProfilingGroupName}', profilingGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.CodeguruProfiler = CodeguruProfiler;
//# sourceMappingURL=data:application/json;base64,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