"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CognitoSync = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cognito-sync](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitosync.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CognitoSync extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cognito-sync](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitosync.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cognito-sync';
        this.actionList = {
            "BulkPublish": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_BulkPublish.html",
                "description": "Initiates a bulk publish of all existing datasets for an Identity Pool to the configured stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "DeleteDataset": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DeleteDataset.html",
                "description": "Deletes the specific dataset.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "DescribeDataset": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeDataset.html",
                "description": "Gets meta data about a dataset by identity and dataset name.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "DescribeIdentityPoolUsage": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeIdentityPoolUsage.html",
                "description": "Gets usage details (for example, data storage) about a particular identity pool.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "DescribeIdentityUsage": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeIdentityUsage.html",
                "description": "Gets usage information for an identity, including number of datasets and data usage.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                }
            },
            "GetBulkPublishDetails": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetBulkPublishDetails.html",
                "description": "Get the status of the last BulkPublish operation for an identity pool.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "GetCognitoEvents": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetCognitoEvents.html",
                "description": "Gets the events and the corresponding Lambda functions associated with an identity pool.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "GetIdentityPoolConfiguration": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetIdentityPoolConfiguration.html",
                "description": "Gets the configuration settings of an identity pool.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "ListDatasets": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListDatasets.html",
                "description": "Lists datasets for an identity.",
                "accessLevel": "List",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "ListIdentityPoolUsage": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListIdentityPoolUsage.html",
                "description": "Gets a list of identity pools registered with Cognito.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "ListRecords": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListRecords.html",
                "description": "Gets paginated records, optionally changed after a particular sync count for a dataset and identity.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "QueryRecords": {
                "url": "",
                "description": "A permission that grants the ability to query records.",
                "accessLevel": "Read"
            },
            "RegisterDevice": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_RegisterDevice.html",
                "description": "Registers a device to receive push sync notifications.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                }
            },
            "SetCognitoEvents": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SetCognitoEvents.html",
                "description": "Sets the AWS Lambda function for a given event type for an identity pool.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "SetDatasetConfiguration": {
                "url": "",
                "description": "A permission that grants ability to configure datasets.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "SetIdentityPoolConfiguration": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SetIdentityPoolConfiguration.html",
                "description": "Sets the necessary configuration for push sync.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "SubscribeToDataset": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SubscribeToDataset.html",
                "description": "Subscribes to receive notifications when a dataset is modified by another device.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "UnsubscribeFromDataset": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_UnsubscribeFromDataset.html",
                "description": "Unsubscribes from receiving notifications when a dataset is modified by another device.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "UpdateRecords": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_UpdateRecords.html",
                "description": "Posts updates to records and adds and deletes records for a dataset and user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "dataset": {
                "name": "dataset",
                "url": "https://docs.aws.amazon.com/cognito/latest/developerguide/synchronizing-data.html#understanding-datasets",
                "arn": "arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}/identity/${IdentityId}/dataset/${DatasetName}",
                "conditionKeys": []
            },
            "identity": {
                "name": "identity",
                "url": "https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html#authenticated-and-unauthenticated-identities",
                "arn": "arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}/identity/${IdentityId}",
                "conditionKeys": []
            },
            "identitypool": {
                "name": "identitypool",
                "url": "https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html",
                "arn": "arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Initiates a bulk publish of all existing datasets for an Identity Pool to the configured stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_BulkPublish.html
     */
    bulkPublish() {
        this.add('cognito-sync:BulkPublish');
        return this;
    }
    /**
     * Deletes the specific dataset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DeleteDataset.html
     */
    deleteDataset() {
        this.add('cognito-sync:DeleteDataset');
        return this;
    }
    /**
     * Gets meta data about a dataset by identity and dataset name.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeDataset.html
     */
    describeDataset() {
        this.add('cognito-sync:DescribeDataset');
        return this;
    }
    /**
     * Gets usage details (for example, data storage) about a particular identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeIdentityPoolUsage.html
     */
    describeIdentityPoolUsage() {
        this.add('cognito-sync:DescribeIdentityPoolUsage');
        return this;
    }
    /**
     * Gets usage information for an identity, including number of datasets and data usage.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeIdentityUsage.html
     */
    describeIdentityUsage() {
        this.add('cognito-sync:DescribeIdentityUsage');
        return this;
    }
    /**
     * Get the status of the last BulkPublish operation for an identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetBulkPublishDetails.html
     */
    getBulkPublishDetails() {
        this.add('cognito-sync:GetBulkPublishDetails');
        return this;
    }
    /**
     * Gets the events and the corresponding Lambda functions associated with an identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetCognitoEvents.html
     */
    getCognitoEvents() {
        this.add('cognito-sync:GetCognitoEvents');
        return this;
    }
    /**
     * Gets the configuration settings of an identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetIdentityPoolConfiguration.html
     */
    getIdentityPoolConfiguration() {
        this.add('cognito-sync:GetIdentityPoolConfiguration');
        return this;
    }
    /**
     * Lists datasets for an identity.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListDatasets.html
     */
    listDatasets() {
        this.add('cognito-sync:ListDatasets');
        return this;
    }
    /**
     * Gets a list of identity pools registered with Cognito.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListIdentityPoolUsage.html
     */
    listIdentityPoolUsage() {
        this.add('cognito-sync:ListIdentityPoolUsage');
        return this;
    }
    /**
     * Gets paginated records, optionally changed after a particular sync count for a dataset and identity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListRecords.html
     */
    listRecords() {
        this.add('cognito-sync:ListRecords');
        return this;
    }
    /**
     * A permission that grants the ability to query records.
     *
     * Access Level: Read
     */
    queryRecords() {
        this.add('cognito-sync:QueryRecords');
        return this;
    }
    /**
     * Registers a device to receive push sync notifications.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_RegisterDevice.html
     */
    registerDevice() {
        this.add('cognito-sync:RegisterDevice');
        return this;
    }
    /**
     * Sets the AWS Lambda function for a given event type for an identity pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SetCognitoEvents.html
     */
    setCognitoEvents() {
        this.add('cognito-sync:SetCognitoEvents');
        return this;
    }
    /**
     * A permission that grants ability to configure datasets.
     *
     * Access Level: Write
     */
    setDatasetConfiguration() {
        this.add('cognito-sync:SetDatasetConfiguration');
        return this;
    }
    /**
     * Sets the necessary configuration for push sync.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SetIdentityPoolConfiguration.html
     */
    setIdentityPoolConfiguration() {
        this.add('cognito-sync:SetIdentityPoolConfiguration');
        return this;
    }
    /**
     * Subscribes to receive notifications when a dataset is modified by another device.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SubscribeToDataset.html
     */
    subscribeToDataset() {
        this.add('cognito-sync:SubscribeToDataset');
        return this;
    }
    /**
     * Unsubscribes from receiving notifications when a dataset is modified by another device.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_UnsubscribeFromDataset.html
     */
    unsubscribeFromDataset() {
        this.add('cognito-sync:UnsubscribeFromDataset');
        return this;
    }
    /**
     * Posts updates to records and adds and deletes records for a dataset and user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_UpdateRecords.html
     */
    updateRecords() {
        this.add('cognito-sync:UpdateRecords');
        return this;
    }
    /**
     * Adds a resource of type dataset to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/synchronizing-data.html#understanding-datasets
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param identityId - Identifier for the identityId.
     * @param datasetName - Identifier for the datasetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDataset(identityPoolId, identityId, datasetName, account, region, partition) {
        var arn = 'arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}/identity/${IdentityId}/dataset/${DatasetName}';
        arn = arn.replace('${IdentityPoolId}', identityPoolId);
        arn = arn.replace('${IdentityId}', identityId);
        arn = arn.replace('${DatasetName}', datasetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type identity to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html#authenticated-and-unauthenticated-identities
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param identityId - Identifier for the identityId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIdentity(identityPoolId, identityId, account, region, partition) {
        var arn = 'arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}/identity/${IdentityId}';
        arn = arn.replace('${IdentityPoolId}', identityPoolId);
        arn = arn.replace('${IdentityId}', identityId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type identitypool to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIdentitypool(identityPoolId, account, region, partition) {
        var arn = 'arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}';
        arn = arn.replace('${IdentityPoolId}', identityPoolId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.CognitoSync = CognitoSync;
//# sourceMappingURL=data:application/json;base64,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