"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Detective = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [detective](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazondetective.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Detective extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [detective](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazondetective.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'detective';
        this.actionList = {
            "AcceptInvitation": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_AcceptInvitation.html",
                "description": "Grants permission to accept an invitation to become a member of a behavior graph",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "CreateGraph": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_CreateGraph.html",
                "description": "Grants permission to create a behavior graph and begin to aggregate security information",
                "accessLevel": "Write"
            },
            "CreateMembers": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_CreateMembers.html",
                "description": "Grants permission to request the membership of one or more accounts in a behavior graph managed by this account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "DeleteGraph": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_DeleteGraph.html",
                "description": "Grants permission to delete a behavior graph and stop aggregating security information",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "DeleteMembers": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_DeleteMembers.html",
                "description": "Grants permission to remove member accounts from a behavior graph managed by this account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "DisassociateMembership": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_DisassociateMembership.html",
                "description": "Grants permission to remove the association of this account with a behavior graph",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "GetFreeTrialEligibility": {
                "url": "",
                "description": "Grants permission to retrieve a behavior graph's eligibility for a free trial period",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "GetGraphIngestState": {
                "url": "",
                "description": "Grants permission to retrieve the data ingestion state of a behavior graph",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "GetMembers": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_GetMembers.html",
                "description": "Grants permission to retrieve details on specified members of a behavior graph",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "GetPricingInformation": {
                "url": "",
                "description": "Grants permission to retrieve information about Amazon Detective's pricing",
                "accessLevel": "Read"
            },
            "GetUsageInformation": {
                "url": "",
                "description": "Grants permission to list usage information of a behavior graph",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "ListGraphs": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_ListGraphs.html",
                "description": "Grants permission to list behavior graphs managed by this account",
                "accessLevel": "List"
            },
            "ListInvitations": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_ListInvitations.html",
                "description": "Grants permission to retrieve details on the behavior graphs to which this account has been invited to join",
                "accessLevel": "List"
            },
            "ListMembers": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_ListMembers.html",
                "description": "Grants permission to retrieve details on all members of a behavior graph",
                "accessLevel": "List",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "RejectInvitation": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_RejectInvitation.html",
                "description": "Grants permission to reject an invitation to become a member of a behavior graph",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "SearchGraph": {
                "url": "",
                "description": "Grants permission to search the data stored in a behavior graph",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            },
            "StartMonitoringMember": {
                "url": "https://docs.aws.amazon.com/detective/latest/APIReference/API_StartMonitoringMember.html",
                "description": "Grants permission to start data ingest for a member account that has a status of ACCEPTED_BUT_DISABLED.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Graph": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "Graph": {
                "name": "Graph",
                "url": "https://docs.aws.amazon.com/detective/latest/adminguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources",
                "arn": "arn:${Partition}:detective:${Region}:${Account}:graph:${ResourceId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to accept an invitation to become a member of a behavior graph
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_AcceptInvitation.html
     */
    acceptInvitation() {
        this.add('detective:AcceptInvitation');
        return this;
    }
    /**
     * Grants permission to create a behavior graph and begin to aggregate security information
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_CreateGraph.html
     */
    createGraph() {
        this.add('detective:CreateGraph');
        return this;
    }
    /**
     * Grants permission to request the membership of one or more accounts in a behavior graph managed by this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_CreateMembers.html
     */
    createMembers() {
        this.add('detective:CreateMembers');
        return this;
    }
    /**
     * Grants permission to delete a behavior graph and stop aggregating security information
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_DeleteGraph.html
     */
    deleteGraph() {
        this.add('detective:DeleteGraph');
        return this;
    }
    /**
     * Grants permission to remove member accounts from a behavior graph managed by this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_DeleteMembers.html
     */
    deleteMembers() {
        this.add('detective:DeleteMembers');
        return this;
    }
    /**
     * Grants permission to remove the association of this account with a behavior graph
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_DisassociateMembership.html
     */
    disassociateMembership() {
        this.add('detective:DisassociateMembership');
        return this;
    }
    /**
     * Grants permission to retrieve a behavior graph's eligibility for a free trial period
     *
     * Access Level: Read
     */
    getFreeTrialEligibility() {
        this.add('detective:GetFreeTrialEligibility');
        return this;
    }
    /**
     * Grants permission to retrieve the data ingestion state of a behavior graph
     *
     * Access Level: Read
     */
    getGraphIngestState() {
        this.add('detective:GetGraphIngestState');
        return this;
    }
    /**
     * Grants permission to retrieve details on specified members of a behavior graph
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_GetMembers.html
     */
    getMembers() {
        this.add('detective:GetMembers');
        return this;
    }
    /**
     * Grants permission to retrieve information about Amazon Detective's pricing
     *
     * Access Level: Read
     */
    getPricingInformation() {
        this.add('detective:GetPricingInformation');
        return this;
    }
    /**
     * Grants permission to list usage information of a behavior graph
     *
     * Access Level: Read
     */
    getUsageInformation() {
        this.add('detective:GetUsageInformation');
        return this;
    }
    /**
     * Grants permission to list behavior graphs managed by this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListGraphs.html
     */
    listGraphs() {
        this.add('detective:ListGraphs');
        return this;
    }
    /**
     * Grants permission to retrieve details on the behavior graphs to which this account has been invited to join
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListInvitations.html
     */
    listInvitations() {
        this.add('detective:ListInvitations');
        return this;
    }
    /**
     * Grants permission to retrieve details on all members of a behavior graph
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_ListMembers.html
     */
    listMembers() {
        this.add('detective:ListMembers');
        return this;
    }
    /**
     * Grants permission to reject an invitation to become a member of a behavior graph
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_RejectInvitation.html
     */
    rejectInvitation() {
        this.add('detective:RejectInvitation');
        return this;
    }
    /**
     * Grants permission to search the data stored in a behavior graph
     *
     * Access Level: Read
     */
    searchGraph() {
        this.add('detective:SearchGraph');
        return this;
    }
    /**
     * Grants permission to start data ingest for a member account that has a status of ACCEPTED_BUT_DISABLED.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/detective/latest/APIReference/API_StartMonitoringMember.html
     */
    startMonitoringMember() {
        this.add('detective:StartMonitoringMember');
        return this;
    }
    /**
     * Adds a resource of type Graph to the statement
     *
     * https://docs.aws.amazon.com/detective/latest/adminguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGraph(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:detective:${Region}:${Account}:graph:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Detective = Detective;
//# sourceMappingURL=data:application/json;base64,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