"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Autoscaling = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [autoscaling](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2autoscaling.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Autoscaling extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [autoscaling](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2autoscaling.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'autoscaling';
        this.actionList = {
            "AttachInstances": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachInstances.html",
                "description": "Attaches one or more EC2 instances to the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AttachLoadBalancerTargetGroups": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachLoadBalancerTargetGroups.html",
                "description": "Attaches one or more target groups to the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "autoscaling:TargetGroupARNs"
                ]
            },
            "AttachLoadBalancers": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachLoadBalancers.html",
                "description": "Attaches one or more load balancers to the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "autoscaling:LoadBalancerNames"
                ]
            },
            "BatchDeleteScheduledAction": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_BatchDeleteScheduledAction.html",
                "description": "Deletes the specified scheduled actions.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "BatchPutScheduledUpdateGroupAction": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_BatchPutScheduledUpdateGroupAction.html",
                "description": "Creates or updates multiple scheduled scaling actions for an Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CancelInstanceRefresh": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CancelInstanceRefresh.html",
                "description": "Grants permission to cancel an instance refresh operation in progress",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CompleteLifecycleAction": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CompleteLifecycleAction.html",
                "description": "Completes the lifecycle action for the specified token or instance with the specified result.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateAutoScalingGroup": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateAutoScalingGroup.html",
                "description": "Creates an Auto Scaling group with the specified name and attributes.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "autoscaling:InstanceTypes",
                    "autoscaling:LaunchConfigurationName",
                    "autoscaling:LaunchTemplateVersionSpecified",
                    "autoscaling:LoadBalancerNames",
                    "autoscaling:MaxSize",
                    "autoscaling:MinSize",
                    "autoscaling:TargetGroupARNs",
                    "autoscaling:VPCZoneIdentifiers",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLaunchConfiguration": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateLaunchConfiguration.html",
                "description": "Creates a launch configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launchConfiguration": {
                        "required": true
                    }
                },
                "conditions": [
                    "autoscaling:ImageId",
                    "autoscaling:InstanceType",
                    "autoscaling:SpotPrice"
                ]
            },
            "CreateOrUpdateTags": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateOrUpdateTags.html",
                "description": "Creates or updates tags for the specified Auto Scaling group.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteAutoScalingGroup": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteAutoScalingGroup.html",
                "description": "Deletes the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteLaunchConfiguration": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteLaunchConfiguration.html",
                "description": "Deletes the specified launch configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launchConfiguration": {
                        "required": true
                    }
                }
            },
            "DeleteLifecycleHook": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteLifecycleHook.html",
                "description": "Deletes the specified lifecycle hook.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteNotificationConfiguration": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteNotificationConfiguration.html",
                "description": "Deletes the specified notification.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeletePolicy": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeletePolicy.html",
                "description": "Deletes the specified Auto Scaling policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteScheduledAction": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteScheduledAction.html",
                "description": "Deletes the specified scheduled action.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTags": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteTags.html",
                "description": "Deletes the specified tags.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DescribeAccountLimits": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAccountLimits.html",
                "description": "Describes the current Auto Scaling resource limits for your AWS account.",
                "accessLevel": "List"
            },
            "DescribeAdjustmentTypes": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAdjustmentTypes.html",
                "description": "Describes the policy adjustment types for use with PutScalingPolicy.",
                "accessLevel": "List"
            },
            "DescribeAutoScalingGroups": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingGroups.html",
                "description": "Describes one or more Auto Scaling groups. If a list of names is not provided, the call describes all Auto Scaling groups.",
                "accessLevel": "List"
            },
            "DescribeAutoScalingInstances": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingInstances.html",
                "description": "Describes one or more Auto Scaling instances. If a list is not provided, the call describes all instances.",
                "accessLevel": "List"
            },
            "DescribeAutoScalingNotificationTypes": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingNotificationTypes.html",
                "description": "Describes the notification types that are supported by Auto Scaling.",
                "accessLevel": "List"
            },
            "DescribeInstanceRefreshes": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeInstanceRefreshes.html",
                "description": "Grants permission to describe one or more instance refreshes for an Auto Scaling group",
                "accessLevel": "List"
            },
            "DescribeLaunchConfigurations": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLaunchConfigurations.html",
                "description": "Describes one or more launch configurations. If you omit the list of names, then the call describes all launch configurations.",
                "accessLevel": "List"
            },
            "DescribeLifecycleHookTypes": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLifecycleHookTypes.html",
                "description": "Describes the available types of lifecycle hooks.",
                "accessLevel": "List"
            },
            "DescribeLifecycleHooks": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLifecycleHooks.html",
                "description": "Describes the lifecycle hooks for the specified Auto Scaling group.",
                "accessLevel": "List"
            },
            "DescribeLoadBalancerTargetGroups": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLoadBalancerTargetGroups.html",
                "description": "Describes the target groups for the specified Auto Scaling group.",
                "accessLevel": "List"
            },
            "DescribeLoadBalancers": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLoadBalancers.html",
                "description": "Describes the load balancers for the specified Auto Scaling group.",
                "accessLevel": "List"
            },
            "DescribeMetricCollectionTypes": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeMetricCollectionTypes.html",
                "description": "Describes the available CloudWatch metrics for Auto Scaling.",
                "accessLevel": "List"
            },
            "DescribeNotificationConfigurations": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeNotificationConfigurations.html",
                "description": "Describes the notification actions associated with the specified Auto Scaling group.",
                "accessLevel": "List"
            },
            "DescribePolicies": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribePolicies.html",
                "description": "Describes the policies for the specified Auto Scaling group.",
                "accessLevel": "List"
            },
            "DescribeScalingActivities": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScalingActivities.html",
                "description": "Describes one or more scaling activities for the specified Auto Scaling group.",
                "accessLevel": "List"
            },
            "DescribeScalingProcessTypes": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScalingProcessTypes.html",
                "description": "Describes the scaling process types for use with ResumeProcesses and SuspendProcesses.",
                "accessLevel": "List"
            },
            "DescribeScheduledActions": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScheduledActions.html",
                "description": "Describes the actions scheduled for your Auto Scaling group that haven't run.",
                "accessLevel": "List"
            },
            "DescribeTags": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeTags.html",
                "description": "Describes the specified tags.",
                "accessLevel": "Read"
            },
            "DescribeTerminationPolicyTypes": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeTerminationPolicyTypes.html",
                "description": "Describes the termination policies supported by Auto Scaling.",
                "accessLevel": "List"
            },
            "DetachInstances": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachInstances.html",
                "description": "Removes one or more instances from the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DetachLoadBalancerTargetGroups": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachLoadBalancerTargetGroups.html",
                "description": "Detaches one or more target groups from the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "autoscaling:TargetGroupARNs"
                ]
            },
            "DetachLoadBalancers": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachLoadBalancers.html",
                "description": "Removes one or more load balancers from the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "autoscaling:LoadBalancerNames"
                ]
            },
            "DisableMetricsCollection": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DisableMetricsCollection.html",
                "description": "Disables monitoring of the specified metrics for the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "EnableMetricsCollection": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_EnableMetricsCollection.html",
                "description": "Enables monitoring of the specified metrics for the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "EnterStandby": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_EnterStandby.html",
                "description": "Moves the specified instances into Standby mode.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ExecutePolicy": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ExecutePolicy.html",
                "description": "Executes the specified policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ExitStandby": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ExitStandby.html",
                "description": "Moves the specified instances out of Standby mode.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "PutLifecycleHook": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutLifecycleHook.html",
                "description": "Creates or updates a lifecycle hook for the specified Auto Scaling Group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "PutNotificationConfiguration": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutNotificationConfiguration.html",
                "description": "Configures an Auto Scaling group to send notifications when specified events take place.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "PutScalingPolicy": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutScalingPolicy.html",
                "description": "Creates or updates a policy for an Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "PutScheduledUpdateGroupAction": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutScheduledUpdateGroupAction.html",
                "description": "Creates or updates a scheduled scaling action for an Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "autoscaling:MaxSize",
                    "autoscaling:MinSize"
                ]
            },
            "RecordLifecycleActionHeartbeat": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_RecordLifecycleActionHeartbeat.html",
                "description": "Records a heartbeat for the lifecycle action associated with the specified token or instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ResumeProcesses": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ResumeProcesses.html",
                "description": "Resumes the specified suspended Auto Scaling processes, or all suspended process, for the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "SetDesiredCapacity": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetDesiredCapacity.html",
                "description": "Sets the size of the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "SetInstanceHealth": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetInstanceHealth.html",
                "description": "Sets the health status of the specified instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "SetInstanceProtection": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetInstanceProtection.html",
                "description": "Updates the instance protection settings of the specified instances.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "StartInstanceRefresh": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_StartInstanceRefresh.html",
                "description": "Grants permission to start a new instance refresh operation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "SuspendProcesses": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SuspendProcesses.html",
                "description": "Suspends the specified Auto Scaling processes, or all processes, for the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "TerminateInstanceInAutoScalingGroup": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_TerminateInstanceInAutoScalingGroup.html",
                "description": "Terminates the specified instance and optionally adjusts the desired group size.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "UpdateAutoScalingGroup": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_UpdateAutoScalingGroup.html",
                "description": "Updates the configuration for the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "autoscaling:InstanceTypes",
                    "autoscaling:LaunchConfigurationName",
                    "autoscaling:LaunchTemplateVersionSpecified",
                    "autoscaling:MaxSize",
                    "autoscaling:MinSize",
                    "autoscaling:VPCZoneIdentifiers"
                ]
            }
        };
        this.resourceTypes = {
            "autoScalingGroup": {
                "name": "autoScalingGroup",
                "url": "https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-resources",
                "arn": "arn:${Partition}:autoscaling:${Region}:${Account}:autoScalingGroup:${GroupId}:autoScalingGroupName/${GroupFriendlyName}",
                "conditionKeys": [
                    "autoscaling:ResourceTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "launchConfiguration": {
                "name": "launchConfiguration",
                "url": "https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-resources",
                "arn": "arn:${Partition}:autoscaling:${Region}:${Account}:launchConfiguration:${Id}:launchConfigurationName/${LaunchConfigurationName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Attaches one or more EC2 instances to the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachInstances.html
     */
    attachInstances() {
        this.add('autoscaling:AttachInstances');
        return this;
    }
    /**
     * Attaches one or more target groups to the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachLoadBalancerTargetGroups.html
     */
    attachLoadBalancerTargetGroups() {
        this.add('autoscaling:AttachLoadBalancerTargetGroups');
        return this;
    }
    /**
     * Attaches one or more load balancers to the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachLoadBalancers.html
     */
    attachLoadBalancers() {
        this.add('autoscaling:AttachLoadBalancers');
        return this;
    }
    /**
     * Deletes the specified scheduled actions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_BatchDeleteScheduledAction.html
     */
    batchDeleteScheduledAction() {
        this.add('autoscaling:BatchDeleteScheduledAction');
        return this;
    }
    /**
     * Creates or updates multiple scheduled scaling actions for an Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_BatchPutScheduledUpdateGroupAction.html
     */
    batchPutScheduledUpdateGroupAction() {
        this.add('autoscaling:BatchPutScheduledUpdateGroupAction');
        return this;
    }
    /**
     * Grants permission to cancel an instance refresh operation in progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CancelInstanceRefresh.html
     */
    cancelInstanceRefresh() {
        this.add('autoscaling:CancelInstanceRefresh');
        return this;
    }
    /**
     * Completes the lifecycle action for the specified token or instance with the specified result.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CompleteLifecycleAction.html
     */
    completeLifecycleAction() {
        this.add('autoscaling:CompleteLifecycleAction');
        return this;
    }
    /**
     * Creates an Auto Scaling group with the specified name and attributes.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateAutoScalingGroup.html
     */
    createAutoScalingGroup() {
        this.add('autoscaling:CreateAutoScalingGroup');
        return this;
    }
    /**
     * Creates a launch configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateLaunchConfiguration.html
     */
    createLaunchConfiguration() {
        this.add('autoscaling:CreateLaunchConfiguration');
        return this;
    }
    /**
     * Creates or updates tags for the specified Auto Scaling group.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateOrUpdateTags.html
     */
    createOrUpdateTags() {
        this.add('autoscaling:CreateOrUpdateTags');
        return this;
    }
    /**
     * Deletes the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteAutoScalingGroup.html
     */
    deleteAutoScalingGroup() {
        this.add('autoscaling:DeleteAutoScalingGroup');
        return this;
    }
    /**
     * Deletes the specified launch configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteLaunchConfiguration.html
     */
    deleteLaunchConfiguration() {
        this.add('autoscaling:DeleteLaunchConfiguration');
        return this;
    }
    /**
     * Deletes the specified lifecycle hook.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteLifecycleHook.html
     */
    deleteLifecycleHook() {
        this.add('autoscaling:DeleteLifecycleHook');
        return this;
    }
    /**
     * Deletes the specified notification.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteNotificationConfiguration.html
     */
    deleteNotificationConfiguration() {
        this.add('autoscaling:DeleteNotificationConfiguration');
        return this;
    }
    /**
     * Deletes the specified Auto Scaling policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeletePolicy.html
     */
    deletePolicy() {
        this.add('autoscaling:DeletePolicy');
        return this;
    }
    /**
     * Deletes the specified scheduled action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteScheduledAction.html
     */
    deleteScheduledAction() {
        this.add('autoscaling:DeleteScheduledAction');
        return this;
    }
    /**
     * Deletes the specified tags.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteTags.html
     */
    deleteTags() {
        this.add('autoscaling:DeleteTags');
        return this;
    }
    /**
     * Describes the current Auto Scaling resource limits for your AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAccountLimits.html
     */
    describeAccountLimits() {
        this.add('autoscaling:DescribeAccountLimits');
        return this;
    }
    /**
     * Describes the policy adjustment types for use with PutScalingPolicy.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAdjustmentTypes.html
     */
    describeAdjustmentTypes() {
        this.add('autoscaling:DescribeAdjustmentTypes');
        return this;
    }
    /**
     * Describes one or more Auto Scaling groups. If a list of names is not provided, the call describes all Auto Scaling groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingGroups.html
     */
    describeAutoScalingGroups() {
        this.add('autoscaling:DescribeAutoScalingGroups');
        return this;
    }
    /**
     * Describes one or more Auto Scaling instances. If a list is not provided, the call describes all instances.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingInstances.html
     */
    describeAutoScalingInstances() {
        this.add('autoscaling:DescribeAutoScalingInstances');
        return this;
    }
    /**
     * Describes the notification types that are supported by Auto Scaling.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingNotificationTypes.html
     */
    describeAutoScalingNotificationTypes() {
        this.add('autoscaling:DescribeAutoScalingNotificationTypes');
        return this;
    }
    /**
     * Grants permission to describe one or more instance refreshes for an Auto Scaling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeInstanceRefreshes.html
     */
    describeInstanceRefreshes() {
        this.add('autoscaling:DescribeInstanceRefreshes');
        return this;
    }
    /**
     * Describes one or more launch configurations. If you omit the list of names, then the call describes all launch configurations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLaunchConfigurations.html
     */
    describeLaunchConfigurations() {
        this.add('autoscaling:DescribeLaunchConfigurations');
        return this;
    }
    /**
     * Describes the available types of lifecycle hooks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLifecycleHookTypes.html
     */
    describeLifecycleHookTypes() {
        this.add('autoscaling:DescribeLifecycleHookTypes');
        return this;
    }
    /**
     * Describes the lifecycle hooks for the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLifecycleHooks.html
     */
    describeLifecycleHooks() {
        this.add('autoscaling:DescribeLifecycleHooks');
        return this;
    }
    /**
     * Describes the target groups for the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLoadBalancerTargetGroups.html
     */
    describeLoadBalancerTargetGroups() {
        this.add('autoscaling:DescribeLoadBalancerTargetGroups');
        return this;
    }
    /**
     * Describes the load balancers for the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLoadBalancers.html
     */
    describeLoadBalancers() {
        this.add('autoscaling:DescribeLoadBalancers');
        return this;
    }
    /**
     * Describes the available CloudWatch metrics for Auto Scaling.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeMetricCollectionTypes.html
     */
    describeMetricCollectionTypes() {
        this.add('autoscaling:DescribeMetricCollectionTypes');
        return this;
    }
    /**
     * Describes the notification actions associated with the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeNotificationConfigurations.html
     */
    describeNotificationConfigurations() {
        this.add('autoscaling:DescribeNotificationConfigurations');
        return this;
    }
    /**
     * Describes the policies for the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribePolicies.html
     */
    describePolicies() {
        this.add('autoscaling:DescribePolicies');
        return this;
    }
    /**
     * Describes one or more scaling activities for the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScalingActivities.html
     */
    describeScalingActivities() {
        this.add('autoscaling:DescribeScalingActivities');
        return this;
    }
    /**
     * Describes the scaling process types for use with ResumeProcesses and SuspendProcesses.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScalingProcessTypes.html
     */
    describeScalingProcessTypes() {
        this.add('autoscaling:DescribeScalingProcessTypes');
        return this;
    }
    /**
     * Describes the actions scheduled for your Auto Scaling group that haven't run.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScheduledActions.html
     */
    describeScheduledActions() {
        this.add('autoscaling:DescribeScheduledActions');
        return this;
    }
    /**
     * Describes the specified tags.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeTags.html
     */
    describeTags() {
        this.add('autoscaling:DescribeTags');
        return this;
    }
    /**
     * Describes the termination policies supported by Auto Scaling.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeTerminationPolicyTypes.html
     */
    describeTerminationPolicyTypes() {
        this.add('autoscaling:DescribeTerminationPolicyTypes');
        return this;
    }
    /**
     * Removes one or more instances from the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachInstances.html
     */
    detachInstances() {
        this.add('autoscaling:DetachInstances');
        return this;
    }
    /**
     * Detaches one or more target groups from the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachLoadBalancerTargetGroups.html
     */
    detachLoadBalancerTargetGroups() {
        this.add('autoscaling:DetachLoadBalancerTargetGroups');
        return this;
    }
    /**
     * Removes one or more load balancers from the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachLoadBalancers.html
     */
    detachLoadBalancers() {
        this.add('autoscaling:DetachLoadBalancers');
        return this;
    }
    /**
     * Disables monitoring of the specified metrics for the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DisableMetricsCollection.html
     */
    disableMetricsCollection() {
        this.add('autoscaling:DisableMetricsCollection');
        return this;
    }
    /**
     * Enables monitoring of the specified metrics for the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_EnableMetricsCollection.html
     */
    enableMetricsCollection() {
        this.add('autoscaling:EnableMetricsCollection');
        return this;
    }
    /**
     * Moves the specified instances into Standby mode.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_EnterStandby.html
     */
    enterStandby() {
        this.add('autoscaling:EnterStandby');
        return this;
    }
    /**
     * Executes the specified policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ExecutePolicy.html
     */
    executePolicy() {
        this.add('autoscaling:ExecutePolicy');
        return this;
    }
    /**
     * Moves the specified instances out of Standby mode.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ExitStandby.html
     */
    exitStandby() {
        this.add('autoscaling:ExitStandby');
        return this;
    }
    /**
     * Creates or updates a lifecycle hook for the specified Auto Scaling Group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutLifecycleHook.html
     */
    putLifecycleHook() {
        this.add('autoscaling:PutLifecycleHook');
        return this;
    }
    /**
     * Configures an Auto Scaling group to send notifications when specified events take place.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutNotificationConfiguration.html
     */
    putNotificationConfiguration() {
        this.add('autoscaling:PutNotificationConfiguration');
        return this;
    }
    /**
     * Creates or updates a policy for an Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutScalingPolicy.html
     */
    putScalingPolicy() {
        this.add('autoscaling:PutScalingPolicy');
        return this;
    }
    /**
     * Creates or updates a scheduled scaling action for an Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutScheduledUpdateGroupAction.html
     */
    putScheduledUpdateGroupAction() {
        this.add('autoscaling:PutScheduledUpdateGroupAction');
        return this;
    }
    /**
     * Records a heartbeat for the lifecycle action associated with the specified token or instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_RecordLifecycleActionHeartbeat.html
     */
    recordLifecycleActionHeartbeat() {
        this.add('autoscaling:RecordLifecycleActionHeartbeat');
        return this;
    }
    /**
     * Resumes the specified suspended Auto Scaling processes, or all suspended process, for the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ResumeProcesses.html
     */
    resumeProcesses() {
        this.add('autoscaling:ResumeProcesses');
        return this;
    }
    /**
     * Sets the size of the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetDesiredCapacity.html
     */
    setDesiredCapacity() {
        this.add('autoscaling:SetDesiredCapacity');
        return this;
    }
    /**
     * Sets the health status of the specified instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetInstanceHealth.html
     */
    setInstanceHealth() {
        this.add('autoscaling:SetInstanceHealth');
        return this;
    }
    /**
     * Updates the instance protection settings of the specified instances.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetInstanceProtection.html
     */
    setInstanceProtection() {
        this.add('autoscaling:SetInstanceProtection');
        return this;
    }
    /**
     * Grants permission to start a new instance refresh operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_StartInstanceRefresh.html
     */
    startInstanceRefresh() {
        this.add('autoscaling:StartInstanceRefresh');
        return this;
    }
    /**
     * Suspends the specified Auto Scaling processes, or all processes, for the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SuspendProcesses.html
     */
    suspendProcesses() {
        this.add('autoscaling:SuspendProcesses');
        return this;
    }
    /**
     * Terminates the specified instance and optionally adjusts the desired group size.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_TerminateInstanceInAutoScalingGroup.html
     */
    terminateInstanceInAutoScalingGroup() {
        this.add('autoscaling:TerminateInstanceInAutoScalingGroup');
        return this;
    }
    /**
     * Updates the configuration for the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_UpdateAutoScalingGroup.html
     */
    updateAutoScalingGroup() {
        this.add('autoscaling:UpdateAutoScalingGroup');
        return this;
    }
    /**
     * Adds a resource of type autoScalingGroup to the statement
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-resources
     *
     * @param groupId - Identifier for the groupId.
     * @param groupFriendlyName - Identifier for the groupFriendlyName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - autoscaling:ResourceTag/${TagKey}
     *  - aws:ResourceTag/${TagKey}
     */
    onAutoScalingGroup(groupId, groupFriendlyName, account, region, partition) {
        var arn = 'arn:${Partition}:autoscaling:${Region}:${Account}:autoScalingGroup:${GroupId}:autoScalingGroupName/${GroupFriendlyName}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${GroupFriendlyName}', groupFriendlyName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type launchConfiguration to the statement
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-resources
     *
     * @param id - Identifier for the id.
     * @param launchConfigurationName - Identifier for the launchConfigurationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLaunchConfiguration(id, launchConfigurationName, account, region, partition) {
        var arn = 'arn:${Partition}:autoscaling:${Region}:${Account}:launchConfiguration:${Id}:launchConfigurationName/${LaunchConfigurationName}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${LaunchConfigurationName}', launchConfigurationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The AMI used to create the instance.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImageId(value, operator) {
        return this.if(`autoscaling:ImageId`, value, operator || 'StringLike');
    }
    /**
     * The type of instance, in terms of the hardware resources available.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceType(value, operator) {
        return this.if(`autoscaling:InstanceType`, value, operator || 'StringLike');
    }
    /**
     * The types of instances, in terms of the hardware resources available.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceTypes(value, operator) {
        return this.if(`autoscaling:InstanceTypes`, value, operator || 'StringLike');
    }
    /**
     * The name of a launch configuration.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLaunchConfigurationName(value, operator) {
        return this.if(`autoscaling:LaunchConfigurationName`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether users can specify any version of a launch template or only the Latest or Default version
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifLaunchTemplateVersionSpecified(value) {
        return this.if(`autoscaling:LaunchTemplateVersionSpecified`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * The name of the load balancer.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLoadBalancerNames(value, operator) {
        return this.if(`autoscaling:LoadBalancerNames`, value, operator || 'StringLike');
    }
    /**
     * The maximum scaling size.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaxSize(value, operator) {
        return this.if(`autoscaling:MaxSize`, value, operator || 'NumericEquals');
    }
    /**
     * The minimum scaling size.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMinSize(value, operator) {
        return this.if(`autoscaling:MinSize`, value, operator || 'NumericEquals');
    }
    /**
     * The value of a tag attached to a resource.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`autoscaling:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * The spot price associated with an instance.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSpotPrice(value, operator) {
        return this.if(`autoscaling:SpotPrice`, value, operator || 'NumericEquals');
    }
    /**
     * The ARN of a target group.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifTargetGroupARNs(value, operator) {
        return this.if(`autoscaling:TargetGroupARNs`, value, operator || 'ArnEquals');
    }
    /**
     * The identifier of a VPC zone.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVPCZoneIdentifiers(value, operator) {
        return this.if(`autoscaling:VPCZoneIdentifiers`, value, operator || 'StringLike');
    }
}
exports.Autoscaling = Autoscaling;
//# sourceMappingURL=data:application/json;base64,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