"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Gamelift = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [gamelift](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazongamelift.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Gamelift extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [gamelift](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazongamelift.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'gamelift';
        this.actionList = {
            "AcceptMatch": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_AcceptMatch.html",
                "description": "Registers player acceptance or rejection of a proposed FlexMatch match.",
                "accessLevel": "Write"
            },
            "CreateAlias": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateAlias.html",
                "description": "Defines a new alias for a fleet.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateBuild": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateBuild.html",
                "description": "Creates a new game build using files stored in an Amazon S3 bucket.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateFleet": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateFleet.html",
                "description": "Creates a new fleet of computing resources to run your game servers.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateGameSession": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateGameSession.html",
                "description": "Starts a new game session on a specified fleet.",
                "accessLevel": "Write"
            },
            "CreateGameSessionQueue": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateGameSessionQueue.html",
                "description": "Sets up a new queue for processing new game session placement requests.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateMatchmakingConfiguration": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateMatchmakingConfiguration.html",
                "description": "Creates a new FlexMatch matchmaker.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateMatchmakingRuleSet": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateMatchmakingRuleSet.html",
                "description": "Creates a new matchmaking rule set for FlexMatch.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePlayerSession": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreatePlayerSession.html",
                "description": "Reserves an available game session slot for a player.",
                "accessLevel": "Write"
            },
            "CreatePlayerSessions": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreatePlayerSessions.html",
                "description": "Reserves available game session slots for multiple players.",
                "accessLevel": "Write"
            },
            "CreateScript": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateScript.html",
                "description": "Creates a new Realtime Servers script.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateVpcPeeringAuthorization": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateVpcPeeringAuthorization.html",
                "description": "Allows GameLift to create or delete a peering connection between a GameLift fleet VPC and a VPC on another AWS account.",
                "accessLevel": "Write"
            },
            "CreateVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateVpcPeeringConnection.html",
                "description": "Establishes a peering connection between your GameLift fleet VPC and a VPC on another account.",
                "accessLevel": "Write"
            },
            "DeleteAlias": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteAlias.html",
                "description": "Deletes an alias.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    }
                }
            },
            "DeleteBuild": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteBuild.html",
                "description": "Deletes a game build.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "build": {
                        "required": true
                    }
                }
            },
            "DeleteFleet": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteFleet.html",
                "description": "Deletes an empty fleet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DeleteGameSessionQueue": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteGameSessionQueue.html",
                "description": "Deletes an existing game session queue.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "gameSessionQueue": {
                        "required": true
                    }
                }
            },
            "DeleteMatchmakingConfiguration": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteMatchmakingConfiguration.html",
                "description": "Deletes an existing FlexMatch matchmaker.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "matchmakingConfiguration": {
                        "required": true
                    }
                }
            },
            "DeleteMatchmakingRuleSet": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteMatchmakingRuleSet.html",
                "description": "Deletes an existing FlexMatch matchmaking rule set.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "matchmakingRuleSet": {
                        "required": true
                    }
                }
            },
            "DeleteScalingPolicy": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteScalingPolicy.html",
                "description": "Deletes a set of auto-scaling rules.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DeleteScript": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteScript.html",
                "description": "Deletes a Realtime Servers script.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "script": {
                        "required": true
                    }
                }
            },
            "DeleteVpcPeeringAuthorization": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteVpcPeeringAuthorization.html",
                "description": "Cancels a VPC peering authorization.",
                "accessLevel": "Write"
            },
            "DeleteVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteVpcPeeringConnection.html",
                "description": "Removes a peering connection between VPCs.",
                "accessLevel": "Write"
            },
            "DescribeAlias": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeAlias.html",
                "description": "Retrieves properties for an alias.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    }
                }
            },
            "DescribeBuild": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeBuild.html",
                "description": "Retrieves properties for a game build.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "build": {
                        "required": true
                    }
                }
            },
            "DescribeEC2InstanceLimits": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeEC2InstanceLimits.html",
                "description": "Retrieves the maximum allowed and current usage for EC2 instance types.",
                "accessLevel": "Read"
            },
            "DescribeFleetAttributes": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetAttributes.html",
                "description": "Retrieves general properties, including status, for fleets.",
                "accessLevel": "Read"
            },
            "DescribeFleetCapacity": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetCapacity.html",
                "description": "Retrieves the current capacity setting for fleets.",
                "accessLevel": "Read"
            },
            "DescribeFleetEvents": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetEvents.html",
                "description": "Retrieves entries from a fleet's event log.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeFleetPortSettings": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetPortSettings.html",
                "description": "Retrieves the inbound connection permissions for a fleet.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeFleetUtilization": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetUtilization.html",
                "description": "Retrieves utilization statistics for fleets.",
                "accessLevel": "Read"
            },
            "DescribeGameSessionDetails": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessionDetails.html",
                "description": "Retrieves properties for game sessions in a fleet, including the protection policy.",
                "accessLevel": "Read"
            },
            "DescribeGameSessionPlacement": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessionPlacement.html",
                "description": "Retrieves details of a game session placement request.",
                "accessLevel": "Read"
            },
            "DescribeGameSessionQueues": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessionQueues.html",
                "description": "Retrieves properties for game session queues.",
                "accessLevel": "Read"
            },
            "DescribeGameSessions": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessions.html",
                "description": "Retrieves properties for game sessions in a fleet.",
                "accessLevel": "Read"
            },
            "DescribeInstances": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeInstances.html",
                "description": "Retrieves information about instances in a fleet.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeMatchmaking": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmaking.html",
                "description": "Retrieves details of matchmaking tickets.",
                "accessLevel": "Read"
            },
            "DescribeMatchmakingConfigurations": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmakingConfigurations.html",
                "description": "Retrieves properties for FlexMatch matchmakers.",
                "accessLevel": "Read"
            },
            "DescribeMatchmakingRuleSets": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmakingRuleSets.html",
                "description": "Retrieves properties for FlexMatch matchmaking rule sets.",
                "accessLevel": "Read"
            },
            "DescribePlayerSessions": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribePlayerSessions.html",
                "description": "Retrieves properties for player sessions in a game session.",
                "accessLevel": "Read"
            },
            "DescribeRuntimeConfiguration": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeRuntimeConfiguration.html",
                "description": "Retrieves the current runtime configuration for a fleet.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeScalingPolicies": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeScalingPolicies.html",
                "description": "Retrieves all scaling policies that are applied to a fleet.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeScript": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeScript.html",
                "description": "Retrieves properties for a Realtime Servers script.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "script": {
                        "required": true
                    }
                }
            },
            "DescribeVpcPeeringAuthorizations": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeVpcPeeringAuthorizations.html",
                "description": "Retrieves valid VPC peering authorizations.",
                "accessLevel": "Read"
            },
            "DescribeVpcPeeringConnections": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeVpcPeeringConnections.html",
                "description": "Retrieves details on active or pending VPC peering connections.",
                "accessLevel": "Read"
            },
            "GetGameSessionLogUrl": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_GetGameSessionLogUrl.html",
                "description": "Retrieves the location of stored logs for a game session.",
                "accessLevel": "Read"
            },
            "GetInstanceAccess": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_GetInstanceAccess.html",
                "description": "Requests remote access to a specified fleet instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "ListAliases": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListAliases.html",
                "description": "Retrieves all aliases that are defined in the current region.",
                "accessLevel": "List"
            },
            "ListBuilds": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListBuilds.html",
                "description": "Retrieves all game build in the current region.",
                "accessLevel": "List"
            },
            "ListFleets": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListFleets.html",
                "description": "Retrieves a list of fleet IDs for all fleets in the current region.",
                "accessLevel": "List"
            },
            "ListScripts": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListScripts.html",
                "description": "Retrieves properties for all Realtime Servers scripts in the current region.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListTagsForResource.html",
                "description": "List Tags for GameLift Resource",
                "accessLevel": "List",
                "resourceTypes": {
                    "alias": {
                        "required": false
                    },
                    "build": {
                        "required": false
                    },
                    "fleet": {
                        "required": false
                    },
                    "gameSessionQueue": {
                        "required": false
                    },
                    "matchmakingConfiguration": {
                        "required": false
                    },
                    "matchmakingRuleSet": {
                        "required": false
                    },
                    "script": {
                        "required": false
                    }
                }
            },
            "PutScalingPolicy": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_PutScalingPolicy.html",
                "description": "Creates or updates a fleet auto-scaling policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "RequestUploadCredentials": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_RequestUploadCredentials.html",
                "description": "Retrieves fresh upload credentials to use when uploading a new game build.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "build": {
                        "required": true
                    }
                }
            },
            "ResolveAlias": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_ResolveAlias.html",
                "description": "Retrieves the fleet ID associated with an alias.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    }
                }
            },
            "SearchGameSessions": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_SearchGameSessions.html",
                "description": "Retrieves game sessions that match a set of search criteria.",
                "accessLevel": "Read"
            },
            "StartFleetActions": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartFleetActions.html",
                "description": "Resumes auto-scaling activity on a fleet after it was suspended with StopFleetActions().",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "StartGameSessionPlacement": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartGameSessionPlacement.html",
                "description": "Sends a game session placement request to a game session queue.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "gameSessionQueue": {
                        "required": true
                    }
                }
            },
            "StartMatchBackfill": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartMatchBackfill.html",
                "description": "Requests FlexMatch matchmaking to fill available player slots in an existing game session.",
                "accessLevel": "Write"
            },
            "StartMatchmaking": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartMatchmaking.html",
                "description": "Requests FlexMatch matchmaking for one or a group of players and game session placement for a resulting match.",
                "accessLevel": "Write"
            },
            "StopFleetActions": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html",
                "description": "Suspends auto-scaling activity on a fleet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "StopGameSessionPlacement": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopGameSessionPlacement.html",
                "description": "Cancels a game session placement request that is in progress.",
                "accessLevel": "Write"
            },
            "StopMatchmaking": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopMatchmaking.html",
                "description": "Cancels a matchmaking or match backfill request that is in progress.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_TagResource.html",
                "description": "Tags GameLift Resources",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "alias": {
                        "required": false
                    },
                    "build": {
                        "required": false
                    },
                    "fleet": {
                        "required": false
                    },
                    "gameSessionQueue": {
                        "required": false
                    },
                    "matchmakingConfiguration": {
                        "required": false
                    },
                    "matchmakingRuleSet": {
                        "required": false
                    },
                    "script": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UntagResource.html",
                "description": "Untagging GameLift Resources",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "alias": {
                        "required": false
                    },
                    "build": {
                        "required": false
                    },
                    "fleet": {
                        "required": false
                    },
                    "gameSessionQueue": {
                        "required": false
                    },
                    "matchmakingConfiguration": {
                        "required": false
                    },
                    "matchmakingRuleSet": {
                        "required": false
                    },
                    "script": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateAlias": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateAlias.html",
                "description": "Updates the properties of an existing alias.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "alias": {
                        "required": true
                    }
                }
            },
            "UpdateBuild": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateBuild.html",
                "description": "Updates an existing build's metadata.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "build": {
                        "required": true
                    }
                }
            },
            "UpdateFleetAttributes": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateFleetAttributes.html",
                "description": "Updates the general properties of an existing fleet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateFleetCapacity": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateFleetCapacity.html",
                "description": "Adjusts a fleet's capacity settings.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateFleetPortSettings": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateFleetPortSettings.html",
                "description": "Adjusts a fleet's port settings.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateGameSession": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateGameSession.html",
                "description": "Updates the properties of an existing game session.",
                "accessLevel": "Write"
            },
            "UpdateGameSessionQueue": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateGameSessionQueue.html",
                "description": "Updates properties of an existing game session queue.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "gameSessionQueue": {
                        "required": true
                    }
                }
            },
            "UpdateMatchmakingConfiguration": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateMatchmakingConfiguration.html",
                "description": "Updates properties of an existing FlexMatch matchmaking configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "matchmakingConfiguration": {
                        "required": true
                    }
                }
            },
            "UpdateRuntimeConfiguration": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateRuntimeConfiguration.html",
                "description": "Updates how server processes are configured on instances in an existing fleet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateScript": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateScript.html",
                "description": "Updates the metadata and content of an existing Realtime Servers script.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "script": {
                        "required": true
                    }
                }
            },
            "ValidateMatchmakingRuleSet": {
                "url": "https://docs.aws.amazon.com/gamelift/latest/apireference/API_ValidateMatchmakingRuleSet.html",
                "description": "Validates the syntax of a FlexMatch matchmaking rule set.",
                "accessLevel": "Read"
            }
        };
        this.resourceTypes = {
            "alias": {
                "name": "alias",
                "url": "https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html",
                "arn": "arn:${Partition}:gamelift:${Region}::alias/${AliasId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "build": {
                "name": "build",
                "url": "https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html",
                "arn": "arn:${Partition}:gamelift:${Region}:${AccountId}:build/${BuildId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "script": {
                "name": "script",
                "url": "https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html",
                "arn": "arn:${Partition}:gamelift:${Region}:${AccountId}:script/${ScriptId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "fleet": {
                "name": "fleet",
                "url": "https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html",
                "arn": "arn:${Partition}:gamelift:${Region}:${Account}:fleet/${FleetId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "gameSessionQueue": {
                "name": "gameSessionQueue",
                "url": "https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html",
                "arn": "arn:${Partition}:gamelift:${Region}:${Account}:gamesessionqueue/${GameSessionQueueName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "matchmakingConfiguration": {
                "name": "matchmakingConfiguration",
                "url": "https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html",
                "arn": "arn:${Partition}:gamelift:${Region}:${Account}:matchmakingconfiguration/${MatchmakingConfigurationName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "matchmakingRuleSet": {
                "name": "matchmakingRuleSet",
                "url": "https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html",
                "arn": "arn:${Partition}:gamelift:${Region}:${Account}:matchmakingruleset/${MatchmakingRuleSetName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Registers player acceptance or rejection of a proposed FlexMatch match.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_AcceptMatch.html
     */
    acceptMatch() {
        this.add('gamelift:AcceptMatch');
        return this;
    }
    /**
     * Defines a new alias for a fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateAlias.html
     */
    createAlias() {
        this.add('gamelift:CreateAlias');
        return this;
    }
    /**
     * Creates a new game build using files stored in an Amazon S3 bucket.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateBuild.html
     */
    createBuild() {
        this.add('gamelift:CreateBuild');
        return this;
    }
    /**
     * Creates a new fleet of computing resources to run your game servers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateFleet.html
     */
    createFleet() {
        this.add('gamelift:CreateFleet');
        return this;
    }
    /**
     * Starts a new game session on a specified fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateGameSession.html
     */
    createGameSession() {
        this.add('gamelift:CreateGameSession');
        return this;
    }
    /**
     * Sets up a new queue for processing new game session placement requests.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateGameSessionQueue.html
     */
    createGameSessionQueue() {
        this.add('gamelift:CreateGameSessionQueue');
        return this;
    }
    /**
     * Creates a new FlexMatch matchmaker.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateMatchmakingConfiguration.html
     */
    createMatchmakingConfiguration() {
        this.add('gamelift:CreateMatchmakingConfiguration');
        return this;
    }
    /**
     * Creates a new matchmaking rule set for FlexMatch.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateMatchmakingRuleSet.html
     */
    createMatchmakingRuleSet() {
        this.add('gamelift:CreateMatchmakingRuleSet');
        return this;
    }
    /**
     * Reserves an available game session slot for a player.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreatePlayerSession.html
     */
    createPlayerSession() {
        this.add('gamelift:CreatePlayerSession');
        return this;
    }
    /**
     * Reserves available game session slots for multiple players.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreatePlayerSessions.html
     */
    createPlayerSessions() {
        this.add('gamelift:CreatePlayerSessions');
        return this;
    }
    /**
     * Creates a new Realtime Servers script.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateScript.html
     */
    createScript() {
        this.add('gamelift:CreateScript');
        return this;
    }
    /**
     * Allows GameLift to create or delete a peering connection between a GameLift fleet VPC and a VPC on another AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateVpcPeeringAuthorization.html
     */
    createVpcPeeringAuthorization() {
        this.add('gamelift:CreateVpcPeeringAuthorization');
        return this;
    }
    /**
     * Establishes a peering connection between your GameLift fleet VPC and a VPC on another account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateVpcPeeringConnection.html
     */
    createVpcPeeringConnection() {
        this.add('gamelift:CreateVpcPeeringConnection');
        return this;
    }
    /**
     * Deletes an alias.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteAlias.html
     */
    deleteAlias() {
        this.add('gamelift:DeleteAlias');
        return this;
    }
    /**
     * Deletes a game build.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteBuild.html
     */
    deleteBuild() {
        this.add('gamelift:DeleteBuild');
        return this;
    }
    /**
     * Deletes an empty fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteFleet.html
     */
    deleteFleet() {
        this.add('gamelift:DeleteFleet');
        return this;
    }
    /**
     * Deletes an existing game session queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteGameSessionQueue.html
     */
    deleteGameSessionQueue() {
        this.add('gamelift:DeleteGameSessionQueue');
        return this;
    }
    /**
     * Deletes an existing FlexMatch matchmaker.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteMatchmakingConfiguration.html
     */
    deleteMatchmakingConfiguration() {
        this.add('gamelift:DeleteMatchmakingConfiguration');
        return this;
    }
    /**
     * Deletes an existing FlexMatch matchmaking rule set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteMatchmakingRuleSet.html
     */
    deleteMatchmakingRuleSet() {
        this.add('gamelift:DeleteMatchmakingRuleSet');
        return this;
    }
    /**
     * Deletes a set of auto-scaling rules.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteScalingPolicy.html
     */
    deleteScalingPolicy() {
        this.add('gamelift:DeleteScalingPolicy');
        return this;
    }
    /**
     * Deletes a Realtime Servers script.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteScript.html
     */
    deleteScript() {
        this.add('gamelift:DeleteScript');
        return this;
    }
    /**
     * Cancels a VPC peering authorization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteVpcPeeringAuthorization.html
     */
    deleteVpcPeeringAuthorization() {
        this.add('gamelift:DeleteVpcPeeringAuthorization');
        return this;
    }
    /**
     * Removes a peering connection between VPCs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteVpcPeeringConnection.html
     */
    deleteVpcPeeringConnection() {
        this.add('gamelift:DeleteVpcPeeringConnection');
        return this;
    }
    /**
     * Retrieves properties for an alias.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeAlias.html
     */
    describeAlias() {
        this.add('gamelift:DescribeAlias');
        return this;
    }
    /**
     * Retrieves properties for a game build.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeBuild.html
     */
    describeBuild() {
        this.add('gamelift:DescribeBuild');
        return this;
    }
    /**
     * Retrieves the maximum allowed and current usage for EC2 instance types.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeEC2InstanceLimits.html
     */
    describeEC2InstanceLimits() {
        this.add('gamelift:DescribeEC2InstanceLimits');
        return this;
    }
    /**
     * Retrieves general properties, including status, for fleets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetAttributes.html
     */
    describeFleetAttributes() {
        this.add('gamelift:DescribeFleetAttributes');
        return this;
    }
    /**
     * Retrieves the current capacity setting for fleets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetCapacity.html
     */
    describeFleetCapacity() {
        this.add('gamelift:DescribeFleetCapacity');
        return this;
    }
    /**
     * Retrieves entries from a fleet's event log.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetEvents.html
     */
    describeFleetEvents() {
        this.add('gamelift:DescribeFleetEvents');
        return this;
    }
    /**
     * Retrieves the inbound connection permissions for a fleet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetPortSettings.html
     */
    describeFleetPortSettings() {
        this.add('gamelift:DescribeFleetPortSettings');
        return this;
    }
    /**
     * Retrieves utilization statistics for fleets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetUtilization.html
     */
    describeFleetUtilization() {
        this.add('gamelift:DescribeFleetUtilization');
        return this;
    }
    /**
     * Retrieves properties for game sessions in a fleet, including the protection policy.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessionDetails.html
     */
    describeGameSessionDetails() {
        this.add('gamelift:DescribeGameSessionDetails');
        return this;
    }
    /**
     * Retrieves details of a game session placement request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessionPlacement.html
     */
    describeGameSessionPlacement() {
        this.add('gamelift:DescribeGameSessionPlacement');
        return this;
    }
    /**
     * Retrieves properties for game session queues.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessionQueues.html
     */
    describeGameSessionQueues() {
        this.add('gamelift:DescribeGameSessionQueues');
        return this;
    }
    /**
     * Retrieves properties for game sessions in a fleet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessions.html
     */
    describeGameSessions() {
        this.add('gamelift:DescribeGameSessions');
        return this;
    }
    /**
     * Retrieves information about instances in a fleet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeInstances.html
     */
    describeInstances() {
        this.add('gamelift:DescribeInstances');
        return this;
    }
    /**
     * Retrieves details of matchmaking tickets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmaking.html
     */
    describeMatchmaking() {
        this.add('gamelift:DescribeMatchmaking');
        return this;
    }
    /**
     * Retrieves properties for FlexMatch matchmakers.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmakingConfigurations.html
     */
    describeMatchmakingConfigurations() {
        this.add('gamelift:DescribeMatchmakingConfigurations');
        return this;
    }
    /**
     * Retrieves properties for FlexMatch matchmaking rule sets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmakingRuleSets.html
     */
    describeMatchmakingRuleSets() {
        this.add('gamelift:DescribeMatchmakingRuleSets');
        return this;
    }
    /**
     * Retrieves properties for player sessions in a game session.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribePlayerSessions.html
     */
    describePlayerSessions() {
        this.add('gamelift:DescribePlayerSessions');
        return this;
    }
    /**
     * Retrieves the current runtime configuration for a fleet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeRuntimeConfiguration.html
     */
    describeRuntimeConfiguration() {
        this.add('gamelift:DescribeRuntimeConfiguration');
        return this;
    }
    /**
     * Retrieves all scaling policies that are applied to a fleet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeScalingPolicies.html
     */
    describeScalingPolicies() {
        this.add('gamelift:DescribeScalingPolicies');
        return this;
    }
    /**
     * Retrieves properties for a Realtime Servers script.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeScript.html
     */
    describeScript() {
        this.add('gamelift:DescribeScript');
        return this;
    }
    /**
     * Retrieves valid VPC peering authorizations.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeVpcPeeringAuthorizations.html
     */
    describeVpcPeeringAuthorizations() {
        this.add('gamelift:DescribeVpcPeeringAuthorizations');
        return this;
    }
    /**
     * Retrieves details on active or pending VPC peering connections.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeVpcPeeringConnections.html
     */
    describeVpcPeeringConnections() {
        this.add('gamelift:DescribeVpcPeeringConnections');
        return this;
    }
    /**
     * Retrieves the location of stored logs for a game session.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_GetGameSessionLogUrl.html
     */
    getGameSessionLogUrl() {
        this.add('gamelift:GetGameSessionLogUrl');
        return this;
    }
    /**
     * Requests remote access to a specified fleet instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_GetInstanceAccess.html
     */
    getInstanceAccess() {
        this.add('gamelift:GetInstanceAccess');
        return this;
    }
    /**
     * Retrieves all aliases that are defined in the current region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListAliases.html
     */
    listAliases() {
        this.add('gamelift:ListAliases');
        return this;
    }
    /**
     * Retrieves all game build in the current region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListBuilds.html
     */
    listBuilds() {
        this.add('gamelift:ListBuilds');
        return this;
    }
    /**
     * Retrieves a list of fleet IDs for all fleets in the current region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListFleets.html
     */
    listFleets() {
        this.add('gamelift:ListFleets');
        return this;
    }
    /**
     * Retrieves properties for all Realtime Servers scripts in the current region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListScripts.html
     */
    listScripts() {
        this.add('gamelift:ListScripts');
        return this;
    }
    /**
     * List Tags for GameLift Resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('gamelift:ListTagsForResource');
        return this;
    }
    /**
     * Creates or updates a fleet auto-scaling policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_PutScalingPolicy.html
     */
    putScalingPolicy() {
        this.add('gamelift:PutScalingPolicy');
        return this;
    }
    /**
     * Retrieves fresh upload credentials to use when uploading a new game build.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_RequestUploadCredentials.html
     */
    requestUploadCredentials() {
        this.add('gamelift:RequestUploadCredentials');
        return this;
    }
    /**
     * Retrieves the fleet ID associated with an alias.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ResolveAlias.html
     */
    resolveAlias() {
        this.add('gamelift:ResolveAlias');
        return this;
    }
    /**
     * Retrieves game sessions that match a set of search criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_SearchGameSessions.html
     */
    searchGameSessions() {
        this.add('gamelift:SearchGameSessions');
        return this;
    }
    /**
     * Resumes auto-scaling activity on a fleet after it was suspended with StopFleetActions().
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartFleetActions.html
     */
    startFleetActions() {
        this.add('gamelift:StartFleetActions');
        return this;
    }
    /**
     * Sends a game session placement request to a game session queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartGameSessionPlacement.html
     */
    startGameSessionPlacement() {
        this.add('gamelift:StartGameSessionPlacement');
        return this;
    }
    /**
     * Requests FlexMatch matchmaking to fill available player slots in an existing game session.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartMatchBackfill.html
     */
    startMatchBackfill() {
        this.add('gamelift:StartMatchBackfill');
        return this;
    }
    /**
     * Requests FlexMatch matchmaking for one or a group of players and game session placement for a resulting match.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartMatchmaking.html
     */
    startMatchmaking() {
        this.add('gamelift:StartMatchmaking');
        return this;
    }
    /**
     * Suspends auto-scaling activity on a fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html
     */
    stopFleetActions() {
        this.add('gamelift:StopFleetActions');
        return this;
    }
    /**
     * Cancels a game session placement request that is in progress.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopGameSessionPlacement.html
     */
    stopGameSessionPlacement() {
        this.add('gamelift:StopGameSessionPlacement');
        return this;
    }
    /**
     * Cancels a matchmaking or match backfill request that is in progress.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopMatchmaking.html
     */
    stopMatchmaking() {
        this.add('gamelift:StopMatchmaking');
        return this;
    }
    /**
     * Tags GameLift Resources
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_TagResource.html
     */
    tagResource() {
        this.add('gamelift:TagResource');
        return this;
    }
    /**
     * Untagging GameLift Resources
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UntagResource.html
     */
    untagResource() {
        this.add('gamelift:UntagResource');
        return this;
    }
    /**
     * Updates the properties of an existing alias.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateAlias.html
     */
    updateAlias() {
        this.add('gamelift:UpdateAlias');
        return this;
    }
    /**
     * Updates an existing build's metadata.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateBuild.html
     */
    updateBuild() {
        this.add('gamelift:UpdateBuild');
        return this;
    }
    /**
     * Updates the general properties of an existing fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateFleetAttributes.html
     */
    updateFleetAttributes() {
        this.add('gamelift:UpdateFleetAttributes');
        return this;
    }
    /**
     * Adjusts a fleet's capacity settings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateFleetCapacity.html
     */
    updateFleetCapacity() {
        this.add('gamelift:UpdateFleetCapacity');
        return this;
    }
    /**
     * Adjusts a fleet's port settings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateFleetPortSettings.html
     */
    updateFleetPortSettings() {
        this.add('gamelift:UpdateFleetPortSettings');
        return this;
    }
    /**
     * Updates the properties of an existing game session.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateGameSession.html
     */
    updateGameSession() {
        this.add('gamelift:UpdateGameSession');
        return this;
    }
    /**
     * Updates properties of an existing game session queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateGameSessionQueue.html
     */
    updateGameSessionQueue() {
        this.add('gamelift:UpdateGameSessionQueue');
        return this;
    }
    /**
     * Updates properties of an existing FlexMatch matchmaking configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateMatchmakingConfiguration.html
     */
    updateMatchmakingConfiguration() {
        this.add('gamelift:UpdateMatchmakingConfiguration');
        return this;
    }
    /**
     * Updates how server processes are configured on instances in an existing fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateRuntimeConfiguration.html
     */
    updateRuntimeConfiguration() {
        this.add('gamelift:UpdateRuntimeConfiguration');
        return this;
    }
    /**
     * Updates the metadata and content of an existing Realtime Servers script.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateScript.html
     */
    updateScript() {
        this.add('gamelift:UpdateScript');
        return this;
    }
    /**
     * Validates the syntax of a FlexMatch matchmaking rule set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ValidateMatchmakingRuleSet.html
     */
    validateMatchmakingRuleSet() {
        this.add('gamelift:ValidateMatchmakingRuleSet');
        return this;
    }
    /**
     * Adds a resource of type alias to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html
     *
     * @param aliasId - Identifier for the aliasId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onAlias(aliasId, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}::alias/${AliasId}';
        arn = arn.replace('${AliasId}', aliasId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type build to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html
     *
     * @param buildId - Identifier for the buildId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onBuild(buildId, accountId, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${AccountId}:build/${BuildId}';
        arn = arn.replace('${BuildId}', buildId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type script to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html
     *
     * @param scriptId - Identifier for the scriptId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onScript(scriptId, accountId, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${AccountId}:script/${ScriptId}';
        arn = arn.replace('${ScriptId}', scriptId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html
     *
     * @param fleetId - Identifier for the fleetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onFleet(fleetId, account, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${Account}:fleet/${FleetId}';
        arn = arn.replace('${FleetId}', fleetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type gameSessionQueue to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html
     *
     * @param gameSessionQueueName - Identifier for the gameSessionQueueName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onGameSessionQueue(gameSessionQueueName, account, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${Account}:gamesessionqueue/${GameSessionQueueName}';
        arn = arn.replace('${GameSessionQueueName}', gameSessionQueueName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type matchmakingConfiguration to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html
     *
     * @param matchmakingConfigurationName - Identifier for the matchmakingConfigurationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onMatchmakingConfiguration(matchmakingConfigurationName, account, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${Account}:matchmakingconfiguration/${MatchmakingConfigurationName}';
        arn = arn.replace('${MatchmakingConfigurationName}', matchmakingConfigurationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type matchmakingRuleSet to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/resources-defined.html
     *
     * @param matchmakingRuleSetName - Identifier for the matchmakingRuleSetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onMatchmakingRuleSet(matchmakingRuleSetName, account, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${Account}:matchmakingruleset/${MatchmakingRuleSetName}';
        arn = arn.replace('${MatchmakingRuleSetName}', matchmakingRuleSetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Gamelift = Gamelift;
//# sourceMappingURL=data:application/json;base64,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