"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Glacier = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [glacier](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonglacier.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Glacier extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [glacier](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonglacier.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'glacier';
        this.actionList = {
            "AbortMultipartUpload": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-abort-upload.html",
                "description": "Aborts a multipart upload identified by the upload ID",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "AbortVaultLock": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-AbortVaultLock.html",
                "description": "Aborts the vault locking process if the vault lock is not in the Locked state",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "AddTagsToVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-AddTagsToVault.html",
                "description": "Adds the specified tags to a vault",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "CompleteMultipartUpload": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-complete-upload.html",
                "description": "Completes a multipart upload process",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "CompleteVaultLock": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-CompleteVaultLock.html",
                "description": "Completes the vault locking process",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "CreateVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-put.html",
                "description": "Creates a new vault with the specified name",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "DeleteArchive": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-delete.html",
                "description": "Deletes an archive from a vault",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                },
                "conditions": [
                    "glacier:ArchiveAgeInDays"
                ]
            },
            "DeleteVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-delete.html",
                "description": "Deletes a vault",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "DeleteVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-DeleteVaultAccessPolicy.html",
                "description": "Deletes the access policy associated with the specified vault",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "DeleteVaultNotifications": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-delete.html",
                "description": "Deletes the notification configuration set for a vault",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "DescribeJob": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-describe-job-get.html",
                "description": "Returns information about a job you previously initiated",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "DescribeVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-get..html",
                "description": "Returns information about a vault",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "GetDataRetrievalPolicy": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetDataRetrievalPolicy.html",
                "description": "Returns the current data retrieval policy for the account and region specified in the GET request",
                "accessLevel": "Read"
            },
            "GetJobOutput": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-job-output-get.html",
                "description": "Downloads the output of the job you initiated",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "GetVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetVaultAccessPolicy.html",
                "description": "Retrieves the access-policy subresource set on the vault",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "GetVaultLock": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetVaultLock.html",
                "description": "Retrieves attributes from the lock-policy subresource set on the specified vault",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "GetVaultNotifications": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-get.html",
                "description": "Retrieves the notification-configuration subresource set on the vault",
                "accessLevel": "Read",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "InitiateJob": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-initiate-job-post.html",
                "description": "Initiates a job of the specified type",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                },
                "conditions": [
                    "glacier:ArchiveAgeInDays"
                ]
            },
            "InitiateMultipartUpload": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-initiate-upload.html",
                "description": "Initiates a multipart upload",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "InitiateVaultLock": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-InitiateVaultLock.html",
                "description": "Initiates the vault locking process",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-jobs-get.html",
                "description": "Lists jobs for a vault that are in-progress and jobs that have recently finished",
                "accessLevel": "List",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "ListMultipartUploads": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-list-uploads.html",
                "description": "Lists in-progress multipart uploads for the specified vault",
                "accessLevel": "List",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "ListParts": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-list-parts.html",
                "description": "Lists the parts of an archive that have been uploaded in a specific multipart upload",
                "accessLevel": "List",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "ListProvisionedCapacity": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-ListProvisionedCapacity.html",
                "description": "This operation lists the provisioned capacity for the specified AWS account.",
                "accessLevel": "List"
            },
            "ListTagsForVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-ListTagsForVault.html",
                "description": "Lists all the tags attached to a vault",
                "accessLevel": "List",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "ListVaults": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vaults-get.html",
                "description": "Lists all vaults",
                "accessLevel": "List"
            },
            "PurchaseProvisionedCapacity": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-PurchaseProvisionedCapacity.html",
                "description": "This operation purchases a provisioned capacity unit for an AWS account.",
                "accessLevel": "Write"
            },
            "RemoveTagsFromVault": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-RemoveTagsFromVault.html",
                "description": "Removes one or more tags from the set of tags attached to a vault",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "SetDataRetrievalPolicy": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-SetDataRetrievalPolicy.html",
                "description": "Sets and then enacts a data retrieval policy in the region specified in the PUT request",
                "accessLevel": "Permissions management"
            },
            "SetVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-SetVaultAccessPolicy.html",
                "description": "Configures an access policy for a vault and will overwrite an existing policy",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "SetVaultNotifications": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-put.html",
                "description": "Configures vault notifications",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "UploadArchive": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-post.html",
                "description": "Adds an archive to a vault",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            },
            "UploadMultipartPart": {
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/api-upload-part.html",
                "description": "Uploads a part of an archive",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vault": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "vault": {
                "name": "vault",
                "url": "https://docs.aws.amazon.com/amazonglacier/latest/dev/working-with-vaults.html",
                "arn": "arn:${Partition}:glacier:${Region}:${Account}:vaults/${VaultName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Aborts a multipart upload identified by the upload ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-abort-upload.html
     */
    abortMultipartUpload() {
        this.add('glacier:AbortMultipartUpload');
        return this;
    }
    /**
     * Aborts the vault locking process if the vault lock is not in the Locked state
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-AbortVaultLock.html
     */
    abortVaultLock() {
        this.add('glacier:AbortVaultLock');
        return this;
    }
    /**
     * Adds the specified tags to a vault
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-AddTagsToVault.html
     */
    addTagsToVault() {
        this.add('glacier:AddTagsToVault');
        return this;
    }
    /**
     * Completes a multipart upload process
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-complete-upload.html
     */
    completeMultipartUpload() {
        this.add('glacier:CompleteMultipartUpload');
        return this;
    }
    /**
     * Completes the vault locking process
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-CompleteVaultLock.html
     */
    completeVaultLock() {
        this.add('glacier:CompleteVaultLock');
        return this;
    }
    /**
     * Creates a new vault with the specified name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-put.html
     */
    createVault() {
        this.add('glacier:CreateVault');
        return this;
    }
    /**
     * Deletes an archive from a vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-delete.html
     */
    deleteArchive() {
        this.add('glacier:DeleteArchive');
        return this;
    }
    /**
     * Deletes a vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-delete.html
     */
    deleteVault() {
        this.add('glacier:DeleteVault');
        return this;
    }
    /**
     * Deletes the access policy associated with the specified vault
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-DeleteVaultAccessPolicy.html
     */
    deleteVaultAccessPolicy() {
        this.add('glacier:DeleteVaultAccessPolicy');
        return this;
    }
    /**
     * Deletes the notification configuration set for a vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-delete.html
     */
    deleteVaultNotifications() {
        this.add('glacier:DeleteVaultNotifications');
        return this;
    }
    /**
     * Returns information about a job you previously initiated
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-describe-job-get.html
     */
    describeJob() {
        this.add('glacier:DescribeJob');
        return this;
    }
    /**
     * Returns information about a vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-get..html
     */
    describeVault() {
        this.add('glacier:DescribeVault');
        return this;
    }
    /**
     * Returns the current data retrieval policy for the account and region specified in the GET request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetDataRetrievalPolicy.html
     */
    getDataRetrievalPolicy() {
        this.add('glacier:GetDataRetrievalPolicy');
        return this;
    }
    /**
     * Downloads the output of the job you initiated
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-job-output-get.html
     */
    getJobOutput() {
        this.add('glacier:GetJobOutput');
        return this;
    }
    /**
     * Retrieves the access-policy subresource set on the vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetVaultAccessPolicy.html
     */
    getVaultAccessPolicy() {
        this.add('glacier:GetVaultAccessPolicy');
        return this;
    }
    /**
     * Retrieves attributes from the lock-policy subresource set on the specified vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-GetVaultLock.html
     */
    getVaultLock() {
        this.add('glacier:GetVaultLock');
        return this;
    }
    /**
     * Retrieves the notification-configuration subresource set on the vault
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-get.html
     */
    getVaultNotifications() {
        this.add('glacier:GetVaultNotifications');
        return this;
    }
    /**
     * Initiates a job of the specified type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-initiate-job-post.html
     */
    initiateJob() {
        this.add('glacier:InitiateJob');
        return this;
    }
    /**
     * Initiates a multipart upload
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-initiate-upload.html
     */
    initiateMultipartUpload() {
        this.add('glacier:InitiateMultipartUpload');
        return this;
    }
    /**
     * Initiates the vault locking process
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-InitiateVaultLock.html
     */
    initiateVaultLock() {
        this.add('glacier:InitiateVaultLock');
        return this;
    }
    /**
     * Lists jobs for a vault that are in-progress and jobs that have recently finished
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-jobs-get.html
     */
    listJobs() {
        this.add('glacier:ListJobs');
        return this;
    }
    /**
     * Lists in-progress multipart uploads for the specified vault
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-list-uploads.html
     */
    listMultipartUploads() {
        this.add('glacier:ListMultipartUploads');
        return this;
    }
    /**
     * Lists the parts of an archive that have been uploaded in a specific multipart upload
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-multipart-list-parts.html
     */
    listParts() {
        this.add('glacier:ListParts');
        return this;
    }
    /**
     * This operation lists the provisioned capacity for the specified AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-ListProvisionedCapacity.html
     */
    listProvisionedCapacity() {
        this.add('glacier:ListProvisionedCapacity');
        return this;
    }
    /**
     * Lists all the tags attached to a vault
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-ListTagsForVault.html
     */
    listTagsForVault() {
        this.add('glacier:ListTagsForVault');
        return this;
    }
    /**
     * Lists all vaults
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vaults-get.html
     */
    listVaults() {
        this.add('glacier:ListVaults');
        return this;
    }
    /**
     * This operation purchases a provisioned capacity unit for an AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-PurchaseProvisionedCapacity.html
     */
    purchaseProvisionedCapacity() {
        this.add('glacier:PurchaseProvisionedCapacity');
        return this;
    }
    /**
     * Removes one or more tags from the set of tags attached to a vault
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-RemoveTagsFromVault.html
     */
    removeTagsFromVault() {
        this.add('glacier:RemoveTagsFromVault');
        return this;
    }
    /**
     * Sets and then enacts a data retrieval policy in the region specified in the PUT request
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-SetDataRetrievalPolicy.html
     */
    setDataRetrievalPolicy() {
        this.add('glacier:SetDataRetrievalPolicy');
        return this;
    }
    /**
     * Configures an access policy for a vault and will overwrite an existing policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-SetVaultAccessPolicy.html
     */
    setVaultAccessPolicy() {
        this.add('glacier:SetVaultAccessPolicy');
        return this;
    }
    /**
     * Configures vault notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-vault-notifications-put.html
     */
    setVaultNotifications() {
        this.add('glacier:SetVaultNotifications');
        return this;
    }
    /**
     * Adds an archive to a vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-archive-post.html
     */
    uploadArchive() {
        this.add('glacier:UploadArchive');
        return this;
    }
    /**
     * Uploads a part of an archive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/api-upload-part.html
     */
    uploadMultipartPart() {
        this.add('glacier:UploadMultipartPart');
        return this;
    }
    /**
     * Adds a resource of type vault to the statement
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/working-with-vaults.html
     *
     * @param vaultName - Identifier for the vaultName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVault(vaultName, account, region, partition) {
        var arn = 'arn:${Partition}:glacier:${Region}:${Account}:vaults/${VaultName}';
        arn = arn.replace('${VaultName}', vaultName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * How long an archive has been stored in the vault, in days.
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/access-control-overview.html#specifying-conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifArchiveAgeInDays(value, operator) {
        return this.if(`glacier:ArchiveAgeInDays`, value, operator || 'StringLike');
    }
    /**
     * A customer-defined tag.
     *
     * https://docs.aws.amazon.com/amazonglacier/latest/dev/access-control-overview.html#specifying-conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTagExists(value, operator) {
        return this.if(`glacier:ResourceTag/`, value, operator || 'StringLike');
    }
}
exports.Glacier = Glacier;
//# sourceMappingURL=data:application/json;base64,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