"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kinesis = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [kinesis](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesis.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kinesis extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [kinesis](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesis.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'kinesis';
        this.actionList = {
            "AddTagsToStream": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_AddTagsToStream.html",
                "description": "Adds or updates tags for the specified Amazon Kinesis stream. Each stream can have up to 10 tags.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "CreateStream": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_CreateStream.html",
                "description": "Creates a Amazon Kinesis stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DecreaseStreamRetentionPeriod": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DecreaseStreamRetentionPeriod.html",
                "description": "Decreases the stream's retention period, which is the length of time data records are accessible after they are added to the stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DeleteStream": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DeleteStream.html",
                "description": "Deletes a stream and all its shards and data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DeregisterStreamConsumer": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DeregisterStreamConsumer.html",
                "description": "Deregisters a stream consumer with a Kinesis data stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "consumer": {
                        "required": true
                    },
                    "stream": {
                        "required": true
                    }
                }
            },
            "DescribeLimits": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeLimits.html",
                "description": "Describes the shard limits and usage for the account.",
                "accessLevel": "Read"
            },
            "DescribeStream": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStream.html",
                "description": "Describes the specified stream.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DescribeStreamConsumer": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStreamConsumer.html",
                "description": "Gets the description of a registered stream consumer.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "consumer": {
                        "required": true
                    },
                    "stream": {
                        "required": true
                    }
                }
            },
            "DescribeStreamSummary": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStreamSummary.html",
                "description": "Provides a summarized description of the specified Kinesis data stream without the shard list.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DisableEnhancedMonitoring": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DisableEnhancedMonitoring.html",
                "description": "Disables enhanced monitoring.",
                "accessLevel": "Write"
            },
            "EnableEnhancedMonitoring": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_EnableEnhancedMonitoring.html",
                "description": "API_EnableEnhancedMonitoring.html",
                "accessLevel": "Write"
            },
            "GetRecords": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetRecords.html",
                "description": "Gets data records from a shard.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "GetShardIterator": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetShardIterator.html",
                "description": "Gets a shard iterator. A shard iterator expires five minutes after it is returned to the requester.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "IncreaseStreamRetentionPeriod": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_IncreaseStreamRetentionPeriod.html",
                "description": "Increases the stream's retention period, which is the length of time data records are accessible after they are added to the stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "ListShards": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListShards.html",
                "description": "Lists the shards in a stream and provides information about each shard.",
                "accessLevel": "List"
            },
            "ListStreamConsumers": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListStreamConsumers.html",
                "description": "Lists the stream consumers registered to receive data from a Kinesis stream using enhanced fan-out, and provides information about each consumer.",
                "accessLevel": "List"
            },
            "ListStreams": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListStreams.html",
                "description": "Lists your streams.",
                "accessLevel": "List"
            },
            "ListTagsForStream": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListTagsForStream.html",
                "description": "Lists the tags for the specified Amazon Kinesis stream.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "MergeShards": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_MergeShards.html",
                "description": "Merges two adjacent shards in a stream and combines them into a single shard to reduce the stream's capacity to ingest and transport data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "PutRecord": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_PutRecord.html",
                "description": "Writes a single data record from a producer into an Amazon Kinesis stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "PutRecords": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_PutRecords.html",
                "description": "Writes multiple data records from a producer into an Amazon Kinesis stream in a single call (also referred to as a PutRecords request).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "RegisterStreamConsumer": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_RegisterStreamConsumer.html",
                "description": "Registers a stream consumer with a Kinesis data stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "consumer": {
                        "required": true
                    },
                    "stream": {
                        "required": true
                    }
                }
            },
            "RemoveTagsFromStream": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_RemoveTagsFromStream.html",
                "description": "Description for SplitShard",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "SplitShard": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_SplitShard.html",
                "description": "Description for SplitShard",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "SubscribeToShard": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_SubscribeToShard.html",
                "description": "Listening to a specific shard with enhanced fan-out.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "consumer": {
                        "required": true
                    },
                    "stream": {
                        "required": true
                    }
                }
            },
            "UpdateShardCount": {
                "url": "https://docs.aws.amazon.com/kinesis/latest/APIReference/API_UpdateShardCount.html",
                "description": "Updates the shard count of the specified stream to the specified number of shards.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "stream": {
                "name": "stream",
                "url": "",
                "arn": "arn:${Partition}:kinesis:${Region}:${Account}:stream/${StreamName}",
                "conditionKeys": []
            },
            "consumer": {
                "name": "consumer",
                "url": "",
                "arn": "arn:${Partition}:kinesis:${Region}:${Account}:${StreamType}/${StreamName}/consumer/${ConsumerName}:${ConsumerCreationTimpstamp}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Adds or updates tags for the specified Amazon Kinesis stream. Each stream can have up to 10 tags.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_AddTagsToStream.html
     */
    addTagsToStream() {
        this.add('kinesis:AddTagsToStream');
        return this;
    }
    /**
     * Creates a Amazon Kinesis stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_CreateStream.html
     */
    createStream() {
        this.add('kinesis:CreateStream');
        return this;
    }
    /**
     * Decreases the stream's retention period, which is the length of time data records are accessible after they are added to the stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DecreaseStreamRetentionPeriod.html
     */
    decreaseStreamRetentionPeriod() {
        this.add('kinesis:DecreaseStreamRetentionPeriod');
        return this;
    }
    /**
     * Deletes a stream and all its shards and data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DeleteStream.html
     */
    deleteStream() {
        this.add('kinesis:DeleteStream');
        return this;
    }
    /**
     * Deregisters a stream consumer with a Kinesis data stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DeregisterStreamConsumer.html
     */
    deregisterStreamConsumer() {
        this.add('kinesis:DeregisterStreamConsumer');
        return this;
    }
    /**
     * Describes the shard limits and usage for the account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeLimits.html
     */
    describeLimits() {
        this.add('kinesis:DescribeLimits');
        return this;
    }
    /**
     * Describes the specified stream.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStream.html
     */
    describeStream() {
        this.add('kinesis:DescribeStream');
        return this;
    }
    /**
     * Gets the description of a registered stream consumer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStreamConsumer.html
     */
    describeStreamConsumer() {
        this.add('kinesis:DescribeStreamConsumer');
        return this;
    }
    /**
     * Provides a summarized description of the specified Kinesis data stream without the shard list.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStreamSummary.html
     */
    describeStreamSummary() {
        this.add('kinesis:DescribeStreamSummary');
        return this;
    }
    /**
     * Disables enhanced monitoring.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DisableEnhancedMonitoring.html
     */
    disableEnhancedMonitoring() {
        this.add('kinesis:DisableEnhancedMonitoring');
        return this;
    }
    /**
     * API_EnableEnhancedMonitoring.html
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_EnableEnhancedMonitoring.html
     */
    enableEnhancedMonitoring() {
        this.add('kinesis:EnableEnhancedMonitoring');
        return this;
    }
    /**
     * Gets data records from a shard.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetRecords.html
     */
    getRecords() {
        this.add('kinesis:GetRecords');
        return this;
    }
    /**
     * Gets a shard iterator. A shard iterator expires five minutes after it is returned to the requester.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetShardIterator.html
     */
    getShardIterator() {
        this.add('kinesis:GetShardIterator');
        return this;
    }
    /**
     * Increases the stream's retention period, which is the length of time data records are accessible after they are added to the stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_IncreaseStreamRetentionPeriod.html
     */
    increaseStreamRetentionPeriod() {
        this.add('kinesis:IncreaseStreamRetentionPeriod');
        return this;
    }
    /**
     * Lists the shards in a stream and provides information about each shard.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListShards.html
     */
    listShards() {
        this.add('kinesis:ListShards');
        return this;
    }
    /**
     * Lists the stream consumers registered to receive data from a Kinesis stream using enhanced fan-out, and provides information about each consumer.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListStreamConsumers.html
     */
    listStreamConsumers() {
        this.add('kinesis:ListStreamConsumers');
        return this;
    }
    /**
     * Lists your streams.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListStreams.html
     */
    listStreams() {
        this.add('kinesis:ListStreams');
        return this;
    }
    /**
     * Lists the tags for the specified Amazon Kinesis stream.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListTagsForStream.html
     */
    listTagsForStream() {
        this.add('kinesis:ListTagsForStream');
        return this;
    }
    /**
     * Merges two adjacent shards in a stream and combines them into a single shard to reduce the stream's capacity to ingest and transport data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_MergeShards.html
     */
    mergeShards() {
        this.add('kinesis:MergeShards');
        return this;
    }
    /**
     * Writes a single data record from a producer into an Amazon Kinesis stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_PutRecord.html
     */
    putRecord() {
        this.add('kinesis:PutRecord');
        return this;
    }
    /**
     * Writes multiple data records from a producer into an Amazon Kinesis stream in a single call (also referred to as a PutRecords request).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_PutRecords.html
     */
    putRecords() {
        this.add('kinesis:PutRecords');
        return this;
    }
    /**
     * Registers a stream consumer with a Kinesis data stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_RegisterStreamConsumer.html
     */
    registerStreamConsumer() {
        this.add('kinesis:RegisterStreamConsumer');
        return this;
    }
    /**
     * Description for SplitShard
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_RemoveTagsFromStream.html
     */
    removeTagsFromStream() {
        this.add('kinesis:RemoveTagsFromStream');
        return this;
    }
    /**
     * Description for SplitShard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_SplitShard.html
     */
    splitShard() {
        this.add('kinesis:SplitShard');
        return this;
    }
    /**
     * Listening to a specific shard with enhanced fan-out.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_SubscribeToShard.html
     */
    subscribeToShard() {
        this.add('kinesis:SubscribeToShard');
        return this;
    }
    /**
     * Updates the shard count of the specified stream to the specified number of shards.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_UpdateShardCount.html
     */
    updateShardCount() {
        this.add('kinesis:UpdateShardCount');
        return this;
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * @param streamName - Identifier for the streamName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onStream(streamName, account, region, partition) {
        var arn = 'arn:${Partition}:kinesis:${Region}:${Account}:stream/${StreamName}';
        arn = arn.replace('${StreamName}', streamName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type consumer to the statement
     *
     * @param streamType - Identifier for the streamType.
     * @param streamName - Identifier for the streamName.
     * @param consumerName - Identifier for the consumerName.
     * @param consumerCreationTimpstamp - Identifier for the consumerCreationTimpstamp.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConsumer(streamType, streamName, consumerName, consumerCreationTimpstamp, account, region, partition) {
        var arn = 'arn:${Partition}:kinesis:${Region}:${Account}:${StreamType}/${StreamName}/consumer/${ConsumerName}:${ConsumerCreationTimpstamp}';
        arn = arn.replace('${StreamType}', streamType);
        arn = arn.replace('${StreamName}', streamName);
        arn = arn.replace('${ConsumerName}', consumerName);
        arn = arn.replace('${ConsumerCreationTimpstamp}', consumerCreationTimpstamp);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Kinesis = Kinesis;
//# sourceMappingURL=data:application/json;base64,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