"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisanalyticsV2 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [kinesisanalytics-v2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesisanalyticsv2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class KinesisanalyticsV2 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [kinesisanalytics-v2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesisanalyticsv2.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'kinesisanalytics-v2';
        this.actionList = {
            "AddApplicationCloudWatchLoggingOption": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationCloudWatchLoggingOption.html",
                "description": "Adds cloudwatch logging option to the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "AddApplicationInput": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationInput.html",
                "description": "Adds input to the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "AddApplicationInputProcessingConfiguration": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationInputProcessingConfiguration.html",
                "description": "Adds input processing configuration to the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "AddApplicationOutput": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationOutput.html",
                "description": "Adds output to the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "AddApplicationReferenceDataSource": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationReferenceDataSource.html",
                "description": "Adds reference data source to the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "AddApplicationVpcConfiguration": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationVpcConfiguration.html",
                "description": "Adds VPC configuration to the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "CreateApplication": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_CreateApplication.html",
                "description": "Creates an application.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateApplicationSnapshot": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_CreateApplicationSnapshot.html",
                "description": "Creates a snapshot for an application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteApplication": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplication.html",
                "description": "Deletes the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteApplicationCloudWatchLoggingOption": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationCloudWatchLoggingOption.html",
                "description": "Deletes the specified cloudwatch logging option of the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteApplicationInputProcessingConfiguration": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationInputProcessingConfiguration.html",
                "description": "Deletes the specified input processing configuration of the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteApplicationOutput": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationOutput.html",
                "description": "Deletes the specified output of the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteApplicationReferenceDataSource": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationReferenceDataSource.html",
                "description": "Deletes the specified reference data source of the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteApplicationSnapshot": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationSnapshot.html",
                "description": "Deletes a snapshot for an application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteApplicationVpcConfiguration": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationVpcConfiguration.html",
                "description": "Deletes the specified VPC configuration of the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DescribeApplication": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DescribeApplication.html",
                "description": "Describes the specified application.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DescribeApplicationSnapshot": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DescribeApplicationSnapshot.html",
                "description": "Describes an application snapshot.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DiscoverInputSchema": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DiscoverInputSchema.html",
                "description": "Discovers the input schema for the application.",
                "accessLevel": "Read"
            },
            "ListApplicationSnapshots": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_ListApplicationSnapshots.html",
                "description": "Lists the snapshots for an application.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "ListApplications": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_ListApplications.html",
                "description": "List applications for the account",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_ListTagsForResource.html",
                "description": "Fetch the tags associated with the application.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "StartApplication": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_StartsApplication.html",
                "description": "Starts the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "StopApplication": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_StopApplication.html",
                "description": "Stops the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_TagResource.html",
                "description": "Add tags to the application.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_UntagResource.html",
                "description": "Remove the specified tags from the application.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateApplication": {
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_UpdateApplication.html",
                "description": "Updates the application.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "application": {
                "name": "application",
                "url": "https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/how-it-works.html",
                "arn": "arn:${Partition}:kinesisanalytics:${Region}:${Account}:application/${ApplicationName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Adds cloudwatch logging option to the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationCloudWatchLoggingOption.html
     */
    addApplicationCloudWatchLoggingOption() {
        this.add('kinesisanalytics-v2:AddApplicationCloudWatchLoggingOption');
        return this;
    }
    /**
     * Adds input to the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationInput.html
     */
    addApplicationInput() {
        this.add('kinesisanalytics-v2:AddApplicationInput');
        return this;
    }
    /**
     * Adds input processing configuration to the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationInputProcessingConfiguration.html
     */
    addApplicationInputProcessingConfiguration() {
        this.add('kinesisanalytics-v2:AddApplicationInputProcessingConfiguration');
        return this;
    }
    /**
     * Adds output to the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationOutput.html
     */
    addApplicationOutput() {
        this.add('kinesisanalytics-v2:AddApplicationOutput');
        return this;
    }
    /**
     * Adds reference data source to the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationReferenceDataSource.html
     */
    addApplicationReferenceDataSource() {
        this.add('kinesisanalytics-v2:AddApplicationReferenceDataSource');
        return this;
    }
    /**
     * Adds VPC configuration to the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_AddApplicationVpcConfiguration.html
     */
    addApplicationVpcConfiguration() {
        this.add('kinesisanalytics-v2:AddApplicationVpcConfiguration');
        return this;
    }
    /**
     * Creates an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_CreateApplication.html
     */
    createApplication() {
        this.add('kinesisanalytics-v2:CreateApplication');
        return this;
    }
    /**
     * Creates a snapshot for an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_CreateApplicationSnapshot.html
     */
    createApplicationSnapshot() {
        this.add('kinesisanalytics-v2:CreateApplicationSnapshot');
        return this;
    }
    /**
     * Deletes the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplication.html
     */
    deleteApplication() {
        this.add('kinesisanalytics-v2:DeleteApplication');
        return this;
    }
    /**
     * Deletes the specified cloudwatch logging option of the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationCloudWatchLoggingOption.html
     */
    deleteApplicationCloudWatchLoggingOption() {
        this.add('kinesisanalytics-v2:DeleteApplicationCloudWatchLoggingOption');
        return this;
    }
    /**
     * Deletes the specified input processing configuration of the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationInputProcessingConfiguration.html
     */
    deleteApplicationInputProcessingConfiguration() {
        this.add('kinesisanalytics-v2:DeleteApplicationInputProcessingConfiguration');
        return this;
    }
    /**
     * Deletes the specified output of the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationOutput.html
     */
    deleteApplicationOutput() {
        this.add('kinesisanalytics-v2:DeleteApplicationOutput');
        return this;
    }
    /**
     * Deletes the specified reference data source of the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationReferenceDataSource.html
     */
    deleteApplicationReferenceDataSource() {
        this.add('kinesisanalytics-v2:DeleteApplicationReferenceDataSource');
        return this;
    }
    /**
     * Deletes a snapshot for an application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationSnapshot.html
     */
    deleteApplicationSnapshot() {
        this.add('kinesisanalytics-v2:DeleteApplicationSnapshot');
        return this;
    }
    /**
     * Deletes the specified VPC configuration of the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DeleteApplicationVpcConfiguration.html
     */
    deleteApplicationVpcConfiguration() {
        this.add('kinesisanalytics-v2:DeleteApplicationVpcConfiguration');
        return this;
    }
    /**
     * Describes the specified application.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DescribeApplication.html
     */
    describeApplication() {
        this.add('kinesisanalytics-v2:DescribeApplication');
        return this;
    }
    /**
     * Describes an application snapshot.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DescribeApplicationSnapshot.html
     */
    describeApplicationSnapshot() {
        this.add('kinesisanalytics-v2:DescribeApplicationSnapshot');
        return this;
    }
    /**
     * Discovers the input schema for the application.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_DiscoverInputSchema.html
     */
    discoverInputSchema() {
        this.add('kinesisanalytics-v2:DiscoverInputSchema');
        return this;
    }
    /**
     * Lists the snapshots for an application.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_ListApplicationSnapshots.html
     */
    listApplicationSnapshots() {
        this.add('kinesisanalytics-v2:ListApplicationSnapshots');
        return this;
    }
    /**
     * List applications for the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_ListApplications.html
     */
    listApplications() {
        this.add('kinesisanalytics-v2:ListApplications');
        return this;
    }
    /**
     * Fetch the tags associated with the application.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('kinesisanalytics-v2:ListTagsForResource');
        return this;
    }
    /**
     * Starts the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_StartsApplication.html
     */
    startApplication() {
        this.add('kinesisanalytics-v2:StartApplication');
        return this;
    }
    /**
     * Stops the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_StopApplication.html
     */
    stopApplication() {
        this.add('kinesisanalytics-v2:StopApplication');
        return this;
    }
    /**
     * Add tags to the application.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_TagResource.html
     */
    tagResource() {
        this.add('kinesisanalytics-v2:TagResource');
        return this;
    }
    /**
     * Remove the specified tags from the application.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_UntagResource.html
     */
    untagResource() {
        this.add('kinesisanalytics-v2:UntagResource');
        return this;
    }
    /**
     * Updates the application.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/API_UpdateApplication.html
     */
    updateApplication() {
        this.add('kinesisanalytics-v2:UpdateApplication');
        return this;
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/kinesisanalytics/latest/apiv2/how-it-works.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onApplication(applicationName, account, region, partition) {
        var arn = 'arn:${Partition}:kinesisanalytics:${Region}:${Account}:application/${ApplicationName}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.KinesisanalyticsV2 = KinesisanalyticsV2;
//# sourceMappingURL=data:application/json;base64,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