"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Firehose = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [firehose](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesisfirehose.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Firehose extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [firehose](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesisfirehose.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'firehose';
        this.actionList = {
            "CreateDeliveryStream": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_CreateDeliveryStream.html",
                "description": "Creates a delivery stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteDeliveryStream": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_DeleteDeliveryStream.html",
                "description": "Deletes a delivery stream and its data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                }
            },
            "DescribeDeliveryStream": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_DescribeDeliveryStream.html",
                "description": "Describes the specified delivery stream and gets the status.",
                "accessLevel": "List",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                }
            },
            "ListDeliveryStreams": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_ListDeliveryStreams.html",
                "description": "Lists your delivery streams.",
                "accessLevel": "List"
            },
            "ListTagsForDeliveryStream": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_ListTagsForDeliveryStream.html",
                "description": "Lists the tags for the specified delivery stream.",
                "accessLevel": "List",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                }
            },
            "PutRecord": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_PutRecord.html",
                "description": "Writes a single data record into an Amazon Kinesis Firehose delivery stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                }
            },
            "PutRecordBatch": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_PutRecordBatch.html",
                "description": "Writes multiple data records into a delivery stream in a single call, which can achieve higher throughput per producer than when writing single records.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                }
            },
            "StartDeliveryStreamEncryption": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_StartDeliveryStreamEncryption.html",
                "description": "Enables server-side encryption (SSE) for the delivery stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                }
            },
            "StopDeliveryStreamEncryption": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_StopDeliveryStreamEncryption.html",
                "description": "Disables the specified destination of the specified delivery stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                }
            },
            "TagDeliveryStream": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_TagDeliveryStream.html",
                "description": "Adds or updates tags for the specified delivery stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagDeliveryStream": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_UntagDeliveryStream.html",
                "description": "Removes tags from the specified delivery stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDestination": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_UpdateDestination.html",
                "description": "Updates the specified destination of the specified delivery stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "deliverystream": {
                "name": "deliverystream",
                "url": "https://docs.aws.amazon.com/firehose/latest/dev/basic-create.html",
                "arn": "arn:${Partition}:firehose:${Region}:${Account}:deliverystream/${DeliveryStreamName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a delivery stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_CreateDeliveryStream.html
     */
    createDeliveryStream() {
        this.add('firehose:CreateDeliveryStream');
        return this;
    }
    /**
     * Deletes a delivery stream and its data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_DeleteDeliveryStream.html
     */
    deleteDeliveryStream() {
        this.add('firehose:DeleteDeliveryStream');
        return this;
    }
    /**
     * Describes the specified delivery stream and gets the status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_DescribeDeliveryStream.html
     */
    describeDeliveryStream() {
        this.add('firehose:DescribeDeliveryStream');
        return this;
    }
    /**
     * Lists your delivery streams.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_ListDeliveryStreams.html
     */
    listDeliveryStreams() {
        this.add('firehose:ListDeliveryStreams');
        return this;
    }
    /**
     * Lists the tags for the specified delivery stream.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_ListTagsForDeliveryStream.html
     */
    listTagsForDeliveryStream() {
        this.add('firehose:ListTagsForDeliveryStream');
        return this;
    }
    /**
     * Writes a single data record into an Amazon Kinesis Firehose delivery stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_PutRecord.html
     */
    putRecord() {
        this.add('firehose:PutRecord');
        return this;
    }
    /**
     * Writes multiple data records into a delivery stream in a single call, which can achieve higher throughput per producer than when writing single records.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_PutRecordBatch.html
     */
    putRecordBatch() {
        this.add('firehose:PutRecordBatch');
        return this;
    }
    /**
     * Enables server-side encryption (SSE) for the delivery stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_StartDeliveryStreamEncryption.html
     */
    startDeliveryStreamEncryption() {
        this.add('firehose:StartDeliveryStreamEncryption');
        return this;
    }
    /**
     * Disables the specified destination of the specified delivery stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_StopDeliveryStreamEncryption.html
     */
    stopDeliveryStreamEncryption() {
        this.add('firehose:StopDeliveryStreamEncryption');
        return this;
    }
    /**
     * Adds or updates tags for the specified delivery stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_TagDeliveryStream.html
     */
    tagDeliveryStream() {
        this.add('firehose:TagDeliveryStream');
        return this;
    }
    /**
     * Removes tags from the specified delivery stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_UntagDeliveryStream.html
     */
    untagDeliveryStream() {
        this.add('firehose:UntagDeliveryStream');
        return this;
    }
    /**
     * Updates the specified destination of the specified delivery stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_UpdateDestination.html
     */
    updateDestination() {
        this.add('firehose:UpdateDestination');
        return this;
    }
    /**
     * Adds a resource of type deliverystream to the statement
     *
     * https://docs.aws.amazon.com/firehose/latest/dev/basic-create.html
     *
     * @param deliveryStreamName - Identifier for the deliveryStreamName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDeliverystream(deliveryStreamName, account, region, partition) {
        var arn = 'arn:${Partition}:firehose:${Region}:${Account}:deliverystream/${DeliveryStreamName}';
        arn = arn.replace('${DeliveryStreamName}', deliveryStreamName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Firehose = Firehose;
//# sourceMappingURL=data:application/json;base64,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